/* packet-ieee17221.c
 * Dissector for IEEE P1722.1
 * Copyright 2011-2013, Thomas Bottom <tom.bottom@labxtechnologies.com>
 *                      Chris Pane <chris.pane@labxtechnologies.com>
 *                      Chris Wulff <chris.wulff@labxtechnologies.com>
 *
 * Copyright 2011, Andy Lucas <andy@xmos.com>
 *
 * Wireshark - Network traffic analyzer
 * By Gerald Combs <gerald@wireshark.org>
 * Copyright 1998 Gerald Combs
 *
 * SPDX-License-Identifier: GPL-2.0-or-later
 *
 */

/* DEV NOTES
 * This file uses 3 space indentation
 */

#include "config.h"

#include <epan/packet.h>

void proto_register_17221(void);
void proto_reg_handoff_17221(void);

static dissector_handle_t avb17221_handle;

/* 17221 Offsets */
#define P1722_HEADER_OFFSET                 12

/* 1722.1 ADP Offsets */
#define ADP_CD_OFFSET                       0
#define ADP_VERSION_OFFSET                  1
#define ADP_VALID_TIME_OFFSET               2
#define ADP_CD_LENGTH_OFFSET                3
#define ADP_ENTITY_ID_OFFSET                4
#define ADP_ENTITY_MODEL_ID_OFFSET          P1722_HEADER_OFFSET+0
#define ADP_ENTITY_CAP_OFFSET               P1722_HEADER_OFFSET+8
#define ADP_TALKER_STREAM_SRCS_OFFSET       P1722_HEADER_OFFSET+12
#define ADP_TALKER_CAP_OFFSET               P1722_HEADER_OFFSET+14
#define ADP_LISTENER_STREAM_SINKS_OFFSET    P1722_HEADER_OFFSET+16
#define ADP_LISTENER_CAP_OFFSET             P1722_HEADER_OFFSET+18
#define ADP_CONTROLLER_CAP_OFFSET           P1722_HEADER_OFFSET+20
#define ADP_AVAIL_INDEX_OFFSET              P1722_HEADER_OFFSET+24
#define ADP_AS_GM_ID_OFFSET                 P1722_HEADER_OFFSET+28
#define ADP_GPTP_DOMAIN_NUMBER_OFFSET       P1722_HEADER_OFFSET+36
#define ADP_IDENTIFY_CONTROL_INDEX          P1722_HEADER_OFFSET+40
#define ADP_INTERFACE_INDEX                 P1722_HEADER_OFFSET+42
#define ADP_ASSOC_ID_OFFSET                 P1722_HEADER_OFFSET+44

/* Bit Field Masks */

#define ADP_MSG_TYPE_MASK                   0x0f
#define ADP_VALID_TIME_MASK                 0xf8
#define ADP_CD_LENGTH_MASK                  0x07ff

/* message_type */

#define ADP_ENTITY_AVAILABLE_MESSAGE        0x00
#define ADP_ENTITY_DEPARTING_MESSAGE        0x01
#define ADP_ENTITY_DISCOVER_MESSAGE         0x02

/* entity_capabilities_flags                            */
#define ADP_EFU_MODE_BITMASK                 0x00000001
#define ADP_ADDRESS_ACCESS_BITMASK           0x00000002
#define ADP_GATEWAY_ENTITY_BITMASK           0x00000004
#define ADP_AEM_SUPPORTED_BITMASK            0x00000008
#define ADP_LEGACY_AVC_BITMASK               0x00000010
#define ADP_ASSOC_ID_SUPPORT_BITMASK         0x00000020
#define ADP_ASSOC_ID_VALID_BITMASK           0x00000040
#define ADP_VENDOR_UNIQUE_BITMASK            0x00000080
#define ADP_CLASS_A_SUPPORTED_BITMASK        0x00000100
#define ADP_CLASS_B_SUPPORTED_BITMASK        0x00000200
#define ADP_AS_SUPPORTED_BITMASK             0x00000400

/* talker capabilities flags                            */
#define ADP_TALK_IMPLEMENTED_BITMASK         0x0001
#define ADP_TALK_OTHER_SRC_BITMASK           0x0200
#define ADP_TALK_CONTROL_SRC_BITMASK         0x0400
#define ADP_TALK_MEDIA_CLK_SRC_BITMASK       0x0800
#define ADP_TALK_SMPTE_SRC_BITMASK           0x1000

#define ADP_TALK_MIDI_SRC_BITMASK            0x2000
#define ADP_TALK_AUDIO_SRC_BITMASK           0x4000
#define ADP_TALK_VIDEO_SRC_BITMASK           0x8000

/* listener capabilities flags                            */
#define ADP_LIST_IMPLEMENTED_BITMASK         0x0001
#define ADP_LIST_OTHER_SINK_BITMASK          0x0200
#define ADP_LIST_CONTROL_SINK_BITMASK        0x0400
#define ADP_LIST_MEDIA_CLK_SINK_BITMASK      0x0800
#define ADP_LIST_SMPTE_SINK_BITMASK          0x1000
#define ADP_LIST_MIDI_SINK_BITMASK           0x2000
#define ADP_LIST_AUDIO_SINK_BITMASK          0x4000
#define ADP_LIST_VIDEO_SINK_BITMASK          0x8000

/* Controller capabilities flags                        */
#define ADP_CONT_IMPLEMENTED_BITMASK         0x0001
#define ADP_CONT_LAYER3_PROXY_BITMASK        0x0002

/* Default audio formats fields */
#define ADP_DEF_AUDIO_SAMPLE_RATES_MASK      0xFC
#define ADP_DEF_AUDIO_MAX_CHANS_MASK         0x03FC
#define ADP_DEF_AUDIO_SAF_MASK               0x0002
#define ADP_DEF_AUDIO_FLOAT_MASK             0x0001

/* Default sample rates flags */
#define ADP_SAMP_RATE_44K1_BITMASK           0x01<<2
#define ADP_SAMP_RATE_48K_BITMASK            0x02<<2
#define ADP_SAMP_RATE_88K2_BITMASK           0x04<<2
#define ADP_SAMP_RATE_96K_BITMASK            0x08<<2
#define ADP_SAMP_RATE_176K4_BITMASK          0x10<<2
#define ADP_SAMP_RATE_192K_BITMASK           0x20<<2

/* channel_formats flags */

#define ADP_CHAN_FORMAT_MONO                        (0x00000001)
#define ADP_CHAN_FORMAT_2CH                         (0x00000002)
#define ADP_CHAN_FORMAT_3CH                         (0x00000004)
#define ADP_CHAN_FORMAT_4CH                         (0x00000008)
#define ADP_CHAN_FORMAT_5CH                         (0x00000010)
#define ADP_CHAN_FORMAT_6CH                         (0x00000020)
#define ADP_CHAN_FORMAT_7CH                         (0x00000040)
#define ADP_CHAN_FORMAT_8CH                         (0x00000080)
#define ADP_CHAN_FORMAT_10CH                        (0x00000100)
#define ADP_CHAN_FORMAT_12CH                        (0x00000200)
#define ADP_CHAN_FORMAT_14CH                        (0x00000400)
#define ADP_CHAN_FORMAT_16CH                        (0x00000800)
#define ADP_CHAN_FORMAT_18CH                        (0x00001000)
#define ADP_CHAN_FORMAT_20CH                        (0x00002000)
#define ADP_CHAN_FORMAT_22CH                        (0x00004000)
#define ADP_CHAN_FORMAT_24CH                        (0x00008000)

/******************************************************************************/
/* 1722.1 ACMP Offsets */
#define ACMP_CD_OFFSET                      0
#define ACMP_VERSION_OFFSET                 1
#define ACMP_STATUS_FIELD_OFFSET            2
#define ACMP_CD_LENGTH_OFFSET               2
#define ACMP_STREAM_ID_OFFSET               4
#define ACMP_CONTROLLER_GUID_OFFSET         12
#define ACMP_TALKER_GUID_OFFSET             20
#define ACMP_LISTENER_GUID_OFFSET           28
#define ACMP_TALKER_UNIQUE_ID_OFFSET        36
#define ACMP_LISTENER_UNIQUE_ID_OFFSET      38
#define ACMP_DEST_MAC_OFFSET                40
#define ACMP_CONNECTION_COUNT_OFFSET        46
#define ACMP_SEQUENCE_ID_OFFSET             48
#define ACMP_FLAGS_OFFSET                   50
#define ACMP_VLAN_ID_OFFSET                 52

/* Bit Field Masks */

#define ACMP_MSG_TYPE_MASK                  0x0F
#define ACMP_STATUS_FIELD_MASK              0xF8
#define ACMP_CD_LENGTH_MASK                 0x07FF

/* message_type */

#define ACMP_CONNECT_TX_COMMAND             0
#define ACMP_CONNECT_TX_RESPONSE            1
#define ACMP_DISCONNECT_TX_COMMAND          2
#define ACMP_DISCONNECT_TX_RESPONSE         3
#define ACMP_GET_TX_STATE_COMMAND           4
#define ACMP_GET_TX_STATE_RESPONSE          5
#define ACMP_CONNECT_RX_COMMAND             6
#define ACMP_CONNECT_RX_RESPONSE            7
#define ACMP_DISCONNECT_RX_COMMAND          8
#define ACMP_DISCONNECT_RX_RESPONSE         9
#define ACMP_GET_RX_STATE_COMMAND           10
#define ACMP_GET_RX_STATE_RESPONSE          11
#define ACMP_GET_TX_CONNECTION_COMMAND      12
#define ACMP_GET_TX_CONNECTION_RESPONSE     13

/* status_field */

#define ACMP_STATUS_SUCCESS                             0
#define ACMP_STATUS_LISTENER_UNKNOWN_ID                 1
#define ACMP_STATUS_TALKER_UNKNOWN_ID                   2
#define ACMP_STATUS_TALKER_DEST_MAC_FAIL                3
#define ACMP_STATUS_TALKER_NO_STREAM_INDEX              4
#define ACMP_STATUS_TALKER_NO_BANDWIDTH                 5
#define ACMP_STATUS_TALKER_EXCLUSIVE                    6
#define ACMP_STATUS_LISTENER_TALKER_TIMEOUT             7
#define ACMP_STATUS_LISTENER_EXCLUSIVE                  8
#define ACMP_STATUS_STATE_UNAVAILABLE                   9
#define ACMP_STATUS_NOT_CONNECTED                       10
#define ACMP_STATUS_NO_SUCH_CONNECTION                  11
#define ACMP_STATUS_COULD_NOT_SEND_MESSAGE              12
#define ACMP_STATUS_DEFAULT_SET_DIFFERENT               15
#define ACMP_STATUS_NOT_SUPPORTED                       31

/* ACMP flags                                   */
#define ACMP_FLAG_CLASS_B_BITMASK               0x0001
#define ACMP_FLAG_FAST_CONNECT_BITMASK          0x0002
#define ACMP_FLAG_SAVED_STATE_BITMASK           0x0004
#define ACMP_FLAG_STREAMING_WAIT_BITMASK        0x0008
#define ACMP_FLAG_SUPPORTS_ENCRYPTED_BITMASK    0x0010
#define ACMP_FLAG_ENCRYPTED_PDU_BITMASK         0x0020
#define ACMP_FLAG_TALKER_FAILED_BITMASK         0x0040

/******************************************************************************/
/* AECP Common Offsets */
#define AECP_OFFSET_VERSION                        1
#define AECP_OFFSET_CD_LENGTH                      2
#define AECP_OFFSET_TARGET_GUID                    4
#define AECP_OFFSET_CONTROLLER_GUID                12
#define AECP_OFFSET_SEQUENCE_ID                    20
#define AECP_OFFSET_U_FLAG                         22
#define AECP_OFFSET_COMMAND_TYPE                   22

#define AECP_OFFSET_STATUS_CODE                    2

/* AECP Address Access Command Specific Offsets */
#define AECP_AA_OFFSET_COUNT                       22
#define AECP_AA_OFFSET_TLVS_START                  24

/* AECP AEM Command Specific Offsets */
/* Note that these values include the 12 byte common header */

/* ACQUIRE_ENTITY */
#define AECP_OFFSET_ACQUIRE_ENTITY_FLAGS              24
#define AECP_OFFSET_ACQUIRE_ENTITY_OWNER_GUID         28
#define AECP_OFFSET_ACQUIRE_ENTITY_DESCRIPTOR_TYPE    36
#define AECP_OFFSET_ACQUIRE_ENTITY_DESCRIPTOR_INDEX   38

/* LOCK_ENTITY */
#define AECP_OFFSET_LOCK_ENTITY_FLAGS                 24
#define AECP_OFFSET_LOCK_ENTITY_LOCKED_GUID           28

/* READ/WRITE_DESCRIPTOR */
#define AECP_OFFSET_DESCRIPTOR_CONFIGURATION_INDEX    24
#define AECP_OFFSET_DESCRIPTOR_DESCRIPTOR             28
#define AECP_OFFSET_DESCRIPTOR_DESCRIPTOR_TYPE        28
#define AECP_OFFSET_DESCRIPTOR_DESCRIPTOR_INDEX       30

/* GET/SET_CONFIGURATION */
#define AECP_OFFSET_CONFIGURATION_CONFIGURATION_INDEX 26

/* GET/SET_STREAM_FORMAT */
#define AECP_OFFSET_STREAM_FORMAT_DESCRIPTOR_TYPE     24
#define AECP_OFFSET_STREAM_FORMAT_DESCRIPTOR_INDEX    26
#define AECP_OFFSET_STREAM_FORMAT_STREAM_FORMAT       28

/* GET/SET_VIDEO_FORMAT */
#define AECP_OFFSET_VIDEO_FORMAT_DESCRIPTOR_TYPE      24
#define AECP_OFFSET_VIDEO_FORMAT_DESCRIPTOR_INDEX     26
#define AECP_OFFSET_VIDEO_FORMAT_FORMAT_SPECIFIC      28
#define AECP_OFFSET_VIDEO_FORMAT_ASPECT_RATIO         32
#define AECP_OFFSET_VIDEO_FORMAT_COLOR_SPACE          34
#define AECP_OFFSET_VIDEO_FORMAT_FRAME_SIZE           36

/* GET/SET_SENSOR_FORMAT */
#define AECP_OFFSET_SENSOR_FORMAT_DESCRIPTOR_TYPE     24
#define AECP_OFFSET_SENSOR_FORMAT_DESCRIPTOR_INDEX    26
#define AECP_OFFSET_SENSOR_FORMAT_SENSOR_FORMAT       28

/* GET/SET_STREAM_INFO */
#define AECP_OFFSET_STREAM_INFO_DESCRIPTOR_TYPE          24
#define AECP_OFFSET_STREAM_INFO_DESCRIPTOR_INDEX         26
#define AECP_OFFSET_STREAM_INFO_FLAGS                    28
#define AECP_OFFSET_STREAM_INFO_STREAM_FORMAT            32
#define AECP_OFFSET_STREAM_INFO_STREAM_ID                40
#define AECP_OFFSET_STREAM_INFO_MSRP_ACCUMULATED_LATENCY 48
#define AECP_OFFSET_STREAM_INFO_STREAM_DEST_MAC          52
#define AECP_OFFSET_STREAM_INFO_MSRP_FAILURE_CODE        58
#define AECP_OFFSET_STREAM_INFO_MSRP_FAILURE_BRIDGE_ID   60
/** #define AECP_OFFSET_STREAM_INFO_STREAM_VLAN_ID           68 **/

/* GET/SET_NAME */
#define AECP_OFFSET_NAME_DESCRIPTOR_TYPE             24
#define AECP_OFFSET_NAME_DESCRIPTOR_INDEX            26
#define AECP_OFFSET_NAME_NAME_INDEX                  28
#define AECP_OFFSET_NAME_CONFIGURATION_INDEX         30
#define AECP_OFFSET_NAME_NAME                        32

/* GET/SET_ASSOCIATION_ID */
#define AECP_OFFSET_ASSOCIATION_ID_ASSOCIATION_ID    24

/* GET/SET_SAMPLING_RATE */
#define AECP_OFFSET_SAMPLING_RATE_DESCRIPTOR_TYPE    24
#define AECP_OFFSET_SAMPLING_RATE_DESCRIPTOR_INDEX   26
#define AECP_OFFSET_SAMPLING_RATE_SAMPLING_RATE      28

/* GET/SET_CLOCK_SOURCE */
#define AECP_OFFSET_CLOCK_SOURCE_DESCRIPTOR_TYPE     24
#define AECP_OFFSET_CLOCK_SOURCE_DESCRIPTOR_INDEX    26
#define AECP_OFFSET_CLOCK_SOURCE_CLOCK_SOURCE_INDEX  28

/* GET/SET_CONTROL, GET/SET_MIXER */
#define AECP_OFFSET_CONTROL_DESCRIPTOR_TYPE          24
#define AECP_OFFSET_CONTROL_DESCRIPTOR_INDEX         26
#define AECP_OFFSET_CONTROL_VALUES                   28

/* INCREMENT/DECREMENT_CONTROL */
#define AECP_OFFSET_INCDEC_CONTROL_DESCRIPTOR_TYPE   24
#define AECP_OFFSET_INCDEC_CONTROL_DESCRIPTOR_INDEX  26
#define AECP_OFFSET_INCDEC_CONTROL_INDEX_COUNT       28
#define AECP_OFFSET_INCDEC_CONTROL_INDEX_LIST        32

/* GET/SET_SIGNAL_SELECTOR */
#define AECP_OFFSET_SIGNAL_SELECTOR_DESCRIPTOR_TYPE  24
#define AECP_OFFSET_SIGNAL_SELECTOR_DESCRIPTOR_INDEX 26
#define AECP_OFFSET_SIGNAL_SELECTOR_SIGNAL_TYPE      28
#define AECP_OFFSET_SIGNAL_SELECTOR_SIGNAL_INDEX     30
#define AECP_OFFSET_SIGNAL_SELECTOR_SIGNAL_OUTPUT    32

/* GET/SET_MATRIX */
#define AECP_OFFSET_MATRIX_DESCRIPTOR_TYPE           24
#define AECP_OFFSET_MATRIX_DESCRIPTOR_INDEX          26
#define AECP_OFFSET_MATRIX_MATRIX_COLUMN             28
#define AECP_OFFSET_MATRIX_MATRIX_ROW                30
#define AECP_OFFSET_MATRIX_REGION_WIDTH              32
#define AECP_OFFSET_MATRIX_REGION_HEIGHT             34
#define AECP_OFFSET_MATRIX_REP                       36
#define AECP_OFFSET_MATRIX_DIRECTION                 36
#define AECP_OFFSET_MATRIX_VALUE_COUNT               36
#define AECP_OFFSET_MATRIX_ITEM_OFFSET               38
#define AECP_OFFSET_MATRIX_VALUES                    40

/* START/STOP_STREAMING */
#define AECP_OFFSET_STREAMING_DESCRIPTOR_TYPE        24
#define AECP_OFFSET_STREAMING_DESCRIPTOR_INDEX       26

/* REGISTER/DEREGISTER_UNSOLICITED_NOTIFICATION */
/* No additional fields */

/* IDENTIFY_NOTIFICATION */
#define AECP_OFFSET_IDENTIFY_NOTIFICATION_DESCRIPTOR_TYPE  24
#define AECP_OFFSET_IDENTIFY_NOTIFICATION_DESCRIPTOR_INDEX 26

/* GET_AVB_INFO */
#define AECP_OFFSET_AVB_INFO_DESCRIPTOR_TYPE         24
#define AECP_OFFSET_AVB_INFO_DESCRIPTOR_INDEX        26
#define AECP_OFFSET_AVB_INFO_AS_GRANDMASTER_ID       28
#define AECP_OFFSET_AVB_INFO_PROPAGATION_DELAY       36
#define AECP_OFFSET_AVB_INFO_AS_DOMAIN_NUMBER        40
#define AECP_OFFSET_AVB_INFO_FLAGS                   41
#define AECP_OFFSET_AVB_INFO_MSRP_MAPPINGS_COUNT     42
#define AECP_OFFSET_AVB_INFO_MSRP_MAPPINGS           44

/* GET_AS_PATH */
#define AECP_OFFSET_AS_PATH_DESCRIPTOR_INDEX         24
#define AECP_OFFSET_AS_PATH_COUNT                    26
#define AECP_OFFSET_AS_PATH_PATH_SEQUENCE            28

/* GET_COUNTERS */
#define AECP_OFFSET_COUNTERS_DESCRIPTOR_TYPE                   24
#define AECP_OFFSET_COUNTERS_DESCRIPTOR_INDEX                  26
#define AECP_OFFSET_COUNTERS_VALID                             28
#define AECP_OFFSET_COUNTERS_AVB_INTERFACE_LINK_UP             32
#define AECP_OFFSET_COUNTERS_AVB_INTERFACE_LINK_DOWN           36
#define AECP_OFFSET_COUNTERS_AVB_INTERFACE_PACKETS_TX          40
#define AECP_OFFSET_COUNTERS_AVB_INTERFACE_PACKETS_RX          44
#define AECP_OFFSET_COUNTERS_AVB_INTERFACE_RX_CRC_ERROR        48
#define AECP_OFFSET_COUNTERS_AVB_INTERFACE_GPTP_GM_CHANGED     52
#define AECP_OFFSET_COUNTERS_CLOCK_DOMAIN_LOCKED               32
#define AECP_OFFSET_COUNTERS_CLOCK_DOMAIN_UNLOCKED             36
#define AECP_OFFSET_COUNTERS_STREAM_INPUT_MEDIA_LOCKED         32
#define AECP_OFFSET_COUNTERS_STREAM_INPUT_MEDIA_UNLOCKED       36
#define AECP_OFFSET_COUNTERS_STREAM_INPUT_STREAM_RESET         40
#define AECP_OFFSET_COUNTERS_STREAM_INPUT_SEQ_NUM_MISMATCH     44
#define AECP_OFFSET_COUNTERS_STREAM_INPUT_MEDIA_RESET          48
#define AECP_OFFSET_COUNTERS_STREAM_INPUT_TIMESTAMP_UNCERTAIN  52
#define AECP_OFFSET_COUNTERS_STREAM_INPUT_TIMESTAMP_VALID      56
#define AECP_OFFSET_COUNTERS_STREAM_INPUT_TIMESTAMP_NOT_VALID  60
#define AECP_OFFSET_COUNTERS_STREAM_INPUT_UNSUPPORTED_FORMAT   64
#define AECP_OFFSET_COUNTERS_STREAM_INPUT_LATE_TIMESTAMP       68
#define AECP_OFFSET_COUNTERS_STREAM_INPUT_EARLY_TIMESTAMP      72
#define AECP_OFFSET_COUNTERS_STREAM_INPUT_PACKETS_TX           76
#define AECP_OFFSET_COUNTERS_STREAM_INPUT_PACKETS_RX           80
#define AECP_OFFSET_COUNTERS_ENTITY_SPECIFIC_8                 128
#define AECP_OFFSET_COUNTERS_ENTITY_SPECIFIC_7                 132
#define AECP_OFFSET_COUNTERS_ENTITY_SPECIFIC_6                 136
#define AECP_OFFSET_COUNTERS_ENTITY_SPECIFIC_5                 140
#define AECP_OFFSET_COUNTERS_ENTITY_SPECIFIC_4                 144
#define AECP_OFFSET_COUNTERS_ENTITY_SPECIFIC_3                 148
#define AECP_OFFSET_COUNTERS_ENTITY_SPECIFIC_2                 152
#define AECP_OFFSET_COUNTERS_ENTITY_SPECIFIC_1                 156

/* GET_AUDIO_MAP, GET_VIDEO_MAP, GET_SENSOR_MAP */
#define AECP_OFFSET_MAP_DESCRIPTOR_TYPE                24
#define AECP_OFFSET_MAP_DESCRIPTOR_INDEX               26
#define AECP_OFFSET_MAP_MAP_INDEX                      28
#define AECP_OFFSET_MAP_NUMBER_OF_MAPS                 30
#define AECP_OFFSET_MAP_NUMBER_OF_MAPPINGS             32
#define AECP_OFFSET_MAP_MAPPINGS                       36

/* ADD/REMOVE_AUDIO_MAPPINGS, ADD/REMOVE_VIDEO_MAPPINGS, ADD/REMOVE_SENSOR_MAPPINGS */
#define AECP_OFFSET_ADDREM_MAP_DESCRIPTOR_TYPE         24
#define AECP_OFFSET_ADDREM_MAP_DESCRIPTOR_INDEX        26
#define AECP_OFFSET_ADDREM_MAP_NUMBER_OF_MAPPINGS      28
#define AECP_OFFSET_ADDREM_MAP_MAPPINGS                32

/* START_OPERATION */
#define AECP_OFFSET_START_OPERATION_DESCRIPTOR_TYPE    24
#define AECP_OFFSET_START_OPERATION_DESCRIPTOR_INDEX   26
#define AECP_OFFSET_START_OPERATION_OPERATION_ID       28
#define AECP_OFFSET_START_OPERATION_OPERATION_TYPE     30
#define AECP_OFFSET_START_OPERATION_VALUES             32

/* ABORT_OPERATION */
#define AECP_OFFSET_ABORT_OPERATION_DESCRIPTOR_TYPE    24
#define AECP_OFFSET_ABORT_OPERATION_DESCRIPTOR_INDEX   26
#define AECP_OFFSET_ABORT_OPERATION_OPERATION_ID       28

/* OPERATION_STATUS */
#define AECP_OFFSET_OPERATION_STATUS_DESCRIPTOR_TYPE   24
#define AECP_OFFSET_OPERATION_STATUS_DESCRIPTOR_INDEX  26
#define AECP_OFFSET_OPERATION_STATUS_OPERATION_ID      28
#define AECP_OFFSET_OPERATION_STATUS_PERCENT_COMPLETE  30

/* AUTH_ADD/DELETE_KEY, AUTH_GET_KEY */
#define AECP_OFFSET_AUTH_KEY_KEY_EUI                   24
#define AECP_OFFSET_AUTH_KEY_KEY_TYPE                  32
#define AECP_OFFSET_AUTH_KEY_KEY_LENGTH                32
#define AECP_OFFSET_AUTH_KEY_KEY                       36

/* AUTH_GET_KEY_LIST */
#define AECP_OFFSET_AUTH_GET_KEY_LIST_KEYCHAIN_ID      24
#define AECP_OFFSET_AUTH_GET_KEY_LIST_NUMBER_OF_KEYS   24

/* AUTH_ADD_KEY_TO_KEYCHAIN, AUTH_DELETE_KEY_FROM_KEYCHAIN */
#define AECP_OFFSET_AUTH_KEYCHAIN_KEYCHAIN_ID          24
#define AECP_OFFSET_AUTH_KEYCHAIN_KEY_EUI              28

/* AUTH_GET_KEYCHAIN_LIST */
#define AECP_OFFSET_AUTH_GET_KEYCHAIN_LIST_KEYCHAIN_ID     24
#define AECP_OFFSET_AUTH_GET_KEYCHAIN_LIST_LIST_INDEX      26
#define AECP_OFFSET_AUTH_GET_KEYCHAIN_LIST_NUMBER_OF_LISTS 28
#define AECP_OFFSET_AUTH_GET_KEYCHAIN_LIST_NUMBER_OF_KEYS  30
#define AECP_OFFSET_AUTH_GET_KEYCHAIN_LIST_KEY_EUIS        32

/* AUTH_GET_IDENTITY */
#define AECP_OFFSET_AUTH_GET_IDENTITY_KEY_GUID         24
#define AECP_OFFSET_AUTH_GET_IDENTITY_SIGNATURE        32

/* AUTH_ADD/DELETE_TOKEN */
#define AECP_OFFSET_AUTH_TOKEN_TOKEN_LENGTH            24
#define AECP_OFFSET_AUTH_TOKEN_AUTHENTICATION_TOKEN    28

/* AUTHENTICATE/DEAUTHENTICATE */
#define AECP_OFFSET_AUTHENTICATE_DESCRIPTOR_TYPE       24
#define AECP_OFFSET_AUTHENTICATE_DESCRIPTOR_INDEX      26
#define AECP_OFFSET_AUTHENTICATE_TOKEN_LENGTH          28
#define AECP_OFFSET_AUTHENTICATE_AUTH_TOKEN            32

/* ENABLE/DISABLE_TRANSPORT_SECURITY */
#define AECP_OFFSET_TRANSPORT_SECURITY_KEY_EUI         24

/* ENABLE/DISABLE_STREAM_ENCRYPTION */
#define AECP_OFFSET_STREAM_ENCRYPTION_DESCRIPTOR_TYPE  24
#define AECP_OFFSET_STREAM_ENCRYPTION_DESCRIPTOR_INDEX 26
#define AECP_OFFSET_STREAM_ENCRYPTION_KEY_EUI          28

/* Set/Get Stream Backup */
#define AECP_OFFSET_STREAM_BACKUP_DESCRIPTOR_TYPE        24
#define AECP_OFFSET_STREAM_BACKUP_DESCRIPTOR_INDEX       26
#define AECP_OFFSET_STREAM_BACKUP_TALKER_ENTITY_ID_0     28
#define AECP_OFFSET_STREAM_BACKUP_TALKER_UNIQUE_ID_0     36
#define AECP_OFFSET_STREAM_BACKUP_TALKER_ENTITY_ID_1     40
#define AECP_OFFSET_STREAM_BACKUP_TALKER_UNIQUE_ID_1     48
#define AECP_OFFSET_STREAM_BACKUP_TALKER_ENTITY_ID_2     52
#define AECP_OFFSET_STREAM_BACKUP_TALKER_UNIQUE_ID_2     60
#define AECP_OFFSET_STREAM_BACKEDUP_TALKER_ENTITY_ID     64
#define AECP_OFFSET_STREAM_BACKEDUP_TALKER_UNIQUE_ID     72

/* aecp_message_type */
/* 1722.1 draft 2011-11-14 table 9.1 */
#define AECP_AEM_COMMAND_MESSAGE                0
#define AECP_AEM_RESPONSE_MESSAGE               1
#define AECP_ADDRESS_ACCESS_COMMAND_MESSAGE     2
#define AECP_ADDRESS_ACCESS_RESPONSE_MESSAGE    3
#define AECP_AVC_COMMAND_MESSAGE                4
#define AECP_AVC_RESPONSE_MESSAGE               5
#define AECP_VENDOR_UNIQUE_COMMAND_MESSAGE      6
#define AECP_VENDOR_UNIQUE_RESPONSE_MESSAGE     7
#define AECP_EXTENDED_COMMAND_MESSAGE           14
#define AECP_EXTENDED_RESPONSE_MESSAGE          15

/* aecp status field */
/* 1722.1 draft 2011-11-14 table 9.2 */
#define AECP_STATUS_SUCCESS          0
#define AECP_STATUS_NOT_IMPLEMENTED  1

/* AECP Address Type */
/* 1722.1 draft 2011-11-14 sec 7.4.35.1 table 7.82 */
#define AECP_ADDRESS_MAC  0x0000
#define AECP_ADDRESS_IPV4 0x0001
#define AECP_ADDRESS_IPV6 0x0002

/* AECP Direction Field Values */
/* 1722.1 draft 2011-11-14 table 7.78 */
#define AECP_DIRECTION_HORIZONTAL 0
#define AECP_DIRECTION_VERTICAL   1

/* AECP Keychain Type Values */
/* 1722.1 draft 2011-11-14 table 7.80 */
#define AECP_KEYCHAIN_NONE         0x0
#define AECP_KEYCHAIN_MANUFACTURER 0x1
#define AECP_KEYCHAIN_PRODUCT      0x2
#define AECP_KEYCHAIN_ENTITY       0x3
#define AECP_KEYCHAIN_USER         0x4

/* AECP Command Codes */
/* IEEE 1722.1 draft D21 Table 7.126 */
#define AECP_COMMAND_ACQUIRE_ENTITY                 0x0000
#define AECP_COMMAND_LOCK_ENTITY                    0x0001
#define AECP_COMMAND_ENTITY_AVAILABLE               0x0002
#define AECP_COMMAND_CONTROLLER_AVAILABLE           0x0003
#define AECP_COMMAND_READ_DESCRIPTOR                0x0004
#define AECP_COMMAND_WRITE_DESCRIPTOR               0x0005
#define AECP_COMMAND_SET_CONFIGURATION              0x0006
#define AECP_COMMAND_GET_CONFIGURATION              0x0007
#define AECP_COMMAND_SET_STREAM_FORMAT              0x0008
#define AECP_COMMAND_GET_STREAM_FORMAT              0x0009
#define AECP_COMMAND_SET_VIDEO_FORMAT               0x000a
#define AECP_COMMAND_GET_VIDEO_FORMAT               0x000b
#define AECP_COMMAND_SET_SENSOR_FORMAT              0x000c
#define AECP_COMMAND_GET_SENSOR_FORMAT              0x000d
#define AECP_COMMAND_SET_STREAM_INFO                0x000e
#define AECP_COMMAND_GET_STREAM_INFO                0x000f
#define AECP_COMMAND_SET_NAME                       0x0010
#define AECP_COMMAND_GET_NAME                       0x0011
#define AECP_COMMAND_SET_ASSOCIATION_ID             0x0012
#define AECP_COMMAND_GET_ASSOCIATION_ID             0x0013
#define AECP_COMMAND_SET_SAMPLING_RATE              0x0014
#define AECP_COMMAND_GET_SAMPLING_RATE              0x0015
#define AECP_COMMAND_SET_CLOCK_SOURCE               0x0016
#define AECP_COMMAND_GET_CLOCK_SOURCE               0x0017
#define AECP_COMMAND_SET_CONTROL_VALUE              0x0018
#define AECP_COMMAND_GET_CONTROL_VALUE              0x0019
#define AECP_COMMAND_INCREMENT_CONTROL              0x001a
#define AECP_COMMAND_DECREMENT_CONTROL              0x001b
#define AECP_COMMAND_SET_SIGNAL_SELECTOR            0x001c
#define AECP_COMMAND_GET_SIGNAL_SELECTOR            0x001d
#define AECP_COMMAND_SET_MIXER                      0x001e
#define AECP_COMMAND_GET_MIXER                      0x001f
#define AECP_COMMAND_SET_MATRIX                     0x0020
#define AECP_COMMAND_GET_MATRIX                     0x0021
#define AECP_COMMAND_START_STREAMING                0x0022
#define AECP_COMMAND_STOP_STREAMING                 0x0023
#define AECP_COMMAND_REGISTER_UNSOL_NOTIFICATION    0x0024
#define AECP_COMMAND_DEREGISTER_UNSOL_NOTIFICATION  0x0025
#define AECP_COMMAND_IDENTIFY_NOTIFICATION          0x0026
#define AECP_COMMAND_GET_AVB_INFO                   0x0027
#define AECP_COMMAND_GET_AS_PATH                    0x0028
#define AECP_COMMAND_GET_COUNTERS                   0x0029
#define AECP_COMMAND_REBOOT                         0x002a
#define AECP_COMMAND_GET_AUDIO_MAP                  0x002b
#define AECP_COMMAND_ADD_AUDIO_MAPPINGS             0x002c
#define AECP_COMMAND_REMOVE_AUDIO_MAPPINGS          0x002d
#define AECP_COMMAND_GET_VIDEO_MAP                  0x002e
#define AECP_COMMAND_ADD_VIDEO_MAPPINGS             0x002f
#define AECP_COMMAND_REMOVE_VIDEO_MAPPINGS          0x0030
#define AECP_COMMAND_GET_SENSOR_MAP                 0x0031
#define AECP_COMMAND_ADD_SENSOR_MAPPINGS            0x0032
#define AECP_COMMAND_REMOVE_SENSOR_MAPPINGS         0x0033
#define AECP_COMMAND_START_OPERATION                0x0034
#define AECP_COMMAND_ABORT_OPERATION                0x0035
#define AECP_COMMAND_OPERATION_STATUS               0x0036
#define AECP_COMMAND_AUTH_ADD_KEY                   0x0037
#define AECP_COMMAND_AUTH_DELETE_KEY                0x0038
#define AECP_COMMAND_AUTH_GET_KEY_LIST              0x0039
#define AECP_COMMAND_AUTH_GET_KEY                   0x003a
#define AECP_COMMAND_AUTH_ADD_KEY_TO_CHAIN          0x003b
#define AECP_COMMAND_AUTH_DELETE_KEY_FROM_CHAIN     0x003c
#define AECP_COMMAND_AUTH_GET_KEYCHAIN_LIST         0x003d
#define AECP_COMMAND_AUTH_GET_IDENTITY              0x003e
#define AECP_COMMAND_AUTH_ADD_TOKEN                 0x003f
#define AECP_COMMAND_AUTH_DELETE_TOKEN              0x0040
#define AECP_COMMAND_AUTHENTICATE                   0x0041
#define AECP_COMMAND_DEAUTHENTICATE                 0x0042
#define AECP_COMMAND_ENABLE_TRANSPORT_SECURITY      0x0043
#define AECP_COMMAND_DISABLE_TRANSPORT_SECURITY     0x0044
#define AECP_COMMAND_ENABLE_STREAM_ENCRYPTION       0x0045
#define AECP_COMMAND_DISABLE_STREAM_ENCRYPTION      0x0046
#define AECP_COMMAND_SET_STREAM_BACKUP              0x0049
#define AECP_COMMAND_GET_STREAM_BACKUP              0x004a
/* 0x002e - 0x7ffe RESERVED for future use */
#define AECP_COMMAND_EXPANSION                      0x7fff /* reserved */

/* AEM common format packet STATUS field values */
/* IEEE 1722.1 draft D21 Table 7.127 */
#define AEM_STATUS_SUCCESS                 0
#define AEM_STATUS_NOT_IMPLEMENTED         1
#define AEM_STATUS_NO_SUCH_DESCRIPTOR      2
#define AEM_STATUS_ENTITY_LOCKED           3
#define AEM_STATUS_ENTITY_ACQUIRED         4
#define AEM_STATUS_NOT_AUTHENTICATED       5
#define AEM_STATUS_AUTHENTICATION_DISABLED 6
#define AEM_STATUS_BAD_ARGUMENTS           7
#define AEM_STATUS_NO_RESOURCES            8
#define AEM_STATUS_IN_PROGRESS             9
#define AEM_STATUS_ENTITY_MISBEHAVING      10
#define AEM_STATUS_NOT_SUPPORTED           11
#define AEM_STATUS_STREAM_IS_RUNNING       12
/* 10 - 31 RESERVED */

/* AEM descriptor indices */
/* IEEE 1722.1 draft D21 TABLE 7.1 */
#define AEM_DESCRIPTOR_ENTITY                0x0000
#define AEM_DESCRIPTOR_CONFIGURATION         0x0001
#define AEM_DESCRIPTOR_AUDIO_UNIT            0x0002
#define AEM_DESCRIPTOR_VIDEO_UNIT            0x0003
#define AEM_DESCRIPTOR_SENSOR_UNIT           0x0004
#define AEM_DESCRIPTOR_STREAM_INPUT          0x0005
#define AEM_DESCRIPTOR_STREAM_OUTPUT         0x0006
#define AEM_DESCRIPTOR_JACK_INPUT            0x0007
#define AEM_DESCRIPTOR_JACK_OUTPUT           0x0008
#define AEM_DESCRIPTOR_AVB_INTERFACE         0x0009
#define AEM_DESCRIPTOR_CLOCK_SOURCE          0x000a
#define AEM_DESCRIPTOR_MEMORY_OBJECT         0x000b
#define AEM_DESCRIPTOR_LOCALE                0x000c
#define AEM_DESCRIPTOR_STRINGS               0x000d
#define AEM_DESCRIPTOR_STREAM_PORT_INPUT     0x000e
#define AEM_DESCRIPTOR_STREAM_PORT_OUTPUT    0x000f
#define AEM_DESCRIPTOR_EXTERNAL_PORT_INPUT   0x0010
#define AEM_DESCRIPTOR_EXTERNAL_PORT_OUTPUT  0x0011
#define AEM_DESCRIPTOR_INTERNAL_PORT_INPUT   0x0012
#define AEM_DESCRIPTOR_INTERNAL_PORT_OUTPUT  0x0013
#define AEM_DESCRIPTOR_AUDIO_CLUSTER         0x0014
#define AEM_DESCRIPTOR_VIDEO_CLUSTER         0x0015
#define AEM_DESCRIPTOR_SENSOR_CLUSTER        0x0016
#define AEM_DESCRIPTOR_AUDIO_MAP             0x0017
#define AEM_DESCRIPTOR_VIDEO_MAP             0x0018
#define AEM_DESCRIPTOR_SENSOR_MAP            0x0019
#define AEM_DESCRIPTOR_CONTROL               0x001a
#define AEM_DESCRIPTOR_SIGNAL_SELECTOR       0x001b
#define AEM_DESCRIPTOR_MIXER                 0x001c
#define AEM_DESCRIPTOR_MATRIX                0x001d
#define AEM_DESCRIPTOR_MATRIX_SIGNAL         0x001e
#define AEM_DESCRIPTOR_SIGNAL_SPLITTER       0x001f
#define AEM_DESCRIPTOR_SIGNAL_COMBINER       0x0020
#define AEM_DESCRIPTOR_SIGNAL_DEMULTIPLEXER  0x0021
#define AEM_DESCRIPTOR_SIGNAL_MULTIPLEXER    0x0022
#define AEM_DESCRIPTOR_SIGNAL_TRANSCODER     0x0023
#define AEM_DESCRIPTOR_CLOCK_DOMAIN          0x0024
#define AEM_DESCRIPTOR_CONTROL_BLOCK         0x0025
#define AEM_DESCRIPTOR_INVALID               0xFFFF

/* AEM JACK TYPES */
/* IEEE 1722.1 draft D21 TABLE 7.12 */
#define AEM_JACKTYPE_SPEAKER              0x0000
#define AEM_JACKTYPE_HEADPHONE            0x0001
#define AEM_JACKTYPE_ANALOG_MICROPHONE    0x0002
#define AEM_JACKTYPE_SPDIF                0x0003
#define AEM_JACKTYPE_ADAT                 0x0004
#define AEM_JACKTYPE_TDIF                 0x0005
#define AEM_JACKTYPE_MADI                 0x0006
#define AEM_JACKTYPE_UNBALANCED_ANALOG    0x0007
#define AEM_JACKTYPE_BALANCED_ANALOG      0x0008
#define AEM_JACKTYPE_DIGITAL              0x0009
#define AEM_JACKTYPE_MIDI                 0x000a
#define AEM_JACKTYPE_AES_EBU              0x000b
#define AEM_JACKTYPE_COMPOSITE_VIDEO      0x000c
#define AEM_JACKTYPE_S_VHS_VIDEO          0x000d
#define AEM_JACKTYPE_COMPONENT_VIDEO      0x000e
#define AEM_JACKTYPE_DVI                  0x000f
#define AEM_JACKTYPE_HDMI                 0x0010
#define AEM_JACKTYPE_UDI                  0x0011
#define AEM_JACKTYPE_DISPLAYPORT          0x0012
#define AEM_JACKTYPE_ANTENNA              0x0013
#define AEM_JACKTYPE_ANALOG_TUNER         0x0014
#define AEM_JACKTYPE_ETHERNET             0x0015
#define AEM_JACKTYPE_WIFI                 0x0016
#define AEM_JACKTYPE_USB                  0x0017
#define AEM_JACKTYPE_PCI                  0x0018
#define AEM_JACKTYPE_PCI_E                0x0019
#define AEM_JACKTYPE_SCSI                 0x001a
#define AEM_JACKTYPE_ATA                  0x001b
#define AEM_JACKTYPE_IMAGER               0x001c
#define AEM_JACKTYPE_IR                   0x001d
#define AEM_JACKTYPE_THUNDERBOLT          0x001e
#define AEM_JACKTYPE_SATA                 0x001f
#define AEM_JACKTYPE_SMPTE_LTC            0x0020
#define AEM_JACKTYPE_DIGITAL_MICROPHONE   0x0021

/* CONTROL TYPES */
/* IEEE 1722.1 draft D21 TABLE 7.94 */
#define AEM_CONTROL_ENABLE                   0x90e0f00000000000
#define AEM_CONTROL_IDENTIFY                 0x90e0f00000000001
#define AEM_CONTROL_MUTE                     0x90e0f00000000002
#define AEM_CONTROL_INVERT                   0x90e0f00000000003
#define AEM_CONTROL_GAIN                     0x90e0f00000000004
#define AEM_CONTROL_ATTENUATE                0x90e0f00000000005
#define AEM_CONTROL_DELAY                    0x90e0f00000000006
#define AEM_CONTROL_SRC_MODE                 0x90e0f00000000007
#define AEM_CONTROL_SNAPSHOT                 0x90e0f00000000008
#define AEM_CONTROL_POW_LINE_FREQ            0x90e0f00000000009
#define AEM_CONTROL_POWER_STATUS             0x90e0f0000000000a
#define AEM_CONTROL_FAN_STATUS               0x90e0f0000000000b
#define AEM_CONTROL_TEMPERATURE              0x90e0f0000000000c
#define AEM_CONTROL_ALTITUDE                 0x90e0f0000000000d
#define AEM_CONTROL_ABSOLUTE_HUMIDITY        0x90e0f0000000000e
#define AEM_CONTROL_RELATIVE_HUMIDITY        0x90e0f0000000000f
#define AEM_CONTROL_ORIENTATION              0x90e0f00000000010
#define AEM_CONTROL_VELOCITY                 0x90e0f00000000011
#define AEM_CONTROL_ACCELERATION             0x90e0f00000000012
#define AEM_CONTROL_FILTER_RESPONSE          0x90e0f00000000013
/* 0x90e0f00000000014 - 0x90e0f0000000ffff reserved */
#define AEM_CONTROL_PANPOT                   0x90e0f00000010000
#define AEM_CONTROL_PHANTOM                  0x90e0f00000010001
#define AEM_CONTROL_AUDIO_SCALE              0x90e0f00000010002
#define AEM_CONTROL_AUDIO_METERS             0x90e0f00000010003
#define AEM_CONTROL_AUDIO_SPECTRUM           0x90e0f00000010004
/* 0x90e0f00000010005 - 0x90e0f0000001ffff reserved */
#define AEM_CONTROL_SCANNING_MODE            0x90e0f00000020000
#define AEM_CONTROL_AUTO_EXP_MODE            0x90e0f00000020001
#define AEM_CONTROL_AUTO_EXP_PRIO            0x90e0f00000020002
#define AEM_CONTROL_EXP_TIME                 0x90e0f00000020003
#define AEM_CONTROL_FOCUS                    0x90e0f00000020004
#define AEM_CONTROL_FOCUS_AUTO               0x90e0f00000020005
#define AEM_CONTROL_IRIS                     0x90e0f00000020006
#define AEM_CONTROL_ZOOM                     0x90e0f00000020007
#define AEM_CONTROL_PRIVACY                  0x90e0f00000020008
#define AEM_CONTROL_BACKLIGHT                0x90e0f00000020009
#define AEM_CONTROL_BRIGHTNESS               0x90e0f0000002000a
#define AEM_CONTROL_CONTRAST                 0x90e0f0000002000b
#define AEM_CONTROL_HUE                      0x90e0f0000002000c
#define AEM_CONTROL_SATURATION               0x90e0f0000002000d
#define AEM_CONTROL_SHARPNESS                0x90e0f0000002000e
#define AEM_CONTROL_GAMMA                    0x90e0f0000002000f
#define AEM_CONTROL_WHITE_BAL_TEMP           0x90e0f00000020010
#define AEM_CONTROL_WHITE_BAL_TEMP_AUTO      0x90e0f00000020011
#define AEM_CONTROL_WHITE_BAL_COMP           0x90e0f00000020012
#define AEM_CONTROL_WHITE_BAL_COMP_AUTO      0x90e0f00000020013
#define AEM_CONTROL_DIGITAL_ZOOM             0x90e0f00000020014
/* 0x90e0f00000020015 - 0x90e0f0000002ffff reserved */
#define AEM_CONTROL_MEDIA_PLAYLIST           0x90e0f00000030000
#define AEM_CONTROL_MEDIA_PLAYLIST_NAME      0x90e0f00000030001
#define AEM_CONTROL_MEDIA_DISK               0x90e0f00000030002
#define AEM_CONTROL_MEDIA_DISK_NAME          0x90e0f00000030003
#define AEM_CONTROL_MEDIA_TRACK              0x90e0f00000030004
#define AEM_CONTROL_MEDIA_TRACK_NAME         0x90e0f00000030005
#define AEM_CONTROL_MEDIA_SPEED              0x90e0f00000030006
#define AEM_CONTROL_MEDIA_SAMPLE_POSITION    0x90e0f00000030007
#define AEM_CONTROL_MEDIA_PLAYBACK_TRANSPORT 0x90e0f00000030008
#define AEM_CONTROL_MEDIA_RECORD_TRANSPORT   0x90e0f00000030009
/* 0x90e0f0000003000a - 0x90e0f0000003ffff reserved */
#define AEM_CONTROL_FREQUENCY                0x90e0f00000040000
#define AEM_CONTROL_MODULATION               0x90e0f00000040001
#define AEM_CONTROL_POLARIZATION             0x90e0f00000040002
/* 0x90e0f00000040003 - 0x90e0f0ffffffffff reserved */

/* AEM Control Value Types */
/* IEEE 1722.1 draft D21 TABLE 7.106 */
#define AEM_CONTROL_LINEAR_INT8       0x0000
#define AEM_CONTROL_LINEAR_UINT8      0x0001
#define AEM_CONTROL_LINEAR_INT16      0x0002
#define AEM_CONTROL_LINEAR_UINT16     0x0003
#define AEM_CONTROL_LINEAR_INT32      0x0004
#define AEM_CONTROL_LINEAR_UINT32     0x0005
#define AEM_CONTROL_LINEAR_INT64      0x0006
#define AEM_CONTROL_LINEAR_UINT64     0x0007
#define AEM_CONTROL_LINEAR_FLOAT      0x0008
#define AEM_CONTROL_LINEAR_DOUBLE     0x0009
#define AEM_CONTROL_SELECTOR_INT8     0x000a
#define AEM_CONTROL_SELECTOR_UINT8    0x000b
#define AEM_CONTROL_SELECTOR_INT16    0x000c
#define AEM_CONTROL_SELECTOR_UINT16   0x000d
#define AEM_CONTROL_SELECTOR_INT32    0x000e
#define AEM_CONTROL_SELECTOR_UINT32   0x000f
#define AEM_CONTROL_SELECTOR_INT64    0x0010
#define AEM_CONTROL_SELECTOR_UINT64   0x0011
#define AEM_CONTROL_SELECTOR_FLOAT    0x0012
#define AEM_CONTROL_SELECTOR_DOUBLE   0x0013
#define AEM_CONTROL_SELECTOR_STRING   0x0014
#define AEM_CONTROL_ARRAY_INT8        0x0015
#define AEM_CONTROL_ARRAY_UINT8       0x0016
#define AEM_CONTROL_ARRAY_INT16       0x0017
#define AEM_CONTROL_ARRAY_UINT16      0x0018
#define AEM_CONTROL_ARRAY_INT32       0x0019
#define AEM_CONTROL_ARRAY_UINT32      0x001a
#define AEM_CONTROL_ARRAY_INT64       0x001b
#define AEM_CONTROL_ARRAY_UINT64      0x001c
#define AEM_CONTROL_ARRAY_FLOAT       0x001d
#define AEM_CONTROL_ARRAY_DOUBLE      0x001e
#define AEM_CONTROL_UTF8              0x001f
#define AEM_CONTROL_BODE_PLOT         0x0020
#define AEM_CONTROL_SMPTE_TIME        0x0021
#define AEM_CONTROL_SAMPLE_RATE       0x0022
#define AEM_CONTROL_GPTP_TIME         0x0023
/* 0x0024-0x3ffd RESERVED Reserved for future use. */
#define AEM_CONTROL_VENDOR            0x3ffe
/* 0x3fff EXPANSION Reserved for future expansion. */

/* AEM CLOCK_SOURCE Types */
/* IEEE 1722.1 draft D21 TABLE 7.17 */
#define AEM_CLOCK_INTERNAL             0x0000
#define AEM_CLOCK_EXTERNAL             0x0001
#define AEM_CLOCK_INPUT_STREAM         0x0002
#define AEM_CLOCK_MEDIA_CLOCK_STREAM   0x0003
/* 0x0004-0xfffe RESERVED Reserved for future use. */
/* 0xffff EXPANSION Reserved for future expansion. */

/* AEM Stream Format Definitions */
/* IEEE 1722.1 draft D21 TABLE 7.65 */
#define IEC_61883_IIDC_SUBTYPE      0x00
#define MMA_SUBTYPE                 0x01
#define AVTP_AUDIO_SUBTYPE          0x02
#define AVTP_VIDEO_SUBTYPE          0x03
#define AVTP_CONTROL_SUBTYPE        0x04
/* 0x05-0x7e RESERVED Reserved for future protocols. */
#define EXPERIMENTAL_SUBTYPE        0x7f


/* AEM Descriptor Offsets */
/* IEEE 1722.1 draft D21 sections 7.2.1 - 7.2.33 */
#define AEM_OFFSET_DESCRIPTOR_TYPE                          0
#define AEM_OFFSET_DESCRIPTOR_ID                            2

#define ENTITY_OFFSET_ENTITY_GUID                           4
#define ENTITY_OFFSET_ENTITY_MODEL_ID                       12
#define ENTITY_OFFSET_ENTITY_CAPABILITIES                   20
#define ENTITY_OFFSET_TALKER_STREAM_SOURCES                 24
#define ENTITY_OFFSET_TALKER_CAPABILITIES                   26
#define ENTITY_OFFSET_LISTENER_STREAM_SINKS                 28
#define ENTITY_OFFSET_LISTENER_CAPABILITIES                 30
#define ENTITY_OFFSET_CONTROLLER_CAPABILITIES               32
#define ENTITY_OFFSET_AVAILABLE_INDEX                       36
#define ENTITY_OFFSET_ASSOCIATION_ID                        40
#define ENTITY_OFFSET_ENTITY_NAME                           48
#define ENTITY_OFFSET_VENDOR_NAME_STRING                    112
#define ENTITY_OFFSET_MODEL_NAME_STRING                     114
#define ENTITY_OFFSET_FIRMWARE_VERSION                      116
#define ENTITY_OFFSET_GROUP_NAME                            180
#define ENTITY_OFFSET_SERIAL_NUMBER                         244
#define ENTITY_OFFSET_CONFIGURATIONS_COUNT                  308
#define ENTITY_OFFSET_CURRENT_CONFIGURATION                 310

#define CONFIGURATION_OFFSET_OBJECT_NAME                    4
#define CONFIGURATION_OFFSET_LOCALIZED_DESCRIPTION          68
#define CONFIGURATION_OFFSET_DESCRIPTOR_COUNTS_COUNT        70
#define CONFIGURATION_OFFSET_DESCRIPTOR_COUNTS_OFFSET       72
#define CONFIGURATION_OFFSET_DESCRIPTOR_COUNTS              74

#define AUDIO_UNIT_OFFSET_OBJECT_NAME                       4
#define AUDIO_UNIT_OFFSET_LOCALIZED_DESCRIPTION             68
#define AUDIO_UNIT_OFFSET_CLOCK_DOMAIN_INDEX                70
#define AUDIO_UNIT_OFFSET_NUMBER_OF_STREAM_INPUT_PORTS      72
#define AUDIO_UNIT_OFFSET_BASE_STREAM_INPUT_PORT            74
#define AUDIO_UNIT_OFFSET_NUMBER_OF_STREAM_OUTPUT_PORTS     76
#define AUDIO_UNIT_OFFSET_BASE_STREAM_OUTPUT_PORT           78
#define AUDIO_UNIT_OFFSET_NUMBER_OF_EXTERNAL_INPUT_PORTS    80
#define AUDIO_UNIT_OFFSET_BASE_EXTERNAL_INPUT_PORT          82
#define AUDIO_UNIT_OFFSET_NUMBER_OF_EXTERNAL_OUTPUT_PORTS   84
#define AUDIO_UNIT_OFFSET_BASE_EXTERNAL_OUTPUT_PORT         86
#define AUDIO_UNIT_OFFSET_NUMBER_OF_INTERNAL_INPUT_PORTS    88
#define AUDIO_UNIT_OFFSET_BASE_INTERNAL_INPUT_PORT          90
#define AUDIO_UNIT_OFFSET_NUMBER_OF_INTERNAL_OUTPUT_PORTS   92
#define AUDIO_UNIT_OFFSET_BASE_INTERNAL_OUTPUT_PORT         94
#define AUDIO_UNIT_OFFSET_NUMBER_OF_CONTROLS                96
#define AUDIO_UNIT_OFFSET_BASE_CONTROL                      98
#define AUDIO_UNIT_OFFSET_NUMBER_SIGNAL_SELECTORS           100
#define AUDIO_UNIT_OFFSET_BASE_SIGNAL_SELECTOR              102
#define AUDIO_UNIT_OFFSET_NUMBER_MIXERS                     104
#define AUDIO_UNIT_OFFSET_BASE_MIXER                        106
#define AUDIO_UNIT_OFFSET_NUMBER_MATRICES                   108
#define AUDIO_UNIT_OFFSET_BASE_MATRIX                       110
#define AUDIO_UNIT_OFFSET_NUMBER_SPLITTERS                  112
#define AUDIO_UNIT_OFFSET_BASE_SPLITTER                     114
#define AUDIO_UNIT_OFFSET_NUMBER_COMBINERS                  116
#define AUDIO_UNIT_OFFSET_BASE_COMBINER                     118
#define AUDIO_UNIT_OFFSET_NUMBER_DEMULTIPLEXERS             120
#define AUDIO_UNIT_OFFSET_BASE_DEMULTIPLEXER                122
#define AUDIO_UNIT_OFFSET_NUMBER_MULTIPLEXERS               124
#define AUDIO_UNIT_OFFSET_BASE_MULTIPLEXER                  126
#define AUDIO_UNIT_OFFSET_NUMBER_TRANSCODERS                128
#define AUDIO_UNIT_OFFSET_BASE_TRANSCODER                   130
#define AUDIO_UNIT_OFFSET_NUMBER_CONTROL_BLOCKS             132
#define AUDIO_UNIT_OFFSET_BASE_CONTROL_BLOCK                134
#define AUDIO_UNIT_OFFSET_CURRENT_SAMPLE_RATE               136
#define AUDIO_UNIT_OFFSET_SAMPLE_RATES_OFFSET               140
#define AUDIO_UNIT_OFFSET_SAMPLE_RATES_COUNT                142

#define VIDEO_UNIT_OFFSET_OBJECT_NAME                       4
#define VIDEO_UNIT_OFFSET_LOCALIZED_DESCRIPTION             68
#define VIDEO_UNIT_OFFSET_CLOCK_DOMAIN_INDEX                70
#define VIDEO_UNIT_OFFSET_NUMBER_OF_STREAM_INPUT_PORTS      72
#define VIDEO_UNIT_OFFSET_BASE_STREAM_INPUT_PORT            74
#define VIDEO_UNIT_OFFSET_NUMBER_OF_STREAM_OUTPUT_PORTS     76
#define VIDEO_UNIT_OFFSET_BASE_STREAM_OUTPUT_PORT           78
#define VIDEO_UNIT_OFFSET_NUMBER_OF_EXTERNAL_INPUT_PORTS    80
#define VIDEO_UNIT_OFFSET_BASE_EXTERNAL_INPUT_PORT          82
#define VIDEO_UNIT_OFFSET_NUMBER_OF_EXTERNAL_OUTPUT_PORTS   84
#define VIDEO_UNIT_OFFSET_BASE_EXTERNAL_OUTPUT_PORT         86
#define VIDEO_UNIT_OFFSET_NUMBER_OF_INTERNAL_INPUT_PORTS    88
#define VIDEO_UNIT_OFFSET_BASE_INTERNAL_INPUT_PORT          90
#define VIDEO_UNIT_OFFSET_NUMBER_OF_INTERNAL_OUTPUT_PORTS   92
#define VIDEO_UNIT_OFFSET_BASE_INTERNAL_OUTPUT_PORT         94
#define VIDEO_UNIT_OFFSET_NUMBER_OF_CONTROLS                96
#define VIDEO_UNIT_OFFSET_BASE_CONTROL                      98
#define VIDEO_UNIT_OFFSET_NUMBER_SIGNAL_SELECTORS           100
#define VIDEO_UNIT_OFFSET_BASE_SIGNAL_SELECTOR              102
#define VIDEO_UNIT_OFFSET_NUMBER_MIXERS                     104
#define VIDEO_UNIT_OFFSET_BASE_MIXER                        106
#define VIDEO_UNIT_OFFSET_NUMBER_MATRICES                   108
#define VIDEO_UNIT_OFFSET_BASE_MATRIX                       110
#define VIDEO_UNIT_OFFSET_NUMBER_SPLITTERS                  112
#define VIDEO_UNIT_OFFSET_BASE_SPLITTER                     114
#define VIDEO_UNIT_OFFSET_NUMBER_COMBINERS                  116
#define VIDEO_UNIT_OFFSET_BASE_COMBINER                     118
#define VIDEO_UNIT_OFFSET_NUMBER_DEMULTIPLEXERS             120
#define VIDEO_UNIT_OFFSET_BASE_DEMULTIPLEXER                122
#define VIDEO_UNIT_OFFSET_NUMBER_MULTIPLEXERS               124
#define VIDEO_UNIT_OFFSET_BASE_MULTIPLEXER                  126
#define VIDEO_UNIT_OFFSET_NUMBER_TRANSCODERS                128
#define VIDEO_UNIT_OFFSET_BASE_TRANSCODER                   130
#define VIDEO_UNIT_OFFSET_NUMBER_CONTROL_BLOCKS             132
#define VIDEO_UNIT_OFFSET_BASE_CONTROL_BLOCK                134

#define SENSOR_UNIT_OFFSET_OBJECT_NAME                      4
#define SENSOR_UNIT_OFFSET_LOCALIZED_DESCRIPTION            68
#define SENSOR_UNIT_OFFSET_CLOCK_DOMAIN_INDEX               70
#define SENSOR_UNIT_OFFSET_NUMBER_OF_STREAM_INPUT_PORTS     72
#define SENSOR_UNIT_OFFSET_BASE_STREAM_INPUT_PORT           74
#define SENSOR_UNIT_OFFSET_NUMBER_OF_STREAM_OUTPUT_PORTS    76
#define SENSOR_UNIT_OFFSET_BASE_STREAM_OUTPUT_PORT          78
#define SENSOR_UNIT_OFFSET_NUMBER_OF_EXTERNAL_INPUT_PORTS   80
#define SENSOR_UNIT_OFFSET_BASE_EXTERNAL_INPUT_PORT         82
#define SENSOR_UNIT_OFFSET_NUMBER_OF_EXTERNAL_OUTPUT_PORTS  84
#define SENSOR_UNIT_OFFSET_BASE_EXTERNAL_OUTPUT_PORT        86
#define SENSOR_UNIT_OFFSET_NUMBER_OF_INTERNAL_INPUT_PORTS   88
#define SENSOR_UNIT_OFFSET_BASE_INTERNAL_INPUT_PORT         90
#define SENSOR_UNIT_OFFSET_NUMBER_OF_INTERNAL_OUTPUT_PORTS  92
#define SENSOR_UNIT_OFFSET_BASE_INTERNAL_OUTPUT_PORT        94
#define SENSOR_UNIT_OFFSET_NUMBER_OF_CONTROLS               96
#define SENSOR_UNIT_OFFSET_BASE_CONTROL                     98
#define SENSOR_UNIT_OFFSET_NUMBER_SIGNAL_SELECTORS          100
#define SENSOR_UNIT_OFFSET_BASE_SIGNAL_SELECTOR             102
#define SENSOR_UNIT_OFFSET_NUMBER_MIXERS                    104
#define SENSOR_UNIT_OFFSET_BASE_MIXER                       106
#define SENSOR_UNIT_OFFSET_NUMBER_MATRICES                  108
#define SENSOR_UNIT_OFFSET_BASE_MATRIX                      110
#define SENSOR_UNIT_OFFSET_NUMBER_SPLITTERS                 112
#define SENSOR_UNIT_OFFSET_BASE_SPLITTER                    114
#define SENSOR_UNIT_OFFSET_NUMBER_COMBINERS                 116
#define SENSOR_UNIT_OFFSET_BASE_COMBINER                    118
#define SENSOR_UNIT_OFFSET_NUMBER_DEMULTIPLEXERS            120
#define SENSOR_UNIT_OFFSET_BASE_DEMULTIPLEXER               122
#define SENSOR_UNIT_OFFSET_NUMBER_MULTIPLEXERS              124
#define SENSOR_UNIT_OFFSET_BASE_MULTIPLEXER                 126
#define SENSOR_UNIT_OFFSET_NUMBER_TRANSCODERS               128
#define SENSOR_UNIT_OFFSET_BASE_TRANSCODER                  130
#define SENSOR_UNIT_OFFSET_NUMBER_CONTROL_BLOCKS            132
#define SENSOR_UNIT_OFFSET_BASE_CONTROL_BLOCK               134

#define STREAM_OFFSET_OBJECT_NAME                           4
#define STREAM_OFFSET_LOCALIZED_DESCRIPTION                 68
#define STREAM_OFFSET_CLOCK_DOMAIN_INDEX                    70
#define STREAM_OFFSET_STREAM_FLAGS                          72
#define STREAM_OFFSET_CURRENT_FORMAT                        74
#define STREAM_OFFSET_FORMATS_OFFSET                        82
#define STREAM_OFFSET_NUMBER_OF_FORMATS                     84
#define STREAM_OFFSET_BACKUP_TALKER_GUID_0                  86
#define STREAM_OFFSET_BACKUP_TALKER_UNIQUE_0                94
#define STREAM_OFFSET_BACKUP_TALKER_GUID_1                  96
#define STREAM_OFFSET_BACKUP_TALKER_UNIQUE_1                104
#define STREAM_OFFSET_BACKUP_TALKER_GUID_2                  106
#define STREAM_OFFSET_BACKUP_TALKER_UNIQUE_2                114
#define STREAM_OFFSET_BACKEDUP_TALKER_GUID                  116
#define STREAM_OFFSET_BACKEDUP_TALKER_UNIQUE_ID             124
#define STREAM_OFFSET_AVB_INTERFACE_ID                      126
#define STREAM_OFFSET_BUFFER_LENGTH                         128

#define JACK_OFFSET_OBJECT_NAME                             4
#define JACK_OFFSET_LOCALIZED_DESCRIPTION                   68
#define JACK_OFFSET_JACK_FLAGS                              70
#define JACK_OFFSET_JACK_TYPE                               72
#define JACK_OFFSET_JACK_NUMBER_OF_CONTROLS                 74
#define JACK_OFFSET_JACK_BASE_CONTROL                       76

#define AVB_INTERFACE_OFFSET_OBJECT_NAME                    4
#define AVB_INTERFACE_OFFSET_LOCALIZED_DESCRIPTION          68
#define AVB_INTERFACE_OFFSET_MAC_ADDRESS                    70
#define AVB_INTERFACE_OFFSET_INTERFACE_FLAGS                76
#define AVB_INTERFACE_OFFSET_CLOCK_IDENTITY                 78
#define AVB_INTERFACE_OFFSET_PRIORITY1                      86
#define AVB_INTERFACE_OFFSET_CLOCK_CLASS                    87
#define AVB_INTERFACE_OFFSET_OFFSET_SCALED_LOG_VARIANCE     88
#define AVB_INTERFACE_OFFSET_CLOCK_ACCURACY                 90
#define AVB_INTERFACE_OFFSET_PRIORITY2                      91
#define AVB_INTERFACE_OFFSET_DOMAIN_NUMBER                  92
#define AVB_INTERFACE_OFFSET_LOG_SYNC_INTERVAL              93

#define CLOCK_SOURCE_OFFSET_OBJECT_NAME                     4
#define CLOCK_SOURCE_OFFSET_LOCALIZED_DESCRIPTION           68
#define CLOCK_SOURCE_OFFSET_CLOCK_SOURCE_FLAGS              70
#define CLOCK_SOURCE_OFFSET_CLOCK_SOURCE_TYPE               72
#define CLOCK_SOURCE_OFFSET_CLOCK_SOURCE_IDENTIFIER         74
#define CLOCK_SOURCE_OFFSET_CLOCK_SOURCE_LOCATION_TYPE      82
#define CLOCK_SOURCE_OFFSET_CLOCK_SOURCE_LOCATION_INDEX     84

#define MEMORY_OBJECT_OFFSET_OBJECT_NAME                    4
#define MEMORY_OBJECT_OFFSET_LOCALIZED_DESCRIPTION          68
#define MEMORY_OBJECT_OFFSET_MEMORY_OBJECT_TYPE             70
#define MEMORY_OBJECT_OFFSET_TARGET_DESCRIPTOR_TYPE         72
#define MEMORY_OBJECT_OFFSET_TARGET_DESCRIPTOR_INDEX        74
#define MEMORY_OBJECT_OFFSET_START_ADDRESS                  76
#define MEMORY_OBJECT_OFFSET_LENGTH                         84

#define LOCALE_OFFSET_LOCALE_IDENTIFIER                     4
#define LOCALE_OFFSET_NUMBER_OF_STRINGS                     68
#define LOCALE_OFFSET_BASE_STRINGS                          70

#define STRINGS_OFFSET_STRING0                              4

#define STREAM_PORT_OFFSET_CLOCK_DOMAIN_INDEX               4
#define STREAM_PORT_OFFSET_PORT_FLAGS                       6
#define STREAM_PORT_OFFSET_NUMBER_OF_CONTROLS               8
#define STREAM_PORT_OFFSET_BASE_CONTROL                     10
#define STREAM_PORT_OFFSET_NUMBER_OF_CLUSTERS               12
#define STREAM_PORT_OFFSET_BASE_CLUSTER                     14
#define STREAM_PORT_OFFSET_NUMBER_AUDIO_MAPS                16
#define STREAM_PORT_OFFSET_BASE_AUDIO_MAP                   18

#define EXTERNAL_PORT_OFFSET_CLOCK_DOMAIN_INDEX             4
#define EXTERNAL_PORT_OFFSET_PORT_FLAGS                     6
#define EXTERNAL_PORT_OFFSET_NUMBER_OF_CONTROLS             8
#define EXTERNAL_PORT_OFFSET_BASE_CONTROL                   10
#define EXTERNAL_PORT_OFFSET_SIGNAL_TYPE                    12
#define EXTERNAL_PORT_OFFSET_SIGNAL_INDEX                   14
#define EXTERNAL_PORT_OFFSET_SIGNAL_OUTPUT                  16
#define EXTERNAL_PORT_OFFSET_BLOCK_LATENCY                  18
#define EXTERNAL_PORT_OFFSET_JACK_INDEX                     22

#define INTERNAL_PORT_OFFSET_CLOCK_DOMAIN_INDEX             4
#define INTERNAL_PORT_OFFSET_PORT_FLAGS                     6
#define INTERNAL_PORT_OFFSET_NUMBER_OF_CONTROLS             8
#define INTERNAL_PORT_OFFSET_BASE_CONTROL                   10
#define INTERNAL_PORT_OFFSET_SIGNAL_TYPE                    12
#define INTERNAL_PORT_OFFSET_SIGNAL_INDEX                   14
#define INTERNAL_PORT_OFFSET_SIGNAL_OUTPUT                  16
#define INTERNAL_PORT_OFFSET_BLOCK_LATENCY                  18
#define INTERNAL_PORT_OFFSET_INTERNAL_INDEX                 22

#define AUDIO_CLUSTER_OFFSET_OBJECT_NAME                    4
#define AUDIO_CLUSTER_OFFSET_LOCALIZED_DESCRIPTION          68
#define AUDIO_CLUSTER_OFFSET_SIGNAL_TYPE                    70
#define AUDIO_CLUSTER_OFFSET_SIGNAL_ID                      72
#define AUDIO_CLUSTER_OFFSET_SIGNAL_OUTPUT                  74
#define AUDIO_CLUSTER_OFFSET_PATH_LATENCY                   76
#define AUDIO_CLUSTER_OFFSET_BLOCK_LATENCY                  80
#define AUDIO_CLUSTER_OFFSET_CHANNEL_COUNT                  84
#define AUDIO_CLUSTER_OFFSET_FORMAT                         86

#define VIDEO_CLUSTER_OFFSET_OBJECT_NAME                       4
#define VIDEO_CLUSTER_OFFSET_LOCALIZED_DESCRIPTION             68
#define VIDEO_CLUSTER_OFFSET_SIGNAL_TYPE                       70
#define VIDEO_CLUSTER_OFFSET_SIGNAL_ID                         72
#define VIDEO_CLUSTER_OFFSET_SIGNAL_OUTPUT                     74
#define VIDEO_CLUSTER_OFFSET_PATH_LATENCY                      76
#define VIDEO_CLUSTER_OFFSET_BLOCK_LATENCY                     80
#define VIDEO_CLUSTER_OFFSET_FORMAT                            84
#define VIDEO_CLUSTER_OFFSET_CURRENT_FORMAT_SPECIFIC           85
#define VIDEO_CLUSTER_OFFSET_SUPPORTED_FORMAT_SPECIFICS_OFFSET 89
#define VIDEO_CLUSTER_OFFSET_SUPPORTED_FORMAT_SPECIFICS_COUNT  91
#define VIDEO_CLUSTER_OFFSET_CURRENT_SAMPLING_RATE             93
#define VIDEO_CLUSTER_OFFSET_SUPPORTED_SAMPLING_RATES_OFFSET   97
#define VIDEO_CLUSTER_OFFSET_SUPPORTED_SAMPLING_RATES_COUNT    99
#define VIDEO_CLUSTER_OFFSET_CURRENT_ASPECT_RATIO              101
#define VIDEO_CLUSTER_OFFSET_SUPPORTED_ASPECT_RATIOS_OFFSET    103
#define VIDEO_CLUSTER_OFFSET_SUPPORTED_ASPECT_RATIOS_COUNT     105
#define VIDEO_CLUSTER_OFFSET_CURRENT_SIZE                      107
#define VIDEO_CLUSTER_OFFSET_SUPPORTED_SIZES_OFFSET            111
#define VIDEO_CLUSTER_OFFSET_SUPPORTED_SIZES_COUNT             113
#define VIDEO_CLUSTER_OFFSET_CURRENT_COLOR_SPACE               115
#define VIDEO_CLUSTER_OFFSET_SUPPORTED_COLOR_SPACES_OFFSET     117
#define VIDEO_CLUSTER_OFFSET_SUPPORTED_COLOR_SPACES_COUNT      119

#define SENSOR_CLUSTER_OFFSET_OBJECT_NAME                      4
#define SENSOR_CLUSTER_OFFSET_LOCALIZED_DESCRIPTION            68
#define SENSOR_CLUSTER_OFFSET_SIGNAL_TYPE                      70
#define SENSOR_CLUSTER_OFFSET_SIGNAL_ID                        72
#define SENSOR_CLUSTER_OFFSET_SIGNAL_OUTPUT                    74
#define SENSOR_CLUSTER_OFFSET_PATH_LATENCY                     76
#define SENSOR_CLUSTER_OFFSET_BLOCK_LATENCY                    80
#define SENSOR_CLUSTER_OFFSET_CURRENT_FORMAT                   84
#define SENSOR_CLUSTER_OFFSET_SUPPORTED_FORMATS_OFFSET         92
#define SENSOR_CLUSTER_OFFSET_SUPPORTED_FORMATS_COUNT          94
#define SENSOR_CLUSTER_OFFSET_CURRENT_SAMPLING_RATE            96
#define SENSOR_CLUSTER_OFFSET_SUPPORTED_SAMPLING_RATES_OFFSET  100
#define SENSOR_CLUSTER_OFFSET_SUPPORTED_SAMPLING_RATES_COUNT   102

#define AUDIO_MAP_OFFSET_MAPPINGS_OFFSET                     4
#define AUDIO_MAP_OFFSET_NUMBER_OF_MAPPINGS                  6

#define VIDEO_MAP_OFFSET_MAPPINGS_OFFSET                     4
#define VIDEO_MAP_OFFSET_NUMBER_OF_MAPPINGS                  6

#define SENSOR_MAP_OFFSET_MAPPINGS_OFFSET                    4
#define SENSOR_MAP_OFFSET_NUMBER_OF_MAPPINGS                 6

#define CONTROL_OFFSET_OBJECT_NAME                           4
#define CONTROL_OFFSET_LOCALIZED_DESCRIPTION                 68
#define CONTROL_OFFSET_BLOCK_LATENCY                         70
#define CONTROL_OFFSET_CONTROL_LATENCY                       74
#define CONTROL_OFFSET_CONTROL_DOMAIN                        78
#define CONTROL_OFFSET_CONTROL_VALUE_TYPE                    80
#define CONTROL_OFFSET_CONTROL_TYPE                          82
#define CONTROL_OFFSET_RESET_TIME                            90
#define CONTROL_OFFSET_VALUES_OFFSET                         94
#define CONTROL_OFFSET_NUMBER_OF_VALUES                      96
#define CONTROL_OFFSET_SIGNAL_TYPE                           98
#define CONTROL_OFFSET_SIGNAL_INDEX                          100
#define CONTROL_OFFSET_SIGNAL_OUTPUT                         102

#define SIGNAL_SELECTOR_OFFSET_OBJECT_NAME                   4
#define SIGNAL_SELECTOR_OFFSET_LOCALIZED_DESCRIPTION         68
#define SIGNAL_SELECTOR_OFFSET_BLOCK_LATENCY                 70
#define SIGNAL_SELECTOR_OFFSET_CONTROL_LATENCY               74
#define SIGNAL_SELECTOR_OFFSET_CONTROL_DOMAIN                78
#define SIGNAL_SELECTOR_OFFSET_SOURCES_OFFSET                80
#define SIGNAL_SELECTOR_OFFSET_NUMBER_OF_SOURCES             82
#define SIGNAL_SELECTOR_OFFSET_CURRENT_SIGNAL_TYPE           84
#define SIGNAL_SELECTOR_OFFSET_CURRENT_SIGNAL_INDEX          86
#define SIGNAL_SELECTOR_OFFSET_CURRENT_SIGNAL_OUTPUT         88
#define SIGNAL_SELECTOR_OFFSET_DEFAULT_SOURCE_TYPE           90
#define SIGNAL_SELECTOR_OFFSET_DEFAULT_SOURCE_INDEX          92
#define SIGNAL_SELECTOR_OFFSET_DEFAULT_SIGNAL_OUTPUT         94

#define MIXER_OFFSET_OBJECT_NAME                             4
#define MIXER_OFFSET_LOCALIZED_DESCRIPTION                   68
#define MIXER_OFFSET_BLOCK_LATENCY                           70
#define MIXER_OFFSET_CONTROL_LATENCY                         74
#define MIXER_OFFSET_CONTROL_DOMAIN                          78
#define MIXER_OFFSET_CONTROL_VALUE_TYPE                      80
#define MIXER_OFFSET_SOURCES_OFFSET                          82
#define MIXER_OFFSET_NUMBER_OF_SOURCES                       84
#define MIXER_OFFSET_VALUE_OFFSET                            86

#define MATRIX_OFFSET_OBJECT_NAME                            4
#define MATRIX_OFFSET_LOCALIZED_DESCRIPTION                  68
#define MATRIX_OFFSET_BLOCK_LATENCY                          70
#define MATRIX_OFFSET_CONTROL_LATENCY                        74
#define MATRIX_OFFSET_CONTROL_DOMAIN                         78
#define MATRIX_OFFSET_CONTROL_VALUE_TYPE                     80
#define MATRIX_OFFSET_CONTROL_TYPE                           82
#define MATRIX_OFFSET_WIDTH                                  90
#define MATRIX_OFFSET_HEIGHT                                 92
#define MATRIX_OFFSET_VALUES_OFFSET                          94
#define MATRIX_OFFSET_NUMBER_OF_VALUES                       96
#define MATRIX_OFFSET_NUMBER_OF_SOURCES                      98
#define MATRIX_OFFSET_BASE_SOURCE                            100

#define MATRIX_SIGNAL_OFFSET_SIGNALS_COUNT                   4
#define MATRIX_SIGNAL_OFFSET_SIGNALS_OFFSET                  6

#define SIGNAL_SPLITTER_OFFSET_OBJECT_NAME                   4
#define SIGNAL_SPLITTER_OFFSET_LOCALIZED_DESCRIPTION         68
#define SIGNAL_SPLITTER_OFFSET_BLOCK_LATENCY                 70
#define SIGNAL_SPLITTER_OFFSET_CONTROL_LATENCY               74
#define SIGNAL_SPLITTER_OFFSET_CONTROL_DOMAIN                78
#define SIGNAL_SPLITTER_OFFSET_SIGNAL_TYPE                   80
#define SIGNAL_SPLITTER_OFFSET_SIGNAL_INDEX                  82
#define SIGNAL_SPLITTER_OFFSET_SIGNAL_OUTPUT                 84
#define SIGNAL_SPLITTER_OFFSET_NUMBER_OF_OUTPUTS             86
#define SIGNAL_SPLITTER_OFFSET_SPLITTER_MAP_COUNT            88
#define SIGNAL_SPLITTER_OFFSET_SPLITTER_MAP_OFFSET           90

#define SIGNAL_COMBINER_OFFSET_OBJECT_NAME                   4
#define SIGNAL_COMBINER_OFFSET_LOCALIZED_DESCRIPTION         68
#define SIGNAL_COMBINER_OFFSET_BLOCK_LATENCY                 70
#define SIGNAL_COMBINER_OFFSET_CONTROL_LATENCY               74
#define SIGNAL_COMBINER_OFFSET_CONTROL_DOMAIN                78
#define SIGNAL_COMBINER_OFFSET_COMBINER_MAP_COUNT            80
#define SIGNAL_COMBINER_OFFSET_COMBINER_MAP_OFFSET           82
#define SIGNAL_COMBINER_OFFSET_SOURCES_OFFSET                84
#define SIGNAL_COMBINER_OFFSET_NUMBER_OF_SOURCES             86

#define SIGNAL_DEMULTIPLEXER_OFFSET_OBJECT_NAME              4
#define SIGNAL_DEMULTIPLEXER_OFFSET_LOCALIZED_DESCRIPTION    68
#define SIGNAL_DEMULTIPLEXER_OFFSET_BLOCK_LATENCY            70
#define SIGNAL_DEMULTIPLEXER_OFFSET_CONTROL_LATENCY          74
#define SIGNAL_DEMULTIPLEXER_OFFSET_CONTROL_DOMAIN           78
#define SIGNAL_DEMULTIPLEXER_OFFSET_SIGNAL_TYPE              80
#define SIGNAL_DEMULTIPLEXER_OFFSET_SIGNAL_INDEX             82
#define SIGNAL_DEMULTIPLEXER_OFFSET_SIGNAL_OUTPUT            84
#define SIGNAL_DEMULTIPLEXER_OFFSET_NUMBER_OF_OUTPUTS        86
#define SIGNAL_DEMULTIPLEXER_OFFSET_DEMULTIPLEXER_MAP_COUNT  88
#define SIGNAL_DEMULTIPLEXER_OFFSET_DEMULTIPLEXER_MAP_OFFSET 90

#define SIGNAL_MULTIPLEXER_OFFSET_OBJECT_NAME                4
#define SIGNAL_MULTIPLEXER_OFFSET_LOCALIZED_DESCRIPTION      68
#define SIGNAL_MULTIPLEXER_OFFSET_BLOCK_LATENCY              70
#define SIGNAL_MULTIPLEXER_OFFSET_CONTROL_LATENCY            74
#define SIGNAL_MULTIPLEXER_OFFSET_CONTROL_DOMAIN             78
#define SIGNAL_MULTIPLEXER_OFFSET_MULTIPLEXER_MAP_COUNT      80
#define SIGNAL_MULTIPLEXER_OFFSET_MULTIPLEXER_MAP_OFFSET     82
#define SIGNAL_MULTIPLEXER_OFFSET_SOURCES_OFFSET             84
#define SIGNAL_MULTIPLEXER_OFFSET_NUMBER_OF_SOURCES          86

#define SIGNAL_TRANSCODER_OFFSET_OBJECT_NAME                 4
#define SIGNAL_TRANSCODER_OFFSET_LOCALIZED_DESCRIPTION       68
#define SIGNAL_TRANSCODER_OFFSET_BLOCK_LATENCY               70
#define SIGNAL_TRANSCODER_OFFSET_CONTROL_LATENCY             74
#define SIGNAL_TRANSCODER_OFFSET_CONTROL_DOMAIN              78
#define SIGNAL_TRANSCODER_OFFSET_CONTROL_VALUE_TYPE          80
#define SIGNAL_TRANSCODER_OFFSET_VALUES_OFFSET               82
#define SIGNAL_TRANSCODER_OFFSET_NUMBER_OF_VALUES            84
#define SIGNAL_TRANSCODER_OFFSET_SIGNAL_TYPE                 86
#define SIGNAL_TRANSCODER_OFFSET_SIGNAL_INDEX                88
#define SIGNAL_TRANSCODER_OFFSET_SIGNAL_OUTPUT               90

#define CLOCK_DOMAIN_OFFSET_OBJECT_NAME                      4
#define CLOCK_DOMAIN_OFFSET_LOCALIZED_DESCRIPTION            68
#define CLOCK_DOMAIN_OFFSET_CLOCK_SOURCE_INDEX               70
#define CLOCK_DOMAIN_OFFSET_CLOCK_SOURCES_OFFSET             72
#define CLOCK_DOMAIN_OFFSET_CLOCK_SOURCES_COUNT              74

#define CONTROL_BLOCK_OFFSET_OBJECT_NAME                     4
#define CONTROL_BLOCK_OFFSET_LOCALIZED_DESCRIPTION           68
#define CONTROL_BLOCK_OFFSET_NUMBER_OF_CONTROLS              70
#define CONTROL_BLOCK_OFFSET_BASE_CONTROL                    72
#define CONTROL_BLOCK_OFFSET_FINAL_CONTROL_INDEX             74


/* AEM Stream Format Offsets */
/* IEEE 1722.1 draft D21 section 7.3.2 */
#define AEM_OFFSET_SF_SUBTYPE                      0

#define AEM_OFFSET_SF_VERSION                      0

/* IIDC */
#define AEM_OFFSET_SF                              1
#define AEM_OFFSET_IIDC_FORMAT                     5
#define AEM_OFFSET_IIDC_MODE                       6
#define AEM_OFFSET_IIDC_RATE                       7

/* 61883 */
#define AEM_OFFSET_FMT                             1

/* 61883-6 */
#define AEM_OFFSET_FDF_EVT                         2
#define AEM_OFFSET_FDF_SFC                         2
#define AEM_OFFSET_DBS                             3
#define AEM_OFFSET_B                               4
#define AEM_OFFSET_NB                              4

/* 61883-6 AM824 */
#define AEM_OFFSET_LABEL_IEC_60958_CNT             5
#define AEM_OFFSET_LABEL_MBLA_CNT                  6
#define AEM_OFFSET_LABEL_MIDI_CNT                  7
#define AEM_OFFSET_LABEL_SMPTE_CNT                 7

/* 61883-8 */
#define AEM_OFFSET_VIDEO_MODE                      5
#define AEM_OFFSET_COMPRESS_MODE                   6
#define AEM_OFFSET_COLOR_SPACE                     7

/* AECP Vendor Unique Command Specific Offsets */
#define AECP_VUC_OFFSET_PROTOCOL_ID                22

/* AECP Video Cluster Format Offsets */
/* IEEE 1722.1 draft D21 section 7.3.7 */
#if 0 /* TODO: These are all different in D21 */
#define AEM_OFFSET_MFD_TYPE                        3
#define AEM_OFFSET_DIV                             4
#define AEM_OFFSET_INTERLACE                       4
#define AEM_OFFSET_CHANNELS                        4
#define AEM_OFFSET_COLOR_FORMAT                    4
#define AEM_OFFSET_BPP                             5
#define AEM_OFFSET_ASPECT_X                        6
#define AEM_OFFSET_ASPECT_Y                        7
#define AEM_OFFSET_FRAME_RATE                      8
#define AEM_OFFSET_COMP1                           9
#define AEM_OFFSET_COMP2                           10
#define AEM_OFFSET_COMP3                           10
#define AEM_OFFSET_COMP4                           11
#define AEM_OFFSET_SVMF_WIDTH                      12
#define AEM_OFFSET_SVMF_HEIGHT                     14

#define AEM_OFFSET_CS_EUI64                        8
#endif

/* Bitmasks */
#define AECP_STATUS_CODE_MASK                   0xf8
#define AECP_TOKEN_LENGTH_MASK                  0x07ff
#define AECP_KEY_PART_MASK                      0x78
#define AECP_CONTINUED_MASK                     0x80
#define AECP_CD_LENGTH_MASK                     0x07ff
#define AECP_COMMAND_TYPE_MASK                  0x7fff
#define AECP_KEYCHAIN_ID_MASK                   0xe0
#define AECP_KEYTYPE_MASK                       0x1c
#define AECP_KEY_COUNT_MASK                     0x0fff
#define AECP_KEY_LENGTH_MASK                    0x07ff
#define AECP_KEY_NUMBER_MASK                    0x04ff
#define AECP_MATRIX_DIRECTION_MASK              0x70
#define AECP_MATRIX_REP_MASK                    0x80
#define AECP_MATRIX_VALUE_COUNT_MASK            0x0fff
#define AECP_MSG_TYPE_MASK                      0x0f
#define AECP_PERSISTENT_FLAG_MASK               0x00000001
#define AECP_RELEASE_FLAG_MASK                  0x80000000
#define AECP_SIGNATURE_ID_MASK                  0x0fff
#define AECP_SIGNATURE_INFO_MASK                0x00f0
#define AECP_SIGNATURE_LENGTH_MASK              0x3ff
#define AECP_UNLOCK_FLAG_MASK                   0x00000001
#define AECP_U_FLAG_MASK                        0x80
#define AECP_MSRP_MAPPINGS_COUNT_MASK           0x00
#define AECP_AS_CAPABLE_FLAG_MASK               0x01
#define AECP_GPTP_ENABLED_FLAG_MASK             0x02
#define AECP_SRP_ENABLED_FLAG_MASK              0x04


/* AECP Sampling Rate masks */
#define AECP_SAMPLING_RATE_PULL_MASK            0xE0000000
#define AECP_SAMPLING_RATE_BASE_FREQ_MASK       0x1FFFFFFF

/* Stream Flags (7.130) */
#define AECP_STREAM_VLAN_ID_VALID_FLAG_MASK     0x02000000
#define AECP_CONNECTED_FLAG_MASK                0x04000000
#define AECP_MSRP_FAILURE_VALID_FLAG_MASK       0x08000000
#define AECP_DEST_MAC_VALID_FLAG_MASK           0x10000000
#define AECP_MSRP_ACC_LAT_VALID_FLAG_MASK       0x20000000
#define AECP_STREAM_ID_VALID_FLAG_MASK          0x40000000
#define AECP_STREAM_FORMAT_VALID_FLAG_MASK      0x80000000

/* key permission flag masks */
#define AECP_PRIVATE_KEY_READ_FLAG_MASK         0x80000000
#define AECP_PRIVATE_KEY_WRITE_FLAG_MASK        0x40000000
#define AECP_PUBLIC_KEY_WRITE_FLAG_MASK         0x20000000
#define AECP_CONNECTION_FLAG_MASK               0x10000000
#define AECP_CONTROL_ADMIN_FLAG_MASK            0x08000000
#define AECP_MEM_OBJ_ADMIN_FLAG_MASK            0x04000000
#define AECP_MEM_OBJ_SETTINGS_FLAG_MASK         0x02000000
#define AECP_CONTROL_USER_L1_FLAG_MASK          0x00000008
#define AECP_CONTROL_USER_L2_FLAG_MASK          0x00000004
#define AECP_CONTROL_USER_L3_FLAG_MASK          0x00000002
#define AECP_CONTROL_USER_L4_FLAG_MASK          0x00000001

/* 7.133-139 counters_valid flag masks */
#define AECP_COUNTERS_VALID_LINK_UP                0x00000001
#define AECP_COUNTERS_VALID_LINK_DOWN              0x00000002
#define AECP_COUNTERS_VALID_PACKETS_TX             0x00000004
#define AECP_COUNTERS_VALID_PACKETS_RX             0x00000008
#define AECP_COUNTERS_VALID_RX_CRC_ERROR           0x00000010
#define AECP_COUNTERS_VALID_GPTP_GM_CHANGED        0x00000020

#define AECP_COUNTERS_VALID_CLOCK_DOMAIN_LOCKED    0x00000001
#define AECP_COUNTERS_VALID_CLOCK_DOMAIN_UNLOCKED  0x00000002

#define AECP_COUNTERS_VALID_MEDIA_LOCKED           0x00000001
#define AECP_COUNTERS_VALID_MEDIA_UNLOCKED         0x00000002
#define AECP_COUNTERS_VALID_STREAM_RESET           0x00000004
#define AECP_COUNTERS_VALID_SEQ_NUM_MISMATCH       0x00000008
#define AECP_COUNTERS_VALID_MEDIA_RESET            0x00000010
#define AECP_COUNTERS_VALID_TIMESTAMP_UNCERTAIN    0x00000020
#define AECP_COUNTERS_VALID_TIMESTAMP_VALID        0x00000040
#define AECP_COUNTERS_VALID_TIMESTAMP_NOT_VALID    0x00000080
#define AECP_COUNTERS_VALID_UNSUPPORTED_FORMAT     0x00000100
#define AECP_COUNTERS_VALID_LATE_TIMESTAMP         0x00000200
#define AECP_COUNTERS_VALID_EARLY_TIMESTAMP        0x00000400
#define AECP_COUNTERS_VALID_STREAM_PACKETS_RX      0x00000800
#define AECP_COUNTERS_VALID_STREAM_PACKETS_TX      0x00001000

#define AECP_COUNTERS_VALID_ENTITY_SPECIFIC_8      0x01000000
#define AECP_COUNTERS_VALID_ENTITY_SPECIFIC_7      0x02000000
#define AECP_COUNTERS_VALID_ENTITY_SPECIFIC_6      0x04000000
#define AECP_COUNTERS_VALID_ENTITY_SPECIFIC_5      0x08000000
#define AECP_COUNTERS_VALID_ENTITY_SPECIFIC_4      0x10000000
#define AECP_COUNTERS_VALID_ENTITY_SPECIFIC_3      0x20000000
#define AECP_COUNTERS_VALID_ENTITY_SPECIFIC_2      0x40000000
#define AECP_COUNTERS_VALID_ENTITY_SPECIFIC_1      0x80000000

#define AEM_CLOCK_SYNC_SOURCE_FLAG_MASK         0x00000001
#define AEM_ASYNC_SAMPLE_RATE_CONV_FLAG_MASK    0x00000002
#define AEM_SYNC_SAMPLE_RATE_CONV_FLAG_MASK     0x00000004

#define AEM_BASE_FREQUENCY_MASK                 0x1fffffff
#define AEM_CAPTIVE_FLAG_MASK                   0x00000002
#define AEM_CLASS_A_FLAG_MASK                   0x00000002
#define AEM_CLASS_B_FLAG_MASK                   0x00000004

#define AEM_MASK_B                              0x80
#define AEM_MASK_BPP                            0x3F
#define AEM_MASK_CHANNELS                       0x3C
#define AEM_MASK_COLOR_FORMAT                   0x03C0
#define AEM_MASK_COMP1                          0x0F80
#define AEM_MASK_COMP2                          0x7C
#define AEM_MASK_COMP3                          0x03E0
#define AEM_MASK_COMP4                          0x1F
#define AEM_MASK_DIV                            0x80
#define AEM_MASK_FDF_EVT                        0xF8
#define AEM_MASK_FDF_SFC                        0x07
#define AEM_MASK_FMT                            0x3F
#define AEM_MASK_INTERLACE                      0x40
#define AEM_MASK_LABEL_MIDI_CNT                 0xF0
#define AEM_MASK_LABEL_SMPTE_CNT                0x0F
#define AEM_MASK_NB                             0x40
#define AEM_MASK_OUI24                          0xFFFFFF00
#define AEM_MASK_SF                             0x80
#define AEM_MASK_SF_SUBTYPE                     0x7F
#define AEM_MASK_SF_VERSION                     0x80
#define AEM_PULL_FIELD_MASK                     0xe0


#define MFD_TYPE_VIDEO                          0x00
#define MFD_TYPE_TRANSPORT_STRM                 0x01
#define MFD_TYPE_MIDI                           0x02
#define MFD_TYPE_TIME_CODE                      0x03
#define MFD_TYPE_CONTROL_PROTO                  0x04

#define COLOR_FORMAT_MONO                       0x0
#define COLOR_FORMAT_RGB                        0x1
#define COLOR_FORMAT_RGBA                       0x2
#define COLOR_FORMAT_BGRA                       0x3
#define COLOR_FORMAT_CMYK                       0x4
#define COLOR_FORMAT_HSL                        0x5
#define COLOR_FORMAT_YUV                        0x6
#define COLOR_FORMAT_YCBCR                      0x7
#define COLOR_FORMAT_YPBPR                      0x08

#define OUI24_STANDARD_MEDIA_FORMAT             0x90e0f0

#define MEMORY_OBJECT_TYPE_FIRMWARE_IMAGE       0x0000
#define MEMORY_OBJECT_TYPE_VENDOR_SPECIFIC      0x0001
#define MEMORY_OBJECT_TYPE_CRASH_DUMP           0x0002
#define MEMORY_OBJECT_TYPE_LOG_OBJECT           0x0003
#define MEMORY_OBJECT_TYPE_AUTOSTART_SETTINGS   0x0004
#define MEMORY_OBJECT_TYPE_SNAPSHOT_SETTINGS    0x0005

#define KEY_TYPE_NONE            0
#define KEY_TYPE_SHA256          1
#define KEY_TYPE_AES128          2
#define KEY_TYPE_AES256          3
#define KEY_TYPE_RSA1024_PUBLIC  4
#define KEY_TYPE_RSA1024_PRIVATE 5
#define KEY_TYPE_X509            6

static const value_string aecp_key_type_type_vals [] = {
   {KEY_TYPE_NONE,            "NO_KEY"},
   {KEY_TYPE_SHA256,          "SHA256"},
   {KEY_TYPE_AES128,          "AES128"},
   {KEY_TYPE_AES256,          "AES256"},
   {KEY_TYPE_RSA1024_PUBLIC,  "RSA1024_PUBLIC"},
   {KEY_TYPE_RSA1024_PRIVATE, "RSA1024_PRIVATE"},
   {KEY_TYPE_X509,            "RSA1024_X509"},
   {0,                         NULL}
};

static const value_string aem_memory_object_type_vals [] = {
   {MEMORY_OBJECT_TYPE_FIRMWARE_IMAGE,       "FIRMWARE_IMAGE"},
   {MEMORY_OBJECT_TYPE_VENDOR_SPECIFIC,      "VENDOR_SPECIFIC"},
   {MEMORY_OBJECT_TYPE_CRASH_DUMP,           "CRASH_DUMP"},
   {MEMORY_OBJECT_TYPE_LOG_OBJECT,           "LOG_OBJECT"},
   {MEMORY_OBJECT_TYPE_AUTOSTART_SETTINGS,   "AUTOSTART_SETTINGS"},
   {MEMORY_OBJECT_TYPE_SNAPSHOT_SETTINGS,    "SNAPSHOT_SETTINGS"},
   {0,                                       NULL}
};

#if 0
static const value_string aem_stream_format_subtype_vals [] = {
   {IEC_61883_IIDC_SUBTYPE,      "IEC_61883_IIDC_SUBTYPE"},
   {MMA_SUBTYPE,                 "MMA_SUBTYPE"},
   {AVTP_AUDIO_SUBTYPE,          "AVTP_AUDIO_SUBTYPE"},
   {AVTP_VIDEO_SUBTYPE,          "AVTP_VIDEO_SUBTYPE"},
   {EXPERIMENTAL_SUBTYPE,        "EXPERIMENTAL_SUBTYPE"},
   {0,                            NULL}
};

static const value_string aem_color_format_type_vals [] = {
   {COLOR_FORMAT_MONO,  "Monochrome"},
   {COLOR_FORMAT_RGB,   "RGB"},
   {COLOR_FORMAT_RGBA,  "RGBA"},
   {COLOR_FORMAT_BGRA,  "BGRA"},
   {COLOR_FORMAT_CMYK,  "CMYK"},
   {COLOR_FORMAT_HSL,   "HSL"},
   {COLOR_FORMAT_YUV,   "YUV"},
   {COLOR_FORMAT_YCBCR, "YCbCr"},
   {COLOR_FORMAT_YPBPR, "YPbPr"},
   {0,                   NULL}
};

static const value_string aem_mfd_type_vals [] = {
   {MFD_TYPE_VIDEO,             "VIDEO"},
   {MFD_TYPE_TRANSPORT_STRM,    "TRANSPORT_STREAM"},
   {MFD_TYPE_MIDI,              "MIDI"},
   {MFD_TYPE_TIME_CODE,         "TIME_CODE"},
   {MFD_TYPE_CONTROL_PROTO,     "CONTROL_PROTOCOL"},
   {0,                           NULL}
};
#endif

static const value_string aem_clock_source_type_vals [] = {
   {AEM_CLOCK_INTERNAL          , "INTERNAL"},
   {AEM_CLOCK_EXTERNAL          , "EXTERNAL"},
   {AEM_CLOCK_INPUT_STREAM      , "INPUT STREAM"},
   {AEM_CLOCK_MEDIA_CLOCK_STREAM, "MEDIA CLOCK STREAM"},
   {0, NULL}
};

/* frequency multipliers from table 7.6 pull field values */
static const value_string aem_frequency_multiplier_type_vals [] = {
   {0, "1.0"},
   {1, "1/1.001"},
   {2, "1.001"},
   {3, "24/25"},
   {4, "25/24"},
   {0, NULL}
};

static const value_string aem_control_value_type_vals [] = {
   {AEM_CONTROL_LINEAR_INT8        ,"CONTROL_LINEAR_INT8"},
   {AEM_CONTROL_LINEAR_UINT8       ,"CONTROL_LINEAR_UINT8"},
   {AEM_CONTROL_LINEAR_INT16       ,"CONTROL_LINEAR_INT16"},
   {AEM_CONTROL_LINEAR_UINT16      ,"CONTROL_LINEAR_UINT16"},
   {AEM_CONTROL_LINEAR_INT32       ,"CONTROL_LINEAR_INT32"},
   {AEM_CONTROL_LINEAR_UINT32      ,"CONTROL_LINEAR_UINT32"},
   {AEM_CONTROL_LINEAR_INT64       ,"CONTROL_LINEAR_INT64"},
   {AEM_CONTROL_LINEAR_UINT64      ,"CONTROL_LINEAR_UINT64"},
   {AEM_CONTROL_LINEAR_FLOAT       ,"CONTROL_LINEAR_FLOAT"},
   {AEM_CONTROL_LINEAR_DOUBLE      ,"CONTROL_LINEAR_DOUBLE"},
   {AEM_CONTROL_SELECTOR_INT8      ,"CONTROL_SELECTOR_INT8"},
   {AEM_CONTROL_SELECTOR_UINT8     ,"CONTROL_SELECTOR_UINT8"},
   {AEM_CONTROL_SELECTOR_INT16     ,"CONTROL_SELECTOR_INT16"},
   {AEM_CONTROL_SELECTOR_UINT16    ,"CONTROL_SELECTOR_UINT16"},
   {AEM_CONTROL_SELECTOR_INT32     ,"CONTROL_SELECTOR_INT32"},
   {AEM_CONTROL_SELECTOR_UINT32    ,"CONTROL_SELECTOR_UINT32"},
   {AEM_CONTROL_SELECTOR_INT64     ,"CONTROL_SELECTOR_INT64"},
   {AEM_CONTROL_SELECTOR_UINT64    ,"CONTROL_SELECTOR_UINT64"},
   {AEM_CONTROL_SELECTOR_FLOAT     ,"CONTROL_SELECTOR_FLOAT"},
   {AEM_CONTROL_SELECTOR_DOUBLE    ,"CONTROL_SELECTOR_DOUBLE"},
   {AEM_CONTROL_UTF8               ,"CONTROL_UTF8"},
   {AEM_CONTROL_BODE_PLOT          ,"CONTROL_BODE_PLOT"},
   {AEM_CONTROL_ARRAY_INT8         ,"CONTROL_ARRAY_INT8"},
   {AEM_CONTROL_ARRAY_UINT8        ,"CONTROL_ARRAY_UINT8"},
   {AEM_CONTROL_ARRAY_INT16        ,"CONTROL_ARRAY_INT16"},
   {AEM_CONTROL_ARRAY_UINT16       ,"CONTROL_ARRAY_UINT16"},
   {AEM_CONTROL_ARRAY_INT32        ,"CONTROL_ARRAY_INT32"},
   {AEM_CONTROL_ARRAY_UINT32       ,"CONTROL_ARRAY_UINT32"},
   {AEM_CONTROL_ARRAY_INT64        ,"CONTROL_ARRAY_INT64"},
   {AEM_CONTROL_ARRAY_UINT64       ,"CONTROL_ARRAY_UINT64"},
   {AEM_CONTROL_ARRAY_FLOAT        ,"CONTROL_ARRAY_FLOAT"},
   {AEM_CONTROL_ARRAY_DOUBLE       ,"CONTROL_ARRAY_DOUBLE"},
   {AEM_CONTROL_VENDOR             ,"CONTROL_CONTROL_VENDOR"},
   {0                              , NULL}
};

static const value_string aem_jack_type_vals [] = {
   {AEM_JACKTYPE_SPEAKER            ,"SPEAKER" },
   {AEM_JACKTYPE_HEADPHONE          ,"HEADPHONE" },
   {AEM_JACKTYPE_ANALOG_MICROPHONE  ,"ANALOG_MICROPHONE" },
   {AEM_JACKTYPE_SPDIF              ,"SPDIF" },
   {AEM_JACKTYPE_ADAT               ,"ADAT" },
   {AEM_JACKTYPE_TDIF               ,"TDIF" },
   {AEM_JACKTYPE_MADI               ,"MADI" },
   {AEM_JACKTYPE_UNBALANCED_ANALOG  ,"UNBALANCED_ANALOG" },
   {AEM_JACKTYPE_BALANCED_ANALOG    ,"BALANCED_ANALOG" },
   {AEM_JACKTYPE_DIGITAL            ,"DIGITAL" },
   {AEM_JACKTYPE_MIDI               ,"MIDI" },
   {AEM_JACKTYPE_AES_EBU            ,"AES_EBU" },
   {AEM_JACKTYPE_COMPOSITE_VIDEO    ,"COMPOSITE_VIDEO" },
   {AEM_JACKTYPE_S_VHS_VIDEO        ,"S_VHS_VIDEO" },
   {AEM_JACKTYPE_COMPONENT_VIDEO    ,"COMPONENT_VIDEO" },
   {AEM_JACKTYPE_DVI                ,"DVI" },
   {AEM_JACKTYPE_HDMI               ,"HDMI" },
   {AEM_JACKTYPE_UDI                ,"UDI" },
   {AEM_JACKTYPE_DISPLAYPORT        ,"DISPLAYPORT" },
   {AEM_JACKTYPE_ANTENNA            ,"ANTENNA" },
   {AEM_JACKTYPE_ANALOG_TUNER       ,"ANALOG_TUNER" },
   {AEM_JACKTYPE_ETHERNET           ,"ETHERNET" },
   {AEM_JACKTYPE_WIFI               ,"WIFI" },
   {AEM_JACKTYPE_USB                ,"USB" },
   {AEM_JACKTYPE_PCI                ,"PCI" },
   {AEM_JACKTYPE_PCI_E              ,"PDI_E" },
   {AEM_JACKTYPE_SCSI               ,"SCSI" },
   {AEM_JACKTYPE_ATA                ,"ATA" },
   {AEM_JACKTYPE_IMAGER             ,"IMAGER" },
   {AEM_JACKTYPE_IR                 ,"IR" },
   {AEM_JACKTYPE_THUNDERBOLT        ,"THUNDERBOLT" },
   {AEM_JACKTYPE_SATA               ,"SATA" },
   {AEM_JACKTYPE_SMPTE_LTC          ,"SMPTE_LTC" },
   {AEM_JACKTYPE_DIGITAL_MICROPHONE ,"DIGITAL_MICROPHONE" },
   {0, NULL }
};

/* value_string uses a 32 bit integer id, control uses 64.
 * TODO - make custom formatter for hf_aem_control_type
static const value_string aem_control_type_vals [] = {
   {AEM_CONTROL_ENABLE              , "ENABLE"},
   {AEM_CONTROL_DELAY               , "DELAY"},
   {AEM_CONTROL_POW_LINE_FREQ       , "POW_LINE_FREQ"},
   {AEM_CONTROL_ROLLPITCHYAW_ABS    , "ROLLPITCHYAW_ABS"},
   {AEM_CONTROL_ROLLPITCHYAW_REL    , "ROLLPITCHYAW_REL"},
   {AEM_CONTROL_SURGESWAYHEAVE_ABS  , "SURGESWAYHEAVE_ABS"},
   {AEM_CONTROL_SURGESWAYHEAVE_REL  , "SURGESWAYHEAVE_REL"},
   {AEM_CONTROL_IDENTIFY            , "IDENTIFY"},
   {AEM_CONTROL_POWER_STATUS        , "POWER_STATUS"},
   {AEM_CONTROL_FAN_STATUS          , "FAN_STATUS"},
   {AEN_CONTROL_TEMPERATURE         , "TEMPERATURE"},
   {AEM_CONTROL_TEMPERATURE_SENSOR  , "TEMPERATURE_SENSOR"},
   {AEM_CONTROL_ALTITUDE            , "ALTITUDE"},
   {AEM_CONTROL_HUMIDITY            , "HUMIDITY"},
   {AEM_CONTROL_MUTE                , "MUTE"},
   {AEM_CONTROL_VOLUME              , "VOLUME"},
   {AEM_CONTROL_INVERT              , "INVERT"},
   {AEM_CONTROL_PANPOT              , "PANPOT"},
   {AEM_CONTROL_ISOLATE             , "ISOLATE"},
   {AEM_CONTROL_POSITION            , "POSITION"},
   {AEM_CONTROL_PHANTOM             , "PHANTOM"},
   {AEM_CONTROL_AUDIO_SCALE         , "AUDIO_SCALE"},
   {AEM_CONTROL_AUDIO_METERS        , "AUDIO_METERS"},
   {AEM_CONTROL_AUDIO_SPECTRUM      , "AUDIO_SPECTRUM"},
   {AEM_CONTROL_FILTER_RESPONSE     , "FILTER_RESPONSE"},
   {AEM_CONTROL_SCANNING_MODE       , "SCANNING_MODE"},
   {AEM_CONTROL_AUTO_EXP_MODE       , "AUTO_EXP_MODE"},
   {AEM_CONTROL_AUTO_EXP_PRIO       , "AUTO_EXP_PRIO"},
   {AEM_CONTROL_EXP_TIME_ABS        , "EXP_TIME_ABS"},
   {AEM_CONTROL_EXP_TIME_REL        , "EXP_TIME_REL"},
   {AEM_CONTROL_FOCUS_ABS           , "FOCUS_ABS"},
   {AEM_CONTROL_FOCUS_REL           , "FOCUS_REL"},
   {AEM_CONTROL_FOCUS_AUTO          , "FOCUS_AUTO"},
   {AEM_CONTROL_IRIS_ABS            , "IRIS_ABS"},
   {AEM_CONTROL_IRIS_REL            , "IRIS_REL"},
   {AEM_CONTROL_ZOOM_ABS            , "ZOOM_ABS"},
   {AEM_CONTROL_ZOOM_REL            , "ZOOM_REL"},
   {AEM_CONTROL_PRIVACY             , "PRIVACY"},
   {AEM_CONTROL_BACKLIGHT           , "BACKLIGHT"},
   {AEM_CONTROL_BRIGHTNESS          , "BRIGHTNESS"},
   {AEM_CONTROL_CONTRAST            , "CONTRAST"},
   {AEM_CONTROL_GAIN                , "GAIN"},
   {AEM_CONTROL_HUE                 , "HUE"},
   {AEM_CONTROL_SATURATION          , "SATURATION"},
   {AEM_CONTROL_SHARPNESS           , "SHARPNESS"},
   {AEM_CONTROL_GAMMA               , "GAMMA"},
   {AEM_CONTROL_WHITE_BAL_TEMP      , "WHITE_BAL_TEMP"},
   {AEM_CONTROL_WHITE_BAL_TENP_AUTO , "WHITE_BAL_TEMP_AUTO"},
   {AEM_CONTROL_WHITE_BAL_COMP      , "WHITE_BAL_COMP"},
   {AEM_CONTROL_WHITE_BAL_COMP_AUTO , "WHITE_BAL_COMP_AUTO"},
   {AEM_CONTROL_DIGITAL_ZOOM        , "DIGITAL_ZOOM"},
   {0                               , NULL}
};
*/

#if 0
static const value_string aecp_address_type_vals [] = {
   {AECP_ADDRESS_MAC  , "MAC"},
   {AECP_ADDRESS_IPV4 , "IPV4"},
   {AECP_ADDRESS_IPV6 , "IPV6"},
   {0                 , NULL}
};
#endif

static const value_string aecp_keychain_id_type_vals [] = {
   {AECP_KEYCHAIN_NONE         , "NONE"},
   {AECP_KEYCHAIN_MANUFACTURER , "MANUFACTURER"},
   {AECP_KEYCHAIN_PRODUCT      , "PRODUCT"},
   {AECP_KEYCHAIN_ENTITY       , "ENTITY"},
   {AECP_KEYCHAIN_USER         , "USER"},
   {0                          ,  NULL}
};

static const value_string aecp_direction_type_vals [] = {
   {AECP_DIRECTION_HORIZONTAL , "HORIZONTAL"},
   {AECP_DIRECTION_VERTICAL   , "VERTICAL"},
   {0                         , NULL}
};

static const value_string aem_descriptor_type_vals[] = {
   {AEM_DESCRIPTOR_ENTITY              , "ENTITY"},
   {AEM_DESCRIPTOR_CONFIGURATION       , "CONFIGURATION"},
   {AEM_DESCRIPTOR_AUDIO_UNIT               , "AUDIO_UNIT"},
   {AEM_DESCRIPTOR_VIDEO_UNIT               , "VIDEO_UNIT"},
   {AEM_DESCRIPTOR_SENSOR_UNIT              , "SENSOR_UNIT"},
   {AEM_DESCRIPTOR_STREAM_INPUT        , "STREAM_INPUT"},
   {AEM_DESCRIPTOR_STREAM_OUTPUT       , "STREAM_OUTPUT"},
   {AEM_DESCRIPTOR_JACK_INPUT , "JACK_INPUT"},
   {AEM_DESCRIPTOR_JACK_OUTPUT, "JACK_OUTPUT"},
   {AEM_DESCRIPTOR_STREAM_PORT_INPUT    , "STREAM_PORT_INPUT"},
   {AEM_DESCRIPTOR_STREAM_PORT_OUTPUT    , "STREAM_PORT_OUTPUT"},
   {AEM_DESCRIPTOR_EXTERNAL_PORT_INPUT , "EXTERNAL_PORT_INPUT"},
   {AEM_DESCRIPTOR_EXTERNAL_PORT_OUTPUT, "EXTERNAL_PORT_OUTPUT"},
   {AEM_DESCRIPTOR_INTERNAL_PORT_INPUT , "INTERNAL_PORT_INPUT"},
   {AEM_DESCRIPTOR_INTERNAL_PORT_OUTPUT, "INTERNAL_PORT_OUTPUT"},
   {AEM_DESCRIPTOR_AVB_INTERFACE       , "AVB_INTERFACE"},
   {AEM_DESCRIPTOR_CLOCK_SOURCE        , "CLOCK_SOURCE"},
   {AEM_DESCRIPTOR_AUDIO_CLUSTER       , "AUDIO_CLUSTER"},
   {AEM_DESCRIPTOR_VIDEO_CLUSTER       , "VIDEO_CLUSTER"},
   {AEM_DESCRIPTOR_SENSOR_CLUSTER      , "SENSOR_CLUSTER"},
   {AEM_DESCRIPTOR_AUDIO_MAP           , "AUDIO_MAP"},
   {AEM_DESCRIPTOR_VIDEO_MAP           , "VIDEO_MAP"},
   {AEM_DESCRIPTOR_SENSOR_MAP          , "SENSOR_MAP"},
   {AEM_DESCRIPTOR_CONTROL             , "CONTROL"},
   {AEM_DESCRIPTOR_SIGNAL_SELECTOR     , "SIGNAL_SELECTOR"},
   {AEM_DESCRIPTOR_MIXER               , "MIXER"},
   {AEM_DESCRIPTOR_MATRIX              , "MATRIX"},
   {AEM_DESCRIPTOR_LOCALE              , "LOCALE"},
   {AEM_DESCRIPTOR_STRINGS             , "STRINGS"},
   {AEM_DESCRIPTOR_MATRIX_SIGNAL       , "MATRIX_SIGNAL"},
   {AEM_DESCRIPTOR_MEMORY_OBJECT       , "MEMORY_OBJECT"},
   {AEM_DESCRIPTOR_SIGNAL_SPLITTER     , "SIGNAL_SPLITTER"},
   {AEM_DESCRIPTOR_SIGNAL_COMBINER     , "SIGNAL_COMBINER"},
   {AEM_DESCRIPTOR_SIGNAL_DEMULTIPLEXER, "SIGNAL_DEMULTIPLEXER"},
   {AEM_DESCRIPTOR_SIGNAL_MULTIPLEXER  , "SIGNAL_MULTIPLEXER"},
   {AEM_DESCRIPTOR_SIGNAL_TRANSCODER   , "SIGNAL_TRANSCODER"},
   {AEM_DESCRIPTOR_CLOCK_DOMAIN        , "CLOCK_DOMAIN"},
   {AEM_DESCRIPTOR_CONTROL_BLOCK       , "CONTROL_BLOCK"},
   {0                                  , NULL}
};

static const value_string aem_status_type_vals[] = {
   {AEM_STATUS_SUCCESS                , "SUCCESS"},
   {AEM_STATUS_NOT_IMPLEMENTED        , "NOT_IMPLEMENTED"},
   {AEM_STATUS_NO_SUCH_DESCRIPTOR     , "NO_SUCH_DESCRIPTOR"},
   {AEM_STATUS_ENTITY_LOCKED          , "ENTITY_LOCKED"},
   {AEM_STATUS_ENTITY_ACQUIRED        , "ENTITY_ACQUIRED"},
   {AEM_STATUS_NOT_AUTHENTICATED      , "NOT_AUTHENTICATED"},
   {AEM_STATUS_AUTHENTICATION_DISABLED, "AUTHENTICATION_DISABLED"},
   {AEM_STATUS_BAD_ARGUMENTS          , "BAD_ARGUMENTS"},
   {AEM_STATUS_NO_RESOURCES           , "NO_RESOURCES"},
   {AEM_STATUS_IN_PROGRESS            , "IN_PROGRESS"},
   {AEM_STATUS_ENTITY_MISBEHAVING     , "ENTITY_MISBEHAVING"},
   {AEM_STATUS_NOT_SUPPORTED          , "NOT_SUPPORTED"},
   {AEM_STATUS_STREAM_IS_RUNNING      , "STREAM_IS_RUNNING"},
   {0                                 , NULL}
};

static const value_string aecp_message_type_vals[] = {
   {AECP_AEM_COMMAND_MESSAGE,             "AEM_COMMAND"},
   {AECP_AEM_RESPONSE_MESSAGE,            "AEM_RESPONSE"},
   {AECP_ADDRESS_ACCESS_COMMAND_MESSAGE,  "ADDRESS_ACCESS_COMMAND"},
   {AECP_ADDRESS_ACCESS_RESPONSE_MESSAGE, "ADDRESS_ACCESS_RESPONSE"},
   {AECP_AVC_COMMAND_MESSAGE,             "AVC_COMMAND"},
   {AECP_AVC_RESPONSE_MESSAGE,            "AVC_RESPONSE"},
   {AECP_VENDOR_UNIQUE_COMMAND_MESSAGE,   "VENDOR_UNIQUE_COMMAND"},
   {AECP_VENDOR_UNIQUE_RESPONSE_MESSAGE,  "VENDOR_UNIQUE_RESPONSE"},
   {AECP_EXTENDED_COMMAND_MESSAGE,        "EXTENDED_COMMAND"},
   {AECP_EXTENDED_RESPONSE_MESSAGE,       "EXTENDED_RESPONSE"},
   {0,                                    NULL }
};

static const value_string aecp_command_type_vals[] = {
   {AECP_COMMAND_ACQUIRE_ENTITY                , "ACQUIRE_ENTITY"},
   {AECP_COMMAND_LOCK_ENTITY                   , "LOCK_ENTITY"},
   {AECP_COMMAND_ENTITY_AVAILABLE              , "ENTITY_AVAILABLE"},
   {AECP_COMMAND_CONTROLLER_AVAILABLE          , "CONTROLLER_AVAILABLE"},
   {AECP_COMMAND_READ_DESCRIPTOR               , "READ_DESCRIPTOR"},
   {AECP_COMMAND_WRITE_DESCRIPTOR              , "WRITE_DESCRIPTOR"},
   {AECP_COMMAND_SET_CONFIGURATION             , "SET_CONFIGURATION"},
   {AECP_COMMAND_GET_CONFIGURATION             , "GET_CONFIGURATION"},
   {AECP_COMMAND_SET_STREAM_FORMAT             , "SET_STREAM_FORMAT"},
   {AECP_COMMAND_GET_STREAM_FORMAT             , "GET_STREAM_FORMAT"},
   {AECP_COMMAND_SET_VIDEO_FORMAT              , "SET_VIDEO_FORMAT"},
   {AECP_COMMAND_GET_VIDEO_FORMAT              , "GET_VIDEO_FORMAT"},
   {AECP_COMMAND_SET_SENSOR_FORMAT             , "SET_SENSOR_FORMAT"},
   {AECP_COMMAND_GET_SENSOR_FORMAT             , "GET_SENSOR_FORMAT"},
   {AECP_COMMAND_SET_STREAM_INFO               , "SET_STREAM_INFO"},
   {AECP_COMMAND_GET_STREAM_INFO               , "GET_STREAM_INFO"},
   {AECP_COMMAND_SET_NAME                      , "SET_NAME"},
   {AECP_COMMAND_GET_NAME                      , "GET_NAME"},
   {AECP_COMMAND_SET_ASSOCIATION_ID            , "SET_ASSOCIATION_ID"},
   {AECP_COMMAND_GET_ASSOCIATION_ID            , "GET_ASSOCIATION_ID"},
   {AECP_COMMAND_SET_SAMPLING_RATE             , "SET_SAMPLING_RATE"},
   {AECP_COMMAND_GET_SAMPLING_RATE             , "GET_SAMPLING_RATE"},
   {AECP_COMMAND_SET_CLOCK_SOURCE              , "SET_CLOCK_SOURCE"},
   {AECP_COMMAND_GET_CLOCK_SOURCE              , "GET_CLOCK_SOURCE"},
   {AECP_COMMAND_SET_CONTROL_VALUE             , "SET_CONTROL_VALUE"},
   {AECP_COMMAND_GET_CONTROL_VALUE             , "GET_CONTROL_VALUE"},
   {AECP_COMMAND_INCREMENT_CONTROL             , "INCREMENT_CONTROL"},
   {AECP_COMMAND_DECREMENT_CONTROL             , "DECREMENT_CONTROL"},
   {AECP_COMMAND_SET_SIGNAL_SELECTOR           , "SET_SIGNAL_SELECTOR"},
   {AECP_COMMAND_GET_SIGNAL_SELECTOR           , "GET_SIGNAL_SELECTOR"},
   {AECP_COMMAND_SET_MIXER                     , "SET_MIXER"},
   {AECP_COMMAND_GET_MIXER                     , "GET_MIXER"},
   {AECP_COMMAND_SET_MATRIX                    , "SET_MATRIX"},
   {AECP_COMMAND_GET_MATRIX                    , "GET_MATRIX"},
   {AECP_COMMAND_START_STREAMING               , "START_STREAMING"},
   {AECP_COMMAND_STOP_STREAMING                , "STOP_STREAMING"},
   {AECP_COMMAND_REGISTER_UNSOL_NOTIFICATION   , "REGISTER_UNSOL_NOTIFICATION"},
   {AECP_COMMAND_DEREGISTER_UNSOL_NOTIFICATION , "DEREGISTER_UNSOL_NOTIFICATION"},
   {AECP_COMMAND_IDENTIFY_NOTIFICATION         , "IDENTIFY_NOTIFICATION"},
   {AECP_COMMAND_GET_AVB_INFO                  , "GET_AVB_INFO"},
   {AECP_COMMAND_GET_AS_PATH                   , "GET_AS_PATH"},
   {AECP_COMMAND_GET_COUNTERS                  , "GET_COUNTERS"},
   {AECP_COMMAND_REBOOT                        , "REBOOT"},
   {AECP_COMMAND_GET_AUDIO_MAP                 , "GET_AUDIO_MAP"},
   {AECP_COMMAND_ADD_AUDIO_MAPPINGS            , "ADD_AUDIO_MAPPINGS"},
   {AECP_COMMAND_REMOVE_AUDIO_MAPPINGS         , "REMOVE_AUDIO_MAPPINGS"},
   {AECP_COMMAND_GET_VIDEO_MAP                 , "GET_VIDEO_MAP"},
   {AECP_COMMAND_ADD_VIDEO_MAPPINGS            , "ADD_VIDEO_MAPPINGS"},
   {AECP_COMMAND_REMOVE_VIDEO_MAPPINGS         , "REMOVE_VIDEO_MAPPINGS"},
   {AECP_COMMAND_GET_SENSOR_MAP                , "GET_SENSOR_MAP"},
   {AECP_COMMAND_ADD_SENSOR_MAPPINGS           , "ADD_SENSOR_MAPPINGS"},
   {AECP_COMMAND_REMOVE_SENSOR_MAPPINGS        , "REMOVE_SENSOR_MAPPINGS"},
   {AECP_COMMAND_START_OPERATION               , "START_OPERATION"},
   {AECP_COMMAND_ABORT_OPERATION               , "ABORT_OPERATION"},
   {AECP_COMMAND_OPERATION_STATUS              , "OPERATION_STATUS"},
   {AECP_COMMAND_AUTH_ADD_KEY                  , "AUTH_ADD_KEY"},
   {AECP_COMMAND_AUTH_DELETE_KEY               , "AUTH_DELETE_KEY"},
   {AECP_COMMAND_AUTH_GET_KEY_LIST             , "AUTH_GET_KEY_LIST"},
   {AECP_COMMAND_AUTH_GET_KEY                  , "AUTH_GET_KEY"},
   {AECP_COMMAND_AUTH_ADD_KEY_TO_CHAIN         , "AUTH_ADD_KEY_TO_CHAIN"},
   {AECP_COMMAND_AUTH_DELETE_KEY_FROM_CHAIN    , "AUTH_DELETE_KEY_FROM_CHAIN"},
   {AECP_COMMAND_AUTH_GET_KEYCHAIN_LIST        , "AUTH_GET_KEYCHAIN_LIST"},
   {AECP_COMMAND_AUTH_GET_IDENTITY             , "AUTH_GET_IDENTITY"},
   {AECP_COMMAND_AUTH_ADD_TOKEN                , "AUTH_ADD_TOKEN"},
   {AECP_COMMAND_AUTH_DELETE_TOKEN             , "AUTH_DELETE_TOKEN"},
   {AECP_COMMAND_AUTHENTICATE                  , "AUTHENTICATE"},
   {AECP_COMMAND_DEAUTHENTICATE                , "DEAUTHENTICATE"},
   {AECP_COMMAND_ENABLE_TRANSPORT_SECURITY     , "ENABLE_TRANSPORT_SECURITY"},
   {AECP_COMMAND_DISABLE_TRANSPORT_SECURITY    , "DISABLE_TRANSPORT_SECURITY"},
   {AECP_COMMAND_ENABLE_STREAM_ENCRYPTION      , "ENABLE_STREAM_ENCRYPTION"},
   {AECP_COMMAND_DISABLE_STREAM_ENCRYPTION     , "DISABLE_STREAM_ENCRYPTION"},
   {AECP_COMMAND_SET_STREAM_BACKUP             , "SET_STREAM_BACKUP"},
   {AECP_COMMAND_GET_STREAM_BACKUP             , "GET_STREAM_BACKUP"},
   {0                                          , NULL}
};


static const value_string adp_message_type_vals[] = {
   {ADP_ENTITY_AVAILABLE_MESSAGE,       "ENTITY_AVAILABLE"},
   {ADP_ENTITY_DEPARTING_MESSAGE,       "ENTITY_DEPARTING"},
   {ADP_ENTITY_DISCOVER_MESSAGE,        "ENTITY_DISCOVER"},
   {0,                                  NULL }
};

static const value_string acmp_message_type_vals[] = {
   {ACMP_CONNECT_TX_COMMAND,           "CONNECT_TX_COMMAND"},
   {ACMP_CONNECT_TX_RESPONSE,          "CONNECT_TX_RESPONSE"},
   {ACMP_DISCONNECT_TX_COMMAND,        "DISCONNECT_TX_COMMAND"},
   {ACMP_DISCONNECT_TX_RESPONSE,       "DISCONNECT_TX_RESPONSE"},
   {ACMP_GET_TX_STATE_COMMAND,         "GET_TX_STATE_COMMAND"},
   {ACMP_GET_TX_STATE_RESPONSE,        "GET_TX_STATE_RESPONSE"},
   {ACMP_CONNECT_RX_COMMAND,           "CONNECT_RX_COMMAND"},
   {ACMP_CONNECT_RX_RESPONSE,          "CONNECT_RX_RESPONSE"},
   {ACMP_DISCONNECT_RX_COMMAND,        "DISCONNECT_RX_COMMAND"},
   {ACMP_DISCONNECT_RX_RESPONSE,       "DISCONNECT_RX_RESPONSE"},
   {ACMP_GET_RX_STATE_COMMAND,         "GET_RX_STATE_COMMAND"},
   {ACMP_GET_RX_STATE_RESPONSE,        "GET_RX_STATE_RESPONSE"},
   {ACMP_GET_TX_CONNECTION_COMMAND,    "GET_TX_CONNECTION_COMMAND"},
   {ACMP_GET_TX_CONNECTION_RESPONSE,   "GET_TX_CONNECTION_RESPONSE"},
   {0,                                  NULL }
};

static const value_string acmp_status_field_vals[] = {
   {ACMP_STATUS_SUCCESS,                               "SUCCESS"},
   {ACMP_STATUS_LISTENER_UNKNOWN_ID,                   "LISTENER_UNKNOWN_ID"},
   {ACMP_STATUS_TALKER_UNKNOWN_ID,                     "TALKER_UNKNOWN_ID"},
   {ACMP_STATUS_TALKER_DEST_MAC_FAIL,                  "TALKER_DEST_MAC_FAIL"},
   {ACMP_STATUS_TALKER_NO_STREAM_INDEX,                "TALKER_NO_STREAM_INDEX"},
   {ACMP_STATUS_TALKER_NO_BANDWIDTH,                   "TALKER_NO_BANDWIDTH"},
   {ACMP_STATUS_TALKER_EXCLUSIVE,                      "TALKER_EXCLUSIVE"},
   {ACMP_STATUS_LISTENER_TALKER_TIMEOUT,               "LISTENER_TALKER_TIMEOUT"},
   {ACMP_STATUS_LISTENER_EXCLUSIVE,                    "LISTENER_EXCLUSIVE"},
   {ACMP_STATUS_STATE_UNAVAILABLE,                     "STATE_UNAVAILABLE"},
   {ACMP_STATUS_NOT_CONNECTED,                         "NOT_CONNECTED"},
   {ACMP_STATUS_NO_SUCH_CONNECTION,                    "NO_SUCH_CONNECTION"},
   {ACMP_STATUS_COULD_NOT_SEND_MESSAGE,                "COULD_NOT_SEND_MESSAGE"},
   {ACMP_STATUS_DEFAULT_SET_DIFFERENT,                 "DEFAULT_SET_DIFFERENT"},
   {ACMP_STATUS_NOT_SUPPORTED,                         "NOT_SUPPORTED"},
   {0,                                  NULL }
};

static const value_string aecp_sampling_rate_pull_field_vals[] = {
   {0,        "Multiply by 1.0"},
   {1,        "Multiply by 1/1.001"},
   {2,        "Multiply by 1.001"},
   {3,        "Multiply by 24/25"},
   {4,        "Multiply by 25/24"},
   {0,        NULL }
};


/**********************************************************/
/* Initialize the protocol and registered fields          */
/**********************************************************/
static int proto_17221;

/* AVDECC Discovery Protocol Data Unit (ADPDU) */
static int hf_adp_message_type;
static int hf_adp_valid_time;
static int hf_adp_cd_length;
static int hf_adp_entity_id;
static int hf_adp_entity_model_id;
static int hf_adp_entity_cap;
static int hf_adp_talker_stream_srcs;
static int hf_adp_talker_cap;
static int hf_adp_listener_stream_sinks;
static int hf_adp_listener_cap;
static int hf_adp_controller_cap;
static int hf_adp_avail_index;
static int hf_adp_gptp_gm_id;
/* static int hf_adp_def_aud_format; */
/* static int hf_adp_def_vid_format; */
static int hf_adp_assoc_id;
/* static int hf_adp_entity_type; */

/* Entity Capabilities Flags */
static int hf_adp_entity_cap_efu_mode;
static int hf_adp_entity_cap_address_access_supported;
static int hf_adp_entity_cap_gateway_entity;
static int hf_adp_entity_cap_aem_supported;
static int hf_adp_entity_cap_legacy_avc;
static int hf_adp_entity_cap_assoc_id_support;
static int hf_adp_entity_cap_assoc_id_valid;
static int hf_adp_entity_cap_vendor_unique;
static int hf_adp_entity_cap_class_a_supported;
static int hf_adp_entity_cap_class_b_supported;
static int hf_adp_entity_cap_gptp_supported;

/* Talker Capabilities Flags */
static int hf_adp_talk_cap_implement;
static int hf_adp_talk_cap_other_src;
static int hf_adp_talk_cap_control_src;
static int hf_adp_talk_cap_media_clk_src;
static int hf_adp_talk_cap_smpte_src;
static int hf_adp_talk_cap_midi_src;
static int hf_adp_talk_cap_audio_src;
static int hf_adp_talk_cap_video_src;

/* Listener Capabilities Flags */
static int hf_adp_list_cap_implement;
static int hf_adp_list_cap_other_sink;
static int hf_adp_list_cap_control_sink;
static int hf_adp_list_cap_media_clk_sink;
static int hf_adp_list_cap_smpte_sink;
static int hf_adp_list_cap_midi_sink;
static int hf_adp_list_cap_audio_sink;
static int hf_adp_list_cap_video_sink;

/* Controller Capabilities Flags */
static int hf_adp_cont_cap_implement;
static int hf_adp_cont_cap_layer3_proxy;

#if 0
/* Default Audio Format */
static int hf_adp_def_aud_sample_rates;
static int hf_adp_def_aud_max_chan;
static int hf_adp_def_aud_saf_flag;
static int hf_adp_def_aud_float_flag;
static int hf_adp_def_aud_chan_formats;

/* Default Audio Sample Rates */
static int hf_adp_samp_rate_44k1;
static int hf_adp_samp_rate_48k;
static int hf_adp_samp_rate_88k2;
static int hf_adp_samp_rate_96k;
static int hf_adp_samp_rate_176k4;
static int hf_adp_samp_rate_192k;

/* Audio Channel Formats */
static int hf_adp_chan_format_mono;
static int hf_adp_chan_format_2ch;
static int hf_adp_chan_format_3ch;
static int hf_adp_chan_format_4ch;
static int hf_adp_chan_format_5ch;
static int hf_adp_chan_format_6ch;
static int hf_adp_chan_format_7ch;
static int hf_adp_chan_format_8ch;
static int hf_adp_chan_format_10ch;
static int hf_adp_chan_format_12ch;
static int hf_adp_chan_format_14ch;
static int hf_adp_chan_format_16ch;
static int hf_adp_chan_format_18ch;
static int hf_adp_chan_format_20ch;
static int hf_adp_chan_format_22ch;
static int hf_adp_chan_format_24ch;
#endif

/* AECP Vendor Unique Protocol Dissector table */
static dissector_table_t vendor_unique_protocol_dissector_table;

/* ***************************************************************** */
/*     AVDECC Enumeration and Control Protocol Data Unit (AECPDU)    */
/* ***************************************************************** */

static int hf_aecp_aa_count;
static int hf_aecp_aa_tlv_mode;
static int hf_aecp_aa_tlv_length;
static int hf_aecp_aa_tlv_address;

/* static int hf_aecp_address_type; */
static int hf_aecp_association_id;
static int hf_aecp_as_path_count;
static int hf_aecp_as_path_sequences;
static int ett_aecp_get_as_path_sequences;
static int hf_aecp_get_as_info_clock_id;
static int hf_aecp_auth_token;
static int hf_aecp_avb_interface_gptp_gm_changed_valid;
static int hf_aecp_avb_interface_gptp_gm_changed;
static int hf_aecp_avb_interface_link_up_valid;
static int hf_aecp_avb_interface_link_up;
static int hf_aecp_avb_interface_link_down_valid;
static int hf_aecp_avb_interface_link_down;
static int hf_aecp_avb_interface_packets_tx_valid;
static int hf_aecp_avb_interface_packets_tx;
static int hf_aecp_avb_interface_packets_rx_valid;
static int hf_aecp_avb_interface_packets_rx;
static int hf_aecp_avb_interface_rx_crc_error_valid;
static int hf_aecp_avb_interface_rx_crc_error;
static int hf_aecp_cd_length;
static int hf_aecp_clock_domain_locked_valid;
static int hf_aecp_clock_domain_locked;
static int hf_aecp_clock_domain_unlocked_valid;
static int hf_aecp_clock_domain_unlocked;
static int hf_aecp_clock_source_id;
static int hf_aecp_command_type;
static int hf_aecp_configuration;
static int hf_aecp_configuration_index;
static int hf_aecp_connected_flag;
/* static int hf_aecp_continued_flag; */
/* static int hf_aecp_control_admin_flag; */
/* static int hf_aecp_control_user_l1; */
/* static int hf_aecp_control_user_l2; */
/* static int hf_aecp_control_user_l3; */
/* static int hf_aecp_control_user_l4; */
static int hf_aecp_controller_guid;
/* static int hf_aecp_count; */
static int hf_aecp_descriptor_index;
static int hf_aecp_descriptor_type;
/* static int hf_aecp_descriptors; */
static int hf_aecp_dest_mac_valid_flag;
static int hf_aecp_entity_specific1;
static int hf_aecp_entity_specific1_valid;
static int hf_aecp_entity_specific2;
static int hf_aecp_entity_specific2_valid;
static int hf_aecp_entity_specific3;
static int hf_aecp_entity_specific3_valid;
static int hf_aecp_entity_specific4;
static int hf_aecp_entity_specific4_valid;
static int hf_aecp_entity_specific5;
static int hf_aecp_entity_specific5_valid;
static int hf_aecp_entity_specific6;
static int hf_aecp_entity_specific6_valid;
static int hf_aecp_entity_specific7;
static int hf_aecp_entity_specific7_valid;
static int hf_aecp_entity_specific8;
static int hf_aecp_entity_specific8_valid;
static int hf_aecp_flags_32;
/* static int hf_aecp_ipv4_address; */
/* static int hf_aecp_ipv6_address; */
static int hf_aecp_key;
static int hf_aecp_key_eui;
static int hf_aecp_key_length;
/* static int hf_aecp_key_number; */
/* static int hf_aecp_key_part; */
static int hf_aecp_keychain_id;
static int hf_aecp_keychain_id_small;
static int hf_aecp_keychain_list_index;
static int hf_aecp_keychain_number_of_keys;
static int hf_aecp_keychain_number_of_lists;
static int hf_aecp_key_signature;
static int hf_aecp_key_type;
static int hf_aecp_locked_guid;
static int hf_aecp_mac_address;
static int hf_aecp_map_index;
/* static int hf_aecp_matrix_affected_item_count; */
static int hf_aecp_matrix_column;
static int hf_aecp_matrix_direction;
static int hf_aecp_matrix_item_offset;
static int hf_aecp_matrix_region_height;
static int hf_aecp_matrix_region_width;
static int hf_aecp_matrix_rep;
static int hf_aecp_matrix_row;
static int hf_aecp_matrix_value_count;
/* static int hf_aecp_media_format; */
/* static int hf_aecp_mem_obj_admin_flag; */
/* static int hf_aecp_mem_obj_settings_flag; */
static int hf_aecp_message_type;
static int hf_aecp_msrp_acc_lat_valid_flag;
static int hf_aecp_msrp_accumulated_latency;
static int hf_aecp_msrp_failure_bridge_id;
static int hf_aecp_msrp_failure_code;
static int hf_aecp_msrp_failure_valid_flag;
/* static int hf_aecp_stream_vlan_id; */
static int hf_aecp_name;
static int hf_aecp_name_index;
static int hf_aecp_number_of_maps;
static int hf_aecp_operation_id;
static int hf_aecp_operation_type;
static int hf_aecp_owner_guid;
static int hf_aecp_percent_complete;
static int hf_aecp_persistent_flag;
/* static int hf_aecp_private_key_read_flag; */
/* static int hf_aecp_private_key_write_flag; */
/* static int hf_aecp_public_key_write_flag; */
/* static int hf_aecp_query_id; */
/* static int hf_aecp_query_limit; */
/* static int hf_aecp_query_period; */
/* static int hf_aecp_query_type; */
static int hf_aecp_release_flag;
static int hf_aecp_sampling_rate_base_frequency;
static int hf_aecp_sampling_rate_pull;
static int hf_aecp_sequence_id;
static int hf_aecp_signal_index;
static int hf_aecp_signal_type;
static int hf_aecp_signal_output;
static int hf_aecp_stream_format;
static int hf_aecp_stream_format_valid_flag;
static int hf_aecp_stream_id_valid_flag;
static int hf_aecp_stream_input_early_timestamp_valid;
static int hf_aecp_stream_input_early_timestamp;
static int hf_aecp_stream_input_late_timestamp_valid;
static int hf_aecp_stream_input_late_timestamp;
static int hf_aecp_stream_input_media_locked_valid;
static int hf_aecp_stream_input_media_locked;
static int hf_aecp_stream_input_media_unlocked_valid;
static int hf_aecp_stream_input_media_unlocked;
static int hf_aecp_stream_input_media_reset_valid;
static int hf_aecp_stream_input_media_reset;
static int hf_aecp_stream_input_packets_rx_valid;
static int hf_aecp_stream_input_packets_rx;
static int hf_aecp_stream_input_packets_tx_valid;
static int hf_aecp_stream_input_packets_tx;
static int hf_aecp_stream_input_seq_num_mismatch_valid;
static int hf_aecp_stream_input_seq_num_mismatch;
static int hf_aecp_stream_input_stream_reset_valid;
static int hf_aecp_stream_input_stream_reset;
static int hf_aecp_stream_input_timestamp_uncertain_valid;
static int hf_aecp_stream_input_timestamp_uncertain;
static int hf_aecp_stream_input_timestamp_valid_valid;
static int hf_aecp_stream_input_timestamp_valid;
static int hf_aecp_stream_input_timestamp_not_valid_valid;
static int hf_aecp_stream_input_timestamp_not_valid;
static int hf_aecp_stream_input_unsupported_format_valid;
static int hf_aecp_stream_input_unsupported_format;
static int hf_aecp_stream_vlan_id_valid_flag;
static int hf_aecp_target_guid;
static int hf_aecp_token_length;
static int hf_aecp_u_flag;
static int hf_aecp_unlock_flag;
/* static int hf_aecp_values; */
/* static int hf_aecp_values_count; */
static int hf_aecp_vendor_unique_protocol_id;
static int hf_aecp_video_format;
static int hf_aecp_status_code;
static int hf_aecp_backup_talker_entity_id_0;
static int hf_aecp_backup_talker_entity_id_1;
static int hf_aecp_backup_talker_entity_id_2;
static int hf_aecp_backup_talker_unique_id_0;
static int hf_aecp_backup_talker_unique_id_1;
static int hf_aecp_backup_talker_unique_id_2;
static int hf_aecp_backedup_talker_entity_id;
static int hf_aecp_backedup_talker_unique_id;
static int hf_aecp_avb_info_ptp_grandmaster_id;
static int hf_aecp_avb_info_propegation_delay;
static int hf_aecp_avb_info_gptp_domain_number;
/* static int hf_aecp_avb_info_flags; */
static int hf_aecp_as_capable_flag;
static int hf_aecp_gptp_enabled_flag;
static int hf_aecp_srp_enabled_flag;
static int hf_aecp_avb_info_msrp_mappings_count;
static int hf_aecp_avb_info_msrp_mappings;
static int hf_aecp_avb_info_msrp_mapping_traffic_class;
static int hf_aecp_avb_info_msrp_mapping_priority;

static int hf_aecp_get_avb_info_msrp_vlan_id;

/* ***************************************************************** */
/*                   AVDECC Entity Model (AEM)                       */
/* ***************************************************************** */
static int hf_aem_am824_label;
/* static int hf_aem_aspect_x; */
/* static int hf_aem_aspect_y; */
/* static int hf_aem_audio_channels; */
static int hf_aem_avb_interface_id;
static int hf_aem_buffer_length;
static int hf_aem_b_flag;
static int hf_aem_backedup_talker_guid;
static int hf_aem_backedup_talker_unique;
static int hf_aem_backup_talker_guid_0;
static int hf_aem_backup_talker_guid_1;
static int hf_aem_backup_talker_guid_2;
static int hf_aem_backup_talker_unique_0;
static int hf_aem_backup_talker_unique_1;
static int hf_aem_backup_talker_unique_2;
static int hf_aem_base_audio_map;
static int hf_aem_base_cluster;
static int hf_aem_base_control;
static int hf_aem_base_control_block;
/* static int hf_aem_base_destination; */
static int hf_aem_base_external_input_port;
static int hf_aem_base_external_output_port;
static int hf_aem_base_frequency;
static int hf_aem_base_internal_input_port;
static int hf_aem_base_internal_output_port;
static int hf_aem_base_matrix;
static int hf_aem_base_mixer;
static int hf_aem_base_signal_selector;
/* static int hf_aem_base_source; */
static int hf_aem_number_splitters;
static int hf_aem_base_splitter;
static int hf_aem_number_combiners;
static int hf_aem_base_combiner;
static int hf_aem_number_demultiplexers;
static int hf_aem_base_demultiplexer;
static int hf_aem_number_multiplexers;
static int hf_aem_base_multiplexer;
static int hf_aem_number_transcoders;
static int hf_aem_base_transcoder;
static int hf_aem_base_stream_input_port;
static int hf_aem_base_stream_output_port;
static int hf_aem_base_strings;
static int hf_aem_binary_blob;
static int hf_aem_blob_size;
static int hf_aem_block_latency;
static int hf_aem_reset_time;
/* static int hf_aem_bpp; */
static int hf_aem_channel_count;
/* static int hf_aem_channels; */
static int hf_aem_clock_source_flags;
static int hf_aem_clock_source_id;
static int hf_aem_clock_domain_id;
static int hf_aem_clock_sources_offset;
static int hf_aem_clock_source_index;
static int hf_aem_clock_sources_count;
static int hf_aem_clock_sources;
static int hf_aem_clock_sources_array;
static int hf_aem_clock_source_location_id;
static int hf_aem_clock_source_location_type;
static int hf_aem_clock_source_name;
static int hf_aem_clock_source_name_string;
static int hf_aem_clock_source_type;
/* static int hf_aem_cluster_name; */
/* static int hf_aem_cluster_name_string; */
/* static int hf_aem_color_format; */
static int hf_aem_color_space;
/* static int hf_aem_comp1; */
/* static int hf_aem_comp2; */
/* static int hf_aem_comp3; */
/* static int hf_aem_comp4; */
static int hf_aem_compress_mode;
static int hf_aem_configuration_name;
static int hf_aem_configuration_name_string;
static int hf_aem_configurations_count;
static int hf_aem_control_domain;
static int hf_aem_control_latency;
/* static int hf_aem_control_location_id; */
/* static int hf_aem_control_location_type; */
static int hf_aem_control_type;
static int hf_aem_control_value_type;
static int hf_aem_count;
/* static int hf_aem_cs_eui64; */
static int hf_aem_ctrl_double;
static int hf_aem_ctrl_float;
static int hf_aem_ctrl_int16;
static int hf_aem_ctrl_int32;
static int hf_aem_ctrl_int64;
static int hf_aem_ctrl_int8;
static int hf_aem_ctrl_uint16;
static int hf_aem_ctrl_uint32;
static int hf_aem_ctrl_uint64;
static int hf_aem_ctrl_uint8;
static int hf_aem_ctrl_vals;
static int hf_aem_current_configuration;
static int hf_aem_current_sample_rate;
/* static int hf_aem_current_signal_id; */
/* static int hf_aem_current_signal_type; */
static int hf_aem_dbs;
/* static int hf_aem_default_signal_id; */
/* static int hf_aem_default_signal_type; */
static int hf_aem_descriptor_counts_count;
static int hf_aem_descriptor_counts_offset;
/* static int hf_aem_div; */
static int hf_aem_entity_id;
static int hf_aem_entity_model_id;
static int hf_aem_entity_name;
static int hf_aem_fdf_evt;
static int hf_aem_fdf_sfc;
static int hf_aem_firmware_version;
static int hf_aem_flags_async_sample_rate_conv;
static int hf_aem_flags_captive;
static int hf_aem_flags_class_a;
static int hf_aem_flags_class_b;
static int hf_aem_flags_clock_sync_source;
static int hf_aem_flags_sync_sample_rate_conv;
static int hf_aem_fmt;
/* static int hf_aem_formats_count; */
static int hf_aem_formats_offset;
/* static int hf_aem_frame_rate; */
static int hf_aem_frequency;
static int hf_aem_group_name;
static int hf_aem_guid;
/* static int hf_aem_height; */
static int hf_aem_iidc_format;
static int hf_aem_iidc_mode;
static int hf_aem_iidc_rate;
static int hf_aem_interface_name;
static int hf_aem_interface_name_string;
/* static int hf_aem_interlace; */
/* static int hf_aem_internal_id; */
static int hf_aem_jack_flags;
static int hf_aem_jack_id;
static int hf_aem_jack_name;
static int hf_aem_jack_name_string;
static int hf_aem_jack_type;
static int hf_aem_label_iec_60958_cnt;
static int hf_aem_label_mbla_cnt;
static int hf_aem_label_midi_cnt;
static int hf_aem_label_smpte_cnt;
static int hf_aem_length;
static int hf_aem_locale_identifier;
static int hf_aem_mapping_cluster_offset;
static int hf_aem_mapping_cluster_channel;
static int hf_aem_mapping_stream_channel;
static int hf_aem_mapping_stream_index;
static int hf_aem_mappings;
static int hf_aem_mappings_offset;
static int hf_aem_memory_object_type;
/* static int hf_aem_mf_height; */
/* static int hf_aem_mf_width; */
/* static int hf_aem_mfd_type; */
static int hf_aem_model_name_string;
static int hf_aem_avb_interface_flags;
static int hf_aem_avb_clock_identity;
static int hf_aem_avb_priority1;
static int hf_aem_avb_clock_class;
static int hf_aem_avb_offset_scaled_log_variance;
static int hf_aem_avb_clock_accuracy;
static int hf_aem_avb_priority2;
static int hf_aem_avb_domain_number;
static int hf_aem_avb_log_sync_interval;
/* static int hf_aem_avb_propagation_delay; */
static int hf_aem_nb_flag;
static int hf_aem_number_audio_maps;
/* static int hf_aem_number_destinations; */
static int hf_aem_number_matrices;
static int hf_aem_number_mixers;
static int hf_aem_number_of_clusters;
static int hf_aem_number_of_controls;
static int hf_aem_number_control_blocks;
static int hf_aem_number_of_external_input_ports;
static int hf_aem_number_of_external_output_ports;
static int hf_aem_number_of_formats;
static int hf_aem_number_of_internal_input_ports;
static int hf_aem_number_of_internal_output_ports;
static int hf_aem_number_of_mappings;
/* static int hf_aem_number_of_sources; */
static int hf_aem_number_of_stream_input_ports;
static int hf_aem_number_of_stream_output_ports;
static int hf_aem_number_of_strings;
static int hf_aem_number_of_values;
static int hf_aem_number_signal_selectors;
static int hf_aem_object_name;
static int hf_aem_localized_description;
/* static int hf_aem_oui24; */
static int hf_aem_path_latency;
static int hf_aem_port_flags;
static int hf_aem_pull_field;
static int hf_aem_sample_rates;
static int hf_aem_sample_rates_count;
static int hf_aem_sample_rates_offset;
static int hf_aem_serial_number;
static int hf_aem_sf;
static int hf_aem_sf_version;
static int hf_aem_sf_subtype;
static int hf_aem_signal_index;
static int hf_aem_signal_output;
static int hf_aem_signal_type;
static int hf_aem_signals_count;
static int hf_aem_signals_offset;
static int hf_aem_sources;
/* static int hf_aem_sources_offset; */
static int hf_aem_start_address;
/* static int hf_aem_stream_channels; */
static int hf_aem_stream_flags;
static int hf_aem_stream_format;
static int hf_aem_stream_formats;
/* static int hf_aem_stream_id; */
static int hf_aem_stream_name;
static int hf_aem_stream_name_string;
static int hf_aem_string;
static int hf_aem_string_ref;
static int hf_aem_target_descriptor_id;
static int hf_aem_target_descriptor_type;
static int hf_aem_unit;
static int hf_aem_unknown_descriptor;
/* static int hf_aem_value_offset; */
static int hf_aem_values_offset;
static int hf_aem_vendor_name_string;
static int hf_aem_video_mode;
/* static int hf_aem_width; */

/****************************************************************** */
/*     AVDECC Connection Management Protocol Data Unit (ACMPDU)     */
/* **************************************************************** */
static int hf_acmp_message_type;
static int hf_acmp_status_field;
static int hf_acmp_cd_length;
static int hf_acmp_stream_id;
static int hf_acmp_controller_guid;
static int hf_acmp_talker_guid;
static int hf_acmp_listener_guid;
static int hf_acmp_talker_unique_id;
static int hf_acmp_listener_unique_id;
static int hf_acmp_stream_dest_mac;
static int hf_acmp_connection_count;
static int hf_acmp_sequence_id;
static int hf_acmp_flags;
static int hf_acmp_vlan_id;

/* ACMP Flags (8.2.1.17) */
static int hf_acmp_flags_class_b;
static int hf_acmp_flags_fast_connect;
static int hf_acmp_flags_saved_state;
static int hf_acmp_flags_streaming_wait;
static int hf_acmp_flags_supports_encrypted;
static int hf_acmp_flags_encrypted_pdu;
static int hf_acmp_flags_talker_failed;

/* Initialize the subtree pointers */
static int ett_17221;
/* ADP */
static int ett_adp_ent_cap;
static int ett_adp_talk_cap;
static int ett_adp_list_cap;
static int ett_adp_cont_cap;
/* static int ett_adp_aud_format; */
static int ett_adp_samp_rates;
static int ett_adp_chan_format;
/* ACMP */
static int ett_acmp_flags;
/* AEM */
static int ett_aem_descriptor;
static int ett_aem_desc_counts;
static int ett_aem_sample_rates;
static int ett_aem_stream_flags;
static int ett_aem_clock_sources;
static int ett_aem_stream_formats;
static int ett_aem_jack_flags;
static int ett_aem_port_flags;
static int ett_aecp_get_avb_info_msrp_mappings;
static int ett_aem_clock_source_flags;
static int ett_aem_mappings;
static int ett_aem_ctrl_vals;
static int ett_aem_sources;
/* static int ett_aem_media_format; */
static int ett_aem_stream_format;

static int ett_aecp_descriptors;
static int ett_aecp_flags_32;

typedef struct {
   int hf;
   guint16 size;
} ctrl_ref_vals;

/* convenience function */
static inline ctrl_ref_vals
get_ctrl_ref_vals(guint16 ctrl_val_type)
{
   ctrl_ref_vals ret;

   switch(ctrl_val_type) {
      case AEM_CONTROL_LINEAR_INT8:
      case AEM_CONTROL_SELECTOR_INT8:
      case AEM_CONTROL_ARRAY_INT8:
         ret.hf = hf_aem_ctrl_int8;
         ret.size = 1;
         break;
      case AEM_CONTROL_LINEAR_UINT8:
      case AEM_CONTROL_SELECTOR_UINT8:
      case AEM_CONTROL_ARRAY_UINT8:
         ret.hf = hf_aem_ctrl_uint8;
         ret.size = 1;
         break;
      case AEM_CONTROL_LINEAR_INT16:
      case AEM_CONTROL_SELECTOR_INT16:
      case AEM_CONTROL_ARRAY_INT16:
         ret.hf = hf_aem_ctrl_int16;
         ret.size = 2;
         break;
      case AEM_CONTROL_LINEAR_UINT16:
      case AEM_CONTROL_SELECTOR_UINT16:
      case AEM_CONTROL_ARRAY_UINT16:
         ret.hf = hf_aem_ctrl_uint16;
         ret.size = 2;
         break;
      case AEM_CONTROL_LINEAR_INT32:
      case AEM_CONTROL_SELECTOR_INT32:
      case AEM_CONTROL_ARRAY_INT32:
         ret.hf = hf_aem_ctrl_int32;
         ret.size = 4;
         break;
      case AEM_CONTROL_LINEAR_UINT32:
      case AEM_CONTROL_SELECTOR_UINT32:
      case AEM_CONTROL_ARRAY_UINT32:
         ret.hf = hf_aem_ctrl_uint32;
         ret.size = 4;
         break;
      case AEM_CONTROL_LINEAR_FLOAT:
      case AEM_CONTROL_SELECTOR_FLOAT:
      case AEM_CONTROL_ARRAY_FLOAT:
         ret.hf = hf_aem_ctrl_float;
         ret.size = 4;
         break;
      case AEM_CONTROL_LINEAR_INT64:
      case AEM_CONTROL_SELECTOR_INT64:
      case AEM_CONTROL_ARRAY_INT64:
         ret.hf = hf_aem_ctrl_int64;
         ret.size = 8;
         break;
      case AEM_CONTROL_LINEAR_UINT64:
      case AEM_CONTROL_SELECTOR_UINT64:
      case AEM_CONTROL_ARRAY_UINT64:
         ret.hf = hf_aem_ctrl_uint64;
         ret.size = 8;
         break;
      case AEM_CONTROL_LINEAR_DOUBLE:
      case AEM_CONTROL_SELECTOR_DOUBLE:
      case AEM_CONTROL_ARRAY_DOUBLE:
         ret.hf = hf_aem_ctrl_double;
         ret.size = 8;
         break;
      case AEM_CONTROL_BODE_PLOT:
         ret.hf = -1;
         ret.size = 12;
         break;
      default:
         ret.size = 0;
         ret.hf = -1;
         break;
   }
   return ret;
}

static void
dissect_17221_stream_format(tvbuff_t *tvb, proto_tree *tree)
{
   proto_item *stream_tree;
   proto_item *stream_ti;
   guint8 version;
   guint8 subtype;
   guint8 sf;
   guint8 fmt;
   guint8 fdf_evt;

   /* subtree */
   stream_ti = proto_tree_add_item(tree, hf_aem_stream_format, tvb,
         0, 8, ENC_NA);
   stream_tree = proto_item_add_subtree(stream_ti, ett_aem_stream_format);

   /* get version */
   version = tvb_get_guint8(tvb, 0) & AEM_MASK_SF_VERSION;

   /* add the version to the tree */
   proto_tree_add_item(stream_tree, hf_aem_sf_version, tvb,
         AEM_OFFSET_SF_VERSION, 1, ENC_BIG_ENDIAN);

   if (version == 0) {       /* stream format version 0 */

      subtype = tvb_get_guint8(tvb, AEM_OFFSET_SF_SUBTYPE) & AEM_MASK_SF_SUBTYPE;

      proto_tree_add_item(stream_tree, hf_aem_sf_subtype, tvb,
            AEM_OFFSET_SF_SUBTYPE, 1, ENC_BIG_ENDIAN);

      switch(subtype) {
         case IEC_61883_IIDC_SUBTYPE:
            /* get sf */
            sf = tvb_get_guint8(tvb, 1) & AEM_MASK_SF;
            proto_tree_add_item(stream_tree, hf_aem_sf, tvb,
                  AEM_OFFSET_SF, 1, ENC_BIG_ENDIAN);

            if (sf == 0) { /* IIDC Stream Format */
               proto_tree_add_item(stream_tree, hf_aem_iidc_format, tvb,
                     AEM_OFFSET_IIDC_FORMAT, 1, ENC_BIG_ENDIAN);
               proto_tree_add_item(stream_tree, hf_aem_iidc_mode, tvb,
                     AEM_OFFSET_IIDC_MODE, 1, ENC_BIG_ENDIAN);
               proto_tree_add_item(stream_tree, hf_aem_iidc_rate, tvb,
                     AEM_OFFSET_IIDC_RATE, 1, ENC_BIG_ENDIAN);

            } else { /* 61883 Stream Format */
               proto_tree_add_item(stream_tree, hf_aem_fmt, tvb,
                     AEM_OFFSET_FMT, 1, ENC_BIG_ENDIAN);
               fmt = tvb_get_guint8(tvb, AEM_OFFSET_FMT) & 0x7F;
               if (fmt == 0x20) {       /* 61883-6 Stream Format */
                  proto_tree_add_item(stream_tree, hf_aem_fdf_evt, tvb,
                        AEM_OFFSET_FDF_EVT, 1, ENC_BIG_ENDIAN);
                  proto_tree_add_item(stream_tree, hf_aem_fdf_sfc, tvb,
                        AEM_OFFSET_FDF_SFC, 1, ENC_BIG_ENDIAN);
                  proto_tree_add_item(stream_tree, hf_aem_dbs, tvb,
                        AEM_OFFSET_DBS, 1, ENC_BIG_ENDIAN);

                  fdf_evt = tvb_get_guint8(tvb, AEM_OFFSET_FDF_EVT) & AEM_MASK_FDF_EVT;

                  proto_tree_add_item(stream_tree, hf_aem_b_flag, tvb,
                        AEM_OFFSET_B, 1, ENC_BIG_ENDIAN);
                  proto_tree_add_item(stream_tree, hf_aem_nb_flag, tvb,
                        AEM_OFFSET_NB, 1, ENC_BIG_ENDIAN);

                 if (fdf_evt == 0x00) { /* 61883-6 AM824 Stream Format  */
                    proto_tree_add_item(stream_tree, hf_aem_label_iec_60958_cnt, tvb,
                          AEM_OFFSET_LABEL_IEC_60958_CNT, 1, ENC_BIG_ENDIAN);
                    proto_tree_add_item(stream_tree, hf_aem_label_mbla_cnt, tvb,
                          AEM_OFFSET_LABEL_MBLA_CNT, 1, ENC_BIG_ENDIAN);
                    proto_tree_add_item(stream_tree, hf_aem_label_midi_cnt, tvb,
                          AEM_OFFSET_LABEL_MIDI_CNT, 1, ENC_BIG_ENDIAN);
                    proto_tree_add_item(stream_tree, hf_aem_label_smpte_cnt, tvb,
                          AEM_OFFSET_LABEL_SMPTE_CNT, 1, ENC_BIG_ENDIAN);
                  }
               } else if (fmt == 0x01) { /* 61883-8 Stream Format */
                  proto_tree_add_item(stream_tree, hf_aem_video_mode, tvb,
                        AEM_OFFSET_VIDEO_MODE, 1, ENC_BIG_ENDIAN);
                  proto_tree_add_item(stream_tree, hf_aem_compress_mode, tvb,
                        AEM_OFFSET_COMPRESS_MODE, 1, ENC_BIG_ENDIAN);
                  proto_tree_add_item(stream_tree, hf_aem_color_space, tvb,
                        AEM_OFFSET_COLOR_SPACE, 1, ENC_BIG_ENDIAN);
               }
            }
            break;
         case MMA_SUBTYPE:
            /* Defined by the MMA */
            break;
         case AVTP_AUDIO_SUBTYPE:
            /* Defined by the AVTP */
            break;
         case AVTP_VIDEO_SUBTYPE:
            /* Defined by the AVTP */
            break;
         case EXPERIMENTAL_SUBTYPE:
            /* used for experimental formats for development purposes only */
            break;
         default:
            /* unknown or unimplemented subtype */
            /* possibly a weather baloon, or swamp gas */
            break;
      }
   }
}

static void
dissect_17221_video_format(tvbuff_t *tvb, proto_tree *tree)
{
   /* TODO */
   proto_tree_add_item(tree, hf_aecp_video_format, tvb,
      0, 4, ENC_NA);
}

static void
dissect_17221_video_aspect_ratio(tvbuff_t *tvb, proto_tree *tree)
{
   /* TODO */
   proto_tree_add_item(tree, hf_aecp_video_format, tvb,
      0, 2, ENC_NA);
}

static void
dissect_17221_video_frame_size(tvbuff_t *tvb, proto_tree *tree)
{
   /* TODO */
   proto_tree_add_item(tree, hf_aecp_video_format, tvb,
      0, 4, ENC_NA);
}

static void
dissect_17221_sensor_format(tvbuff_t *tvb, proto_tree *tree)
{
   /* TODO */
   proto_tree_add_item(tree, hf_aecp_video_format, tvb,
      0, 8, ENC_NA);
}

#if 0 /* old, prior to D21. Fix for D21 video format */
static void
dissect_17221_media_format(tvbuff_t *tvb, proto_tree *tree)
{
   proto_item *media_tree;
   proto_item *media_ti;
   guint32 oui24;
   guint8  mfd_type;

   /* grab the oui24 and mfd_type */
   oui24 = tvb_get_ntoh24(tvb, 0);
   mfd_type = tvb_get_guint8(tvb, 3);

   /* subtree */
   media_ti = proto_tree_add_item(tree, hf_aecp_media_format, tvb,
         0, 16, ENC_NA);
   media_tree = proto_item_add_subtree(media_ti, ett_aem_media_format);

   /* standard media formats */
   if (oui24 == OUI24_STANDARD_MEDIA_FORMAT) {
      /* Standard Media Format Fields */
      proto_tree_add_item(media_tree, hf_aem_oui24, tvb,
            0, 3, ENC_BIG_ENDIAN);
      proto_tree_add_item(media_tree, hf_aem_mfd_type, tvb,
            AEM_OFFSET_MFD_TYPE, 1, ENC_BIG_ENDIAN);

      /* break down the standard media format types */
      switch(mfd_type) {
         case MFD_TYPE_VIDEO:
            proto_tree_add_item(media_tree, hf_aem_div, tvb,
                  AEM_OFFSET_DIV, 1, ENC_BIG_ENDIAN);
            proto_tree_add_item(media_tree, hf_aem_interlace, tvb,
                  AEM_OFFSET_INTERLACE, 2, ENC_BIG_ENDIAN);
            proto_tree_add_item(media_tree, hf_aem_channels, tvb,
                  AEM_OFFSET_CHANNELS, 1, ENC_BIG_ENDIAN);
            proto_tree_add_item(media_tree, hf_aem_color_format, tvb,
                  AEM_OFFSET_COLOR_FORMAT, 2, ENC_BIG_ENDIAN);
            proto_tree_add_item(media_tree, hf_aem_bpp, tvb,
                  AEM_OFFSET_BPP, 1, ENC_BIG_ENDIAN);
            proto_tree_add_item(media_tree, hf_aem_aspect_x, tvb,
                  AEM_OFFSET_ASPECT_X, 1, ENC_BIG_ENDIAN);
            proto_tree_add_item(media_tree, hf_aem_aspect_y, tvb,
                  AEM_OFFSET_ASPECT_Y, 1 ,ENC_BIG_ENDIAN);
            proto_tree_add_item(media_tree, hf_aem_frame_rate, tvb,
                  AEM_OFFSET_FRAME_RATE, 1, ENC_BIG_ENDIAN);
            proto_tree_add_item(media_tree, hf_aem_comp1, tvb,
                  AEM_OFFSET_COMP1, 2, ENC_BIG_ENDIAN);
            proto_tree_add_item(media_tree, hf_aem_comp2, tvb,
                  AEM_OFFSET_COMP2, 1, ENC_BIG_ENDIAN);
            proto_tree_add_item(media_tree, hf_aem_comp3, tvb,
                  AEM_OFFSET_COMP3, 2, ENC_BIG_ENDIAN);
            proto_tree_add_item(media_tree, hf_aem_comp4, tvb,
                  AEM_OFFSET_COMP4, 1, ENC_BIG_ENDIAN);
            proto_tree_add_item(media_tree, hf_aem_mf_width, tvb,
                  AEM_OFFSET_SVMF_WIDTH, 2, ENC_BIG_ENDIAN);
            proto_tree_add_item(media_tree, hf_aem_mf_height, tvb,
                  AEM_OFFSET_SVMF_HEIGHT, 2, ENC_BIG_ENDIAN);
            break;
         case MFD_TYPE_TRANSPORT_STRM:
            break;
         case MFD_TYPE_MIDI:
            break;
         case MFD_TYPE_TIME_CODE:
            break;
         case MFD_TYPE_CONTROL_PROTO:
            proto_tree_add_item(media_tree, hf_aem_cs_eui64, tvb,
                  AEM_OFFSET_CS_EUI64, 8, ENC_BIG_ENDIAN);
            break;
         default:
            /* unsupported MFD type */
            break;
      }
   } else { /* vendor specific media formats */

      /* these are not the media formats you are looking for */

   }
}
#endif

/* TODO following updates in Draft 18 and the pending Draft 19 this section will require major overhaul */
static void
dissect_17221_ctrl_val(tvbuff_t *tvb, proto_tree *tree, guint16 num_ctrl_vals, guint16 ctrl_val_type,
                       guint16 ctrl_offset)
{
   proto_item *ctrl_item;
   proto_item *ctrl_subtree;
   int i;
   guint32 bin_blob_size;
   ctrl_ref_vals ref;

   /* set up control values tree */
   ctrl_item = proto_tree_add_item(tree, hf_aem_ctrl_vals, tvb,
         0, 0, ENC_NA);
   ctrl_subtree = proto_item_add_subtree(ctrl_item, ett_aem_ctrl_vals);

   /* ctrl_val_type's are dissected below in this if/else block */
   /* for now only a few value types are in use, if I have time to come
      back to it I will add more fields to this but for now when viewing
      control_values you will need a copy of the spec handy to figure
      out what you are looking at, the get_ctrl_ref_vals function above
      will ideally be eliminated and this section will be a lot cleaner
      if/when that happens */

   ref = get_ctrl_ref_vals(ctrl_val_type);

   /* LINEAR TYPES */
   if (ctrl_val_type < 0xa) {
      for(i = 0; i < num_ctrl_vals; ++i) {
         proto_tree_add_item(ctrl_subtree, ref.hf, tvb,
               ctrl_offset, ref.size, ENC_BIG_ENDIAN);
         ctrl_offset += ref.size;
         proto_tree_add_item(ctrl_subtree, ref.hf, tvb,
               ctrl_offset, ref.size, ENC_BIG_ENDIAN);
         ctrl_offset += ref.size;
         proto_tree_add_item(ctrl_subtree, ref.hf, tvb,
               ctrl_offset, ref.size, ENC_BIG_ENDIAN);
         ctrl_offset += ref.size;
         proto_tree_add_item(ctrl_subtree, ref.hf, tvb,
               ctrl_offset, ref.size, ENC_BIG_ENDIAN);
         ctrl_offset += ref.size;
         proto_tree_add_item(ctrl_subtree, ref.hf, tvb,
               ctrl_offset, ref.size, ENC_BIG_ENDIAN);
         ctrl_offset += ref.size;
         proto_tree_add_item(ctrl_subtree, hf_aem_unit, tvb,
               ctrl_offset, 2, ENC_BIG_ENDIAN);
         ctrl_offset += 2;
         proto_tree_add_item(ctrl_subtree, hf_aem_string_ref, tvb,
               ctrl_offset, 2, ENC_BIG_ENDIAN);
         ctrl_offset += 2;
      }

      /* SELECTOR TYPES */
   } else if (ctrl_val_type > 0x9 && ctrl_val_type < 0x14) {
      proto_tree_add_item(ctrl_subtree, ref.hf, tvb,
            ctrl_offset, ref.size, ENC_BIG_ENDIAN);
      ctrl_offset += 2;
      proto_tree_add_item(ctrl_subtree, ref.hf, tvb,
            ctrl_offset, ref.size, ENC_BIG_ENDIAN);
      ctrl_offset += 2;
      for(i = 0; i < num_ctrl_vals; ++i) {
         proto_tree_add_item(ctrl_subtree, ref.hf, tvb,
               ctrl_offset, ref.size, ENC_BIG_ENDIAN);
         ctrl_offset += 2;
      }
      proto_tree_add_item(ctrl_subtree, hf_aem_unit, tvb,
            ctrl_offset, 2, ENC_BIG_ENDIAN);

      /* UTF8 STRING TYPE */
   } else if (ctrl_val_type == 0x14) {
      proto_tree_add_item(ctrl_subtree, hf_aem_string, tvb,
            ctrl_offset, -1, ENC_ASCII);

      /* BODE_PLOT TYPE */
   } else if (ctrl_val_type == 0x15) {
      for(i = 0; i < 12 + (num_ctrl_vals * 3); ++i) {
         proto_tree_add_item(ctrl_subtree, hf_aem_ctrl_float, tvb,
               ctrl_offset, 4, ENC_BIG_ENDIAN);
         ctrl_offset += 4;
      }

      /* ARRAY TYPES */
   } else if (ctrl_val_type > 0x15 && ctrl_val_type < 0x1f) {
      /* VENDOR CONTROL TYPE */
   } else if (ctrl_val_type == 0xfffe) {
      proto_tree_add_item(ctrl_subtree, hf_aem_guid, tvb,
            ctrl_offset, 8, ENC_BIG_ENDIAN);
      ctrl_offset += 8;
      bin_blob_size = tvb_get_ntohl(tvb, ctrl_offset);
      proto_tree_add_item(ctrl_subtree, hf_aem_blob_size, tvb,
            ctrl_offset, 4, ENC_BIG_ENDIAN);
      ctrl_offset += 4;
      proto_tree_add_item(ctrl_subtree, hf_aem_binary_blob, tvb,
            ctrl_offset, bin_blob_size, ENC_NA);
   }
}

/* dissect descriptors from the AVDECC Entity Model (AEM) */
/* this dissector is not registered */
static void
dissect_17221_aem(tvbuff_t *tvb, packet_info *pinfo _U_, proto_tree *tree)
{
   guint16 desc_type;
   guint16 ctrl_val_type;
   guint16 num_ctrl_vals;
   tvbuff_t *next_tvb;
   int i;

   /* Mr. Subtree and Mr. Counter will be used as the loop limit and
    * subtree object for arrays in the descriptors, rather than declaring
    * a separate variable for each possible case in the switch which uses
    * arrays.
    */
   proto_item *mr_subtree;
   proto_item *mr_item;
   guint32 mr_offset;
   guint16 mr_counter;

   gfloat frequency;
   gint freq_mult;
   gint base_freq;

   proto_item *aem_tree;
   /* used in creation of descriptor subtree */
   proto_item *desc_ti;


   /* get the type of this descriptor */
   desc_type = tvb_get_ntohs(tvb, 0);

   /* Load the descriptor type and id fields, add subtree */
   desc_ti = proto_tree_add_item(tree, hf_aecp_descriptor_type, tvb,
         AEM_OFFSET_DESCRIPTOR_TYPE, 2, ENC_BIG_ENDIAN);
   aem_tree = proto_item_add_subtree(desc_ti, ett_aem_descriptor);

   proto_tree_add_item(aem_tree, hf_aecp_descriptor_index, tvb,
         AEM_OFFSET_DESCRIPTOR_ID, 2, ENC_BIG_ENDIAN);

   /* Dissect descriptors based on type. Where possible multiple cases *
    * will fall through to the same code                               */
   switch(desc_type) {
      case AEM_DESCRIPTOR_ENTITY:
         proto_tree_add_item(aem_tree, hf_aem_entity_id, tvb,
               ENTITY_OFFSET_ENTITY_GUID, 8, ENC_BIG_ENDIAN);
         proto_tree_add_item(aem_tree, hf_aem_entity_model_id, tvb,
               ENTITY_OFFSET_ENTITY_MODEL_ID, 8, ENC_BIG_ENDIAN);

         proto_tree_add_item(aem_tree, hf_adp_entity_cap, tvb,
               ENTITY_OFFSET_ENTITY_CAPABILITIES, 4, ENC_BIG_ENDIAN);
         proto_tree_add_item(aem_tree, hf_adp_entity_cap_efu_mode, tvb,
               ENTITY_OFFSET_ENTITY_CAPABILITIES, 4, ENC_BIG_ENDIAN);
         proto_tree_add_item(aem_tree, hf_adp_entity_cap_address_access_supported, tvb,
               ENTITY_OFFSET_ENTITY_CAPABILITIES, 4, ENC_BIG_ENDIAN);
         proto_tree_add_item(aem_tree, hf_adp_entity_cap_gateway_entity, tvb,
               ENTITY_OFFSET_ENTITY_CAPABILITIES, 4, ENC_BIG_ENDIAN);
         proto_tree_add_item(aem_tree, hf_adp_entity_cap_aem_supported, tvb,
               ENTITY_OFFSET_ENTITY_CAPABILITIES, 4, ENC_BIG_ENDIAN);
         proto_tree_add_item(aem_tree, hf_adp_entity_cap_legacy_avc, tvb,
               ENTITY_OFFSET_ENTITY_CAPABILITIES, 4, ENC_BIG_ENDIAN);
         proto_tree_add_item(aem_tree, hf_adp_entity_cap_assoc_id_support, tvb,
               ENTITY_OFFSET_ENTITY_CAPABILITIES, 4, ENC_BIG_ENDIAN);
         proto_tree_add_item(aem_tree, hf_adp_entity_cap_assoc_id_valid, tvb,
               ENTITY_OFFSET_ENTITY_CAPABILITIES, 4, ENC_BIG_ENDIAN);
         proto_tree_add_item(aem_tree, hf_adp_entity_cap_vendor_unique, tvb,
               ENTITY_OFFSET_ENTITY_CAPABILITIES, 4, ENC_BIG_ENDIAN);
         proto_tree_add_item(aem_tree, hf_adp_entity_cap_class_a_supported, tvb,
               ENTITY_OFFSET_ENTITY_CAPABILITIES, 4, ENC_BIG_ENDIAN);
         proto_tree_add_item(aem_tree, hf_adp_entity_cap_class_b_supported, tvb,
               ENTITY_OFFSET_ENTITY_CAPABILITIES, 4, ENC_BIG_ENDIAN);
         proto_tree_add_item(aem_tree, hf_adp_entity_cap_gptp_supported, tvb,
               ENTITY_OFFSET_ENTITY_CAPABILITIES, 4, ENC_BIG_ENDIAN);

         proto_tree_add_item(aem_tree, hf_adp_talker_stream_srcs, tvb,
               ENTITY_OFFSET_TALKER_STREAM_SOURCES, 2, ENC_BIG_ENDIAN);

         proto_tree_add_item(aem_tree, hf_adp_talker_cap, tvb,
               ENTITY_OFFSET_TALKER_CAPABILITIES, 2, ENC_BIG_ENDIAN);
         proto_tree_add_item(aem_tree, hf_adp_talk_cap_implement, tvb,
               ENTITY_OFFSET_TALKER_CAPABILITIES, 2, ENC_BIG_ENDIAN);
         proto_tree_add_item(aem_tree, hf_adp_talk_cap_other_src, tvb,
               ENTITY_OFFSET_TALKER_CAPABILITIES, 2, ENC_BIG_ENDIAN);
         proto_tree_add_item(aem_tree, hf_adp_talk_cap_control_src, tvb,
               ENTITY_OFFSET_TALKER_CAPABILITIES, 2, ENC_BIG_ENDIAN);
         proto_tree_add_item(aem_tree, hf_adp_talk_cap_media_clk_src, tvb,
               ENTITY_OFFSET_TALKER_CAPABILITIES, 2, ENC_BIG_ENDIAN);
         proto_tree_add_item(aem_tree, hf_adp_talk_cap_smpte_src, tvb,
               ENTITY_OFFSET_TALKER_CAPABILITIES, 2, ENC_BIG_ENDIAN);
         proto_tree_add_item(aem_tree, hf_adp_talk_cap_midi_src, tvb,
               ENTITY_OFFSET_TALKER_CAPABILITIES, 2, ENC_BIG_ENDIAN);
         proto_tree_add_item(aem_tree, hf_adp_talk_cap_audio_src, tvb,
               ENTITY_OFFSET_TALKER_CAPABILITIES, 2, ENC_BIG_ENDIAN);
         proto_tree_add_item(aem_tree, hf_adp_talk_cap_video_src, tvb,
               ENTITY_OFFSET_TALKER_CAPABILITIES, 2, ENC_BIG_ENDIAN);

         proto_tree_add_item(aem_tree, hf_adp_listener_stream_sinks, tvb,
               ENTITY_OFFSET_LISTENER_STREAM_SINKS, 2, ENC_BIG_ENDIAN);

         proto_tree_add_item(aem_tree, hf_adp_listener_cap, tvb,
               ENTITY_OFFSET_LISTENER_CAPABILITIES, 2, ENC_BIG_ENDIAN);
         proto_tree_add_item(aem_tree, hf_adp_list_cap_implement, tvb,
               ENTITY_OFFSET_LISTENER_CAPABILITIES, 2, ENC_BIG_ENDIAN);
         proto_tree_add_item(aem_tree, hf_adp_list_cap_other_sink, tvb,
               ENTITY_OFFSET_LISTENER_CAPABILITIES, 2, ENC_BIG_ENDIAN);
         proto_tree_add_item(aem_tree, hf_adp_list_cap_control_sink, tvb,
               ENTITY_OFFSET_LISTENER_CAPABILITIES, 2, ENC_BIG_ENDIAN);
         proto_tree_add_item(aem_tree, hf_adp_list_cap_media_clk_sink, tvb,
               ENTITY_OFFSET_LISTENER_CAPABILITIES, 2, ENC_BIG_ENDIAN);
         proto_tree_add_item(aem_tree, hf_adp_list_cap_smpte_sink, tvb,
               ENTITY_OFFSET_LISTENER_CAPABILITIES, 2, ENC_BIG_ENDIAN);
         proto_tree_add_item(aem_tree, hf_adp_list_cap_midi_sink, tvb,
               ENTITY_OFFSET_LISTENER_CAPABILITIES, 2, ENC_BIG_ENDIAN);
         proto_tree_add_item(aem_tree, hf_adp_list_cap_audio_sink, tvb,
               ENTITY_OFFSET_LISTENER_CAPABILITIES, 2, ENC_BIG_ENDIAN);
         proto_tree_add_item(aem_tree, hf_adp_list_cap_video_sink, tvb,
               ENTITY_OFFSET_LISTENER_CAPABILITIES, 2, ENC_BIG_ENDIAN);

         proto_tree_add_item(aem_tree, hf_adp_controller_cap, tvb,
               ENTITY_OFFSET_CONTROLLER_CAPABILITIES, 4, ENC_BIG_ENDIAN);
         proto_tree_add_item(aem_tree, hf_adp_cont_cap_implement, tvb,
               ENTITY_OFFSET_CONTROLLER_CAPABILITIES, 4, ENC_BIG_ENDIAN);
         proto_tree_add_item(aem_tree, hf_adp_cont_cap_layer3_proxy, tvb,
               ENTITY_OFFSET_CONTROLLER_CAPABILITIES, 4, ENC_BIG_ENDIAN);

         proto_tree_add_item(aem_tree, hf_adp_avail_index, tvb,
               ENTITY_OFFSET_AVAILABLE_INDEX, 4, ENC_BIG_ENDIAN);
         proto_tree_add_item(aem_tree, hf_aecp_association_id, tvb,
               ENTITY_OFFSET_ASSOCIATION_ID, 8, ENC_BIG_ENDIAN);
         proto_tree_add_item(aem_tree, hf_aem_entity_name, tvb,
               ENTITY_OFFSET_ENTITY_NAME, 64, ENC_ASCII);
         proto_tree_add_item(aem_tree, hf_aem_vendor_name_string, tvb,
               ENTITY_OFFSET_VENDOR_NAME_STRING, 2, ENC_BIG_ENDIAN);
         proto_tree_add_item(aem_tree, hf_aem_model_name_string, tvb,
               ENTITY_OFFSET_MODEL_NAME_STRING, 2, ENC_BIG_ENDIAN);
         proto_tree_add_item(aem_tree, hf_aem_firmware_version, tvb,
               ENTITY_OFFSET_FIRMWARE_VERSION, 64, ENC_ASCII);
         proto_tree_add_item(aem_tree, hf_aem_group_name, tvb,
               ENTITY_OFFSET_GROUP_NAME, 64, ENC_ASCII);
         proto_tree_add_item(aem_tree, hf_aem_serial_number, tvb,
               ENTITY_OFFSET_SERIAL_NUMBER, 64, ENC_ASCII);
         proto_tree_add_item(aem_tree, hf_aem_configurations_count, tvb,
               ENTITY_OFFSET_CONFIGURATIONS_COUNT, 2, ENC_BIG_ENDIAN);
         proto_tree_add_item(aem_tree, hf_aem_current_configuration, tvb,
               ENTITY_OFFSET_CURRENT_CONFIGURATION, 2, ENC_BIG_ENDIAN);
         break;
      case AEM_DESCRIPTOR_CONFIGURATION:
         proto_tree_add_item(aem_tree, hf_aem_configuration_name, tvb,
               CONFIGURATION_OFFSET_OBJECT_NAME, 64, ENC_ASCII);
         proto_tree_add_item(aem_tree, hf_aem_configuration_name_string, tvb,
               CONFIGURATION_OFFSET_LOCALIZED_DESCRIPTION, 2, ENC_BIG_ENDIAN);
         proto_tree_add_item(aem_tree, hf_aem_descriptor_counts_count, tvb,
               CONFIGURATION_OFFSET_DESCRIPTOR_COUNTS_COUNT, 2, ENC_BIG_ENDIAN);

         /* set up subtree, counter, and offset for sample rates array */
         mr_item = proto_tree_add_item(aem_tree, hf_aem_descriptor_counts_offset, tvb,
               CONFIGURATION_OFFSET_DESCRIPTOR_COUNTS_OFFSET, 2, ENC_BIG_ENDIAN);
         mr_counter = tvb_get_ntohs(tvb, CONFIGURATION_OFFSET_DESCRIPTOR_COUNTS_COUNT);
         mr_subtree = proto_item_add_subtree(mr_item, ett_aem_desc_counts);
         mr_offset = CONFIGURATION_OFFSET_DESCRIPTOR_COUNTS;

         for(i = 0; i < mr_counter; ++i)
         {
            proto_tree_add_item(mr_subtree, hf_aecp_descriptor_type, tvb, mr_offset, 2, ENC_BIG_ENDIAN);
            mr_offset += 2;
            proto_tree_add_item(mr_subtree, hf_aem_count, tvb, mr_offset, 2, ENC_BIG_ENDIAN);
            mr_offset += 2;
         }
         break;
      case AEM_DESCRIPTOR_AUDIO_UNIT:
         proto_tree_add_item(aem_tree, hf_aem_object_name, tvb,
               AUDIO_UNIT_OFFSET_OBJECT_NAME, 64, ENC_ASCII);
         proto_tree_add_item(aem_tree, hf_aem_localized_description, tvb,
               AUDIO_UNIT_OFFSET_LOCALIZED_DESCRIPTION, 2, ENC_BIG_ENDIAN);

         /* D20 clock domain */
         proto_tree_add_item(aem_tree, hf_aem_clock_domain_id, tvb,
               AUDIO_UNIT_OFFSET_CLOCK_DOMAIN_INDEX, 2, ENC_BIG_ENDIAN);

         proto_tree_add_item(aem_tree, hf_aem_number_of_stream_input_ports, tvb,
               AUDIO_UNIT_OFFSET_NUMBER_OF_STREAM_INPUT_PORTS, 2, ENC_BIG_ENDIAN);
         proto_tree_add_item(aem_tree, hf_aem_base_stream_input_port, tvb,
               AUDIO_UNIT_OFFSET_BASE_STREAM_INPUT_PORT, 2, ENC_BIG_ENDIAN);
         proto_tree_add_item(aem_tree, hf_aem_number_of_stream_output_ports, tvb,
               AUDIO_UNIT_OFFSET_NUMBER_OF_STREAM_OUTPUT_PORTS, 2, ENC_BIG_ENDIAN);
         proto_tree_add_item(aem_tree, hf_aem_base_stream_output_port, tvb,
               AUDIO_UNIT_OFFSET_BASE_STREAM_OUTPUT_PORT, 2, ENC_BIG_ENDIAN);
         proto_tree_add_item(aem_tree, hf_aem_number_of_external_input_ports, tvb,
               AUDIO_UNIT_OFFSET_NUMBER_OF_EXTERNAL_INPUT_PORTS, 2, ENC_BIG_ENDIAN);
         proto_tree_add_item(aem_tree, hf_aem_base_external_input_port, tvb,
               AUDIO_UNIT_OFFSET_BASE_EXTERNAL_INPUT_PORT, 2, ENC_BIG_ENDIAN);
         proto_tree_add_item(aem_tree, hf_aem_number_of_external_output_ports, tvb,
               AUDIO_UNIT_OFFSET_NUMBER_OF_EXTERNAL_OUTPUT_PORTS, 2, ENC_BIG_ENDIAN);
         proto_tree_add_item(aem_tree, hf_aem_base_external_output_port, tvb,
               AUDIO_UNIT_OFFSET_BASE_EXTERNAL_OUTPUT_PORT, 2, ENC_BIG_ENDIAN);
         proto_tree_add_item(aem_tree, hf_aem_number_of_internal_input_ports, tvb,
               AUDIO_UNIT_OFFSET_NUMBER_OF_INTERNAL_INPUT_PORTS, 2, ENC_BIG_ENDIAN);
         proto_tree_add_item(aem_tree, hf_aem_base_internal_input_port, tvb,
               AUDIO_UNIT_OFFSET_BASE_INTERNAL_INPUT_PORT, 2, ENC_BIG_ENDIAN);
         proto_tree_add_item(aem_tree, hf_aem_number_of_internal_output_ports, tvb,
               AUDIO_UNIT_OFFSET_NUMBER_OF_INTERNAL_OUTPUT_PORTS, 2, ENC_BIG_ENDIAN);
         proto_tree_add_item(aem_tree, hf_aem_base_internal_output_port, tvb,
               AUDIO_UNIT_OFFSET_BASE_INTERNAL_OUTPUT_PORT, 2, ENC_BIG_ENDIAN);
         proto_tree_add_item(aem_tree, hf_aem_number_of_controls, tvb,
               AUDIO_UNIT_OFFSET_NUMBER_OF_CONTROLS, 2, ENC_BIG_ENDIAN);
         proto_tree_add_item(aem_tree, hf_aem_base_control, tvb,
               AUDIO_UNIT_OFFSET_BASE_CONTROL, 2, ENC_BIG_ENDIAN);
         proto_tree_add_item(aem_tree, hf_aem_number_signal_selectors, tvb,
               AUDIO_UNIT_OFFSET_NUMBER_SIGNAL_SELECTORS, 2, ENC_BIG_ENDIAN);
         proto_tree_add_item(aem_tree, hf_aem_base_signal_selector, tvb,
               AUDIO_UNIT_OFFSET_BASE_SIGNAL_SELECTOR, 2, ENC_BIG_ENDIAN);
         proto_tree_add_item(aem_tree, hf_aem_number_mixers, tvb,
               AUDIO_UNIT_OFFSET_NUMBER_MIXERS, 2, ENC_BIG_ENDIAN);
         proto_tree_add_item(aem_tree, hf_aem_base_mixer, tvb,
               AUDIO_UNIT_OFFSET_BASE_MIXER, 2, ENC_BIG_ENDIAN);
         proto_tree_add_item(aem_tree, hf_aem_number_matrices, tvb,
               AUDIO_UNIT_OFFSET_NUMBER_MATRICES, 2, ENC_BIG_ENDIAN);
         proto_tree_add_item(aem_tree, hf_aem_base_matrix, tvb,
               AUDIO_UNIT_OFFSET_BASE_MATRIX, 2, ENC_BIG_ENDIAN);
         proto_tree_add_item(aem_tree, hf_aem_number_splitters, tvb,
               AUDIO_UNIT_OFFSET_NUMBER_SPLITTERS, 2, ENC_BIG_ENDIAN);
         proto_tree_add_item(aem_tree, hf_aem_base_splitter, tvb,
               AUDIO_UNIT_OFFSET_BASE_SPLITTER, 2, ENC_BIG_ENDIAN);
         proto_tree_add_item(aem_tree, hf_aem_number_combiners, tvb,
               AUDIO_UNIT_OFFSET_NUMBER_COMBINERS, 2, ENC_BIG_ENDIAN);
         proto_tree_add_item(aem_tree, hf_aem_base_combiner, tvb,
               AUDIO_UNIT_OFFSET_BASE_COMBINER, 2, ENC_BIG_ENDIAN);
         proto_tree_add_item(aem_tree, hf_aem_number_demultiplexers, tvb,
               AUDIO_UNIT_OFFSET_NUMBER_DEMULTIPLEXERS, 2, ENC_BIG_ENDIAN);
         proto_tree_add_item(aem_tree, hf_aem_base_demultiplexer, tvb,
               AUDIO_UNIT_OFFSET_BASE_DEMULTIPLEXER, 2, ENC_BIG_ENDIAN);
         proto_tree_add_item(aem_tree, hf_aem_number_multiplexers, tvb,
               AUDIO_UNIT_OFFSET_NUMBER_MULTIPLEXERS, 2, ENC_BIG_ENDIAN);
         proto_tree_add_item(aem_tree, hf_aem_base_multiplexer, tvb,
               AUDIO_UNIT_OFFSET_BASE_MULTIPLEXER, 2, ENC_BIG_ENDIAN);
         proto_tree_add_item(aem_tree, hf_aem_number_transcoders, tvb,
               AUDIO_UNIT_OFFSET_NUMBER_TRANSCODERS, 2, ENC_BIG_ENDIAN);
         proto_tree_add_item(aem_tree, hf_aem_base_transcoder, tvb,
               AUDIO_UNIT_OFFSET_BASE_TRANSCODER, 2, ENC_BIG_ENDIAN);


         proto_tree_add_item(aem_tree, hf_aem_number_control_blocks, tvb,
               AUDIO_UNIT_OFFSET_NUMBER_CONTROL_BLOCKS, 2, ENC_BIG_ENDIAN);
         proto_tree_add_item(aem_tree, hf_aem_base_control_block, tvb,
               AUDIO_UNIT_OFFSET_BASE_CONTROL_BLOCK, 2, ENC_BIG_ENDIAN);


         proto_tree_add_item(aem_tree, hf_aem_current_sample_rate, tvb,
               AUDIO_UNIT_OFFSET_CURRENT_SAMPLE_RATE, 4, ENC_BIG_ENDIAN);

         proto_tree_add_item(aem_tree, hf_aem_sample_rates_offset, tvb,
               AUDIO_UNIT_OFFSET_SAMPLE_RATES_OFFSET, 2, ENC_BIG_ENDIAN);
         proto_tree_add_item(aem_tree, hf_aem_sample_rates_count, tvb,
               AUDIO_UNIT_OFFSET_SAMPLE_RATES_COUNT, 2, ENC_BIG_ENDIAN);

         /* set up subtree, counter, and offset for sample rates array */
         mr_offset = tvb_get_ntohs(tvb, AUDIO_UNIT_OFFSET_SAMPLE_RATES_OFFSET);
         mr_counter = tvb_get_ntohs(tvb, AUDIO_UNIT_OFFSET_SAMPLE_RATES_COUNT);
         mr_item = proto_tree_add_item(aem_tree, hf_aem_sample_rates, tvb,
               mr_offset, mr_counter * 4, ENC_NA);
         mr_subtree = proto_item_add_subtree(mr_item, ett_aem_sample_rates);

         /* loop to get the array values */
         for(i = 0; i < mr_counter; ++i) {
            proto_tree_add_item(mr_subtree, hf_aem_pull_field, tvb,
                  mr_offset, 1, ENC_BIG_ENDIAN);
            proto_tree_add_item(mr_subtree, hf_aem_base_frequency, tvb,
                  mr_offset, 4, ENC_BIG_ENDIAN);
            base_freq = tvb_get_ntohl(tvb, mr_offset);
            freq_mult = base_freq;

            freq_mult &= 0xe0000000;
            freq_mult = freq_mult >> 29;
            base_freq &= 0x1fffffff;
            /* replace this with something not horrible */
            frequency = freq_mult == 0 ? 1 :
               freq_mult == 1 ? 1 / (gfloat)1.001 :
               freq_mult == 2 ? (gfloat)1.001 :
               freq_mult == 3 ? 24 / 25 :
               freq_mult == 4 ? 54 / 24 : 0;

            frequency *= base_freq;
            proto_tree_add_float(mr_subtree, hf_aem_frequency, tvb, mr_offset, 4, frequency);

            mr_offset += 4;
         }
         break;
      case AEM_DESCRIPTOR_VIDEO_UNIT:
      case AEM_DESCRIPTOR_SENSOR_UNIT:
         proto_tree_add_item(aem_tree, hf_aem_object_name, tvb,
               VIDEO_UNIT_OFFSET_OBJECT_NAME, 64, ENC_ASCII);
         proto_tree_add_item(aem_tree, hf_aem_localized_description, tvb,
               VIDEO_UNIT_OFFSET_LOCALIZED_DESCRIPTION, 2, ENC_BIG_ENDIAN);

         /* D20 clock domain */
         proto_tree_add_item(aem_tree, hf_aem_clock_domain_id, tvb,
               AUDIO_UNIT_OFFSET_CLOCK_DOMAIN_INDEX, 2, ENC_BIG_ENDIAN);

         proto_tree_add_item(aem_tree, hf_aem_number_of_stream_input_ports, tvb,
               VIDEO_UNIT_OFFSET_NUMBER_OF_STREAM_INPUT_PORTS, 2, ENC_BIG_ENDIAN);
         proto_tree_add_item(aem_tree, hf_aem_base_stream_input_port, tvb,
               VIDEO_UNIT_OFFSET_BASE_STREAM_INPUT_PORT, 2, ENC_BIG_ENDIAN);
         proto_tree_add_item(aem_tree, hf_aem_number_of_stream_output_ports, tvb,
               VIDEO_UNIT_OFFSET_NUMBER_OF_STREAM_OUTPUT_PORTS, 2, ENC_BIG_ENDIAN);
         proto_tree_add_item(aem_tree, hf_aem_base_stream_output_port, tvb,
               VIDEO_UNIT_OFFSET_BASE_STREAM_OUTPUT_PORT, 2, ENC_BIG_ENDIAN);
         proto_tree_add_item(aem_tree, hf_aem_number_of_external_input_ports, tvb,
               VIDEO_UNIT_OFFSET_NUMBER_OF_EXTERNAL_INPUT_PORTS, 2, ENC_BIG_ENDIAN);
         proto_tree_add_item(aem_tree, hf_aem_base_external_input_port, tvb,
               VIDEO_UNIT_OFFSET_BASE_EXTERNAL_INPUT_PORT, 2, ENC_BIG_ENDIAN);
         proto_tree_add_item(aem_tree, hf_aem_number_of_external_output_ports, tvb,
               VIDEO_UNIT_OFFSET_NUMBER_OF_EXTERNAL_OUTPUT_PORTS, 2, ENC_BIG_ENDIAN);
         proto_tree_add_item(aem_tree, hf_aem_base_external_output_port, tvb,
               VIDEO_UNIT_OFFSET_BASE_EXTERNAL_OUTPUT_PORT, 2, ENC_BIG_ENDIAN);
         proto_tree_add_item(aem_tree, hf_aem_number_of_internal_input_ports, tvb,
               VIDEO_UNIT_OFFSET_NUMBER_OF_INTERNAL_INPUT_PORTS, 2, ENC_BIG_ENDIAN);
         proto_tree_add_item(aem_tree, hf_aem_base_internal_input_port, tvb,
               VIDEO_UNIT_OFFSET_BASE_INTERNAL_INPUT_PORT, 2, ENC_BIG_ENDIAN);
         proto_tree_add_item(aem_tree, hf_aem_number_of_internal_output_ports, tvb,
               VIDEO_UNIT_OFFSET_NUMBER_OF_INTERNAL_OUTPUT_PORTS, 2, ENC_BIG_ENDIAN);
         proto_tree_add_item(aem_tree, hf_aem_base_internal_output_port, tvb,
               VIDEO_UNIT_OFFSET_BASE_INTERNAL_OUTPUT_PORT, 2, ENC_BIG_ENDIAN);
         proto_tree_add_item(aem_tree, hf_aem_number_of_controls, tvb,
               VIDEO_UNIT_OFFSET_NUMBER_OF_CONTROLS, 2, ENC_BIG_ENDIAN);
         proto_tree_add_item(aem_tree, hf_aem_base_control, tvb,
               VIDEO_UNIT_OFFSET_BASE_CONTROL, 2, ENC_BIG_ENDIAN);
         proto_tree_add_item(aem_tree, hf_aem_number_signal_selectors, tvb,
               VIDEO_UNIT_OFFSET_NUMBER_SIGNAL_SELECTORS, 2, ENC_BIG_ENDIAN);
         proto_tree_add_item(aem_tree, hf_aem_base_signal_selector, tvb,
               VIDEO_UNIT_OFFSET_BASE_SIGNAL_SELECTOR, 2, ENC_BIG_ENDIAN);
         proto_tree_add_item(aem_tree, hf_aem_number_mixers, tvb,
               VIDEO_UNIT_OFFSET_NUMBER_MIXERS, 2, ENC_BIG_ENDIAN);
         proto_tree_add_item(aem_tree, hf_aem_base_mixer, tvb,
               VIDEO_UNIT_OFFSET_BASE_MIXER, 2, ENC_BIG_ENDIAN);
         proto_tree_add_item(aem_tree, hf_aem_number_matrices, tvb,
               VIDEO_UNIT_OFFSET_NUMBER_MATRICES, 2, ENC_BIG_ENDIAN);
         proto_tree_add_item(aem_tree, hf_aem_base_matrix, tvb,
               VIDEO_UNIT_OFFSET_BASE_MATRIX, 2, ENC_BIG_ENDIAN);
         proto_tree_add_item(aem_tree, hf_aem_number_splitters, tvb,
               VIDEO_UNIT_OFFSET_NUMBER_SPLITTERS, 2, ENC_BIG_ENDIAN);
         proto_tree_add_item(aem_tree, hf_aem_base_splitter, tvb,
               VIDEO_UNIT_OFFSET_BASE_SPLITTER, 2, ENC_BIG_ENDIAN);
         proto_tree_add_item(aem_tree, hf_aem_number_combiners, tvb,
               VIDEO_UNIT_OFFSET_NUMBER_COMBINERS, 2, ENC_BIG_ENDIAN);
         proto_tree_add_item(aem_tree, hf_aem_base_combiner, tvb,
               VIDEO_UNIT_OFFSET_BASE_COMBINER, 2, ENC_BIG_ENDIAN);
         proto_tree_add_item(aem_tree, hf_aem_number_demultiplexers, tvb,
               VIDEO_UNIT_OFFSET_NUMBER_DEMULTIPLEXERS, 2, ENC_BIG_ENDIAN);
         proto_tree_add_item(aem_tree, hf_aem_base_demultiplexer, tvb,
               VIDEO_UNIT_OFFSET_BASE_DEMULTIPLEXER, 2, ENC_BIG_ENDIAN);
         proto_tree_add_item(aem_tree, hf_aem_number_multiplexers, tvb,
               VIDEO_UNIT_OFFSET_NUMBER_MULTIPLEXERS, 2, ENC_BIG_ENDIAN);
         proto_tree_add_item(aem_tree, hf_aem_base_multiplexer, tvb,
               VIDEO_UNIT_OFFSET_BASE_MULTIPLEXER, 2, ENC_BIG_ENDIAN);
         proto_tree_add_item(aem_tree, hf_aem_number_transcoders, tvb,
               VIDEO_UNIT_OFFSET_NUMBER_TRANSCODERS, 2, ENC_BIG_ENDIAN);
         proto_tree_add_item(aem_tree, hf_aem_base_transcoder, tvb,
               VIDEO_UNIT_OFFSET_BASE_TRANSCODER, 2, ENC_BIG_ENDIAN);


         proto_tree_add_item(aem_tree, hf_aem_number_control_blocks, tvb,
               VIDEO_UNIT_OFFSET_NUMBER_CONTROL_BLOCKS, 2, ENC_BIG_ENDIAN);
         proto_tree_add_item(aem_tree, hf_aem_base_control_block, tvb,
               VIDEO_UNIT_OFFSET_BASE_CONTROL_BLOCK, 2, ENC_BIG_ENDIAN);
         break;
      case AEM_DESCRIPTOR_STREAM_INPUT:
      case AEM_DESCRIPTOR_STREAM_OUTPUT:
         proto_tree_add_item(aem_tree, hf_aem_stream_name, tvb,
               STREAM_OFFSET_OBJECT_NAME, 64, ENC_ASCII);
         proto_tree_add_item(aem_tree, hf_aem_stream_name_string, tvb,
               STREAM_OFFSET_LOCALIZED_DESCRIPTION, 2, ENC_BIG_ENDIAN);

         /* D20 clock domain */
         proto_tree_add_item(aem_tree, hf_aem_clock_domain_id, tvb,
               STREAM_OFFSET_CLOCK_DOMAIN_INDEX, 2, ENC_BIG_ENDIAN);

         /* set up a flags subtree */
         mr_item = proto_tree_add_item(aem_tree, hf_aem_stream_flags, tvb,
               STREAM_OFFSET_STREAM_FLAGS, 2, ENC_BIG_ENDIAN);
         mr_subtree = proto_item_add_subtree(mr_item, ett_aem_stream_flags);
         /* add flags to new subtree */
         proto_tree_add_item(mr_subtree, hf_aem_flags_clock_sync_source, tvb,
               STREAM_OFFSET_STREAM_FLAGS, 2, ENC_BIG_ENDIAN);
         proto_tree_add_item(mr_subtree, hf_aem_flags_class_a, tvb,
               STREAM_OFFSET_STREAM_FLAGS, 2, ENC_BIG_ENDIAN);
         proto_tree_add_item(mr_subtree, hf_aem_flags_class_b, tvb,
               STREAM_OFFSET_STREAM_FLAGS, 2, ENC_BIG_ENDIAN);
         /* done adding flags, continue with fields */
         /* stream format dissection */
         next_tvb = tvb_new_subset_length(tvb, STREAM_OFFSET_CURRENT_FORMAT, 8);
         dissect_17221_stream_format(next_tvb, aem_tree);

         proto_tree_add_item(aem_tree, hf_aem_formats_offset, tvb,
               STREAM_OFFSET_FORMATS_OFFSET, 2, ENC_BIG_ENDIAN);
         proto_tree_add_item(aem_tree, hf_aem_number_of_formats, tvb,
               STREAM_OFFSET_NUMBER_OF_FORMATS, 2, ENC_BIG_ENDIAN);
         proto_tree_add_item(aem_tree, hf_aem_backup_talker_guid_0, tvb,
               STREAM_OFFSET_BACKUP_TALKER_GUID_0, 8, ENC_BIG_ENDIAN);
         proto_tree_add_item(aem_tree, hf_aem_backup_talker_unique_0, tvb,
               STREAM_OFFSET_BACKUP_TALKER_UNIQUE_0, 2, ENC_BIG_ENDIAN);
         proto_tree_add_item(aem_tree, hf_aem_backup_talker_guid_1, tvb,
               STREAM_OFFSET_BACKUP_TALKER_GUID_1, 8, ENC_BIG_ENDIAN);
         proto_tree_add_item(aem_tree, hf_aem_backup_talker_unique_1, tvb,
               STREAM_OFFSET_BACKUP_TALKER_UNIQUE_1, 2, ENC_BIG_ENDIAN);
         proto_tree_add_item(aem_tree, hf_aem_backup_talker_guid_2, tvb,
               STREAM_OFFSET_BACKUP_TALKER_GUID_2, 8, ENC_BIG_ENDIAN);
         proto_tree_add_item(aem_tree, hf_aem_backup_talker_unique_2, tvb,
               STREAM_OFFSET_BACKUP_TALKER_UNIQUE_2, 2, ENC_BIG_ENDIAN);
         proto_tree_add_item(aem_tree, hf_aem_backedup_talker_guid, tvb,
               STREAM_OFFSET_BACKEDUP_TALKER_GUID, 8, ENC_BIG_ENDIAN);
         proto_tree_add_item(aem_tree, hf_aem_backedup_talker_unique, tvb,
               STREAM_OFFSET_BACKEDUP_TALKER_UNIQUE_ID, 2, ENC_BIG_ENDIAN);
         proto_tree_add_item(aem_tree, hf_aem_avb_interface_id, tvb,
               STREAM_OFFSET_AVB_INTERFACE_ID, 2, ENC_BIG_ENDIAN);

         proto_tree_add_item(aem_tree, hf_aem_buffer_length, tvb,
               STREAM_OFFSET_BUFFER_LENGTH, 2, ENC_BIG_ENDIAN);

         mr_offset = tvb_get_ntohs(tvb, STREAM_OFFSET_FORMATS_OFFSET);
         mr_counter = tvb_get_ntohs(tvb, STREAM_OFFSET_NUMBER_OF_FORMATS);

         /* set up subtree, counter, and offset for formats array */
         mr_item = proto_tree_add_item(aem_tree, hf_aem_stream_formats, tvb,
               mr_offset, mr_counter * 8, ENC_NA);
         mr_subtree = proto_item_add_subtree(mr_item, ett_aem_stream_formats);

         for(i = 0; i < mr_counter; ++i) {
            next_tvb = tvb_new_subset_length(tvb, mr_offset, 8);
            dissect_17221_stream_format(next_tvb, mr_subtree);
            mr_offset += 8;
         }
         break;
      case AEM_DESCRIPTOR_JACK_INPUT:
      case AEM_DESCRIPTOR_JACK_OUTPUT:
         proto_tree_add_item(aem_tree, hf_aem_jack_name, tvb,
               JACK_OFFSET_OBJECT_NAME, 64, ENC_ASCII);
         proto_tree_add_item(aem_tree, hf_aem_jack_name_string, tvb,
               JACK_OFFSET_LOCALIZED_DESCRIPTION, 2, ENC_BIG_ENDIAN);

         /* set up jack flags subtree */
         mr_item = proto_tree_add_item(aem_tree, hf_aem_jack_flags, tvb,
               JACK_OFFSET_JACK_FLAGS, 2, ENC_BIG_ENDIAN);
         mr_subtree = proto_item_add_subtree(mr_item, ett_aem_jack_flags);
         proto_tree_add_item(mr_subtree, hf_aem_flags_captive, tvb,
               JACK_OFFSET_JACK_FLAGS, 2, ENC_BIG_ENDIAN);
         proto_tree_add_item(mr_subtree, hf_aem_flags_clock_sync_source, tvb,
               JACK_OFFSET_JACK_FLAGS, 2, ENC_BIG_ENDIAN);
         /* end jack flags subtree */

         proto_tree_add_item(aem_tree, hf_aem_jack_type, tvb,
               JACK_OFFSET_JACK_TYPE, 2, ENC_BIG_ENDIAN);

         proto_tree_add_item(aem_tree, hf_aem_number_of_controls, tvb,
               JACK_OFFSET_JACK_NUMBER_OF_CONTROLS, 2, ENC_BIG_ENDIAN);
         proto_tree_add_item(aem_tree, hf_aem_base_control, tvb,
               JACK_OFFSET_JACK_BASE_CONTROL, 2, ENC_BIG_ENDIAN);
         break;
      case AEM_DESCRIPTOR_STREAM_PORT_INPUT:
      case AEM_DESCRIPTOR_STREAM_PORT_OUTPUT:
         /* D20 clock domain */
         proto_tree_add_item(aem_tree, hf_aem_clock_domain_id, tvb,
               STREAM_PORT_OFFSET_CLOCK_DOMAIN_INDEX, 2, ENC_BIG_ENDIAN);

         /* set up port_flags subtree */
         mr_item = proto_tree_add_item(aem_tree, hf_aem_port_flags, tvb,
               STREAM_PORT_OFFSET_PORT_FLAGS, 2, ENC_BIG_ENDIAN);
         mr_subtree = proto_item_add_subtree(mr_item, ett_aem_port_flags);
         proto_tree_add_item(mr_subtree, hf_aem_flags_clock_sync_source, tvb,
               STREAM_PORT_OFFSET_PORT_FLAGS, 2, ENC_BIG_ENDIAN);
         proto_tree_add_item(mr_subtree, hf_aem_flags_async_sample_rate_conv, tvb,
               STREAM_PORT_OFFSET_PORT_FLAGS, 2, ENC_BIG_ENDIAN);
         proto_tree_add_item(mr_subtree, hf_aem_flags_sync_sample_rate_conv, tvb,
               STREAM_PORT_OFFSET_PORT_FLAGS, 2, ENC_BIG_ENDIAN);
         /* end port_flags subtree */

         proto_tree_add_item(aem_tree, hf_aem_number_of_controls, tvb,
               STREAM_PORT_OFFSET_NUMBER_OF_CONTROLS, 2, ENC_BIG_ENDIAN);
         proto_tree_add_item(aem_tree, hf_aem_base_control, tvb,
               STREAM_PORT_OFFSET_BASE_CONTROL, 2, ENC_BIG_ENDIAN);
         proto_tree_add_item(aem_tree, hf_aem_number_of_clusters, tvb,
               STREAM_PORT_OFFSET_NUMBER_OF_CLUSTERS, 2, ENC_BIG_ENDIAN);
         proto_tree_add_item(aem_tree, hf_aem_base_cluster, tvb,
               STREAM_PORT_OFFSET_BASE_CLUSTER, 2, ENC_BIG_ENDIAN);
         proto_tree_add_item(aem_tree, hf_aem_number_audio_maps, tvb,
               STREAM_PORT_OFFSET_NUMBER_AUDIO_MAPS, 2, ENC_BIG_ENDIAN);
         proto_tree_add_item(aem_tree, hf_aem_base_audio_map, tvb,
               STREAM_PORT_OFFSET_BASE_AUDIO_MAP, 2, ENC_BIG_ENDIAN);
         break;
      case AEM_DESCRIPTOR_EXTERNAL_PORT_INPUT:
      case AEM_DESCRIPTOR_EXTERNAL_PORT_OUTPUT:
         proto_tree_add_item(aem_tree, hf_aem_clock_domain_id, tvb,
               EXTERNAL_PORT_OFFSET_CLOCK_DOMAIN_INDEX, 2, ENC_BIG_ENDIAN);

         /* set up port_flags subtree */
         mr_item = proto_tree_add_item(aem_tree, hf_aem_port_flags, tvb,
               EXTERNAL_PORT_OFFSET_PORT_FLAGS, 2, ENC_BIG_ENDIAN);
         mr_subtree = proto_item_add_subtree(mr_item, ett_aem_port_flags);
         proto_tree_add_item(mr_subtree, hf_aem_flags_clock_sync_source, tvb,
               EXTERNAL_PORT_OFFSET_PORT_FLAGS, 2, ENC_BIG_ENDIAN);
         proto_tree_add_item(mr_subtree, hf_aem_flags_async_sample_rate_conv, tvb,
               EXTERNAL_PORT_OFFSET_PORT_FLAGS, 2, ENC_BIG_ENDIAN);
         proto_tree_add_item(mr_subtree, hf_aem_flags_sync_sample_rate_conv, tvb,
               EXTERNAL_PORT_OFFSET_PORT_FLAGS, 2, ENC_BIG_ENDIAN);
         /* end port_flags subtree */

         proto_tree_add_item(aem_tree, hf_aem_number_of_controls, tvb,
               EXTERNAL_PORT_OFFSET_NUMBER_OF_CONTROLS, 2, ENC_BIG_ENDIAN);
         proto_tree_add_item(aem_tree, hf_aem_base_control, tvb,
               EXTERNAL_PORT_OFFSET_BASE_CONTROL, 2, ENC_BIG_ENDIAN);
         proto_tree_add_item(aem_tree, hf_aem_signal_type, tvb,
               EXTERNAL_PORT_OFFSET_SIGNAL_TYPE, 2, ENC_BIG_ENDIAN);
         proto_tree_add_item(aem_tree, hf_aem_signal_index, tvb,
               EXTERNAL_PORT_OFFSET_SIGNAL_INDEX, 2, ENC_BIG_ENDIAN);
         proto_tree_add_item(aem_tree, hf_aem_signal_output, tvb,
               EXTERNAL_PORT_OFFSET_SIGNAL_OUTPUT, 2, ENC_BIG_ENDIAN);
         proto_tree_add_item(aem_tree, hf_aem_block_latency, tvb,
               EXTERNAL_PORT_OFFSET_BLOCK_LATENCY, 4, ENC_BIG_ENDIAN);
         proto_tree_add_item(aem_tree, hf_aem_jack_id, tvb,
               EXTERNAL_PORT_OFFSET_JACK_INDEX, 2, ENC_BIG_ENDIAN);
         break;
      case AEM_DESCRIPTOR_INTERNAL_PORT_INPUT:
      case AEM_DESCRIPTOR_INTERNAL_PORT_OUTPUT:
         proto_tree_add_item(aem_tree, hf_aem_clock_domain_id, tvb,
               INTERNAL_PORT_OFFSET_CLOCK_DOMAIN_INDEX, 2, ENC_BIG_ENDIAN);

         /* set up port_flags subtree */
         mr_item = proto_tree_add_item(aem_tree, hf_aem_port_flags, tvb,
               INTERNAL_PORT_OFFSET_PORT_FLAGS, 2, ENC_BIG_ENDIAN);
         mr_subtree = proto_item_add_subtree(mr_item, ett_aem_port_flags);
         proto_tree_add_item(mr_subtree, hf_aem_flags_clock_sync_source, tvb,
               INTERNAL_PORT_OFFSET_PORT_FLAGS, 2, ENC_BIG_ENDIAN);
         proto_tree_add_item(mr_subtree, hf_aem_flags_async_sample_rate_conv, tvb,
               INTERNAL_PORT_OFFSET_PORT_FLAGS, 2, ENC_BIG_ENDIAN);
         proto_tree_add_item(mr_subtree, hf_aem_flags_sync_sample_rate_conv, tvb,
               INTERNAL_PORT_OFFSET_PORT_FLAGS, 2, ENC_BIG_ENDIAN);
         /* end port_flags subtree */

         proto_tree_add_item(aem_tree, hf_aem_number_of_controls, tvb,
               INTERNAL_PORT_OFFSET_NUMBER_OF_CONTROLS, 2, ENC_BIG_ENDIAN);
         proto_tree_add_item(aem_tree, hf_aem_base_control, tvb,
               INTERNAL_PORT_OFFSET_BASE_CONTROL, 2, ENC_BIG_ENDIAN);
         proto_tree_add_item(aem_tree, hf_aem_signal_type, tvb,
               INTERNAL_PORT_OFFSET_SIGNAL_TYPE, 2, ENC_BIG_ENDIAN);
         proto_tree_add_item(aem_tree, hf_aem_signal_index, tvb,
               INTERNAL_PORT_OFFSET_SIGNAL_INDEX, 2, ENC_BIG_ENDIAN);
         proto_tree_add_item(aem_tree, hf_aem_signal_output, tvb,
               INTERNAL_PORT_OFFSET_SIGNAL_OUTPUT, 2, ENC_BIG_ENDIAN);
         proto_tree_add_item(aem_tree, hf_aem_block_latency, tvb,
               INTERNAL_PORT_OFFSET_BLOCK_LATENCY, 4, ENC_BIG_ENDIAN);
         proto_tree_add_item(aem_tree, hf_aem_jack_id, tvb,
               INTERNAL_PORT_OFFSET_INTERNAL_INDEX, 2, ENC_BIG_ENDIAN);
         break;
      case AEM_DESCRIPTOR_AVB_INTERFACE:
         proto_tree_add_item(aem_tree, hf_aem_interface_name, tvb,
               AVB_INTERFACE_OFFSET_OBJECT_NAME, 64, ENC_ASCII);
         proto_tree_add_item(aem_tree, hf_aem_interface_name_string, tvb,
               AVB_INTERFACE_OFFSET_LOCALIZED_DESCRIPTION, 2, ENC_BIG_ENDIAN);

         proto_tree_add_item(aem_tree, hf_aecp_mac_address, tvb,
               AVB_INTERFACE_OFFSET_MAC_ADDRESS, 6, ENC_NA);

         proto_tree_add_item(aem_tree, hf_aem_avb_interface_flags, tvb,
               AVB_INTERFACE_OFFSET_INTERFACE_FLAGS, 2, ENC_BIG_ENDIAN);
         proto_tree_add_item(aem_tree, hf_aem_avb_clock_identity, tvb,
               AVB_INTERFACE_OFFSET_CLOCK_IDENTITY, 8, ENC_BIG_ENDIAN);
         proto_tree_add_item(aem_tree, hf_aem_avb_priority1, tvb,
               AVB_INTERFACE_OFFSET_PRIORITY1, 1, ENC_BIG_ENDIAN);
         proto_tree_add_item(aem_tree, hf_aem_avb_clock_class, tvb,
               AVB_INTERFACE_OFFSET_CLOCK_CLASS, 1, ENC_BIG_ENDIAN);
         proto_tree_add_item(aem_tree, hf_aem_avb_offset_scaled_log_variance, tvb,
               AVB_INTERFACE_OFFSET_OFFSET_SCALED_LOG_VARIANCE, 2, ENC_BIG_ENDIAN);
         proto_tree_add_item(aem_tree, hf_aem_avb_clock_accuracy, tvb,
               AVB_INTERFACE_OFFSET_CLOCK_ACCURACY, 1, ENC_BIG_ENDIAN);
         proto_tree_add_item(aem_tree, hf_aem_avb_priority2, tvb,
               AVB_INTERFACE_OFFSET_PRIORITY2, 1, ENC_BIG_ENDIAN);
         proto_tree_add_item(aem_tree, hf_aem_avb_domain_number, tvb,
               AVB_INTERFACE_OFFSET_DOMAIN_NUMBER, 1, ENC_BIG_ENDIAN);
         proto_tree_add_item(aem_tree, hf_aem_avb_log_sync_interval, tvb,
               AVB_INTERFACE_OFFSET_LOG_SYNC_INTERVAL, 1, ENC_BIG_ENDIAN);
         break;
      case AEM_DESCRIPTOR_CLOCK_SOURCE:
         proto_tree_add_item(aem_tree, hf_aem_clock_source_name, tvb,
               CLOCK_SOURCE_OFFSET_OBJECT_NAME, 64, ENC_ASCII);
         proto_tree_add_item(aem_tree, hf_aem_clock_source_name_string, tvb,
               CLOCK_SOURCE_OFFSET_LOCALIZED_DESCRIPTION, 2, ENC_BIG_ENDIAN);
         /* set up clock_source_flags subtree */
         /*mr_item = */proto_tree_add_item(aem_tree, hf_aem_clock_source_flags, tvb,
               CLOCK_SOURCE_OFFSET_CLOCK_SOURCE_FLAGS, 2, ENC_BIG_ENDIAN);
         /* mr_subtree = proto_item_add_subtree(mr_item, ett_aem_clock_source_flags);*/
         /* all flags reserved */
         /* end clock_source_flags subtree */
         proto_tree_add_item(aem_tree, hf_aem_clock_source_type, tvb,
               CLOCK_SOURCE_OFFSET_CLOCK_SOURCE_TYPE, 2, ENC_BIG_ENDIAN);
         proto_tree_add_item(aem_tree, hf_aecp_clock_source_id, tvb,
               CLOCK_SOURCE_OFFSET_CLOCK_SOURCE_IDENTIFIER, 8, ENC_BIG_ENDIAN);
         proto_tree_add_item(aem_tree, hf_aem_clock_source_location_type, tvb,
               CLOCK_SOURCE_OFFSET_CLOCK_SOURCE_LOCATION_TYPE, 2, ENC_BIG_ENDIAN);
         proto_tree_add_item(aem_tree, hf_aem_clock_source_location_id, tvb,
               CLOCK_SOURCE_OFFSET_CLOCK_SOURCE_LOCATION_INDEX, 2, ENC_BIG_ENDIAN);
         break;
      case AEM_DESCRIPTOR_AUDIO_MAP:
         proto_tree_add_item(aem_tree, hf_aem_mappings_offset, tvb,
               AUDIO_MAP_OFFSET_MAPPINGS_OFFSET, 2, ENC_BIG_ENDIAN);
         proto_tree_add_item(aem_tree, hf_aem_number_of_mappings, tvb,
               AUDIO_MAP_OFFSET_NUMBER_OF_MAPPINGS, 2, ENC_BIG_ENDIAN);

         /* prepare mappings subtree */
         mr_item = proto_tree_add_item(aem_tree, hf_aem_mappings, tvb,
               0, 0, ENC_NA);
         mr_subtree = proto_item_add_subtree(mr_item, ett_aem_mappings);
         mr_offset = tvb_get_ntohs(tvb, AUDIO_MAP_OFFSET_MAPPINGS_OFFSET);
         mr_counter = tvb_get_ntohs(tvb, AUDIO_MAP_OFFSET_NUMBER_OF_MAPPINGS);
         proto_item_set_len(mr_item, mr_counter * 8);

         for(i = 0; i < mr_counter; i++) {
            proto_tree_add_item(mr_subtree, hf_aem_mapping_stream_index, tvb,
               mr_offset, 2, ENC_BIG_ENDIAN);
            mr_offset += 2;
            proto_tree_add_item(mr_subtree, hf_aem_mapping_stream_channel, tvb,
               mr_offset, 2, ENC_BIG_ENDIAN);
            mr_offset += 2;
            proto_tree_add_item(mr_subtree, hf_aem_mapping_cluster_offset, tvb,
               mr_offset, 2, ENC_BIG_ENDIAN);
            mr_offset += 2;
            proto_tree_add_item(mr_subtree, hf_aem_mapping_cluster_channel, tvb,
               mr_offset, 2, ENC_BIG_ENDIAN);
            mr_offset += 2;
         }
         break;
      case AEM_DESCRIPTOR_AUDIO_CLUSTER:
         proto_tree_add_item(aem_tree, hf_aem_configuration_name, tvb,
               AUDIO_CLUSTER_OFFSET_OBJECT_NAME, 64, ENC_ASCII);
         proto_tree_add_item(aem_tree, hf_aem_configuration_name_string, tvb,
               AUDIO_CLUSTER_OFFSET_LOCALIZED_DESCRIPTION, 2, ENC_BIG_ENDIAN);

         proto_tree_add_item(aem_tree, hf_aem_signal_type, tvb,
               AUDIO_CLUSTER_OFFSET_SIGNAL_TYPE, 2, ENC_BIG_ENDIAN);
         proto_tree_add_item(aem_tree, hf_aem_signal_index, tvb,
               AUDIO_CLUSTER_OFFSET_SIGNAL_ID, 2, ENC_BIG_ENDIAN);
         proto_tree_add_item(aem_tree, hf_aem_signal_output, tvb,
               AUDIO_CLUSTER_OFFSET_SIGNAL_OUTPUT, 2, ENC_BIG_ENDIAN);

         proto_tree_add_item(aem_tree, hf_aem_path_latency, tvb,
               AUDIO_CLUSTER_OFFSET_PATH_LATENCY, 4, ENC_BIG_ENDIAN);
         proto_tree_add_item(aem_tree, hf_aem_block_latency, tvb,
               AUDIO_CLUSTER_OFFSET_BLOCK_LATENCY, 4, ENC_BIG_ENDIAN);

         proto_tree_add_item(aem_tree, hf_aem_channel_count, tvb,
               AUDIO_CLUSTER_OFFSET_CHANNEL_COUNT, 2, ENC_BIG_ENDIAN);

         proto_tree_add_item(aem_tree, hf_aem_am824_label, tvb,
               AUDIO_CLUSTER_OFFSET_FORMAT, 1, ENC_BIG_ENDIAN);

         break;
      case AEM_DESCRIPTOR_CONTROL:
         proto_tree_add_item(aem_tree, hf_aem_object_name, tvb,
               CONTROL_OFFSET_OBJECT_NAME, 64, ENC_ASCII);
         proto_tree_add_item(aem_tree, hf_aem_localized_description, tvb,
               CONTROL_OFFSET_LOCALIZED_DESCRIPTION, 2, ENC_BIG_ENDIAN);

         proto_tree_add_item(aem_tree, hf_aem_block_latency, tvb,
               CONTROL_OFFSET_BLOCK_LATENCY, 4, ENC_BIG_ENDIAN);
         proto_tree_add_item(aem_tree, hf_aem_control_latency, tvb,
               CONTROL_OFFSET_CONTROL_LATENCY, 4, ENC_BIG_ENDIAN);
         proto_tree_add_item(aem_tree, hf_aem_control_domain, tvb,
               CONTROL_OFFSET_CONTROL_DOMAIN, 2, ENC_BIG_ENDIAN);
         proto_tree_add_item(aem_tree, hf_aem_control_value_type, tvb,
               CONTROL_OFFSET_CONTROL_VALUE_TYPE, 2, ENC_BIG_ENDIAN);
         proto_tree_add_item(aem_tree, hf_aem_control_type, tvb,
               CONTROL_OFFSET_CONTROL_TYPE, 8, ENC_BIG_ENDIAN);

         proto_tree_add_item(aem_tree, hf_aem_reset_time, tvb,
               CONTROL_OFFSET_RESET_TIME, 4, ENC_BIG_ENDIAN);

         proto_tree_add_item(aem_tree, hf_aem_values_offset, tvb,
               CONTROL_OFFSET_VALUES_OFFSET, 2, ENC_BIG_ENDIAN);
         proto_tree_add_item(aem_tree, hf_aem_number_of_values, tvb,
               CONTROL_OFFSET_NUMBER_OF_VALUES, 2, ENC_BIG_ENDIAN);
         proto_tree_add_item(aem_tree, hf_aem_signal_type, tvb,
               CONTROL_OFFSET_SIGNAL_TYPE, 2, ENC_BIG_ENDIAN);
         proto_tree_add_item(aem_tree, hf_aem_signal_index, tvb,
               CONTROL_OFFSET_SIGNAL_INDEX, 2, ENC_BIG_ENDIAN);
         proto_tree_add_item(aem_tree, hf_aem_signal_output, tvb,
               CONTROL_OFFSET_SIGNAL_OUTPUT, 2, ENC_BIG_ENDIAN);

         ctrl_val_type = tvb_get_ntohs(tvb, CONTROL_OFFSET_CONTROL_VALUE_TYPE);
         num_ctrl_vals = tvb_get_ntohs(tvb, CONTROL_OFFSET_NUMBER_OF_VALUES);
         mr_offset = tvb_get_ntohs(tvb, CONTROL_OFFSET_VALUES_OFFSET);
         dissect_17221_ctrl_val(tvb, aem_tree, num_ctrl_vals, ctrl_val_type,
                                mr_offset);

         break;
      case AEM_DESCRIPTOR_SIGNAL_SELECTOR:
         proto_tree_add_item(aem_tree, hf_aem_object_name, tvb,
               SIGNAL_SELECTOR_OFFSET_OBJECT_NAME, 64, ENC_ASCII);
         proto_tree_add_item(aem_tree, hf_aem_localized_description, tvb,
               SIGNAL_SELECTOR_OFFSET_LOCALIZED_DESCRIPTION, 2, ENC_BIG_ENDIAN);

         proto_tree_add_item(aem_tree, hf_aem_block_latency, tvb,
               SIGNAL_SELECTOR_OFFSET_BLOCK_LATENCY, 4, ENC_BIG_ENDIAN);
         proto_tree_add_item(aem_tree, hf_aem_control_latency, tvb,
               SIGNAL_SELECTOR_OFFSET_CONTROL_LATENCY, 4, ENC_BIG_ENDIAN);
         proto_tree_add_item(aem_tree, hf_aem_control_domain, tvb,
               SIGNAL_SELECTOR_OFFSET_CONTROL_DOMAIN, 2, ENC_BIG_ENDIAN);

#if 0
         proto_tree_add_item(aem_tree, hf_aem_control_location_type, tvb,
               AEM_OFFSET_CONTROL_LOCATION_TYPE_SIGS, 2, ENC_BIG_ENDIAN);
         proto_tree_add_item(aem_tree, hf_aem_control_location_id, tvb,
               AEM_OFFSET_CONTROL_LOCATION_ID_SIGS, 2, ENC_BIG_ENDIAN);
         proto_tree_add_item(aem_tree, hf_aem_control_domain, tvb,
               AEM_OFFSET_CONTROL_DOMAIN_SIGS, 2, ENC_BIG_ENDIAN);
         proto_tree_add_item(aem_tree, hf_aem_object_name, tvb,
               AEM_OFFSET_CONTROL_NAME_SIGS, 64, ENC_ASCII);
         proto_tree_add_item(aem_tree, hf_aem_localized_description, tvb,
               AEM_OFFSET_CONTROL_NAME_STRING_SIGS, 2, ENC_BIG_ENDIAN);
         proto_tree_add_item(aem_tree, hf_aem_sources_offset, tvb,
               AEM_OFFSET_SOURCES_OFFSET_SIGS, 2, ENC_BIG_ENDIAN);
         proto_tree_add_item(aem_tree, hf_aem_number_of_sources, tvb,
               AEM_OFFSET_NUMBER_OF_SOURCES_SIGS, 2, ENC_BIG_ENDIAN);
         proto_tree_add_item(aem_tree, hf_aem_current_signal_type, tvb,
               AEM_OFFSET_CURRENT_SOURCE_TYPE_SIGS, 2, ENC_BIG_ENDIAN);
         proto_tree_add_item(aem_tree, hf_aem_current_signal_index, tvb,
               AEM_OFFSET_CURRENT_SOURCE_ID_SIGS, 2, ENC_BIG_ENDIAN);
         proto_tree_add_item(aem_tree, hf_aem_default_signal_type, tvb,
               AEM_OFFSET_DEFAULT_SOURCE_TYPE_SIGS, 2, ENC_BIG_ENDIAN);
         proto_tree_add_item(aem_tree, hf_aem_default_signal_index, tvb,
               AEM_OFFSET_DEFAULT_SOURCE_ID_SIGS, 2, ENC_BIG_ENDIAN);
         proto_tree_add_item(aem_tree, hf_aem_block_latency, tvb,
               SIGNAL_SELECTOR_OFFSET_BLOCK_LATENCY, 4, ENC_BIG_ENDIAN);
         proto_tree_add_item(aem_tree, hf_aem_control_latency, tvb,
               SIGNAL_SELECTOR_OFFSET_CONTROL_LATENCY, 4, ENC_BIG_ENDIAN);

         /* set up sources subtree */
         mr_item = proto_tree_add_item(aem_tree, hf_aem_sources, tvb,
               0, 0, ENC_NA);
         mr_subtree = proto_item_add_subtree(mr_item, ett_aem_sources);
         mr_counter = tvb_get_ntohs(tvb, AEM_OFFSET_NUMBER_OF_SOURCES_SIGS);
         mr_offset = SIGNAL_SELECTOR_OFFSET_SOURCES;
         proto_item_set_len(mr_item, mr_counter * 4);

         for(i = 0; i < mr_counter; ++i) {
           proto_tree_add_item(mr_subtree, hf_aem_signal_type, tvb,
                 mr_offset, 2, ENC_BIG_ENDIAN);
           mr_offset += 2;
           proto_tree_add_item(mr_subtree, hf_aem_signal_index, tvb,
                 mr_offset, 2, ENC_BIG_ENDIAN);
           mr_offset += 2;
         }
#endif
         break;
      case AEM_DESCRIPTOR_MIXER:
         proto_tree_add_item(aem_tree, hf_aem_object_name, tvb,
               MIXER_OFFSET_OBJECT_NAME, 64, ENC_ASCII);
         proto_tree_add_item(aem_tree, hf_aem_localized_description, tvb,
               MIXER_OFFSET_LOCALIZED_DESCRIPTION, 2, ENC_BIG_ENDIAN);

         proto_tree_add_item(aem_tree, hf_aem_block_latency, tvb,
               MIXER_OFFSET_BLOCK_LATENCY, 4, ENC_BIG_ENDIAN);
         proto_tree_add_item(aem_tree, hf_aem_control_latency, tvb,
               MIXER_OFFSET_CONTROL_LATENCY, 4, ENC_BIG_ENDIAN);
         proto_tree_add_item(aem_tree, hf_aem_control_domain, tvb,
               MIXER_OFFSET_CONTROL_DOMAIN, 2, ENC_BIG_ENDIAN);

#if 0
         proto_tree_add_item(aem_tree, hf_aem_control_location_type, tvb,
               AEM_OFFSET_CONTROL_LOCATION_TYPE_MXR, 2, ENC_BIG_ENDIAN);
         proto_tree_add_item(aem_tree, hf_aem_control_location_id, tvb,
               AEM_OFFSET_CONTROL_LOCATION_ID_MXR, 2, ENC_BIG_ENDIAN);
         proto_tree_add_item(aem_tree, hf_aem_control_value_type, tvb,
               AEM_OFFSET_CONTROL_VALUE_TYPE_MXR, 2, ENC_BIG_ENDIAN);
         proto_tree_add_item(aem_tree, hf_aem_control_domain, tvb,
               AEM_OFFSET_CONTROL_DOMAIN_MXR, 2 ,ENC_BIG_ENDIAN);
         proto_tree_add_item(aem_tree, hf_aem_object_name, tvb,
               AEM_OFFSET_CONTROL_NAME_MXR, 64, ENC_ASCII);
         proto_tree_add_item(aem_tree, hf_aem_localized_description, tvb,
               AEM_OFFSET_CONTROL_NAME_STRING_MXR, 2, ENC_BIG_ENDIAN);
         proto_tree_add_item(aem_tree, hf_aem_sources_offset, tvb,
               AEM_OFFSET_SOURCES_OFFSET_MXR, 2, ENC_BIG_ENDIAN);
         proto_tree_add_item(aem_tree, hf_aem_number_of_sources, tvb,
               AEM_OFFSET_NUMBER_OF_SOURCES_MXR, 2, ENC_BIG_ENDIAN);
         proto_tree_add_item(aem_tree, hf_aem_value_offset, tvb,
               AEM_OFFSET_VALUE_OFFSET_MXR, 2, ENC_BIG_ENDIAN);
         proto_tree_add_item(aem_tree, hf_aem_block_latency, tvb,
               MIXER_OFFSET_BLOCK_LATENCY, 2, ENC_BIG_ENDIAN);
         proto_tree_add_item(aem_tree, hf_aem_control_latency, tvb,
               MIXER_OFFSET_CONTROL_LATENCY, 4, ENC_BIG_ENDIAN);

         /* set up subtree for sources */
         mr_item = proto_tree_add_item(aem_tree, hf_aem_sources, tvb,
               0, 0, ENC_NA);
         mr_subtree = proto_item_add_subtree(mr_item, ett_aem_sources);
         mr_counter = tvb_get_ntohs(tvb, AEM_OFFSET_NUMBER_OF_SOURCES_MXR);
         mr_offset = MIXER_OFFSET_SOURCES;
         proto_item_set_len(mr_item, mr_counter * 4);

         for(i = 0; i < mr_counter; ++i) {
           proto_tree_add_item(mr_subtree, hf_aem_signal_type, tvb,
                 mr_offset, 2, ENC_BIG_ENDIAN);
           mr_offset += 2;
           proto_tree_add_item(mr_subtree, hf_aem_signal_index, tvb,
                 mr_offset, 2, ENC_BIG_ENDIAN);
           mr_offset += 2;
         }

         /* end sources subtree */

         ctrl_val_type = tvb_get_ntohs(tvb, AEM_OFFSET_CONTROL_VALUE_TYPE_MXR);
         num_ctrl_vals = 1;
         dissect_17221_ctrl_val(tvb, aem_tree, num_ctrl_vals, ctrl_val_type,
               MIXER_OFFSET_SOURCES + (tvb_get_ntohs(tvb, AEM_OFFSET_NUMBER_OF_SOURCES_MXR) * 4));
#endif
         break;
      case AEM_DESCRIPTOR_MATRIX:
         proto_tree_add_item(aem_tree, hf_aem_object_name, tvb,
               MATRIX_OFFSET_OBJECT_NAME, 64, ENC_ASCII);
         proto_tree_add_item(aem_tree, hf_aem_localized_description, tvb,
               MATRIX_OFFSET_LOCALIZED_DESCRIPTION, 2, ENC_BIG_ENDIAN);

         proto_tree_add_item(aem_tree, hf_aem_block_latency, tvb,
               MATRIX_OFFSET_BLOCK_LATENCY, 4, ENC_BIG_ENDIAN);
         proto_tree_add_item(aem_tree, hf_aem_control_latency, tvb,
               MATRIX_OFFSET_CONTROL_LATENCY, 4, ENC_BIG_ENDIAN);
         proto_tree_add_item(aem_tree, hf_aem_control_domain, tvb,
               MATRIX_OFFSET_CONTROL_DOMAIN, 2, ENC_BIG_ENDIAN);

         /*
         proto_tree_add_item(aem_tree, hf_aem_control_type, tvb,
               AEM_OFFSET_CONTROL_TYPE_MTRX, 8, ENC_BIG_ENDIAN);
         proto_tree_add_item(aem_tree, hf_aem_control_location_type, tvb,
               AEM_OFFSET_CONTROL_LOCATION_TYPE_MTRX, 2, ENC_BIG_ENDIAN);
         proto_tree_add_item(aem_tree, hf_aem_control_location_id, tvb,
               AEM_OFFSET_CONTROL_LOCATION_ID_MTRX, 2, ENC_BIG_ENDIAN);
         proto_tree_add_item(aem_tree, hf_aem_control_value_type, tvb,
               AEM_OFFSET_CONTROL_VALUE_TYPE_MTRX, 2, ENC_BIG_ENDIAN);
         proto_tree_add_item(aem_tree, hf_aem_control_domain, tvb,
               AEM_OFFSET_CONTROL_DOMAIN_MTRX, 2 ,ENC_BIG_ENDIAN);
         proto_tree_add_item(aem_tree, hf_aem_object_name, tvb,
               AEM_OFFSET_CONTROL_NAME_MTRX, 64, ENC_ASCII);
         proto_tree_add_item(aem_tree, hf_aem_localized_description, tvb,
               AEM_OFFSET_CONTROL_NAME_STRING_MTRX, 2, ENC_BIG_ENDIAN);
         proto_tree_add_item(aem_tree, hf_aem_width, tvb,
               AEM_OFFSET_WIDTH_MTRX, 2, ENC_BIG_ENDIAN);
         proto_tree_add_item(aem_tree, hf_aem_height, tvb,
               AEM_OFFSET_HEIGHT_MTRX, 2, ENC_BIG_ENDIAN);
         proto_tree_add_item(aem_tree, hf_aem_values_offset, tvb,
               AEM_OFFSET_VALUES_OFFSET_MTRX, 2, ENC_BIG_ENDIAN);
         proto_tree_add_item(aem_tree, hf_aem_number_of_values, tvb,
               AEM_OFFSET_NUMBER_OF_VALUES_MTRX, 2, ENC_BIG_ENDIAN);

         proto_tree_add_item(aem_tree, hf_aem_block_latency, tvb,
               MATRIX_OFFSET_BLOCK_LATENCY, 4, ENC_BIG_ENDIAN);
         proto_tree_add_item(aem_tree, hf_aem_control_latency, tvb,
               MATRIX_OFFSET_CONTROL_LATENCY, 4, ENC_BIG_ENDIAN);
         proto_tree_add_item(aem_tree, hf_aem_number_of_sources, tvb,
               MATRIX_OFFSET_NUMBER_SOURCES, 2, ENC_BIG_ENDIAN);
         proto_tree_add_item(aem_tree, hf_aem_base_source, tvb,
               MATRIX_OFFSET_BASE_SOURCE, 2, ENC_BIG_ENDIAN);
         proto_tree_add_item(aem_tree, hf_aem_number_destinations, tvb,
               MATRIX_OFFSET_NUMBER_DESTINATIONS, 2, ENC_BIG_ENDIAN);
         proto_tree_add_item(aem_tree, hf_aem_base_destination, tvb,
               MATRIX_OFFSET_BASE_DESTINATION, 2, ENC_BIG_ENDIAN);

         ctrl_val_type = tvb_get_ntohs(tvb, AEM_OFFSET_CONTROL_VALUE_TYPE_MTRX);
         num_ctrl_vals = tvb_get_ntohs(tvb, AEM_OFFSET_NUMBER_OF_VALUES_MTRX);
         dissect_17221_ctrl_val(tvb, aem_tree, num_ctrl_vals, ctrl_val_type,
                                MATRIX_OFFSET_VALUE_DETAILS);
         */
         break;
      case AEM_DESCRIPTOR_LOCALE:
         proto_tree_add_item(aem_tree, hf_aem_locale_identifier, tvb,
               LOCALE_OFFSET_LOCALE_IDENTIFIER, 64, ENC_ASCII);
         proto_tree_add_item(aem_tree, hf_aem_number_of_strings, tvb,
               LOCALE_OFFSET_NUMBER_OF_STRINGS, 2, ENC_BIG_ENDIAN);
         proto_tree_add_item(aem_tree, hf_aem_base_strings, tvb,
               LOCALE_OFFSET_BASE_STRINGS, 2, ENC_BIG_ENDIAN);
         break;
      case AEM_DESCRIPTOR_STRINGS:
         mr_offset = STRINGS_OFFSET_STRING0;
         for(i = 0; i < 7; ++i) {
            proto_tree_add_item(aem_tree, hf_aem_string, tvb,
                  mr_offset, 64, ENC_ASCII);
            mr_offset += 64;
         }
         break;
      case AEM_DESCRIPTOR_MATRIX_SIGNAL:
         proto_tree_add_item(aem_tree, hf_aem_signals_count, tvb,
               MATRIX_SIGNAL_OFFSET_SIGNALS_COUNT, 2, ENC_BIG_ENDIAN);
         proto_tree_add_item(aem_tree, hf_aem_signals_offset, tvb,
               MATRIX_SIGNAL_OFFSET_SIGNALS_OFFSET, 2, ENC_BIG_ENDIAN);
         /* set up subtree for signals */
         mr_item = proto_tree_add_item(aem_tree, hf_aem_sources, tvb,
               0, 0, ENC_NA);
         mr_subtree = proto_item_add_subtree(mr_item, ett_aem_sources);
         mr_counter = tvb_get_ntohs(tvb, MATRIX_SIGNAL_OFFSET_SIGNALS_COUNT);
         mr_offset = MATRIX_SIGNAL_OFFSET_SIGNALS_OFFSET;
         proto_item_set_len(mr_item, mr_counter * 4);

         for(i = 0; i < mr_counter; ++i)
         {
           proto_tree_add_item(mr_subtree, hf_aem_signal_type, tvb,
                 mr_offset, 2, ENC_BIG_ENDIAN);
           mr_offset += 2;
           proto_tree_add_item(mr_subtree, hf_aem_signal_index, tvb,
                 mr_offset, 2, ENC_BIG_ENDIAN);
           mr_offset += 2;
         }
         break;
      case AEM_DESCRIPTOR_MEMORY_OBJECT:
         proto_tree_add_item(aem_tree, hf_aem_memory_object_type, tvb,
               MEMORY_OBJECT_OFFSET_MEMORY_OBJECT_TYPE, 2, ENC_BIG_ENDIAN);
         proto_tree_add_item(aem_tree, hf_aem_target_descriptor_type, tvb,
               MEMORY_OBJECT_OFFSET_TARGET_DESCRIPTOR_TYPE, 2, ENC_BIG_ENDIAN);
         proto_tree_add_item(aem_tree, hf_aem_target_descriptor_id, tvb,
               MEMORY_OBJECT_OFFSET_TARGET_DESCRIPTOR_INDEX, 2, ENC_BIG_ENDIAN);
         proto_tree_add_item(aem_tree, hf_aem_object_name, tvb,
               MEMORY_OBJECT_OFFSET_OBJECT_NAME, 64, ENC_ASCII);
         proto_tree_add_item(aem_tree, hf_aem_localized_description, tvb,
               MEMORY_OBJECT_OFFSET_LOCALIZED_DESCRIPTION, 2, ENC_BIG_ENDIAN);
         proto_tree_add_item(aem_tree, hf_aem_start_address, tvb,
               MEMORY_OBJECT_OFFSET_START_ADDRESS, 8, ENC_BIG_ENDIAN);
         proto_tree_add_item(aem_tree, hf_aem_length, tvb,
               MEMORY_OBJECT_OFFSET_LENGTH, 8, ENC_BIG_ENDIAN);
         break;
      case AEM_DESCRIPTOR_CLOCK_DOMAIN:
         proto_tree_add_item(aem_tree, hf_aem_object_name, tvb,
               CLOCK_DOMAIN_OFFSET_OBJECT_NAME, 64, ENC_ASCII);
         proto_tree_add_item(aem_tree, hf_aem_localized_description, tvb,
               CLOCK_DOMAIN_OFFSET_LOCALIZED_DESCRIPTION, 2, ENC_BIG_ENDIAN);
         proto_tree_add_item(aem_tree, hf_aem_clock_source_index, tvb,
               CLOCK_DOMAIN_OFFSET_CLOCK_SOURCE_INDEX, 2, ENC_BIG_ENDIAN);
         proto_tree_add_item(aem_tree, hf_aem_clock_sources_offset, tvb,
               CLOCK_DOMAIN_OFFSET_CLOCK_SOURCES_OFFSET, 2, ENC_BIG_ENDIAN);
         proto_tree_add_item(aem_tree, hf_aem_clock_sources_count, tvb,
               CLOCK_DOMAIN_OFFSET_CLOCK_SOURCES_COUNT, 2, ENC_BIG_ENDIAN);

         /* set up a clock sources subtree */
         mr_counter = tvb_get_ntohs(tvb, CLOCK_DOMAIN_OFFSET_CLOCK_SOURCES_COUNT);
         mr_offset =  tvb_get_ntohs(tvb, CLOCK_DOMAIN_OFFSET_CLOCK_SOURCES_OFFSET);
         mr_item = proto_tree_add_item(aem_tree, hf_aem_clock_sources_array, tvb,
               mr_offset, mr_counter * 2, ENC_NA);
         mr_subtree = proto_item_add_subtree(mr_item, ett_aem_clock_sources);
         /* add clocks to new subtree */
         for(i = 0; i < mr_counter; ++i)
         {
            proto_tree_add_item(mr_subtree, hf_aem_clock_sources, tvb,
               mr_offset, 2, ENC_BIG_ENDIAN);
            mr_offset += 2;
         }
         break;

      default:
         proto_tree_add_item(aem_tree, hf_aem_unknown_descriptor, tvb,
               4, tvb_captured_length(tvb) - 4, ENC_NA);
         break;
   }
}

/* dissect enumeration and control packets */
static void
dissect_17221_aecp(tvbuff_t *tvb, packet_info *pinfo, proto_tree *aecp_tree)
{
   guint16 c_type;
   /*guint16 addr_type;*/
   /*guint16 ctrl_data_len;*/
   guint16 mess_status;
   guint16 mess_type;
   guint16 mr_counter;
   guint32 mr_offset;
   proto_item *mr_subtree;
   proto_item *mr_item;
   int i;
   guint64 vendor_unique_protocol_id;
   gchar *vendor_unique_protocol_id_string;
   /* next tvb for use in subdissection */
   tvbuff_t *next_tvb;
   proto_tree *flags_tree;
   proto_item *flags_ti;


   /* AECP Common Format Fields */
   proto_tree_add_item(aecp_tree, hf_aecp_message_type, tvb,
         AECP_OFFSET_VERSION, 1, ENC_BIG_ENDIAN);
   proto_tree_add_item(aecp_tree, hf_aecp_status_code, tvb,
         AECP_OFFSET_STATUS_CODE, 1, ENC_BIG_ENDIAN);
   proto_tree_add_item(aecp_tree, hf_aecp_cd_length, tvb,
         AECP_OFFSET_CD_LENGTH, 2, ENC_BIG_ENDIAN);
   proto_tree_add_item(aecp_tree, hf_aecp_target_guid, tvb,
         AECP_OFFSET_TARGET_GUID, 8, ENC_BIG_ENDIAN);
   proto_tree_add_item(aecp_tree, hf_aecp_controller_guid, tvb,
         AECP_OFFSET_CONTROLLER_GUID, 8, ENC_BIG_ENDIAN);
   proto_tree_add_item(aecp_tree, hf_aecp_sequence_id, tvb,
         AECP_OFFSET_SEQUENCE_ID, 2, ENC_BIG_ENDIAN);

   /* get the message type */
   mess_type = tvb_get_ntohs(tvb, 0) & AECP_MSG_TYPE_MASK;

   if  ((mess_type == AECP_AEM_COMMAND_MESSAGE) || (mess_type == AECP_AEM_RESPONSE_MESSAGE))
   {
      proto_tree_add_item(aecp_tree, hf_aecp_u_flag, tvb,
         AECP_OFFSET_U_FLAG, 1, ENC_BIG_ENDIAN);
      proto_tree_add_item(aecp_tree, hf_aecp_command_type, tvb,
         AECP_OFFSET_COMMAND_TYPE, 2, ENC_BIG_ENDIAN);

    /* get the command type for detailed dissection */
      c_type = tvb_get_ntohs(tvb, AECP_OFFSET_COMMAND_TYPE) & AECP_COMMAND_TYPE_MASK;

      /* get the control data length field - number of octets following target_guid */
      /*ctrl_data_len = tvb_get_ntohs(tvb, AECP_OFFSET_CD_LENGTH) & AECP_CD_LENGTH_MASK;*/


      /* get the status */
      mess_status = tvb_get_ntohs(tvb, 2) & 0xF800;

      /* break dissection down by command type */
      /* fields are added in the order they are listed by 1722.1 */
      switch(c_type) {
        case AECP_COMMAND_ACQUIRE_ENTITY:
           /* set up the flags subtree */
           flags_ti = proto_tree_add_item(aecp_tree, hf_aecp_flags_32, tvb,
                 AECP_OFFSET_ACQUIRE_ENTITY_FLAGS, 4, ENC_BIG_ENDIAN);
           flags_tree = proto_item_add_subtree(flags_ti, ett_acmp_flags);
           proto_tree_add_item(flags_tree, hf_aecp_persistent_flag, tvb,
                 AECP_OFFSET_ACQUIRE_ENTITY_FLAGS, 4, ENC_BIG_ENDIAN);
           proto_tree_add_item(aecp_tree, hf_aecp_release_flag, tvb,
                 AECP_OFFSET_ACQUIRE_ENTITY_FLAGS, 4, ENC_BIG_ENDIAN);
           /* end flags subtree */

           proto_tree_add_item(aecp_tree, hf_aecp_owner_guid, tvb,
                 AECP_OFFSET_ACQUIRE_ENTITY_OWNER_GUID, 8, ENC_BIG_ENDIAN);
           proto_tree_add_item(aecp_tree, hf_aecp_descriptor_type, tvb,
                 AECP_OFFSET_ACQUIRE_ENTITY_DESCRIPTOR_TYPE, 2, ENC_BIG_ENDIAN);
           proto_tree_add_item(aecp_tree, hf_aecp_descriptor_index, tvb,
                 AECP_OFFSET_ACQUIRE_ENTITY_DESCRIPTOR_INDEX, 2, ENC_BIG_ENDIAN);
           break;
        case AECP_COMMAND_LOCK_ENTITY:
           proto_tree_add_item(aecp_tree, hf_aecp_unlock_flag, tvb,
                 AECP_OFFSET_LOCK_ENTITY_FLAGS, 4, ENC_BIG_ENDIAN);
           proto_tree_add_item(aecp_tree, hf_aecp_locked_guid, tvb,
                 AECP_OFFSET_LOCK_ENTITY_LOCKED_GUID, 8, ENC_BIG_ENDIAN);
           break;
        case AECP_COMMAND_READ_DESCRIPTOR:
           proto_tree_add_item(aecp_tree, hf_aecp_configuration, tvb,
                 AECP_OFFSET_DESCRIPTOR_CONFIGURATION_INDEX, 2, ENC_BIG_ENDIAN);

           if ((mess_type == AECP_AEM_COMMAND_MESSAGE)||(mess_status != 0))
           {
              proto_tree_add_item(aecp_tree, hf_aecp_descriptor_type, tvb,
                    AECP_OFFSET_DESCRIPTOR_DESCRIPTOR_TYPE, 2, ENC_BIG_ENDIAN);
              proto_tree_add_item(aecp_tree, hf_aecp_descriptor_index, tvb,
                    AECP_OFFSET_DESCRIPTOR_DESCRIPTOR_INDEX, 2, ENC_BIG_ENDIAN);
           }
           else
           {
              next_tvb = tvb_new_subset_remaining(tvb, AECP_OFFSET_DESCRIPTOR_DESCRIPTOR);
              dissect_17221_aem(next_tvb, pinfo, aecp_tree);
           }
           break;
        case AECP_COMMAND_WRITE_DESCRIPTOR:
           proto_tree_add_item(aecp_tree, hf_aecp_configuration, tvb,
                 AECP_OFFSET_DESCRIPTOR_CONFIGURATION_INDEX, 2, ENC_BIG_ENDIAN);

           /* on command descriptor is value to write
            * on response descriptor is command value if successful
            * or old value if unsuccessful */
           next_tvb = tvb_new_subset_remaining(tvb, AECP_OFFSET_DESCRIPTOR_DESCRIPTOR);
           dissect_17221_aem(next_tvb, pinfo, aecp_tree);
           break;
        case AECP_COMMAND_SET_CONFIGURATION:
        case AECP_COMMAND_GET_CONFIGURATION:
           proto_tree_add_item(aecp_tree, hf_aecp_configuration,
                 tvb, AECP_OFFSET_CONFIGURATION_CONFIGURATION_INDEX, 2, ENC_BIG_ENDIAN);
           break;
        case AECP_COMMAND_SET_STREAM_FORMAT:
        case AECP_COMMAND_GET_STREAM_FORMAT:
           proto_tree_add_item(aecp_tree, hf_aecp_descriptor_type, tvb,
                 AECP_OFFSET_STREAM_FORMAT_DESCRIPTOR_TYPE, 2, ENC_BIG_ENDIAN);
           proto_tree_add_item(aecp_tree, hf_aecp_descriptor_index, tvb,
                 AECP_OFFSET_STREAM_FORMAT_DESCRIPTOR_INDEX, 2, ENC_BIG_ENDIAN);

           if ((mess_type != AECP_AEM_COMMAND_MESSAGE)||(c_type == AECP_COMMAND_SET_STREAM_FORMAT))
           {
              next_tvb = tvb_new_subset_length(tvb, AECP_OFFSET_STREAM_FORMAT_STREAM_FORMAT, 8);
              dissect_17221_stream_format(next_tvb, aecp_tree);
           }
           break;
        case AECP_COMMAND_SET_VIDEO_FORMAT:
        case AECP_COMMAND_GET_VIDEO_FORMAT:
            proto_tree_add_item(aecp_tree, hf_aecp_descriptor_type, tvb,
               AECP_OFFSET_VIDEO_FORMAT_DESCRIPTOR_TYPE, 2, ENC_BIG_ENDIAN);
            proto_tree_add_item(aecp_tree, hf_aecp_descriptor_index, tvb,
               AECP_OFFSET_VIDEO_FORMAT_DESCRIPTOR_INDEX, 2, ENC_BIG_ENDIAN);

            if ((mess_type != AECP_AEM_COMMAND_MESSAGE)||(c_type == AECP_COMMAND_SET_VIDEO_FORMAT))
            {
                next_tvb = tvb_new_subset_length(tvb, AECP_OFFSET_VIDEO_FORMAT_FORMAT_SPECIFIC, 4);
                dissect_17221_video_format(next_tvb, aecp_tree);

                next_tvb = tvb_new_subset_length(tvb, AECP_OFFSET_VIDEO_FORMAT_ASPECT_RATIO, 2);
                dissect_17221_video_aspect_ratio(next_tvb, aecp_tree);

                proto_tree_add_item(aecp_tree, hf_aem_color_space, tvb,
                     AECP_OFFSET_VIDEO_FORMAT_COLOR_SPACE, 2, ENC_BIG_ENDIAN);

                next_tvb = tvb_new_subset_length(tvb, AECP_OFFSET_VIDEO_FORMAT_FRAME_SIZE, 4);
                dissect_17221_video_frame_size(next_tvb, aecp_tree);
            }
            break;
          case AECP_COMMAND_SET_SENSOR_FORMAT:
          case AECP_COMMAND_GET_SENSOR_FORMAT:
            proto_tree_add_item(aecp_tree, hf_aecp_descriptor_type, tvb,
                  AECP_OFFSET_SENSOR_FORMAT_DESCRIPTOR_TYPE, 2, ENC_BIG_ENDIAN);
            proto_tree_add_item(aecp_tree, hf_aecp_descriptor_index, tvb,
                  AECP_OFFSET_SENSOR_FORMAT_DESCRIPTOR_INDEX, 2, ENC_BIG_ENDIAN);

            if ((mess_type != AECP_AEM_COMMAND_MESSAGE)||(c_type == AECP_COMMAND_SET_SENSOR_FORMAT))
            {
                next_tvb = tvb_new_subset_length(tvb, AECP_OFFSET_SENSOR_FORMAT_SENSOR_FORMAT, 8);
                dissect_17221_sensor_format(next_tvb, aecp_tree);
            }
            break;
          case AECP_COMMAND_SET_STREAM_INFO:
          case AECP_COMMAND_GET_STREAM_INFO:
            proto_tree_add_item(aecp_tree, hf_aecp_descriptor_type, tvb,
                  AECP_OFFSET_STREAM_INFO_DESCRIPTOR_TYPE, 2, ENC_BIG_ENDIAN);
            proto_tree_add_item(aecp_tree, hf_aecp_descriptor_index, tvb,
                  AECP_OFFSET_STREAM_INFO_DESCRIPTOR_INDEX, 2, ENC_BIG_ENDIAN);
            if ((mess_type == AECP_AEM_RESPONSE_MESSAGE) || (c_type == AECP_COMMAND_SET_STREAM_INFO)) {
               proto_tree_add_item(aecp_tree, hf_acmp_flags_class_b, tvb,
                     AECP_OFFSET_STREAM_INFO_FLAGS, 4, ENC_BIG_ENDIAN);
                proto_tree_add_item(aecp_tree, hf_acmp_flags_fast_connect, tvb,
                     AECP_OFFSET_STREAM_INFO_FLAGS, 4, ENC_BIG_ENDIAN);
                proto_tree_add_item(aecp_tree, hf_acmp_flags_saved_state, tvb,
                     AECP_OFFSET_STREAM_INFO_FLAGS, 4, ENC_BIG_ENDIAN);
                proto_tree_add_item(aecp_tree, hf_acmp_flags_streaming_wait, tvb,
                     AECP_OFFSET_STREAM_INFO_FLAGS, 4, ENC_BIG_ENDIAN);
                proto_tree_add_item(aecp_tree, hf_acmp_flags_supports_encrypted, tvb,
                     AECP_OFFSET_STREAM_INFO_FLAGS, 4, ENC_BIG_ENDIAN);
                proto_tree_add_item(aecp_tree, hf_acmp_flags_encrypted_pdu, tvb,
                     AECP_OFFSET_STREAM_INFO_FLAGS, 4, ENC_BIG_ENDIAN);
                proto_tree_add_item(aecp_tree, hf_acmp_flags_talker_failed, tvb,
                     AECP_OFFSET_STREAM_INFO_FLAGS, 4, ENC_BIG_ENDIAN);

                proto_tree_add_item(aecp_tree, hf_aecp_stream_vlan_id_valid_flag, tvb,
                     AECP_OFFSET_STREAM_INFO_FLAGS, 4, ENC_BIG_ENDIAN);
                proto_tree_add_item(aecp_tree, hf_aecp_connected_flag, tvb,
                     AECP_OFFSET_STREAM_INFO_FLAGS, 4, ENC_BIG_ENDIAN);
                proto_tree_add_item(aecp_tree, hf_aecp_msrp_failure_valid_flag, tvb,
                     AECP_OFFSET_STREAM_INFO_FLAGS, 4, ENC_BIG_ENDIAN);
                proto_tree_add_item(aecp_tree, hf_aecp_dest_mac_valid_flag, tvb,
                     AECP_OFFSET_STREAM_INFO_FLAGS, 4, ENC_BIG_ENDIAN);
                proto_tree_add_item(aecp_tree, hf_aecp_msrp_acc_lat_valid_flag, tvb,
                     AECP_OFFSET_STREAM_INFO_FLAGS, 4, ENC_BIG_ENDIAN);
                proto_tree_add_item(aecp_tree, hf_aecp_stream_id_valid_flag, tvb,
                     AECP_OFFSET_STREAM_INFO_FLAGS, 4, ENC_BIG_ENDIAN);
                proto_tree_add_item(aecp_tree, hf_aecp_stream_format_valid_flag, tvb,
                     AECP_OFFSET_STREAM_INFO_FLAGS, 4, ENC_BIG_ENDIAN);

                proto_tree_add_item(aecp_tree, hf_aecp_stream_format, tvb,
                     AECP_OFFSET_STREAM_INFO_STREAM_FORMAT, 8, ENC_BIG_ENDIAN);
                proto_tree_add_item(aecp_tree, hf_acmp_stream_id, tvb,
                     AECP_OFFSET_STREAM_INFO_STREAM_ID, 8, ENC_BIG_ENDIAN);

                proto_tree_add_item(aecp_tree, hf_aecp_msrp_accumulated_latency, tvb,
                     AECP_OFFSET_STREAM_INFO_MSRP_ACCUMULATED_LATENCY, 4, ENC_BIG_ENDIAN);
                proto_tree_add_item(aecp_tree, hf_acmp_stream_dest_mac, tvb,
                     AECP_OFFSET_STREAM_INFO_STREAM_DEST_MAC, 6, ENC_NA);
                proto_tree_add_item(aecp_tree, hf_aecp_msrp_failure_code, tvb,
                     AECP_OFFSET_STREAM_INFO_MSRP_FAILURE_CODE, 1, ENC_BIG_ENDIAN);
                proto_tree_add_item(aecp_tree, hf_aecp_msrp_failure_bridge_id, tvb,
                     AECP_OFFSET_STREAM_INFO_MSRP_FAILURE_BRIDGE_ID, 8, ENC_NA);
            }
            break;
          case AECP_COMMAND_SET_NAME:
          case AECP_COMMAND_GET_NAME:
            proto_tree_add_item(aecp_tree, hf_aecp_descriptor_type, tvb,
                  AECP_OFFSET_NAME_DESCRIPTOR_TYPE, 2, ENC_BIG_ENDIAN);
            proto_tree_add_item(aecp_tree, hf_aecp_descriptor_index, tvb,
                  AECP_OFFSET_NAME_DESCRIPTOR_INDEX, 2, ENC_BIG_ENDIAN);
            proto_tree_add_item(aecp_tree, hf_aecp_name_index, tvb,
                  AECP_OFFSET_NAME_NAME_INDEX, 2, ENC_BIG_ENDIAN);
            proto_tree_add_item(aecp_tree, hf_aecp_configuration_index, tvb,
                  AECP_OFFSET_NAME_CONFIGURATION_INDEX, 2, ENC_BIG_ENDIAN);

            if ((mess_type == AECP_AEM_RESPONSE_MESSAGE) || (c_type == AECP_COMMAND_SET_NAME)) {
               proto_tree_add_item(aecp_tree, hf_aecp_name, tvb,
                     AECP_OFFSET_NAME_NAME, 64, ENC_ASCII);
            }
            break;
          case AECP_COMMAND_SET_ASSOCIATION_ID:
          case AECP_COMMAND_GET_ASSOCIATION_ID:
            proto_tree_add_item(aecp_tree, hf_aecp_association_id, tvb,
                  AECP_OFFSET_ASSOCIATION_ID_ASSOCIATION_ID, 8, ENC_BIG_ENDIAN);
            break;
          case AECP_COMMAND_SET_SAMPLING_RATE:
          case AECP_COMMAND_GET_SAMPLING_RATE:
            proto_tree_add_item(aecp_tree, hf_aecp_descriptor_type, tvb,
                  AECP_OFFSET_SAMPLING_RATE_DESCRIPTOR_TYPE, 2, ENC_BIG_ENDIAN);
            proto_tree_add_item(aecp_tree, hf_aecp_descriptor_index, tvb,
                  AECP_OFFSET_SAMPLING_RATE_DESCRIPTOR_INDEX, 2, ENC_BIG_ENDIAN);
            if ((mess_type == AECP_AEM_RESPONSE_MESSAGE) || (c_type == AECP_COMMAND_SET_SAMPLING_RATE)) {
               proto_tree_add_item(aecp_tree,hf_aecp_sampling_rate_pull , tvb,
                     AECP_OFFSET_SAMPLING_RATE_SAMPLING_RATE, 1, ENC_BIG_ENDIAN);
               proto_tree_add_item(aecp_tree, hf_aecp_sampling_rate_base_frequency, tvb,
                     AECP_OFFSET_SAMPLING_RATE_SAMPLING_RATE, 4, ENC_BIG_ENDIAN);
            }
            break;
          case AECP_COMMAND_SET_CLOCK_SOURCE:
          case AECP_COMMAND_GET_CLOCK_SOURCE:
            proto_tree_add_item(aecp_tree, hf_aecp_descriptor_type, tvb,
                  AECP_OFFSET_CLOCK_SOURCE_DESCRIPTOR_TYPE, 2, ENC_BIG_ENDIAN);
            proto_tree_add_item(aecp_tree, hf_aecp_descriptor_index, tvb,
                  AECP_OFFSET_CLOCK_SOURCE_DESCRIPTOR_INDEX, 2, ENC_BIG_ENDIAN);
            if ((mess_type == AECP_AEM_RESPONSE_MESSAGE) || (c_type == AECP_COMMAND_SET_CLOCK_SOURCE)) {
               proto_tree_add_item(aecp_tree, hf_aem_clock_source_id, tvb,
                     AECP_OFFSET_CLOCK_SOURCE_CLOCK_SOURCE_INDEX, 2, ENC_BIG_ENDIAN);
            }
            break;
          case AECP_COMMAND_SET_CONTROL_VALUE:
          case AECP_COMMAND_GET_CONTROL_VALUE:
          case AECP_COMMAND_SET_MIXER:
          case AECP_COMMAND_GET_MIXER:
            proto_tree_add_item(aecp_tree, hf_aecp_descriptor_type, tvb,
                  AECP_OFFSET_CONTROL_DESCRIPTOR_TYPE, 2, ENC_BIG_ENDIAN);
            proto_tree_add_item(aecp_tree, hf_aecp_descriptor_index, tvb,
                  AECP_OFFSET_CONTROL_DESCRIPTOR_INDEX, 2, ENC_BIG_ENDIAN);
            /* TODO: dissect control/mixer values */
            break;
          case AECP_COMMAND_INCREMENT_CONTROL:
          case AECP_COMMAND_DECREMENT_CONTROL:
            proto_tree_add_item(aecp_tree, hf_aecp_descriptor_type, tvb,
                  AECP_OFFSET_INCDEC_CONTROL_DESCRIPTOR_TYPE, 2, ENC_BIG_ENDIAN);
            proto_tree_add_item(aecp_tree, hf_aecp_descriptor_index, tvb,
                  AECP_OFFSET_INCDEC_CONTROL_DESCRIPTOR_INDEX, 2, ENC_BIG_ENDIAN);
            /* TODO: dissect control index list (message) or values (response) */
            break;
          case AECP_COMMAND_SET_SIGNAL_SELECTOR:
          case AECP_COMMAND_GET_SIGNAL_SELECTOR:
            proto_tree_add_item(aecp_tree, hf_aecp_descriptor_type, tvb,
                  AECP_OFFSET_SIGNAL_SELECTOR_DESCRIPTOR_TYPE, 2, ENC_BIG_ENDIAN);
            proto_tree_add_item(aecp_tree, hf_aecp_descriptor_index, tvb,
                  AECP_OFFSET_SIGNAL_SELECTOR_DESCRIPTOR_INDEX, 2, ENC_BIG_ENDIAN);
            if ((mess_type == AECP_AEM_RESPONSE_MESSAGE) || (c_type == AECP_COMMAND_SET_SIGNAL_SELECTOR)) {
               proto_tree_add_item(aecp_tree, hf_aecp_signal_type, tvb,
                     AECP_OFFSET_SIGNAL_SELECTOR_SIGNAL_TYPE, 2, ENC_BIG_ENDIAN);
                proto_tree_add_item(aecp_tree, hf_aecp_signal_index, tvb,
                     AECP_OFFSET_SIGNAL_SELECTOR_SIGNAL_INDEX, 2, ENC_BIG_ENDIAN);
                proto_tree_add_item(aecp_tree, hf_aecp_signal_output, tvb,
                     AECP_OFFSET_SIGNAL_SELECTOR_SIGNAL_OUTPUT, 2, ENC_BIG_ENDIAN);
            }
            break;
          case AECP_COMMAND_GET_MATRIX:
          case AECP_COMMAND_SET_MATRIX:
            proto_tree_add_item(aecp_tree, hf_aecp_descriptor_type, tvb,
                  AECP_OFFSET_MATRIX_DESCRIPTOR_TYPE, 2, ENC_BIG_ENDIAN);
            proto_tree_add_item(aecp_tree, hf_aecp_descriptor_index, tvb,
                  AECP_OFFSET_MATRIX_DESCRIPTOR_INDEX, 2, ENC_BIG_ENDIAN);
            proto_tree_add_item(aecp_tree, hf_aecp_matrix_column, tvb,
                  AECP_OFFSET_MATRIX_MATRIX_COLUMN, 2, ENC_BIG_ENDIAN);
            proto_tree_add_item(aecp_tree, hf_aecp_matrix_row, tvb,
                  AECP_OFFSET_MATRIX_MATRIX_ROW, 2, ENC_BIG_ENDIAN);
            proto_tree_add_item(aecp_tree, hf_aecp_matrix_region_width, tvb,
                  AECP_OFFSET_MATRIX_REGION_WIDTH, 2, ENC_BIG_ENDIAN);
            proto_tree_add_item(aecp_tree, hf_aecp_matrix_region_height, tvb, AECP_OFFSET_MATRIX_REGION_HEIGHT, 2, ENC_BIG_ENDIAN); proto_tree_add_item(aecp_tree, hf_aecp_matrix_rep, tvb,
                  AECP_OFFSET_MATRIX_REP, 1, ENC_BIG_ENDIAN);
            proto_tree_add_item(aecp_tree, hf_aecp_matrix_direction, tvb,
                  AECP_OFFSET_MATRIX_DIRECTION, 1, ENC_BIG_ENDIAN);
            proto_tree_add_item(aecp_tree, hf_aecp_matrix_value_count, tvb,
                  AECP_OFFSET_MATRIX_VALUE_COUNT, 2, ENC_BIG_ENDIAN);
            proto_tree_add_item(aecp_tree, hf_aecp_matrix_item_offset, tvb,
                  AECP_OFFSET_MATRIX_ITEM_OFFSET, 2, ENC_BIG_ENDIAN);

            if ((mess_type == AECP_AEM_RESPONSE_MESSAGE) || (c_type == AECP_COMMAND_GET_MATRIX)) {
               /* TODO: dissect matrix values */
            }
            break;
         case AECP_COMMAND_START_STREAMING:
         case AECP_COMMAND_STOP_STREAMING:
         case AECP_COMMAND_REBOOT:
            proto_tree_add_item(aecp_tree, hf_aecp_descriptor_type, tvb,
                  AECP_OFFSET_STREAMING_DESCRIPTOR_TYPE, 2, ENC_BIG_ENDIAN);
            proto_tree_add_item(aecp_tree, hf_aecp_descriptor_index, tvb,
                  AECP_OFFSET_STREAMING_DESCRIPTOR_INDEX, 2, ENC_BIG_ENDIAN);
            break;
         case AECP_COMMAND_REGISTER_UNSOL_NOTIFICATION:
         case AECP_COMMAND_DEREGISTER_UNSOL_NOTIFICATION:
            /* No additional fields in these command types */
            break;
         case AECP_COMMAND_IDENTIFY_NOTIFICATION:
            proto_tree_add_item(aecp_tree, hf_aecp_descriptor_type, tvb,
                  AECP_OFFSET_IDENTIFY_NOTIFICATION_DESCRIPTOR_TYPE, 2, ENC_BIG_ENDIAN);
            proto_tree_add_item(aecp_tree, hf_aecp_descriptor_index, tvb,
                  AECP_OFFSET_IDENTIFY_NOTIFICATION_DESCRIPTOR_INDEX, 2, ENC_BIG_ENDIAN);
            break;
         case AECP_COMMAND_GET_AVB_INFO:
            proto_tree_add_item(aecp_tree, hf_aecp_descriptor_type, tvb,
                  AECP_OFFSET_AVB_INFO_DESCRIPTOR_TYPE, 2, ENC_BIG_ENDIAN);
            proto_tree_add_item(aecp_tree, hf_aecp_descriptor_index, tvb,
                  AECP_OFFSET_AVB_INFO_DESCRIPTOR_INDEX, 2, ENC_BIG_ENDIAN);
            if (mess_type == AECP_AEM_RESPONSE_MESSAGE) {
                proto_tree_add_item(aecp_tree, hf_aecp_avb_info_ptp_grandmaster_id, tvb,
                        AECP_OFFSET_AVB_INFO_AS_GRANDMASTER_ID, 8, ENC_BIG_ENDIAN);
                proto_tree_add_item(aecp_tree, hf_aecp_avb_info_propegation_delay, tvb,
                        AECP_OFFSET_AVB_INFO_PROPAGATION_DELAY, 4, ENC_BIG_ENDIAN);
                proto_tree_add_item(aecp_tree, hf_aecp_avb_info_gptp_domain_number, tvb,
                        AECP_OFFSET_AVB_INFO_AS_DOMAIN_NUMBER, 1, ENC_BIG_ENDIAN);
                proto_tree_add_item(aecp_tree, hf_aecp_as_capable_flag, tvb,
                        AECP_OFFSET_AVB_INFO_FLAGS, 1, ENC_BIG_ENDIAN);
                proto_tree_add_item(aecp_tree, hf_aecp_gptp_enabled_flag, tvb,
                        AECP_OFFSET_AVB_INFO_FLAGS, 1, ENC_BIG_ENDIAN);
                proto_tree_add_item(aecp_tree, hf_aecp_srp_enabled_flag, tvb,
                        AECP_OFFSET_AVB_INFO_FLAGS, 1, ENC_BIG_ENDIAN);
                proto_tree_add_item(aecp_tree, hf_aecp_avb_info_msrp_mappings_count, tvb,
                        AECP_OFFSET_AVB_INFO_MSRP_MAPPINGS_COUNT, 2, ENC_BIG_ENDIAN);

                /* prepare msrp mappings subtree */
                mr_item = proto_tree_add_item(aecp_tree, hf_aecp_avb_info_msrp_mappings, tvb,
                               0, 0, ENC_NA);
                mr_subtree = proto_item_add_subtree(mr_item, ett_aecp_get_avb_info_msrp_mappings);
                mr_counter = tvb_get_ntohs(tvb, AECP_OFFSET_AVB_INFO_MSRP_MAPPINGS_COUNT);

                mr_offset = AECP_OFFSET_AVB_INFO_MSRP_MAPPINGS;
                for (i = 0; i < mr_counter; i++) {
                    proto_tree_add_item(mr_subtree, hf_aecp_avb_info_msrp_mapping_traffic_class, tvb,
                       mr_offset, 1, ENC_BIG_ENDIAN);
                    mr_offset += 1;
                    proto_tree_add_item(mr_subtree, hf_aecp_avb_info_msrp_mapping_priority, tvb,
                       mr_offset, 1, ENC_BIG_ENDIAN);
                    mr_offset += 1;
                    proto_tree_add_item(mr_subtree, hf_aecp_get_avb_info_msrp_vlan_id, tvb,
                       mr_offset, 2, ENC_BIG_ENDIAN);
                    mr_offset += 2;
                }
            }
            break;
         case AECP_COMMAND_GET_AS_PATH:
            proto_tree_add_item(aecp_tree, hf_aecp_descriptor_index, tvb,
                  AECP_OFFSET_AS_PATH_DESCRIPTOR_INDEX, 2, ENC_BIG_ENDIAN);
            if (mess_type == AECP_AEM_RESPONSE_MESSAGE) {
                proto_tree_add_item(aecp_tree, hf_aecp_as_path_count, tvb,
                    AECP_OFFSET_AS_PATH_COUNT, 2, ENC_BIG_ENDIAN);
                mr_item = proto_tree_add_item(aecp_tree, hf_aecp_as_path_sequences, tvb,
                               0, 0, ENC_NA);
                mr_subtree = proto_item_add_subtree(mr_item, ett_aecp_get_as_path_sequences);
                mr_counter = tvb_get_ntohs(tvb, AECP_OFFSET_AS_PATH_COUNT);
                mr_offset = AECP_OFFSET_AS_PATH_PATH_SEQUENCE;
                for (i = 0; i < mr_counter; i++) {
                    proto_tree_add_item(mr_subtree, hf_aecp_get_as_info_clock_id, tvb,
                       mr_offset, 8, ENC_BIG_ENDIAN);
                    mr_offset += 8;
                }
            }
            break;
         case AECP_COMMAND_GET_COUNTERS:
            proto_tree_add_item(aecp_tree, hf_aecp_descriptor_type, tvb,
                  AECP_OFFSET_COUNTERS_DESCRIPTOR_TYPE, 2, ENC_BIG_ENDIAN);
            proto_tree_add_item(aecp_tree, hf_aecp_descriptor_index, tvb,
                  AECP_OFFSET_COUNTERS_DESCRIPTOR_INDEX, 2, ENC_BIG_ENDIAN);
            if (mess_type == AECP_AEM_RESPONSE_MESSAGE) {

               /*flags_ti = */ proto_tree_add_item(aecp_tree, hf_aecp_flags_32, tvb,
                     AECP_OFFSET_COUNTERS_VALID, 4, ENC_BIG_ENDIAN);
               /*flags_tree = proto_item_add_subtree(flags_ti, ett_acmp_flags);*/

               /* begin counters_valid flags field */
               switch (tvb_get_ntohs(tvb, AECP_OFFSET_COUNTERS_DESCRIPTOR_TYPE)) {
               case AEM_DESCRIPTOR_ENTITY:
                  break;
               case AEM_DESCRIPTOR_AVB_INTERFACE:
                  proto_tree_add_item(aecp_tree, hf_aecp_avb_interface_link_up_valid, tvb,
                        AECP_OFFSET_COUNTERS_VALID, 4, ENC_BIG_ENDIAN);
                  proto_tree_add_item(aecp_tree, hf_aecp_avb_interface_link_down_valid, tvb,
                        AECP_OFFSET_COUNTERS_VALID, 4, ENC_BIG_ENDIAN);
                  proto_tree_add_item(aecp_tree, hf_aecp_avb_interface_packets_tx_valid, tvb,
                        AECP_OFFSET_COUNTERS_VALID, 4, ENC_BIG_ENDIAN);
                  proto_tree_add_item(aecp_tree, hf_aecp_avb_interface_packets_rx_valid, tvb,
                        AECP_OFFSET_COUNTERS_VALID, 4, ENC_BIG_ENDIAN);
                  proto_tree_add_item(aecp_tree, hf_aecp_avb_interface_rx_crc_error_valid, tvb,
                        AECP_OFFSET_COUNTERS_VALID, 4, ENC_BIG_ENDIAN);
                  proto_tree_add_item(aecp_tree, hf_aecp_avb_interface_gptp_gm_changed_valid, tvb,
                        AECP_OFFSET_COUNTERS_VALID, 4, ENC_BIG_ENDIAN);
                  break;
               case AEM_DESCRIPTOR_CLOCK_DOMAIN:
                  proto_tree_add_item(aecp_tree, hf_aecp_clock_domain_locked_valid, tvb,
                        AECP_OFFSET_COUNTERS_VALID, 4, ENC_BIG_ENDIAN);
                  proto_tree_add_item(aecp_tree, hf_aecp_clock_domain_unlocked_valid, tvb,
                        AECP_OFFSET_COUNTERS_VALID, 4, ENC_BIG_ENDIAN);
                  break;
               case AEM_DESCRIPTOR_STREAM_INPUT:
                  proto_tree_add_item(aecp_tree, hf_aecp_stream_input_media_locked_valid, tvb,
                        AECP_OFFSET_COUNTERS_VALID, 4, ENC_BIG_ENDIAN);
                  proto_tree_add_item(aecp_tree, hf_aecp_stream_input_media_unlocked_valid, tvb,
                        AECP_OFFSET_COUNTERS_VALID, 4, ENC_BIG_ENDIAN);
                  proto_tree_add_item(aecp_tree, hf_aecp_stream_input_stream_reset_valid, tvb,
                        AECP_OFFSET_COUNTERS_VALID, 4, ENC_BIG_ENDIAN);
                  proto_tree_add_item(aecp_tree, hf_aecp_stream_input_seq_num_mismatch_valid, tvb,
                        AECP_OFFSET_COUNTERS_VALID, 4, ENC_BIG_ENDIAN);
                  proto_tree_add_item(aecp_tree, hf_aecp_stream_input_media_reset_valid, tvb,
                        AECP_OFFSET_COUNTERS_VALID, 4, ENC_BIG_ENDIAN);
                  proto_tree_add_item(aecp_tree, hf_aecp_stream_input_timestamp_uncertain_valid, tvb,
                        AECP_OFFSET_COUNTERS_VALID, 4, ENC_BIG_ENDIAN);
                  proto_tree_add_item(aecp_tree, hf_aecp_stream_input_timestamp_valid_valid, tvb,
                        AECP_OFFSET_COUNTERS_VALID, 4, ENC_BIG_ENDIAN);
                  proto_tree_add_item(aecp_tree, hf_aecp_stream_input_timestamp_not_valid_valid, tvb,
                        AECP_OFFSET_COUNTERS_VALID, 4, ENC_BIG_ENDIAN);
                  proto_tree_add_item(aecp_tree, hf_aecp_stream_input_unsupported_format_valid, tvb,
                        AECP_OFFSET_COUNTERS_VALID, 4, ENC_BIG_ENDIAN);
                  proto_tree_add_item(aecp_tree, hf_aecp_stream_input_late_timestamp_valid, tvb,
                        AECP_OFFSET_COUNTERS_VALID, 4, ENC_BIG_ENDIAN);
                  proto_tree_add_item(aecp_tree, hf_aecp_stream_input_early_timestamp_valid, tvb,
                        AECP_OFFSET_COUNTERS_VALID, 4, ENC_BIG_ENDIAN);
                  proto_tree_add_item(aecp_tree, hf_aecp_stream_input_packets_tx_valid, tvb,
                        AECP_OFFSET_COUNTERS_VALID, 4, ENC_BIG_ENDIAN);
                  proto_tree_add_item(aecp_tree, hf_aecp_stream_input_packets_rx_valid, tvb,
                        AECP_OFFSET_COUNTERS_VALID, 4, ENC_BIG_ENDIAN);
                  break;
               default:
                  break;
               }

               proto_tree_add_item(aecp_tree, hf_aecp_entity_specific1_valid, tvb,
                     AECP_OFFSET_COUNTERS_VALID, 4, ENC_BIG_ENDIAN);
               proto_tree_add_item(aecp_tree, hf_aecp_entity_specific2_valid, tvb,
                     AECP_OFFSET_COUNTERS_VALID, 4, ENC_BIG_ENDIAN);
               proto_tree_add_item(aecp_tree, hf_aecp_entity_specific3_valid, tvb,
                     AECP_OFFSET_COUNTERS_VALID, 4, ENC_BIG_ENDIAN);
               proto_tree_add_item(aecp_tree, hf_aecp_entity_specific4_valid, tvb,
                     AECP_OFFSET_COUNTERS_VALID, 4, ENC_BIG_ENDIAN);
               proto_tree_add_item(aecp_tree, hf_aecp_entity_specific5_valid, tvb,
                     AECP_OFFSET_COUNTERS_VALID, 4, ENC_BIG_ENDIAN);
               proto_tree_add_item(aecp_tree, hf_aecp_entity_specific6_valid, tvb,
                     AECP_OFFSET_COUNTERS_VALID, 4, ENC_BIG_ENDIAN);
               proto_tree_add_item(aecp_tree, hf_aecp_entity_specific7_valid, tvb,
                     AECP_OFFSET_COUNTERS_VALID, 4, ENC_BIG_ENDIAN);
               proto_tree_add_item(aecp_tree, hf_aecp_entity_specific8_valid, tvb,
                     AECP_OFFSET_COUNTERS_VALID, 4, ENC_BIG_ENDIAN);


               /* end counters_valid flags field */


               switch (tvb_get_ntohs(tvb, AECP_OFFSET_COUNTERS_DESCRIPTOR_TYPE)) {
               case AEM_DESCRIPTOR_ENTITY:
                  break;
               case AEM_DESCRIPTOR_AVB_INTERFACE:
                  proto_tree_add_item(aecp_tree, hf_aecp_avb_interface_link_up, tvb,
                        AECP_OFFSET_COUNTERS_AVB_INTERFACE_LINK_UP, 4, ENC_BIG_ENDIAN);
                  proto_tree_add_item(aecp_tree, hf_aecp_avb_interface_link_down, tvb,
                        AECP_OFFSET_COUNTERS_AVB_INTERFACE_LINK_DOWN, 4, ENC_BIG_ENDIAN);
                  proto_tree_add_item(aecp_tree, hf_aecp_avb_interface_packets_tx, tvb,
                        AECP_OFFSET_COUNTERS_AVB_INTERFACE_PACKETS_TX, 4, ENC_BIG_ENDIAN);
                  proto_tree_add_item(aecp_tree, hf_aecp_avb_interface_packets_rx, tvb,
                        AECP_OFFSET_COUNTERS_AVB_INTERFACE_PACKETS_RX, 4, ENC_BIG_ENDIAN);
                  proto_tree_add_item(aecp_tree, hf_aecp_avb_interface_rx_crc_error, tvb,
                        AECP_OFFSET_COUNTERS_AVB_INTERFACE_RX_CRC_ERROR, 4, ENC_BIG_ENDIAN);
                  proto_tree_add_item(aecp_tree, hf_aecp_avb_interface_gptp_gm_changed, tvb,
                        AECP_OFFSET_COUNTERS_AVB_INTERFACE_GPTP_GM_CHANGED, 4, ENC_BIG_ENDIAN);
                  break;
               case AEM_DESCRIPTOR_CLOCK_DOMAIN:
                  proto_tree_add_item(aecp_tree, hf_aecp_clock_domain_locked, tvb,
                        AECP_OFFSET_COUNTERS_CLOCK_DOMAIN_LOCKED, 4, ENC_BIG_ENDIAN);
                  proto_tree_add_item(aecp_tree, hf_aecp_clock_domain_unlocked, tvb,
                        AECP_OFFSET_COUNTERS_CLOCK_DOMAIN_UNLOCKED, 4, ENC_BIG_ENDIAN);
                  break;
               case AEM_DESCRIPTOR_STREAM_INPUT:
                  proto_tree_add_item(aecp_tree, hf_aecp_stream_input_media_locked, tvb,
                        AECP_OFFSET_COUNTERS_STREAM_INPUT_MEDIA_LOCKED, 4, ENC_BIG_ENDIAN);
                  proto_tree_add_item(aecp_tree, hf_aecp_stream_input_media_unlocked, tvb,
                        AECP_OFFSET_COUNTERS_STREAM_INPUT_MEDIA_UNLOCKED, 4, ENC_BIG_ENDIAN);
                  proto_tree_add_item(aecp_tree, hf_aecp_stream_input_stream_reset, tvb,
                        AECP_OFFSET_COUNTERS_STREAM_INPUT_STREAM_RESET, 4, ENC_BIG_ENDIAN);
                  proto_tree_add_item(aecp_tree, hf_aecp_stream_input_seq_num_mismatch, tvb,
                        AECP_OFFSET_COUNTERS_STREAM_INPUT_SEQ_NUM_MISMATCH, 4, ENC_BIG_ENDIAN);
                  proto_tree_add_item(aecp_tree, hf_aecp_stream_input_media_reset, tvb,
                        AECP_OFFSET_COUNTERS_STREAM_INPUT_MEDIA_RESET, 4, ENC_BIG_ENDIAN);
                  proto_tree_add_item(aecp_tree, hf_aecp_stream_input_timestamp_uncertain, tvb,
                        AECP_OFFSET_COUNTERS_STREAM_INPUT_TIMESTAMP_UNCERTAIN, 4, ENC_BIG_ENDIAN);
                  proto_tree_add_item(aecp_tree, hf_aecp_stream_input_timestamp_valid, tvb,
                        AECP_OFFSET_COUNTERS_STREAM_INPUT_TIMESTAMP_VALID, 4, ENC_BIG_ENDIAN);
                  proto_tree_add_item(aecp_tree, hf_aecp_stream_input_timestamp_not_valid, tvb,
                        AECP_OFFSET_COUNTERS_STREAM_INPUT_TIMESTAMP_NOT_VALID, 4, ENC_BIG_ENDIAN);
                  proto_tree_add_item(aecp_tree, hf_aecp_stream_input_unsupported_format, tvb,
                        AECP_OFFSET_COUNTERS_STREAM_INPUT_UNSUPPORTED_FORMAT, 4, ENC_BIG_ENDIAN);
                  proto_tree_add_item(aecp_tree, hf_aecp_stream_input_late_timestamp, tvb,
                        AECP_OFFSET_COUNTERS_STREAM_INPUT_LATE_TIMESTAMP, 4, ENC_BIG_ENDIAN);
                  proto_tree_add_item(aecp_tree, hf_aecp_stream_input_early_timestamp, tvb,
                        AECP_OFFSET_COUNTERS_STREAM_INPUT_EARLY_TIMESTAMP, 4, ENC_BIG_ENDIAN);
                  proto_tree_add_item(aecp_tree, hf_aecp_stream_input_packets_tx, tvb,
                        AECP_OFFSET_COUNTERS_STREAM_INPUT_PACKETS_TX, 4, ENC_BIG_ENDIAN);
                  proto_tree_add_item(aecp_tree, hf_aecp_stream_input_packets_rx, tvb,
                        AECP_OFFSET_COUNTERS_STREAM_INPUT_PACKETS_RX, 4, ENC_BIG_ENDIAN);
                  break;
               default:
                  break;
               }

               proto_tree_add_item(aecp_tree, hf_aecp_entity_specific1, tvb,
                     AECP_OFFSET_COUNTERS_ENTITY_SPECIFIC_1, 4, ENC_BIG_ENDIAN);
               proto_tree_add_item(aecp_tree, hf_aecp_entity_specific2, tvb,
                     AECP_OFFSET_COUNTERS_ENTITY_SPECIFIC_2, 4, ENC_BIG_ENDIAN);
               proto_tree_add_item(aecp_tree, hf_aecp_entity_specific3, tvb,
                     AECP_OFFSET_COUNTERS_ENTITY_SPECIFIC_3, 4, ENC_BIG_ENDIAN);
               proto_tree_add_item(aecp_tree, hf_aecp_entity_specific4, tvb,
                     AECP_OFFSET_COUNTERS_ENTITY_SPECIFIC_4, 4, ENC_BIG_ENDIAN);
               proto_tree_add_item(aecp_tree, hf_aecp_entity_specific5, tvb,
                     AECP_OFFSET_COUNTERS_ENTITY_SPECIFIC_5, 4, ENC_BIG_ENDIAN);
               proto_tree_add_item(aecp_tree, hf_aecp_entity_specific6, tvb,
                     AECP_OFFSET_COUNTERS_ENTITY_SPECIFIC_6, 4, ENC_BIG_ENDIAN);
               proto_tree_add_item(aecp_tree, hf_aecp_entity_specific7, tvb,
                     AECP_OFFSET_COUNTERS_ENTITY_SPECIFIC_7, 4, ENC_BIG_ENDIAN);
               proto_tree_add_item(aecp_tree, hf_aecp_entity_specific8, tvb,
                     AECP_OFFSET_COUNTERS_ENTITY_SPECIFIC_8, 4, ENC_BIG_ENDIAN);

            }
            break;
         case AECP_COMMAND_GET_AUDIO_MAP:
         case AECP_COMMAND_GET_VIDEO_MAP:
         case AECP_COMMAND_GET_SENSOR_MAP:
            proto_tree_add_item(aecp_tree, hf_aecp_descriptor_type, tvb,
                  AECP_OFFSET_MAP_DESCRIPTOR_TYPE, 2, ENC_BIG_ENDIAN);
            proto_tree_add_item(aecp_tree, hf_aecp_descriptor_index, tvb,
                  AECP_OFFSET_MAP_DESCRIPTOR_INDEX, 2, ENC_BIG_ENDIAN);
            proto_tree_add_item(aecp_tree, hf_aecp_map_index, tvb,
                  AECP_OFFSET_MAP_MAP_INDEX, 2, ENC_BIG_ENDIAN);
            if (mess_type == AECP_AEM_RESPONSE_MESSAGE) {
               proto_tree_add_item(aecp_tree, hf_aecp_number_of_maps, tvb,
                     AECP_OFFSET_MAP_NUMBER_OF_MAPS, 2, ENC_BIG_ENDIAN);
               proto_tree_add_item(aecp_tree, hf_aecp_number_of_maps, tvb,
                     AECP_OFFSET_MAP_NUMBER_OF_MAPPINGS, 2, ENC_BIG_ENDIAN);
               /* TODO: dissect mappings */
            }
            break;
         case AECP_COMMAND_ADD_AUDIO_MAPPINGS:
         case AECP_COMMAND_REMOVE_AUDIO_MAPPINGS:
         case AECP_COMMAND_ADD_VIDEO_MAPPINGS:
         case AECP_COMMAND_REMOVE_VIDEO_MAPPINGS:
         case AECP_COMMAND_ADD_SENSOR_MAPPINGS:
         case AECP_COMMAND_REMOVE_SENSOR_MAPPINGS:
            proto_tree_add_item(aecp_tree, hf_aecp_descriptor_type, tvb,
                  AECP_OFFSET_ADDREM_MAP_DESCRIPTOR_TYPE, 2, ENC_BIG_ENDIAN);
            proto_tree_add_item(aecp_tree, hf_aecp_descriptor_index, tvb,
                  AECP_OFFSET_ADDREM_MAP_DESCRIPTOR_INDEX, 2, ENC_BIG_ENDIAN);
            proto_tree_add_item(aecp_tree, hf_aecp_number_of_maps, tvb,
                  AECP_OFFSET_ADDREM_MAP_NUMBER_OF_MAPPINGS, 2, ENC_BIG_ENDIAN);
            /* TODO: dissect mappings */
            break;
         case AECP_COMMAND_START_OPERATION:
            proto_tree_add_item(aecp_tree, hf_aecp_descriptor_type, tvb,
                  AECP_OFFSET_START_OPERATION_DESCRIPTOR_TYPE, 2, ENC_BIG_ENDIAN);
            proto_tree_add_item(aecp_tree, hf_aecp_descriptor_index, tvb,
                  AECP_OFFSET_START_OPERATION_DESCRIPTOR_INDEX, 2, ENC_BIG_ENDIAN);
            proto_tree_add_item(aecp_tree, hf_aecp_operation_id, tvb,
                  AECP_OFFSET_START_OPERATION_OPERATION_ID, 2, ENC_BIG_ENDIAN);
            proto_tree_add_item(aecp_tree, hf_aecp_operation_type, tvb,
                  AECP_OFFSET_START_OPERATION_OPERATION_TYPE, 2, ENC_BIG_ENDIAN);
            /* TODO: add values support when operation types are defined */
            break;
         case AECP_COMMAND_ABORT_OPERATION:
            proto_tree_add_item(aecp_tree, hf_aecp_descriptor_type, tvb,
                  AECP_OFFSET_ABORT_OPERATION_DESCRIPTOR_TYPE, 2, ENC_BIG_ENDIAN);
            proto_tree_add_item(aecp_tree, hf_aecp_descriptor_index, tvb,
                  AECP_OFFSET_ABORT_OPERATION_DESCRIPTOR_INDEX, 2, ENC_BIG_ENDIAN);
            proto_tree_add_item(aecp_tree, hf_aecp_operation_id, tvb,
                  AECP_OFFSET_ABORT_OPERATION_OPERATION_ID, 2, ENC_BIG_ENDIAN);
            break;
         case AECP_COMMAND_OPERATION_STATUS:
            proto_tree_add_item(aecp_tree, hf_aecp_descriptor_type, tvb,
                  AECP_OFFSET_OPERATION_STATUS_DESCRIPTOR_TYPE, 2, ENC_BIG_ENDIAN);
            proto_tree_add_item(aecp_tree, hf_aecp_descriptor_index, tvb,
                  AECP_OFFSET_OPERATION_STATUS_DESCRIPTOR_INDEX, 2, ENC_BIG_ENDIAN);
            proto_tree_add_item(aecp_tree, hf_aecp_operation_id, tvb,
                  AECP_OFFSET_OPERATION_STATUS_OPERATION_ID, 2, ENC_BIG_ENDIAN);
            proto_tree_add_item(aecp_tree, hf_aecp_percent_complete, tvb,
                  AECP_OFFSET_OPERATION_STATUS_PERCENT_COMPLETE, 2, ENC_BIG_ENDIAN);
            break;
         case AECP_COMMAND_AUTH_ADD_KEY:
         case AECP_COMMAND_AUTH_DELETE_KEY:
         case AECP_COMMAND_AUTH_GET_KEY:
            proto_tree_add_item(aecp_tree, hf_aecp_key_eui, tvb,
                  AECP_OFFSET_AUTH_KEY_KEY_EUI, 8, ENC_BIG_ENDIAN);
            proto_tree_add_item(aecp_tree, hf_aecp_key_type, tvb,
                  AECP_OFFSET_AUTH_KEY_KEY_TYPE, 1, ENC_BIG_ENDIAN);
            proto_tree_add_item(aecp_tree, hf_aecp_key_length, tvb,
                  AECP_OFFSET_AUTH_KEY_KEY_LENGTH, 2, ENC_BIG_ENDIAN);

            if (((mess_type == AECP_AEM_RESPONSE_MESSAGE) && (c_type == AECP_COMMAND_AUTH_GET_KEY)) ||
                ((mess_type == AECP_AEM_COMMAND_MESSAGE) && (c_type == AECP_COMMAND_AUTH_ADD_KEY))) {

               mr_counter = tvb_get_ntohs(tvb, AECP_OFFSET_AUTH_KEY_KEY_LENGTH) & AECP_KEY_LENGTH_MASK;
               proto_tree_add_item(aecp_tree, hf_aecp_key, tvb,
                     AECP_OFFSET_AUTH_KEY_KEY, mr_counter, ENC_NA);
            }
            break;
         case AECP_COMMAND_AUTH_GET_KEY_LIST:
            proto_tree_add_item(aecp_tree, hf_aecp_keychain_id_small, tvb,
                  AECP_OFFSET_AUTH_GET_KEY_LIST_KEYCHAIN_ID, 1, ENC_BIG_ENDIAN);
            if (mess_type == AECP_AEM_RESPONSE_MESSAGE) {
               proto_tree_add_item(aecp_tree, hf_aecp_keychain_id_small, tvb,
                     AECP_OFFSET_AUTH_GET_KEY_LIST_NUMBER_OF_KEYS, 1, ENC_BIG_ENDIAN);
            }
            break;
         case AECP_COMMAND_AUTH_ADD_KEY_TO_CHAIN:
         case AECP_COMMAND_AUTH_DELETE_KEY_FROM_CHAIN:
            proto_tree_add_item(aecp_tree, hf_aecp_keychain_id, tvb,
                  AECP_OFFSET_AUTH_KEYCHAIN_KEYCHAIN_ID, 2, ENC_BIG_ENDIAN);
            proto_tree_add_item(aecp_tree, hf_aecp_key_eui, tvb,
                  AECP_OFFSET_AUTH_KEYCHAIN_KEY_EUI, 8, ENC_BIG_ENDIAN);
            break;
         case AECP_COMMAND_AUTH_GET_KEYCHAIN_LIST:
            proto_tree_add_item(aecp_tree, hf_aecp_keychain_id, tvb,
                  AECP_OFFSET_AUTH_GET_KEYCHAIN_LIST_KEYCHAIN_ID, 2, ENC_BIG_ENDIAN);
            proto_tree_add_item(aecp_tree, hf_aecp_keychain_list_index, tvb,
                  AECP_OFFSET_AUTH_GET_KEYCHAIN_LIST_LIST_INDEX, 2, ENC_BIG_ENDIAN);
            if (mess_type == AECP_AEM_RESPONSE_MESSAGE) {
               proto_tree_add_item(aecp_tree, hf_aecp_keychain_number_of_lists, tvb,
                     AECP_OFFSET_AUTH_GET_KEYCHAIN_LIST_NUMBER_OF_LISTS, 2, ENC_BIG_ENDIAN);
               proto_tree_add_item(aecp_tree, hf_aecp_keychain_number_of_keys, tvb,
                     AECP_OFFSET_AUTH_GET_KEYCHAIN_LIST_NUMBER_OF_KEYS, 2, ENC_BIG_ENDIAN);
               /* TODO: dissect key euis */
            }
            break;
         case AECP_COMMAND_AUTH_GET_IDENTITY:
            proto_tree_add_item(aecp_tree, hf_aecp_key_eui, tvb,
                  AECP_OFFSET_AUTH_GET_IDENTITY_KEY_GUID, 8, ENC_BIG_ENDIAN);
            proto_tree_add_item(aecp_tree, hf_aecp_key_signature, tvb,
                  AECP_OFFSET_AUTH_GET_IDENTITY_SIGNATURE, 28, ENC_NA);
            break;
         case AECP_COMMAND_AUTH_ADD_TOKEN:
            if (mess_type == AECP_AEM_COMMAND_MESSAGE) {
               guint32 token_length = tvb_get_ntohl(tvb, AECP_OFFSET_AUTH_TOKEN_TOKEN_LENGTH);
               proto_tree_add_item(aecp_tree, hf_aecp_key_length, tvb,
                     AECP_OFFSET_AUTH_TOKEN_TOKEN_LENGTH, 2, ENC_BIG_ENDIAN);
               proto_tree_add_item(aecp_tree, hf_aecp_key_signature, tvb,
                     AECP_OFFSET_AUTH_TOKEN_AUTHENTICATION_TOKEN, token_length, ENC_NA);
            }
            break;
         case AECP_COMMAND_AUTH_DELETE_TOKEN:
            /* No command specific fields */
            break;
         case AECP_COMMAND_AUTHENTICATE:
         case AECP_COMMAND_DEAUTHENTICATE:
            proto_tree_add_item(aecp_tree, hf_aecp_descriptor_type, tvb,
                  AECP_OFFSET_AUTHENTICATE_DESCRIPTOR_TYPE, 2, ENC_BIG_ENDIAN);
            proto_tree_add_item(aecp_tree, hf_aecp_descriptor_index, tvb,
                  AECP_OFFSET_AUTHENTICATE_DESCRIPTOR_INDEX, 2, ENC_BIG_ENDIAN);
            proto_tree_add_item(aecp_tree, hf_aecp_token_length, tvb,
                  AECP_OFFSET_AUTHENTICATE_TOKEN_LENGTH, 2, ENC_BIG_ENDIAN);

            mr_counter = tvb_get_ntohs(tvb, AECP_OFFSET_AUTHENTICATE_TOKEN_LENGTH)
               & AECP_TOKEN_LENGTH_MASK;
            proto_tree_add_item(aecp_tree, hf_aecp_auth_token, tvb,
                  AECP_OFFSET_AUTHENTICATE_AUTH_TOKEN, mr_counter, ENC_NA);
            break;
         case AECP_COMMAND_ENABLE_TRANSPORT_SECURITY:
            proto_tree_add_item(aecp_tree, hf_aecp_key_eui, tvb,
                  AECP_OFFSET_TRANSPORT_SECURITY_KEY_EUI, 8, ENC_BIG_ENDIAN);
            break;
         case AECP_COMMAND_DISABLE_TRANSPORT_SECURITY:
            /* No command specific fields */
            break;
         case AECP_COMMAND_ENABLE_STREAM_ENCRYPTION:
         case AECP_COMMAND_DISABLE_STREAM_ENCRYPTION:
            proto_tree_add_item(aecp_tree, hf_aecp_descriptor_type, tvb,
                  AECP_OFFSET_STREAM_ENCRYPTION_DESCRIPTOR_TYPE, 2, ENC_BIG_ENDIAN);
            proto_tree_add_item(aecp_tree, hf_aecp_descriptor_index, tvb,
                  AECP_OFFSET_STREAM_ENCRYPTION_DESCRIPTOR_INDEX, 2, ENC_BIG_ENDIAN);
            if (c_type == AECP_COMMAND_ENABLE_STREAM_ENCRYPTION) {
              proto_tree_add_item(aecp_tree, hf_aecp_key_eui, tvb,
                    AECP_OFFSET_STREAM_ENCRYPTION_KEY_EUI, 8, ENC_BIG_ENDIAN);
            }
            break;

         case AECP_COMMAND_SET_STREAM_BACKUP:
         case AECP_COMMAND_GET_STREAM_BACKUP:
            proto_tree_add_item(aecp_tree, hf_aecp_descriptor_type, tvb,
                  AECP_OFFSET_STREAM_BACKUP_DESCRIPTOR_TYPE, 2, ENC_BIG_ENDIAN);
            proto_tree_add_item(aecp_tree, hf_aecp_descriptor_index, tvb,
                  AECP_OFFSET_STREAM_BACKUP_DESCRIPTOR_INDEX, 2, ENC_BIG_ENDIAN);
            proto_tree_add_item(aecp_tree, hf_aecp_backup_talker_entity_id_0, tvb,
                 AECP_OFFSET_STREAM_BACKUP_TALKER_ENTITY_ID_0, 8, ENC_BIG_ENDIAN);
            proto_tree_add_item(aecp_tree, hf_aecp_backup_talker_unique_id_0, tvb,
                  AECP_OFFSET_STREAM_BACKUP_TALKER_UNIQUE_ID_0, 2, ENC_BIG_ENDIAN);
            proto_tree_add_item(aecp_tree, hf_aecp_backup_talker_entity_id_1, tvb,
                 AECP_OFFSET_STREAM_BACKUP_TALKER_ENTITY_ID_1, 8, ENC_BIG_ENDIAN);
            proto_tree_add_item(aecp_tree, hf_aecp_backup_talker_unique_id_1, tvb,
                  AECP_OFFSET_STREAM_BACKUP_TALKER_UNIQUE_ID_1, 2, ENC_BIG_ENDIAN);
            proto_tree_add_item(aecp_tree, hf_aecp_backup_talker_entity_id_2, tvb,
                 AECP_OFFSET_STREAM_BACKUP_TALKER_ENTITY_ID_2, 8, ENC_BIG_ENDIAN);
            proto_tree_add_item(aecp_tree, hf_aecp_backup_talker_unique_id_2, tvb,
                  AECP_OFFSET_STREAM_BACKUP_TALKER_UNIQUE_ID_2, 2, ENC_BIG_ENDIAN);
            proto_tree_add_item(aecp_tree, hf_aecp_backedup_talker_entity_id, tvb,
                 AECP_OFFSET_STREAM_BACKEDUP_TALKER_ENTITY_ID, 8, ENC_BIG_ENDIAN);
            proto_tree_add_item(aecp_tree, hf_aecp_backedup_talker_unique_id, tvb,
                  AECP_OFFSET_STREAM_BACKEDUP_TALKER_UNIQUE_ID, 2, ENC_BIG_ENDIAN);
            break;
            /* * * * AEM COMMON FORMAT PACKETS * * * */
         case AECP_COMMAND_CONTROLLER_AVAILABLE:
            break;
         default:
            /* the command type is not one of the valid spec values */
            break;
      }
   } /* AECP AEM Command */
   else if ((mess_type == AECP_ADDRESS_ACCESS_COMMAND_MESSAGE) || (mess_type == AECP_ADDRESS_ACCESS_RESPONSE_MESSAGE))
   {
      proto_tree_add_item(aecp_tree, hf_aecp_aa_count, tvb,
         AECP_AA_OFFSET_COUNT, 2, ENC_BIG_ENDIAN);
      proto_tree_add_item(aecp_tree, hf_aecp_aa_tlv_mode, tvb,
         AECP_AA_OFFSET_TLVS_START, 1, ENC_BIG_ENDIAN);
      proto_tree_add_item(aecp_tree, hf_aecp_aa_tlv_length, tvb,
         AECP_AA_OFFSET_TLVS_START, 2, ENC_BIG_ENDIAN);
      proto_tree_add_item(aecp_tree, hf_aecp_aa_tlv_address, tvb,
         AECP_AA_OFFSET_TLVS_START+2, 8, ENC_BIG_ENDIAN);
   } /* AECP Vendor Unique Command/Response */
   else if ((mess_type == AECP_VENDOR_UNIQUE_COMMAND_MESSAGE) || (mess_type == AECP_VENDOR_UNIQUE_RESPONSE_MESSAGE))
   {
      proto_tree_add_item(aecp_tree, hf_aecp_vendor_unique_protocol_id, tvb,
         AECP_VUC_OFFSET_PROTOCOL_ID, 6, ENC_BIG_ENDIAN);

      /* attempt to dissect the payload specific data */
      next_tvb = tvb_new_subset_remaining(tvb, AECP_VUC_OFFSET_PROTOCOL_ID);
      vendor_unique_protocol_id = tvb_get_guint48(tvb, AECP_VUC_OFFSET_PROTOCOL_ID, ENC_BIG_ENDIAN);
      vendor_unique_protocol_id_string = wmem_strdup_printf(pinfo->pool, "%012" PRIx64, vendor_unique_protocol_id);
      dissector_try_string(vendor_unique_protocol_dissector_table, vendor_unique_protocol_id_string, next_tvb, pinfo, aecp_tree, NULL);
    }
}

static void
dissect_17221_adp(tvbuff_t *tvb, packet_info *pinfo _U_, proto_tree *adp_tree)
{
   proto_item *ent_cap_ti;
   proto_item *talk_cap_ti;
   proto_item *list_cap_ti;
   proto_item *cont_cap_ti;

   proto_tree *ent_cap_flags_tree;
   proto_tree *talk_cap_flags_tree;
   proto_tree *list_cap_flags_tree;
   proto_tree *cont_cap_flags_tree;


   proto_tree_add_item(adp_tree, hf_adp_message_type, tvb, ADP_VERSION_OFFSET, 1, ENC_BIG_ENDIAN);
   proto_tree_add_item(adp_tree, hf_adp_valid_time, tvb, ADP_VALID_TIME_OFFSET, 1, ENC_BIG_ENDIAN);
   proto_tree_add_item(adp_tree, hf_adp_cd_length, tvb, ADP_CD_LENGTH_OFFSET, 1, ENC_BIG_ENDIAN);
   proto_tree_add_item(adp_tree, hf_adp_entity_id, tvb, ADP_ENTITY_ID_OFFSET, 8, ENC_BIG_ENDIAN);
   proto_tree_add_item(adp_tree, hf_adp_entity_model_id, tvb, ADP_ENTITY_MODEL_ID_OFFSET, 8, ENC_BIG_ENDIAN);

   /* Subtree for entity_capabilities field */
   ent_cap_ti = proto_tree_add_item(adp_tree, hf_adp_entity_cap, tvb, ADP_ENTITY_CAP_OFFSET, 4, ENC_BIG_ENDIAN);
   ent_cap_flags_tree = proto_item_add_subtree(ent_cap_ti, ett_adp_ent_cap);

   proto_tree_add_item(ent_cap_flags_tree,
         hf_adp_entity_cap_efu_mode, tvb, ADP_ENTITY_CAP_OFFSET, 4, ENC_BIG_ENDIAN);
   proto_tree_add_item(ent_cap_flags_tree,
         hf_adp_entity_cap_address_access_supported, tvb, ADP_ENTITY_CAP_OFFSET, 4, ENC_BIG_ENDIAN);
   proto_tree_add_item(ent_cap_flags_tree,
         hf_adp_entity_cap_gateway_entity, tvb, ADP_ENTITY_CAP_OFFSET, 4, ENC_BIG_ENDIAN);
   proto_tree_add_item(ent_cap_flags_tree,
         hf_adp_entity_cap_aem_supported, tvb, ADP_ENTITY_CAP_OFFSET, 4, ENC_BIG_ENDIAN);
   proto_tree_add_item(ent_cap_flags_tree,
         hf_adp_entity_cap_legacy_avc, tvb, ADP_ENTITY_CAP_OFFSET, 4, ENC_BIG_ENDIAN);
   proto_tree_add_item(ent_cap_flags_tree,
         hf_adp_entity_cap_assoc_id_support, tvb, ADP_ENTITY_CAP_OFFSET, 4, ENC_BIG_ENDIAN);
   proto_tree_add_item(ent_cap_flags_tree,
         hf_adp_entity_cap_assoc_id_valid, tvb, ADP_ENTITY_CAP_OFFSET, 4, ENC_BIG_ENDIAN);
   proto_tree_add_item(ent_cap_flags_tree,
         hf_adp_entity_cap_vendor_unique, tvb, ADP_ENTITY_CAP_OFFSET, 4, ENC_BIG_ENDIAN);
   proto_tree_add_item(ent_cap_flags_tree,
         hf_adp_entity_cap_class_a_supported, tvb, ADP_ENTITY_CAP_OFFSET, 4, ENC_BIG_ENDIAN);
   proto_tree_add_item(ent_cap_flags_tree,
         hf_adp_entity_cap_class_b_supported, tvb, ADP_ENTITY_CAP_OFFSET, 4, ENC_BIG_ENDIAN);
   proto_tree_add_item(ent_cap_flags_tree,
         hf_adp_entity_cap_gptp_supported, tvb, ADP_ENTITY_CAP_OFFSET, 4, ENC_BIG_ENDIAN);

   proto_tree_add_item(adp_tree, hf_adp_talker_stream_srcs, tvb, ADP_TALKER_STREAM_SRCS_OFFSET, 2, ENC_BIG_ENDIAN);

   talk_cap_ti = proto_tree_add_item(adp_tree, hf_adp_talker_cap, tvb, ADP_TALKER_CAP_OFFSET, 2, ENC_BIG_ENDIAN);
   talk_cap_flags_tree = proto_item_add_subtree(talk_cap_ti, ett_adp_talk_cap);

   proto_tree_add_item(talk_cap_flags_tree,
         hf_adp_talk_cap_implement, tvb, ADP_TALKER_CAP_OFFSET, 2, ENC_BIG_ENDIAN);
   proto_tree_add_item(talk_cap_flags_tree,
         hf_adp_talk_cap_other_src, tvb, ADP_TALKER_CAP_OFFSET, 2, ENC_BIG_ENDIAN);
   proto_tree_add_item(talk_cap_flags_tree,
         hf_adp_talk_cap_control_src, tvb, ADP_TALKER_CAP_OFFSET, 2, ENC_BIG_ENDIAN);
   proto_tree_add_item(talk_cap_flags_tree,
         hf_adp_talk_cap_media_clk_src, tvb, ADP_TALKER_CAP_OFFSET, 2, ENC_BIG_ENDIAN);
   proto_tree_add_item(talk_cap_flags_tree,
         hf_adp_talk_cap_smpte_src, tvb, ADP_TALKER_CAP_OFFSET, 2, ENC_BIG_ENDIAN);
   proto_tree_add_item(talk_cap_flags_tree,
         hf_adp_talk_cap_midi_src, tvb, ADP_TALKER_CAP_OFFSET, 2, ENC_BIG_ENDIAN);
   proto_tree_add_item(talk_cap_flags_tree,
         hf_adp_talk_cap_audio_src, tvb, ADP_TALKER_CAP_OFFSET, 2, ENC_BIG_ENDIAN);
   proto_tree_add_item(talk_cap_flags_tree,
         hf_adp_talk_cap_video_src, tvb, ADP_TALKER_CAP_OFFSET, 2, ENC_BIG_ENDIAN);

   proto_tree_add_item(adp_tree, hf_adp_listener_stream_sinks,
         tvb, ADP_LISTENER_STREAM_SINKS_OFFSET, 2, ENC_BIG_ENDIAN);

   list_cap_ti = proto_tree_add_item(adp_tree, hf_adp_listener_cap, tvb, ADP_LISTENER_CAP_OFFSET, 2, ENC_BIG_ENDIAN);
   list_cap_flags_tree = proto_item_add_subtree(list_cap_ti, ett_adp_list_cap);

   proto_tree_add_item(list_cap_flags_tree,
         hf_adp_list_cap_implement, tvb, ADP_LISTENER_CAP_OFFSET, 2, ENC_BIG_ENDIAN);
   proto_tree_add_item(list_cap_flags_tree,
         hf_adp_list_cap_other_sink, tvb, ADP_LISTENER_CAP_OFFSET, 2, ENC_BIG_ENDIAN);
   proto_tree_add_item(list_cap_flags_tree,
         hf_adp_list_cap_control_sink, tvb, ADP_LISTENER_CAP_OFFSET, 2, ENC_BIG_ENDIAN);
   proto_tree_add_item(list_cap_flags_tree,
         hf_adp_list_cap_media_clk_sink, tvb, ADP_LISTENER_CAP_OFFSET, 2, ENC_BIG_ENDIAN);
   proto_tree_add_item(list_cap_flags_tree,
         hf_adp_list_cap_smpte_sink, tvb, ADP_LISTENER_CAP_OFFSET, 2, ENC_BIG_ENDIAN);
   proto_tree_add_item(list_cap_flags_tree,
         hf_adp_list_cap_midi_sink, tvb, ADP_LISTENER_CAP_OFFSET, 2, ENC_BIG_ENDIAN);
   proto_tree_add_item(list_cap_flags_tree,
         hf_adp_list_cap_audio_sink, tvb, ADP_LISTENER_CAP_OFFSET, 2, ENC_BIG_ENDIAN);
   proto_tree_add_item(list_cap_flags_tree,
         hf_adp_list_cap_video_sink, tvb, ADP_LISTENER_CAP_OFFSET, 2, ENC_BIG_ENDIAN);

   cont_cap_ti = proto_tree_add_item(adp_tree, hf_adp_controller_cap, tvb, ADP_CONTROLLER_CAP_OFFSET, 4, ENC_BIG_ENDIAN);
   cont_cap_flags_tree = proto_item_add_subtree(cont_cap_ti, ett_adp_cont_cap);

   proto_tree_add_item(cont_cap_flags_tree,
         hf_adp_cont_cap_implement, tvb, ADP_CONTROLLER_CAP_OFFSET, 4, ENC_BIG_ENDIAN);
   proto_tree_add_item(cont_cap_flags_tree,
         hf_adp_cont_cap_layer3_proxy, tvb, ADP_CONTROLLER_CAP_OFFSET, 4, ENC_BIG_ENDIAN);

   proto_tree_add_item(adp_tree, hf_adp_avail_index, tvb, ADP_AVAIL_INDEX_OFFSET, 4, ENC_BIG_ENDIAN);
   proto_tree_add_item(adp_tree, hf_adp_gptp_gm_id, tvb, ADP_AS_GM_ID_OFFSET, 8, ENC_BIG_ENDIAN);

   proto_tree_add_item(adp_tree, hf_adp_assoc_id, tvb, ADP_ASSOC_ID_OFFSET, 8, ENC_BIG_ENDIAN);
}

static void
dissect_17221_acmp(tvbuff_t *tvb, packet_info *pinfo _U_, proto_tree *acmp_tree)
{
   proto_item *flags_ti;
   proto_tree *flags_tree;

   proto_tree_add_item(acmp_tree, hf_acmp_message_type, tvb, ACMP_VERSION_OFFSET, 1, ENC_BIG_ENDIAN);
   proto_tree_add_item(acmp_tree, hf_acmp_status_field, tvb, ACMP_STATUS_FIELD_OFFSET, 1, ENC_BIG_ENDIAN);
   proto_tree_add_item(acmp_tree, hf_acmp_cd_length, tvb, ACMP_CD_LENGTH_OFFSET, 2, ENC_BIG_ENDIAN);
   proto_tree_add_item(acmp_tree, hf_acmp_stream_id, tvb, ACMP_STREAM_ID_OFFSET, 8, ENC_BIG_ENDIAN);
   proto_tree_add_item(acmp_tree, hf_acmp_controller_guid, tvb, ACMP_CONTROLLER_GUID_OFFSET, 8, ENC_BIG_ENDIAN);
   proto_tree_add_item(acmp_tree, hf_acmp_talker_guid, tvb, ACMP_TALKER_GUID_OFFSET, 8, ENC_BIG_ENDIAN);
   proto_tree_add_item(acmp_tree, hf_acmp_listener_guid, tvb, ACMP_LISTENER_GUID_OFFSET, 8, ENC_BIG_ENDIAN);
   proto_tree_add_item(acmp_tree, hf_acmp_talker_unique_id, tvb, ACMP_TALKER_UNIQUE_ID_OFFSET, 2, ENC_BIG_ENDIAN);
   proto_tree_add_item(acmp_tree, hf_acmp_listener_unique_id, tvb, ACMP_LISTENER_UNIQUE_ID_OFFSET, 2, ENC_BIG_ENDIAN);
   proto_tree_add_item(acmp_tree, hf_acmp_stream_dest_mac, tvb, ACMP_DEST_MAC_OFFSET, 6, ENC_NA);
   proto_tree_add_item(acmp_tree, hf_acmp_connection_count, tvb, ACMP_CONNECTION_COUNT_OFFSET, 2, ENC_BIG_ENDIAN);
   proto_tree_add_item(acmp_tree, hf_acmp_sequence_id, tvb, ACMP_SEQUENCE_ID_OFFSET, 2, ENC_BIG_ENDIAN);

   flags_ti = proto_tree_add_item(acmp_tree, hf_acmp_flags, tvb, ACMP_FLAGS_OFFSET, 2, ENC_BIG_ENDIAN);
   flags_tree = proto_item_add_subtree(flags_ti, ett_acmp_flags);

   proto_tree_add_item(flags_tree, hf_acmp_flags_class_b, tvb, ACMP_FLAGS_OFFSET, 2, ENC_BIG_ENDIAN);
   proto_tree_add_item(flags_tree, hf_acmp_flags_fast_connect, tvb, ACMP_FLAGS_OFFSET, 2, ENC_BIG_ENDIAN);
   proto_tree_add_item(flags_tree, hf_acmp_flags_saved_state, tvb, ACMP_FLAGS_OFFSET, 2, ENC_BIG_ENDIAN);
   proto_tree_add_item(flags_tree, hf_acmp_flags_streaming_wait, tvb, ACMP_FLAGS_OFFSET, 2, ENC_BIG_ENDIAN);
   proto_tree_add_item(flags_tree, hf_acmp_flags_supports_encrypted, tvb, ACMP_FLAGS_OFFSET, 2, ENC_BIG_ENDIAN);
   proto_tree_add_item(flags_tree, hf_acmp_flags_encrypted_pdu, tvb, ACMP_FLAGS_OFFSET, 2, ENC_BIG_ENDIAN);
   proto_tree_add_item(flags_tree, hf_acmp_flags_talker_failed, tvb, ACMP_FLAGS_OFFSET, 2, ENC_BIG_ENDIAN);

   proto_tree_add_item(acmp_tree, hf_acmp_vlan_id, tvb, ACMP_VLAN_ID_OFFSET, 2, ENC_BIG_ENDIAN);
}

static int
dissect_17221(tvbuff_t *tvb, packet_info *pinfo, proto_tree *tree, void* data _U_)
{
   guint8 subtype = 0;
   proto_item *ieee17221_item;
   proto_tree *ieee17221_tree;
   subtype = tvb_get_guint8(tvb, 0);
   subtype &= 0x7F;

   /* Make entries in Protocol column and Info column on summary display */
   col_set_str(pinfo->cinfo, COL_PROTOCOL, "IEEE1722-1");

   ieee17221_item = proto_tree_add_item(tree, proto_17221, tvb, 0, -1, ENC_NA);
   ieee17221_tree = proto_item_add_subtree(ieee17221_item, ett_17221);

   switch (subtype)
   {
      case 0x7A:
         {
            col_set_str(pinfo->cinfo, COL_INFO, "AVDECC Discovery Protocol");
            if (tree)
               dissect_17221_adp(tvb, pinfo, ieee17221_tree);
            break;
         }
      case 0x7B:
         {
            col_set_str(pinfo->cinfo, COL_INFO, "AVDECC Enumeration and Control Protocol");
            if (tree)
               dissect_17221_aecp(tvb, pinfo, ieee17221_tree);
            break;
         }
      case 0x7C:
         {
            col_set_str(pinfo->cinfo, COL_INFO, "AVDECC Connection Management Protocol");
            if (tree)
               dissect_17221_acmp(tvb, pinfo, ieee17221_tree);
            break;
         }
      default:
         {
            /* Shouldn't get here */
            col_set_str(pinfo->cinfo, COL_INFO, "1722.1 Unknown");
            return 0;
         }
   }

   return tvb_captured_length(tvb);
}

/* Register the protocol with Wireshark */
void
proto_register_17221(void)
{
   static hf_register_info hf[] = {
      { &hf_adp_message_type,
         { "Message Type", "ieee17221.message_type",
            FT_UINT8, BASE_DEC, VALS(adp_message_type_vals), ADP_MSG_TYPE_MASK, NULL, HFILL }
      },
      { &hf_adp_valid_time,
         { "Valid Time", "ieee17221.valid_time",
            FT_UINT8, BASE_DEC, NULL, ADP_VALID_TIME_MASK, NULL, HFILL }
      },
      { &hf_adp_cd_length,
         { "Control Data Length", "ieee17221.control_data_length",
            FT_UINT16, BASE_DEC, NULL, ADP_CD_LENGTH_MASK, NULL, HFILL }
      },
      { &hf_adp_entity_id,
         { "Entity ID", "ieee17221.entity_id",
            FT_UINT64, BASE_HEX, NULL, 0x0, NULL, HFILL }
      },
      { &hf_adp_entity_model_id,
         { "Entity Model ID", "ieee17221.entity_model_id",
            FT_UINT64, BASE_HEX, NULL, 0x0, NULL, HFILL }
      },
      { &hf_adp_entity_cap,
         { "Entity Capabilities", "ieee17221.entity_capabilities",
            FT_UINT32, BASE_HEX, NULL, 0x0, NULL, HFILL }
      },
      /* Entity Capability Flags Begin */
      { &hf_adp_entity_cap_efu_mode,
         { "EFU_MODE", "ieee17221.entity_capabilities.efu_mode",
            FT_BOOLEAN, 32, NULL, ADP_EFU_MODE_BITMASK, NULL, HFILL }
      },
      { &hf_adp_entity_cap_address_access_supported,
         { "ADDRESS_ACCESS", "ieee17221.entity_capabilities.address_access",
            FT_BOOLEAN, 32, NULL, ADP_ADDRESS_ACCESS_BITMASK, NULL, HFILL }
      },
      { &hf_adp_entity_cap_gateway_entity,
         { "GATEWAY_ENTITY", "ieee17221.entity_capabilities.gateway_entity",
            FT_BOOLEAN, 32, NULL, ADP_GATEWAY_ENTITY_BITMASK, NULL, HFILL }
      },
      { &hf_adp_entity_cap_aem_supported,
         { "AEM", "ieee17221.entity_capabilities.aem_supported",
            FT_BOOLEAN, 32, NULL, ADP_AEM_SUPPORTED_BITMASK, NULL, HFILL }
      },
      { &hf_adp_entity_cap_legacy_avc,
         { "LEGACY_AVC", "ieee17221.entity_capabilities.legacy_avc",
            FT_BOOLEAN, 32, NULL, ADP_LEGACY_AVC_BITMASK, NULL, HFILL }
      },
      { &hf_adp_entity_cap_assoc_id_support,
         { "ASSOCIATION_ID_SUPPORTED", "ieee17221.entity_capabilities.association_id_supported",
            FT_BOOLEAN, 32, NULL, ADP_ASSOC_ID_SUPPORT_BITMASK, NULL, HFILL }
      },
      { &hf_adp_entity_cap_assoc_id_valid,
         { "ASSOCIATION_ID_VALID", "ieee17221.entity_capabilities.association_id_valid",
            FT_BOOLEAN, 32, NULL, ADP_ASSOC_ID_VALID_BITMASK, NULL, HFILL }
      },
      { &hf_adp_entity_cap_vendor_unique,
         { "VENDOR_UNIQUE", "ieee17221.entity_capabilities.vendor_unique",
            FT_BOOLEAN, 32, NULL, ADP_VENDOR_UNIQUE_BITMASK, NULL, HFILL }
      },
      { &hf_adp_entity_cap_class_a_supported,
         { "CLASS_A", "ieee17221.entity_capabilities.class_a",
            FT_BOOLEAN, 32, NULL, ADP_CLASS_A_SUPPORTED_BITMASK, NULL, HFILL }
      },
      { &hf_adp_entity_cap_class_b_supported,
         { "CLASS_B", "ieee17221.entity_capabilities.class_b",
            FT_BOOLEAN, 32, NULL, ADP_CLASS_B_SUPPORTED_BITMASK, NULL, HFILL }
      },
      { &hf_adp_entity_cap_gptp_supported,
         { "gPTP Supported", "ieee17221.entity_capabilities.gptp_supported",
            FT_BOOLEAN, 32, NULL, ADP_AS_SUPPORTED_BITMASK, NULL, HFILL }
      },
      /* Entity Capability Flags End */
      { &hf_adp_talker_stream_srcs,
         { "Talker Stream Sources", "ieee17221.talker_stream_sources",
            FT_UINT16, BASE_DEC, NULL, 0x0, NULL, HFILL }
      },
      { &hf_adp_talker_cap,
         { "Talker Capabilities", "ieee17221.talker_capabilities",
            FT_UINT16, BASE_HEX, NULL, 0x0, NULL, HFILL }
      },
      /* Talker Capability Flags Begin */
      { &hf_adp_talk_cap_implement,
         { "IMPLEMENTED", "ieee17221.talker_capabilities.implemented",
            FT_BOOLEAN, 16, NULL, ADP_TALK_IMPLEMENTED_BITMASK, NULL, HFILL }
      },
      { &hf_adp_talk_cap_other_src,
         { "OTHER_SOURCE", "ieee17221.talker_capabilities.other_source",
            FT_BOOLEAN, 16, NULL, ADP_TALK_OTHER_SRC_BITMASK, NULL, HFILL }
      },
      { &hf_adp_talk_cap_control_src,
         { "CONTROL_SOURCE", "ieee17221.talker_capabilities.control_source",
            FT_BOOLEAN, 16, NULL, ADP_TALK_CONTROL_SRC_BITMASK, NULL, HFILL }
      },
      { &hf_adp_talk_cap_media_clk_src,
         { "MEDIA_CLOCK_SOURCE", "ieee17221.talker_capabilities.media_clock_source",
            FT_BOOLEAN, 16, NULL, ADP_TALK_MEDIA_CLK_SRC_BITMASK, NULL, HFILL }
      },
      { &hf_adp_talk_cap_smpte_src,
         { "SMPTE_SOURCE", "ieee17221.talker_capabilities.smpte_source",
            FT_BOOLEAN, 16, NULL, ADP_TALK_SMPTE_SRC_BITMASK, NULL, HFILL }
      },
      { &hf_adp_talk_cap_midi_src,
         { "MIDI_SOURCE", "ieee17221.talker_capabilities.midi_source",
            FT_BOOLEAN, 16, NULL, ADP_TALK_MIDI_SRC_BITMASK, NULL, HFILL }
      },
      { &hf_adp_talk_cap_audio_src,
         { "AUDIO_SOURCE", "ieee17221.talker_capabilities.audio_source",
            FT_BOOLEAN, 16, NULL, ADP_TALK_AUDIO_SRC_BITMASK, NULL, HFILL }
      },
      { &hf_adp_talk_cap_video_src,
         { "VIDEO_SOURCE", "ieee17221.talker_capabilities.video_source",
            FT_BOOLEAN, 16, NULL, ADP_TALK_VIDEO_SRC_BITMASK, NULL, HFILL }
      },
      /* Talker Capability Flags End */
      { &hf_adp_listener_stream_sinks,
         { "Listener Stream Sinks", "ieee17221.listener_stream_sinks",
            FT_UINT16, BASE_DEC, NULL, 0x0, NULL, HFILL }
      },
      { &hf_adp_listener_cap,
         { "Listener Capabilities", "ieee17221.listener_capabilities",
            FT_UINT16, BASE_HEX, NULL, 0x0, NULL, HFILL }
      },
      /* Listener Capability Flags Begin */
      { &hf_adp_list_cap_implement,
         { "IMPLEMENTED", "ieee17221.listener_capabilities.implemented",
            FT_BOOLEAN, 16, NULL, ADP_LIST_IMPLEMENTED_BITMASK, NULL, HFILL }
      },
      { &hf_adp_list_cap_other_sink,
         { "OTHER_SINK", "ieee17221.listener_capabilities.other_source",
            FT_BOOLEAN, 16, NULL, ADP_LIST_OTHER_SINK_BITMASK, NULL, HFILL }
      },
      { &hf_adp_list_cap_control_sink,
         { "CONTROL_SINK", "ieee17221.listener_capabilities.control_source",
            FT_BOOLEAN, 16, NULL, ADP_LIST_CONTROL_SINK_BITMASK, NULL, HFILL }
      },
      { &hf_adp_list_cap_media_clk_sink,
         { "MEDIA_CLOCK_SINK", "ieee17221.listener_capabilities.media_clock_source",
            FT_BOOLEAN, 16, NULL, ADP_LIST_MEDIA_CLK_SINK_BITMASK, NULL, HFILL }
      },
      { &hf_adp_list_cap_smpte_sink,
         { "SMPTE_SINK", "ieee17221.listener_capabilities.smpte_source",
            FT_BOOLEAN, 16, NULL, ADP_LIST_SMPTE_SINK_BITMASK, NULL, HFILL }
      },
      { &hf_adp_list_cap_midi_sink,
         { "MIDI_SINK", "ieee17221.listener_capabilities.midi_source",
            FT_BOOLEAN, 16, NULL, ADP_LIST_MIDI_SINK_BITMASK, NULL, HFILL }
      },
      { &hf_adp_list_cap_audio_sink,
         { "AUDIO_SINK", "ieee17221.listener_capabilities.audio_source",
            FT_BOOLEAN, 16, NULL, ADP_LIST_AUDIO_SINK_BITMASK, NULL, HFILL }
      },
      { &hf_adp_list_cap_video_sink,
         { "VIDEO_SINK", "ieee17221.listener_capabilities.video_source",
            FT_BOOLEAN, 16, NULL, ADP_LIST_VIDEO_SINK_BITMASK, NULL, HFILL }
      },
      /* Listener Capability Flags End */
      { &hf_adp_controller_cap,
         { "Controller Capabilities", "ieee17221.controller_capabilities",
            FT_UINT32, BASE_HEX, NULL, 0x0, NULL, HFILL }
      },
      /* Controller Capability Flags Begin */
      { &hf_adp_cont_cap_implement,
         { "IMPLEMENTED", "ieee17221.controller_capabilities.implemented",
            FT_BOOLEAN, 16, NULL, ADP_CONT_IMPLEMENTED_BITMASK, NULL, HFILL }
      },
      { &hf_adp_cont_cap_layer3_proxy,
         { "LAYER3_PROXY", "ieee17221.controller_capabilities.layer3_proxy",
            FT_BOOLEAN, 16, NULL, ADP_CONT_LAYER3_PROXY_BITMASK, NULL, HFILL }
      },
      /* Controller Capability Flags End */
      { &hf_adp_avail_index,
         { "Available Index", "ieee17221.available_index",
            FT_UINT32, BASE_HEX, NULL, 0x0, NULL, HFILL }
      },
      { &hf_adp_gptp_gm_id,
         { "gPTP Grandmaster ID", "ieee17221.gptp_grandmaster_id",
            FT_UINT64, BASE_HEX, NULL, 0x0, NULL, HFILL }
      },
#if 0
      { &hf_adp_def_aud_format,
         { "Default Audio Format", "ieee17221.default_audio_format",
            FT_UINT32, BASE_HEX, NULL, 0x0, NULL, HFILL }
      },
#endif
#if 0
      /* Default Audio Formats Fields Begin */
      { &hf_adp_def_aud_sample_rates,
         { "Sample Rates", "ieee17221.default_audio_format.sample_rates",
            FT_UINT8, BASE_HEX, NULL, ADP_DEF_AUDIO_SAMPLE_RATES_MASK, NULL, HFILL }
      },
      /* Sample rates Begin */
      { &hf_adp_samp_rate_44k1,
         { "44.1kHz", "ieee17221.default_audio_format.sample_rates.44k1",
            FT_BOOLEAN, 8, NULL, ADP_SAMP_RATE_44K1_BITMASK, NULL, HFILL }
      },
      { &hf_adp_samp_rate_48k,
         { "48kHz", "ieee17221.default_audio_format.sample_rates.48k",
            FT_BOOLEAN, 8, NULL, ADP_SAMP_RATE_48K_BITMASK, NULL, HFILL }
      },
      { &hf_adp_samp_rate_88k2,
         { "88.2kHz", "ieee17221.default_audio_format.sample_rates.88k2",
            FT_BOOLEAN, 8, NULL, ADP_SAMP_RATE_88K2_BITMASK, NULL, HFILL }
      },
      { &hf_adp_samp_rate_96k,
         { "96kHz", "ieee17221.default_audio_format.sample_rates.96k",
            FT_BOOLEAN, 8, NULL, ADP_SAMP_RATE_96K_BITMASK, NULL, HFILL }
      },
      { &hf_adp_samp_rate_176k4,
         { "176.4kHz", "ieee17221.default_audio_format.sample_rates.176k4",
            FT_BOOLEAN, 8, NULL, ADP_SAMP_RATE_176K4_BITMASK, NULL, HFILL }
      },
      { &hf_adp_samp_rate_192k,
         { "192kHz", "ieee17221.default_audio_format.sample_rates.192k",
            FT_BOOLEAN, 8, NULL, ADP_SAMP_RATE_192K_BITMASK, NULL, HFILL }
      },
      /* Sample rates End */
      { &hf_adp_def_aud_max_chan,
         { "Max Channels", "ieee17221.default_audio_format.max_channels",
            FT_UINT16, BASE_DEC, NULL, ADP_DEF_AUDIO_MAX_CHANS_MASK, NULL, HFILL }
      },
      { &hf_adp_def_aud_saf_flag,
         { "saf", "ieee17221.default_audio_format.saf",
            FT_BOOLEAN, 16, NULL, ADP_DEF_AUDIO_SAF_MASK, NULL, HFILL }
      },
      { &hf_adp_def_aud_float_flag,
         { "float", "ieee17221.default_audio_format.float",
            FT_BOOLEAN, 16, NULL, ADP_DEF_AUDIO_FLOAT_MASK, NULL, HFILL }
      },
      { &hf_adp_def_aud_chan_formats,
         { "Channel Formats", "ieee17221.default_audio_format.channel_formats",
            FT_UINT16, BASE_HEX, NULL, 0x0, NULL, HFILL }
      },
      /* Channel Formats Fields Start */
      { &hf_adp_chan_format_mono,
         { "MONO", "ieee17221.default_audio_format.channel_formats.mono",
            FT_BOOLEAN, 16, NULL, ADP_CHAN_FORMAT_MONO, NULL, HFILL }
      },
      { &hf_adp_chan_format_2ch,
         { "2_CH", "ieee17221.default_audio_format.channel_formats.2_ch",
            FT_BOOLEAN, 16, NULL, ADP_CHAN_FORMAT_2CH, NULL, HFILL }
      },
      { &hf_adp_chan_format_3ch,
         { "3_CH", "ieee17221.default_audio_format.channel_formats.3_ch",
            FT_BOOLEAN, 16, NULL, ADP_CHAN_FORMAT_3CH, NULL, HFILL }
      },
      { &hf_adp_chan_format_4ch,
         { "4_CH", "ieee17221.default_audio_format.channel_formats.4_ch",
            FT_BOOLEAN, 16, NULL, ADP_CHAN_FORMAT_4CH, NULL, HFILL }
      },
      { &hf_adp_chan_format_5ch,
         { "5_CH", "ieee17221.default_audio_format.channel_formats.5_ch",
            FT_BOOLEAN, 16, NULL, ADP_CHAN_FORMAT_5CH, NULL, HFILL }
      },
      { &hf_adp_chan_format_6ch,
         { "6_CH", "ieee17221.default_audio_format.channel_formats.6_ch",
            FT_BOOLEAN, 16, NULL, ADP_CHAN_FORMAT_6CH, NULL, HFILL }
      },
      { &hf_adp_chan_format_7ch,
         { "7_CH", "ieee17221.default_audio_format.channel_formats.7_ch",
            FT_BOOLEAN, 16, NULL, ADP_CHAN_FORMAT_7CH, NULL, HFILL }
      },
      { &hf_adp_chan_format_8ch,
         { "8_CH", "ieee17221.default_audio_format.channel_formats.8_ch",
            FT_BOOLEAN, 16, NULL, ADP_CHAN_FORMAT_8CH, NULL, HFILL }
      },
      { &hf_adp_chan_format_10ch,
         { "10_CH", "ieee17221.default_audio_format.channel_formats.10_ch",
            FT_BOOLEAN, 16, NULL, ADP_CHAN_FORMAT_10CH, NULL, HFILL }
      },
      { &hf_adp_chan_format_12ch,
         { "12_CH", "ieee17221.default_audio_format.channel_formats.12_ch",
            FT_BOOLEAN, 16, NULL, ADP_CHAN_FORMAT_12CH, NULL, HFILL }
      },
      { &hf_adp_chan_format_14ch,
         { "14_CH", "ieee17221.default_audio_format.channel_formats.14_ch",
            FT_BOOLEAN, 16, NULL, ADP_CHAN_FORMAT_14CH, NULL, HFILL }
      },
      { &hf_adp_chan_format_16ch,
         { "16_CH", "ieee17221.default_audio_format.channel_formats.16_ch",
            FT_BOOLEAN, 16, NULL, ADP_CHAN_FORMAT_16CH, NULL, HFILL }
      },
      { &hf_adp_chan_format_18ch,
         { "18_CH", "ieee17221.default_audio_format.channel_formats.18_ch",
            FT_BOOLEAN, 16, NULL, ADP_CHAN_FORMAT_18CH, NULL, HFILL }
      },
      { &hf_adp_chan_format_20ch,
         { "20_CH", "ieee17221.default_audio_format.channel_formats.20_ch",
            FT_BOOLEAN, 16, NULL, ADP_CHAN_FORMAT_20CH, NULL, HFILL }
      },
      { &hf_adp_chan_format_22ch,
         { "22_CH", "ieee17221.default_audio_format.channel_formats.22_ch",
            FT_BOOLEAN, 16, NULL, ADP_CHAN_FORMAT_22CH, NULL, HFILL }
      },
      { &hf_adp_chan_format_24ch,
         { "24_CH", "ieee17221.default_audio_format.channel_formats.24_ch",
            FT_BOOLEAN, 16, NULL, ADP_CHAN_FORMAT_24CH, NULL, HFILL }
      },
#endif
      /* Channel Formats Fields End */
      /* Default Audio Formats Fields End */
#if 0
      { &hf_adp_def_vid_format,
         { "Default Video Format", "ieee17221.default_video_format",
            FT_UINT32, BASE_HEX, NULL, 0x0, NULL, HFILL }
      },
#endif
      { &hf_adp_assoc_id,
         { "Association ID", "ieee17221.association_id",
            FT_UINT64, BASE_HEX, NULL, 0x0, NULL, HFILL }
      },
#if 0
      { &hf_adp_entity_type,
         { "Entity Type", "ieee17221.entity_type",
            FT_UINT32, BASE_HEX, NULL, 0x0, NULL, HFILL }
      },
#endif
      /*******************************************************************/
      { &hf_acmp_message_type,
         { "Message Type", "ieee17221.message_type",
            FT_UINT8, BASE_DEC, VALS(acmp_message_type_vals), ACMP_MSG_TYPE_MASK, NULL, HFILL }
      },
      { &hf_acmp_status_field,
         { "Status Field", "ieee17221.status_field",
            FT_UINT8, BASE_DEC, VALS(acmp_status_field_vals), ACMP_STATUS_FIELD_MASK, NULL, HFILL }
      },
      { &hf_acmp_cd_length,
         { "Control Data Length", "ieee17221.control_data_length",
           FT_UINT16, BASE_DEC, NULL, ACMP_CD_LENGTH_MASK, NULL, HFILL }
      },
      { &hf_acmp_stream_id,
         { "Stream ID", "ieee17221.stream_id",
            FT_UINT64, BASE_HEX, NULL, 0x0, NULL, HFILL }
      },
      { &hf_acmp_controller_guid,
         { "Controller GUID", "ieee17221.controller_guid",
            FT_UINT64, BASE_HEX, NULL, 0x0, NULL, HFILL }
      },
      { &hf_acmp_talker_guid,
         { "Talker GUID", "ieee17221.talker_guid",
            FT_UINT64, BASE_HEX, NULL, 0x0, NULL, HFILL }
      },
      { &hf_acmp_listener_guid,
         { "Listener GUID", "ieee17221.listener_guid",
            FT_UINT64, BASE_HEX, NULL, 0x0, NULL, HFILL }
      },
      { &hf_acmp_talker_unique_id,
         { "Talker Unique ID", "ieee17221.talker_unique_id",
            FT_UINT16, BASE_HEX, NULL, 0x0, NULL, HFILL }
      },
      { &hf_acmp_listener_unique_id,
         { "Listener Unique ID", "ieee17221.listener_unique_id",
            FT_UINT16, BASE_HEX, NULL, 0x0, NULL, HFILL }
      },
      { &hf_acmp_stream_dest_mac,
         { "Destination MAC address", "ieee17221.dest_mac",
            FT_ETHER, BASE_NONE, NULL, 0x0, NULL, HFILL }
      },
      { &hf_acmp_connection_count,
         { "Connection Count", "ieee17221.connection_count",
            FT_UINT16, BASE_DEC, NULL, 0x0, NULL, HFILL }
      },
      { &hf_acmp_sequence_id,
         { "Sequence ID", "ieee17221.sequence_id",
            FT_UINT16, BASE_HEX, NULL, 0x0, NULL, HFILL }
      },
      { &hf_acmp_flags,
         { "Flags", "ieee17221.flags",
            FT_UINT16, BASE_HEX, NULL, 0x0, NULL, HFILL }
      },
      /* ACMP Flags Begin */
      { &hf_acmp_flags_class_b,
         { "CLASS_B", "ieee17221.flags.class_b",
            FT_BOOLEAN, 16, NULL, ACMP_FLAG_CLASS_B_BITMASK, NULL, HFILL }
      },
      { &hf_acmp_flags_fast_connect,
         { "FAST_CONNECT", "ieee17221.flags.fast_connect",
            FT_BOOLEAN, 16, NULL, ACMP_FLAG_FAST_CONNECT_BITMASK, NULL, HFILL }
      },
      { &hf_acmp_flags_saved_state,
         { "SAVED_STATE", "ieee17221.flags.saved_state",
            FT_BOOLEAN, 16, NULL, ACMP_FLAG_SAVED_STATE_BITMASK, NULL, HFILL }
      },
      { &hf_acmp_flags_streaming_wait,
         { "STREAMING_WAIT", "ieee17221.flags.streaming_wait",
            FT_BOOLEAN, 16, NULL, ACMP_FLAG_STREAMING_WAIT_BITMASK, NULL, HFILL }
      },
      { &hf_acmp_flags_supports_encrypted,
         { "SUPPORTS_ENCRYPTED", "ieee17221.flags.supports_encrypted",
            FT_BOOLEAN, 16, NULL, ACMP_FLAG_SUPPORTS_ENCRYPTED_BITMASK, NULL, HFILL }
      },
      { &hf_acmp_flags_encrypted_pdu,
         { "ENCRYPTED_PDU", "ieee17221.flags.encrypted_pdu",
            FT_BOOLEAN, 16, NULL, ACMP_FLAG_ENCRYPTED_PDU_BITMASK, NULL, HFILL }
      },
      { &hf_acmp_flags_talker_failed,
         { "TALKER_FAILED", "ieee17221.flags.talker_failed",
            FT_BOOLEAN, 16, NULL, ACMP_FLAG_TALKER_FAILED_BITMASK, NULL, HFILL }
      },
      /* ACMP Flags End */
      { &hf_acmp_vlan_id,
         { "Stream VLAN Id", "ieee17221.vlan_id",
            FT_UINT16, BASE_HEX, NULL, 0x0, NULL, HFILL }
      },
      /******* AECP ******/
      /* COMMON FIELDS */
      { &hf_aecp_message_type,
         { "Message Type", "ieee17221.message_type",
            FT_UINT8, BASE_DEC, VALS(aecp_message_type_vals), AECP_MSG_TYPE_MASK, NULL, HFILL }
      },
      { &hf_aecp_cd_length,
         { "Control Data Length", "ieee17221.control_data_length",
            FT_UINT16, BASE_DEC, NULL, AECP_CD_LENGTH_MASK, NULL, HFILL }
      },
      { &hf_aecp_target_guid,
         { "Target GUID", "ieee17221.target_guid",
            FT_UINT64, BASE_HEX, NULL, 0x0, NULL, HFILL }
      },
      { &hf_aecp_controller_guid,
         { "Controller GUID", "ieee17221.controller_guid",
            FT_UINT64, BASE_HEX, NULL, 0x0, NULL, HFILL }
      },
      { &hf_aecp_sequence_id,
         { "Sequence ID", "ieee17221.sequence_id",
            FT_UINT16, BASE_DEC, NULL, 0x0, NULL, HFILL }
      },
      { &hf_aecp_u_flag,
         { "U Flag", "ieee17221.u_flag",
            FT_BOOLEAN, 8, NULL, AECP_U_FLAG_MASK, NULL, HFILL }
      },
      { &hf_aecp_command_type,
         { "Command Type", "ieee17221.command_type",
            FT_UINT16, BASE_HEX, VALS(aecp_command_type_vals), AECP_COMMAND_TYPE_MASK, NULL, HFILL }
      },
      /* Address Access Fields */
      { &hf_aecp_aa_count,
         { "Count", "ieee17221.count",
            FT_UINT16, BASE_DEC, NULL, 0x0, NULL, HFILL }
      },
      { &hf_aecp_aa_tlv_mode,
         { "Tlv Mode", "ieee17221.tlv_mode",
            FT_UINT8, BASE_DEC, NULL, 0xF0, NULL, HFILL }
      },
      { &hf_aecp_aa_tlv_length,
         { "Tlv Length", "ieee17221.tlv_length",
            FT_UINT16, BASE_DEC, NULL, 0x0FFF, NULL, HFILL }
      },
       { &hf_aecp_aa_tlv_address,
         { "Tlv Address", "ieee17221.tlv_address",
            FT_UINT64, BASE_HEX, NULL, 0, NULL, HFILL }
      },

      /* SLIGHTLY LESS COMMON FIELDS */
      { &hf_aecp_descriptor_type,
         { "Descriptor Type", "ieee17221.descriptor_type",
            FT_UINT16, BASE_HEX, VALS(aem_descriptor_type_vals), 0x00, NULL, HFILL }
      },
      { &hf_aecp_descriptor_index,
         {"Descriptor Index", "ieee17221.descriptor_index",
            FT_UINT16, BASE_HEX, NULL, 0x0, NULL, HFILL }
      },

      /* AECP Commands and Responses 1722.1 Sec 7.4 */
      /* LOCK_ENTITY */
      { &hf_aecp_unlock_flag,
         { "UNLOCK Flag", "ieee17221.flags.unlock",
            FT_BOOLEAN, 8, NULL, AECP_UNLOCK_FLAG_MASK, NULL, HFILL }
      },
      { &hf_aecp_locked_guid,
         { "Locked GUID", "ieee17221.locked_guid",
            FT_UINT64, BASE_HEX, NULL, 0x0, NULL, HFILL}
      },

      /* READ_DESCRIPTOR */
      { &hf_aecp_configuration,
         { "Configuration", "ieee17221.configuration",
            FT_UINT16, BASE_DEC, NULL, 0x0, NULL, HFILL }
      },

      /* WRITE_DESCRIPTOR */

      /* ACQUIRE_ENTITY */
      { &hf_aecp_persistent_flag,
         { "Persistent Flag", "ieee17221.flags.persistent",
            FT_BOOLEAN, 32, NULL, AECP_PERSISTENT_FLAG_MASK, NULL, HFILL }
      },
      { &hf_aecp_release_flag,
         { "Release Flag", "ieee17221.flags.release",
            FT_BOOLEAN, 32, NULL, AECP_RELEASE_FLAG_MASK, NULL, HFILL }
      },
      { &hf_aecp_owner_guid,
         { "Owner GUID", "ieee17221.owner_guid",
            FT_UINT64, BASE_HEX, NULL, 0x0, NULL, HFILL }
      },

      /* CONTROLLER_AVAILABLE */

      /* SET_CLOCK_SOURCE / GET_CLOCK_SOURCE */
      { &hf_aecp_clock_source_id,
         { "Clock Source ID", "ieee17221.clock_source_id64",
            FT_UINT64, BASE_HEX, NULL, 0x0, NULL, HFILL }
      },

      /* SET_STREAM_FORMAT */
      { &hf_aecp_stream_format,
         {"Stream Format", "ieee17221.stream_format64",
            FT_UINT64, BASE_HEX, NULL, 0x0, NULL, HFILL }
      },
      /* GET_STREAM_FORMAT */

      /* SET_CONFIGURATION / GET_CONFIGURATION */

      /* SET_CONTROL_VALUE / GET_CONTROL_VALUE */

      /* SET_SIGNAL_SELECTOR / GET_SIGNAL_SELECTOR */
      { &hf_aecp_signal_type,
         {"Signal Type", "ieee17221.signal_type",
            FT_UINT16, BASE_HEX, NULL, 0x0, NULL, HFILL }
      },
      { &hf_aecp_signal_index,
         {"Signal Index", "ieee17221.signal_index",
            FT_UINT16, BASE_HEX, NULL, 0x0, NULL, HFILL }
      },
      { &hf_aecp_signal_output,
         {"Signal Output", "ieee17221.signal_output",
            FT_UINT16, BASE_HEX, NULL, 0x0, NULL, HFILL }
      },

      /* SET_MIXER / GET_MIXER */

      /* SET_MATRIX / GET_MATRIX */
      { &hf_aecp_matrix_column,
         {"Matrix Column", "ieee17221.matrix_column",
            FT_UINT16, BASE_DEC, NULL, 0x0, NULL, HFILL }
      },
      { &hf_aecp_matrix_row,
         {"Matrix Row", "ieee17221.matrix_row",
            FT_UINT16, BASE_DEC, NULL, 0x0, NULL, HFILL }
      },
      { &hf_aecp_matrix_region_width,
         {"Region Width", "ieee17221.matrix_region_width",
            FT_UINT16, BASE_DEC, NULL, 0x0, NULL, HFILL }
      },
      { &hf_aecp_matrix_region_height,
         {"Region Height", "ieee17221.matrix_region_height",
            FT_UINT16, BASE_DEC, NULL, 0x0, NULL, HFILL }
      },
      { &hf_aecp_matrix_rep,
         {"Rep", "ieee17221.matrix_rep",
            FT_BOOLEAN, 8, NULL, AECP_MATRIX_REP_MASK, NULL, HFILL }
      },
      { &hf_aecp_matrix_direction,
         {"Direction", "ieee17221.matrix_direction",
            FT_UINT8, BASE_DEC, VALS(aecp_direction_type_vals), AECP_MATRIX_DIRECTION_MASK, NULL, HFILL }
      },
      { &hf_aecp_matrix_value_count,
         {"Value Count", "ieee17221.matrix_value_count",
            FT_UINT16, BASE_DEC, NULL, AECP_MATRIX_VALUE_COUNT_MASK, NULL, HFILL }
      },
      { &hf_aecp_matrix_item_offset,
         {"Item Offset", "ieee17221.matrix_item_offset",
            FT_UINT16, BASE_DEC, NULL, 0x0, NULL, HFILL }
      },
#if 0
      { &hf_aecp_matrix_affected_item_count,
         {"Affected Item Count", "ieee17221.matrix_affected_item_count",
            FT_UINT32, BASE_DEC, NULL, 0x0, NULL, HFILL }
      },
#endif

      /* START_STREAMING */

      /* STOP_STREAMING */

      /* GET_AVB_INFO */
      { &hf_aecp_avb_info_ptp_grandmaster_id,
         {"gPTP Grandmaster ID", "ieee17221.avb_info_gptp_grandmaster_id",
            FT_UINT64, BASE_HEX, NULL, 0x0, NULL, HFILL }
      },
      { &hf_aecp_avb_info_propegation_delay,
         {"Propagation Delay", "ieee17221.avb_info_propagation_delay",
            FT_UINT32, BASE_DEC, NULL, 0x0, NULL, HFILL }
      },
      { &hf_aecp_avb_info_gptp_domain_number,
         {"gPTP Domain Number", "ieee17221.avb_info_gptp_domain_number",
            FT_UINT8, BASE_DEC, NULL, 0x0, NULL, HFILL }
      },
#if 0
      { &hf_aecp_avb_info_flags,
         {"Flags", "ieee17221.avbinfo_flags",
            FT_UINT8, BASE_HEX, NULL, 0x0, NULL, HFILL }
      },
#endif

      { &hf_aecp_as_capable_flag,
         { "AS Capable Flag", "ieee17221.as_capable_flag",
            FT_BOOLEAN, 8, NULL, AECP_AS_CAPABLE_FLAG_MASK, NULL, HFILL }
      },

      { &hf_aecp_gptp_enabled_flag,
         { "gPTP Enabled Flag", "ieee17221.gptp_enabled_flag",
            FT_BOOLEAN, 8, NULL, AECP_GPTP_ENABLED_FLAG_MASK, NULL, HFILL }
      },

      { &hf_aecp_srp_enabled_flag,
         { "SRP Enabled Flag", "ieee17221.srp_enabled_flag",
            FT_BOOLEAN, 8, NULL, AECP_SRP_ENABLED_FLAG_MASK, NULL, HFILL }
      },

      { &hf_aecp_avb_info_msrp_mappings_count,
         {"MSRP Mappings Count", "ieee17221.msrp_mappings_count",
            FT_UINT16, BASE_DEC, NULL, 0x0, NULL, HFILL }
      },

      { &hf_aecp_avb_info_msrp_mappings,
         {"MSRP Mappings", "ieee17221.msrp_mappings",
            FT_NONE, BASE_NONE, NULL, 0x0, NULL, HFILL }
      },

      { &hf_aecp_avb_info_msrp_mapping_traffic_class,
         {"MSRP Mapping Traffic Class", "ieee17221.msrp_mapping_traffic_class",
            FT_UINT8, BASE_DEC, NULL, 0x0, NULL, HFILL }
      },
      { &hf_aecp_avb_info_msrp_mapping_priority,
         {"MSRP Mapping Priority", "ieee17221.msrp_mapping_priority",
            FT_UINT8, BASE_DEC, NULL, 0x0, NULL, HFILL }
      },
      { &hf_aecp_get_avb_info_msrp_vlan_id,
         {"MSRP VLAN ID", "ieee17221.msrp_vlan_id",
            FT_UINT16, BASE_DEC, NULL, 0x0, NULL, HFILL }
      },

      { &hf_aecp_map_index,
         {"Map Index", "ieee17221.map_index",
            FT_UINT16, BASE_DEC, NULL, 0x0, NULL, HFILL }
      },
      { &hf_aecp_number_of_maps,
         {"Number of Maps", "ieee17221.number_of_maps",
            FT_UINT16, BASE_DEC, NULL, 0x0, NULL, HFILL }
      },

      /* GET_STREAM_INFO */
      { &hf_aecp_msrp_accumulated_latency,
         {"MSRP Accumulated Latency", "ieee17221.msrp_accumulated_latency",
            FT_UINT32, BASE_DEC, NULL, 0x0, NULL, HFILL }
      },
      { &hf_aecp_msrp_failure_code,
         {"MSRP Failure Code", "ieee17221.msrp_failure_code",
            FT_UINT32, BASE_DEC, NULL, 0x0, NULL, HFILL }
      },
      { &hf_aecp_msrp_failure_bridge_id,
         {"MSRP Failure Bridge ID", "ieee17221.msrp_failure_bridge_id",
            FT_BYTES, BASE_NONE, NULL, 0x0, NULL, HFILL }
      },
#if 0
      { &hf_aecp_stream_vlan_id,
         {"Stream VLAN ID", "ieee17221.stream_vlan_id",
             FT_UINT16, BASE_DEC, NULL, 0x0, NULL, HFILL }
      },
#endif
      { &hf_aecp_stream_vlan_id_valid_flag,
         {"Stream VLAN ID Valid Flag", "ieee17221.flags.stream_vlan_id_valid",
            FT_BOOLEAN, 32, NULL, AECP_STREAM_VLAN_ID_VALID_FLAG_MASK, NULL, HFILL }
      },
      { &hf_aecp_connected_flag,
         {"Connected Flag", "ieee17221.flags.connected",
            FT_BOOLEAN, 32, NULL, AECP_CONNECTED_FLAG_MASK, NULL, HFILL }
      },
      { &hf_aecp_msrp_failure_valid_flag,
         {"MSRP Failure Valid Flag", "ieee17221.flags.msrp_failure_valid",
            FT_BOOLEAN, 32, NULL, AECP_MSRP_FAILURE_VALID_FLAG_MASK, NULL, HFILL }
      },
      { &hf_aecp_dest_mac_valid_flag,
         {"Dest MAC Valid Flag", "ieee17221.flags.dest_mac_valid",
            FT_BOOLEAN, 32, NULL, AECP_DEST_MAC_VALID_FLAG_MASK, NULL, HFILL }
      },
      { &hf_aecp_msrp_acc_lat_valid_flag,
         {"MSRP Accumulated Latency Field Valid Flag", "ieee17221.flags.msrp_acc_lat_valid",
            FT_BOOLEAN, 32, NULL, AECP_MSRP_ACC_LAT_VALID_FLAG_MASK, NULL, HFILL }
      },
      { &hf_aecp_stream_id_valid_flag,
         {"Stream ID Valid Flag", "ieee17221.flags.stream_id_valid",
            FT_BOOLEAN, 32, NULL, AECP_STREAM_ID_VALID_FLAG_MASK, NULL, HFILL }
      },
      { &hf_aecp_stream_format_valid_flag,
         {"Stream Format Valid Flag", "ieee17221.flags.stream_format_valid",
            FT_BOOLEAN, 32, NULL, AECP_STREAM_FORMAT_VALID_FLAG_MASK, NULL, HFILL }
      },

      /* SET_NAME / GET_NAME */
      { &hf_aecp_name_index,
         {"Name Index", "ieee17221.name_index",
            FT_UINT16, BASE_DEC, NULL, 0x0, NULL, HFILL }
      },
      { &hf_aecp_configuration_index,
         {"Configuration Index", "ieee17221.configuration_index",
            FT_UINT16, BASE_DEC, NULL, 0x0, NULL, HFILL }
      },

      { &hf_aecp_name,
         {"Name", "ieee17221.name",
            FT_STRING, BASE_NONE, NULL, 0x0, NULL, HFILL }
      },

      /* SET_ASSOCIATION_ID / GET_ASSOCIATION_ID */
      { &hf_aecp_association_id,
         {"Association ID", "ieee17221.association_id",
            FT_UINT64, BASE_HEX, NULL, 0x0, NULL, HFILL }
      },

      /* GET_AS_PATH */
      { &hf_aecp_as_path_count,
         {"Count", "ieee17221.as_path_count",
            FT_UINT16, BASE_DEC, NULL, 0x0, NULL, HFILL }
      },

      { &hf_aecp_as_path_sequences,
         {"Path Sequence", "ieee17221.as_path_sequences",
            FT_NONE, BASE_NONE, NULL, 0x0, NULL, HFILL }
      },

      { &hf_aecp_get_as_info_clock_id,
         { "ClockId", "ieee17221.get_as_info_clock_id",
            FT_UINT64, BASE_HEX, NULL, 0x0, NULL, HFILL }
      },

      /* AUTH_ADD_KEY */
      { &hf_aecp_keychain_id,
         {"Keychain ID", "ieee17221.keychain_id",
            FT_UINT16, BASE_HEX, NULL, 0x0, NULL, HFILL }
      },
      { &hf_aecp_keychain_id_small,
         {"Keychain ID", "ieee17221.keychain_id_small",
            FT_UINT8, BASE_HEX, VALS(aecp_keychain_id_type_vals), AECP_KEYCHAIN_ID_MASK, NULL, HFILL }
      },
      { &hf_aecp_key_type,
         {"Key Type", "ieee17221.key_type",
            FT_UINT8, BASE_HEX, VALS(aecp_key_type_type_vals), AECP_KEYTYPE_MASK, NULL, HFILL }
      },
      { &hf_aecp_key_signature,
         {"Key Signature", "ieee17221.key_signature",
            FT_BYTES, BASE_NONE, NULL, 0x0, NULL, HFILL }
      },
#if 0
      { &hf_aecp_key_number,
         {"Key ID", "ieee17221.key_id",
            FT_UINT16, BASE_HEX, NULL, AECP_KEY_NUMBER_MASK, NULL, HFILL }
      },
      { &hf_aecp_continued_flag,
         {"Continued", "ieee17221.continued",
            FT_BOOLEAN, 8, NULL, AECP_CONTINUED_MASK, NULL, HFILL }
      },
      { &hf_aecp_key_part,
         {"Key Part", "ieee17221.key_part",
            FT_UINT8, BASE_DEC, NULL, AECP_KEY_PART_MASK, NULL, HFILL }
      },
#endif

      { &hf_aecp_key_length,
         {"Key Length", "ieee17221.key_length",
            FT_UINT16, BASE_DEC, NULL, AECP_KEY_LENGTH_MASK, NULL, HFILL }
      },
#if 0
      { &hf_aecp_private_key_read_flag,
         {"Private Key Read Flag", "ieee17221.flags.private_key_read",
            FT_BOOLEAN, 32, NULL, AECP_PRIVATE_KEY_READ_FLAG_MASK, NULL, HFILL }
      },
      { &hf_aecp_private_key_write_flag,
         {"Private Key Write Flag", "ieee17221.flags.private_key_write",
            FT_BOOLEAN, 32, NULL, AECP_PRIVATE_KEY_WRITE_FLAG_MASK, NULL, HFILL }
      },
      { &hf_aecp_public_key_write_flag,
         {"Public Key Write Flag", "ieee17221.flags.public_key_write",
            FT_BOOLEAN, 32, NULL, AECP_PUBLIC_KEY_WRITE_FLAG_MASK, NULL, HFILL }
      },
      { &hf_aecp_control_admin_flag,
         {"Control Admin Flag", "ieee17221.flags.control_admin",
            FT_BOOLEAN, 32, NULL, AECP_CONTROL_ADMIN_FLAG_MASK, NULL, HFILL }
      },
      { &hf_aecp_mem_obj_admin_flag,
         {"Memory Object Admin", "ieee17221.mem_obj_admin",
            FT_BOOLEAN, 32, NULL, AECP_MEM_OBJ_ADMIN_FLAG_MASK, NULL, HFILL }
      },
      { &hf_aecp_mem_obj_settings_flag,
         {"Memory Object Settings", "ieee17221.mem_obj_settings",
            FT_BOOLEAN, 32, NULL, AECP_MEM_OBJ_SETTINGS_FLAG_MASK, NULL, HFILL }
      },
      { &hf_aecp_control_user_l1,
         {"Control User L1 flag", "ieee17221.flags.control_user_l1",
            FT_BOOLEAN, 32, NULL, AECP_CONTROL_USER_L1_FLAG_MASK, NULL, HFILL }
      },
      { &hf_aecp_control_user_l2,
         {"Control User L2 flag", "ieee17221.flags.control_user_l2",
            FT_BOOLEAN, 32, NULL, AECP_CONTROL_USER_L2_FLAG_MASK, NULL, HFILL }
      },
      { &hf_aecp_control_user_l3,
         {"Control User L3 flag", "ieee17221.flags.control_user_l3",
            FT_BOOLEAN, 32, NULL, AECP_CONTROL_USER_L3_FLAG_MASK, NULL, HFILL }
      },
      { &hf_aecp_control_user_l4,
         {"Control User L4 flag", "ieee17221.flags.control_user_l4",
            FT_BOOLEAN, 32, NULL, AECP_CONTROL_USER_L4_FLAG_MASK, NULL, HFILL }
      },
#endif

      { &hf_aecp_keychain_list_index,
         {"Keychain List Index", "ieee17221.keychain_list_index",
            FT_UINT16, BASE_HEX, NULL, 0x0, NULL, HFILL }
      },
      { &hf_aecp_keychain_number_of_keys,
         {"Keychain Number of Keys", "ieee17221.keychain_number_of_keys",
            FT_UINT16, BASE_DEC, NULL, 0x0, NULL, HFILL }
      },
      { &hf_aecp_keychain_number_of_lists,
         {"Keychain Number of Lists", "ieee17221.keychain_number_of_lists",
            FT_UINT16, BASE_DEC, NULL, 0x0, NULL, HFILL }
      },

      { &hf_aecp_key_eui,
         {"Key EUI", "ieee17221.key_eui",
            FT_UINT64, BASE_HEX, NULL, 0x0, NULL, HFILL }
      },
      { &hf_aecp_token_length,
         {"Token Length", "ieee17221.token_length",
            FT_UINT16, BASE_DEC, NULL, AECP_TOKEN_LENGTH_MASK, NULL, HFILL }
      },
      { &hf_aecp_key,
         {"Key", "ieee17221.key",
            FT_BYTES, BASE_NONE, NULL, 0x0, NULL, HFILL }
      },
      { &hf_aecp_auth_token,
         {"Auth Token", "ieee17221.auth_token",
            FT_BYTES, BASE_NONE, NULL, 0x0, NULL, HFILL }
      },
      { &hf_aecp_flags_32,
         {"Flags", "ieee17221.flags",
            FT_UINT32, BASE_HEX, NULL, 0x0, NULL, HFILL }
      },

      /* AUTH_GET_KEY */
      /* AUTHENTICATE */
      /* GET_COUNTERS */

      /* Counter valid bits */
      { &hf_aecp_avb_interface_link_up_valid,
         {"Link Up Valid", "ieee17221.flags.link_up_valid",
            FT_BOOLEAN, 32, NULL, AECP_COUNTERS_VALID_LINK_UP, NULL, HFILL }
      },
      { &hf_aecp_avb_interface_link_down_valid,
         {"Link Down Valid", "ieee17221.flags.link_down_valid",
            FT_BOOLEAN, 32, NULL, AECP_COUNTERS_VALID_LINK_DOWN, NULL, HFILL }
      },
      { &hf_aecp_avb_interface_packets_tx_valid,
         {"Packets TX Valid", "ieee17221.flags.packets_tx_valid",
            FT_BOOLEAN, 32, NULL, AECP_COUNTERS_VALID_PACKETS_TX, NULL, HFILL }
      },
      { &hf_aecp_avb_interface_packets_rx_valid,
         {"Packets RX Valid", "ieee17221.flags.packets_rx_valid",
            FT_BOOLEAN, 32, NULL, AECP_COUNTERS_VALID_PACKETS_RX, NULL, HFILL }
      },
      { &hf_aecp_avb_interface_rx_crc_error_valid,
         {"RX CRC Error Valid", "ieee17221.flags.rx_crc_error_valid",
            FT_BOOLEAN, 32, NULL, AECP_COUNTERS_VALID_RX_CRC_ERROR, NULL, HFILL }
      },
      { &hf_aecp_avb_interface_gptp_gm_changed_valid,
         {"GPTP GM Changed Valid", "ieee17221.flags.gptp_gm_changed_valid",
            FT_BOOLEAN, 32, NULL, AECP_COUNTERS_VALID_GPTP_GM_CHANGED, NULL, HFILL }
      },

      { &hf_aecp_clock_domain_unlocked_valid,
         {"Clock Domain Unlocked Valid", "ieee17221.flags.clock_domain_unlocked_valid",
            FT_BOOLEAN, 32, NULL, AECP_COUNTERS_VALID_CLOCK_DOMAIN_UNLOCKED, NULL, HFILL }
      },
      { &hf_aecp_clock_domain_locked_valid,
         {"Clock_Domain Locked Valid", "ieee17221.flags.clock_domain_locked_valid",
            FT_BOOLEAN, 32, NULL, AECP_COUNTERS_VALID_CLOCK_DOMAIN_LOCKED, NULL, HFILL }
      },

      { &hf_aecp_stream_input_media_unlocked_valid,
         {"Media Unlocked Valid", "ieee17221.flags.media_unlocked_valid",
            FT_BOOLEAN, 32, NULL, AECP_COUNTERS_VALID_MEDIA_UNLOCKED, NULL, HFILL }
      },
      { &hf_aecp_stream_input_media_locked_valid,
         {"Media Locked Valid", "ieee17221.flags.media_locked_valid",
            FT_BOOLEAN, 32, NULL, AECP_COUNTERS_VALID_MEDIA_LOCKED, NULL, HFILL }
      },
      { &hf_aecp_stream_input_stream_reset_valid,
         {"Stream Reset Valid", "ieee17221.flags.stream_reset_valid",
            FT_BOOLEAN, 32, NULL, AECP_COUNTERS_VALID_STREAM_RESET, NULL, HFILL }
      },
      { &hf_aecp_stream_input_seq_num_mismatch_valid,
         {"Seq Num Mismatch Valid", "ieee17221.flags.seq_num_mismatch_valid",
            FT_BOOLEAN, 32, NULL, AECP_COUNTERS_VALID_SEQ_NUM_MISMATCH, NULL, HFILL }
      },
      { &hf_aecp_stream_input_media_reset_valid,
         {"Media Reset Valid", "ieee17221.flags.media_reset_valid",
            FT_BOOLEAN, 32, NULL, AECP_COUNTERS_VALID_MEDIA_RESET, NULL, HFILL }
      },
      { &hf_aecp_stream_input_timestamp_uncertain_valid,
         {"Timestamp Uncertain Valid", "ieee17221.flags.timestamp_uncertain_valid",
            FT_BOOLEAN, 32, NULL, AECP_COUNTERS_VALID_TIMESTAMP_UNCERTAIN, NULL, HFILL }
      },
      { &hf_aecp_stream_input_timestamp_valid_valid,
         {"Timestamp Valid Valid", "ieee17221.flags.timestamp_valid_valid",
            FT_BOOLEAN, 32, NULL, AECP_COUNTERS_VALID_TIMESTAMP_VALID, NULL, HFILL }
      },
      { &hf_aecp_stream_input_timestamp_not_valid_valid,
         {"Timestamp Not Valid Valid", "ieee17221.flags.timestamp_not_valid_valid",
            FT_BOOLEAN, 32, NULL, AECP_COUNTERS_VALID_TIMESTAMP_NOT_VALID, NULL, HFILL }
      },
      { &hf_aecp_stream_input_unsupported_format_valid,
         {"Unsupported Format Valid", "ieee17221.flags.unsupported_format_valid",
            FT_BOOLEAN, 32, NULL, AECP_COUNTERS_VALID_UNSUPPORTED_FORMAT, NULL, HFILL }
      },
      { &hf_aecp_stream_input_late_timestamp_valid,
         {"Late Timestamp Valid", "ieee17221.flags.late_timestamp_valid",
            FT_BOOLEAN, 32, NULL, AECP_COUNTERS_VALID_LATE_TIMESTAMP, NULL, HFILL }
      },
      { &hf_aecp_stream_input_early_timestamp_valid,
         {"Early Timestamp Valid", "ieee17221.flags.early_timestamp_valid",
            FT_BOOLEAN, 32, NULL, AECP_COUNTERS_VALID_EARLY_TIMESTAMP, NULL, HFILL }
      },
      { &hf_aecp_stream_input_packets_tx_valid,
         {"Stream Packets TX Valid", "ieee17221.flags.stream_packets_tx_valid",
            FT_BOOLEAN, 32, NULL, AECP_COUNTERS_VALID_STREAM_PACKETS_TX, NULL, HFILL }
      },
      { &hf_aecp_stream_input_packets_rx_valid,
         {"Stream Packets RX Valid", "ieee17221.flags.stream_packets_rx_valid",
            FT_BOOLEAN, 32, NULL, AECP_COUNTERS_VALID_STREAM_PACKETS_RX, NULL, HFILL }
      },

      { &hf_aecp_entity_specific1_valid,
         {"Entity Specific 1", "ieee17221.flags.entity_specific1_valid",
            FT_BOOLEAN, 32, NULL, AECP_COUNTERS_VALID_ENTITY_SPECIFIC_1, NULL, HFILL }
      },
      { &hf_aecp_entity_specific2_valid,
         {"Entity Specific 2", "ieee17221.flags.entity_specific2_valid",
            FT_BOOLEAN, 32, NULL, AECP_COUNTERS_VALID_ENTITY_SPECIFIC_2, NULL, HFILL }
      },
      { &hf_aecp_entity_specific3_valid,
         {"Entity Specific 3", "ieee17221.flags.entity_specific3_valid",
            FT_BOOLEAN, 32, NULL, AECP_COUNTERS_VALID_ENTITY_SPECIFIC_3, NULL, HFILL }
      },
      { &hf_aecp_entity_specific4_valid,
         {"Entity Specific 4", "ieee17221.flags.entity_specific4_valid",
            FT_BOOLEAN, 32, NULL, AECP_COUNTERS_VALID_ENTITY_SPECIFIC_4, NULL, HFILL }
      },
      { &hf_aecp_entity_specific5_valid,
         {"Entity Specific 5", "ieee17221.flags.entity_specific5_valid",
            FT_BOOLEAN, 32, NULL, AECP_COUNTERS_VALID_ENTITY_SPECIFIC_5, NULL, HFILL }
      },
      { &hf_aecp_entity_specific6_valid,
         {"Entity Specific 6", "ieee17221.flags.entity_specific6_valid",
            FT_BOOLEAN, 32, NULL, AECP_COUNTERS_VALID_ENTITY_SPECIFIC_6, NULL, HFILL }
      },
      { &hf_aecp_entity_specific7_valid,
         {"Entity Specific 7", "ieee17221.flags.entity_specific7_valid",
            FT_BOOLEAN, 32, NULL, AECP_COUNTERS_VALID_ENTITY_SPECIFIC_7, NULL, HFILL }
      },
      { &hf_aecp_entity_specific8_valid,
         {"Entity Specific 8", "ieee17221.flags.entity_specific8_valid",
            FT_BOOLEAN, 32, NULL, AECP_COUNTERS_VALID_ENTITY_SPECIFIC_8, NULL, HFILL }
      },

      /* Counters */
      { &hf_aecp_avb_interface_link_up,
         {"Link Up", "ieee17221.link_up",
            FT_UINT32, BASE_DEC, NULL, 0x0, NULL, HFILL }
      },
      { &hf_aecp_avb_interface_link_down,
         {"Link Down", "ieee17221.link_down",
            FT_UINT32, BASE_DEC, NULL, 0x0, NULL, HFILL }
      },
      { &hf_aecp_avb_interface_packets_tx,
         {"Frames TX", "ieee17221.frames_tx",
            FT_UINT32, BASE_DEC, NULL, 0x0, NULL, HFILL }
      },
      { &hf_aecp_avb_interface_packets_rx,
         {"Frames RX", "ieee17221.frames_rx",
            FT_UINT32, BASE_DEC, NULL, 0x0, NULL, HFILL }
      },
      { &hf_aecp_avb_interface_rx_crc_error,
         {"RX CRC Error", "ieee17221.rx_crc_error",
            FT_UINT32, BASE_DEC, NULL, 0x0, NULL, HFILL }
      },
      { &hf_aecp_avb_interface_gptp_gm_changed,
         {"GPTP GM Changed", "ieee17221.gptp_gm_changed",
            FT_UINT32, BASE_DEC, NULL, 0x0, NULL, HFILL }
      },

      { &hf_aecp_clock_domain_unlocked,
         {"Clock Domain Unlocked", "ieee17221.clock_domain_unlocked",
            FT_UINT32, BASE_DEC, NULL, 0x0, NULL, HFILL }
      },
      { &hf_aecp_clock_domain_locked,
         {"Clock_Domain Locked", "ieee17221.clock_domain_locked",
            FT_UINT32, BASE_DEC, NULL, 0x0, NULL, HFILL }
      },

      { &hf_aecp_stream_input_media_unlocked,
         {"Media Unlocked", "ieee17221.media_unlocked",
            FT_UINT32, BASE_DEC, NULL, 0x0, NULL, HFILL }
      },
      { &hf_aecp_stream_input_media_locked,
         {"Media Locked", "ieee17221.media_locked",
            FT_UINT32, BASE_DEC, NULL, 0x0, NULL, HFILL }
      },
      { &hf_aecp_stream_input_stream_reset,
         {"Stream Reset", "ieee17221.stream_reset",
            FT_UINT32, BASE_DEC, NULL, 0x0, NULL, HFILL }
      },
      { &hf_aecp_stream_input_seq_num_mismatch,
         {"Seq Num Mismatch", "ieee17221.seq_num_mismatch",
            FT_UINT32, BASE_DEC, NULL, 0x0, NULL, HFILL }
      },
      { &hf_aecp_stream_input_media_reset,
         {"Media Reset", "ieee17221.media_reset",
            FT_UINT32, BASE_DEC, NULL, 0x0, NULL, HFILL }
      },
      { &hf_aecp_stream_input_timestamp_uncertain,
         {"Timestamp Uncertain", "ieee17221.timestamp_uncertain",
            FT_UINT32, BASE_DEC, NULL, 0x0, NULL, HFILL }
      },
      { &hf_aecp_stream_input_timestamp_valid,
         {"Timestamp Valid", "ieee17221.timestamp",
            FT_UINT32, BASE_DEC, NULL, 0x0, NULL, HFILL }
      },
      { &hf_aecp_stream_input_timestamp_not_valid,
         {"Timestamp Not Valid", "ieee17221.timestamp_not",
            FT_UINT32, BASE_DEC, NULL, 0x0, NULL, HFILL }
      },
      { &hf_aecp_stream_input_unsupported_format,
         {"Unsupported Format", "ieee17221.unsupported_format",
            FT_UINT32, BASE_DEC, NULL, 0x0, NULL, HFILL }
      },
      { &hf_aecp_stream_input_late_timestamp,
         {"Late Timestamp", "ieee17221.late_timestamp",
            FT_UINT32, BASE_DEC, NULL, 0x0, NULL, HFILL }
      },
      { &hf_aecp_stream_input_early_timestamp,
         {"Early Timestamp", "ieee17221.early_timestamp",
            FT_UINT32, BASE_DEC, NULL, 0x0, NULL, HFILL }
      },
      { &hf_aecp_stream_input_packets_tx,
         {"Stream Packets TX", "ieee17221.stream_packets_tx",
            FT_UINT32, BASE_DEC, NULL, 0x0, NULL, HFILL }
      },
      { &hf_aecp_stream_input_packets_rx,
         {"Stream Packets RX", "ieee17221.stream_packets_rx",
            FT_UINT32, BASE_DEC, NULL, 0x0, NULL, HFILL }
      },

      { &hf_aecp_entity_specific1,
         {"Entity Specific 1", "ieee17221.entity_specific1",
            FT_UINT32, BASE_DEC, NULL, 0x0, NULL, HFILL }
      },
      { &hf_aecp_entity_specific2,
         {"Entity Specific 2", "ieee17221.entity_specific2",
            FT_UINT32, BASE_DEC, NULL, 0x0, NULL, HFILL }
      },
      { &hf_aecp_entity_specific3,
         {"Entity Specific 3", "ieee17221.entity_specific3",
            FT_UINT32, BASE_DEC, NULL, 0x0, NULL, HFILL }
      },
      { &hf_aecp_entity_specific4,
         {"Entity Specific 4", "ieee17221.entity_specific4",
            FT_UINT32, BASE_DEC, NULL, 0x0, NULL, HFILL }
      },
      { &hf_aecp_entity_specific5,
         {"Entity Specific 5", "ieee17221.entity_specific5",
            FT_UINT32, BASE_DEC, NULL, 0x0, NULL, HFILL }
      },
      { &hf_aecp_entity_specific6,
         {"Entity Specific 6", "ieee17221.entity_specific6",
            FT_UINT32, BASE_DEC, NULL, 0x0, NULL, HFILL }
      },
      { &hf_aecp_entity_specific7,
         {"Entity Specific 7", "ieee17221.entity_specific7",
            FT_UINT32, BASE_DEC, NULL, 0x0, NULL, HFILL }
      },
      { &hf_aecp_entity_specific8,
         {"Entity Specific 8", "ieee17221.entity_specific8",
            FT_UINT32, BASE_DEC, NULL, 0x0, NULL, HFILL }
      },

      /* REBOOT */

      /* SET_MEDIA_FORMAT / GET_MEDIA_FORMAT */
#if 0
      { &hf_aecp_media_format,
         {"Media Format", "ieee17221.media_format",
            FT_BYTES, BASE_NONE, NULL, 0x0, NULL, HFILL }
      },
#endif

      { &hf_aecp_video_format,
         {"Video Format", "ieee17221.video_format",
            FT_BYTES, BASE_NONE, NULL, 0x0, NULL, HFILL }
      },

      { &hf_aecp_sampling_rate_pull,
         { "Sampling Rate Pull Value", "ieee17221.sampling_rate_pull",
            FT_UINT32, BASE_HEX, VALS(aecp_sampling_rate_pull_field_vals), AECP_SAMPLING_RATE_PULL_MASK, NULL, HFILL }
      },

      { &hf_aecp_sampling_rate_base_frequency,
         {"Sampling Rate Base Frequency", "ieee17221.sampling_rate_base_frequency",
            FT_UINT32, BASE_DEC, NULL, AECP_SAMPLING_RATE_BASE_FREQ_MASK, NULL, HFILL }
      },

      /* REGISTER_STATE_NOTIFICATION */
#if 0
      { &hf_aecp_address_type,
         {"Address Type", "ieee17221.address_type",
            FT_UINT16, BASE_HEX, VALS(aecp_address_type_vals), 0x00, NULL, HFILL }
      },
#endif
      { &hf_aecp_mac_address,
         { "MAC address", "ieee17221.mac_address",
            FT_ETHER, BASE_NONE, NULL, 0x0, NULL, HFILL }
      },
#if 0
      { &hf_aecp_ipv4_address,
         {"IPV4 Address", "ieee17221.ipv4_address",
            FT_IPv4, BASE_NONE, NULL, 0x0, NULL, HFILL }
      },
      { &hf_aecp_ipv6_address,
         {"IPv6 Address", "ieee17221.ipv6_address",
            FT_IPv6, BASE_NONE, NULL, 0x0, NULL, HFILL }
      },
#endif

      /* DEREGISTER_STATE_NOTIFICATION */

      /* REGISTER_QUERY_NOTIFICATION / DEREGISTER_QUERY_NOTIFICATION */
#if 0
      { &hf_aecp_query_period,
         {"Query Period (ms)", "ieee17221.query_period",
            FT_UINT16, BASE_DEC, NULL, 0x0, NULL, HFILL }
      },
      { &hf_aecp_query_limit,
         {"Query Limit", "ieee17221.query_limit",
            FT_UINT16, BASE_DEC, NULL, 0x0, NULL, HFILL }
      },
      { &hf_aecp_query_type,
         {"Query Type", "ieee17221.query_type",
            FT_UINT16, BASE_HEX, VALS(aecp_command_type_vals), 0x00, NULL, HFILL }
      },
      { &hf_aecp_query_id,
         {"Query ID", "ieee17221.query_id",
            FT_UINT16, BASE_DEC, NULL, 0x0, NULL, HFILL }
      },
#endif

      /* IDENTIFY_NOTIFICATION */

      /* STATE_CHANGE_NOTIFICATION */
#if 0
      { &hf_aecp_count,
         {"Count", "ieee17221.count",
            FT_UINT16, BASE_DEC, NULL, 0x0, NULL, HFILL }
      },
      { &hf_aecp_descriptors,
         {"Descriptors Array", "ieee17221.descriptors",
            FT_BYTES, BASE_NONE, NULL, 0x0, NULL, HFILL }
      },

      /* INCREMENT_CONTROL_VALUE / DECREMENT_CONTROL_VALUE */
      { &hf_aecp_values_count,
         {"Values Count", "ieee17221.values_count",
            FT_UINT16, BASE_DEC, NULL, 0x0, NULL, HFILL }
      },
#endif

      /* START_OPERATION */
      { &hf_aecp_operation_id,
         {"Operation ID", "ieee17221.operation_id",
            FT_UINT16, BASE_DEC, NULL, 0x0, NULL, HFILL }
      },
      { &hf_aecp_operation_type,
         {"Operation Type", "ieee17221.operation_type",
            FT_UINT16, BASE_HEX, NULL, 0x0, NULL, HFILL }
      },/* draft spec says this is defined by control_type field *
         * start_operation does not include a control type field *
         * There is an operation type table 7.83 that has not    *
         * yet been defined. control_type may be part of a       *
         * descriptor; will check                                */

      /* ABORT_OPERATION */

      /* OPERATION_STATUS */
      { &hf_aecp_percent_complete,
         {"Percent Complete", "ieee17221.percent_complete",
            FT_UINT16, BASE_DEC, NULL, 0x0, NULL, HFILL }
      },

      /* AUTH_GET_KEY_COUNT */

      /* * AVDECC ENTITY MODEL DESCRIPTOR FIELDS * */

      /* ENTITY */
      /* hf_aecp_descriptor_type */
      /* hf_aecp_descriptor_index */
      { &hf_aem_entity_id,
         {"Entity ID", "ieee17221.entity_id",
            FT_UINT64, BASE_HEX, NULL, 0x0, NULL, HFILL }
      },
      { &hf_aem_entity_model_id,
         {"Entity Model ID", "ieee17221.entity_model_id",
            FT_UINT64, BASE_HEX, NULL, 0x0, NULL, HFILL }
      },
      /* hf_adp_entity_cap
       * hf_adp_entity_cap_avdecc_ip
       * hf_adp_entity_cap_zero_conf
       * hf_adp_entity_cap_gateway_entity
       * hf_adp_entity_cap_avdecc_control
       * hf_adp_entity_cap_legacy_avc
       * hf_adp_entity_cap_assoc_id_support
       * hf_adp_entity_cap_assoc_id_valid
       */
      /* hf_adp_talker_stream_srcs */
      /* hf_adp_talker_cap & flags */
      /* hf_adp_listener_stream_sinks */
      /* hf_adp_listener_cap & flags */
      /* hf_adp_controller_cap & flags */
      /* hf_adp_avail_index */
      /* where appropriate use adp values */
      { &hf_aem_entity_name,
         {"Entity Name", "ieee17221.entity_name",
            FT_STRING, BASE_NONE, NULL, 0x0, NULL, HFILL }
      },
      { &hf_aem_vendor_name_string,
         {"Vendor Name String (ptr)", "ieee17221.vendor_name_string",
            FT_UINT16, BASE_DEC, NULL, 0x0, NULL, HFILL }
      },
      { &hf_aem_model_name_string,
         {"Model Name String (ptr)", "ieee17221.model_name_string",
            FT_UINT16, BASE_DEC, NULL, 0x0, NULL, HFILL }
      },
      { &hf_aem_firmware_version,
         {"Firmware Version", "ieee17221.firmware_version",
            FT_STRING, BASE_NONE, NULL, 0x0, NULL, HFILL }
      },
      { &hf_aem_group_name,
         {"Group Name", "ieee17221.group_name",
            FT_STRING, BASE_NONE, NULL, 0x0, NULL, HFILL }
      },
      { &hf_aem_serial_number,
         {"Serial Number", "ieee17221.serial_number",
            FT_STRING, BASE_NONE, NULL, 0x0, NULL, HFILL }
      },
      { &hf_aem_configurations_count,
         {"Configurations Count", "ieee17221.configurations_count",
            FT_UINT16, BASE_DEC, NULL, 0x0, NULL, HFILL }
      },
      { &hf_aem_current_configuration,
         {"Current Configuration", "ieee17221.current_configuration",
            FT_UINT16, BASE_DEC, NULL, 0x0, NULL, HFILL }
      },

      /* CONFIGURATION */
      { &hf_aem_configuration_name,
         {"Object Name", "ieee17221.configuration_name",
            FT_STRING, BASE_NONE, NULL, 0x0, NULL, HFILL }
      },
      { &hf_aem_configuration_name_string,
         {"Localized Description", "ieee17221.configuration_name_string",
            FT_UINT16, BASE_DEC, NULL, 0x0, NULL, HFILL }
      },
      { &hf_aem_descriptor_counts_count,
         {"Descriptor Counts Count", "ieee17221.descriptor_counts_count",
            FT_UINT16, BASE_DEC, NULL, 0x0, NULL, HFILL }
      },
      { &hf_aem_descriptor_counts_offset,
         {"Descriptor Counts Offset", "ieee17221.descriptor_counts_offset",
            FT_UINT16, BASE_DEC, NULL, 0x0, NULL, HFILL }
      },
      { &hf_aem_count,
         {"Count", "ieee17221.count",
            FT_UINT16, BASE_DEC, NULL, 0x0, NULL, HFILL }
      },

      /* AUDIO */
      { &hf_aem_number_of_stream_input_ports,
         {"Number Of Stream Input Ports", "ieee17221.number_of_stream_input_ports",
            FT_UINT16, BASE_DEC, NULL, 0x0, NULL, HFILL }
      },
      { &hf_aem_base_stream_input_port,
         {"Base Stream Input Port", "ieee17221.base_stream_input_port",
            FT_UINT16, BASE_DEC, NULL, 0x0, NULL, HFILL }
      },
      { &hf_aem_number_of_stream_output_ports,
         {"Number Of Stream Output Ports", "ieee17221.number_of_stream_output_ports",
            FT_UINT16, BASE_DEC, NULL, 0x0, NULL, HFILL }
      },
      { &hf_aem_base_stream_output_port,
         {"Base Stream Output Port", "ieee17221.base_stream_output_port",
            FT_UINT16, BASE_DEC, NULL, 0x0, NULL, HFILL }
      },
      { &hf_aem_number_of_external_input_ports,
         {"Number Of External Input Ports", "ieee17221.number_of_external_input_ports",
            FT_UINT16, BASE_DEC, NULL, 0x0, NULL, HFILL }
      },
      { &hf_aem_base_external_input_port,
         {"Base External Input Port", "ieee17221.base_external_input_port",
            FT_UINT16, BASE_DEC, NULL, 0x0, NULL, HFILL }
      },
      { &hf_aem_number_of_external_output_ports,
         {"Number Of External Output Ports", "ieee17221.number_of_external_output_ports",
            FT_UINT16, BASE_DEC, NULL, 0x0, NULL, HFILL }
      },
      { &hf_aem_base_external_output_port,
         {"Base External Output Port", "ieee17221.base_external_output_port",
            FT_UINT16, BASE_DEC, NULL, 0x0, NULL, HFILL }
      },
      { &hf_aem_number_of_internal_input_ports,
         {"Number Of Internal Input Ports", "ieee17221.number_of_internal_input_ports",
            FT_UINT16, BASE_DEC, NULL, 0x0, NULL, HFILL }
      },
      { &hf_aem_base_internal_input_port,
         {"Base Internal Input Port", "ieee17221.base_internal_input_port",
            FT_UINT16, BASE_DEC, NULL, 0x0, NULL, HFILL }
      },
      { &hf_aem_number_of_internal_output_ports,
         {"Number Of Internal Output Ports", "ieee17221.number_of_internal_output_ports",
            FT_UINT16, BASE_DEC, NULL, 0x0, NULL, HFILL }
      },
      { &hf_aem_base_internal_output_port,
         {"Base Internal Output Port", "ieee17221.base_internal_output_port",
            FT_UINT16, BASE_DEC, NULL, 0x0, NULL, HFILL }
      },
      { &hf_aem_clock_source_id,
         {"Clock Source ID", "ieee17221.clock_source_id",
            FT_UINT16, BASE_DEC, NULL, 0x0, NULL, HFILL }
      },
      { &hf_aem_clock_domain_id,
         {"Clock Domain ID", "ieee17221.clock_domain_id",
            FT_UINT16, BASE_DEC, NULL, 0x0, NULL, HFILL }
      },
      { &hf_aem_clock_sources_offset,
         {"Clock Sources Offset", "ieee17221.clock_sources_offset",
            FT_UINT16, BASE_DEC, NULL, 0x0, NULL, HFILL }
      },
      { &hf_aem_clock_source_index,
         {"Clock Source Index", "ieee17221.clock_source_index",
            FT_UINT16, BASE_DEC, NULL, 0x0, NULL, HFILL }
      },
      { &hf_aem_clock_sources_count,
         {"Clock Sources Count", "ieee17221.clock_sources_count",
            FT_UINT16, BASE_DEC, NULL, 0x0, NULL, HFILL }
      },
      { &hf_aem_clock_sources,
         {"Clock Sources", "ieee17221.clock_sources",
            FT_UINT16, BASE_DEC, NULL, 0x0, NULL, HFILL }
      },
      { &hf_aem_clock_sources_array,
         {"Clock Sources Array", "ieee17221.clock_sources_array",
            FT_NONE, BASE_NONE, NULL, 0x0, NULL, HFILL }
      },
      { &hf_aem_number_of_controls,
         {"Number Of Controls", "ieee17221.number_of_controls",
            FT_UINT16, BASE_DEC, NULL, 0x0, NULL, HFILL }
      },
      { &hf_aem_base_control,
         {"Base Control", "ieee17221.base_control",
            FT_UINT16, BASE_DEC, NULL, 0x0, NULL, HFILL }
      },
      { &hf_aem_number_control_blocks,
         {"Number Of Control Blocks", "ieee17221.number_of_control_blocks",
            FT_UINT16, BASE_DEC, NULL, 0x0, NULL, HFILL }
      },
      { &hf_aem_base_control_block,
         {"Base Control Block", "ieee17221.base_control_block",
            FT_UINT16, BASE_DEC, NULL, 0x0, NULL, HFILL }
      },
      { &hf_aem_object_name,
         {"Object Name", "ieee17221.object_name",
            FT_STRING, BASE_NONE, NULL, 0x0, NULL, HFILL }
      },
      { &hf_aem_localized_description,
         {"Localized Description", "ieee17221.localized_description",
            FT_UINT16, BASE_DEC, NULL, 0x0, NULL, HFILL }
      },
      { &hf_aem_current_sample_rate,
         {"Current Sample Rate", "ieee17221.current_sample_rate",
            FT_UINT32, BASE_DEC, NULL, 0x0, NULL, HFILL }
      },
      { &hf_aem_number_signal_selectors,
         {"Number of Signal Selectors", "ieee17221.num_signal_selectors",
            FT_UINT16, BASE_DEC, NULL, 0x0, NULL, HFILL }
      },
      { &hf_aem_base_signal_selector,
         {"Base Signal Selector", "ieee17221.base_signal_selector",
            FT_UINT16, BASE_DEC, NULL, 0x0, NULL, HFILL }
      },
      { &hf_aem_number_mixers,
         {"Number of Mixers", "ieee17221.num_mixers",
            FT_UINT16, BASE_DEC, NULL, 0x0, NULL, HFILL }
      },
      { &hf_aem_base_mixer,
         {"Base Mixer", "ieee17221.base_mixer",
            FT_UINT16, BASE_DEC, NULL, 0x0, NULL, HFILL }
      },
      { &hf_aem_number_matrices,
         {"Number of Matrices", "ieee17221.num_matrices",
            FT_UINT16, BASE_DEC, NULL, 0x0, NULL, HFILL }
      },
      { &hf_aem_base_matrix,
         {"Base Matrix", "ieee17221.base_matrix",
            FT_UINT16, BASE_DEC, NULL, 0x0, NULL, HFILL }
      },
      { &hf_aem_number_splitters,
         {"Number of Splitters", "ieee17221.num_splitters",
            FT_UINT16, BASE_DEC, NULL, 0x0, NULL, HFILL }
      },
      { &hf_aem_base_splitter,
         {"Base Splitter", "ieee17221.base_splitter",
            FT_UINT16, BASE_DEC, NULL, 0x0, NULL, HFILL }
      },
      { &hf_aem_number_combiners,
         {"Number of Combiners", "ieee17221.num_combiners",
            FT_UINT16, BASE_DEC, NULL, 0x0, NULL, HFILL }
      },
      { &hf_aem_base_combiner,
         {"Base Combiner", "ieee17221.base_combiner",
            FT_UINT16, BASE_DEC, NULL, 0x0, NULL, HFILL }
      },
      { &hf_aem_number_demultiplexers,
         {"Number of Demultiplexers", "ieee17221.num_demultiplexer",
            FT_UINT16, BASE_DEC, NULL, 0x0, NULL, HFILL }
      },
      { &hf_aem_base_demultiplexer,
         {"Base Demultiplexer", "ieee17221.base_demultiplexer",
            FT_UINT16, BASE_DEC, NULL, 0x0, NULL, HFILL }
      },
      { &hf_aem_number_multiplexers,
         {"Number of Multiplexers", "ieee17221.num_multiplexers",
            FT_UINT16, BASE_DEC, NULL, 0x0, NULL, HFILL }
      },
      { &hf_aem_base_multiplexer,
         {"Base Multiplexer", "ieee17221.base_multiplexer",
            FT_UINT16, BASE_DEC, NULL, 0x0, NULL, HFILL }
      },
      { &hf_aem_number_transcoders,
         {"Number of Transcoders", "ieee17221.num_transcoders",
            FT_UINT16, BASE_DEC, NULL, 0x0, NULL, HFILL }
      },
      { &hf_aem_base_transcoder,
         {"Base Transcoder", "ieee17221.base_transcoder",
            FT_UINT16, BASE_DEC, NULL, 0x0, NULL, HFILL }
      },
      /* sample rate is 3 bit pull field multiplier and 29 bit base freq in Hz */
      { &hf_aem_sample_rates_offset,
         {"Sample Rates Offset", "ieee17221.sample_rates_offset",
            FT_UINT16, BASE_DEC, NULL, 0x0, NULL, HFILL }
      },
      { &hf_aem_sample_rates_count,
         {"Sample Rates Count", "ieee17221.sample_rates_count",
            FT_UINT16, BASE_DEC, NULL, 0x0, NULL, HFILL }
      },
      { &hf_aem_sample_rates,
         {"Sample Rates", "ieee17221.sample_rates",
            FT_NONE, BASE_NONE, NULL, 0x0, NULL, HFILL }
      },

      { &hf_aem_base_frequency,
         {"Base Frequency", "ieee17221.base_frequency",
            FT_UINT32, BASE_DEC, NULL, AEM_BASE_FREQUENCY_MASK, NULL, HFILL }
      },
      { &hf_aem_pull_field,
         {"Pull Field (frequency multiplier)", "ieee17221.pull_field",
            FT_UINT8, BASE_HEX, VALS(aem_frequency_multiplier_type_vals), AEM_PULL_FIELD_MASK, NULL, HFILL }
      },

      /* VIDEO */

      /* SENSOR */

      /* STREAM_INPUT */
      /* STREAM_OUTPUT */
      { &hf_aem_stream_name,
         {"Object Name", "ieee17221.stream_name",
            FT_STRING, BASE_NONE, NULL, 0x0, NULL, HFILL }
      },
      { &hf_aem_stream_name_string,
         {"Localized Description", "ieee17221.stream_name_string",
            FT_UINT16, BASE_DEC, NULL, 0x0, NULL, HFILL }
      },
      { &hf_aem_stream_flags,
         {"Stream Flags", "ieee17221.stream_flags",
            FT_UINT16, BASE_HEX, NULL, 0x0, NULL, HFILL }
      },
      { &hf_aem_flags_clock_sync_source,
         {"Clock Sync Source Flag", "ieee17221.flags.clock_sync_source",
            FT_BOOLEAN, 16, NULL, AEM_CLOCK_SYNC_SOURCE_FLAG_MASK, NULL, HFILL }
      },
      { &hf_aem_flags_class_a,
         {"Class A Flag", "ieee17221.flags.class_a",
            FT_BOOLEAN, 16, NULL, AEM_CLASS_A_FLAG_MASK, NULL, HFILL }
      },
      { &hf_aem_flags_class_b,
         {"Class B Flag", "ieee17221.flags.class_b",
            FT_BOOLEAN, 16, NULL, AEM_CLASS_B_FLAG_MASK, NULL, HFILL }
      },
#if 0
      { &hf_aem_stream_channels,
         {"Stream Channels", "ieee17221.stream_channels",
            FT_UINT16, BASE_DEC, NULL, 0x0, NULL, HFILL }
      },
#endif
      { &hf_aem_formats_offset,
         {"Formats Offset", "ieee17221.formats_offset",
            FT_UINT16, BASE_DEC, NULL, 0x0, NULL, HFILL }
      },
      { &hf_aem_number_of_formats,
         {"Number Of Formats", "ieee17221.number_of_formats",
            FT_UINT16, BASE_DEC, NULL, 0x0, NULL, HFILL }
      },
      { &hf_aem_backup_talker_guid_0,
         {"Primary Backup Talker GUID", "ieee17221.backup_talker_guid_0",
            FT_UINT64, BASE_HEX, NULL, 0x0, NULL, HFILL }
      },
      { &hf_aem_backup_talker_unique_0,
         {"Primary Backup Talker Unique ID", "ieee17221.backup_talker_unique_0",
            FT_UINT16, BASE_HEX, NULL, 0x0, NULL, HFILL }
      },
      { &hf_aem_backup_talker_guid_1,
         {"Secondary Backup Talker GUID", "ieee17221.backup_talker_guid_1",
            FT_UINT64, BASE_HEX, NULL, 0x0, NULL, HFILL }
      },
      { &hf_aem_backup_talker_unique_1,
         {"Secondary Backup Talker Unique ID", "ieee17221.backup_talker_unique_1",
            FT_UINT16, BASE_HEX, NULL, 0x0, NULL, HFILL }
      },
      { &hf_aem_backup_talker_guid_2,
         {"Tertiary Backup Talker GUID", "ieee17221.backup_talker_guid_2",
            FT_UINT64, BASE_HEX, NULL, 0x0, NULL, HFILL }
      },
      { &hf_aem_backup_talker_unique_2,
         {"Tertiary Backup Talker Unique ID", "ieee17221.backup_talker_unique_2",
            FT_UINT16, BASE_HEX, NULL, 0x0, NULL, HFILL }
      },
      { &hf_aem_backedup_talker_guid,
         {"Backedup Talker GUID", "ieee17221.backedup_talker_guid",
            FT_UINT64, BASE_HEX, NULL, 0x0, NULL, HFILL }
      },
      { &hf_aem_backedup_talker_unique,
         {"Backedup Talker Unique ID", "ieee17221.backedup_talker_unique",
            FT_UINT16, BASE_HEX, NULL, 0x0, NULL, HFILL }
      },
      { &hf_aem_avb_interface_id,
         {"AVB Interface ID", "ieee17221.avb_interface_id",
            FT_UINT16, BASE_DEC, NULL, 0x0, NULL, HFILL }
      },
      { &hf_aem_buffer_length,
         {"Buffer Length", "ieee17221.buffer_length",
            FT_UINT32, BASE_DEC, NULL, 0x0, NULL, HFILL }
      },

      /* array head. uses ett_aem_stream_formats */
      { &hf_aem_stream_formats,
         {"Stream Formats Array", "ieee17221.stream_formats",
            FT_NONE, BASE_NONE, NULL, 0x0, NULL, HFILL }
      },

      /* EXTERNAL_JACK_INPUT */
      /* EXTERNAL_JACK_OUTPUT*/
      { &hf_aem_jack_name,
         {"Object Name", "ieee17221.jack_name",
            FT_STRING, BASE_NONE, NULL, 0x0, NULL, HFILL }
      },
      { &hf_aem_jack_name_string,
         {"Localized Description", "ieee17221.jack_name_string",
            FT_UINT16, BASE_DEC, NULL, 0x0, NULL, HFILL }
      },
      { &hf_aem_interface_name,
         {"Interface Name", "ieee17221.interface_name",
            FT_STRING, BASE_NONE, NULL, 0x0, NULL, HFILL }
      },
      { &hf_aem_interface_name_string,
         {"Interface Name String", "ieee17221.interface_name_string",
            FT_UINT16, BASE_DEC, NULL, 0x0, NULL, HFILL }
      },
      { &hf_aem_jack_flags,
         {"Jack Flags", "ieee17221.jack_flags",
            FT_UINT16, BASE_HEX, NULL, 0x0, NULL, HFILL }
      },
      { &hf_aem_flags_captive,
         {"Captive Flag", "ieee17221.flags.captive",
            FT_BOOLEAN, 16, NULL, AEM_CAPTIVE_FLAG_MASK, NULL, HFILL }
      },
      { &hf_aem_jack_type,
         {"Jack Type", "ieee17221.jack_type",
            FT_UINT16, BASE_HEX, VALS(aem_jack_type_vals), 0x00, NULL, HFILL }
      },
      /* AUDIO_PORT_INPUT */
      /* AUDIO_PORT_OUTPUT */
      { &hf_aem_port_flags,
         {"Port Flags", "ieee17221.port_flags",
            FT_UINT16, BASE_HEX, NULL, 0x0, NULL, HFILL }
      },
      { &hf_aem_flags_async_sample_rate_conv,
         {"Asynchronous Sample Rate Converter Flag", "ieee17221.flags.async_sample_rate_conv",
            FT_BOOLEAN, 16, NULL, AEM_ASYNC_SAMPLE_RATE_CONV_FLAG_MASK, NULL, HFILL }
      },
      { &hf_aem_flags_sync_sample_rate_conv,
         {"Synchronous Sample Rate Converter Flag", "ieee17221.flags.sync_sample_rate_conv",
            FT_BOOLEAN, 16, NULL, AEM_SYNC_SAMPLE_RATE_CONV_FLAG_MASK, NULL, HFILL }
      },
#if 0
      { &hf_aem_audio_channels,
         {"Audio Channels", "ieee17221.audio_channels",
            FT_UINT16, BASE_DEC, NULL, 0x0, NULL, HFILL }
      },
#endif
      { &hf_aem_number_of_clusters,
         {"Number of Clusters", "ieee17221.number_of_clusters",
            FT_UINT16, BASE_DEC, NULL, 0x0, NULL, HFILL }
      },
      { &hf_aem_base_cluster,
         {"Base Cluster", "ieee17221.base_cluster",
            FT_UINT16, BASE_DEC, NULL, 0x0, NULL, HFILL }
      },
      { &hf_aem_base_audio_map,
         {"Base Audio Map", "ieee17221.base_audio_map",
            FT_UINT16, BASE_DEC, NULL, 0x0, NULL, HFILL }
      },
      { &hf_aem_number_audio_maps,
         {"Number of Audio Maps", "ieee17221.num_audio_maps",
            FT_UINT16, BASE_DEC, NULL, 0x0, NULL, HFILL }
      },
      /* VIDEO_PORT_INPUT */
      /* VIDEO_PORT_OUTPUT */
#if 0
      { &hf_aem_stream_id,
         {"Stream Descriptor ID", "ieee17221.stream_descriptor_id",
            FT_UINT16, BASE_DEC, NULL, 0x0, NULL, HFILL }
      },
      { &hf_aem_formats_count,
         {"Formats Count", "ieee17221.formats_count",
            FT_UINT16, BASE_DEC, NULL, 0x0, NULL, HFILL }
      },
#endif

      /* EXTERNAL_PORT_INPUT */
      /* EXTERNAL_PORT_OUTPUT */
      { &hf_aem_jack_id,
         {"Jack ID", "ieee17221.jack_id",
            FT_UINT16, BASE_HEX, NULL, 0x0, NULL, HFILL }
      },
      /* SENSOR_PORT_INPUT */
      /* SENSOR_PORT_OUTPUT */
      /* INTERNAL_PORT_INPUT */
      /* INTERNAL_PORT_OUTPUT */
#if 0
      { &hf_aem_internal_id,
         {"Internal ID", "ieee17221.internal_id",
            FT_UINT16, BASE_DEC, NULL, 0x0, NULL, HFILL }
      },
#endif
      /* AVB_INTERFACE */
     { &hf_aem_avb_interface_flags,
         {"Interface Flags", "ieee17221.interface_flags",
            FT_UINT16, BASE_DEC, NULL, 0x0, NULL, HFILL }
      },
      { &hf_aem_avb_clock_identity,
         {"Clock Identity", "ieee17221.clock_identity",
            FT_UINT64, BASE_HEX, NULL, 0x0, NULL, HFILL }
      },
      { &hf_aem_avb_priority1,
         {"Priority1", "ieee17221.priority1",
            FT_UINT8, BASE_DEC, NULL, 0x0, NULL, HFILL }
      },
      { &hf_aem_avb_clock_class,
         {"Clock Class", "ieee17221.clock_class",
            FT_UINT8, BASE_DEC, NULL, 0x0, NULL, HFILL }
      },
      { &hf_aem_avb_offset_scaled_log_variance,
         {"Scaled Log Variance", "ieee17221.offset_scaled_log_variance",
            FT_UINT16, BASE_DEC, NULL, 0x0, NULL, HFILL }
      },
      { &hf_aem_avb_clock_accuracy,
         {"Clock Accuracy", "ieee17221.clock_accuracy",
            FT_UINT8, BASE_DEC, NULL, 0x0, NULL, HFILL }
      },
      { &hf_aem_avb_priority2,
         {"Priority2", "ieee17221.priority2",
            FT_UINT8, BASE_DEC, NULL, 0x0, NULL, HFILL }
      },
      { &hf_aem_avb_domain_number,
         {"Domain Number", "ieee17221.domain_number",
            FT_UINT8, BASE_DEC, NULL, 0x0, NULL, HFILL }
      },
      { &hf_aem_avb_log_sync_interval,
         {"Log sync Interval", "ieee17221.log_sync_interval",
            FT_INT8, BASE_DEC, NULL, 0x0, NULL, HFILL }
      },
#if 0
      { &hf_aem_avb_propagation_delay,
         {"Propagation Delay", "ieee17221.propagation_delay",
            FT_UINT32, BASE_DEC, NULL, 0x0, NULL, HFILL }
      },
#endif

      /* CLOCK_SOURCE */
      { &hf_aem_clock_source_name,
         {"Object Name", "ieee17221.clock_source_name",
            FT_STRING, BASE_NONE, NULL, 0x0, NULL, HFILL }
      },
      { &hf_aem_clock_source_name_string,
         {"Localized Description", "ieee17221.clock_source_name_string",
            FT_UINT16, BASE_DEC, NULL, 0x0, NULL, HFILL }
      },
      { &hf_aem_clock_source_flags,
         {"Clock Source Flags", "ieee17221.clock_source_flags",
            FT_UINT16, BASE_HEX, NULL, 0x0, NULL, HFILL }
      }, /* no flags currently defined */
      { &hf_aem_clock_source_type,
         {"Clock Source Type", "ieee17221.clock_source_type",
            FT_UINT16, BASE_HEX, VALS(aem_clock_source_type_vals), 0x00, NULL, HFILL }
      },
      { &hf_aem_clock_source_location_type,
         {"Clock Source Location Type", "ieee17221.clock_source_location_type",
            FT_UINT16, BASE_HEX, VALS(aem_descriptor_type_vals), 0x00, NULL, HFILL }
      },
      { &hf_aem_clock_source_location_id,
         {"Clock Source Location ID", "ieee17221.clock_source_location_id",
            FT_UINT16, BASE_DEC, NULL, 0x0, NULL, HFILL }
      },
      /* AUDIO_MAP */
      { &hf_aem_mappings_offset,
         {"Mappings Offset", "ieee17221.mappings_offset",
            FT_UINT16, BASE_DEC, NULL, 0x0, NULL, HFILL }
      },
      { &hf_aem_number_of_mappings,
         {"Number of Mappings", "ieee17221.number_of_mappings",
            FT_UINT16, BASE_DEC, NULL, 0x0, NULL, HFILL }
      },
      { &hf_aem_mappings,
         {"Mappings", "ieee17221.mappings",
            FT_NONE, BASE_NONE, NULL, 0x0, NULL, HFILL }
      },
      { &hf_aem_mapping_stream_index,
         {"Mapping Stream Index", "ieee17221.mapping_stream_index",
            FT_UINT16, BASE_DEC, NULL, 0x0, NULL, HFILL }
      },
      { &hf_aem_mapping_stream_channel,
         {"Mapping Stream Channel", "ieee17221.mapping_stream_channel",
            FT_UINT16, BASE_DEC, NULL, 0x0, NULL, HFILL }
      },
      { &hf_aem_mapping_cluster_offset,
         {"Mapping Cluster Offset", "ieee17221.mapping_cluster_offset",
            FT_UINT16, BASE_DEC, NULL, 0x0, NULL, HFILL }
      },
      { &hf_aem_mapping_cluster_channel,
         {"Mapping Cluster Channel", "ieee17221.mapping_cluster_channel",
            FT_UINT16, BASE_DEC, NULL, 0x0, NULL, HFILL }
      },
      /* AUDIO_CLUSTER */
      { &hf_aem_channel_count,
         {"Channel Count", "ieee17221.channel_count",
            FT_UINT16, BASE_DEC, NULL, 0x0, NULL, HFILL }
      },
      { &hf_aem_path_latency,
         {"Path Latency", "ieee17221.path_latency",
            FT_UINT32, BASE_DEC, NULL, 0x0, NULL, HFILL }
      },
      { &hf_aem_am824_label,
         {"AM824 Label", "ieee17221.am824_label",
            FT_UINT8, BASE_DEC, NULL, 0x0, NULL, HFILL }
      },
#if 0
      { &hf_aem_cluster_name,
         {"Cluster Name", "ieee17221.cluster_name",
            FT_STRING, BASE_NONE, NULL, 0x0, NULL, HFILL }
      },
      { &hf_aem_cluster_name_string,
         {"Cluster Name String", "ieee17221.cluster_name_string",
            FT_UINT16, BASE_DEC, NULL, 0x0, NULL, HFILL }
      },
#endif

      /* CONTROL */
      { &hf_aem_control_type,
         {"Control Type", "ieee17221.control_type",
            FT_UINT64, BASE_HEX, NULL, 0x0, NULL, HFILL }
      },
#if 0
      { &hf_aem_control_location_type,
         {"Control Location Type", "ieee17221.control_location_type",
            FT_UINT16, BASE_HEX, VALS(aem_descriptor_type_vals), 0x00, NULL, HFILL }
      },
      { &hf_aem_control_location_id,
         {"Control Location ID", "ieee17221.control_location_id",
            FT_UINT16, BASE_DEC, NULL, 0x0, NULL, HFILL }
      },
#endif
      { &hf_aem_control_value_type,
         {"Control Value Type", "ieee17221.control_value_type",
            FT_UINT16, BASE_HEX, VALS(aem_control_value_type_vals), 0x00, NULL, HFILL }
      },
      { &hf_aem_control_domain,
         {"Control Domain", "ieee17221.control_domain",
            FT_UINT16, BASE_DEC, NULL, 0x0, NULL, HFILL }
      },
      { &hf_aem_values_offset,
         {"Values Offset", "ieee17221.values_offset",
            FT_UINT16, BASE_DEC, NULL, 0x0, NULL, HFILL }
      },
      { &hf_aem_number_of_values,
         {"Number Of Values", "ieee17221.number_of_values",
            FT_UINT16, BASE_DEC, NULL, 0x0, NULL, HFILL }
      },
      { &hf_aem_control_latency,
         {"Control Latency", "ieee17221.control_latency",
            FT_UINT32, BASE_DEC, NULL, 0x0, NULL, HFILL }
      },

      /* SIGNAL_SELECTOR */
#if 0
      { &hf_aem_sources_offset,
         {"Sources Offset", "ieee17221.sources_offset",
            FT_UINT16, BASE_DEC, NULL, 0x0, NULL, HFILL }
      },
      { &hf_aem_current_signal_type,
         {"Current Signal Type", "ieee17221.current_signal_type",
            FT_UINT16, BASE_HEX, VALS(aem_descriptor_type_vals), 0x00, NULL, HFILL }
      },
      { &hf_aem_current_signal_id,
         {"Current Signal ID", "ieee17221.current_signal_id",
            FT_UINT16, BASE_DEC, NULL, 0x0, NULL, HFILL }
      },
      { &hf_aem_default_signal_type,
         {"Default Signal Type", "ieee17221.default_signal_type",
            FT_UINT16, BASE_HEX, VALS(aem_descriptor_type_vals), 0x00, NULL, HFILL }
      },
      { &hf_aem_default_signal_id,
         {"Default Signal ID", "ieee17221.default_signal_id",
            FT_UINT16, BASE_DEC, NULL, 0x0, NULL, HFILL }
      },
#endif
      { &hf_aem_block_latency,
         {"Block Latency", "ieee17221.block_latency",
            FT_UINT32, BASE_DEC, NULL, 0x0, NULL, HFILL }
      },
      { &hf_aem_reset_time,
         {"Reset Time", "ieee17221.reset_time",
            FT_UINT32, BASE_DEC, NULL, 0x0, NULL, HFILL }
      },
      { &hf_aem_signal_type,
         {"Signal Type", "ieee17221.signal_type",
            FT_UINT16, BASE_HEX, VALS(aem_descriptor_type_vals), 0x00, NULL, HFILL }
      },
      { &hf_aem_signal_index,
         {"Signal Index", "ieee17221.signal_index",
            FT_UINT16, BASE_DEC, NULL, 0x0, NULL, HFILL }
      },
      { &hf_aem_signal_output,
         {"Signal Output", "ieee17221.signal_output",
            FT_UINT16, BASE_DEC, NULL, 0x0, NULL, HFILL }
      },

      /* MIXER */
#if 0
      { &hf_aem_number_of_sources,
         {"Number of Sources", "ieee17221.number_of_sources",
            FT_UINT16, BASE_DEC, NULL, 0x0, NULL, HFILL }
      },
      { &hf_aem_value_offset,
         {"Value Offset", "ieee17221.value_offset",
            FT_UINT16, BASE_DEC, NULL, 0x0, NULL, HFILL }
      },

      /* MATRIX */
      { &hf_aem_width,
         {"Width", "ieee17221.width",
            FT_UINT16, BASE_DEC, NULL, 0x0, NULL, HFILL }
      },
      { &hf_aem_height,
         {"Height", "ieee17221.height",
            FT_UINT16, BASE_DEC, NULL, 0x0, NULL, HFILL }
      },
      { &hf_aem_base_source,
         {"Base Source", "ieee17221.base_source",
            FT_UINT16, BASE_HEX, NULL, 0x0, NULL, HFILL }
      },
      { &hf_aem_number_destinations,
         {"Number of Destinations", "ieee17221.num_destinations",
            FT_UINT16, BASE_DEC, NULL, 0x0, NULL, HFILL }
      },
      { &hf_aem_base_destination,
         {"Base Destination", "ieee17221.base_destination",
            FT_UINT16, BASE_HEX, NULL, 0x0, NULL, HFILL }
      },
#endif

      /* LOCALE */
      { &hf_aem_locale_identifier,
         {"Locale Identifier", "ieee17221.locale_identifier",
            FT_STRING, BASE_NONE, NULL, 0x0, NULL, HFILL }
      },
      { &hf_aem_number_of_strings,
         {"Number of Strings", "ieee17221.number_of_strings",
            FT_UINT16, BASE_DEC, NULL, 0x0, NULL, HFILL }
      },
      { &hf_aem_base_strings,
         {"Base Strings", "ieee17221.base_strings",
            FT_UINT16, BASE_DEC, NULL, 0x0, NULL, HFILL }
      },

      /* STRINGS */
      { &hf_aem_string,
         {"String", "ieee17221.string",
            FT_STRINGZ, BASE_NONE, NULL, 0x0, NULL, HFILL }
      },

      /* MATRIX SIGNAL */
      { &hf_aem_signals_count,
         {"Signals Count", "ieee17221.signals_count",
            FT_UINT16, BASE_DEC, NULL, 0x0, NULL, HFILL }
      },
      { &hf_aem_signals_offset,
         {"Signals Offset", "ieee17221.signals_offset",
            FT_UINT16, BASE_HEX, NULL, 0x0, NULL, HFILL }
      },

      /* MEMORY OBJECT */
      { &hf_aem_memory_object_type,
         {"Memory Object Type", "ieee17221.memory_object_type",
            FT_UINT16, BASE_HEX, VALS(aem_memory_object_type_vals), 0x00, NULL, HFILL }
      },
      { &hf_aem_target_descriptor_type,
         {"Target Descriptor Type", "ieee17221.target_descriptor_type",
            FT_UINT16, BASE_HEX, VALS(aem_descriptor_type_vals), 0x00, NULL, HFILL }
      },
      { &hf_aem_target_descriptor_id,
         {"Target Descriptor ID", "ieee17221.target_descriptor_id",
            FT_UINT16, BASE_HEX, NULL, 0x0, NULL, HFILL }
      },
      { &hf_aem_start_address,
         {"Start Address", "ieee17221.start_address",
            FT_UINT64, BASE_HEX, NULL, 0x0, NULL, HFILL }
      },
      { &hf_aem_length,
         {"Length", "ieee17221.length",
            FT_UINT64, BASE_DEC, NULL, 0x0, NULL, HFILL }
      },


      /* CONTROL VALUE TYPES */
      { &hf_aem_ctrl_int8,
         {"Control INT8", "ieee17221.ctrl_int8",
            FT_INT8, BASE_DEC, NULL, 0x0, NULL, HFILL }
      },
      { &hf_aem_ctrl_uint8,
         {"Control UINT8", "ieee17221.ctrl_uint8",
            FT_UINT8, BASE_DEC, NULL, 0x0, NULL, HFILL }
      },
      { &hf_aem_ctrl_int16,
         {"Control INT16", "ieee17221.ctrl_int16",
            FT_INT16, BASE_DEC, NULL, 0x0, NULL, HFILL }
      },
      { &hf_aem_ctrl_uint16,
         {"Control UINT16", "ieee17221.ctrl_uint16",
            FT_UINT16, BASE_DEC, NULL, 0x0, NULL, HFILL }
      },
      { &hf_aem_ctrl_int32,
         {"Control INT32", "ieee17221.ctrl_int32",
            FT_INT32, BASE_DEC, NULL, 0x0, NULL, HFILL }
      },
      { &hf_aem_ctrl_uint32,
         {"Control UINT32", "ieee17221.ctrl_uint32",
            FT_UINT32, BASE_DEC, NULL, 0x0, NULL, HFILL }
      },
      { &hf_aem_ctrl_int64,
         {"Control INT64", "ieee17221.ctrl_int64",
            FT_INT64, BASE_DEC, NULL, 0x0, NULL, HFILL }
      },
      { &hf_aem_ctrl_uint64,
         {"Control UINT64", "ieee17221.ctrl_uint64",
            FT_UINT64, BASE_DEC, NULL, 0x0, NULL, HFILL }
      },
      { &hf_aem_ctrl_float,
         {"Control FLOAT", "ieee17221.ctrl_float",
            FT_FLOAT, BASE_NONE, NULL, 0x0, NULL, HFILL }
      },
      { &hf_aem_ctrl_double,
         {"Control DOUBLE", "ieee17221.ctrl_double",
            FT_DOUBLE, BASE_NONE, NULL, 0x0, NULL, HFILL }
      },
      { &hf_aem_ctrl_vals,
         {"Control Values", "ieee17221.ctrl_vals",
            FT_NONE, BASE_NONE, NULL, 0x0, NULL, HFILL }
      },
      { &hf_aem_unit,
         {"Control Value Units", "ieee17221.units",
            FT_UINT16, BASE_HEX, NULL, 0x0, NULL, HFILL }
      },
      { &hf_aem_string_ref,
         {"String Reference", "ieee17221.string_ref",
            FT_UINT16, BASE_DEC, NULL, 0x0, NULL, HFILL }
      },
      { &hf_aem_guid,
         {"GUID", "ieee17221.guid",
            FT_UINT64, BASE_HEX, NULL, 0x0, NULL, HFILL }
      },
      { &hf_aem_blob_size,
         {"Blob Size", "ieee17221.blob_size",
            FT_UINT32, BASE_DEC, NULL, 0x0, NULL, HFILL }
      },
      { &hf_aem_binary_blob,
         {"Binary Blob", "ieee17221.binary_blob",
            FT_BYTES, BASE_NONE, NULL, 0x0, NULL, HFILL }
      },
      { &hf_aem_sources,
         {"Sources", "ieee17221.sources",
            FT_NONE, BASE_NONE, NULL, 0x0, NULL, HFILL }
      },
      { &hf_aem_unknown_descriptor,
         {"Unknown or Malformed Descriptor", "ieee17221.unknown_descriptor",
            FT_BYTES, BASE_NONE, NULL, 0x0, NULL, HFILL }
      },
      { &hf_aem_frequency,
         {"Frequency", "ieee17221.frequency",
            FT_FLOAT, BASE_NONE, NULL, 0x0, NULL, HFILL }
      },

      /* AEM MEDIA FORMAT FIELDS */
      /* Standard media formats are defined by always having the 24 most significant bits
       * of the EUI64 set to 0x90e0f0
       */
#if 0
      { &hf_aem_oui24,
         {"OUI-24", "ieee17221.oui24",
            FT_UINT24, BASE_HEX, NULL, 0x0, NULL, HFILL }
      },
      { &hf_aem_mfd_type,
         {"MFD Type", "ieee17221.mfd_type",
            FT_UINT8, BASE_HEX, VALS(aem_mfd_type_vals), 0x00, NULL, HFILL }
      },
      { &hf_aem_div,
         {"Div Flag", "ieee17221.div",
            FT_BOOLEAN, BASE_NONE, NULL, 0x0, NULL, HFILL }
      },
      { &hf_aem_interlace,
         {"Interlace Flag", "ieee17221.interlace",
            FT_BOOLEAN, BASE_NONE, NULL, 0x0, NULL, HFILL }
      },
      { &hf_aem_channels,
         {"Video Channel Count", "ieee17221.channels",
            FT_UINT8, BASE_DEC, NULL, AEM_MASK_CHANNELS, NULL, HFILL }
      },
      { &hf_aem_color_format,
         {"Color Format", "ieee17221.color_format",
            FT_UINT16, BASE_HEX, VALS(aem_color_format_type_vals), AEM_MASK_COLOR_FORMAT, NULL, HFILL }
      },
      { &hf_aem_bpp,
         {"Bits Per Pixel", "ieee17221.bpp",
            FT_UINT8, BASE_DEC, NULL, AEM_MASK_BPP, NULL, HFILL }
      },
      { &hf_aem_aspect_x,
         {"Aspect X", "ieee17221.aspect_x",
            FT_UINT8, BASE_DEC, NULL, 0x0, NULL, HFILL }
      },
      { &hf_aem_aspect_y,
         {"Aspect Y", "ieee17221.aspect_y",
            FT_UINT8, BASE_DEC, NULL, 0x0, NULL, HFILL }
      },
      { &hf_aem_frame_rate,
         {"Frame Rate", "ieee17221.frame_rate",
            FT_UINT8, BASE_DEC, NULL, 0x0, NULL, HFILL }
      },
      { &hf_aem_comp1,
         {"Comp 1", "ieee17221.comp1",
            FT_UINT16, BASE_DEC, NULL, AEM_MASK_COMP1, NULL, HFILL }
      },
      { &hf_aem_comp2,
         {"Comp 2", "ieee17221.comp2",
            FT_UINT8, BASE_DEC, NULL, AEM_MASK_COMP2, NULL, HFILL }
      },
      { &hf_aem_comp3,
         {"Comp 3", "ieee17221.comp3",
            FT_UINT16, BASE_DEC, NULL, AEM_MASK_COMP3, NULL, HFILL }
      },
      { &hf_aem_comp4,
         {"Comp 4", "ieee17221.comp4",
            FT_UINT8, BASE_DEC, NULL, AEM_MASK_COMP4, NULL, HFILL }
      },
      { &hf_aem_mf_width,
         {"Width", "ieee17221.width",
            FT_UINT16, BASE_DEC, NULL, 0x0, NULL, HFILL }
      },
      { &hf_aem_mf_height,
         {"Height", "ieee17221.height",
            FT_UINT16, BASE_DEC, NULL, 0x0, NULL, HFILL }
      },
      { &hf_aem_cs_eui64,
         {"CS EUI64", "ieee17221.cs_eui64",
            FT_UINT64, BASE_HEX, NULL, 0x0, NULL, HFILL }
      },
#endif
      /* BEGIN STREAM FORMAT (SF) FIELDS */
      { &hf_aem_stream_format,
         {"Stream Format", "ieee17221.stream_format",
            FT_NONE, BASE_NONE, NULL, 0x0, NULL, HFILL }
      },
      { &hf_aem_sf_version,
         {"Version", "ieee17221.sf_version",
            FT_UINT8, BASE_HEX, NULL, AEM_MASK_SF_VERSION, NULL, HFILL }
      },
      { &hf_aem_sf_subtype,
         {"Subtype", "ieee17221.sf_subtype",
            FT_UINT8, BASE_HEX, NULL, AEM_MASK_SF_SUBTYPE, NULL, HFILL }
      },

      { &hf_aem_sf,
         {"SF", "ieee17221.sf",
            FT_BOOLEAN, 8, NULL, AEM_MASK_SF, NULL, HFILL }
      },
      { &hf_aem_iidc_format,
         {"IIDC Format", "ieee17221.iidc_format",
            FT_UINT8, BASE_HEX, NULL, 0x0, NULL, HFILL }
      },
      { &hf_aem_iidc_mode,
         {"IIDC Mode", "ieee17221.iidc_mode",
            FT_UINT8, BASE_HEX, NULL, 0x0, NULL, HFILL }
      },
      { &hf_aem_iidc_rate,
         {"IIDC Rate", "ieee17221.iidc_rate",
            FT_UINT8, BASE_HEX, NULL, 0x0, NULL, HFILL }
      },
      { &hf_aem_fmt,
         {"FMT", "ieee17221.fmt",
            FT_UINT8, BASE_HEX, NULL, AEM_MASK_FMT, NULL, HFILL }
      },
      { &hf_aem_fdf_evt,
         {"FDF EVT", "ieee17221.fdf_evt",
            FT_UINT8, BASE_HEX, NULL, AEM_MASK_FDF_EVT, NULL, HFILL }
      },
      { &hf_aem_fdf_sfc,
         {"FDF SFC", "ieee17221.fdf_sfc",
            FT_UINT8, BASE_HEX, NULL, AEM_MASK_FDF_SFC, NULL, HFILL }
      },
      { &hf_aem_dbs,
         {"DBS", "ieee17221.dbs",
            FT_UINT8, BASE_HEX, NULL, 0x0, NULL, HFILL }
      },
      { &hf_aem_b_flag,
         {"Blocking Flag", "ieee17221.flags.b",
            FT_BOOLEAN, 8, NULL, AEM_MASK_B, NULL, HFILL }
      },
      { &hf_aem_nb_flag,
         {"NonBlocking Flag", "ieee17221.flags.nb",
            FT_BOOLEAN, 8, NULL, AEM_MASK_NB, NULL, HFILL }
      },
      { &hf_aem_label_iec_60958_cnt,
         {"Label IEC 60958 Count", "ieee17221.label_iec_60958_cnt",
            FT_UINT8, BASE_DEC, NULL, 0x0, NULL, HFILL }
      },
      { &hf_aem_label_mbla_cnt,
         {"Label Multi-Bit Linear Audio Count", "ieee17221.label_mbla_cnt",
            FT_UINT8, BASE_DEC, NULL, 0x0, NULL, HFILL }
      },
      { &hf_aem_label_midi_cnt,
         {"Label Midi Slot Count", "ieee17221.label_midi_cnt",
            FT_UINT8, BASE_DEC, NULL, AEM_MASK_LABEL_MIDI_CNT, NULL, HFILL }
      },
      { &hf_aem_label_smpte_cnt,
         {"Label SMPTE Slot Count", "ieee17221.label_smpte_cnt",
            FT_UINT8, BASE_DEC, NULL, AEM_MASK_LABEL_SMPTE_CNT, NULL, HFILL }
      },
      { &hf_aem_video_mode,
         {"Video Mode", "ieee17221.video_mode",
            FT_UINT8, BASE_HEX, NULL, 0x0, NULL, HFILL }
      },
      { &hf_aem_compress_mode,
         {"Compress Mode", "ieee17221.compress_mode",
            FT_UINT8, BASE_HEX, NULL, 0x0, NULL, HFILL }
      },
      { &hf_aem_color_space,
         {"Color Space", "ieee17221.color_space",
            FT_UINT16, BASE_HEX, NULL, 0x0, NULL, HFILL }
      },
#if 0
      { &hf_aecp_values,
         {"Values", "ieee17221.values",
            FT_BYTES, BASE_NONE, NULL, 0x0, NULL, HFILL }
      },
#endif
      { &hf_aecp_status_code,
         {"Status", "ieee17221.status",
            FT_UINT8, BASE_HEX, VALS(aem_status_type_vals), AECP_STATUS_CODE_MASK, NULL, HFILL }
      },
      /* END STREAM FORMAT (SF) FIELDS */

      /* SET/GET_STREAM_BACKUP */
      { &hf_aecp_backup_talker_entity_id_0,
         {"Backup Talker Entity ID 0", "ieee17221.backup_talker_entity_id_0",
            FT_UINT64, BASE_HEX, NULL, 0x0, NULL, HFILL }
      },
      { &hf_aecp_backup_talker_unique_id_0,
         {"Backup Talker Unique ID 0", "ieee17221.backup_talker_unique_id_0",
            FT_UINT16, BASE_DEC, NULL, 0x0, NULL, HFILL }
      },
      { &hf_aecp_backup_talker_entity_id_1,
         {"Backup Talker Entity ID 1", "ieee17221.backup_talker_entity_id_1",
            FT_UINT64, BASE_HEX, NULL, 0x0, NULL, HFILL }
      },
      { &hf_aecp_backup_talker_unique_id_1,
         {"Backup Talker Unique ID 1", "ieee17221.backup_talker_unique_id_1",
            FT_UINT16, BASE_DEC, NULL, 0x0, NULL, HFILL }
      },
      { &hf_aecp_backup_talker_entity_id_2,
         {"Backup Talker Entity ID 2", "ieee17221.backup_talker_entity_id_2",
            FT_UINT64, BASE_HEX, NULL, 0x0, NULL, HFILL }
      },
      { &hf_aecp_backup_talker_unique_id_2,
         {"Backup Talker Unique ID 2", "ieee17221.backup_talker_unique_id_2",
            FT_UINT16, BASE_DEC, NULL, 0x0, NULL, HFILL }
      },
      { &hf_aecp_backedup_talker_entity_id,
         {"Backed up Talker Entity ID", "ieee17221.backedup_talker_entity_id",
            FT_UINT64, BASE_HEX, NULL, 0x0, NULL, HFILL }
      },
      { &hf_aecp_backedup_talker_unique_id,
         {"Backedup Talker Unique ID", "ieee17221.backedup_talker_unique_id",
            FT_UINT16, BASE_DEC, NULL, 0x0, NULL, HFILL }
      },
      { &hf_aecp_vendor_unique_protocol_id,
         {"Vendor Unique Protocol ID", "ieee17221.protocol_id",
            FT_UINT48, BASE_HEX, NULL, 0x0, NULL, HFILL }
      }
   };

   /* Setup protocol subtree array */
   static gint *ett[] = {
      &ett_17221,
      &ett_adp_ent_cap,
      &ett_adp_talk_cap,
      &ett_adp_list_cap,
      &ett_adp_cont_cap,
/*       &ett_adp_aud_format, */
      &ett_adp_samp_rates,
      &ett_adp_chan_format,
      &ett_acmp_flags,
      &ett_aem_desc_counts,
      &ett_aem_descriptor,
      &ett_aem_sample_rates,
      &ett_aem_stream_flags,
      &ett_aem_clock_sources,
      &ett_aem_stream_formats,
      &ett_aem_jack_flags,
      &ett_aem_port_flags,
      &ett_aecp_get_avb_info_msrp_mappings,
      &ett_aecp_get_as_path_sequences,
      &ett_aem_clock_source_flags,
      &ett_aem_mappings,
      &ett_aem_ctrl_vals,
      &ett_aem_sources,
/*       &ett_aem_media_format, */
      &ett_aecp_descriptors,
      &ett_aecp_flags_32,
      &ett_aem_stream_format,
   };

   /* Register the protocol name and description */
   proto_17221 = proto_register_protocol("IEEE 1722.1 Protocol", "IEEE1722.1", "ieee17221");
   avb17221_handle = register_dissector("ieee17221", dissect_17221, proto_17221);

   /* Required function calls to register the header fields and subtrees used */
   proto_register_field_array(proto_17221, hf, array_length(hf));
   proto_register_subtree_array(ett, array_length(ett));

   /* Register the custom handler for vendor specific messages */
   vendor_unique_protocol_dissector_table = register_dissector_table("ieee17221.protocol_id", "Vendor Unique Protocol ID", proto_17221, FT_STRING, STRING_CASE_SENSITIVE);
}

void
proto_reg_handoff_17221(void)
{
   dissector_add_uint("ieee1722.subtype", 0xFA, avb17221_handle);
   dissector_add_uint("ieee1722.subtype", 0xFB, avb17221_handle);
   dissector_add_uint("ieee1722.subtype", 0xFC, avb17221_handle);
}

/*
 * Editor modelines  -  https://www.wireshark.org/tools/modelines.html
 *
 * Local variables:
 * c-basic-offset: 3
 * tab-width: 8
 * indent-tabs-mode: nil
 * End:
 *
 * vi: set shiftwidth=3 tabstop=8 expandtab:
 * :indentSize=3:tabSize=8:noTabs=true:
 */
