/* wireshark_preference.cpp
 *
 * Wireshark - Network traffic analyzer
 * By Gerald Combs <gerald@wireshark.org>
 * Copyright 1998 Gerald Combs
 *
 * SPDX-License-Identifier: GPL-2.0-or-later
 */

#include "config.h"

#include <epan/prefs.h>
#include <epan/prefs-int.h>

#include <ui/qt/manager/wireshark_preference.h>
#include <ui/qt/manager/preference_manager.h>
#include <ui/qt/widgets/range_syntax_lineedit.h>
#include "ui/qt/widgets/wireshark_file_dialog.h"
#include <ui/qt/wireshark_application.h>
#include <ui/qt/uat_dialog.h>

#include <QDir>
#include <QLineEdit>
#include <QComboBox>
#include <QColorDialog>

WiresharkPreference::WiresharkPreference(QObject * parent) : QObject(parent), _prefsItem(NULL)
{}

QWidget * WiresharkPreference::editor(QWidget * /*parent*/, const QStyleOptionViewItem &/*option*/, const QModelIndex &/*index*/)
{
    return Q_NULLPTR;
}

void WiresharkPreference::setData(QWidget * /*editor*/, const QModelIndex &/*index*/) {}
void WiresharkPreference::setModelData(QWidget * /*editor*/, QAbstractItemModel * /*model*/, const QModelIndex &/*index*/) {}

void WiresharkPreference::setPrefsItem(PrefsItem * item)
{
    _prefsItem = item;
}

PrefsItem * WiresharkPreference::prefsItem() const
{
    return _prefsItem;
}

class BoolPreference : public WiresharkPreference
{
public:
    BoolPreference(QObject * parent = Q_NULLPTR) : WiresharkPreference(parent) {}
    virtual QWidget * editor(QWidget *parent, const QStyleOptionViewItem &option, const QModelIndex &index)
    {
        ((QAbstractItemModel*)index.model())->setData(index, QString("BOOL"), Qt::EditRole);
        return WiresharkPreference::editor(parent, option, index);
    }
};
REGISTER_PREFERENCE_TYPE(PREF_BOOL, BoolPreference)

class StringPreference : public WiresharkPreference
{
public:
    StringPreference(QObject * parent = Q_NULLPTR) : WiresharkPreference(parent) {}
    virtual QWidget * editor(QWidget *parent, const QStyleOptionViewItem &/*option*/, const QModelIndex &/*index*/)
    {
        return new QLineEdit(parent);
    }

    virtual void setData(QWidget *editor, const QModelIndex &index)
    {
        QLineEdit* line = static_cast<QLineEdit*>(editor);
        line->setText(index.model()->data(index, Qt::DisplayRole).toString());
    }

    virtual void setModelData(QWidget *editor, QAbstractItemModel *model, const QModelIndex &index)
    {
        QLineEdit* line = static_cast<QLineEdit*>(editor);
        model->setData(index, line->text(), Qt::EditRole);
    }
};
REGISTER_PREFERENCE_TYPE(PREF_STRING, StringPreference)
REGISTER_PREFERENCE_TYPE(PREF_CUSTOM, StringPreference)

class UIntPreference : public StringPreference
{
public:
    UIntPreference(QObject * parent = Q_NULLPTR) : StringPreference(parent) {}
};
REGISTER_PREFERENCE_TYPE(PREF_UINT, UIntPreference)
REGISTER_PREFERENCE_TYPE(PREF_DECODE_AS_UINT, UIntPreference)

class EnumPreference : public WiresharkPreference
{
public:
    EnumPreference(QObject * parent = Q_NULLPTR) : WiresharkPreference(parent) {}
    virtual QWidget * editor(QWidget *parent, const QStyleOptionViewItem &/*option*/, const QModelIndex &/*index*/)
    {
        return new QComboBox(parent);
    }

    virtual void setData(QWidget *editor, const QModelIndex &index)
    {
        QComboBox* combo = static_cast<QComboBox*>(editor);
        const enum_val_t *ev;
        PrefsItem* pref = VariantPointer<PrefsItem>::asPtr(index.model()->data(index, Qt::UserRole));
        for (ev = prefs_get_enumvals(pref->getPref()); ev && ev->description; ev++) {
            combo->addItem(ev->description, QVariant(ev->value));
            if (prefs_get_enum_value(pref->getPref(), pref_stashed) == ev->value)
                combo->setCurrentIndex(combo->count() - 1);
        }
    }

    virtual void setModelData(QWidget *editor, QAbstractItemModel *model, const QModelIndex &index)
    {
        QComboBox* combo = static_cast<QComboBox*>(editor);
        model->setData(index, combo->itemData(combo->currentIndex()), Qt::EditRole);
    }
};
REGISTER_PREFERENCE_TYPE(PREF_ENUM, EnumPreference)

class RangePreference : public WiresharkPreference
{
public:
    RangePreference(QObject * parent = Q_NULLPTR) : WiresharkPreference(parent) {}
    virtual QWidget * editor(QWidget *parent, const QStyleOptionViewItem &/*option*/, const QModelIndex &/*index*/)
    {
        return new RangeSyntaxLineEdit(parent);
    }

    virtual void setData(QWidget *editor, const QModelIndex &index)
    {
        RangeSyntaxLineEdit* syntax = static_cast<RangeSyntaxLineEdit*>(editor);
        syntax->setText(index.model()->data(index, Qt::DisplayRole).toString());
    }

    virtual void setModelData(QWidget *editor, QAbstractItemModel *model, const QModelIndex &index)
    {
        RangeSyntaxLineEdit* syntax = static_cast<RangeSyntaxLineEdit*>(editor);
        model->setData(index, syntax->text(), Qt::EditRole);
    }
};
REGISTER_PREFERENCE_TYPE(PREF_RANGE, RangePreference)
REGISTER_PREFERENCE_TYPE(PREF_DECODE_AS_RANGE, RangePreference)

class ColorPreference : public WiresharkPreference
{
public:
    ColorPreference(QObject * parent = Q_NULLPTR) : WiresharkPreference(parent) {}
    virtual QWidget * editor(QWidget * parent, const QStyleOptionViewItem &option, const QModelIndex &index)
    {
        QColorDialog color_dlg;
        color_t color = *prefs_get_color_value(prefsItem()->getPref(), pref_stashed);

        color_dlg.setCurrentColor(QColor(
                                      color.red >> 8,
                                      color.green >> 8,
                                      color.blue >> 8
                                      ));
        if (color_dlg.exec() == QDialog::Accepted)
            ((QAbstractItemModel*)index.model())->setData(index, color_dlg.currentColor().name(), Qt::EditRole);

        return WiresharkPreference::editor(parent, option, index);
    }
};
REGISTER_PREFERENCE_TYPE(PREF_COLOR, ColorPreference)

class SaveFilePreference : public WiresharkPreference
{
public:
    SaveFilePreference(QObject * parent = Q_NULLPTR) : WiresharkPreference(parent) {}
    virtual QWidget * editor(QWidget * parent, const QStyleOptionViewItem &option, const QModelIndex &index)
    {
        QString filename = WiresharkFileDialog::getSaveFileName(parent, wsApp->windowTitleString(prefs_get_title(prefsItem()->getPref())),
                                                    index.model()->data(index, Qt::DisplayRole).toString());
        if (!filename.isEmpty()) {
            ((QAbstractItemModel*)index.model())->setData(index, QDir::toNativeSeparators(filename), Qt::EditRole);
        }
        return WiresharkPreference::editor(parent, option, index);
    }
};
REGISTER_PREFERENCE_TYPE(PREF_SAVE_FILENAME, SaveFilePreference)

class OpenFilePreference : public WiresharkPreference
{
public:
    OpenFilePreference(QObject * parent = Q_NULLPTR) : WiresharkPreference(parent) {}
    virtual QWidget * editor(QWidget * parent, const QStyleOptionViewItem &option, const QModelIndex &index)
    {
        QString filename = WiresharkFileDialog::getOpenFileName(parent, wsApp->windowTitleString(prefs_get_title(prefsItem()->getPref())),
                                                        index.model()->data(index, Qt::DisplayRole).toString());
        if (!filename.isEmpty()) {
            ((QAbstractItemModel*)index.model())->setData(index, QDir::toNativeSeparators(filename), Qt::EditRole);
        }
        return WiresharkPreference::editor(parent, option, index);
    }
};
REGISTER_PREFERENCE_TYPE(PREF_OPEN_FILENAME, OpenFilePreference)

class DirNamePreference : public WiresharkPreference
{
public:
    DirNamePreference(QObject * parent = Q_NULLPTR) : WiresharkPreference(parent) {}
    virtual QWidget * editor(QWidget * parent, const QStyleOptionViewItem &option, const QModelIndex &index)
    {
        QString filename = WiresharkFileDialog::getExistingDirectory(parent, wsApp->windowTitleString(prefs_get_title(prefsItem()->getPref())),
                                                    index.model()->data(index, Qt::DisplayRole).toString());
        if (!filename.isEmpty()) {
            ((QAbstractItemModel*)index.model())->setData(index, QDir::toNativeSeparators(filename), Qt::EditRole);
        }
        return WiresharkPreference::editor(parent, option, index);
    }
};
REGISTER_PREFERENCE_TYPE(PREF_DIRNAME, DirNamePreference)

class UatPreference : public WiresharkPreference
{
public:
    UatPreference(QObject * parent = Q_NULLPTR) : WiresharkPreference(parent) {}
    virtual QWidget * editor(QWidget * parent, const QStyleOptionViewItem &option, const QModelIndex &index)
    {
        if (prefsItem()->getPrefGUIType() == GUI_ALL || prefsItem()->getPrefGUIType() == GUI_QT) {
            UatDialog uat_dlg(parent, prefs_get_uat_value(prefsItem()->getPref()));
            uat_dlg.exec();
        }
        return WiresharkPreference::editor(parent, option, index);
    }
};
REGISTER_PREFERENCE_TYPE(PREF_UAT, UatPreference)


/*
 * Editor modelines
 *
 * Local Variables:
 * c-basic-offset: 4
 * tab-width: 8
 * indent-tabs-mode: nil
 * End:
 *
 * ex: set shiftwidth=4 tabstop=8 expandtab:
 * :indentSize=4:tabSize=8:noTabs=true:
 */
