/* packet-netlink-nl80211.c
 * Dissector for nl80211 (over Netlink).
 *
 * Copyright (c) 2017, Peter Wu <peter@lekensteyn.nl>
 * Copyright (c) 2018, Mikael Kanstrup <mikael.kanstrup@sony.com>
 *
 * Wireshark - Network traffic analyzer
 * By Gerald Combs <gerald@wireshark.org>
 * Copyright 1998 Gerald Combs
 *
 * SPDX-License-Identifier: GPL-2.0-or-later
 */

#define NEW_PROTO_TREE_API

#include "config.h"

#include <epan/packet.h>
#include "packet-ieee80211.h"
#include "packet-netlink.h"

void proto_register_netlink_nl80211(void);
void proto_reg_handoff_netlink_nl80211(void);

static dissector_handle_t ieee80211_handle;
static dissector_table_t ieee80211_tag_dissector_table;

#define NETLINK_NL80211_HFI_INIT HFI_INIT(proto_netlink_generic)

/* Extracted using tools/generate-nl80211-fields.py */
/* Definitions from linux/nl80211.h {{{ */
enum ws_nl80211_commands {
    WS_NL80211_CMD_UNSPEC,
    WS_NL80211_CMD_GET_WIPHY,
    WS_NL80211_CMD_SET_WIPHY,
    WS_NL80211_CMD_NEW_WIPHY,
    WS_NL80211_CMD_DEL_WIPHY,
    WS_NL80211_CMD_GET_INTERFACE,
    WS_NL80211_CMD_SET_INTERFACE,
    WS_NL80211_CMD_NEW_INTERFACE,
    WS_NL80211_CMD_DEL_INTERFACE,
    WS_NL80211_CMD_GET_KEY,
    WS_NL80211_CMD_SET_KEY,
    WS_NL80211_CMD_NEW_KEY,
    WS_NL80211_CMD_DEL_KEY,
    WS_NL80211_CMD_GET_BEACON,
    WS_NL80211_CMD_SET_BEACON,
    WS_NL80211_CMD_START_AP,
    WS_NL80211_CMD_STOP_AP,
    WS_NL80211_CMD_GET_STATION,
    WS_NL80211_CMD_SET_STATION,
    WS_NL80211_CMD_NEW_STATION,
    WS_NL80211_CMD_DEL_STATION,
    WS_NL80211_CMD_GET_MPATH,
    WS_NL80211_CMD_SET_MPATH,
    WS_NL80211_CMD_NEW_MPATH,
    WS_NL80211_CMD_DEL_MPATH,
    WS_NL80211_CMD_SET_BSS,
    WS_NL80211_CMD_SET_REG,
    WS_NL80211_CMD_REQ_SET_REG,
    WS_NL80211_CMD_GET_MESH_CONFIG,
    WS_NL80211_CMD_SET_MESH_CONFIG,
    WS_NL80211_CMD_SET_MGMT_EXTRA_IE,
    WS_NL80211_CMD_GET_REG,
    WS_NL80211_CMD_GET_SCAN,
    WS_NL80211_CMD_TRIGGER_SCAN,
    WS_NL80211_CMD_NEW_SCAN_RESULTS,
    WS_NL80211_CMD_SCAN_ABORTED,
    WS_NL80211_CMD_REG_CHANGE,
    WS_NL80211_CMD_AUTHENTICATE,
    WS_NL80211_CMD_ASSOCIATE,
    WS_NL80211_CMD_DEAUTHENTICATE,
    WS_NL80211_CMD_DISASSOCIATE,
    WS_NL80211_CMD_MICHAEL_MIC_FAILURE,
    WS_NL80211_CMD_REG_BEACON_HINT,
    WS_NL80211_CMD_JOIN_IBSS,
    WS_NL80211_CMD_LEAVE_IBSS,
    WS_NL80211_CMD_TESTMODE,
    WS_NL80211_CMD_CONNECT,
    WS_NL80211_CMD_ROAM,
    WS_NL80211_CMD_DISCONNECT,
    WS_NL80211_CMD_SET_WIPHY_NETNS,
    WS_NL80211_CMD_GET_SURVEY,
    WS_NL80211_CMD_NEW_SURVEY_RESULTS,
    WS_NL80211_CMD_SET_PMKSA,
    WS_NL80211_CMD_DEL_PMKSA,
    WS_NL80211_CMD_FLUSH_PMKSA,
    WS_NL80211_CMD_REMAIN_ON_CHANNEL,
    WS_NL80211_CMD_CANCEL_REMAIN_ON_CHANNEL,
    WS_NL80211_CMD_SET_TX_BITRATE_MASK,
    WS_NL80211_CMD_REGISTER_FRAME,
    WS_NL80211_CMD_FRAME,
    WS_NL80211_CMD_FRAME_TX_STATUS,
    WS_NL80211_CMD_SET_POWER_SAVE,
    WS_NL80211_CMD_GET_POWER_SAVE,
    WS_NL80211_CMD_SET_CQM,
    WS_NL80211_CMD_NOTIFY_CQM,
    WS_NL80211_CMD_SET_CHANNEL,
    WS_NL80211_CMD_SET_WDS_PEER,
    WS_NL80211_CMD_FRAME_WAIT_CANCEL,
    WS_NL80211_CMD_JOIN_MESH,
    WS_NL80211_CMD_LEAVE_MESH,
    WS_NL80211_CMD_UNPROT_DEAUTHENTICATE,
    WS_NL80211_CMD_UNPROT_DISASSOCIATE,
    WS_NL80211_CMD_NEW_PEER_CANDIDATE,
    WS_NL80211_CMD_GET_WOWLAN,
    WS_NL80211_CMD_SET_WOWLAN,
    WS_NL80211_CMD_START_SCHED_SCAN,
    WS_NL80211_CMD_STOP_SCHED_SCAN,
    WS_NL80211_CMD_SCHED_SCAN_RESULTS,
    WS_NL80211_CMD_SCHED_SCAN_STOPPED,
    WS_NL80211_CMD_SET_REKEY_OFFLOAD,
    WS_NL80211_CMD_PMKSA_CANDIDATE,
    WS_NL80211_CMD_TDLS_OPER,
    WS_NL80211_CMD_TDLS_MGMT,
    WS_NL80211_CMD_UNEXPECTED_FRAME,
    WS_NL80211_CMD_PROBE_CLIENT,
    WS_NL80211_CMD_REGISTER_BEACONS,
    WS_NL80211_CMD_UNEXPECTED_4ADDR_FRAME,
    WS_NL80211_CMD_SET_NOACK_MAP,
    WS_NL80211_CMD_CH_SWITCH_NOTIFY,
    WS_NL80211_CMD_START_P2P_DEVICE,
    WS_NL80211_CMD_STOP_P2P_DEVICE,
    WS_NL80211_CMD_CONN_FAILED,
    WS_NL80211_CMD_SET_MCAST_RATE,
    WS_NL80211_CMD_SET_MAC_ACL,
    WS_NL80211_CMD_RADAR_DETECT,
    WS_NL80211_CMD_GET_PROTOCOL_FEATURES,
    WS_NL80211_CMD_UPDATE_FT_IES,
    WS_NL80211_CMD_FT_EVENT,
    WS_NL80211_CMD_CRIT_PROTOCOL_START,
    WS_NL80211_CMD_CRIT_PROTOCOL_STOP,
    WS_NL80211_CMD_GET_COALESCE,
    WS_NL80211_CMD_SET_COALESCE,
    WS_NL80211_CMD_CHANNEL_SWITCH,
    WS_NL80211_CMD_VENDOR,
    WS_NL80211_CMD_SET_QOS_MAP,
    WS_NL80211_CMD_ADD_TX_TS,
    WS_NL80211_CMD_DEL_TX_TS,
    WS_NL80211_CMD_GET_MPP,
    WS_NL80211_CMD_JOIN_OCB,
    WS_NL80211_CMD_LEAVE_OCB,
    WS_NL80211_CMD_CH_SWITCH_STARTED_NOTIFY,
    WS_NL80211_CMD_TDLS_CHANNEL_SWITCH,
    WS_NL80211_CMD_TDLS_CANCEL_CHANNEL_SWITCH,
    WS_NL80211_CMD_WIPHY_REG_CHANGE,
    WS_NL80211_CMD_ABORT_SCAN,
    WS_NL80211_CMD_START_NAN,
    WS_NL80211_CMD_STOP_NAN,
    WS_NL80211_CMD_ADD_NAN_FUNCTION,
    WS_NL80211_CMD_DEL_NAN_FUNCTION,
    WS_NL80211_CMD_CHANGE_NAN_CONFIG,
    WS_NL80211_CMD_NAN_MATCH,
    WS_NL80211_CMD_SET_MULTICAST_TO_UNICAST,
    WS_NL80211_CMD_UPDATE_CONNECT_PARAMS,
    WS_NL80211_CMD_SET_PMK,
    WS_NL80211_CMD_DEL_PMK,
    WS_NL80211_CMD_PORT_AUTHORIZED,
    WS_NL80211_CMD_RELOAD_REGDB,
    WS_NL80211_CMD_EXTERNAL_AUTH,
    WS_NL80211_CMD_STA_OPMODE_CHANGED,
    WS_NL80211_CMD_CONTROL_PORT_FRAME,
    WS_NL80211_CMD_GET_FTM_RESPONDER_STATS,
};

enum ws_nl80211_attrs {
    WS_NL80211_ATTR_UNSPEC,
    WS_NL80211_ATTR_WIPHY,
    WS_NL80211_ATTR_WIPHY_NAME,
    WS_NL80211_ATTR_IFINDEX,
    WS_NL80211_ATTR_IFNAME,
    WS_NL80211_ATTR_IFTYPE,
    WS_NL80211_ATTR_MAC,
    WS_NL80211_ATTR_KEY_DATA,
    WS_NL80211_ATTR_KEY_IDX,
    WS_NL80211_ATTR_KEY_CIPHER,
    WS_NL80211_ATTR_KEY_SEQ,
    WS_NL80211_ATTR_KEY_DEFAULT,
    WS_NL80211_ATTR_BEACON_INTERVAL,
    WS_NL80211_ATTR_DTIM_PERIOD,
    WS_NL80211_ATTR_BEACON_HEAD,
    WS_NL80211_ATTR_BEACON_TAIL,
    WS_NL80211_ATTR_STA_AID,
    WS_NL80211_ATTR_STA_FLAGS,
    WS_NL80211_ATTR_STA_LISTEN_INTERVAL,
    WS_NL80211_ATTR_STA_SUPPORTED_RATES,
    WS_NL80211_ATTR_STA_VLAN,
    WS_NL80211_ATTR_STA_INFO,
    WS_NL80211_ATTR_WIPHY_BANDS,
    WS_NL80211_ATTR_MNTR_FLAGS,
    WS_NL80211_ATTR_MESH_ID,
    WS_NL80211_ATTR_STA_PLINK_ACTION,
    WS_NL80211_ATTR_MPATH_NEXT_HOP,
    WS_NL80211_ATTR_MPATH_INFO,
    WS_NL80211_ATTR_BSS_CTS_PROT,
    WS_NL80211_ATTR_BSS_SHORT_PREAMBLE,
    WS_NL80211_ATTR_BSS_SHORT_SLOT_TIME,
    WS_NL80211_ATTR_HT_CAPABILITY,
    WS_NL80211_ATTR_SUPPORTED_IFTYPES,
    WS_NL80211_ATTR_REG_ALPHA2,
    WS_NL80211_ATTR_REG_RULES,
    WS_NL80211_ATTR_MESH_CONFIG,
    WS_NL80211_ATTR_BSS_BASIC_RATES,
    WS_NL80211_ATTR_WIPHY_TXQ_PARAMS,
    WS_NL80211_ATTR_WIPHY_FREQ,
    WS_NL80211_ATTR_WIPHY_CHANNEL_TYPE,
    WS_NL80211_ATTR_KEY_DEFAULT_MGMT,
    WS_NL80211_ATTR_MGMT_SUBTYPE,
    WS_NL80211_ATTR_IE,
    WS_NL80211_ATTR_MAX_NUM_SCAN_SSIDS,
    WS_NL80211_ATTR_SCAN_FREQUENCIES,
    WS_NL80211_ATTR_SCAN_SSIDS,
    WS_NL80211_ATTR_GENERATION,
    WS_NL80211_ATTR_BSS,
    WS_NL80211_ATTR_REG_INITIATOR,
    WS_NL80211_ATTR_REG_TYPE,
    WS_NL80211_ATTR_SUPPORTED_COMMANDS,
    WS_NL80211_ATTR_FRAME,
    WS_NL80211_ATTR_SSID,
    WS_NL80211_ATTR_AUTH_TYPE,
    WS_NL80211_ATTR_REASON_CODE,
    WS_NL80211_ATTR_KEY_TYPE,
    WS_NL80211_ATTR_MAX_SCAN_IE_LEN,
    WS_NL80211_ATTR_CIPHER_SUITES,
    WS_NL80211_ATTR_FREQ_BEFORE,
    WS_NL80211_ATTR_FREQ_AFTER,
    WS_NL80211_ATTR_FREQ_FIXED,
    WS_NL80211_ATTR_WIPHY_RETRY_SHORT,
    WS_NL80211_ATTR_WIPHY_RETRY_LONG,
    WS_NL80211_ATTR_WIPHY_FRAG_THRESHOLD,
    WS_NL80211_ATTR_WIPHY_RTS_THRESHOLD,
    WS_NL80211_ATTR_TIMED_OUT,
    WS_NL80211_ATTR_USE_MFP,
    WS_NL80211_ATTR_STA_FLAGS2,
    WS_NL80211_ATTR_CONTROL_PORT,
    WS_NL80211_ATTR_TESTDATA,
    WS_NL80211_ATTR_PRIVACY,
    WS_NL80211_ATTR_DISCONNECTED_BY_AP,
    WS_NL80211_ATTR_STATUS_CODE,
    WS_NL80211_ATTR_CIPHER_SUITES_PAIRWISE,
    WS_NL80211_ATTR_CIPHER_SUITE_GROUP,
    WS_NL80211_ATTR_WPA_VERSIONS,
    WS_NL80211_ATTR_AKM_SUITES,
    WS_NL80211_ATTR_REQ_IE,
    WS_NL80211_ATTR_RESP_IE,
    WS_NL80211_ATTR_PREV_BSSID,
    WS_NL80211_ATTR_KEY,
    WS_NL80211_ATTR_KEYS,
    WS_NL80211_ATTR_PID,
    WS_NL80211_ATTR_4ADDR,
    WS_NL80211_ATTR_SURVEY_INFO,
    WS_NL80211_ATTR_PMKID,
    WS_NL80211_ATTR_MAX_NUM_PMKIDS,
    WS_NL80211_ATTR_DURATION,
    WS_NL80211_ATTR_COOKIE,
    WS_NL80211_ATTR_WIPHY_COVERAGE_CLASS,
    WS_NL80211_ATTR_TX_RATES,
    WS_NL80211_ATTR_FRAME_MATCH,
    WS_NL80211_ATTR_ACK,
    WS_NL80211_ATTR_PS_STATE,
    WS_NL80211_ATTR_CQM,
    WS_NL80211_ATTR_LOCAL_STATE_CHANGE,
    WS_NL80211_ATTR_AP_ISOLATE,
    WS_NL80211_ATTR_WIPHY_TX_POWER_SETTING,
    WS_NL80211_ATTR_WIPHY_TX_POWER_LEVEL,
    WS_NL80211_ATTR_TX_FRAME_TYPES,
    WS_NL80211_ATTR_RX_FRAME_TYPES,
    WS_NL80211_ATTR_FRAME_TYPE,
    WS_NL80211_ATTR_CONTROL_PORT_ETHERTYPE,
    WS_NL80211_ATTR_CONTROL_PORT_NO_ENCRYPT,
    WS_NL80211_ATTR_SUPPORT_IBSS_RSN,
    WS_NL80211_ATTR_WIPHY_ANTENNA_TX,
    WS_NL80211_ATTR_WIPHY_ANTENNA_RX,
    WS_NL80211_ATTR_MCAST_RATE,
    WS_NL80211_ATTR_OFFCHANNEL_TX_OK,
    WS_NL80211_ATTR_BSS_HT_OPMODE,
    WS_NL80211_ATTR_KEY_DEFAULT_TYPES,
    WS_NL80211_ATTR_MAX_REMAIN_ON_CHANNEL_DURATION,
    WS_NL80211_ATTR_MESH_SETUP,
    WS_NL80211_ATTR_WIPHY_ANTENNA_AVAIL_TX,
    WS_NL80211_ATTR_WIPHY_ANTENNA_AVAIL_RX,
    WS_NL80211_ATTR_SUPPORT_MESH_AUTH,
    WS_NL80211_ATTR_STA_PLINK_STATE,
    WS_NL80211_ATTR_WOWLAN_TRIGGERS,
    WS_NL80211_ATTR_WOWLAN_TRIGGERS_SUPPORTED,
    WS_NL80211_ATTR_SCHED_SCAN_INTERVAL,
    WS_NL80211_ATTR_INTERFACE_COMBINATIONS,
    WS_NL80211_ATTR_SOFTWARE_IFTYPES,
    WS_NL80211_ATTR_REKEY_DATA,
    WS_NL80211_ATTR_MAX_NUM_SCHED_SCAN_SSIDS,
    WS_NL80211_ATTR_MAX_SCHED_SCAN_IE_LEN,
    WS_NL80211_ATTR_SCAN_SUPP_RATES,
    WS_NL80211_ATTR_HIDDEN_SSID,
    WS_NL80211_ATTR_IE_PROBE_RESP,
    WS_NL80211_ATTR_IE_ASSOC_RESP,
    WS_NL80211_ATTR_STA_WME,
    WS_NL80211_ATTR_SUPPORT_AP_UAPSD,
    WS_NL80211_ATTR_ROAM_SUPPORT,
    WS_NL80211_ATTR_SCHED_SCAN_MATCH,
    WS_NL80211_ATTR_MAX_MATCH_SETS,
    WS_NL80211_ATTR_PMKSA_CANDIDATE,
    WS_NL80211_ATTR_TX_NO_CCK_RATE,
    WS_NL80211_ATTR_TDLS_ACTION,
    WS_NL80211_ATTR_TDLS_DIALOG_TOKEN,
    WS_NL80211_ATTR_TDLS_OPERATION,
    WS_NL80211_ATTR_TDLS_SUPPORT,
    WS_NL80211_ATTR_TDLS_EXTERNAL_SETUP,
    WS_NL80211_ATTR_DEVICE_AP_SME,
    WS_NL80211_ATTR_DONT_WAIT_FOR_ACK,
    WS_NL80211_ATTR_FEATURE_FLAGS,
    WS_NL80211_ATTR_PROBE_RESP_OFFLOAD,
    WS_NL80211_ATTR_PROBE_RESP,
    WS_NL80211_ATTR_DFS_REGION,
    WS_NL80211_ATTR_DISABLE_HT,
    WS_NL80211_ATTR_HT_CAPABILITY_MASK,
    WS_NL80211_ATTR_NOACK_MAP,
    WS_NL80211_ATTR_INACTIVITY_TIMEOUT,
    WS_NL80211_ATTR_RX_SIGNAL_DBM,
    WS_NL80211_ATTR_BG_SCAN_PERIOD,
    WS_NL80211_ATTR_WDEV,
    WS_NL80211_ATTR_USER_REG_HINT_TYPE,
    WS_NL80211_ATTR_CONN_FAILED_REASON,
    WS_NL80211_ATTR_AUTH_DATA,
    WS_NL80211_ATTR_VHT_CAPABILITY,
    WS_NL80211_ATTR_SCAN_FLAGS,
    WS_NL80211_ATTR_CHANNEL_WIDTH,
    WS_NL80211_ATTR_CENTER_FREQ1,
    WS_NL80211_ATTR_CENTER_FREQ2,
    WS_NL80211_ATTR_P2P_CTWINDOW,
    WS_NL80211_ATTR_P2P_OPPPS,
    WS_NL80211_ATTR_LOCAL_MESH_POWER_MODE,
    WS_NL80211_ATTR_ACL_POLICY,
    WS_NL80211_ATTR_MAC_ADDRS,
    WS_NL80211_ATTR_MAC_ACL_MAX,
    WS_NL80211_ATTR_RADAR_EVENT,
    WS_NL80211_ATTR_EXT_CAPA,
    WS_NL80211_ATTR_EXT_CAPA_MASK,
    WS_NL80211_ATTR_STA_CAPABILITY,
    WS_NL80211_ATTR_STA_EXT_CAPABILITY,
    WS_NL80211_ATTR_PROTOCOL_FEATURES,
    WS_NL80211_ATTR_SPLIT_WIPHY_DUMP,
    WS_NL80211_ATTR_DISABLE_VHT,
    WS_NL80211_ATTR_VHT_CAPABILITY_MASK,
    WS_NL80211_ATTR_MDID,
    WS_NL80211_ATTR_IE_RIC,
    WS_NL80211_ATTR_CRIT_PROT_ID,
    WS_NL80211_ATTR_MAX_CRIT_PROT_DURATION,
    WS_NL80211_ATTR_PEER_AID,
    WS_NL80211_ATTR_COALESCE_RULE,
    WS_NL80211_ATTR_CH_SWITCH_COUNT,
    WS_NL80211_ATTR_CH_SWITCH_BLOCK_TX,
    WS_NL80211_ATTR_CSA_IES,
    WS_NL80211_ATTR_CSA_C_OFF_BEACON,
    WS_NL80211_ATTR_CSA_C_OFF_PRESP,
    WS_NL80211_ATTR_RXMGMT_FLAGS,
    WS_NL80211_ATTR_STA_SUPPORTED_CHANNELS,
    WS_NL80211_ATTR_STA_SUPPORTED_OPER_CLASSES,
    WS_NL80211_ATTR_HANDLE_DFS,
    WS_NL80211_ATTR_SUPPORT_5_MHZ,
    WS_NL80211_ATTR_SUPPORT_10_MHZ,
    WS_NL80211_ATTR_OPMODE_NOTIF,
    WS_NL80211_ATTR_VENDOR_ID,
    WS_NL80211_ATTR_VENDOR_SUBCMD,
    WS_NL80211_ATTR_VENDOR_DATA,
    WS_NL80211_ATTR_VENDOR_EVENTS,
    WS_NL80211_ATTR_QOS_MAP,
    WS_NL80211_ATTR_MAC_HINT,
    WS_NL80211_ATTR_WIPHY_FREQ_HINT,
    WS_NL80211_ATTR_MAX_AP_ASSOC_STA,
    WS_NL80211_ATTR_TDLS_PEER_CAPABILITY,
    WS_NL80211_ATTR_SOCKET_OWNER,
    WS_NL80211_ATTR_CSA_C_OFFSETS_TX,
    WS_NL80211_ATTR_MAX_CSA_COUNTERS,
    WS_NL80211_ATTR_TDLS_INITIATOR,
    WS_NL80211_ATTR_USE_RRM,
    WS_NL80211_ATTR_WIPHY_DYN_ACK,
    WS_NL80211_ATTR_TSID,
    WS_NL80211_ATTR_USER_PRIO,
    WS_NL80211_ATTR_ADMITTED_TIME,
    WS_NL80211_ATTR_SMPS_MODE,
    WS_NL80211_ATTR_OPER_CLASS,
    WS_NL80211_ATTR_MAC_MASK,
    WS_NL80211_ATTR_WIPHY_SELF_MANAGED_REG,
    WS_NL80211_ATTR_EXT_FEATURES,
    WS_NL80211_ATTR_SURVEY_RADIO_STATS,
    WS_NL80211_ATTR_NETNS_FD,
    WS_NL80211_ATTR_SCHED_SCAN_DELAY,
    WS_NL80211_ATTR_REG_INDOOR,
    WS_NL80211_ATTR_MAX_NUM_SCHED_SCAN_PLANS,
    WS_NL80211_ATTR_MAX_SCAN_PLAN_INTERVAL,
    WS_NL80211_ATTR_MAX_SCAN_PLAN_ITERATIONS,
    WS_NL80211_ATTR_SCHED_SCAN_PLANS,
    WS_NL80211_ATTR_PBSS,
    WS_NL80211_ATTR_BSS_SELECT,
    WS_NL80211_ATTR_STA_SUPPORT_P2P_PS,
    WS_NL80211_ATTR_PAD,
    WS_NL80211_ATTR_IFTYPE_EXT_CAPA,
    WS_NL80211_ATTR_MU_MIMO_GROUP_DATA,
    WS_NL80211_ATTR_MU_MIMO_FOLLOW_MAC_ADDR,
    WS_NL80211_ATTR_SCAN_START_TIME_TSF,
    WS_NL80211_ATTR_SCAN_START_TIME_TSF_BSSID,
    WS_NL80211_ATTR_MEASUREMENT_DURATION,
    WS_NL80211_ATTR_MEASUREMENT_DURATION_MANDATORY,
    WS_NL80211_ATTR_MESH_PEER_AID,
    WS_NL80211_ATTR_NAN_MASTER_PREF,
    WS_NL80211_ATTR_BANDS,
    WS_NL80211_ATTR_NAN_FUNC,
    WS_NL80211_ATTR_NAN_MATCH,
    WS_NL80211_ATTR_FILS_KEK,
    WS_NL80211_ATTR_FILS_NONCES,
    WS_NL80211_ATTR_MULTICAST_TO_UNICAST_ENABLED,
    WS_NL80211_ATTR_BSSID,
    WS_NL80211_ATTR_SCHED_SCAN_RELATIVE_RSSI,
    WS_NL80211_ATTR_SCHED_SCAN_RSSI_ADJUST,
    WS_NL80211_ATTR_TIMEOUT_REASON,
    WS_NL80211_ATTR_FILS_ERP_USERNAME,
    WS_NL80211_ATTR_FILS_ERP_REALM,
    WS_NL80211_ATTR_FILS_ERP_NEXT_SEQ_NUM,
    WS_NL80211_ATTR_FILS_ERP_RRK,
    WS_NL80211_ATTR_FILS_CACHE_ID,
    WS_NL80211_ATTR_PMK,
    WS_NL80211_ATTR_SCHED_SCAN_MULTI,
    WS_NL80211_ATTR_SCHED_SCAN_MAX_REQS,
    WS_NL80211_ATTR_WANT_1X_4WAY_HS,
    WS_NL80211_ATTR_PMKR0_NAME,
    WS_NL80211_ATTR_PORT_AUTHORIZED,
    WS_NL80211_ATTR_EXTERNAL_AUTH_ACTION,
    WS_NL80211_ATTR_EXTERNAL_AUTH_SUPPORT,
    WS_NL80211_ATTR_NSS,
    WS_NL80211_ATTR_ACK_SIGNAL,
    WS_NL80211_ATTR_CONTROL_PORT_OVER_NL80211,
    WS_NL80211_ATTR_TXQ_STATS,
    WS_NL80211_ATTR_TXQ_LIMIT,
    WS_NL80211_ATTR_TXQ_MEMORY_LIMIT,
    WS_NL80211_ATTR_TXQ_QUANTUM,
    WS_NL80211_ATTR_HE_CAPABILITY,
    WS_NL80211_ATTR_FTM_RESPONDER,
    WS_NL80211_ATTR_FTM_RESPONDER_STATS,
};

enum ws_nl80211_iftype {
    WS_NL80211_IFTYPE_UNSPECIFIED,
    WS_NL80211_IFTYPE_ADHOC,
    WS_NL80211_IFTYPE_STATION,
    WS_NL80211_IFTYPE_AP,
    WS_NL80211_IFTYPE_AP_VLAN,
    WS_NL80211_IFTYPE_WDS,
    WS_NL80211_IFTYPE_MONITOR,
    WS_NL80211_IFTYPE_MESH_POINT,
    WS_NL80211_IFTYPE_P2P_CLIENT,
    WS_NL80211_IFTYPE_P2P_GO,
    WS_NL80211_IFTYPE_P2P_DEVICE,
    WS_NL80211_IFTYPE_OCB,
    WS_NL80211_IFTYPE_NAN,
};

enum ws_nl80211_sta_flags {
    WS___NL80211_STA_FLAG_INVALID,
    WS_NL80211_STA_FLAG_AUTHORIZED,
    WS_NL80211_STA_FLAG_SHORT_PREAMBLE,
    WS_NL80211_STA_FLAG_WME,
    WS_NL80211_STA_FLAG_MFP,
    WS_NL80211_STA_FLAG_AUTHENTICATED,
    WS_NL80211_STA_FLAG_TDLS_PEER,
    WS_NL80211_STA_FLAG_ASSOCIATED,
};

enum ws_nl80211_sta_p2p_ps_status {
    WS_NL80211_P2P_PS_UNSUPPORTED = 0,
    WS_NL80211_P2P_PS_SUPPORTED,
};

enum ws_nl80211_rate_info {
    WS___NL80211_RATE_INFO_INVALID,
    WS_NL80211_RATE_INFO_BITRATE,
    WS_NL80211_RATE_INFO_MCS,
    WS_NL80211_RATE_INFO_40_MHZ_WIDTH,
    WS_NL80211_RATE_INFO_SHORT_GI,
    WS_NL80211_RATE_INFO_BITRATE32,
    WS_NL80211_RATE_INFO_VHT_MCS,
    WS_NL80211_RATE_INFO_VHT_NSS,
    WS_NL80211_RATE_INFO_80_MHZ_WIDTH,
    WS_NL80211_RATE_INFO_80P80_MHZ_WIDTH,
    WS_NL80211_RATE_INFO_160_MHZ_WIDTH,
    WS_NL80211_RATE_INFO_10_MHZ_WIDTH,
    WS_NL80211_RATE_INFO_5_MHZ_WIDTH,
    WS_NL80211_RATE_INFO_HE_MCS,
    WS_NL80211_RATE_INFO_HE_NSS,
    WS_NL80211_RATE_INFO_HE_GI,
    WS_NL80211_RATE_INFO_HE_DCM,
    WS_NL80211_RATE_INFO_HE_RU_ALLOC,
};

enum ws_nl80211_sta_bss_param {
    WS___NL80211_STA_BSS_PARAM_INVALID,
    WS_NL80211_STA_BSS_PARAM_CTS_PROT,
    WS_NL80211_STA_BSS_PARAM_SHORT_PREAMBLE,
    WS_NL80211_STA_BSS_PARAM_SHORT_SLOT_TIME,
    WS_NL80211_STA_BSS_PARAM_DTIM_PERIOD,
    WS_NL80211_STA_BSS_PARAM_BEACON_INTERVAL,
};

enum ws_nl80211_sta_info {
    WS___NL80211_STA_INFO_INVALID,
    WS_NL80211_STA_INFO_INACTIVE_TIME,
    WS_NL80211_STA_INFO_RX_BYTES,
    WS_NL80211_STA_INFO_TX_BYTES,
    WS_NL80211_STA_INFO_LLID,
    WS_NL80211_STA_INFO_PLID,
    WS_NL80211_STA_INFO_PLINK_STATE,
    WS_NL80211_STA_INFO_SIGNAL,
    WS_NL80211_STA_INFO_TX_BITRATE,
    WS_NL80211_STA_INFO_RX_PACKETS,
    WS_NL80211_STA_INFO_TX_PACKETS,
    WS_NL80211_STA_INFO_TX_RETRIES,
    WS_NL80211_STA_INFO_TX_FAILED,
    WS_NL80211_STA_INFO_SIGNAL_AVG,
    WS_NL80211_STA_INFO_RX_BITRATE,
    WS_NL80211_STA_INFO_BSS_PARAM,
    WS_NL80211_STA_INFO_CONNECTED_TIME,
    WS_NL80211_STA_INFO_STA_FLAGS,
    WS_NL80211_STA_INFO_BEACON_LOSS,
    WS_NL80211_STA_INFO_T_OFFSET,
    WS_NL80211_STA_INFO_LOCAL_PM,
    WS_NL80211_STA_INFO_PEER_PM,
    WS_NL80211_STA_INFO_NONPEER_PM,
    WS_NL80211_STA_INFO_RX_BYTES64,
    WS_NL80211_STA_INFO_TX_BYTES64,
    WS_NL80211_STA_INFO_CHAIN_SIGNAL,
    WS_NL80211_STA_INFO_CHAIN_SIGNAL_AVG,
    WS_NL80211_STA_INFO_EXPECTED_THROUGHPUT,
    WS_NL80211_STA_INFO_RX_DROP_MISC,
    WS_NL80211_STA_INFO_BEACON_RX,
    WS_NL80211_STA_INFO_BEACON_SIGNAL_AVG,
    WS_NL80211_STA_INFO_TID_STATS,
    WS_NL80211_STA_INFO_RX_DURATION,
    WS_NL80211_STA_INFO_PAD,
    WS_NL80211_STA_INFO_ACK_SIGNAL,
    WS_NL80211_STA_INFO_ACK_SIGNAL_AVG,
    WS_NL80211_STA_INFO_RX_MPDUS,
    WS_NL80211_STA_INFO_FCS_ERROR_COUNT,
};

enum ws_nl80211_tid_stats {
    WS___NL80211_TID_STATS_INVALID,
    WS_NL80211_TID_STATS_RX_MSDU,
    WS_NL80211_TID_STATS_TX_MSDU,
    WS_NL80211_TID_STATS_TX_MSDU_RETRIES,
    WS_NL80211_TID_STATS_TX_MSDU_FAILED,
    WS_NL80211_TID_STATS_PAD,
    WS_NL80211_TID_STATS_TXQ_STATS,
};

enum ws_nl80211_txq_stats {
    WS___NL80211_TXQ_STATS_INVALID,
    WS_NL80211_TXQ_STATS_BACKLOG_BYTES,
    WS_NL80211_TXQ_STATS_BACKLOG_PACKETS,
    WS_NL80211_TXQ_STATS_FLOWS,
    WS_NL80211_TXQ_STATS_DROPS,
    WS_NL80211_TXQ_STATS_ECN_MARKS,
    WS_NL80211_TXQ_STATS_OVERLIMIT,
    WS_NL80211_TXQ_STATS_OVERMEMORY,
    WS_NL80211_TXQ_STATS_COLLISIONS,
    WS_NL80211_TXQ_STATS_TX_BYTES,
    WS_NL80211_TXQ_STATS_TX_PACKETS,
    WS_NL80211_TXQ_STATS_MAX_FLOWS,
};

enum ws_nl80211_mpath_info {
    WS___NL80211_MPATH_INFO_INVALID,
    WS_NL80211_MPATH_INFO_FRAME_QLEN,
    WS_NL80211_MPATH_INFO_SN,
    WS_NL80211_MPATH_INFO_METRIC,
    WS_NL80211_MPATH_INFO_EXPTIME,
    WS_NL80211_MPATH_INFO_FLAGS,
    WS_NL80211_MPATH_INFO_DISCOVERY_TIMEOUT,
    WS_NL80211_MPATH_INFO_DISCOVERY_RETRIES,
};

enum ws_nl80211_band_iftype_attr {
    WS___NL80211_BAND_IFTYPE_ATTR_INVALID,
    WS_NL80211_BAND_IFTYPE_ATTR_IFTYPES,
    WS_NL80211_BAND_IFTYPE_ATTR_HE_CAP_MAC,
    WS_NL80211_BAND_IFTYPE_ATTR_HE_CAP_PHY,
    WS_NL80211_BAND_IFTYPE_ATTR_HE_CAP_MCS_SET,
    WS_NL80211_BAND_IFTYPE_ATTR_HE_CAP_PPE,
};

enum ws_nl80211_band_attr {
    WS___NL80211_BAND_ATTR_INVALID,
    WS_NL80211_BAND_ATTR_FREQS,
    WS_NL80211_BAND_ATTR_RATES,
    WS_NL80211_BAND_ATTR_HT_MCS_SET,
    WS_NL80211_BAND_ATTR_HT_CAPA,
    WS_NL80211_BAND_ATTR_HT_AMPDU_FACTOR,
    WS_NL80211_BAND_ATTR_HT_AMPDU_DENSITY,
    WS_NL80211_BAND_ATTR_VHT_MCS_SET,
    WS_NL80211_BAND_ATTR_VHT_CAPA,
    WS_NL80211_BAND_ATTR_IFTYPE_DATA,
};

enum ws_nl80211_wmm_rule {
    WS___NL80211_WMMR_INVALID,
    WS_NL80211_WMMR_CW_MIN,
    WS_NL80211_WMMR_CW_MAX,
    WS_NL80211_WMMR_AIFSN,
    WS_NL80211_WMMR_TXOP,
};

enum ws_nl80211_frequency_attr {
    WS___NL80211_FREQUENCY_ATTR_INVALID,
    WS_NL80211_FREQUENCY_ATTR_FREQ,
    WS_NL80211_FREQUENCY_ATTR_DISABLED,
    WS_NL80211_FREQUENCY_ATTR_NO_IR,
    WS___NL80211_FREQUENCY_ATTR_NO_IBSS,
    WS_NL80211_FREQUENCY_ATTR_RADAR,
    WS_NL80211_FREQUENCY_ATTR_MAX_TX_POWER,
    WS_NL80211_FREQUENCY_ATTR_DFS_STATE,
    WS_NL80211_FREQUENCY_ATTR_DFS_TIME,
    WS_NL80211_FREQUENCY_ATTR_NO_HT40_MINUS,
    WS_NL80211_FREQUENCY_ATTR_NO_HT40_PLUS,
    WS_NL80211_FREQUENCY_ATTR_NO_80MHZ,
    WS_NL80211_FREQUENCY_ATTR_NO_160MHZ,
    WS_NL80211_FREQUENCY_ATTR_DFS_CAC_TIME,
    WS_NL80211_FREQUENCY_ATTR_INDOOR_ONLY,
    WS_NL80211_FREQUENCY_ATTR_IR_CONCURRENT,
    WS_NL80211_FREQUENCY_ATTR_NO_20MHZ,
    WS_NL80211_FREQUENCY_ATTR_NO_10MHZ,
    WS_NL80211_FREQUENCY_ATTR_WMM,
};

enum ws_nl80211_bitrate_attr {
    WS___NL80211_BITRATE_ATTR_INVALID,
    WS_NL80211_BITRATE_ATTR_RATE,
    WS_NL80211_BITRATE_ATTR_2GHZ_SHORTPREAMBLE,
};

enum ws_nl80211_reg_initiator {
    WS_NL80211_REGDOM_SET_BY_CORE,
    WS_NL80211_REGDOM_SET_BY_USER,
    WS_NL80211_REGDOM_SET_BY_DRIVER,
    WS_NL80211_REGDOM_SET_BY_COUNTRY_IE,
};

enum ws_nl80211_reg_type {
    WS_NL80211_REGDOM_TYPE_COUNTRY,
    WS_NL80211_REGDOM_TYPE_WORLD,
    WS_NL80211_REGDOM_TYPE_CUSTOM_WORLD,
    WS_NL80211_REGDOM_TYPE_INTERSECTION,
};

enum ws_nl80211_reg_rule_attr {
    WS___NL80211_REG_RULE_ATTR_INVALID,
    WS_NL80211_ATTR_REG_RULE_FLAGS,
    WS_NL80211_ATTR_FREQ_RANGE_START,
    WS_NL80211_ATTR_FREQ_RANGE_END,
    WS_NL80211_ATTR_FREQ_RANGE_MAX_BW,
    WS_NL80211_ATTR_POWER_RULE_MAX_ANT_GAIN,
    WS_NL80211_ATTR_POWER_RULE_MAX_EIRP,
    WS_NL80211_ATTR_DFS_CAC_TIME,
};

enum ws_nl80211_sched_scan_match_attr {
    WS___NL80211_SCHED_SCAN_MATCH_ATTR_INVALID,
    WS_NL80211_SCHED_SCAN_MATCH_ATTR_SSID,
    WS_NL80211_SCHED_SCAN_MATCH_ATTR_RSSI,
    WS_NL80211_SCHED_SCAN_MATCH_ATTR_RELATIVE_RSSI,
    WS_NL80211_SCHED_SCAN_MATCH_ATTR_RSSI_ADJUST,
    WS_NL80211_SCHED_SCAN_MATCH_ATTR_BSSID,
};

enum ws_nl80211_dfs_regions {
    WS_NL80211_DFS_UNSET = 0,
    WS_NL80211_DFS_FCC = 1,
    WS_NL80211_DFS_ETSI = 2,
    WS_NL80211_DFS_JP = 3,
};

enum ws_nl80211_user_reg_hint_type {
    WS_NL80211_USER_REG_HINT_USER = 0,
    WS_NL80211_USER_REG_HINT_CELL_BASE = 1,
    WS_NL80211_USER_REG_HINT_INDOOR = 2,
};

enum ws_nl80211_survey_info {
    WS___NL80211_SURVEY_INFO_INVALID,
    WS_NL80211_SURVEY_INFO_FREQUENCY,
    WS_NL80211_SURVEY_INFO_NOISE,
    WS_NL80211_SURVEY_INFO_IN_USE,
    WS_NL80211_SURVEY_INFO_TIME,
    WS_NL80211_SURVEY_INFO_TIME_BUSY,
    WS_NL80211_SURVEY_INFO_TIME_EXT_BUSY,
    WS_NL80211_SURVEY_INFO_TIME_RX,
    WS_NL80211_SURVEY_INFO_TIME_TX,
    WS_NL80211_SURVEY_INFO_TIME_SCAN,
    WS_NL80211_SURVEY_INFO_PAD,
};

enum ws_nl80211_mntr_flags {
    WS___NL80211_MNTR_FLAG_INVALID,
    WS_NL80211_MNTR_FLAG_FCSFAIL,
    WS_NL80211_MNTR_FLAG_PLCPFAIL,
    WS_NL80211_MNTR_FLAG_CONTROL,
    WS_NL80211_MNTR_FLAG_OTHER_BSS,
    WS_NL80211_MNTR_FLAG_COOK_FRAMES,
    WS_NL80211_MNTR_FLAG_ACTIVE,
};

enum ws_nl80211_mesh_power_mode {
    WS_NL80211_MESH_POWER_UNKNOWN,
    WS_NL80211_MESH_POWER_ACTIVE,
    WS_NL80211_MESH_POWER_LIGHT_SLEEP,
    WS_NL80211_MESH_POWER_DEEP_SLEEP,
};

enum ws_nl80211_meshconf_params {
    WS___NL80211_MESHCONF_INVALID,
    WS_NL80211_MESHCONF_RETRY_TIMEOUT,
    WS_NL80211_MESHCONF_CONFIRM_TIMEOUT,
    WS_NL80211_MESHCONF_HOLDING_TIMEOUT,
    WS_NL80211_MESHCONF_MAX_PEER_LINKS,
    WS_NL80211_MESHCONF_MAX_RETRIES,
    WS_NL80211_MESHCONF_TTL,
    WS_NL80211_MESHCONF_AUTO_OPEN_PLINKS,
    WS_NL80211_MESHCONF_HWMP_MAX_PREQ_RETRIES,
    WS_NL80211_MESHCONF_PATH_REFRESH_TIME,
    WS_NL80211_MESHCONF_MIN_DISCOVERY_TIMEOUT,
    WS_NL80211_MESHCONF_HWMP_ACTIVE_PATH_TIMEOUT,
    WS_NL80211_MESHCONF_HWMP_PREQ_MIN_INTERVAL,
    WS_NL80211_MESHCONF_HWMP_NET_DIAM_TRVS_TIME,
    WS_NL80211_MESHCONF_HWMP_ROOTMODE,
    WS_NL80211_MESHCONF_ELEMENT_TTL,
    WS_NL80211_MESHCONF_HWMP_RANN_INTERVAL,
    WS_NL80211_MESHCONF_GATE_ANNOUNCEMENTS,
    WS_NL80211_MESHCONF_HWMP_PERR_MIN_INTERVAL,
    WS_NL80211_MESHCONF_FORWARDING,
    WS_NL80211_MESHCONF_RSSI_THRESHOLD,
    WS_NL80211_MESHCONF_SYNC_OFFSET_MAX_NEIGHBOR,
    WS_NL80211_MESHCONF_HT_OPMODE,
    WS_NL80211_MESHCONF_HWMP_PATH_TO_ROOT_TIMEOUT,
    WS_NL80211_MESHCONF_HWMP_ROOT_INTERVAL,
    WS_NL80211_MESHCONF_HWMP_CONFIRMATION_INTERVAL,
    WS_NL80211_MESHCONF_POWER_MODE,
    WS_NL80211_MESHCONF_AWAKE_WINDOW,
    WS_NL80211_MESHCONF_PLINK_TIMEOUT,
};

enum ws_nl80211_mesh_setup_params {
    WS___NL80211_MESH_SETUP_INVALID,
    WS_NL80211_MESH_SETUP_ENABLE_VENDOR_PATH_SEL,
    WS_NL80211_MESH_SETUP_ENABLE_VENDOR_METRIC,
    WS_NL80211_MESH_SETUP_IE,
    WS_NL80211_MESH_SETUP_USERSPACE_AUTH,
    WS_NL80211_MESH_SETUP_USERSPACE_AMPE,
    WS_NL80211_MESH_SETUP_ENABLE_VENDOR_SYNC,
    WS_NL80211_MESH_SETUP_USERSPACE_MPM,
    WS_NL80211_MESH_SETUP_AUTH_PROTOCOL,
};

enum ws_nl80211_txq_attr {
    WS___NL80211_TXQ_ATTR_INVALID,
    WS_NL80211_TXQ_ATTR_AC,
    WS_NL80211_TXQ_ATTR_TXOP,
    WS_NL80211_TXQ_ATTR_CWMIN,
    WS_NL80211_TXQ_ATTR_CWMAX,
    WS_NL80211_TXQ_ATTR_AIFS,
};

enum ws_nl80211_channel_type {
    WS_NL80211_CHAN_NO_HT,
    WS_NL80211_CHAN_HT20,
    WS_NL80211_CHAN_HT40MINUS,
    WS_NL80211_CHAN_HT40PLUS,
};

enum ws_nl80211_chan_width {
    WS_NL80211_CHAN_WIDTH_20_NOHT,
    WS_NL80211_CHAN_WIDTH_20,
    WS_NL80211_CHAN_WIDTH_40,
    WS_NL80211_CHAN_WIDTH_80,
    WS_NL80211_CHAN_WIDTH_80P80,
    WS_NL80211_CHAN_WIDTH_160,
    WS_NL80211_CHAN_WIDTH_5,
    WS_NL80211_CHAN_WIDTH_10,
};

enum ws_nl80211_bss_scan_width {
    WS_NL80211_BSS_CHAN_WIDTH_20,
    WS_NL80211_BSS_CHAN_WIDTH_10,
    WS_NL80211_BSS_CHAN_WIDTH_5,
};

enum ws_nl80211_bss {
    WS___NL80211_BSS_INVALID,
    WS_NL80211_BSS_BSSID,
    WS_NL80211_BSS_FREQUENCY,
    WS_NL80211_BSS_TSF,
    WS_NL80211_BSS_BEACON_INTERVAL,
    WS_NL80211_BSS_CAPABILITY,
    WS_NL80211_BSS_INFORMATION_ELEMENTS,
    WS_NL80211_BSS_SIGNAL_MBM,
    WS_NL80211_BSS_SIGNAL_UNSPEC,
    WS_NL80211_BSS_STATUS,
    WS_NL80211_BSS_SEEN_MS_AGO,
    WS_NL80211_BSS_BEACON_IES,
    WS_NL80211_BSS_CHAN_WIDTH,
    WS_NL80211_BSS_BEACON_TSF,
    WS_NL80211_BSS_PRESP_DATA,
    WS_NL80211_BSS_LAST_SEEN_BOOTTIME,
    WS_NL80211_BSS_PAD,
    WS_NL80211_BSS_PARENT_TSF,
    WS_NL80211_BSS_PARENT_BSSID,
    WS_NL80211_BSS_CHAIN_SIGNAL,
};

enum ws_nl80211_bss_status {
    WS_NL80211_BSS_STATUS_AUTHENTICATED,
    WS_NL80211_BSS_STATUS_ASSOCIATED,
    WS_NL80211_BSS_STATUS_IBSS_JOINED,
};

enum ws_nl80211_auth_type {
    WS_NL80211_AUTHTYPE_OPEN_SYSTEM,
    WS_NL80211_AUTHTYPE_SHARED_KEY,
    WS_NL80211_AUTHTYPE_FT,
    WS_NL80211_AUTHTYPE_NETWORK_EAP,
    WS_NL80211_AUTHTYPE_SAE,
    WS_NL80211_AUTHTYPE_FILS_SK,
    WS_NL80211_AUTHTYPE_FILS_SK_PFS,
    WS_NL80211_AUTHTYPE_FILS_PK,
};

enum ws_nl80211_key_type {
    WS_NL80211_KEYTYPE_GROUP,
    WS_NL80211_KEYTYPE_PAIRWISE,
    WS_NL80211_KEYTYPE_PEERKEY,
};

enum ws_nl80211_mfp {
    WS_NL80211_MFP_NO,
    WS_NL80211_MFP_REQUIRED,
    WS_NL80211_MFP_OPTIONAL,
};

enum ws_nl80211_key_default_types {
    WS___NL80211_KEY_DEFAULT_TYPE_INVALID,
    WS_NL80211_KEY_DEFAULT_TYPE_UNICAST,
    WS_NL80211_KEY_DEFAULT_TYPE_MULTICAST,
};

enum ws_nl80211_key_attributes {
    WS___NL80211_KEY_INVALID,
    WS_NL80211_KEY_DATA,
    WS_NL80211_KEY_IDX,
    WS_NL80211_KEY_CIPHER,
    WS_NL80211_KEY_SEQ,
    WS_NL80211_KEY_DEFAULT,
    WS_NL80211_KEY_DEFAULT_MGMT,
    WS_NL80211_KEY_TYPE,
    WS_NL80211_KEY_DEFAULT_TYPES,
};

enum ws_nl80211_tx_rate_attributes {
    WS___NL80211_TXRATE_INVALID,
    WS_NL80211_TXRATE_LEGACY,
    WS_NL80211_TXRATE_HT,
    WS_NL80211_TXRATE_VHT,
    WS_NL80211_TXRATE_GI,
};

enum ws_nl80211_ps_state {
    WS_NL80211_PS_DISABLED,
    WS_NL80211_PS_ENABLED,
};

enum ws_nl80211_attr_cqm {
    WS___NL80211_ATTR_CQM_INVALID,
    WS_NL80211_ATTR_CQM_RSSI_THOLD,
    WS_NL80211_ATTR_CQM_RSSI_HYST,
    WS_NL80211_ATTR_CQM_RSSI_THRESHOLD_EVENT,
    WS_NL80211_ATTR_CQM_PKT_LOSS_EVENT,
    WS_NL80211_ATTR_CQM_TXE_RATE,
    WS_NL80211_ATTR_CQM_TXE_PKTS,
    WS_NL80211_ATTR_CQM_TXE_INTVL,
    WS_NL80211_ATTR_CQM_BEACON_LOSS_EVENT,
    WS_NL80211_ATTR_CQM_RSSI_LEVEL,
};

enum ws_nl80211_tx_power_setting {
    WS_NL80211_TX_POWER_AUTOMATIC,
    WS_NL80211_TX_POWER_LIMITED,
    WS_NL80211_TX_POWER_FIXED,
};

enum ws_nl80211_if_combination_attrs {
    WS_NL80211_IFACE_COMB_UNSPEC,
    WS_NL80211_IFACE_COMB_LIMITS,
    WS_NL80211_IFACE_COMB_MAXNUM,
    WS_NL80211_IFACE_COMB_STA_AP_BI_MATCH,
    WS_NL80211_IFACE_COMB_NUM_CHANNELS,
    WS_NL80211_IFACE_COMB_RADAR_DETECT_WIDTHS,
    WS_NL80211_IFACE_COMB_RADAR_DETECT_REGIONS,
    WS_NL80211_IFACE_COMB_BI_MIN_GCD,
};

enum ws_nl80211_plink_state {
    WS_NL80211_PLINK_LISTEN,
    WS_NL80211_PLINK_OPN_SNT,
    WS_NL80211_PLINK_OPN_RCVD,
    WS_NL80211_PLINK_CNF_RCVD,
    WS_NL80211_PLINK_ESTAB,
    WS_NL80211_PLINK_HOLDING,
    WS_NL80211_PLINK_BLOCKED,
};

enum ws_plink_actions {
    WS_NL80211_PLINK_ACTION_NO_ACTION,
    WS_NL80211_PLINK_ACTION_OPEN,
    WS_NL80211_PLINK_ACTION_BLOCK,
};

enum ws_nl80211_rekey_data {
    WS___NL80211_REKEY_DATA_INVALID,
    WS_NL80211_REKEY_DATA_KEK,
    WS_NL80211_REKEY_DATA_KCK,
    WS_NL80211_REKEY_DATA_REPLAY_CTR,
};

enum ws_nl80211_sta_wme_attr {
    WS___NL80211_STA_WME_INVALID,
    WS_NL80211_STA_WME_UAPSD_QUEUES,
    WS_NL80211_STA_WME_MAX_SP,
};

enum ws_nl80211_pmksa_candidate_attr {
    WS___NL80211_PMKSA_CANDIDATE_INVALID,
    WS_NL80211_PMKSA_CANDIDATE_INDEX,
    WS_NL80211_PMKSA_CANDIDATE_BSSID,
    WS_NL80211_PMKSA_CANDIDATE_PREAUTH,
};

enum ws_nl80211_tdls_operation {
    WS_NL80211_TDLS_DISCOVERY_REQ,
    WS_NL80211_TDLS_SETUP,
    WS_NL80211_TDLS_TEARDOWN,
    WS_NL80211_TDLS_ENABLE_LINK,
    WS_NL80211_TDLS_DISABLE_LINK,
};

enum ws_nl80211_connect_failed_reason {
    WS_NL80211_CONN_FAIL_MAX_CLIENTS,
    WS_NL80211_CONN_FAIL_BLOCKED_CLIENT,
};

enum ws_nl80211_timeout_reason {
    WS_NL80211_TIMEOUT_UNSPECIFIED,
    WS_NL80211_TIMEOUT_SCAN,
    WS_NL80211_TIMEOUT_AUTH,
    WS_NL80211_TIMEOUT_ASSOC,
};

enum ws_nl80211_acl_policy {
    WS_NL80211_ACL_POLICY_ACCEPT_UNLESS_LISTED,
    WS_NL80211_ACL_POLICY_DENY_UNLESS_LISTED,
};

enum ws_nl80211_smps_mode {
    WS_NL80211_SMPS_OFF,
    WS_NL80211_SMPS_STATIC,
    WS_NL80211_SMPS_DYNAMIC,
};

enum ws_nl80211_radar_event {
    WS_NL80211_RADAR_DETECTED,
    WS_NL80211_RADAR_CAC_FINISHED,
    WS_NL80211_RADAR_CAC_ABORTED,
    WS_NL80211_RADAR_NOP_FINISHED,
    WS_NL80211_RADAR_PRE_CAC_EXPIRED,
    WS_NL80211_RADAR_CAC_STARTED,
};

enum ws_nl80211_dfs_state {
    WS_NL80211_DFS_USABLE,
    WS_NL80211_DFS_UNAVAILABLE,
    WS_NL80211_DFS_AVAILABLE,
};

enum ws_nl80211_crit_proto_id {
    WS_NL80211_CRIT_PROTO_UNSPEC,
    WS_NL80211_CRIT_PROTO_DHCP,
    WS_NL80211_CRIT_PROTO_EAPOL,
    WS_NL80211_CRIT_PROTO_APIPA,
};

enum ws_nl80211_sched_scan_plan {
    WS___NL80211_SCHED_SCAN_PLAN_INVALID,
    WS_NL80211_SCHED_SCAN_PLAN_INTERVAL,
    WS_NL80211_SCHED_SCAN_PLAN_ITERATIONS,
};

enum ws_nl80211_bss_select_attr {
    WS___NL80211_BSS_SELECT_ATTR_INVALID,
    WS_NL80211_BSS_SELECT_ATTR_RSSI,
    WS_NL80211_BSS_SELECT_ATTR_BAND_PREF,
    WS_NL80211_BSS_SELECT_ATTR_RSSI_ADJUST,
};

enum ws_nl80211_nan_func_attributes {
    WS___NL80211_NAN_FUNC_INVALID,
    WS_NL80211_NAN_FUNC_TYPE,
    WS_NL80211_NAN_FUNC_SERVICE_ID,
    WS_NL80211_NAN_FUNC_PUBLISH_TYPE,
    WS_NL80211_NAN_FUNC_PUBLISH_BCAST,
    WS_NL80211_NAN_FUNC_SUBSCRIBE_ACTIVE,
    WS_NL80211_NAN_FUNC_FOLLOW_UP_ID,
    WS_NL80211_NAN_FUNC_FOLLOW_UP_REQ_ID,
    WS_NL80211_NAN_FUNC_FOLLOW_UP_DEST,
    WS_NL80211_NAN_FUNC_CLOSE_RANGE,
    WS_NL80211_NAN_FUNC_TTL,
    WS_NL80211_NAN_FUNC_SERVICE_INFO,
    WS_NL80211_NAN_FUNC_SRF,
    WS_NL80211_NAN_FUNC_RX_MATCH_FILTER,
    WS_NL80211_NAN_FUNC_TX_MATCH_FILTER,
    WS_NL80211_NAN_FUNC_INSTANCE_ID,
    WS_NL80211_NAN_FUNC_TERM_REASON,
};

enum ws_nl80211_nan_match_attributes {
    WS___NL80211_NAN_MATCH_INVALID,
    WS_NL80211_NAN_MATCH_FUNC_LOCAL,
    WS_NL80211_NAN_MATCH_FUNC_PEER,
};

enum ws_nl80211_external_auth_action {
    WS_NL80211_EXTERNAL_AUTH_START,
    WS_NL80211_EXTERNAL_AUTH_ABORT,
};

static const value_string ws_nl80211_commands_vals[] = {
    { WS_NL80211_CMD_UNSPEC,                "NL80211_CMD_UNSPEC" },
    { WS_NL80211_CMD_GET_WIPHY,             "NL80211_CMD_GET_WIPHY" },
    { WS_NL80211_CMD_SET_WIPHY,             "NL80211_CMD_SET_WIPHY" },
    { WS_NL80211_CMD_NEW_WIPHY,             "NL80211_CMD_NEW_WIPHY" },
    { WS_NL80211_CMD_DEL_WIPHY,             "NL80211_CMD_DEL_WIPHY" },
    { WS_NL80211_CMD_GET_INTERFACE,         "NL80211_CMD_GET_INTERFACE" },
    { WS_NL80211_CMD_SET_INTERFACE,         "NL80211_CMD_SET_INTERFACE" },
    { WS_NL80211_CMD_NEW_INTERFACE,         "NL80211_CMD_NEW_INTERFACE" },
    { WS_NL80211_CMD_DEL_INTERFACE,         "NL80211_CMD_DEL_INTERFACE" },
    { WS_NL80211_CMD_GET_KEY,               "NL80211_CMD_GET_KEY" },
    { WS_NL80211_CMD_SET_KEY,               "NL80211_CMD_SET_KEY" },
    { WS_NL80211_CMD_NEW_KEY,               "NL80211_CMD_NEW_KEY" },
    { WS_NL80211_CMD_DEL_KEY,               "NL80211_CMD_DEL_KEY" },
    { WS_NL80211_CMD_GET_BEACON,            "NL80211_CMD_GET_BEACON" },
    { WS_NL80211_CMD_SET_BEACON,            "NL80211_CMD_SET_BEACON" },
    { WS_NL80211_CMD_START_AP,              "NL80211_CMD_START_AP" },
    { WS_NL80211_CMD_STOP_AP,               "NL80211_CMD_STOP_AP" },
    { WS_NL80211_CMD_GET_STATION,           "NL80211_CMD_GET_STATION" },
    { WS_NL80211_CMD_SET_STATION,           "NL80211_CMD_SET_STATION" },
    { WS_NL80211_CMD_NEW_STATION,           "NL80211_CMD_NEW_STATION" },
    { WS_NL80211_CMD_DEL_STATION,           "NL80211_CMD_DEL_STATION" },
    { WS_NL80211_CMD_GET_MPATH,             "NL80211_CMD_GET_MPATH" },
    { WS_NL80211_CMD_SET_MPATH,             "NL80211_CMD_SET_MPATH" },
    { WS_NL80211_CMD_NEW_MPATH,             "NL80211_CMD_NEW_MPATH" },
    { WS_NL80211_CMD_DEL_MPATH,             "NL80211_CMD_DEL_MPATH" },
    { WS_NL80211_CMD_SET_BSS,               "NL80211_CMD_SET_BSS" },
    { WS_NL80211_CMD_SET_REG,               "NL80211_CMD_SET_REG" },
    { WS_NL80211_CMD_REQ_SET_REG,           "NL80211_CMD_REQ_SET_REG" },
    { WS_NL80211_CMD_GET_MESH_CONFIG,       "NL80211_CMD_GET_MESH_CONFIG" },
    { WS_NL80211_CMD_SET_MESH_CONFIG,       "NL80211_CMD_SET_MESH_CONFIG" },
    { WS_NL80211_CMD_SET_MGMT_EXTRA_IE,     "NL80211_CMD_SET_MGMT_EXTRA_IE" },
    { WS_NL80211_CMD_GET_REG,               "NL80211_CMD_GET_REG" },
    { WS_NL80211_CMD_GET_SCAN,              "NL80211_CMD_GET_SCAN" },
    { WS_NL80211_CMD_TRIGGER_SCAN,          "NL80211_CMD_TRIGGER_SCAN" },
    { WS_NL80211_CMD_NEW_SCAN_RESULTS,      "NL80211_CMD_NEW_SCAN_RESULTS" },
    { WS_NL80211_CMD_SCAN_ABORTED,          "NL80211_CMD_SCAN_ABORTED" },
    { WS_NL80211_CMD_REG_CHANGE,            "NL80211_CMD_REG_CHANGE" },
    { WS_NL80211_CMD_AUTHENTICATE,          "NL80211_CMD_AUTHENTICATE" },
    { WS_NL80211_CMD_ASSOCIATE,             "NL80211_CMD_ASSOCIATE" },
    { WS_NL80211_CMD_DEAUTHENTICATE,        "NL80211_CMD_DEAUTHENTICATE" },
    { WS_NL80211_CMD_DISASSOCIATE,          "NL80211_CMD_DISASSOCIATE" },
    { WS_NL80211_CMD_MICHAEL_MIC_FAILURE,   "NL80211_CMD_MICHAEL_MIC_FAILURE" },
    { WS_NL80211_CMD_REG_BEACON_HINT,       "NL80211_CMD_REG_BEACON_HINT" },
    { WS_NL80211_CMD_JOIN_IBSS,             "NL80211_CMD_JOIN_IBSS" },
    { WS_NL80211_CMD_LEAVE_IBSS,            "NL80211_CMD_LEAVE_IBSS" },
    { WS_NL80211_CMD_TESTMODE,              "NL80211_CMD_TESTMODE" },
    { WS_NL80211_CMD_CONNECT,               "NL80211_CMD_CONNECT" },
    { WS_NL80211_CMD_ROAM,                  "NL80211_CMD_ROAM" },
    { WS_NL80211_CMD_DISCONNECT,            "NL80211_CMD_DISCONNECT" },
    { WS_NL80211_CMD_SET_WIPHY_NETNS,       "NL80211_CMD_SET_WIPHY_NETNS" },
    { WS_NL80211_CMD_GET_SURVEY,            "NL80211_CMD_GET_SURVEY" },
    { WS_NL80211_CMD_NEW_SURVEY_RESULTS,    "NL80211_CMD_NEW_SURVEY_RESULTS" },
    { WS_NL80211_CMD_SET_PMKSA,             "NL80211_CMD_SET_PMKSA" },
    { WS_NL80211_CMD_DEL_PMKSA,             "NL80211_CMD_DEL_PMKSA" },
    { WS_NL80211_CMD_FLUSH_PMKSA,           "NL80211_CMD_FLUSH_PMKSA" },
    { WS_NL80211_CMD_REMAIN_ON_CHANNEL,     "NL80211_CMD_REMAIN_ON_CHANNEL" },
    { WS_NL80211_CMD_CANCEL_REMAIN_ON_CHANNEL, "NL80211_CMD_CANCEL_REMAIN_ON_CHANNEL" },
    { WS_NL80211_CMD_SET_TX_BITRATE_MASK,   "NL80211_CMD_SET_TX_BITRATE_MASK" },
    { WS_NL80211_CMD_REGISTER_FRAME,        "NL80211_CMD_REGISTER_FRAME" },
    { WS_NL80211_CMD_FRAME,                 "NL80211_CMD_FRAME" },
    { WS_NL80211_CMD_FRAME_TX_STATUS,       "NL80211_CMD_FRAME_TX_STATUS" },
    { WS_NL80211_CMD_SET_POWER_SAVE,        "NL80211_CMD_SET_POWER_SAVE" },
    { WS_NL80211_CMD_GET_POWER_SAVE,        "NL80211_CMD_GET_POWER_SAVE" },
    { WS_NL80211_CMD_SET_CQM,               "NL80211_CMD_SET_CQM" },
    { WS_NL80211_CMD_NOTIFY_CQM,            "NL80211_CMD_NOTIFY_CQM" },
    { WS_NL80211_CMD_SET_CHANNEL,           "NL80211_CMD_SET_CHANNEL" },
    { WS_NL80211_CMD_SET_WDS_PEER,          "NL80211_CMD_SET_WDS_PEER" },
    { WS_NL80211_CMD_FRAME_WAIT_CANCEL,     "NL80211_CMD_FRAME_WAIT_CANCEL" },
    { WS_NL80211_CMD_JOIN_MESH,             "NL80211_CMD_JOIN_MESH" },
    { WS_NL80211_CMD_LEAVE_MESH,            "NL80211_CMD_LEAVE_MESH" },
    { WS_NL80211_CMD_UNPROT_DEAUTHENTICATE, "NL80211_CMD_UNPROT_DEAUTHENTICATE" },
    { WS_NL80211_CMD_UNPROT_DISASSOCIATE,   "NL80211_CMD_UNPROT_DISASSOCIATE" },
    { WS_NL80211_CMD_NEW_PEER_CANDIDATE,    "NL80211_CMD_NEW_PEER_CANDIDATE" },
    { WS_NL80211_CMD_GET_WOWLAN,            "NL80211_CMD_GET_WOWLAN" },
    { WS_NL80211_CMD_SET_WOWLAN,            "NL80211_CMD_SET_WOWLAN" },
    { WS_NL80211_CMD_START_SCHED_SCAN,      "NL80211_CMD_START_SCHED_SCAN" },
    { WS_NL80211_CMD_STOP_SCHED_SCAN,       "NL80211_CMD_STOP_SCHED_SCAN" },
    { WS_NL80211_CMD_SCHED_SCAN_RESULTS,    "NL80211_CMD_SCHED_SCAN_RESULTS" },
    { WS_NL80211_CMD_SCHED_SCAN_STOPPED,    "NL80211_CMD_SCHED_SCAN_STOPPED" },
    { WS_NL80211_CMD_SET_REKEY_OFFLOAD,     "NL80211_CMD_SET_REKEY_OFFLOAD" },
    { WS_NL80211_CMD_PMKSA_CANDIDATE,       "NL80211_CMD_PMKSA_CANDIDATE" },
    { WS_NL80211_CMD_TDLS_OPER,             "NL80211_CMD_TDLS_OPER" },
    { WS_NL80211_CMD_TDLS_MGMT,             "NL80211_CMD_TDLS_MGMT" },
    { WS_NL80211_CMD_UNEXPECTED_FRAME,      "NL80211_CMD_UNEXPECTED_FRAME" },
    { WS_NL80211_CMD_PROBE_CLIENT,          "NL80211_CMD_PROBE_CLIENT" },
    { WS_NL80211_CMD_REGISTER_BEACONS,      "NL80211_CMD_REGISTER_BEACONS" },
    { WS_NL80211_CMD_UNEXPECTED_4ADDR_FRAME, "NL80211_CMD_UNEXPECTED_4ADDR_FRAME" },
    { WS_NL80211_CMD_SET_NOACK_MAP,         "NL80211_CMD_SET_NOACK_MAP" },
    { WS_NL80211_CMD_CH_SWITCH_NOTIFY,      "NL80211_CMD_CH_SWITCH_NOTIFY" },
    { WS_NL80211_CMD_START_P2P_DEVICE,      "NL80211_CMD_START_P2P_DEVICE" },
    { WS_NL80211_CMD_STOP_P2P_DEVICE,       "NL80211_CMD_STOP_P2P_DEVICE" },
    { WS_NL80211_CMD_CONN_FAILED,           "NL80211_CMD_CONN_FAILED" },
    { WS_NL80211_CMD_SET_MCAST_RATE,        "NL80211_CMD_SET_MCAST_RATE" },
    { WS_NL80211_CMD_SET_MAC_ACL,           "NL80211_CMD_SET_MAC_ACL" },
    { WS_NL80211_CMD_RADAR_DETECT,          "NL80211_CMD_RADAR_DETECT" },
    { WS_NL80211_CMD_GET_PROTOCOL_FEATURES, "NL80211_CMD_GET_PROTOCOL_FEATURES" },
    { WS_NL80211_CMD_UPDATE_FT_IES,         "NL80211_CMD_UPDATE_FT_IES" },
    { WS_NL80211_CMD_FT_EVENT,              "NL80211_CMD_FT_EVENT" },
    { WS_NL80211_CMD_CRIT_PROTOCOL_START,   "NL80211_CMD_CRIT_PROTOCOL_START" },
    { WS_NL80211_CMD_CRIT_PROTOCOL_STOP,    "NL80211_CMD_CRIT_PROTOCOL_STOP" },
    { WS_NL80211_CMD_GET_COALESCE,          "NL80211_CMD_GET_COALESCE" },
    { WS_NL80211_CMD_SET_COALESCE,          "NL80211_CMD_SET_COALESCE" },
    { WS_NL80211_CMD_CHANNEL_SWITCH,        "NL80211_CMD_CHANNEL_SWITCH" },
    { WS_NL80211_CMD_VENDOR,                "NL80211_CMD_VENDOR" },
    { WS_NL80211_CMD_SET_QOS_MAP,           "NL80211_CMD_SET_QOS_MAP" },
    { WS_NL80211_CMD_ADD_TX_TS,             "NL80211_CMD_ADD_TX_TS" },
    { WS_NL80211_CMD_DEL_TX_TS,             "NL80211_CMD_DEL_TX_TS" },
    { WS_NL80211_CMD_GET_MPP,               "NL80211_CMD_GET_MPP" },
    { WS_NL80211_CMD_JOIN_OCB,              "NL80211_CMD_JOIN_OCB" },
    { WS_NL80211_CMD_LEAVE_OCB,             "NL80211_CMD_LEAVE_OCB" },
    { WS_NL80211_CMD_CH_SWITCH_STARTED_NOTIFY, "NL80211_CMD_CH_SWITCH_STARTED_NOTIFY" },
    { WS_NL80211_CMD_TDLS_CHANNEL_SWITCH,   "NL80211_CMD_TDLS_CHANNEL_SWITCH" },
    { WS_NL80211_CMD_TDLS_CANCEL_CHANNEL_SWITCH, "NL80211_CMD_TDLS_CANCEL_CHANNEL_SWITCH" },
    { WS_NL80211_CMD_WIPHY_REG_CHANGE,      "NL80211_CMD_WIPHY_REG_CHANGE" },
    { WS_NL80211_CMD_ABORT_SCAN,            "NL80211_CMD_ABORT_SCAN" },
    { WS_NL80211_CMD_START_NAN,             "NL80211_CMD_START_NAN" },
    { WS_NL80211_CMD_STOP_NAN,              "NL80211_CMD_STOP_NAN" },
    { WS_NL80211_CMD_ADD_NAN_FUNCTION,      "NL80211_CMD_ADD_NAN_FUNCTION" },
    { WS_NL80211_CMD_DEL_NAN_FUNCTION,      "NL80211_CMD_DEL_NAN_FUNCTION" },
    { WS_NL80211_CMD_CHANGE_NAN_CONFIG,     "NL80211_CMD_CHANGE_NAN_CONFIG" },
    { WS_NL80211_CMD_NAN_MATCH,             "NL80211_CMD_NAN_MATCH" },
    { WS_NL80211_CMD_SET_MULTICAST_TO_UNICAST, "NL80211_CMD_SET_MULTICAST_TO_UNICAST" },
    { WS_NL80211_CMD_UPDATE_CONNECT_PARAMS, "NL80211_CMD_UPDATE_CONNECT_PARAMS" },
    { WS_NL80211_CMD_SET_PMK,               "NL80211_CMD_SET_PMK" },
    { WS_NL80211_CMD_DEL_PMK,               "NL80211_CMD_DEL_PMK" },
    { WS_NL80211_CMD_PORT_AUTHORIZED,       "NL80211_CMD_PORT_AUTHORIZED" },
    { WS_NL80211_CMD_RELOAD_REGDB,          "NL80211_CMD_RELOAD_REGDB" },
    { WS_NL80211_CMD_EXTERNAL_AUTH,         "NL80211_CMD_EXTERNAL_AUTH" },
    { WS_NL80211_CMD_STA_OPMODE_CHANGED,    "NL80211_CMD_STA_OPMODE_CHANGED" },
    { WS_NL80211_CMD_CONTROL_PORT_FRAME,    "NL80211_CMD_CONTROL_PORT_FRAME" },
    { WS_NL80211_CMD_GET_FTM_RESPONDER_STATS, "NL80211_CMD_GET_FTM_RESPONDER_STATS" },
    { 0, NULL }
};
static value_string_ext ws_nl80211_commands_vals_ext = VALUE_STRING_EXT_INIT(ws_nl80211_commands_vals);

static const value_string ws_nl80211_attrs_vals[] = {
    { WS_NL80211_ATTR_UNSPEC,               "NL80211_ATTR_UNSPEC" },
    { WS_NL80211_ATTR_WIPHY,                "NL80211_ATTR_WIPHY" },
    { WS_NL80211_ATTR_WIPHY_NAME,           "NL80211_ATTR_WIPHY_NAME" },
    { WS_NL80211_ATTR_IFINDEX,              "NL80211_ATTR_IFINDEX" },
    { WS_NL80211_ATTR_IFNAME,               "NL80211_ATTR_IFNAME" },
    { WS_NL80211_ATTR_IFTYPE,               "NL80211_ATTR_IFTYPE" },
    { WS_NL80211_ATTR_MAC,                  "NL80211_ATTR_MAC" },
    { WS_NL80211_ATTR_KEY_DATA,             "NL80211_ATTR_KEY_DATA" },
    { WS_NL80211_ATTR_KEY_IDX,              "NL80211_ATTR_KEY_IDX" },
    { WS_NL80211_ATTR_KEY_CIPHER,           "NL80211_ATTR_KEY_CIPHER" },
    { WS_NL80211_ATTR_KEY_SEQ,              "NL80211_ATTR_KEY_SEQ" },
    { WS_NL80211_ATTR_KEY_DEFAULT,          "NL80211_ATTR_KEY_DEFAULT" },
    { WS_NL80211_ATTR_BEACON_INTERVAL,      "NL80211_ATTR_BEACON_INTERVAL" },
    { WS_NL80211_ATTR_DTIM_PERIOD,          "NL80211_ATTR_DTIM_PERIOD" },
    { WS_NL80211_ATTR_BEACON_HEAD,          "NL80211_ATTR_BEACON_HEAD" },
    { WS_NL80211_ATTR_BEACON_TAIL,          "NL80211_ATTR_BEACON_TAIL" },
    { WS_NL80211_ATTR_STA_AID,              "NL80211_ATTR_STA_AID" },
    { WS_NL80211_ATTR_STA_FLAGS,            "NL80211_ATTR_STA_FLAGS" },
    { WS_NL80211_ATTR_STA_LISTEN_INTERVAL,  "NL80211_ATTR_STA_LISTEN_INTERVAL" },
    { WS_NL80211_ATTR_STA_SUPPORTED_RATES,  "NL80211_ATTR_STA_SUPPORTED_RATES" },
    { WS_NL80211_ATTR_STA_VLAN,             "NL80211_ATTR_STA_VLAN" },
    { WS_NL80211_ATTR_STA_INFO,             "NL80211_ATTR_STA_INFO" },
    { WS_NL80211_ATTR_WIPHY_BANDS,          "NL80211_ATTR_WIPHY_BANDS" },
    { WS_NL80211_ATTR_MNTR_FLAGS,           "NL80211_ATTR_MNTR_FLAGS" },
    { WS_NL80211_ATTR_MESH_ID,              "NL80211_ATTR_MESH_ID" },
    { WS_NL80211_ATTR_STA_PLINK_ACTION,     "NL80211_ATTR_STA_PLINK_ACTION" },
    { WS_NL80211_ATTR_MPATH_NEXT_HOP,       "NL80211_ATTR_MPATH_NEXT_HOP" },
    { WS_NL80211_ATTR_MPATH_INFO,           "NL80211_ATTR_MPATH_INFO" },
    { WS_NL80211_ATTR_BSS_CTS_PROT,         "NL80211_ATTR_BSS_CTS_PROT" },
    { WS_NL80211_ATTR_BSS_SHORT_PREAMBLE,   "NL80211_ATTR_BSS_SHORT_PREAMBLE" },
    { WS_NL80211_ATTR_BSS_SHORT_SLOT_TIME,  "NL80211_ATTR_BSS_SHORT_SLOT_TIME" },
    { WS_NL80211_ATTR_HT_CAPABILITY,        "NL80211_ATTR_HT_CAPABILITY" },
    { WS_NL80211_ATTR_SUPPORTED_IFTYPES,    "NL80211_ATTR_SUPPORTED_IFTYPES" },
    { WS_NL80211_ATTR_REG_ALPHA2,           "NL80211_ATTR_REG_ALPHA2" },
    { WS_NL80211_ATTR_REG_RULES,            "NL80211_ATTR_REG_RULES" },
    { WS_NL80211_ATTR_MESH_CONFIG,          "NL80211_ATTR_MESH_CONFIG" },
    { WS_NL80211_ATTR_BSS_BASIC_RATES,      "NL80211_ATTR_BSS_BASIC_RATES" },
    { WS_NL80211_ATTR_WIPHY_TXQ_PARAMS,     "NL80211_ATTR_WIPHY_TXQ_PARAMS" },
    { WS_NL80211_ATTR_WIPHY_FREQ,           "NL80211_ATTR_WIPHY_FREQ" },
    { WS_NL80211_ATTR_WIPHY_CHANNEL_TYPE,   "NL80211_ATTR_WIPHY_CHANNEL_TYPE" },
    { WS_NL80211_ATTR_KEY_DEFAULT_MGMT,     "NL80211_ATTR_KEY_DEFAULT_MGMT" },
    { WS_NL80211_ATTR_MGMT_SUBTYPE,         "NL80211_ATTR_MGMT_SUBTYPE" },
    { WS_NL80211_ATTR_IE,                   "NL80211_ATTR_IE" },
    { WS_NL80211_ATTR_MAX_NUM_SCAN_SSIDS,   "NL80211_ATTR_MAX_NUM_SCAN_SSIDS" },
    { WS_NL80211_ATTR_SCAN_FREQUENCIES,     "NL80211_ATTR_SCAN_FREQUENCIES" },
    { WS_NL80211_ATTR_SCAN_SSIDS,           "NL80211_ATTR_SCAN_SSIDS" },
    { WS_NL80211_ATTR_GENERATION,           "NL80211_ATTR_GENERATION" },
    { WS_NL80211_ATTR_BSS,                  "NL80211_ATTR_BSS" },
    { WS_NL80211_ATTR_REG_INITIATOR,        "NL80211_ATTR_REG_INITIATOR" },
    { WS_NL80211_ATTR_REG_TYPE,             "NL80211_ATTR_REG_TYPE" },
    { WS_NL80211_ATTR_SUPPORTED_COMMANDS,   "NL80211_ATTR_SUPPORTED_COMMANDS" },
    { WS_NL80211_ATTR_FRAME,                "NL80211_ATTR_FRAME" },
    { WS_NL80211_ATTR_SSID,                 "NL80211_ATTR_SSID" },
    { WS_NL80211_ATTR_AUTH_TYPE,            "NL80211_ATTR_AUTH_TYPE" },
    { WS_NL80211_ATTR_REASON_CODE,          "NL80211_ATTR_REASON_CODE" },
    { WS_NL80211_ATTR_KEY_TYPE,             "NL80211_ATTR_KEY_TYPE" },
    { WS_NL80211_ATTR_MAX_SCAN_IE_LEN,      "NL80211_ATTR_MAX_SCAN_IE_LEN" },
    { WS_NL80211_ATTR_CIPHER_SUITES,        "NL80211_ATTR_CIPHER_SUITES" },
    { WS_NL80211_ATTR_FREQ_BEFORE,          "NL80211_ATTR_FREQ_BEFORE" },
    { WS_NL80211_ATTR_FREQ_AFTER,           "NL80211_ATTR_FREQ_AFTER" },
    { WS_NL80211_ATTR_FREQ_FIXED,           "NL80211_ATTR_FREQ_FIXED" },
    { WS_NL80211_ATTR_WIPHY_RETRY_SHORT,    "NL80211_ATTR_WIPHY_RETRY_SHORT" },
    { WS_NL80211_ATTR_WIPHY_RETRY_LONG,     "NL80211_ATTR_WIPHY_RETRY_LONG" },
    { WS_NL80211_ATTR_WIPHY_FRAG_THRESHOLD, "NL80211_ATTR_WIPHY_FRAG_THRESHOLD" },
    { WS_NL80211_ATTR_WIPHY_RTS_THRESHOLD,  "NL80211_ATTR_WIPHY_RTS_THRESHOLD" },
    { WS_NL80211_ATTR_TIMED_OUT,            "NL80211_ATTR_TIMED_OUT" },
    { WS_NL80211_ATTR_USE_MFP,              "NL80211_ATTR_USE_MFP" },
    { WS_NL80211_ATTR_STA_FLAGS2,           "NL80211_ATTR_STA_FLAGS2" },
    { WS_NL80211_ATTR_CONTROL_PORT,         "NL80211_ATTR_CONTROL_PORT" },
    { WS_NL80211_ATTR_TESTDATA,             "NL80211_ATTR_TESTDATA" },
    { WS_NL80211_ATTR_PRIVACY,              "NL80211_ATTR_PRIVACY" },
    { WS_NL80211_ATTR_DISCONNECTED_BY_AP,   "NL80211_ATTR_DISCONNECTED_BY_AP" },
    { WS_NL80211_ATTR_STATUS_CODE,          "NL80211_ATTR_STATUS_CODE" },
    { WS_NL80211_ATTR_CIPHER_SUITES_PAIRWISE, "NL80211_ATTR_CIPHER_SUITES_PAIRWISE" },
    { WS_NL80211_ATTR_CIPHER_SUITE_GROUP,   "NL80211_ATTR_CIPHER_SUITE_GROUP" },
    { WS_NL80211_ATTR_WPA_VERSIONS,         "NL80211_ATTR_WPA_VERSIONS" },
    { WS_NL80211_ATTR_AKM_SUITES,           "NL80211_ATTR_AKM_SUITES" },
    { WS_NL80211_ATTR_REQ_IE,               "NL80211_ATTR_REQ_IE" },
    { WS_NL80211_ATTR_RESP_IE,              "NL80211_ATTR_RESP_IE" },
    { WS_NL80211_ATTR_PREV_BSSID,           "NL80211_ATTR_PREV_BSSID" },
    { WS_NL80211_ATTR_KEY,                  "NL80211_ATTR_KEY" },
    { WS_NL80211_ATTR_KEYS,                 "NL80211_ATTR_KEYS" },
    { WS_NL80211_ATTR_PID,                  "NL80211_ATTR_PID" },
    { WS_NL80211_ATTR_4ADDR,                "NL80211_ATTR_4ADDR" },
    { WS_NL80211_ATTR_SURVEY_INFO,          "NL80211_ATTR_SURVEY_INFO" },
    { WS_NL80211_ATTR_PMKID,                "NL80211_ATTR_PMKID" },
    { WS_NL80211_ATTR_MAX_NUM_PMKIDS,       "NL80211_ATTR_MAX_NUM_PMKIDS" },
    { WS_NL80211_ATTR_DURATION,             "NL80211_ATTR_DURATION" },
    { WS_NL80211_ATTR_COOKIE,               "NL80211_ATTR_COOKIE" },
    { WS_NL80211_ATTR_WIPHY_COVERAGE_CLASS, "NL80211_ATTR_WIPHY_COVERAGE_CLASS" },
    { WS_NL80211_ATTR_TX_RATES,             "NL80211_ATTR_TX_RATES" },
    { WS_NL80211_ATTR_FRAME_MATCH,          "NL80211_ATTR_FRAME_MATCH" },
    { WS_NL80211_ATTR_ACK,                  "NL80211_ATTR_ACK" },
    { WS_NL80211_ATTR_PS_STATE,             "NL80211_ATTR_PS_STATE" },
    { WS_NL80211_ATTR_CQM,                  "NL80211_ATTR_CQM" },
    { WS_NL80211_ATTR_LOCAL_STATE_CHANGE,   "NL80211_ATTR_LOCAL_STATE_CHANGE" },
    { WS_NL80211_ATTR_AP_ISOLATE,           "NL80211_ATTR_AP_ISOLATE" },
    { WS_NL80211_ATTR_WIPHY_TX_POWER_SETTING, "NL80211_ATTR_WIPHY_TX_POWER_SETTING" },
    { WS_NL80211_ATTR_WIPHY_TX_POWER_LEVEL, "NL80211_ATTR_WIPHY_TX_POWER_LEVEL" },
    { WS_NL80211_ATTR_TX_FRAME_TYPES,       "NL80211_ATTR_TX_FRAME_TYPES" },
    { WS_NL80211_ATTR_RX_FRAME_TYPES,       "NL80211_ATTR_RX_FRAME_TYPES" },
    { WS_NL80211_ATTR_FRAME_TYPE,           "NL80211_ATTR_FRAME_TYPE" },
    { WS_NL80211_ATTR_CONTROL_PORT_ETHERTYPE, "NL80211_ATTR_CONTROL_PORT_ETHERTYPE" },
    { WS_NL80211_ATTR_CONTROL_PORT_NO_ENCRYPT, "NL80211_ATTR_CONTROL_PORT_NO_ENCRYPT" },
    { WS_NL80211_ATTR_SUPPORT_IBSS_RSN,     "NL80211_ATTR_SUPPORT_IBSS_RSN" },
    { WS_NL80211_ATTR_WIPHY_ANTENNA_TX,     "NL80211_ATTR_WIPHY_ANTENNA_TX" },
    { WS_NL80211_ATTR_WIPHY_ANTENNA_RX,     "NL80211_ATTR_WIPHY_ANTENNA_RX" },
    { WS_NL80211_ATTR_MCAST_RATE,           "NL80211_ATTR_MCAST_RATE" },
    { WS_NL80211_ATTR_OFFCHANNEL_TX_OK,     "NL80211_ATTR_OFFCHANNEL_TX_OK" },
    { WS_NL80211_ATTR_BSS_HT_OPMODE,        "NL80211_ATTR_BSS_HT_OPMODE" },
    { WS_NL80211_ATTR_KEY_DEFAULT_TYPES,    "NL80211_ATTR_KEY_DEFAULT_TYPES" },
    { WS_NL80211_ATTR_MAX_REMAIN_ON_CHANNEL_DURATION, "NL80211_ATTR_MAX_REMAIN_ON_CHANNEL_DURATION" },
    { WS_NL80211_ATTR_MESH_SETUP,           "NL80211_ATTR_MESH_SETUP" },
    { WS_NL80211_ATTR_WIPHY_ANTENNA_AVAIL_TX, "NL80211_ATTR_WIPHY_ANTENNA_AVAIL_TX" },
    { WS_NL80211_ATTR_WIPHY_ANTENNA_AVAIL_RX, "NL80211_ATTR_WIPHY_ANTENNA_AVAIL_RX" },
    { WS_NL80211_ATTR_SUPPORT_MESH_AUTH,    "NL80211_ATTR_SUPPORT_MESH_AUTH" },
    { WS_NL80211_ATTR_STA_PLINK_STATE,      "NL80211_ATTR_STA_PLINK_STATE" },
    { WS_NL80211_ATTR_WOWLAN_TRIGGERS,      "NL80211_ATTR_WOWLAN_TRIGGERS" },
    { WS_NL80211_ATTR_WOWLAN_TRIGGERS_SUPPORTED, "NL80211_ATTR_WOWLAN_TRIGGERS_SUPPORTED" },
    { WS_NL80211_ATTR_SCHED_SCAN_INTERVAL,  "NL80211_ATTR_SCHED_SCAN_INTERVAL" },
    { WS_NL80211_ATTR_INTERFACE_COMBINATIONS, "NL80211_ATTR_INTERFACE_COMBINATIONS" },
    { WS_NL80211_ATTR_SOFTWARE_IFTYPES,     "NL80211_ATTR_SOFTWARE_IFTYPES" },
    { WS_NL80211_ATTR_REKEY_DATA,           "NL80211_ATTR_REKEY_DATA" },
    { WS_NL80211_ATTR_MAX_NUM_SCHED_SCAN_SSIDS, "NL80211_ATTR_MAX_NUM_SCHED_SCAN_SSIDS" },
    { WS_NL80211_ATTR_MAX_SCHED_SCAN_IE_LEN, "NL80211_ATTR_MAX_SCHED_SCAN_IE_LEN" },
    { WS_NL80211_ATTR_SCAN_SUPP_RATES,      "NL80211_ATTR_SCAN_SUPP_RATES" },
    { WS_NL80211_ATTR_HIDDEN_SSID,          "NL80211_ATTR_HIDDEN_SSID" },
    { WS_NL80211_ATTR_IE_PROBE_RESP,        "NL80211_ATTR_IE_PROBE_RESP" },
    { WS_NL80211_ATTR_IE_ASSOC_RESP,        "NL80211_ATTR_IE_ASSOC_RESP" },
    { WS_NL80211_ATTR_STA_WME,              "NL80211_ATTR_STA_WME" },
    { WS_NL80211_ATTR_SUPPORT_AP_UAPSD,     "NL80211_ATTR_SUPPORT_AP_UAPSD" },
    { WS_NL80211_ATTR_ROAM_SUPPORT,         "NL80211_ATTR_ROAM_SUPPORT" },
    { WS_NL80211_ATTR_SCHED_SCAN_MATCH,     "NL80211_ATTR_SCHED_SCAN_MATCH" },
    { WS_NL80211_ATTR_MAX_MATCH_SETS,       "NL80211_ATTR_MAX_MATCH_SETS" },
    { WS_NL80211_ATTR_PMKSA_CANDIDATE,      "NL80211_ATTR_PMKSA_CANDIDATE" },
    { WS_NL80211_ATTR_TX_NO_CCK_RATE,       "NL80211_ATTR_TX_NO_CCK_RATE" },
    { WS_NL80211_ATTR_TDLS_ACTION,          "NL80211_ATTR_TDLS_ACTION" },
    { WS_NL80211_ATTR_TDLS_DIALOG_TOKEN,    "NL80211_ATTR_TDLS_DIALOG_TOKEN" },
    { WS_NL80211_ATTR_TDLS_OPERATION,       "NL80211_ATTR_TDLS_OPERATION" },
    { WS_NL80211_ATTR_TDLS_SUPPORT,         "NL80211_ATTR_TDLS_SUPPORT" },
    { WS_NL80211_ATTR_TDLS_EXTERNAL_SETUP,  "NL80211_ATTR_TDLS_EXTERNAL_SETUP" },
    { WS_NL80211_ATTR_DEVICE_AP_SME,        "NL80211_ATTR_DEVICE_AP_SME" },
    { WS_NL80211_ATTR_DONT_WAIT_FOR_ACK,    "NL80211_ATTR_DONT_WAIT_FOR_ACK" },
    { WS_NL80211_ATTR_FEATURE_FLAGS,        "NL80211_ATTR_FEATURE_FLAGS" },
    { WS_NL80211_ATTR_PROBE_RESP_OFFLOAD,   "NL80211_ATTR_PROBE_RESP_OFFLOAD" },
    { WS_NL80211_ATTR_PROBE_RESP,           "NL80211_ATTR_PROBE_RESP" },
    { WS_NL80211_ATTR_DFS_REGION,           "NL80211_ATTR_DFS_REGION" },
    { WS_NL80211_ATTR_DISABLE_HT,           "NL80211_ATTR_DISABLE_HT" },
    { WS_NL80211_ATTR_HT_CAPABILITY_MASK,   "NL80211_ATTR_HT_CAPABILITY_MASK" },
    { WS_NL80211_ATTR_NOACK_MAP,            "NL80211_ATTR_NOACK_MAP" },
    { WS_NL80211_ATTR_INACTIVITY_TIMEOUT,   "NL80211_ATTR_INACTIVITY_TIMEOUT" },
    { WS_NL80211_ATTR_RX_SIGNAL_DBM,        "NL80211_ATTR_RX_SIGNAL_DBM" },
    { WS_NL80211_ATTR_BG_SCAN_PERIOD,       "NL80211_ATTR_BG_SCAN_PERIOD" },
    { WS_NL80211_ATTR_WDEV,                 "NL80211_ATTR_WDEV" },
    { WS_NL80211_ATTR_USER_REG_HINT_TYPE,   "NL80211_ATTR_USER_REG_HINT_TYPE" },
    { WS_NL80211_ATTR_CONN_FAILED_REASON,   "NL80211_ATTR_CONN_FAILED_REASON" },
    { WS_NL80211_ATTR_AUTH_DATA,            "NL80211_ATTR_AUTH_DATA" },
    { WS_NL80211_ATTR_VHT_CAPABILITY,       "NL80211_ATTR_VHT_CAPABILITY" },
    { WS_NL80211_ATTR_SCAN_FLAGS,           "NL80211_ATTR_SCAN_FLAGS" },
    { WS_NL80211_ATTR_CHANNEL_WIDTH,        "NL80211_ATTR_CHANNEL_WIDTH" },
    { WS_NL80211_ATTR_CENTER_FREQ1,         "NL80211_ATTR_CENTER_FREQ1" },
    { WS_NL80211_ATTR_CENTER_FREQ2,         "NL80211_ATTR_CENTER_FREQ2" },
    { WS_NL80211_ATTR_P2P_CTWINDOW,         "NL80211_ATTR_P2P_CTWINDOW" },
    { WS_NL80211_ATTR_P2P_OPPPS,            "NL80211_ATTR_P2P_OPPPS" },
    { WS_NL80211_ATTR_LOCAL_MESH_POWER_MODE, "NL80211_ATTR_LOCAL_MESH_POWER_MODE" },
    { WS_NL80211_ATTR_ACL_POLICY,           "NL80211_ATTR_ACL_POLICY" },
    { WS_NL80211_ATTR_MAC_ADDRS,            "NL80211_ATTR_MAC_ADDRS" },
    { WS_NL80211_ATTR_MAC_ACL_MAX,          "NL80211_ATTR_MAC_ACL_MAX" },
    { WS_NL80211_ATTR_RADAR_EVENT,          "NL80211_ATTR_RADAR_EVENT" },
    { WS_NL80211_ATTR_EXT_CAPA,             "NL80211_ATTR_EXT_CAPA" },
    { WS_NL80211_ATTR_EXT_CAPA_MASK,        "NL80211_ATTR_EXT_CAPA_MASK" },
    { WS_NL80211_ATTR_STA_CAPABILITY,       "NL80211_ATTR_STA_CAPABILITY" },
    { WS_NL80211_ATTR_STA_EXT_CAPABILITY,   "NL80211_ATTR_STA_EXT_CAPABILITY" },
    { WS_NL80211_ATTR_PROTOCOL_FEATURES,    "NL80211_ATTR_PROTOCOL_FEATURES" },
    { WS_NL80211_ATTR_SPLIT_WIPHY_DUMP,     "NL80211_ATTR_SPLIT_WIPHY_DUMP" },
    { WS_NL80211_ATTR_DISABLE_VHT,          "NL80211_ATTR_DISABLE_VHT" },
    { WS_NL80211_ATTR_VHT_CAPABILITY_MASK,  "NL80211_ATTR_VHT_CAPABILITY_MASK" },
    { WS_NL80211_ATTR_MDID,                 "NL80211_ATTR_MDID" },
    { WS_NL80211_ATTR_IE_RIC,               "NL80211_ATTR_IE_RIC" },
    { WS_NL80211_ATTR_CRIT_PROT_ID,         "NL80211_ATTR_CRIT_PROT_ID" },
    { WS_NL80211_ATTR_MAX_CRIT_PROT_DURATION, "NL80211_ATTR_MAX_CRIT_PROT_DURATION" },
    { WS_NL80211_ATTR_PEER_AID,             "NL80211_ATTR_PEER_AID" },
    { WS_NL80211_ATTR_COALESCE_RULE,        "NL80211_ATTR_COALESCE_RULE" },
    { WS_NL80211_ATTR_CH_SWITCH_COUNT,      "NL80211_ATTR_CH_SWITCH_COUNT" },
    { WS_NL80211_ATTR_CH_SWITCH_BLOCK_TX,   "NL80211_ATTR_CH_SWITCH_BLOCK_TX" },
    { WS_NL80211_ATTR_CSA_IES,              "NL80211_ATTR_CSA_IES" },
    { WS_NL80211_ATTR_CSA_C_OFF_BEACON,     "NL80211_ATTR_CSA_C_OFF_BEACON" },
    { WS_NL80211_ATTR_CSA_C_OFF_PRESP,      "NL80211_ATTR_CSA_C_OFF_PRESP" },
    { WS_NL80211_ATTR_RXMGMT_FLAGS,         "NL80211_ATTR_RXMGMT_FLAGS" },
    { WS_NL80211_ATTR_STA_SUPPORTED_CHANNELS, "NL80211_ATTR_STA_SUPPORTED_CHANNELS" },
    { WS_NL80211_ATTR_STA_SUPPORTED_OPER_CLASSES, "NL80211_ATTR_STA_SUPPORTED_OPER_CLASSES" },
    { WS_NL80211_ATTR_HANDLE_DFS,           "NL80211_ATTR_HANDLE_DFS" },
    { WS_NL80211_ATTR_SUPPORT_5_MHZ,        "NL80211_ATTR_SUPPORT_5_MHZ" },
    { WS_NL80211_ATTR_SUPPORT_10_MHZ,       "NL80211_ATTR_SUPPORT_10_MHZ" },
    { WS_NL80211_ATTR_OPMODE_NOTIF,         "NL80211_ATTR_OPMODE_NOTIF" },
    { WS_NL80211_ATTR_VENDOR_ID,            "NL80211_ATTR_VENDOR_ID" },
    { WS_NL80211_ATTR_VENDOR_SUBCMD,        "NL80211_ATTR_VENDOR_SUBCMD" },
    { WS_NL80211_ATTR_VENDOR_DATA,          "NL80211_ATTR_VENDOR_DATA" },
    { WS_NL80211_ATTR_VENDOR_EVENTS,        "NL80211_ATTR_VENDOR_EVENTS" },
    { WS_NL80211_ATTR_QOS_MAP,              "NL80211_ATTR_QOS_MAP" },
    { WS_NL80211_ATTR_MAC_HINT,             "NL80211_ATTR_MAC_HINT" },
    { WS_NL80211_ATTR_WIPHY_FREQ_HINT,      "NL80211_ATTR_WIPHY_FREQ_HINT" },
    { WS_NL80211_ATTR_MAX_AP_ASSOC_STA,     "NL80211_ATTR_MAX_AP_ASSOC_STA" },
    { WS_NL80211_ATTR_TDLS_PEER_CAPABILITY, "NL80211_ATTR_TDLS_PEER_CAPABILITY" },
    { WS_NL80211_ATTR_SOCKET_OWNER,         "NL80211_ATTR_SOCKET_OWNER" },
    { WS_NL80211_ATTR_CSA_C_OFFSETS_TX,     "NL80211_ATTR_CSA_C_OFFSETS_TX" },
    { WS_NL80211_ATTR_MAX_CSA_COUNTERS,     "NL80211_ATTR_MAX_CSA_COUNTERS" },
    { WS_NL80211_ATTR_TDLS_INITIATOR,       "NL80211_ATTR_TDLS_INITIATOR" },
    { WS_NL80211_ATTR_USE_RRM,              "NL80211_ATTR_USE_RRM" },
    { WS_NL80211_ATTR_WIPHY_DYN_ACK,        "NL80211_ATTR_WIPHY_DYN_ACK" },
    { WS_NL80211_ATTR_TSID,                 "NL80211_ATTR_TSID" },
    { WS_NL80211_ATTR_USER_PRIO,            "NL80211_ATTR_USER_PRIO" },
    { WS_NL80211_ATTR_ADMITTED_TIME,        "NL80211_ATTR_ADMITTED_TIME" },
    { WS_NL80211_ATTR_SMPS_MODE,            "NL80211_ATTR_SMPS_MODE" },
    { WS_NL80211_ATTR_OPER_CLASS,           "NL80211_ATTR_OPER_CLASS" },
    { WS_NL80211_ATTR_MAC_MASK,             "NL80211_ATTR_MAC_MASK" },
    { WS_NL80211_ATTR_WIPHY_SELF_MANAGED_REG, "NL80211_ATTR_WIPHY_SELF_MANAGED_REG" },
    { WS_NL80211_ATTR_EXT_FEATURES,         "NL80211_ATTR_EXT_FEATURES" },
    { WS_NL80211_ATTR_SURVEY_RADIO_STATS,   "NL80211_ATTR_SURVEY_RADIO_STATS" },
    { WS_NL80211_ATTR_NETNS_FD,             "NL80211_ATTR_NETNS_FD" },
    { WS_NL80211_ATTR_SCHED_SCAN_DELAY,     "NL80211_ATTR_SCHED_SCAN_DELAY" },
    { WS_NL80211_ATTR_REG_INDOOR,           "NL80211_ATTR_REG_INDOOR" },
    { WS_NL80211_ATTR_MAX_NUM_SCHED_SCAN_PLANS, "NL80211_ATTR_MAX_NUM_SCHED_SCAN_PLANS" },
    { WS_NL80211_ATTR_MAX_SCAN_PLAN_INTERVAL, "NL80211_ATTR_MAX_SCAN_PLAN_INTERVAL" },
    { WS_NL80211_ATTR_MAX_SCAN_PLAN_ITERATIONS, "NL80211_ATTR_MAX_SCAN_PLAN_ITERATIONS" },
    { WS_NL80211_ATTR_SCHED_SCAN_PLANS,     "NL80211_ATTR_SCHED_SCAN_PLANS" },
    { WS_NL80211_ATTR_PBSS,                 "NL80211_ATTR_PBSS" },
    { WS_NL80211_ATTR_BSS_SELECT,           "NL80211_ATTR_BSS_SELECT" },
    { WS_NL80211_ATTR_STA_SUPPORT_P2P_PS,   "NL80211_ATTR_STA_SUPPORT_P2P_PS" },
    { WS_NL80211_ATTR_PAD,                  "NL80211_ATTR_PAD" },
    { WS_NL80211_ATTR_IFTYPE_EXT_CAPA,      "NL80211_ATTR_IFTYPE_EXT_CAPA" },
    { WS_NL80211_ATTR_MU_MIMO_GROUP_DATA,   "NL80211_ATTR_MU_MIMO_GROUP_DATA" },
    { WS_NL80211_ATTR_MU_MIMO_FOLLOW_MAC_ADDR, "NL80211_ATTR_MU_MIMO_FOLLOW_MAC_ADDR" },
    { WS_NL80211_ATTR_SCAN_START_TIME_TSF,  "NL80211_ATTR_SCAN_START_TIME_TSF" },
    { WS_NL80211_ATTR_SCAN_START_TIME_TSF_BSSID, "NL80211_ATTR_SCAN_START_TIME_TSF_BSSID" },
    { WS_NL80211_ATTR_MEASUREMENT_DURATION, "NL80211_ATTR_MEASUREMENT_DURATION" },
    { WS_NL80211_ATTR_MEASUREMENT_DURATION_MANDATORY, "NL80211_ATTR_MEASUREMENT_DURATION_MANDATORY" },
    { WS_NL80211_ATTR_MESH_PEER_AID,        "NL80211_ATTR_MESH_PEER_AID" },
    { WS_NL80211_ATTR_NAN_MASTER_PREF,      "NL80211_ATTR_NAN_MASTER_PREF" },
    { WS_NL80211_ATTR_BANDS,                "NL80211_ATTR_BANDS" },
    { WS_NL80211_ATTR_NAN_FUNC,             "NL80211_ATTR_NAN_FUNC" },
    { WS_NL80211_ATTR_NAN_MATCH,            "NL80211_ATTR_NAN_MATCH" },
    { WS_NL80211_ATTR_FILS_KEK,             "NL80211_ATTR_FILS_KEK" },
    { WS_NL80211_ATTR_FILS_NONCES,          "NL80211_ATTR_FILS_NONCES" },
    { WS_NL80211_ATTR_MULTICAST_TO_UNICAST_ENABLED, "NL80211_ATTR_MULTICAST_TO_UNICAST_ENABLED" },
    { WS_NL80211_ATTR_BSSID,                "NL80211_ATTR_BSSID" },
    { WS_NL80211_ATTR_SCHED_SCAN_RELATIVE_RSSI, "NL80211_ATTR_SCHED_SCAN_RELATIVE_RSSI" },
    { WS_NL80211_ATTR_SCHED_SCAN_RSSI_ADJUST, "NL80211_ATTR_SCHED_SCAN_RSSI_ADJUST" },
    { WS_NL80211_ATTR_TIMEOUT_REASON,       "NL80211_ATTR_TIMEOUT_REASON" },
    { WS_NL80211_ATTR_FILS_ERP_USERNAME,    "NL80211_ATTR_FILS_ERP_USERNAME" },
    { WS_NL80211_ATTR_FILS_ERP_REALM,       "NL80211_ATTR_FILS_ERP_REALM" },
    { WS_NL80211_ATTR_FILS_ERP_NEXT_SEQ_NUM, "NL80211_ATTR_FILS_ERP_NEXT_SEQ_NUM" },
    { WS_NL80211_ATTR_FILS_ERP_RRK,         "NL80211_ATTR_FILS_ERP_RRK" },
    { WS_NL80211_ATTR_FILS_CACHE_ID,        "NL80211_ATTR_FILS_CACHE_ID" },
    { WS_NL80211_ATTR_PMK,                  "NL80211_ATTR_PMK" },
    { WS_NL80211_ATTR_SCHED_SCAN_MULTI,     "NL80211_ATTR_SCHED_SCAN_MULTI" },
    { WS_NL80211_ATTR_SCHED_SCAN_MAX_REQS,  "NL80211_ATTR_SCHED_SCAN_MAX_REQS" },
    { WS_NL80211_ATTR_WANT_1X_4WAY_HS,      "NL80211_ATTR_WANT_1X_4WAY_HS" },
    { WS_NL80211_ATTR_PMKR0_NAME,           "NL80211_ATTR_PMKR0_NAME" },
    { WS_NL80211_ATTR_PORT_AUTHORIZED,      "NL80211_ATTR_PORT_AUTHORIZED" },
    { WS_NL80211_ATTR_EXTERNAL_AUTH_ACTION, "NL80211_ATTR_EXTERNAL_AUTH_ACTION" },
    { WS_NL80211_ATTR_EXTERNAL_AUTH_SUPPORT, "NL80211_ATTR_EXTERNAL_AUTH_SUPPORT" },
    { WS_NL80211_ATTR_NSS,                  "NL80211_ATTR_NSS" },
    { WS_NL80211_ATTR_ACK_SIGNAL,           "NL80211_ATTR_ACK_SIGNAL" },
    { WS_NL80211_ATTR_CONTROL_PORT_OVER_NL80211, "NL80211_ATTR_CONTROL_PORT_OVER_NL80211" },
    { WS_NL80211_ATTR_TXQ_STATS,            "NL80211_ATTR_TXQ_STATS" },
    { WS_NL80211_ATTR_TXQ_LIMIT,            "NL80211_ATTR_TXQ_LIMIT" },
    { WS_NL80211_ATTR_TXQ_MEMORY_LIMIT,     "NL80211_ATTR_TXQ_MEMORY_LIMIT" },
    { WS_NL80211_ATTR_TXQ_QUANTUM,          "NL80211_ATTR_TXQ_QUANTUM" },
    { WS_NL80211_ATTR_HE_CAPABILITY,        "NL80211_ATTR_HE_CAPABILITY" },
    { WS_NL80211_ATTR_FTM_RESPONDER,        "NL80211_ATTR_FTM_RESPONDER" },
    { WS_NL80211_ATTR_FTM_RESPONDER_STATS,  "NL80211_ATTR_FTM_RESPONDER_STATS" },
    { 0, NULL }
};
static value_string_ext ws_nl80211_attrs_vals_ext = VALUE_STRING_EXT_INIT(ws_nl80211_attrs_vals);

static const value_string ws_nl80211_iftype_vals[] = {
    { WS_NL80211_IFTYPE_UNSPECIFIED,        "NL80211_IFTYPE_UNSPECIFIED" },
    { WS_NL80211_IFTYPE_ADHOC,              "NL80211_IFTYPE_ADHOC" },
    { WS_NL80211_IFTYPE_STATION,            "NL80211_IFTYPE_STATION" },
    { WS_NL80211_IFTYPE_AP,                 "NL80211_IFTYPE_AP" },
    { WS_NL80211_IFTYPE_AP_VLAN,            "NL80211_IFTYPE_AP_VLAN" },
    { WS_NL80211_IFTYPE_WDS,                "NL80211_IFTYPE_WDS" },
    { WS_NL80211_IFTYPE_MONITOR,            "NL80211_IFTYPE_MONITOR" },
    { WS_NL80211_IFTYPE_MESH_POINT,         "NL80211_IFTYPE_MESH_POINT" },
    { WS_NL80211_IFTYPE_P2P_CLIENT,         "NL80211_IFTYPE_P2P_CLIENT" },
    { WS_NL80211_IFTYPE_P2P_GO,             "NL80211_IFTYPE_P2P_GO" },
    { WS_NL80211_IFTYPE_P2P_DEVICE,         "NL80211_IFTYPE_P2P_DEVICE" },
    { WS_NL80211_IFTYPE_OCB,                "NL80211_IFTYPE_OCB" },
    { WS_NL80211_IFTYPE_NAN,                "NL80211_IFTYPE_NAN" },
    { 0, NULL }
};
static value_string_ext ws_nl80211_iftype_vals_ext = VALUE_STRING_EXT_INIT(ws_nl80211_iftype_vals);

static const value_string ws_nl80211_sta_flags_vals[] = {
    { WS___NL80211_STA_FLAG_INVALID,        "__NL80211_STA_FLAG_INVALID" },
    { WS_NL80211_STA_FLAG_AUTHORIZED,       "NL80211_STA_FLAG_AUTHORIZED" },
    { WS_NL80211_STA_FLAG_SHORT_PREAMBLE,   "NL80211_STA_FLAG_SHORT_PREAMBLE" },
    { WS_NL80211_STA_FLAG_WME,              "NL80211_STA_FLAG_WME" },
    { WS_NL80211_STA_FLAG_MFP,              "NL80211_STA_FLAG_MFP" },
    { WS_NL80211_STA_FLAG_AUTHENTICATED,    "NL80211_STA_FLAG_AUTHENTICATED" },
    { WS_NL80211_STA_FLAG_TDLS_PEER,        "NL80211_STA_FLAG_TDLS_PEER" },
    { WS_NL80211_STA_FLAG_ASSOCIATED,       "NL80211_STA_FLAG_ASSOCIATED" },
    { 0, NULL }
};
static value_string_ext ws_nl80211_sta_flags_vals_ext = VALUE_STRING_EXT_INIT(ws_nl80211_sta_flags_vals);

static const value_string ws_nl80211_sta_p2p_ps_status_vals[] = {
    { WS_NL80211_P2P_PS_UNSUPPORTED,        "NL80211_P2P_PS_UNSUPPORTED" },
    { WS_NL80211_P2P_PS_SUPPORTED,          "NL80211_P2P_PS_SUPPORTED" },
    { 0, NULL }
};
static value_string_ext ws_nl80211_sta_p2p_ps_status_vals_ext = VALUE_STRING_EXT_INIT(ws_nl80211_sta_p2p_ps_status_vals);

static const value_string ws_nl80211_rate_info_vals[] = {
    { WS___NL80211_RATE_INFO_INVALID,       "__NL80211_RATE_INFO_INVALID" },
    { WS_NL80211_RATE_INFO_BITRATE,         "NL80211_RATE_INFO_BITRATE" },
    { WS_NL80211_RATE_INFO_MCS,             "NL80211_RATE_INFO_MCS" },
    { WS_NL80211_RATE_INFO_40_MHZ_WIDTH,    "NL80211_RATE_INFO_40_MHZ_WIDTH" },
    { WS_NL80211_RATE_INFO_SHORT_GI,        "NL80211_RATE_INFO_SHORT_GI" },
    { WS_NL80211_RATE_INFO_BITRATE32,       "NL80211_RATE_INFO_BITRATE32" },
    { WS_NL80211_RATE_INFO_VHT_MCS,         "NL80211_RATE_INFO_VHT_MCS" },
    { WS_NL80211_RATE_INFO_VHT_NSS,         "NL80211_RATE_INFO_VHT_NSS" },
    { WS_NL80211_RATE_INFO_80_MHZ_WIDTH,    "NL80211_RATE_INFO_80_MHZ_WIDTH" },
    { WS_NL80211_RATE_INFO_80P80_MHZ_WIDTH, "NL80211_RATE_INFO_80P80_MHZ_WIDTH" },
    { WS_NL80211_RATE_INFO_160_MHZ_WIDTH,   "NL80211_RATE_INFO_160_MHZ_WIDTH" },
    { WS_NL80211_RATE_INFO_10_MHZ_WIDTH,    "NL80211_RATE_INFO_10_MHZ_WIDTH" },
    { WS_NL80211_RATE_INFO_5_MHZ_WIDTH,     "NL80211_RATE_INFO_5_MHZ_WIDTH" },
    { WS_NL80211_RATE_INFO_HE_MCS,          "NL80211_RATE_INFO_HE_MCS" },
    { WS_NL80211_RATE_INFO_HE_NSS,          "NL80211_RATE_INFO_HE_NSS" },
    { WS_NL80211_RATE_INFO_HE_GI,           "NL80211_RATE_INFO_HE_GI" },
    { WS_NL80211_RATE_INFO_HE_DCM,          "NL80211_RATE_INFO_HE_DCM" },
    { WS_NL80211_RATE_INFO_HE_RU_ALLOC,     "NL80211_RATE_INFO_HE_RU_ALLOC" },
    { 0, NULL }
};
static value_string_ext ws_nl80211_rate_info_vals_ext = VALUE_STRING_EXT_INIT(ws_nl80211_rate_info_vals);

static const value_string ws_nl80211_sta_bss_param_vals[] = {
    { WS___NL80211_STA_BSS_PARAM_INVALID,   "__NL80211_STA_BSS_PARAM_INVALID" },
    { WS_NL80211_STA_BSS_PARAM_CTS_PROT,    "NL80211_STA_BSS_PARAM_CTS_PROT" },
    { WS_NL80211_STA_BSS_PARAM_SHORT_PREAMBLE, "NL80211_STA_BSS_PARAM_SHORT_PREAMBLE" },
    { WS_NL80211_STA_BSS_PARAM_SHORT_SLOT_TIME, "NL80211_STA_BSS_PARAM_SHORT_SLOT_TIME" },
    { WS_NL80211_STA_BSS_PARAM_DTIM_PERIOD, "NL80211_STA_BSS_PARAM_DTIM_PERIOD" },
    { WS_NL80211_STA_BSS_PARAM_BEACON_INTERVAL, "NL80211_STA_BSS_PARAM_BEACON_INTERVAL" },
    { 0, NULL }
};
static value_string_ext ws_nl80211_sta_bss_param_vals_ext = VALUE_STRING_EXT_INIT(ws_nl80211_sta_bss_param_vals);

static const value_string ws_nl80211_sta_info_vals[] = {
    { WS___NL80211_STA_INFO_INVALID,        "__NL80211_STA_INFO_INVALID" },
    { WS_NL80211_STA_INFO_INACTIVE_TIME,    "NL80211_STA_INFO_INACTIVE_TIME" },
    { WS_NL80211_STA_INFO_RX_BYTES,         "NL80211_STA_INFO_RX_BYTES" },
    { WS_NL80211_STA_INFO_TX_BYTES,         "NL80211_STA_INFO_TX_BYTES" },
    { WS_NL80211_STA_INFO_LLID,             "NL80211_STA_INFO_LLID" },
    { WS_NL80211_STA_INFO_PLID,             "NL80211_STA_INFO_PLID" },
    { WS_NL80211_STA_INFO_PLINK_STATE,      "NL80211_STA_INFO_PLINK_STATE" },
    { WS_NL80211_STA_INFO_SIGNAL,           "NL80211_STA_INFO_SIGNAL" },
    { WS_NL80211_STA_INFO_TX_BITRATE,       "NL80211_STA_INFO_TX_BITRATE" },
    { WS_NL80211_STA_INFO_RX_PACKETS,       "NL80211_STA_INFO_RX_PACKETS" },
    { WS_NL80211_STA_INFO_TX_PACKETS,       "NL80211_STA_INFO_TX_PACKETS" },
    { WS_NL80211_STA_INFO_TX_RETRIES,       "NL80211_STA_INFO_TX_RETRIES" },
    { WS_NL80211_STA_INFO_TX_FAILED,        "NL80211_STA_INFO_TX_FAILED" },
    { WS_NL80211_STA_INFO_SIGNAL_AVG,       "NL80211_STA_INFO_SIGNAL_AVG" },
    { WS_NL80211_STA_INFO_RX_BITRATE,       "NL80211_STA_INFO_RX_BITRATE" },
    { WS_NL80211_STA_INFO_BSS_PARAM,        "NL80211_STA_INFO_BSS_PARAM" },
    { WS_NL80211_STA_INFO_CONNECTED_TIME,   "NL80211_STA_INFO_CONNECTED_TIME" },
    { WS_NL80211_STA_INFO_STA_FLAGS,        "NL80211_STA_INFO_STA_FLAGS" },
    { WS_NL80211_STA_INFO_BEACON_LOSS,      "NL80211_STA_INFO_BEACON_LOSS" },
    { WS_NL80211_STA_INFO_T_OFFSET,         "NL80211_STA_INFO_T_OFFSET" },
    { WS_NL80211_STA_INFO_LOCAL_PM,         "NL80211_STA_INFO_LOCAL_PM" },
    { WS_NL80211_STA_INFO_PEER_PM,          "NL80211_STA_INFO_PEER_PM" },
    { WS_NL80211_STA_INFO_NONPEER_PM,       "NL80211_STA_INFO_NONPEER_PM" },
    { WS_NL80211_STA_INFO_RX_BYTES64,       "NL80211_STA_INFO_RX_BYTES64" },
    { WS_NL80211_STA_INFO_TX_BYTES64,       "NL80211_STA_INFO_TX_BYTES64" },
    { WS_NL80211_STA_INFO_CHAIN_SIGNAL,     "NL80211_STA_INFO_CHAIN_SIGNAL" },
    { WS_NL80211_STA_INFO_CHAIN_SIGNAL_AVG, "NL80211_STA_INFO_CHAIN_SIGNAL_AVG" },
    { WS_NL80211_STA_INFO_EXPECTED_THROUGHPUT, "NL80211_STA_INFO_EXPECTED_THROUGHPUT" },
    { WS_NL80211_STA_INFO_RX_DROP_MISC,     "NL80211_STA_INFO_RX_DROP_MISC" },
    { WS_NL80211_STA_INFO_BEACON_RX,        "NL80211_STA_INFO_BEACON_RX" },
    { WS_NL80211_STA_INFO_BEACON_SIGNAL_AVG, "NL80211_STA_INFO_BEACON_SIGNAL_AVG" },
    { WS_NL80211_STA_INFO_TID_STATS,        "NL80211_STA_INFO_TID_STATS" },
    { WS_NL80211_STA_INFO_RX_DURATION,      "NL80211_STA_INFO_RX_DURATION" },
    { WS_NL80211_STA_INFO_PAD,              "NL80211_STA_INFO_PAD" },
    { WS_NL80211_STA_INFO_ACK_SIGNAL,       "NL80211_STA_INFO_ACK_SIGNAL" },
    { WS_NL80211_STA_INFO_ACK_SIGNAL_AVG,   "NL80211_STA_INFO_ACK_SIGNAL_AVG" },
    { WS_NL80211_STA_INFO_RX_MPDUS,         "NL80211_STA_INFO_RX_MPDUS" },
    { WS_NL80211_STA_INFO_FCS_ERROR_COUNT,  "NL80211_STA_INFO_FCS_ERROR_COUNT" },
    { 0, NULL }
};
static value_string_ext ws_nl80211_sta_info_vals_ext = VALUE_STRING_EXT_INIT(ws_nl80211_sta_info_vals);

static const value_string ws_nl80211_tid_stats_vals[] = {
    { WS___NL80211_TID_STATS_INVALID,       "__NL80211_TID_STATS_INVALID" },
    { WS_NL80211_TID_STATS_RX_MSDU,         "NL80211_TID_STATS_RX_MSDU" },
    { WS_NL80211_TID_STATS_TX_MSDU,         "NL80211_TID_STATS_TX_MSDU" },
    { WS_NL80211_TID_STATS_TX_MSDU_RETRIES, "NL80211_TID_STATS_TX_MSDU_RETRIES" },
    { WS_NL80211_TID_STATS_TX_MSDU_FAILED,  "NL80211_TID_STATS_TX_MSDU_FAILED" },
    { WS_NL80211_TID_STATS_PAD,             "NL80211_TID_STATS_PAD" },
    { WS_NL80211_TID_STATS_TXQ_STATS,       "NL80211_TID_STATS_TXQ_STATS" },
    { 0, NULL }
};
static value_string_ext ws_nl80211_tid_stats_vals_ext = VALUE_STRING_EXT_INIT(ws_nl80211_tid_stats_vals);

static const value_string ws_nl80211_txq_stats_vals[] = {
    { WS___NL80211_TXQ_STATS_INVALID,       "__NL80211_TXQ_STATS_INVALID" },
    { WS_NL80211_TXQ_STATS_BACKLOG_BYTES,   "NL80211_TXQ_STATS_BACKLOG_BYTES" },
    { WS_NL80211_TXQ_STATS_BACKLOG_PACKETS, "NL80211_TXQ_STATS_BACKLOG_PACKETS" },
    { WS_NL80211_TXQ_STATS_FLOWS,           "NL80211_TXQ_STATS_FLOWS" },
    { WS_NL80211_TXQ_STATS_DROPS,           "NL80211_TXQ_STATS_DROPS" },
    { WS_NL80211_TXQ_STATS_ECN_MARKS,       "NL80211_TXQ_STATS_ECN_MARKS" },
    { WS_NL80211_TXQ_STATS_OVERLIMIT,       "NL80211_TXQ_STATS_OVERLIMIT" },
    { WS_NL80211_TXQ_STATS_OVERMEMORY,      "NL80211_TXQ_STATS_OVERMEMORY" },
    { WS_NL80211_TXQ_STATS_COLLISIONS,      "NL80211_TXQ_STATS_COLLISIONS" },
    { WS_NL80211_TXQ_STATS_TX_BYTES,        "NL80211_TXQ_STATS_TX_BYTES" },
    { WS_NL80211_TXQ_STATS_TX_PACKETS,      "NL80211_TXQ_STATS_TX_PACKETS" },
    { WS_NL80211_TXQ_STATS_MAX_FLOWS,       "NL80211_TXQ_STATS_MAX_FLOWS" },
    { 0, NULL }
};
static value_string_ext ws_nl80211_txq_stats_vals_ext = VALUE_STRING_EXT_INIT(ws_nl80211_txq_stats_vals);

static const value_string ws_nl80211_mpath_info_vals[] = {
    { WS___NL80211_MPATH_INFO_INVALID,      "__NL80211_MPATH_INFO_INVALID" },
    { WS_NL80211_MPATH_INFO_FRAME_QLEN,     "NL80211_MPATH_INFO_FRAME_QLEN" },
    { WS_NL80211_MPATH_INFO_SN,             "NL80211_MPATH_INFO_SN" },
    { WS_NL80211_MPATH_INFO_METRIC,         "NL80211_MPATH_INFO_METRIC" },
    { WS_NL80211_MPATH_INFO_EXPTIME,        "NL80211_MPATH_INFO_EXPTIME" },
    { WS_NL80211_MPATH_INFO_FLAGS,          "NL80211_MPATH_INFO_FLAGS" },
    { WS_NL80211_MPATH_INFO_DISCOVERY_TIMEOUT, "NL80211_MPATH_INFO_DISCOVERY_TIMEOUT" },
    { WS_NL80211_MPATH_INFO_DISCOVERY_RETRIES, "NL80211_MPATH_INFO_DISCOVERY_RETRIES" },
    { 0, NULL }
};
static value_string_ext ws_nl80211_mpath_info_vals_ext = VALUE_STRING_EXT_INIT(ws_nl80211_mpath_info_vals);

static const value_string ws_nl80211_band_iftype_attr_vals[] = {
    { WS___NL80211_BAND_IFTYPE_ATTR_INVALID, "__NL80211_BAND_IFTYPE_ATTR_INVALID" },
    { WS_NL80211_BAND_IFTYPE_ATTR_IFTYPES,  "NL80211_BAND_IFTYPE_ATTR_IFTYPES" },
    { WS_NL80211_BAND_IFTYPE_ATTR_HE_CAP_MAC, "NL80211_BAND_IFTYPE_ATTR_HE_CAP_MAC" },
    { WS_NL80211_BAND_IFTYPE_ATTR_HE_CAP_PHY, "NL80211_BAND_IFTYPE_ATTR_HE_CAP_PHY" },
    { WS_NL80211_BAND_IFTYPE_ATTR_HE_CAP_MCS_SET, "NL80211_BAND_IFTYPE_ATTR_HE_CAP_MCS_SET" },
    { WS_NL80211_BAND_IFTYPE_ATTR_HE_CAP_PPE, "NL80211_BAND_IFTYPE_ATTR_HE_CAP_PPE" },
    { 0, NULL }
};
static value_string_ext ws_nl80211_band_iftype_attr_vals_ext = VALUE_STRING_EXT_INIT(ws_nl80211_band_iftype_attr_vals);

static const value_string ws_nl80211_band_attr_vals[] = {
    { WS___NL80211_BAND_ATTR_INVALID,       "__NL80211_BAND_ATTR_INVALID" },
    { WS_NL80211_BAND_ATTR_FREQS,           "NL80211_BAND_ATTR_FREQS" },
    { WS_NL80211_BAND_ATTR_RATES,           "NL80211_BAND_ATTR_RATES" },
    { WS_NL80211_BAND_ATTR_HT_MCS_SET,      "NL80211_BAND_ATTR_HT_MCS_SET" },
    { WS_NL80211_BAND_ATTR_HT_CAPA,         "NL80211_BAND_ATTR_HT_CAPA" },
    { WS_NL80211_BAND_ATTR_HT_AMPDU_FACTOR, "NL80211_BAND_ATTR_HT_AMPDU_FACTOR" },
    { WS_NL80211_BAND_ATTR_HT_AMPDU_DENSITY, "NL80211_BAND_ATTR_HT_AMPDU_DENSITY" },
    { WS_NL80211_BAND_ATTR_VHT_MCS_SET,     "NL80211_BAND_ATTR_VHT_MCS_SET" },
    { WS_NL80211_BAND_ATTR_VHT_CAPA,        "NL80211_BAND_ATTR_VHT_CAPA" },
    { WS_NL80211_BAND_ATTR_IFTYPE_DATA,     "NL80211_BAND_ATTR_IFTYPE_DATA" },
    { 0, NULL }
};
static value_string_ext ws_nl80211_band_attr_vals_ext = VALUE_STRING_EXT_INIT(ws_nl80211_band_attr_vals);

static const value_string ws_nl80211_wmm_rule_vals[] = {
    { WS___NL80211_WMMR_INVALID,            "__NL80211_WMMR_INVALID" },
    { WS_NL80211_WMMR_CW_MIN,               "NL80211_WMMR_CW_MIN" },
    { WS_NL80211_WMMR_CW_MAX,               "NL80211_WMMR_CW_MAX" },
    { WS_NL80211_WMMR_AIFSN,                "NL80211_WMMR_AIFSN" },
    { WS_NL80211_WMMR_TXOP,                 "NL80211_WMMR_TXOP" },
    { 0, NULL }
};
static value_string_ext ws_nl80211_wmm_rule_vals_ext = VALUE_STRING_EXT_INIT(ws_nl80211_wmm_rule_vals);

static const value_string ws_nl80211_frequency_attr_vals[] = {
    { WS___NL80211_FREQUENCY_ATTR_INVALID,  "__NL80211_FREQUENCY_ATTR_INVALID" },
    { WS_NL80211_FREQUENCY_ATTR_FREQ,       "NL80211_FREQUENCY_ATTR_FREQ" },
    { WS_NL80211_FREQUENCY_ATTR_DISABLED,   "NL80211_FREQUENCY_ATTR_DISABLED" },
    { WS_NL80211_FREQUENCY_ATTR_NO_IR,      "NL80211_FREQUENCY_ATTR_NO_IR" },
    { WS___NL80211_FREQUENCY_ATTR_NO_IBSS,  "__NL80211_FREQUENCY_ATTR_NO_IBSS" },
    { WS_NL80211_FREQUENCY_ATTR_RADAR,      "NL80211_FREQUENCY_ATTR_RADAR" },
    { WS_NL80211_FREQUENCY_ATTR_MAX_TX_POWER, "NL80211_FREQUENCY_ATTR_MAX_TX_POWER" },
    { WS_NL80211_FREQUENCY_ATTR_DFS_STATE,  "NL80211_FREQUENCY_ATTR_DFS_STATE" },
    { WS_NL80211_FREQUENCY_ATTR_DFS_TIME,   "NL80211_FREQUENCY_ATTR_DFS_TIME" },
    { WS_NL80211_FREQUENCY_ATTR_NO_HT40_MINUS, "NL80211_FREQUENCY_ATTR_NO_HT40_MINUS" },
    { WS_NL80211_FREQUENCY_ATTR_NO_HT40_PLUS, "NL80211_FREQUENCY_ATTR_NO_HT40_PLUS" },
    { WS_NL80211_FREQUENCY_ATTR_NO_80MHZ,   "NL80211_FREQUENCY_ATTR_NO_80MHZ" },
    { WS_NL80211_FREQUENCY_ATTR_NO_160MHZ,  "NL80211_FREQUENCY_ATTR_NO_160MHZ" },
    { WS_NL80211_FREQUENCY_ATTR_DFS_CAC_TIME, "NL80211_FREQUENCY_ATTR_DFS_CAC_TIME" },
    { WS_NL80211_FREQUENCY_ATTR_INDOOR_ONLY, "NL80211_FREQUENCY_ATTR_INDOOR_ONLY" },
    { WS_NL80211_FREQUENCY_ATTR_IR_CONCURRENT, "NL80211_FREQUENCY_ATTR_IR_CONCURRENT" },
    { WS_NL80211_FREQUENCY_ATTR_NO_20MHZ,   "NL80211_FREQUENCY_ATTR_NO_20MHZ" },
    { WS_NL80211_FREQUENCY_ATTR_NO_10MHZ,   "NL80211_FREQUENCY_ATTR_NO_10MHZ" },
    { WS_NL80211_FREQUENCY_ATTR_WMM,        "NL80211_FREQUENCY_ATTR_WMM" },
    { 0, NULL }
};
static value_string_ext ws_nl80211_frequency_attr_vals_ext = VALUE_STRING_EXT_INIT(ws_nl80211_frequency_attr_vals);

static const value_string ws_nl80211_bitrate_attr_vals[] = {
    { WS___NL80211_BITRATE_ATTR_INVALID,    "__NL80211_BITRATE_ATTR_INVALID" },
    { WS_NL80211_BITRATE_ATTR_RATE,         "NL80211_BITRATE_ATTR_RATE" },
    { WS_NL80211_BITRATE_ATTR_2GHZ_SHORTPREAMBLE, "NL80211_BITRATE_ATTR_2GHZ_SHORTPREAMBLE" },
    { 0, NULL }
};
static value_string_ext ws_nl80211_bitrate_attr_vals_ext = VALUE_STRING_EXT_INIT(ws_nl80211_bitrate_attr_vals);

static const value_string ws_nl80211_reg_initiator_vals[] = {
    { WS_NL80211_REGDOM_SET_BY_CORE,        "NL80211_REGDOM_SET_BY_CORE" },
    { WS_NL80211_REGDOM_SET_BY_USER,        "NL80211_REGDOM_SET_BY_USER" },
    { WS_NL80211_REGDOM_SET_BY_DRIVER,      "NL80211_REGDOM_SET_BY_DRIVER" },
    { WS_NL80211_REGDOM_SET_BY_COUNTRY_IE,  "NL80211_REGDOM_SET_BY_COUNTRY_IE" },
    { 0, NULL }
};
static value_string_ext ws_nl80211_reg_initiator_vals_ext = VALUE_STRING_EXT_INIT(ws_nl80211_reg_initiator_vals);

static const value_string ws_nl80211_reg_type_vals[] = {
    { WS_NL80211_REGDOM_TYPE_COUNTRY,       "NL80211_REGDOM_TYPE_COUNTRY" },
    { WS_NL80211_REGDOM_TYPE_WORLD,         "NL80211_REGDOM_TYPE_WORLD" },
    { WS_NL80211_REGDOM_TYPE_CUSTOM_WORLD,  "NL80211_REGDOM_TYPE_CUSTOM_WORLD" },
    { WS_NL80211_REGDOM_TYPE_INTERSECTION,  "NL80211_REGDOM_TYPE_INTERSECTION" },
    { 0, NULL }
};
static value_string_ext ws_nl80211_reg_type_vals_ext = VALUE_STRING_EXT_INIT(ws_nl80211_reg_type_vals);

static const value_string ws_nl80211_reg_rule_attr_vals[] = {
    { WS___NL80211_REG_RULE_ATTR_INVALID,   "__NL80211_REG_RULE_ATTR_INVALID" },
    { WS_NL80211_ATTR_REG_RULE_FLAGS,       "NL80211_ATTR_REG_RULE_FLAGS" },
    { WS_NL80211_ATTR_FREQ_RANGE_START,     "NL80211_ATTR_FREQ_RANGE_START" },
    { WS_NL80211_ATTR_FREQ_RANGE_END,       "NL80211_ATTR_FREQ_RANGE_END" },
    { WS_NL80211_ATTR_FREQ_RANGE_MAX_BW,    "NL80211_ATTR_FREQ_RANGE_MAX_BW" },
    { WS_NL80211_ATTR_POWER_RULE_MAX_ANT_GAIN, "NL80211_ATTR_POWER_RULE_MAX_ANT_GAIN" },
    { WS_NL80211_ATTR_POWER_RULE_MAX_EIRP,  "NL80211_ATTR_POWER_RULE_MAX_EIRP" },
    { WS_NL80211_ATTR_DFS_CAC_TIME,         "NL80211_ATTR_DFS_CAC_TIME" },
    { 0, NULL }
};
static value_string_ext ws_nl80211_reg_rule_attr_vals_ext = VALUE_STRING_EXT_INIT(ws_nl80211_reg_rule_attr_vals);

static const value_string ws_nl80211_sched_scan_match_attr_vals[] = {
    { WS___NL80211_SCHED_SCAN_MATCH_ATTR_INVALID, "__NL80211_SCHED_SCAN_MATCH_ATTR_INVALID" },
    { WS_NL80211_SCHED_SCAN_MATCH_ATTR_SSID, "NL80211_SCHED_SCAN_MATCH_ATTR_SSID" },
    { WS_NL80211_SCHED_SCAN_MATCH_ATTR_RSSI, "NL80211_SCHED_SCAN_MATCH_ATTR_RSSI" },
    { WS_NL80211_SCHED_SCAN_MATCH_ATTR_RELATIVE_RSSI, "NL80211_SCHED_SCAN_MATCH_ATTR_RELATIVE_RSSI" },
    { WS_NL80211_SCHED_SCAN_MATCH_ATTR_RSSI_ADJUST, "NL80211_SCHED_SCAN_MATCH_ATTR_RSSI_ADJUST" },
    { WS_NL80211_SCHED_SCAN_MATCH_ATTR_BSSID, "NL80211_SCHED_SCAN_MATCH_ATTR_BSSID" },
    { 0, NULL }
};
static value_string_ext ws_nl80211_sched_scan_match_attr_vals_ext = VALUE_STRING_EXT_INIT(ws_nl80211_sched_scan_match_attr_vals);

static const value_string ws_nl80211_dfs_regions_vals[] = {
    { WS_NL80211_DFS_UNSET,                 "NL80211_DFS_UNSET" },
    { WS_NL80211_DFS_FCC,                   "NL80211_DFS_FCC" },
    { WS_NL80211_DFS_ETSI,                  "NL80211_DFS_ETSI" },
    { WS_NL80211_DFS_JP,                    "NL80211_DFS_JP" },
    { 0, NULL }
};
static value_string_ext ws_nl80211_dfs_regions_vals_ext = VALUE_STRING_EXT_INIT(ws_nl80211_dfs_regions_vals);

static const value_string ws_nl80211_user_reg_hint_type_vals[] = {
    { WS_NL80211_USER_REG_HINT_USER,        "NL80211_USER_REG_HINT_USER" },
    { WS_NL80211_USER_REG_HINT_CELL_BASE,   "NL80211_USER_REG_HINT_CELL_BASE" },
    { WS_NL80211_USER_REG_HINT_INDOOR,      "NL80211_USER_REG_HINT_INDOOR" },
    { 0, NULL }
};
static value_string_ext ws_nl80211_user_reg_hint_type_vals_ext = VALUE_STRING_EXT_INIT(ws_nl80211_user_reg_hint_type_vals);

static const value_string ws_nl80211_survey_info_vals[] = {
    { WS___NL80211_SURVEY_INFO_INVALID,     "__NL80211_SURVEY_INFO_INVALID" },
    { WS_NL80211_SURVEY_INFO_FREQUENCY,     "NL80211_SURVEY_INFO_FREQUENCY" },
    { WS_NL80211_SURVEY_INFO_NOISE,         "NL80211_SURVEY_INFO_NOISE" },
    { WS_NL80211_SURVEY_INFO_IN_USE,        "NL80211_SURVEY_INFO_IN_USE" },
    { WS_NL80211_SURVEY_INFO_TIME,          "NL80211_SURVEY_INFO_TIME" },
    { WS_NL80211_SURVEY_INFO_TIME_BUSY,     "NL80211_SURVEY_INFO_TIME_BUSY" },
    { WS_NL80211_SURVEY_INFO_TIME_EXT_BUSY, "NL80211_SURVEY_INFO_TIME_EXT_BUSY" },
    { WS_NL80211_SURVEY_INFO_TIME_RX,       "NL80211_SURVEY_INFO_TIME_RX" },
    { WS_NL80211_SURVEY_INFO_TIME_TX,       "NL80211_SURVEY_INFO_TIME_TX" },
    { WS_NL80211_SURVEY_INFO_TIME_SCAN,     "NL80211_SURVEY_INFO_TIME_SCAN" },
    { WS_NL80211_SURVEY_INFO_PAD,           "NL80211_SURVEY_INFO_PAD" },
    { 0, NULL }
};
static value_string_ext ws_nl80211_survey_info_vals_ext = VALUE_STRING_EXT_INIT(ws_nl80211_survey_info_vals);

static const value_string ws_nl80211_mntr_flags_vals[] = {
    { WS___NL80211_MNTR_FLAG_INVALID,       "__NL80211_MNTR_FLAG_INVALID" },
    { WS_NL80211_MNTR_FLAG_FCSFAIL,         "NL80211_MNTR_FLAG_FCSFAIL" },
    { WS_NL80211_MNTR_FLAG_PLCPFAIL,        "NL80211_MNTR_FLAG_PLCPFAIL" },
    { WS_NL80211_MNTR_FLAG_CONTROL,         "NL80211_MNTR_FLAG_CONTROL" },
    { WS_NL80211_MNTR_FLAG_OTHER_BSS,       "NL80211_MNTR_FLAG_OTHER_BSS" },
    { WS_NL80211_MNTR_FLAG_COOK_FRAMES,     "NL80211_MNTR_FLAG_COOK_FRAMES" },
    { WS_NL80211_MNTR_FLAG_ACTIVE,          "NL80211_MNTR_FLAG_ACTIVE" },
    { 0, NULL }
};
static value_string_ext ws_nl80211_mntr_flags_vals_ext = VALUE_STRING_EXT_INIT(ws_nl80211_mntr_flags_vals);

static const value_string ws_nl80211_mesh_power_mode_vals[] = {
    { WS_NL80211_MESH_POWER_UNKNOWN,        "NL80211_MESH_POWER_UNKNOWN" },
    { WS_NL80211_MESH_POWER_ACTIVE,         "NL80211_MESH_POWER_ACTIVE" },
    { WS_NL80211_MESH_POWER_LIGHT_SLEEP,    "NL80211_MESH_POWER_LIGHT_SLEEP" },
    { WS_NL80211_MESH_POWER_DEEP_SLEEP,     "NL80211_MESH_POWER_DEEP_SLEEP" },
    { 0, NULL }
};
static value_string_ext ws_nl80211_mesh_power_mode_vals_ext = VALUE_STRING_EXT_INIT(ws_nl80211_mesh_power_mode_vals);

static const value_string ws_nl80211_meshconf_params_vals[] = {
    { WS___NL80211_MESHCONF_INVALID,        "__NL80211_MESHCONF_INVALID" },
    { WS_NL80211_MESHCONF_RETRY_TIMEOUT,    "NL80211_MESHCONF_RETRY_TIMEOUT" },
    { WS_NL80211_MESHCONF_CONFIRM_TIMEOUT,  "NL80211_MESHCONF_CONFIRM_TIMEOUT" },
    { WS_NL80211_MESHCONF_HOLDING_TIMEOUT,  "NL80211_MESHCONF_HOLDING_TIMEOUT" },
    { WS_NL80211_MESHCONF_MAX_PEER_LINKS,   "NL80211_MESHCONF_MAX_PEER_LINKS" },
    { WS_NL80211_MESHCONF_MAX_RETRIES,      "NL80211_MESHCONF_MAX_RETRIES" },
    { WS_NL80211_MESHCONF_TTL,              "NL80211_MESHCONF_TTL" },
    { WS_NL80211_MESHCONF_AUTO_OPEN_PLINKS, "NL80211_MESHCONF_AUTO_OPEN_PLINKS" },
    { WS_NL80211_MESHCONF_HWMP_MAX_PREQ_RETRIES, "NL80211_MESHCONF_HWMP_MAX_PREQ_RETRIES" },
    { WS_NL80211_MESHCONF_PATH_REFRESH_TIME, "NL80211_MESHCONF_PATH_REFRESH_TIME" },
    { WS_NL80211_MESHCONF_MIN_DISCOVERY_TIMEOUT, "NL80211_MESHCONF_MIN_DISCOVERY_TIMEOUT" },
    { WS_NL80211_MESHCONF_HWMP_ACTIVE_PATH_TIMEOUT, "NL80211_MESHCONF_HWMP_ACTIVE_PATH_TIMEOUT" },
    { WS_NL80211_MESHCONF_HWMP_PREQ_MIN_INTERVAL, "NL80211_MESHCONF_HWMP_PREQ_MIN_INTERVAL" },
    { WS_NL80211_MESHCONF_HWMP_NET_DIAM_TRVS_TIME, "NL80211_MESHCONF_HWMP_NET_DIAM_TRVS_TIME" },
    { WS_NL80211_MESHCONF_HWMP_ROOTMODE,    "NL80211_MESHCONF_HWMP_ROOTMODE" },
    { WS_NL80211_MESHCONF_ELEMENT_TTL,      "NL80211_MESHCONF_ELEMENT_TTL" },
    { WS_NL80211_MESHCONF_HWMP_RANN_INTERVAL, "NL80211_MESHCONF_HWMP_RANN_INTERVAL" },
    { WS_NL80211_MESHCONF_GATE_ANNOUNCEMENTS, "NL80211_MESHCONF_GATE_ANNOUNCEMENTS" },
    { WS_NL80211_MESHCONF_HWMP_PERR_MIN_INTERVAL, "NL80211_MESHCONF_HWMP_PERR_MIN_INTERVAL" },
    { WS_NL80211_MESHCONF_FORWARDING,       "NL80211_MESHCONF_FORWARDING" },
    { WS_NL80211_MESHCONF_RSSI_THRESHOLD,   "NL80211_MESHCONF_RSSI_THRESHOLD" },
    { WS_NL80211_MESHCONF_SYNC_OFFSET_MAX_NEIGHBOR, "NL80211_MESHCONF_SYNC_OFFSET_MAX_NEIGHBOR" },
    { WS_NL80211_MESHCONF_HT_OPMODE,        "NL80211_MESHCONF_HT_OPMODE" },
    { WS_NL80211_MESHCONF_HWMP_PATH_TO_ROOT_TIMEOUT, "NL80211_MESHCONF_HWMP_PATH_TO_ROOT_TIMEOUT" },
    { WS_NL80211_MESHCONF_HWMP_ROOT_INTERVAL, "NL80211_MESHCONF_HWMP_ROOT_INTERVAL" },
    { WS_NL80211_MESHCONF_HWMP_CONFIRMATION_INTERVAL, "NL80211_MESHCONF_HWMP_CONFIRMATION_INTERVAL" },
    { WS_NL80211_MESHCONF_POWER_MODE,       "NL80211_MESHCONF_POWER_MODE" },
    { WS_NL80211_MESHCONF_AWAKE_WINDOW,     "NL80211_MESHCONF_AWAKE_WINDOW" },
    { WS_NL80211_MESHCONF_PLINK_TIMEOUT,    "NL80211_MESHCONF_PLINK_TIMEOUT" },
    { 0, NULL }
};
static value_string_ext ws_nl80211_meshconf_params_vals_ext = VALUE_STRING_EXT_INIT(ws_nl80211_meshconf_params_vals);

static const value_string ws_nl80211_mesh_setup_params_vals[] = {
    { WS___NL80211_MESH_SETUP_INVALID,      "__NL80211_MESH_SETUP_INVALID" },
    { WS_NL80211_MESH_SETUP_ENABLE_VENDOR_PATH_SEL, "NL80211_MESH_SETUP_ENABLE_VENDOR_PATH_SEL" },
    { WS_NL80211_MESH_SETUP_ENABLE_VENDOR_METRIC, "NL80211_MESH_SETUP_ENABLE_VENDOR_METRIC" },
    { WS_NL80211_MESH_SETUP_IE,             "NL80211_MESH_SETUP_IE" },
    { WS_NL80211_MESH_SETUP_USERSPACE_AUTH, "NL80211_MESH_SETUP_USERSPACE_AUTH" },
    { WS_NL80211_MESH_SETUP_USERSPACE_AMPE, "NL80211_MESH_SETUP_USERSPACE_AMPE" },
    { WS_NL80211_MESH_SETUP_ENABLE_VENDOR_SYNC, "NL80211_MESH_SETUP_ENABLE_VENDOR_SYNC" },
    { WS_NL80211_MESH_SETUP_USERSPACE_MPM,  "NL80211_MESH_SETUP_USERSPACE_MPM" },
    { WS_NL80211_MESH_SETUP_AUTH_PROTOCOL,  "NL80211_MESH_SETUP_AUTH_PROTOCOL" },
    { 0, NULL }
};
static value_string_ext ws_nl80211_mesh_setup_params_vals_ext = VALUE_STRING_EXT_INIT(ws_nl80211_mesh_setup_params_vals);

static const value_string ws_nl80211_txq_attr_vals[] = {
    { WS___NL80211_TXQ_ATTR_INVALID,        "__NL80211_TXQ_ATTR_INVALID" },
    { WS_NL80211_TXQ_ATTR_AC,               "NL80211_TXQ_ATTR_AC" },
    { WS_NL80211_TXQ_ATTR_TXOP,             "NL80211_TXQ_ATTR_TXOP" },
    { WS_NL80211_TXQ_ATTR_CWMIN,            "NL80211_TXQ_ATTR_CWMIN" },
    { WS_NL80211_TXQ_ATTR_CWMAX,            "NL80211_TXQ_ATTR_CWMAX" },
    { WS_NL80211_TXQ_ATTR_AIFS,             "NL80211_TXQ_ATTR_AIFS" },
    { 0, NULL }
};
static value_string_ext ws_nl80211_txq_attr_vals_ext = VALUE_STRING_EXT_INIT(ws_nl80211_txq_attr_vals);

static const value_string ws_nl80211_channel_type_vals[] = {
    { WS_NL80211_CHAN_NO_HT,                "NL80211_CHAN_NO_HT" },
    { WS_NL80211_CHAN_HT20,                 "NL80211_CHAN_HT20" },
    { WS_NL80211_CHAN_HT40MINUS,            "NL80211_CHAN_HT40MINUS" },
    { WS_NL80211_CHAN_HT40PLUS,             "NL80211_CHAN_HT40PLUS" },
    { 0, NULL }
};
static value_string_ext ws_nl80211_channel_type_vals_ext = VALUE_STRING_EXT_INIT(ws_nl80211_channel_type_vals);

static const value_string ws_nl80211_chan_width_vals[] = {
    { WS_NL80211_CHAN_WIDTH_20_NOHT,        "NL80211_CHAN_WIDTH_20_NOHT" },
    { WS_NL80211_CHAN_WIDTH_20,             "NL80211_CHAN_WIDTH_20" },
    { WS_NL80211_CHAN_WIDTH_40,             "NL80211_CHAN_WIDTH_40" },
    { WS_NL80211_CHAN_WIDTH_80,             "NL80211_CHAN_WIDTH_80" },
    { WS_NL80211_CHAN_WIDTH_80P80,          "NL80211_CHAN_WIDTH_80P80" },
    { WS_NL80211_CHAN_WIDTH_160,            "NL80211_CHAN_WIDTH_160" },
    { WS_NL80211_CHAN_WIDTH_5,              "NL80211_CHAN_WIDTH_5" },
    { WS_NL80211_CHAN_WIDTH_10,             "NL80211_CHAN_WIDTH_10" },
    { 0, NULL }
};
static value_string_ext ws_nl80211_chan_width_vals_ext = VALUE_STRING_EXT_INIT(ws_nl80211_chan_width_vals);

static const value_string ws_nl80211_bss_scan_width_vals[] = {
    { WS_NL80211_BSS_CHAN_WIDTH_20,         "NL80211_BSS_CHAN_WIDTH_20" },
    { WS_NL80211_BSS_CHAN_WIDTH_10,         "NL80211_BSS_CHAN_WIDTH_10" },
    { WS_NL80211_BSS_CHAN_WIDTH_5,          "NL80211_BSS_CHAN_WIDTH_5" },
    { 0, NULL }
};
static value_string_ext ws_nl80211_bss_scan_width_vals_ext = VALUE_STRING_EXT_INIT(ws_nl80211_bss_scan_width_vals);

static const value_string ws_nl80211_bss_vals[] = {
    { WS___NL80211_BSS_INVALID,             "__NL80211_BSS_INVALID" },
    { WS_NL80211_BSS_BSSID,                 "NL80211_BSS_BSSID" },
    { WS_NL80211_BSS_FREQUENCY,             "NL80211_BSS_FREQUENCY" },
    { WS_NL80211_BSS_TSF,                   "NL80211_BSS_TSF" },
    { WS_NL80211_BSS_BEACON_INTERVAL,       "NL80211_BSS_BEACON_INTERVAL" },
    { WS_NL80211_BSS_CAPABILITY,            "NL80211_BSS_CAPABILITY" },
    { WS_NL80211_BSS_INFORMATION_ELEMENTS,  "NL80211_BSS_INFORMATION_ELEMENTS" },
    { WS_NL80211_BSS_SIGNAL_MBM,            "NL80211_BSS_SIGNAL_MBM" },
    { WS_NL80211_BSS_SIGNAL_UNSPEC,         "NL80211_BSS_SIGNAL_UNSPEC" },
    { WS_NL80211_BSS_STATUS,                "NL80211_BSS_STATUS" },
    { WS_NL80211_BSS_SEEN_MS_AGO,           "NL80211_BSS_SEEN_MS_AGO" },
    { WS_NL80211_BSS_BEACON_IES,            "NL80211_BSS_BEACON_IES" },
    { WS_NL80211_BSS_CHAN_WIDTH,            "NL80211_BSS_CHAN_WIDTH" },
    { WS_NL80211_BSS_BEACON_TSF,            "NL80211_BSS_BEACON_TSF" },
    { WS_NL80211_BSS_PRESP_DATA,            "NL80211_BSS_PRESP_DATA" },
    { WS_NL80211_BSS_LAST_SEEN_BOOTTIME,    "NL80211_BSS_LAST_SEEN_BOOTTIME" },
    { WS_NL80211_BSS_PAD,                   "NL80211_BSS_PAD" },
    { WS_NL80211_BSS_PARENT_TSF,            "NL80211_BSS_PARENT_TSF" },
    { WS_NL80211_BSS_PARENT_BSSID,          "NL80211_BSS_PARENT_BSSID" },
    { WS_NL80211_BSS_CHAIN_SIGNAL,          "NL80211_BSS_CHAIN_SIGNAL" },
    { 0, NULL }
};
static value_string_ext ws_nl80211_bss_vals_ext = VALUE_STRING_EXT_INIT(ws_nl80211_bss_vals);

static const value_string ws_nl80211_bss_status_vals[] = {
    { WS_NL80211_BSS_STATUS_AUTHENTICATED,  "NL80211_BSS_STATUS_AUTHENTICATED" },
    { WS_NL80211_BSS_STATUS_ASSOCIATED,     "NL80211_BSS_STATUS_ASSOCIATED" },
    { WS_NL80211_BSS_STATUS_IBSS_JOINED,    "NL80211_BSS_STATUS_IBSS_JOINED" },
    { 0, NULL }
};
static value_string_ext ws_nl80211_bss_status_vals_ext = VALUE_STRING_EXT_INIT(ws_nl80211_bss_status_vals);

static const value_string ws_nl80211_auth_type_vals[] = {
    { WS_NL80211_AUTHTYPE_OPEN_SYSTEM,      "NL80211_AUTHTYPE_OPEN_SYSTEM" },
    { WS_NL80211_AUTHTYPE_SHARED_KEY,       "NL80211_AUTHTYPE_SHARED_KEY" },
    { WS_NL80211_AUTHTYPE_FT,               "NL80211_AUTHTYPE_FT" },
    { WS_NL80211_AUTHTYPE_NETWORK_EAP,      "NL80211_AUTHTYPE_NETWORK_EAP" },
    { WS_NL80211_AUTHTYPE_SAE,              "NL80211_AUTHTYPE_SAE" },
    { WS_NL80211_AUTHTYPE_FILS_SK,          "NL80211_AUTHTYPE_FILS_SK" },
    { WS_NL80211_AUTHTYPE_FILS_SK_PFS,      "NL80211_AUTHTYPE_FILS_SK_PFS" },
    { WS_NL80211_AUTHTYPE_FILS_PK,          "NL80211_AUTHTYPE_FILS_PK" },
    { 0, NULL }
};
static value_string_ext ws_nl80211_auth_type_vals_ext = VALUE_STRING_EXT_INIT(ws_nl80211_auth_type_vals);

static const value_string ws_nl80211_key_type_vals[] = {
    { WS_NL80211_KEYTYPE_GROUP,             "NL80211_KEYTYPE_GROUP" },
    { WS_NL80211_KEYTYPE_PAIRWISE,          "NL80211_KEYTYPE_PAIRWISE" },
    { WS_NL80211_KEYTYPE_PEERKEY,           "NL80211_KEYTYPE_PEERKEY" },
    { 0, NULL }
};
static value_string_ext ws_nl80211_key_type_vals_ext = VALUE_STRING_EXT_INIT(ws_nl80211_key_type_vals);

static const value_string ws_nl80211_mfp_vals[] = {
    { WS_NL80211_MFP_NO,                    "NL80211_MFP_NO" },
    { WS_NL80211_MFP_REQUIRED,              "NL80211_MFP_REQUIRED" },
    { WS_NL80211_MFP_OPTIONAL,              "NL80211_MFP_OPTIONAL" },
    { 0, NULL }
};
static value_string_ext ws_nl80211_mfp_vals_ext = VALUE_STRING_EXT_INIT(ws_nl80211_mfp_vals);

static const value_string ws_nl80211_key_default_types_vals[] = {
    { WS___NL80211_KEY_DEFAULT_TYPE_INVALID, "__NL80211_KEY_DEFAULT_TYPE_INVALID" },
    { WS_NL80211_KEY_DEFAULT_TYPE_UNICAST,  "NL80211_KEY_DEFAULT_TYPE_UNICAST" },
    { WS_NL80211_KEY_DEFAULT_TYPE_MULTICAST, "NL80211_KEY_DEFAULT_TYPE_MULTICAST" },
    { 0, NULL }
};
static value_string_ext ws_nl80211_key_default_types_vals_ext = VALUE_STRING_EXT_INIT(ws_nl80211_key_default_types_vals);

static const value_string ws_nl80211_key_attributes_vals[] = {
    { WS___NL80211_KEY_INVALID,             "__NL80211_KEY_INVALID" },
    { WS_NL80211_KEY_DATA,                  "NL80211_KEY_DATA" },
    { WS_NL80211_KEY_IDX,                   "NL80211_KEY_IDX" },
    { WS_NL80211_KEY_CIPHER,                "NL80211_KEY_CIPHER" },
    { WS_NL80211_KEY_SEQ,                   "NL80211_KEY_SEQ" },
    { WS_NL80211_KEY_DEFAULT,               "NL80211_KEY_DEFAULT" },
    { WS_NL80211_KEY_DEFAULT_MGMT,          "NL80211_KEY_DEFAULT_MGMT" },
    { WS_NL80211_KEY_TYPE,                  "NL80211_KEY_TYPE" },
    { WS_NL80211_KEY_DEFAULT_TYPES,         "NL80211_KEY_DEFAULT_TYPES" },
    { 0, NULL }
};
static value_string_ext ws_nl80211_key_attributes_vals_ext = VALUE_STRING_EXT_INIT(ws_nl80211_key_attributes_vals);

static const value_string ws_nl80211_tx_rate_attributes_vals[] = {
    { WS___NL80211_TXRATE_INVALID,          "__NL80211_TXRATE_INVALID" },
    { WS_NL80211_TXRATE_LEGACY,             "NL80211_TXRATE_LEGACY" },
    { WS_NL80211_TXRATE_HT,                 "NL80211_TXRATE_HT" },
    { WS_NL80211_TXRATE_VHT,                "NL80211_TXRATE_VHT" },
    { WS_NL80211_TXRATE_GI,                 "NL80211_TXRATE_GI" },
    { 0, NULL }
};
static value_string_ext ws_nl80211_tx_rate_attributes_vals_ext = VALUE_STRING_EXT_INIT(ws_nl80211_tx_rate_attributes_vals);

static const value_string ws_nl80211_ps_state_vals[] = {
    { WS_NL80211_PS_DISABLED,               "NL80211_PS_DISABLED" },
    { WS_NL80211_PS_ENABLED,                "NL80211_PS_ENABLED" },
    { 0, NULL }
};
static value_string_ext ws_nl80211_ps_state_vals_ext = VALUE_STRING_EXT_INIT(ws_nl80211_ps_state_vals);

static const value_string ws_nl80211_attr_cqm_vals[] = {
    { WS___NL80211_ATTR_CQM_INVALID,        "__NL80211_ATTR_CQM_INVALID" },
    { WS_NL80211_ATTR_CQM_RSSI_THOLD,       "NL80211_ATTR_CQM_RSSI_THOLD" },
    { WS_NL80211_ATTR_CQM_RSSI_HYST,        "NL80211_ATTR_CQM_RSSI_HYST" },
    { WS_NL80211_ATTR_CQM_RSSI_THRESHOLD_EVENT, "NL80211_ATTR_CQM_RSSI_THRESHOLD_EVENT" },
    { WS_NL80211_ATTR_CQM_PKT_LOSS_EVENT,   "NL80211_ATTR_CQM_PKT_LOSS_EVENT" },
    { WS_NL80211_ATTR_CQM_TXE_RATE,         "NL80211_ATTR_CQM_TXE_RATE" },
    { WS_NL80211_ATTR_CQM_TXE_PKTS,         "NL80211_ATTR_CQM_TXE_PKTS" },
    { WS_NL80211_ATTR_CQM_TXE_INTVL,        "NL80211_ATTR_CQM_TXE_INTVL" },
    { WS_NL80211_ATTR_CQM_BEACON_LOSS_EVENT, "NL80211_ATTR_CQM_BEACON_LOSS_EVENT" },
    { WS_NL80211_ATTR_CQM_RSSI_LEVEL,       "NL80211_ATTR_CQM_RSSI_LEVEL" },
    { 0, NULL }
};
static value_string_ext ws_nl80211_attr_cqm_vals_ext = VALUE_STRING_EXT_INIT(ws_nl80211_attr_cqm_vals);

static const value_string ws_nl80211_tx_power_setting_vals[] = {
    { WS_NL80211_TX_POWER_AUTOMATIC,        "NL80211_TX_POWER_AUTOMATIC" },
    { WS_NL80211_TX_POWER_LIMITED,          "NL80211_TX_POWER_LIMITED" },
    { WS_NL80211_TX_POWER_FIXED,            "NL80211_TX_POWER_FIXED" },
    { 0, NULL }
};
static value_string_ext ws_nl80211_tx_power_setting_vals_ext = VALUE_STRING_EXT_INIT(ws_nl80211_tx_power_setting_vals);

static const value_string ws_nl80211_if_combination_attrs_vals[] = {
    { WS_NL80211_IFACE_COMB_UNSPEC,         "NL80211_IFACE_COMB_UNSPEC" },
    { WS_NL80211_IFACE_COMB_LIMITS,         "NL80211_IFACE_COMB_LIMITS" },
    { WS_NL80211_IFACE_COMB_MAXNUM,         "NL80211_IFACE_COMB_MAXNUM" },
    { WS_NL80211_IFACE_COMB_STA_AP_BI_MATCH, "NL80211_IFACE_COMB_STA_AP_BI_MATCH" },
    { WS_NL80211_IFACE_COMB_NUM_CHANNELS,   "NL80211_IFACE_COMB_NUM_CHANNELS" },
    { WS_NL80211_IFACE_COMB_RADAR_DETECT_WIDTHS, "NL80211_IFACE_COMB_RADAR_DETECT_WIDTHS" },
    { WS_NL80211_IFACE_COMB_RADAR_DETECT_REGIONS, "NL80211_IFACE_COMB_RADAR_DETECT_REGIONS" },
    { WS_NL80211_IFACE_COMB_BI_MIN_GCD,     "NL80211_IFACE_COMB_BI_MIN_GCD" },
    { 0, NULL }
};
static value_string_ext ws_nl80211_if_combination_attrs_vals_ext = VALUE_STRING_EXT_INIT(ws_nl80211_if_combination_attrs_vals);

static const value_string ws_nl80211_plink_state_vals[] = {
    { WS_NL80211_PLINK_LISTEN,              "NL80211_PLINK_LISTEN" },
    { WS_NL80211_PLINK_OPN_SNT,             "NL80211_PLINK_OPN_SNT" },
    { WS_NL80211_PLINK_OPN_RCVD,            "NL80211_PLINK_OPN_RCVD" },
    { WS_NL80211_PLINK_CNF_RCVD,            "NL80211_PLINK_CNF_RCVD" },
    { WS_NL80211_PLINK_ESTAB,               "NL80211_PLINK_ESTAB" },
    { WS_NL80211_PLINK_HOLDING,             "NL80211_PLINK_HOLDING" },
    { WS_NL80211_PLINK_BLOCKED,             "NL80211_PLINK_BLOCKED" },
    { 0, NULL }
};
static value_string_ext ws_nl80211_plink_state_vals_ext = VALUE_STRING_EXT_INIT(ws_nl80211_plink_state_vals);

static const value_string ws_plink_actions_vals[] = {
    { WS_NL80211_PLINK_ACTION_NO_ACTION,    "NL80211_PLINK_ACTION_NO_ACTION" },
    { WS_NL80211_PLINK_ACTION_OPEN,         "NL80211_PLINK_ACTION_OPEN" },
    { WS_NL80211_PLINK_ACTION_BLOCK,        "NL80211_PLINK_ACTION_BLOCK" },
    { 0, NULL }
};
static value_string_ext ws_plink_actions_vals_ext = VALUE_STRING_EXT_INIT(ws_plink_actions_vals);

static const value_string ws_nl80211_rekey_data_vals[] = {
    { WS___NL80211_REKEY_DATA_INVALID,      "__NL80211_REKEY_DATA_INVALID" },
    { WS_NL80211_REKEY_DATA_KEK,            "NL80211_REKEY_DATA_KEK" },
    { WS_NL80211_REKEY_DATA_KCK,            "NL80211_REKEY_DATA_KCK" },
    { WS_NL80211_REKEY_DATA_REPLAY_CTR,     "NL80211_REKEY_DATA_REPLAY_CTR" },
    { 0, NULL }
};
static value_string_ext ws_nl80211_rekey_data_vals_ext = VALUE_STRING_EXT_INIT(ws_nl80211_rekey_data_vals);

static const value_string ws_nl80211_sta_wme_attr_vals[] = {
    { WS___NL80211_STA_WME_INVALID,         "__NL80211_STA_WME_INVALID" },
    { WS_NL80211_STA_WME_UAPSD_QUEUES,      "NL80211_STA_WME_UAPSD_QUEUES" },
    { WS_NL80211_STA_WME_MAX_SP,            "NL80211_STA_WME_MAX_SP" },
    { 0, NULL }
};
static value_string_ext ws_nl80211_sta_wme_attr_vals_ext = VALUE_STRING_EXT_INIT(ws_nl80211_sta_wme_attr_vals);

static const value_string ws_nl80211_pmksa_candidate_attr_vals[] = {
    { WS___NL80211_PMKSA_CANDIDATE_INVALID, "__NL80211_PMKSA_CANDIDATE_INVALID" },
    { WS_NL80211_PMKSA_CANDIDATE_INDEX,     "NL80211_PMKSA_CANDIDATE_INDEX" },
    { WS_NL80211_PMKSA_CANDIDATE_BSSID,     "NL80211_PMKSA_CANDIDATE_BSSID" },
    { WS_NL80211_PMKSA_CANDIDATE_PREAUTH,   "NL80211_PMKSA_CANDIDATE_PREAUTH" },
    { 0, NULL }
};
static value_string_ext ws_nl80211_pmksa_candidate_attr_vals_ext = VALUE_STRING_EXT_INIT(ws_nl80211_pmksa_candidate_attr_vals);

static const value_string ws_nl80211_tdls_operation_vals[] = {
    { WS_NL80211_TDLS_DISCOVERY_REQ,        "NL80211_TDLS_DISCOVERY_REQ" },
    { WS_NL80211_TDLS_SETUP,                "NL80211_TDLS_SETUP" },
    { WS_NL80211_TDLS_TEARDOWN,             "NL80211_TDLS_TEARDOWN" },
    { WS_NL80211_TDLS_ENABLE_LINK,          "NL80211_TDLS_ENABLE_LINK" },
    { WS_NL80211_TDLS_DISABLE_LINK,         "NL80211_TDLS_DISABLE_LINK" },
    { 0, NULL }
};
static value_string_ext ws_nl80211_tdls_operation_vals_ext = VALUE_STRING_EXT_INIT(ws_nl80211_tdls_operation_vals);

static const value_string ws_nl80211_connect_failed_reason_vals[] = {
    { WS_NL80211_CONN_FAIL_MAX_CLIENTS,     "NL80211_CONN_FAIL_MAX_CLIENTS" },
    { WS_NL80211_CONN_FAIL_BLOCKED_CLIENT,  "NL80211_CONN_FAIL_BLOCKED_CLIENT" },
    { 0, NULL }
};
static value_string_ext ws_nl80211_connect_failed_reason_vals_ext = VALUE_STRING_EXT_INIT(ws_nl80211_connect_failed_reason_vals);

static const value_string ws_nl80211_timeout_reason_vals[] = {
    { WS_NL80211_TIMEOUT_UNSPECIFIED,       "NL80211_TIMEOUT_UNSPECIFIED" },
    { WS_NL80211_TIMEOUT_SCAN,              "NL80211_TIMEOUT_SCAN" },
    { WS_NL80211_TIMEOUT_AUTH,              "NL80211_TIMEOUT_AUTH" },
    { WS_NL80211_TIMEOUT_ASSOC,             "NL80211_TIMEOUT_ASSOC" },
    { 0, NULL }
};
static value_string_ext ws_nl80211_timeout_reason_vals_ext = VALUE_STRING_EXT_INIT(ws_nl80211_timeout_reason_vals);

static const value_string ws_nl80211_acl_policy_vals[] = {
    { WS_NL80211_ACL_POLICY_ACCEPT_UNLESS_LISTED, "NL80211_ACL_POLICY_ACCEPT_UNLESS_LISTED" },
    { WS_NL80211_ACL_POLICY_DENY_UNLESS_LISTED, "NL80211_ACL_POLICY_DENY_UNLESS_LISTED" },
    { 0, NULL }
};
static value_string_ext ws_nl80211_acl_policy_vals_ext = VALUE_STRING_EXT_INIT(ws_nl80211_acl_policy_vals);

static const value_string ws_nl80211_smps_mode_vals[] = {
    { WS_NL80211_SMPS_OFF,                  "NL80211_SMPS_OFF" },
    { WS_NL80211_SMPS_STATIC,               "NL80211_SMPS_STATIC" },
    { WS_NL80211_SMPS_DYNAMIC,              "NL80211_SMPS_DYNAMIC" },
    { 0, NULL }
};
static value_string_ext ws_nl80211_smps_mode_vals_ext = VALUE_STRING_EXT_INIT(ws_nl80211_smps_mode_vals);

static const value_string ws_nl80211_radar_event_vals[] = {
    { WS_NL80211_RADAR_DETECTED,            "NL80211_RADAR_DETECTED" },
    { WS_NL80211_RADAR_CAC_FINISHED,        "NL80211_RADAR_CAC_FINISHED" },
    { WS_NL80211_RADAR_CAC_ABORTED,         "NL80211_RADAR_CAC_ABORTED" },
    { WS_NL80211_RADAR_NOP_FINISHED,        "NL80211_RADAR_NOP_FINISHED" },
    { WS_NL80211_RADAR_PRE_CAC_EXPIRED,     "NL80211_RADAR_PRE_CAC_EXPIRED" },
    { WS_NL80211_RADAR_CAC_STARTED,         "NL80211_RADAR_CAC_STARTED" },
    { 0, NULL }
};
static value_string_ext ws_nl80211_radar_event_vals_ext = VALUE_STRING_EXT_INIT(ws_nl80211_radar_event_vals);

static const value_string ws_nl80211_dfs_state_vals[] = {
    { WS_NL80211_DFS_USABLE,                "NL80211_DFS_USABLE" },
    { WS_NL80211_DFS_UNAVAILABLE,           "NL80211_DFS_UNAVAILABLE" },
    { WS_NL80211_DFS_AVAILABLE,             "NL80211_DFS_AVAILABLE" },
    { 0, NULL }
};
static value_string_ext ws_nl80211_dfs_state_vals_ext = VALUE_STRING_EXT_INIT(ws_nl80211_dfs_state_vals);

static const value_string ws_nl80211_crit_proto_id_vals[] = {
    { WS_NL80211_CRIT_PROTO_UNSPEC,         "NL80211_CRIT_PROTO_UNSPEC" },
    { WS_NL80211_CRIT_PROTO_DHCP,           "NL80211_CRIT_PROTO_DHCP" },
    { WS_NL80211_CRIT_PROTO_EAPOL,          "NL80211_CRIT_PROTO_EAPOL" },
    { WS_NL80211_CRIT_PROTO_APIPA,          "NL80211_CRIT_PROTO_APIPA" },
    { 0, NULL }
};
static value_string_ext ws_nl80211_crit_proto_id_vals_ext = VALUE_STRING_EXT_INIT(ws_nl80211_crit_proto_id_vals);

static const value_string ws_nl80211_sched_scan_plan_vals[] = {
    { WS___NL80211_SCHED_SCAN_PLAN_INVALID, "__NL80211_SCHED_SCAN_PLAN_INVALID" },
    { WS_NL80211_SCHED_SCAN_PLAN_INTERVAL,  "NL80211_SCHED_SCAN_PLAN_INTERVAL" },
    { WS_NL80211_SCHED_SCAN_PLAN_ITERATIONS, "NL80211_SCHED_SCAN_PLAN_ITERATIONS" },
    { 0, NULL }
};
static value_string_ext ws_nl80211_sched_scan_plan_vals_ext = VALUE_STRING_EXT_INIT(ws_nl80211_sched_scan_plan_vals);

static const value_string ws_nl80211_bss_select_attr_vals[] = {
    { WS___NL80211_BSS_SELECT_ATTR_INVALID, "__NL80211_BSS_SELECT_ATTR_INVALID" },
    { WS_NL80211_BSS_SELECT_ATTR_RSSI,      "NL80211_BSS_SELECT_ATTR_RSSI" },
    { WS_NL80211_BSS_SELECT_ATTR_BAND_PREF, "NL80211_BSS_SELECT_ATTR_BAND_PREF" },
    { WS_NL80211_BSS_SELECT_ATTR_RSSI_ADJUST, "NL80211_BSS_SELECT_ATTR_RSSI_ADJUST" },
    { 0, NULL }
};
static value_string_ext ws_nl80211_bss_select_attr_vals_ext = VALUE_STRING_EXT_INIT(ws_nl80211_bss_select_attr_vals);

static const value_string ws_nl80211_nan_func_attributes_vals[] = {
    { WS___NL80211_NAN_FUNC_INVALID,        "__NL80211_NAN_FUNC_INVALID" },
    { WS_NL80211_NAN_FUNC_TYPE,             "NL80211_NAN_FUNC_TYPE" },
    { WS_NL80211_NAN_FUNC_SERVICE_ID,       "NL80211_NAN_FUNC_SERVICE_ID" },
    { WS_NL80211_NAN_FUNC_PUBLISH_TYPE,     "NL80211_NAN_FUNC_PUBLISH_TYPE" },
    { WS_NL80211_NAN_FUNC_PUBLISH_BCAST,    "NL80211_NAN_FUNC_PUBLISH_BCAST" },
    { WS_NL80211_NAN_FUNC_SUBSCRIBE_ACTIVE, "NL80211_NAN_FUNC_SUBSCRIBE_ACTIVE" },
    { WS_NL80211_NAN_FUNC_FOLLOW_UP_ID,     "NL80211_NAN_FUNC_FOLLOW_UP_ID" },
    { WS_NL80211_NAN_FUNC_FOLLOW_UP_REQ_ID, "NL80211_NAN_FUNC_FOLLOW_UP_REQ_ID" },
    { WS_NL80211_NAN_FUNC_FOLLOW_UP_DEST,   "NL80211_NAN_FUNC_FOLLOW_UP_DEST" },
    { WS_NL80211_NAN_FUNC_CLOSE_RANGE,      "NL80211_NAN_FUNC_CLOSE_RANGE" },
    { WS_NL80211_NAN_FUNC_TTL,              "NL80211_NAN_FUNC_TTL" },
    { WS_NL80211_NAN_FUNC_SERVICE_INFO,     "NL80211_NAN_FUNC_SERVICE_INFO" },
    { WS_NL80211_NAN_FUNC_SRF,              "NL80211_NAN_FUNC_SRF" },
    { WS_NL80211_NAN_FUNC_RX_MATCH_FILTER,  "NL80211_NAN_FUNC_RX_MATCH_FILTER" },
    { WS_NL80211_NAN_FUNC_TX_MATCH_FILTER,  "NL80211_NAN_FUNC_TX_MATCH_FILTER" },
    { WS_NL80211_NAN_FUNC_INSTANCE_ID,      "NL80211_NAN_FUNC_INSTANCE_ID" },
    { WS_NL80211_NAN_FUNC_TERM_REASON,      "NL80211_NAN_FUNC_TERM_REASON" },
    { 0, NULL }
};
static value_string_ext ws_nl80211_nan_func_attributes_vals_ext = VALUE_STRING_EXT_INIT(ws_nl80211_nan_func_attributes_vals);

static const value_string ws_nl80211_nan_match_attributes_vals[] = {
    { WS___NL80211_NAN_MATCH_INVALID,       "__NL80211_NAN_MATCH_INVALID" },
    { WS_NL80211_NAN_MATCH_FUNC_LOCAL,      "NL80211_NAN_MATCH_FUNC_LOCAL" },
    { WS_NL80211_NAN_MATCH_FUNC_PEER,       "NL80211_NAN_MATCH_FUNC_PEER" },
    { 0, NULL }
};
static value_string_ext ws_nl80211_nan_match_attributes_vals_ext = VALUE_STRING_EXT_INIT(ws_nl80211_nan_match_attributes_vals);

static const value_string ws_nl80211_external_auth_action_vals[] = {
    { WS_NL80211_EXTERNAL_AUTH_START,       "NL80211_EXTERNAL_AUTH_START" },
    { WS_NL80211_EXTERNAL_AUTH_ABORT,       "NL80211_EXTERNAL_AUTH_ABORT" },
    { 0, NULL }
};
static value_string_ext ws_nl80211_external_auth_action_vals_ext = VALUE_STRING_EXT_INIT(ws_nl80211_external_auth_action_vals);

static header_field_info hfi_nl80211_commands NETLINK_NL80211_HFI_INIT =
    { "Command", "nl80211.cmd", FT_UINT8, BASE_DEC | BASE_EXT_STRING,
      VALS_EXT_PTR(&ws_nl80211_commands_vals_ext), 0x00, "Generic Netlink Command", HFILL };

static header_field_info hfi_nl80211_attrs NETLINK_NL80211_HFI_INIT =
    { "Attribute Type", "nl80211.attr_type", FT_UINT16, BASE_DEC | BASE_EXT_STRING,
      VALS_EXT_PTR(&ws_nl80211_attrs_vals_ext), 0x00, NULL, HFILL };

static header_field_info hfi_nl80211_iftype NETLINK_NL80211_HFI_INIT =
    { "Attribute Type", "nl80211.iftype", FT_UINT16, BASE_DEC | BASE_EXT_STRING,
      VALS_EXT_PTR(&ws_nl80211_iftype_vals_ext), 0x00, NULL, HFILL };

static header_field_info hfi_nl80211_sta_flags NETLINK_NL80211_HFI_INIT =
    { "Attribute Type", "nl80211.sta_flags", FT_UINT16, BASE_DEC | BASE_EXT_STRING,
      VALS_EXT_PTR(&ws_nl80211_sta_flags_vals_ext), 0x00, NULL, HFILL };

static header_field_info hfi_nl80211_sta_p2p_ps_status NETLINK_NL80211_HFI_INIT =
    { "Attribute Value", "nl80211.sta_p2p_ps_status", FT_UINT8, BASE_DEC | BASE_EXT_STRING,
      VALS_EXT_PTR(&ws_nl80211_sta_p2p_ps_status_vals_ext), 0x00, NULL, HFILL };

static header_field_info hfi_nl80211_rate_info NETLINK_NL80211_HFI_INIT =
    { "Attribute Type", "nl80211.rate_info", FT_UINT16, BASE_DEC | BASE_EXT_STRING,
      VALS_EXT_PTR(&ws_nl80211_rate_info_vals_ext), 0x00, NULL, HFILL };

static header_field_info hfi_nl80211_sta_bss_param NETLINK_NL80211_HFI_INIT =
    { "Attribute Type", "nl80211.sta_bss_param", FT_UINT16, BASE_DEC | BASE_EXT_STRING,
      VALS_EXT_PTR(&ws_nl80211_sta_bss_param_vals_ext), 0x00, NULL, HFILL };

static header_field_info hfi_nl80211_sta_info NETLINK_NL80211_HFI_INIT =
    { "Attribute Type", "nl80211.sta_info", FT_UINT16, BASE_DEC | BASE_EXT_STRING,
      VALS_EXT_PTR(&ws_nl80211_sta_info_vals_ext), 0x00, NULL, HFILL };

static header_field_info hfi_nl80211_tid_stats NETLINK_NL80211_HFI_INIT =
    { "Attribute Type", "nl80211.tid_stats", FT_UINT16, BASE_DEC | BASE_EXT_STRING,
      VALS_EXT_PTR(&ws_nl80211_tid_stats_vals_ext), 0x00, NULL, HFILL };

static header_field_info hfi_nl80211_txq_stats NETLINK_NL80211_HFI_INIT =
    { "Attribute Type", "nl80211.txq_stats", FT_UINT16, BASE_DEC | BASE_EXT_STRING,
      VALS_EXT_PTR(&ws_nl80211_txq_stats_vals_ext), 0x00, NULL, HFILL };

static header_field_info hfi_nl80211_mpath_info NETLINK_NL80211_HFI_INIT =
    { "Attribute Type", "nl80211.mpath_info", FT_UINT16, BASE_DEC | BASE_EXT_STRING,
      VALS_EXT_PTR(&ws_nl80211_mpath_info_vals_ext), 0x00, NULL, HFILL };

static header_field_info hfi_nl80211_band_iftype_attr NETLINK_NL80211_HFI_INIT =
    { "Attribute Type", "nl80211.band_iftype_attr", FT_UINT16, BASE_DEC | BASE_EXT_STRING,
      VALS_EXT_PTR(&ws_nl80211_band_iftype_attr_vals_ext), 0x00, NULL, HFILL };

static header_field_info hfi_nl80211_band_attr NETLINK_NL80211_HFI_INIT =
    { "Attribute Type", "nl80211.band_attr", FT_UINT16, BASE_DEC | BASE_EXT_STRING,
      VALS_EXT_PTR(&ws_nl80211_band_attr_vals_ext), 0x00, NULL, HFILL };

static header_field_info hfi_nl80211_wmm_rule NETLINK_NL80211_HFI_INIT =
    { "Attribute Type", "nl80211.wmm_rule", FT_UINT16, BASE_DEC | BASE_EXT_STRING,
      VALS_EXT_PTR(&ws_nl80211_wmm_rule_vals_ext), 0x00, NULL, HFILL };

static header_field_info hfi_nl80211_frequency_attr NETLINK_NL80211_HFI_INIT =
    { "Attribute Type", "nl80211.frequency_attr", FT_UINT16, BASE_DEC | BASE_EXT_STRING,
      VALS_EXT_PTR(&ws_nl80211_frequency_attr_vals_ext), 0x00, NULL, HFILL };

static header_field_info hfi_nl80211_bitrate_attr NETLINK_NL80211_HFI_INIT =
    { "Attribute Type", "nl80211.bitrate_attr", FT_UINT16, BASE_DEC | BASE_EXT_STRING,
      VALS_EXT_PTR(&ws_nl80211_bitrate_attr_vals_ext), 0x00, NULL, HFILL };

static header_field_info hfi_nl80211_reg_initiator NETLINK_NL80211_HFI_INIT =
    { "Attribute Value", "nl80211.reg_initiator", FT_UINT8, BASE_DEC | BASE_EXT_STRING,
      VALS_EXT_PTR(&ws_nl80211_reg_initiator_vals_ext), 0x00, NULL, HFILL };

static header_field_info hfi_nl80211_reg_type NETLINK_NL80211_HFI_INIT =
    { "Attribute Value", "nl80211.reg_type", FT_UINT8, BASE_DEC | BASE_EXT_STRING,
      VALS_EXT_PTR(&ws_nl80211_reg_type_vals_ext), 0x00, NULL, HFILL };

static header_field_info hfi_nl80211_reg_rule_attr NETLINK_NL80211_HFI_INIT =
    { "Attribute Type", "nl80211.reg_rule_attr", FT_UINT16, BASE_DEC | BASE_EXT_STRING,
      VALS_EXT_PTR(&ws_nl80211_reg_rule_attr_vals_ext), 0x00, NULL, HFILL };

static header_field_info hfi_nl80211_sched_scan_match_attr NETLINK_NL80211_HFI_INIT =
    { "Attribute Type", "nl80211.sched_scan_match_attr", FT_UINT16, BASE_DEC | BASE_EXT_STRING,
      VALS_EXT_PTR(&ws_nl80211_sched_scan_match_attr_vals_ext), 0x00, NULL, HFILL };

static header_field_info hfi_nl80211_dfs_regions NETLINK_NL80211_HFI_INIT =
    { "Attribute Value", "nl80211.dfs_regions", FT_UINT8, BASE_DEC | BASE_EXT_STRING,
      VALS_EXT_PTR(&ws_nl80211_dfs_regions_vals_ext), 0x00, NULL, HFILL };

static header_field_info hfi_nl80211_user_reg_hint_type NETLINK_NL80211_HFI_INIT =
    { "Attribute Value", "nl80211.user_reg_hint_type", FT_UINT32, BASE_DEC | BASE_EXT_STRING,
      VALS_EXT_PTR(&ws_nl80211_user_reg_hint_type_vals_ext), 0x00, NULL, HFILL };

static header_field_info hfi_nl80211_survey_info NETLINK_NL80211_HFI_INIT =
    { "Attribute Type", "nl80211.survey_info", FT_UINT16, BASE_DEC | BASE_EXT_STRING,
      VALS_EXT_PTR(&ws_nl80211_survey_info_vals_ext), 0x00, NULL, HFILL };

static header_field_info hfi_nl80211_mntr_flags NETLINK_NL80211_HFI_INIT =
    { "Attribute Type", "nl80211.mntr_flags", FT_UINT16, BASE_DEC | BASE_EXT_STRING,
      VALS_EXT_PTR(&ws_nl80211_mntr_flags_vals_ext), 0x00, NULL, HFILL };

static header_field_info hfi_nl80211_mesh_power_mode NETLINK_NL80211_HFI_INIT =
    { "Attribute Value", "nl80211.mesh_power_mode", FT_UINT32, BASE_DEC | BASE_EXT_STRING,
      VALS_EXT_PTR(&ws_nl80211_mesh_power_mode_vals_ext), 0x00, NULL, HFILL };

static header_field_info hfi_nl80211_meshconf_params NETLINK_NL80211_HFI_INIT =
    { "Attribute Type", "nl80211.meshconf_params", FT_UINT16, BASE_DEC | BASE_EXT_STRING,
      VALS_EXT_PTR(&ws_nl80211_meshconf_params_vals_ext), 0x00, NULL, HFILL };

static header_field_info hfi_nl80211_mesh_setup_params NETLINK_NL80211_HFI_INIT =
    { "Attribute Type", "nl80211.mesh_setup_params", FT_UINT16, BASE_DEC | BASE_EXT_STRING,
      VALS_EXT_PTR(&ws_nl80211_mesh_setup_params_vals_ext), 0x00, NULL, HFILL };

static header_field_info hfi_nl80211_txq_attr NETLINK_NL80211_HFI_INIT =
    { "Attribute Type", "nl80211.txq_attr", FT_UINT16, BASE_DEC | BASE_EXT_STRING,
      VALS_EXT_PTR(&ws_nl80211_txq_attr_vals_ext), 0x00, NULL, HFILL };

static header_field_info hfi_nl80211_channel_type NETLINK_NL80211_HFI_INIT =
    { "Attribute Value", "nl80211.channel_type", FT_UINT32, BASE_DEC | BASE_EXT_STRING,
      VALS_EXT_PTR(&ws_nl80211_channel_type_vals_ext), 0x00, NULL, HFILL };

static header_field_info hfi_nl80211_chan_width NETLINK_NL80211_HFI_INIT =
    { "Attribute Value", "nl80211.chan_width", FT_UINT32, BASE_DEC | BASE_EXT_STRING,
      VALS_EXT_PTR(&ws_nl80211_chan_width_vals_ext), 0x00, NULL, HFILL };

static header_field_info hfi_nl80211_bss_scan_width NETLINK_NL80211_HFI_INIT =
    { "Attribute Value", "nl80211.bss_scan_width", FT_UINT32, BASE_DEC | BASE_EXT_STRING,
      VALS_EXT_PTR(&ws_nl80211_bss_scan_width_vals_ext), 0x00, NULL, HFILL };

static header_field_info hfi_nl80211_bss NETLINK_NL80211_HFI_INIT =
    { "Attribute Type", "nl80211.bss", FT_UINT16, BASE_DEC | BASE_EXT_STRING,
      VALS_EXT_PTR(&ws_nl80211_bss_vals_ext), 0x00, NULL, HFILL };

static header_field_info hfi_nl80211_bss_status NETLINK_NL80211_HFI_INIT =
    { "Attribute Value", "nl80211.bss_status", FT_UINT32, BASE_DEC | BASE_EXT_STRING,
      VALS_EXT_PTR(&ws_nl80211_bss_status_vals_ext), 0x00, NULL, HFILL };

static header_field_info hfi_nl80211_auth_type NETLINK_NL80211_HFI_INIT =
    { "Attribute Value", "nl80211.auth_type", FT_UINT32, BASE_DEC | BASE_EXT_STRING,
      VALS_EXT_PTR(&ws_nl80211_auth_type_vals_ext), 0x00, NULL, HFILL };

static header_field_info hfi_nl80211_key_type NETLINK_NL80211_HFI_INIT =
    { "Attribute Value", "nl80211.key_type", FT_UINT32, BASE_DEC | BASE_EXT_STRING,
      VALS_EXT_PTR(&ws_nl80211_key_type_vals_ext), 0x00, NULL, HFILL };

static header_field_info hfi_nl80211_mfp NETLINK_NL80211_HFI_INIT =
    { "Attribute Value", "nl80211.mfp", FT_UINT32, BASE_DEC | BASE_EXT_STRING,
      VALS_EXT_PTR(&ws_nl80211_mfp_vals_ext), 0x00, NULL, HFILL };

static header_field_info hfi_nl80211_key_default_types NETLINK_NL80211_HFI_INIT =
    { "Attribute Type", "nl80211.key_default_types", FT_UINT16, BASE_DEC | BASE_EXT_STRING,
      VALS_EXT_PTR(&ws_nl80211_key_default_types_vals_ext), 0x00, NULL, HFILL };

static header_field_info hfi_nl80211_key_attributes NETLINK_NL80211_HFI_INIT =
    { "Attribute Type", "nl80211.key_attributes", FT_UINT16, BASE_DEC | BASE_EXT_STRING,
      VALS_EXT_PTR(&ws_nl80211_key_attributes_vals_ext), 0x00, NULL, HFILL };

static header_field_info hfi_nl80211_tx_rate_attributes NETLINK_NL80211_HFI_INIT =
    { "Attribute Type", "nl80211.tx_rate_attributes", FT_UINT16, BASE_DEC | BASE_EXT_STRING,
      VALS_EXT_PTR(&ws_nl80211_tx_rate_attributes_vals_ext), 0x00, NULL, HFILL };

static header_field_info hfi_nl80211_ps_state NETLINK_NL80211_HFI_INIT =
    { "Attribute Value", "nl80211.ps_state", FT_UINT32, BASE_DEC | BASE_EXT_STRING,
      VALS_EXT_PTR(&ws_nl80211_ps_state_vals_ext), 0x00, NULL, HFILL };

static header_field_info hfi_nl80211_attr_cqm NETLINK_NL80211_HFI_INIT =
    { "Attribute Type", "nl80211.attr_cqm", FT_UINT16, BASE_DEC | BASE_EXT_STRING,
      VALS_EXT_PTR(&ws_nl80211_attr_cqm_vals_ext), 0x00, NULL, HFILL };

static header_field_info hfi_nl80211_tx_power_setting NETLINK_NL80211_HFI_INIT =
    { "Attribute Value", "nl80211.tx_power_setting", FT_UINT32, BASE_DEC | BASE_EXT_STRING,
      VALS_EXT_PTR(&ws_nl80211_tx_power_setting_vals_ext), 0x00, NULL, HFILL };

static header_field_info hfi_nl80211_if_combination_attrs NETLINK_NL80211_HFI_INIT =
    { "Attribute Type", "nl80211.if_combination_attrs", FT_UINT16, BASE_DEC | BASE_EXT_STRING,
      VALS_EXT_PTR(&ws_nl80211_if_combination_attrs_vals_ext), 0x00, NULL, HFILL };

static header_field_info hfi_nl80211_plink_state NETLINK_NL80211_HFI_INIT =
    { "Attribute Value", "nl80211.plink_state", FT_UINT8, BASE_DEC | BASE_EXT_STRING,
      VALS_EXT_PTR(&ws_nl80211_plink_state_vals_ext), 0x00, NULL, HFILL };

static header_field_info hfi_plink_actions NETLINK_NL80211_HFI_INIT =
    { "Attribute Value", "nl80211.plink_actions", FT_UINT8, BASE_DEC | BASE_EXT_STRING,
      VALS_EXT_PTR(&ws_plink_actions_vals_ext), 0x00, NULL, HFILL };

static header_field_info hfi_nl80211_rekey_data NETLINK_NL80211_HFI_INIT =
    { "Attribute Type", "nl80211.rekey_data", FT_UINT16, BASE_DEC | BASE_EXT_STRING,
      VALS_EXT_PTR(&ws_nl80211_rekey_data_vals_ext), 0x00, NULL, HFILL };

static header_field_info hfi_nl80211_sta_wme_attr NETLINK_NL80211_HFI_INIT =
    { "Attribute Type", "nl80211.sta_wme_attr", FT_UINT16, BASE_DEC | BASE_EXT_STRING,
      VALS_EXT_PTR(&ws_nl80211_sta_wme_attr_vals_ext), 0x00, NULL, HFILL };

static header_field_info hfi_nl80211_pmksa_candidate_attr NETLINK_NL80211_HFI_INIT =
    { "Attribute Type", "nl80211.pmksa_candidate_attr", FT_UINT16, BASE_DEC | BASE_EXT_STRING,
      VALS_EXT_PTR(&ws_nl80211_pmksa_candidate_attr_vals_ext), 0x00, NULL, HFILL };

static header_field_info hfi_nl80211_tdls_operation NETLINK_NL80211_HFI_INIT =
    { "Attribute Value", "nl80211.tdls_operation", FT_UINT8, BASE_DEC | BASE_EXT_STRING,
      VALS_EXT_PTR(&ws_nl80211_tdls_operation_vals_ext), 0x00, NULL, HFILL };

static header_field_info hfi_nl80211_connect_failed_reason NETLINK_NL80211_HFI_INIT =
    { "Attribute Value", "nl80211.connect_failed_reason", FT_UINT32, BASE_DEC | BASE_EXT_STRING,
      VALS_EXT_PTR(&ws_nl80211_connect_failed_reason_vals_ext), 0x00, NULL, HFILL };

static header_field_info hfi_nl80211_timeout_reason NETLINK_NL80211_HFI_INIT =
    { "Attribute Value", "nl80211.timeout_reason", FT_UINT32, BASE_DEC | BASE_EXT_STRING,
      VALS_EXT_PTR(&ws_nl80211_timeout_reason_vals_ext), 0x00, NULL, HFILL };

static header_field_info hfi_nl80211_acl_policy NETLINK_NL80211_HFI_INIT =
    { "Attribute Value", "nl80211.acl_policy", FT_UINT32, BASE_DEC | BASE_EXT_STRING,
      VALS_EXT_PTR(&ws_nl80211_acl_policy_vals_ext), 0x00, NULL, HFILL };

static header_field_info hfi_nl80211_smps_mode NETLINK_NL80211_HFI_INIT =
    { "Attribute Value", "nl80211.smps_mode", FT_UINT8, BASE_DEC | BASE_EXT_STRING,
      VALS_EXT_PTR(&ws_nl80211_smps_mode_vals_ext), 0x00, NULL, HFILL };

static header_field_info hfi_nl80211_radar_event NETLINK_NL80211_HFI_INIT =
    { "Attribute Value", "nl80211.radar_event", FT_UINT32, BASE_DEC | BASE_EXT_STRING,
      VALS_EXT_PTR(&ws_nl80211_radar_event_vals_ext), 0x00, NULL, HFILL };

static header_field_info hfi_nl80211_dfs_state NETLINK_NL80211_HFI_INIT =
    { "Attribute Type", "nl80211.dfs_state", FT_UINT16, BASE_DEC | BASE_EXT_STRING,
      VALS_EXT_PTR(&ws_nl80211_dfs_state_vals_ext), 0x00, NULL, HFILL };

static header_field_info hfi_nl80211_crit_proto_id NETLINK_NL80211_HFI_INIT =
    { "Attribute Value", "nl80211.crit_proto_id", FT_UINT16, BASE_DEC | BASE_EXT_STRING,
      VALS_EXT_PTR(&ws_nl80211_crit_proto_id_vals_ext), 0x00, NULL, HFILL };

static header_field_info hfi_nl80211_sched_scan_plan NETLINK_NL80211_HFI_INIT =
    { "Attribute Type", "nl80211.sched_scan_plan", FT_UINT16, BASE_DEC | BASE_EXT_STRING,
      VALS_EXT_PTR(&ws_nl80211_sched_scan_plan_vals_ext), 0x00, NULL, HFILL };

static header_field_info hfi_nl80211_bss_select_attr NETLINK_NL80211_HFI_INIT =
    { "Attribute Type", "nl80211.bss_select_attr", FT_UINT16, BASE_DEC | BASE_EXT_STRING,
      VALS_EXT_PTR(&ws_nl80211_bss_select_attr_vals_ext), 0x00, NULL, HFILL };

static header_field_info hfi_nl80211_nan_func_attributes NETLINK_NL80211_HFI_INIT =
    { "Attribute Type", "nl80211.an_func_attributes", FT_UINT16, BASE_DEC | BASE_EXT_STRING,
      VALS_EXT_PTR(&ws_nl80211_nan_func_attributes_vals_ext), 0x00, NULL, HFILL };

static header_field_info hfi_nl80211_nan_match_attributes NETLINK_NL80211_HFI_INIT =
    { "Attribute Type", "nl80211.an_match_attributes", FT_UINT16, BASE_DEC | BASE_EXT_STRING,
      VALS_EXT_PTR(&ws_nl80211_nan_match_attributes_vals_ext), 0x00, NULL, HFILL };

static header_field_info hfi_nl80211_external_auth_action NETLINK_NL80211_HFI_INIT =
    { "Attribute Value", "nl80211.external_auth_action", FT_UINT32, BASE_DEC | BASE_EXT_STRING,
      VALS_EXT_PTR(&ws_nl80211_external_auth_action_vals_ext), 0x00, NULL, HFILL };

static gint ett_nl80211_commands = -1;
static gint ett_nl80211_attrs = -1;
static gint ett_nl80211_iftype = -1;
static gint ett_nl80211_sta_flags = -1;
static gint ett_nl80211_sta_p2p_ps_status = -1;
static gint ett_nl80211_rate_info = -1;
static gint ett_nl80211_sta_bss_param = -1;
static gint ett_nl80211_sta_info = -1;
static gint ett_nl80211_tid_stats = -1;
static gint ett_nl80211_txq_stats = -1;
static gint ett_nl80211_mpath_info = -1;
static gint ett_nl80211_band_iftype_attr = -1;
static gint ett_nl80211_band_attr = -1;
static gint ett_nl80211_wmm_rule = -1;
static gint ett_nl80211_frequency_attr = -1;
static gint ett_nl80211_bitrate_attr = -1;
static gint ett_nl80211_reg_initiator = -1;
static gint ett_nl80211_reg_type = -1;
static gint ett_nl80211_reg_rule_attr = -1;
static gint ett_nl80211_sched_scan_match_attr = -1;
static gint ett_nl80211_dfs_regions = -1;
static gint ett_nl80211_user_reg_hint_type = -1;
static gint ett_nl80211_survey_info = -1;
static gint ett_nl80211_mntr_flags = -1;
static gint ett_nl80211_mesh_power_mode = -1;
static gint ett_nl80211_meshconf_params = -1;
static gint ett_nl80211_mesh_setup_params = -1;
static gint ett_nl80211_txq_attr = -1;
static gint ett_nl80211_channel_type = -1;
static gint ett_nl80211_chan_width = -1;
static gint ett_nl80211_bss_scan_width = -1;
static gint ett_nl80211_bss = -1;
static gint ett_nl80211_bss_status = -1;
static gint ett_nl80211_auth_type = -1;
static gint ett_nl80211_key_type = -1;
static gint ett_nl80211_mfp = -1;
static gint ett_nl80211_key_default_types = -1;
static gint ett_nl80211_key_attributes = -1;
static gint ett_nl80211_tx_rate_attributes = -1;
static gint ett_nl80211_ps_state = -1;
static gint ett_nl80211_attr_cqm = -1;
static gint ett_nl80211_tx_power_setting = -1;
static gint ett_nl80211_if_combination_attrs = -1;
static gint ett_nl80211_plink_state = -1;
static gint ett_plink_actions = -1;
static gint ett_nl80211_rekey_data = -1;
static gint ett_nl80211_sta_wme_attr = -1;
static gint ett_nl80211_pmksa_candidate_attr = -1;
static gint ett_nl80211_tdls_operation = -1;
static gint ett_nl80211_connect_failed_reason = -1;
static gint ett_nl80211_timeout_reason = -1;
static gint ett_nl80211_acl_policy = -1;
static gint ett_nl80211_smps_mode = -1;
static gint ett_nl80211_radar_event = -1;
static gint ett_nl80211_dfs_state = -1;
static gint ett_nl80211_crit_proto_id = -1;
static gint ett_nl80211_sched_scan_plan = -1;
static gint ett_nl80211_bss_select_attr = -1;
static gint ett_nl80211_nan_func_attributes = -1;
static gint ett_nl80211_nan_match_attributes = -1;
static gint ett_nl80211_external_auth_action = -1;
/* }}} */


static int proto_netlink_nl80211;

static dissector_handle_t netlink_nl80211_handle;

static header_field_info *hfi_netlink_nl80211 = NULL;

static gint ett_nl80211 = -1;
static gint ett_nl80211_frame = -1;
static gint ett_nl80211_tag = -1;

static header_field_info hfi_nl80211_attr_value NETLINK_NL80211_HFI_INIT =
    { "Attribute Value", "nl80211.attr_value", FT_BYTES, BASE_NONE,
      NULL, 0x00, NULL, HFILL };

static header_field_info hfi_nl80211_attr_value16 NETLINK_NL80211_HFI_INIT =
    { "Attribute Value", "nl80211.attr_value16", FT_UINT16, BASE_HEX_DEC,
      NULL, 0x00, NULL, HFILL };

static header_field_info hfi_nl80211_attr_value32 NETLINK_NL80211_HFI_INIT =
    { "Attribute Value", "nl80211.attr_value32", FT_UINT32, BASE_HEX_DEC,
      NULL, 0x00, NULL, HFILL };

static header_field_info hfi_nl80211_attr_value64 NETLINK_NL80211_HFI_INIT =
    { "Attribute Value", "nl80211.attr_value64", FT_UINT64, BASE_HEX_DEC,
      NULL, 0x00, NULL, HFILL };

static header_field_info hfi_nl80211_wiphy_name NETLINK_NL80211_HFI_INIT =
    { "Wiphy Name", "nl80211.wiphy_name", FT_STRINGZ, STR_ASCII,
      NULL, 0x00, NULL, HFILL };

static header_field_info hfi_nl80211_ifname NETLINK_NL80211_HFI_INIT =
    { "Interface Name", "nl80211.ifname", FT_STRINGZ, STR_ASCII,
      NULL, 0x00, NULL, HFILL };

static header_field_info hfi_nl80211_mac NETLINK_NL80211_HFI_INIT =
    { "MAC address", "nl80211.mac", FT_ETHER, BASE_NONE,
      NULL, 0x00, NULL, HFILL };

static header_field_info hfi_nl80211_alpha2 NETLINK_NL80211_HFI_INIT =
    { "Alpha2", "nl80211.alpha2", FT_STRINGZ, STR_ASCII,
      NULL, 0x00, NULL, HFILL };

static header_field_info hfi_nl80211_dbm NETLINK_NL80211_HFI_INIT =
    { "dBm", "nl80211.dbm", FT_INT32, BASE_DEC,
      NULL, 0x00, NULL, HFILL };

static int
dissect_nl80211_generic(tvbuff_t *tvb, void *data, proto_tree *tree, _U_ int nla_type, int offset, int len)
{
    genl_info_t *genl_info = (genl_info_t *)data;
    /*
     * No specific dissection available, apply arbitrary heuristics to
     * determine whether we have an u16 or u32 field and treat others as
     * opaque bytes.
     */
    if (len) {
        if (len == 2) {
            proto_tree_add_item(tree, &hfi_nl80211_attr_value16, tvb, offset, len, genl_info->encoding);
        } else if (len == 4) {
            proto_tree_add_item(tree, &hfi_nl80211_attr_value32, tvb, offset, len, genl_info->encoding);
        } else if (len == 8) {
            proto_tree_add_item(tree, &hfi_nl80211_attr_value64, tvb, offset, len, genl_info->encoding);
        } else {
            proto_tree_add_item(tree, &hfi_nl80211_attr_value, tvb, offset, len, genl_info->encoding);
        }
        offset += len;
    }
    return offset;
}

struct attr_lookup {
    unsigned int attr_type;
    header_field_info* hfi;
    gint* ett;
    int (*func)(tvbuff_t *tvb, void *data, proto_tree *tree, int nla_type, int offset, int len);
};

static int
dissect_nested_attr(tvbuff_t *tvb, void *data, proto_tree *tree, int nla_type, int offset, int len, const struct attr_lookup *nested)
{
    genl_info_t *genl_info = (genl_info_t *)data;
    for (int i = 0; nested[i].hfi; i++) {
        if (nested[i].attr_type != (nla_type & NLA_TYPE_MASK)) {
            continue;
        }
        offset = dissect_netlink_attributes(tvb, nested[i].hfi, *nested[i].ett, genl_info,
                                            genl_info->data, tree, offset, len,
                                            nested[i].func ? nested[i].func : dissect_nl80211_generic);
        break;
    }
    return offset;
}

static int
dissect_nested_attr_array(tvbuff_t *tvb, void *data, proto_tree *tree, int nla_type, int offset, int len, const struct attr_lookup *nested_arr)
{
    genl_info_t *genl_info = (genl_info_t *)data;
    for (int i = 0; nested_arr[i].hfi; i++) {
        if (nested_arr[i].attr_type != (nla_type & NLA_TYPE_MASK)) {
            continue;
        }
        offset = dissect_netlink_attributes_array(tvb, nested_arr[i].hfi, *nested_arr[i].ett,
                                                  *nested_arr[i].ett, genl_info,
                                                  genl_info->data, tree, offset, len,
                                                  nested_arr[i].func ?
                                                   nested_arr[i].func : dissect_nl80211_generic);
       break;
    }
    return offset;
}

static int
dissect_value(tvbuff_t *tvb, void *data, proto_tree *tree, int nla_type, int offset, int len, const struct attr_lookup *values)
{
    genl_info_t *genl_info = (genl_info_t *)data;
    for (int i = 0; values[i].hfi; i++) {
        if (values[i].attr_type != (nla_type & NLA_TYPE_MASK)) {
            continue;
        }
        proto_tree_add_item(tree, values[i].hfi, tvb, offset, len, genl_info->encoding);
        return offset + len;
    }
    return offset;
}

static packet_info *m_pinfo = NULL; /* TODO find a better way to pass pinfo to functions */

static int
dissect_tag(tvbuff_t *tvb, proto_tree *tree _U_, int offset, int len, guint8 tag)
{
    proto_item *item;
    ieee80211_tagged_field_data_t field_data = { 0 };
    tvbuff_t *next_tvb = tvb_new_subset_length(tvb, offset, len);
    proto_tree *subtree = proto_tree_add_subtree(tree, next_tvb, 0, -1, ett_nl80211_tag, &item, "Attribute Value");
    dissector_try_uint_new(ieee80211_tag_dissector_table, tag, next_tvb, m_pinfo, subtree, FALSE, &field_data);
    return offset +  len;
}

static int
dissect_information_elements(tvbuff_t *tvb, proto_tree *tree, int offset, int len)
{
    int offset_end = offset + len;
    while (offset < offset_end) {
        int tlen = add_tagged_field(m_pinfo, tree, tvb, offset, 0, NULL, 0, NULL);
        if (tlen == 0) {
            break;
        }
        offset += tlen;
    }
    return offset;
}

static int
dissect_nl80211_frequency_attr(tvbuff_t *tvb, void *data, proto_tree *tree, int nla_type, int offset, int len)
{
    static const struct attr_lookup nested[] = {
        { WS_NL80211_FREQUENCY_ATTR_DFS_STATE, &hfi_nl80211_dfs_state, &ett_nl80211_dfs_state, NULL},
        { WS_NL80211_FREQUENCY_ATTR_WMM, &hfi_nl80211_wmm_rule, &ett_nl80211_wmm_rule, NULL},
        { 0, NULL, NULL, NULL }
    };
    enum ws_nl80211_frequency_attr type = (enum ws_nl80211_frequency_attr) nla_type & NLA_TYPE_MASK;
    int offset_end = offset + len;

    if (offset < offset_end) {
        offset = dissect_nested_attr(tvb, data, tree, nla_type, offset, len, nested);
    }
    if (offset < offset_end) {
        switch (type) {
        default:
            offset = dissect_nl80211_generic(tvb, data, tree, nla_type, offset, len);
            break;
        }
    }
    return offset;
}

static int
dissect_nl80211_band_attr(tvbuff_t *tvb, void *data, proto_tree *tree, int nla_type, int offset, int len)
{
    static const struct attr_lookup nested_arr[] = {
        { WS_NL80211_BAND_ATTR_FREQS, &hfi_nl80211_frequency_attr, &ett_nl80211_frequency_attr, dissect_nl80211_frequency_attr },
        { WS_NL80211_BAND_ATTR_RATES, &hfi_nl80211_bitrate_attr, &ett_nl80211_bitrate_attr, NULL },
        { WS_NL80211_BAND_ATTR_IFTYPE_DATA, &hfi_nl80211_band_iftype_attr, &ett_nl80211_band_iftype_attr, NULL },
        { 0, NULL, NULL, NULL }
    };
    enum ws_nl80211_band_attr type = (enum ws_nl80211_band_attr) nla_type & NLA_TYPE_MASK;
    int offset_end = offset + len;

    if (offset < offset_end) {
        offset = dissect_nested_attr_array(tvb, data, tree, nla_type, offset, len, nested_arr);
    }
    if (offset < offset_end) {
        switch (type) {
        /* TODO add more fields here? */
        default:
            offset = dissect_nl80211_generic(tvb, data, tree, nla_type, offset, len);
            break;
        }
    }
    return offset;
}

static int
dissect_nl80211_bss(tvbuff_t *tvb, void *data, proto_tree *tree, int nla_type, int offset, int len)
{
    static const struct attr_lookup values[] = {
        { WS_NL80211_BSS_STATUS, &hfi_nl80211_bss_status, NULL, NULL },
        { WS_NL80211_BSS_CHAN_WIDTH, &hfi_nl80211_bss_scan_width, NULL, NULL },
        { 0, NULL, NULL, NULL }
    };
    enum ws_nl80211_bss type = (enum ws_nl80211_bss) nla_type & NLA_TYPE_MASK;
    int offset_end = offset + len;

    if (offset < offset_end) {
        offset = dissect_value(tvb, data, tree, nla_type, offset, len, values);
    }
    if (offset < offset_end) {
        switch (type) {
        case WS_NL80211_BSS_INFORMATION_ELEMENTS:
        case WS_NL80211_BSS_BEACON_IES:
            offset = dissect_information_elements(tvb, tree, offset, len);
            break;
        default:
            offset = dissect_nl80211_generic(tvb, data, tree, nla_type, offset, len);
            break;
        }
    }
    return offset;
}

static int
dissect_nl80211_tid_stats(tvbuff_t *tvb, void *data, proto_tree *tree, int nla_type, int offset, int len)
{
    static const struct attr_lookup nested[] = {
        { WS_NL80211_TID_STATS_TXQ_STATS, &hfi_nl80211_txq_stats, &ett_nl80211_txq_stats, NULL},
        { 0, NULL, NULL, NULL }
    };

    enum ws_nl80211_tid_stats type = (enum ws_nl80211_tid_stats) nla_type & NLA_TYPE_MASK;
    int offset_end = offset + len;
    if (offset < offset_end) {
        offset = dissect_nested_attr(tvb, data, tree, nla_type, offset, len, nested);
    }
    if (offset < offset_end) {
        switch (type) {
        default:
            offset = dissect_nl80211_generic(tvb, data, tree, nla_type, offset, len);
            break;
        }
    }
    return offset;
}

static int
dissect_nl80211_sta_info(tvbuff_t *tvb, void *data, proto_tree *tree, int nla_type, int offset, int len)
{
    static const struct attr_lookup nested[] = {
        { WS_NL80211_STA_INFO_TX_BITRATE, &hfi_nl80211_rate_info, &ett_nl80211_rate_info, NULL},
        { WS_NL80211_STA_INFO_RX_BITRATE, &hfi_nl80211_rate_info, &ett_nl80211_rate_info, NULL},
        { WS_NL80211_STA_INFO_BSS_PARAM, &hfi_nl80211_sta_bss_param, &ett_nl80211_sta_bss_param, NULL },
        { 0, NULL, NULL, NULL }
    };
    static const struct attr_lookup nested_arr[] = {
        { WS_NL80211_STA_INFO_TID_STATS, &hfi_nl80211_tid_stats, &ett_nl80211_tid_stats, dissect_nl80211_tid_stats},
        { 0, NULL, NULL, NULL }
    };

    enum ws_nl80211_sta_info type = (enum ws_nl80211_sta_info) nla_type & NLA_TYPE_MASK;
    int offset_end = offset + len;
    if (offset < offset_end) {
        offset = dissect_nested_attr(tvb, data, tree, nla_type, offset, len, nested);
    }
    if (offset < offset_end) {
        offset = dissect_nested_attr_array(tvb, data, tree, nla_type, offset, len, nested_arr);
    }
    if (offset < offset_end) {
        genl_info_t *genl_info = (genl_info_t *)data;
        switch (type) {
        case WS_NL80211_STA_INFO_SIGNAL:
        case WS_NL80211_STA_INFO_SIGNAL_AVG:
        case WS_NL80211_STA_INFO_BEACON_SIGNAL_AVG:
        case WS_NL80211_STA_INFO_ACK_SIGNAL:
        case WS_NL80211_STA_INFO_ACK_SIGNAL_AVG:
            proto_tree_add_item(tree, &hfi_nl80211_dbm, tvb, offset, len, genl_info->encoding);
            offset += len;
            break;
        default:
            offset = dissect_nl80211_generic(tvb, data, tree, nla_type, offset, len);
            break;
        }
    }
    return offset;
}


static int
dissect_nl80211_attrs(tvbuff_t *tvb, void *data, proto_tree *tree, int nla_type, int offset, int len)
{
    static const struct attr_lookup nested[] = {
        { WS_NL80211_ATTR_SUPPORTED_IFTYPES, &hfi_nl80211_iftype, &ett_nl80211_iftype, NULL },
        { WS_NL80211_ATTR_STA_FLAGS, &hfi_nl80211_sta_flags, &ett_nl80211_sta_flags, NULL },
        { WS_NL80211_ATTR_STA_INFO, &hfi_nl80211_sta_info, &ett_nl80211_sta_info, dissect_nl80211_sta_info },
        { WS_NL80211_ATTR_MPATH_INFO, &hfi_nl80211_mpath_info, &ett_nl80211_mpath_info, NULL },
        { WS_NL80211_ATTR_MNTR_FLAGS, &hfi_nl80211_mntr_flags, &ett_nl80211_mntr_flags, NULL },
        { WS_NL80211_ATTR_BSS, &hfi_nl80211_bss, &ett_nl80211_bss, dissect_nl80211_bss },
        { WS_NL80211_ATTR_KEY, &hfi_nl80211_key_attributes, &ett_nl80211_key_attributes, NULL },
        { WS_NL80211_ATTR_SURVEY_INFO, &hfi_nl80211_survey_info, &ett_nl80211_survey_info, NULL },
        { WS_NL80211_ATTR_FREQ_BEFORE, &hfi_nl80211_frequency_attr, &ett_nl80211_frequency_attr, NULL },
        { WS_NL80211_ATTR_FREQ_AFTER, &hfi_nl80211_frequency_attr, &ett_nl80211_frequency_attr, NULL },
        { WS_NL80211_ATTR_TX_RATES, &hfi_nl80211_tx_rate_attributes, &ett_nl80211_tx_rate_attributes, NULL },
        { WS_NL80211_ATTR_CQM, &hfi_nl80211_attr_cqm, &ett_nl80211_attr_cqm, NULL },
        { WS_NL80211_ATTR_KEY_DEFAULT_TYPES, &hfi_nl80211_key_default_types, &ett_nl80211_key_default_types, NULL },
        { WS_NL80211_ATTR_MESH_SETUP, &hfi_nl80211_mesh_setup_params, &ett_nl80211_mesh_setup_params, NULL },
        { WS_NL80211_ATTR_MESH_CONFIG, &hfi_nl80211_meshconf_params, &ett_nl80211_meshconf_params, NULL },
        { WS_NL80211_ATTR_SCHED_SCAN_MATCH, &hfi_nl80211_sched_scan_match_attr, &ett_nl80211_sched_scan_match_attr, NULL },
        { WS_NL80211_ATTR_INTERFACE_COMBINATIONS, &hfi_nl80211_if_combination_attrs, &ett_nl80211_if_combination_attrs, NULL },
        { WS_NL80211_ATTR_REKEY_DATA, &hfi_nl80211_rekey_data, &ett_nl80211_rekey_data, NULL },
        { WS_NL80211_ATTR_STA_WME, &hfi_nl80211_sta_wme_attr, &ett_nl80211_sta_wme_attr, NULL },
        { WS_NL80211_ATTR_PMKSA_CANDIDATE, &hfi_nl80211_pmksa_candidate_attr, &ett_nl80211_pmksa_candidate_attr, NULL },
        { WS_NL80211_ATTR_SCHED_SCAN_PLANS, &hfi_nl80211_sched_scan_plan, &ett_nl80211_sched_scan_plan, NULL },
        { WS_NL80211_ATTR_BSS_SELECT, &hfi_nl80211_bss_select_attr, &ett_nl80211_bss_select_attr, NULL },
        { WS_NL80211_ATTR_IFTYPE_EXT_CAPA, &hfi_nl80211_attrs, &ett_nl80211_attrs, dissect_nl80211_attrs },
        { WS_NL80211_ATTR_NAN_FUNC, &hfi_nl80211_nan_func_attributes, &ett_nl80211_nan_func_attributes, NULL },
        { WS_NL80211_ATTR_NAN_MATCH, &hfi_nl80211_nan_match_attributes, &ett_nl80211_nan_match_attributes, NULL },
        { WS_NL80211_ATTR_TXQ_STATS, &hfi_nl80211_txq_stats, &ett_nl80211_txq_stats, NULL },
        { 0, NULL, NULL, NULL }
    };
    static const struct attr_lookup nested_arr[] = {
        { WS_NL80211_ATTR_WIPHY_TXQ_PARAMS, &hfi_nl80211_txq_attr, &ett_nl80211_txq_attr, NULL },
        { WS_NL80211_ATTR_WIPHY_BANDS, &hfi_nl80211_band_attr, &ett_nl80211_band_attr, dissect_nl80211_band_attr },
        { WS_NL80211_ATTR_REG_RULES, &hfi_nl80211_reg_rule_attr, &ett_nl80211_reg_rule_attr, NULL },
        { 0, NULL, NULL, NULL }
    };
    static const struct attr_lookup values[] = {
        { WS_NL80211_ATTR_CHANNEL_WIDTH, &hfi_nl80211_chan_width, NULL, NULL },
        { WS_NL80211_ATTR_WIPHY_NAME, &hfi_nl80211_wiphy_name, NULL, NULL },
        { WS_NL80211_ATTR_WIPHY_CHANNEL_TYPE, &hfi_nl80211_channel_type, NULL, NULL },
        { WS_NL80211_ATTR_IFNAME, &hfi_nl80211_ifname, NULL, NULL },
        { WS_NL80211_ATTR_IFTYPE, &hfi_nl80211_iftype, NULL, NULL },
        { WS_NL80211_ATTR_MAC, &hfi_nl80211_mac, NULL, NULL },
        { WS_NL80211_ATTR_STA_PLINK_ACTION, &hfi_plink_actions, NULL, NULL },
        { WS_NL80211_ATTR_MPATH_INFO, &hfi_nl80211_mpath_info, NULL, NULL },
        { WS_NL80211_ATTR_REG_ALPHA2, &hfi_nl80211_alpha2, NULL, NULL },
        { WS_NL80211_ATTR_REG_INITIATOR, &hfi_nl80211_reg_initiator, NULL, NULL },
        { WS_NL80211_ATTR_REG_TYPE, &hfi_nl80211_reg_type, NULL, NULL },
        { WS_NL80211_ATTR_AUTH_TYPE, &hfi_nl80211_auth_type, NULL, NULL },
        { WS_NL80211_ATTR_KEY_TYPE, &hfi_nl80211_key_type, NULL, NULL },
        { WS_NL80211_ATTR_USE_MFP, &hfi_nl80211_mfp, NULL, NULL },
        { WS_NL80211_ATTR_PS_STATE, &hfi_nl80211_ps_state, NULL, NULL },
        { WS_NL80211_ATTR_WIPHY_TX_POWER_SETTING, &hfi_nl80211_tx_power_setting, NULL, NULL },
        { WS_NL80211_ATTR_STA_PLINK_STATE, &hfi_nl80211_plink_state, NULL, NULL },
        { WS_NL80211_ATTR_TDLS_OPERATION, &hfi_nl80211_tdls_operation, NULL, NULL },
        { WS_NL80211_ATTR_DFS_REGION, &hfi_nl80211_dfs_regions, NULL, NULL },
        { WS_NL80211_ATTR_RX_SIGNAL_DBM, &hfi_nl80211_dbm, NULL, NULL},
        { WS_NL80211_ATTR_USER_REG_HINT_TYPE, &hfi_nl80211_user_reg_hint_type, NULL, NULL },
        { WS_NL80211_ATTR_CONN_FAILED_REASON, &hfi_nl80211_connect_failed_reason, NULL, NULL },
        { WS_NL80211_ATTR_LOCAL_MESH_POWER_MODE, &hfi_nl80211_mesh_power_mode, NULL, NULL },
        { WS_NL80211_ATTR_ACL_POLICY, &hfi_nl80211_acl_policy, NULL, NULL },
        { WS_NL80211_ATTR_RADAR_EVENT, &hfi_nl80211_radar_event, NULL, NULL },
        { WS_NL80211_ATTR_CRIT_PROT_ID, &hfi_nl80211_crit_proto_id, NULL, NULL },
        { WS_NL80211_ATTR_SMPS_MODE, &hfi_nl80211_smps_mode, NULL, NULL },
        { WS_NL80211_ATTR_STA_SUPPORT_P2P_PS, &hfi_nl80211_sta_p2p_ps_status, NULL, NULL },
        { WS_NL80211_ATTR_TIMEOUT_REASON, &hfi_nl80211_timeout_reason, NULL, NULL },
        { WS_NL80211_ATTR_EXTERNAL_AUTH_ACTION, &hfi_nl80211_external_auth_action, NULL, NULL },
        { 0, NULL, NULL, NULL }
    };
    enum ws_nl80211_attrs type = (enum ws_nl80211_attrs) nla_type & NLA_TYPE_MASK;
    int offset_end = offset + len;
    if (offset < offset_end) {
        offset = dissect_nested_attr(tvb, data, tree, nla_type, offset, len, nested);
    }
    if (offset < offset_end) {
        offset = dissect_nested_attr_array(tvb, data, tree, nla_type, offset, len, nested_arr);
    }
    if (offset < offset_end) {
        offset = dissect_value(tvb, data, tree, nla_type, offset, len, values);
    }
    if (offset < offset_end) {
        tvbuff_t *next_tvb;
        proto_tree *subtree;
        proto_item *item;

        switch (type) {
        case WS_NL80211_ATTR_IE:
        case WS_NL80211_ATTR_REQ_IE:
        case WS_NL80211_ATTR_RESP_IE:
        case WS_NL80211_ATTR_IE_PROBE_RESP:
        case WS_NL80211_ATTR_IE_ASSOC_RESP:
        case WS_NL80211_ATTR_CSA_IES:
            offset = dissect_information_elements(tvb, tree, offset, len);
            break;
        case WS_NL80211_ATTR_HT_CAPABILITY:
            offset = dissect_tag(tvb, tree, offset, len, 45 /* TAG_HT_CAPABILITY */);
            break;
        case WS_NL80211_ATTR_VHT_CAPABILITY:
            offset = dissect_tag(tvb, tree, offset, len, 191 /* TAG_VHT_CAPABILITY */);
            break;
        case WS_NL80211_ATTR_FRAME:
            next_tvb = tvb_new_subset_length(tvb, offset, len);
            subtree = proto_tree_add_subtree(tree, next_tvb, 0, -1, ett_nl80211_frame,
                                             &item, "Attribute Value");
            call_dissector(ieee80211_handle, next_tvb, m_pinfo, subtree);
            break;
        /* TODO add more fields here? */
        default:
            offset = dissect_nl80211_generic(tvb, data, tree, nla_type, offset, len);
            break;
        }
    }
    return offset;
}

static int
dissect_netlink_nl80211(tvbuff_t *tvb, packet_info *pinfo, proto_tree *tree, void *data)
{
    genl_info_t *genl_info = (genl_info_t *)data;
    proto_tree *nlmsg_tree;
    proto_item *pi;
    int offset;

    DISSECTOR_ASSERT(genl_info);

    col_set_str(pinfo->cinfo, COL_PROTOCOL, "nl80211");
    col_clear(pinfo->cinfo, COL_INFO);
    m_pinfo = pinfo;

    offset = dissect_genl_header(tvb, genl_info, &hfi_nl80211_commands);

    pi = proto_tree_add_item(tree, proto_registrar_get_nth(proto_netlink_nl80211), tvb, offset, -1, ENC_NA);
    nlmsg_tree = proto_item_add_subtree(pi, ett_nl80211);

    offset = dissect_netlink_attributes(tvb, &hfi_nl80211_attrs, ett_nl80211_attrs, genl_info, genl_info->data, nlmsg_tree, offset, -1, dissect_nl80211_attrs);

    return offset;
}

void
proto_register_netlink_nl80211(void)
{
#ifndef HAVE_HFI_SECTION_INIT
    static header_field_info *hfi[] = {
        &hfi_nl80211_attr_value,
        &hfi_nl80211_attr_value16,
        &hfi_nl80211_attr_value32,
        &hfi_nl80211_attr_value64,
        &hfi_nl80211_wiphy_name,
        &hfi_nl80211_ifname,
        &hfi_nl80211_mac,
        &hfi_nl80211_alpha2,
        &hfi_nl80211_dbm,
/* Extracted using tools/generate-nl80211-fields.py */
/* Definitions from linux/nl80211.h {{{ */
        &hfi_nl80211_commands,
        &hfi_nl80211_attrs,
        &hfi_nl80211_iftype,
        &hfi_nl80211_sta_flags,
        &hfi_nl80211_sta_p2p_ps_status,
        &hfi_nl80211_rate_info,
        &hfi_nl80211_sta_bss_param,
        &hfi_nl80211_sta_info,
        &hfi_nl80211_tid_stats,
        &hfi_nl80211_txq_stats,
        &hfi_nl80211_mpath_info,
        &hfi_nl80211_band_iftype_attr,
        &hfi_nl80211_band_attr,
        &hfi_nl80211_wmm_rule,
        &hfi_nl80211_frequency_attr,
        &hfi_nl80211_bitrate_attr,
        &hfi_nl80211_reg_initiator,
        &hfi_nl80211_reg_type,
        &hfi_nl80211_reg_rule_attr,
        &hfi_nl80211_sched_scan_match_attr,
        &hfi_nl80211_dfs_regions,
        &hfi_nl80211_user_reg_hint_type,
        &hfi_nl80211_survey_info,
        &hfi_nl80211_mntr_flags,
        &hfi_nl80211_mesh_power_mode,
        &hfi_nl80211_meshconf_params,
        &hfi_nl80211_mesh_setup_params,
        &hfi_nl80211_txq_attr,
        &hfi_nl80211_channel_type,
        &hfi_nl80211_chan_width,
        &hfi_nl80211_bss_scan_width,
        &hfi_nl80211_bss,
        &hfi_nl80211_bss_status,
        &hfi_nl80211_auth_type,
        &hfi_nl80211_key_type,
        &hfi_nl80211_mfp,
        &hfi_nl80211_key_default_types,
        &hfi_nl80211_key_attributes,
        &hfi_nl80211_tx_rate_attributes,
        &hfi_nl80211_ps_state,
        &hfi_nl80211_attr_cqm,
        &hfi_nl80211_tx_power_setting,
        &hfi_nl80211_if_combination_attrs,
        &hfi_nl80211_plink_state,
        &hfi_plink_actions,
        &hfi_nl80211_rekey_data,
        &hfi_nl80211_sta_wme_attr,
        &hfi_nl80211_pmksa_candidate_attr,
        &hfi_nl80211_tdls_operation,
        &hfi_nl80211_connect_failed_reason,
        &hfi_nl80211_timeout_reason,
        &hfi_nl80211_acl_policy,
        &hfi_nl80211_smps_mode,
        &hfi_nl80211_radar_event,
        &hfi_nl80211_dfs_state,
        &hfi_nl80211_crit_proto_id,
        &hfi_nl80211_sched_scan_plan,
        &hfi_nl80211_bss_select_attr,
        &hfi_nl80211_nan_func_attributes,
        &hfi_nl80211_nan_match_attributes,
        &hfi_nl80211_external_auth_action,
/* }}} */
    };
#endif

    static gint *ett[] = {
        &ett_nl80211,
        &ett_nl80211_frame,
        &ett_nl80211_tag,
/* Extracted using tools/generate-nl80211-fields.py */
/* Definitions from linux/nl80211.h {{{ */
        &ett_nl80211_commands,
        &ett_nl80211_attrs,
        &ett_nl80211_iftype,
        &ett_nl80211_sta_flags,
        &ett_nl80211_sta_p2p_ps_status,
        &ett_nl80211_rate_info,
        &ett_nl80211_sta_bss_param,
        &ett_nl80211_sta_info,
        &ett_nl80211_tid_stats,
        &ett_nl80211_txq_stats,
        &ett_nl80211_mpath_info,
        &ett_nl80211_band_iftype_attr,
        &ett_nl80211_band_attr,
        &ett_nl80211_wmm_rule,
        &ett_nl80211_frequency_attr,
        &ett_nl80211_bitrate_attr,
        &ett_nl80211_reg_initiator,
        &ett_nl80211_reg_type,
        &ett_nl80211_reg_rule_attr,
        &ett_nl80211_sched_scan_match_attr,
        &ett_nl80211_dfs_regions,
        &ett_nl80211_user_reg_hint_type,
        &ett_nl80211_survey_info,
        &ett_nl80211_mntr_flags,
        &ett_nl80211_mesh_power_mode,
        &ett_nl80211_meshconf_params,
        &ett_nl80211_mesh_setup_params,
        &ett_nl80211_txq_attr,
        &ett_nl80211_channel_type,
        &ett_nl80211_chan_width,
        &ett_nl80211_bss_scan_width,
        &ett_nl80211_bss,
        &ett_nl80211_bss_status,
        &ett_nl80211_auth_type,
        &ett_nl80211_key_type,
        &ett_nl80211_mfp,
        &ett_nl80211_key_default_types,
        &ett_nl80211_key_attributes,
        &ett_nl80211_tx_rate_attributes,
        &ett_nl80211_ps_state,
        &ett_nl80211_attr_cqm,
        &ett_nl80211_tx_power_setting,
        &ett_nl80211_if_combination_attrs,
        &ett_nl80211_plink_state,
        &ett_plink_actions,
        &ett_nl80211_rekey_data,
        &ett_nl80211_sta_wme_attr,
        &ett_nl80211_pmksa_candidate_attr,
        &ett_nl80211_tdls_operation,
        &ett_nl80211_connect_failed_reason,
        &ett_nl80211_timeout_reason,
        &ett_nl80211_acl_policy,
        &ett_nl80211_smps_mode,
        &ett_nl80211_radar_event,
        &ett_nl80211_dfs_state,
        &ett_nl80211_crit_proto_id,
        &ett_nl80211_sched_scan_plan,
        &ett_nl80211_bss_select_attr,
        &ett_nl80211_nan_func_attributes,
        &ett_nl80211_nan_match_attributes,
        &ett_nl80211_external_auth_action,
/* }}} */
    };

    proto_netlink_nl80211 = proto_register_protocol("Linux 802.11 Netlink", "nl80211", "nl80211");
    hfi_netlink_nl80211 = proto_registrar_get_nth(proto_netlink_nl80211);

    proto_register_fields(proto_netlink_nl80211, hfi, array_length(hfi));
    proto_register_subtree_array(ett, array_length(ett));

    netlink_nl80211_handle = create_dissector_handle(dissect_netlink_nl80211, proto_netlink_nl80211);
    ieee80211_handle = find_dissector_add_dependency("wlan", proto_netlink_nl80211);
    ieee80211_tag_dissector_table = find_dissector_table("wlan.tag.number");
}

void
proto_reg_handoff_netlink_nl80211(void)
{
    dissector_add_string("genl.family", "nl80211", netlink_nl80211_handle);
}

/*
 * Editor modelines  -  https://www.wireshark.org/tools/modelines.html
 *
 * Local variables:
 * c-basic-offset: 4
 * tab-width: 8
 * indent-tabs-mode: nil
 * End:
 *
 * vi: set shiftwidth=4 tabstop=8 expandtab:
 * :indentSize=4:tabSize=8:noTabs=true:
 */
