/* packet-e212.c
 * Routines for output and filtering of E.212 numbers common
 * to many dissectors.
 * Copyright 2006, Anders Broman <anders.broman@ericsson.com>
 *
 * Wireshark - Network traffic analyzer
 * By Gerald Combs <gerald@wireshark.org>
 * Copyright 1998 Gerald Combs
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 */
/*
 * Updates and corrections for Polish networks MCC=260
 * according to www.uke.gov.pl status from January 2011
 * Copyright 2011, Grzegorz Szczytowski <grzegorz.szczytowski@gmail.com>
 */
/* Japanese PLMN codes updated according to http://ja.wikipedia.org/wiki/Mobile_Network_Code
 * as E.212 is still not updated
 * Chinese PLMN codes updated according to http://www.mcc-mnc.com
 */
/* Iranian PLMN codes (MCC 432) updated according to
 * http://mcclist.com/mobile-network-codes-country-codes.asp#Iran
 * due to lack of update in E.212 list
 */

#include "config.h"

#include <stdlib.h>

#include <epan/packet.h>

#include <wsutil/strtoi.h>

#include "packet-e212.h"
#include "expert.h"

void proto_register_e212(void);

/*
 * Annex to ITU Operational Bulletin
 * No. 1121 - 1.IV.2017
 *
 * COMPLEMENT TO ITU-T RECOMMENDATION E.212 (05/2008)
 *
 * Find the bulletins here:
 * http://www.itu.int/pub/T-SP-OB
 */
static const value_string E212_codes[] = {
    {  202, "Greece" },
    {  203, "Unassigned" },
    {  204, "Netherlands" },
    {  205, "Unassigned" },
    {  206, "Belgium" },
    {  207, "Unassigned" },
    {  208, "France" },
    {  209, "Unassigned" },
    {  210, "Unassigned" },
    {  211, "Unassigned" },
    {  212, "Unassigned" },
    {  213, "Andorra" },
    {  214, "Spain" },
    {  215, "Unassigned" },
    {  216, "Hungary" },
    {  217, "Unassigned" },
    {  218, "Bosnia and Herzegovina" },
    {  219, "Croatia" },
    {  220, "Serbia" },
    {  221, "Unassigned" },
    {  222, "Italy" },
    {  223, "Unassigned" },
    {  224, "Unassigned" },
    {  225, "Unassigned" },
    {  226, "Romania" },
    {  227, "Unassigned" },
    {  228, "Switzerland / Liechtenstein" },
    {  229, "Unassigned" },
    {  230, "Czech Rep." },
    {  231, "Slovakia" },
    {  232, "Austria" },
    {  233, "Unassigned" },
    {  234, "United Kingdom" },
    {  235, "United Kingdom" },
    {  236, "Unassigned" },
    {  237, "Unassigned" },
    {  238, "Denmark" },
    {  239, "Unassigned" },
    {  240, "Sweden" },
    {  241, "Unassigned" },
    {  242, "Norway" },
    {  243, "Unassigned" },
    {  244, "Finland" },
    {  245, "Unassigned" },
    {  246, "Lithuania" },
    {  247, "Latvia" },
    {  248, "Estonia" },
    {  249, "Unassigned" },
    {  250, "Russian Federation" },
    {  251, "Unassigned" },
    {  252, "Unassigned" },
    {  253, "Unassigned" },
    {  254, "Unassigned" },
    {  255, "Ukraine" },
    {  256, "Unassigned" },
    {  257, "Belarus" },
    {  258, "Unassigned" },
    {  259, "Moldova (Republic of)" },
    {  260, "Poland" },
    {  261, "Unassigned" },
    {  262, "Germany" },
    {  263, "Unassigned" },
    {  264, "Unassigned" },
    {  265, "Unassigned" },
    {  266, "Gibraltar" },
    {  267, "Unassigned" },
    {  268, "Portugal" },
    {  269, "Unassigned" },
    {  270, "Luxembourg" },
    {  271, "Unassigned" },
    {  272, "Ireland" },
    {  273, "Unassigned" },
    {  274, "Iceland / Faroe Islands" },
    {  275, "Unassigned" },
    {  276, "Albania" },
    {  277, "Unassigned" },
    {  278, "Malta" },
    {  279, "Unassigned" },
    {  280, "Cyprus" },
    {  281, "Unassigned" },
    {  282, "Georgia" },
    {  283, "Unassigned" },
    {  284, "Bulgaria" },
    {  285, "Unassigned" },
    {  286, "Turkey" },
    {  287, "Unassigned" },
    {  288, "Faroe Islands" },
    {  289, "Unassigned" },
    {  290, "Greenland" },
    {  291, "Unassigned" },
    {  292, "San Marino" },
    {  293, "Slovenia" },
    {  294, "The Former Yugoslav Republic of Macedonia" },
    {  295, "Liechtenstein" },
    {  296, "Unassigned" },
    {  297, "Montenegro" },
    {  298, "Unassigned" },
    {  299, "Unassigned" },
    {  300, "Unassigned" },
    {  301, "Unassigned" },
    {  302, "Canada" },
    {  303, "Unassigned" },
    {  304, "Unassigned" },
    {  305, "Unassigned" },
    {  306, "Unassigned" },
    {  307, "Unassigned" },
    {  308, "Saint Pierre and Miquelon" },
    {  309, "Unassigned" },
    {  310, "United States" },
    {  311, "United States" },
    {  312, "United States" },
    {  313, "United States" },
    {  314, "Unassigned" },
    {  315, "Unassigned" },
    {  316, "United States" },
    {  317, "Unassigned" },
    {  318, "Unassigned" },
    {  319, "Unassigned" },
    {  320, "Unassigned" },
    {  321, "Unassigned" },
    {  322, "Unassigned" },
    {  323, "Unassigned" },
    {  324, "Unassigned" },
    {  325, "Unassigned" },
    {  326, "Unassigned" },
    {  327, "Unassigned" },
    {  328, "Unassigned" },
    {  329, "Unassigned" },
    {  330, "Unassigned" },
    {  331, "Unassigned" },
    {  332, "Unassigned" },
    {  333, "Unassigned" },
    {  334, "Mexico" },
    {  335, "Unassigned" },
    {  336, "Unassigned" },
    {  337, "Unassigned" },
    {  338, "Jamaica" },
    {  339, "Unassigned" },
    {  340, "French Guiana / Guadeloupe / Martinique" },
    {  341, "Unassigned" },
    {  342, "Barbados" },
    {  343, "Unassigned" },
    {  344, "Antigua and Barbuda" },
    {  345, "Unassigned" },
    {  346, "Cayman Islands" },
    {  347, "Unassigned" },
    {  348, "British Virgin Islands" },
    {  349, "Unassigned" },
    {  350, "Bermuda" },
    {  351, "Unassigned" },
    {  352, "Grenada" },
    {  353, "Unassigned" },
    {  354, "Montserrat" },
    {  355, "Unassigned" },
    {  356, "Saint Kitts and Nevis" },
    {  357, "Unassigned" },
    {  358, "Saint Lucia" },
    {  359, "Unassigned" },
    {  360, "Saint Vincent and the Grenadines" },
    {  361, "Unassigned" },
    {  362, "Curacao" },
    {  363, "Aruba" },
    {  364, "Bahamas" },
    {  365, "Anguilla" },
    {  366, "Dominica" },
    {  367, "Unassigned" },
    {  368, "Cuba" },
    {  369, "Unassigned" },
    {  370, "Dominican Rep." },
    {  371, "Unassigned" },
    {  372, "Haiti" },
    {  373, "Unassigned" },
    {  374, "Trinidad and Tobago" },
    {  375, "Unassigned" },
    {  376, "Turks and Caicos Islands" },
    {  377, "Unassigned" },
    {  378, "Unassigned" },
    {  379, "Unassigned" },
    {  380, "Unassigned" },
    {  381, "Unassigned" },
    {  382, "Unassigned" },
    {  383, "Unassigned" },
    {  384, "Unassigned" },
    {  385, "Unassigned" },
    {  386, "Unassigned" },
    {  387, "Unassigned" },
    {  388, "Unassigned" },
    {  389, "Unassigned" },
    {  390, "Unassigned" },
    {  391, "Unassigned" },
    {  392, "Unassigned" },
    {  393, "Unassigned" },
    {  394, "Unassigned" },
    {  395, "Unassigned" },
    {  396, "Unassigned" },
    {  397, "Unassigned" },
    {  398, "Unassigned" },
    {  399, "Unassigned" },
    {  400, "Azerbaijan" },
    {  401, "Kazakhstan" },
    {  402, "Bhutan" },
    {  403, "Unassigned" },
    {  404, "India" },
    {  405, "India" },
    {  406, "Unassigned" },
    {  407, "Unassigned" },
    {  408, "Unassigned" },
    {  409, "Unassigned" },
    {  410, "Pakistan" },
    {  411, "Unassigned" },
    {  412, "Afghanistan" },
    {  413, "Sri Lanka" },
    {  414, "Myanmar" },
    {  415, "Lebanon" },
    {  416, "Jordan" },
    {  417, "Syrian Arab Republic" },
    {  418, "Iraq" },
    {  419, "Kuwait" },
    {  420, "Saudi Arabia" },
    {  421, "Yemen" },
    {  422, "Oman" },
    {  423, "Unassigned" },
    {  424, "United Arab Emirates" },
    {  425, "Israel" },
    {  426, "Bahrain" },
    {  427, "Qatar" },
    {  428, "Mongolia" },
    {  429, "Nepal" },
    {  430, "Unassigned" },
    {  431, "Unassigned" },
    {  432, "Iran (Islamic Republic of)" },
    {  433, "Unassigned" },
    {  434, "Uzbekistan" },
    {  435, "Unassigned" },
    {  436, "Tajikistan" },
    {  437, "Kyrgyzstan" },
    {  438, "Turkmenistan" },
    {  439, "Unassigned" },
    {  440, "Japan" },
    {  441, "Japan" },
    {  442, "Unassigned" },
    {  443, "Unassigned" },
    {  444, "Unassigned" },
    {  445, "Unassigned" },
    {  446, "Unassigned" },
    {  447, "Unassigned" },
    {  448, "Unassigned" },
    {  449, "Unassigned" },
    {  450, "Korea (Rep. of)" },
    {  451, "Unassigned" },
    {  452, "Viet Nam" },
    {  453, "Unassigned" },
    {  454, "Hong Kong, China" },
    {  455, "Macao, China" },
    {  456, "Cambodia" },
    {  457, "Lao P.D.R." },
    {  458, "Unassigned" },
    {  459, "Unassigned" },
    {  460, "China" },
    {  461, "Unassigned" },
    {  462, "Unassigned" },
    {  463, "Unassigned" },
    {  464, "Unassigned" },
    {  465, "Unassigned" },
    {  466, "Unassigned" },
    {  467, "Unassigned" },
    {  468, "Unassigned" },
    {  469, "Unassigned" },
    {  470, "Bangladesh" },
    {  471, "Unassigned" },
    {  472, "Maldives" },
    {  473, "Unassigned" },
    {  474, "Unassigned" },
    {  475, "Unassigned" },
    {  476, "Unassigned" },
    {  477, "Unassigned" },
    {  478, "Unassigned" },
    {  479, "Unassigned" },
    {  480, "Unassigned" },
    {  481, "Unassigned" },
    {  482, "Unassigned" },
    {  483, "Unassigned" },
    {  484, "Unassigned" },
    {  485, "Unassigned" },
    {  486, "Unassigned" },
    {  487, "Unassigned" },
    {  488, "Unassigned" },
    {  489, "Unassigned" },
    {  490, "Unassigned" },
    {  491, "Unassigned" },
    {  492, "Unassigned" },
    {  493, "Unassigned" },
    {  494, "Unassigned" },
    {  495, "Unassigned" },
    {  496, "Unassigned" },
    {  497, "Unassigned" },
    {  498, "Unassigned" },
    {  499, "Unassigned" },
    {  500, "Unassigned" },
    {  501, "Unassigned" },
    {  502, "Malaysia" },
    {  503, "Unassigned" },
    {  504, "Unassigned" },
    {  505, "Australia" },
    {  506, "Unassigned" },
    {  507, "Unassigned" },
    {  508, "Unassigned" },
    {  509, "Unassigned" },
    {  510, "Indonesia" },
    {  511, "Unassigned" },
    {  512, "Unassigned" },
    {  513, "Unassigned" },
    {  514, "Timor-Leste" },
    {  515, "Philippines" },
    {  516, "Unassigned" },
    {  517, "Unassigned" },
    {  518, "Unassigned" },
    {  519, "Unassigned" },
    {  520, "Thailand" },
    {  521, "Unassigned" },
    {  522, "Unassigned" },
    {  523, "Unassigned" },
    {  524, "Unassigned" },
    {  525, "Singapore" },
    {  526, "Unassigned" },
    {  527, "Unassigned" },
    {  528, "Brunei Darussalam" },
    {  529, "Unassigned" },
    {  530, "New Zealand" },
    {  531, "Unassigned" },
    {  532, "Unassigned" },
    {  533, "Unassigned" },
    {  534, "Unassigned" },
    {  535, "Unassigned" },
    {  536, "Unassigned" },
    {  537, "Papua New Guinea" },
    {  538, "Unassigned" },
    {  539, "Tonga" },
    {  540, "Solomon Islands" },
    {  541, "Vanuatu" },
    {  542, "Fiji / Nauru" },
    {  543, "Wallis and Futuna" },
    {  544, "Unassigned" },
    {  545, "Kiribati" },
    {  546, "New Caledonia" },
    {  547, "French Polynesia" },
    {  548, "Cook Islands" },
    {  549, "Samoa" },
    {  550, "Micronesia" },
    {  551, "Unassigned" },
    {  552, "Palau" },
    {  553, "Tuvalu" },
    {  554, "Tokelau" },
    {  555, "Niue" },
    {  556, "Unassigned" },
    {  557, "Unassigned" },
    {  558, "Unassigned" },
    {  559, "Unassigned" },
    {  560, "Unassigned" },
    {  561, "Unassigned" },
    {  562, "Unassigned" },
    {  563, "Unassigned" },
    {  564, "Unassigned" },
    {  565, "Unassigned" },
    {  566, "Unassigned" },
    {  567, "Unassigned" },
    {  568, "Unassigned" },
    {  569, "Unassigned" },
    {  570, "Unassigned" },
    {  571, "Unassigned" },
    {  572, "Unassigned" },
    {  573, "Unassigned" },
    {  574, "Unassigned" },
    {  575, "Unassigned" },
    {  576, "Unassigned" },
    {  577, "Unassigned" },
    {  578, "Unassigned" },
    {  579, "Unassigned" },
    {  580, "Unassigned" },
    {  581, "Unassigned" },
    {  582, "Unassigned" },
    {  583, "Unassigned" },
    {  584, "Unassigned" },
    {  585, "Unassigned" },
    {  586, "Unassigned" },
    {  587, "Unassigned" },
    {  588, "Unassigned" },
    {  589, "Unassigned" },
    {  590, "Unassigned" },
    {  591, "Unassigned" },
    {  592, "Unassigned" },
    {  593, "Unassigned" },
    {  594, "Unassigned" },
    {  595, "Unassigned" },
    {  596, "Unassigned" },
    {  597, "Unassigned" },
    {  598, "Unassigned" },
    {  599, "Unassigned" },
    {  600, "Unassigned" },
    {  601, "Unassigned" },
    {  602, "Egypt" },
    {  603, "Algeria" },
    {  604, "Morocco" },
    {  605, "Tunisia" },
    {  606, "Unassigned" },
    {  607, "Gambia" },
    {  608, "Senegal" },
    {  609, "Mauritania" },
    {  610, "Mali" },
    {  611, "Guinea" },
    {  612, "Cote d'Ivoire" },
    {  613, "Burkina Faso" },
    {  614, "Niger" },
    {  615, "Togo" },
    {  616, "Benin" },
    {  617, "Mauritius" },
    {  618, "Liberia" },
    {  619, "Sierra Leone" },
    {  620, "Ghana" },
    {  621, "Nigeria" },
    {  622, "Chad" },
    {  623, "Central African Rep." },
    {  624, "Cameroon" },
    {  625, "Cabo Verde" },
    {  626, "Sao Tome and Principe" },
    {  627, "Equatorial Guinea" },
    {  628, "Gabon" },
    {  629, "Congo" },
    {  630, "Dem. Rep. of the Congo" },
    {  631, "Angola" },
    {  632, "Guinea-Bissau" },
    {  633, "Seychelles" },
    {  634, "Sudan" },
    {  635, "Rwanda" },
    {  636, "Ethiopia" },
    {  637, "Unassigned" },
    {  638, "Djibouti" },
    {  639, "Kenya" },
    {  640, "Tanzania" },
    {  641, "Uganda" },
    {  642, "Burundi" },
    {  643, "Mozambique" },
    {  644, "Unassigned" },
    {  645, "Zambia" },
    {  646, "Madagascar" },
    {  647, "French Departments and Territories in the Indian Ocean" },
    {  648, "Zimbabwe" },
    {  649, "Namibia" },
    {  650, "Malawi" },
    {  651, "Lesotho" },
    {  652, "Botswana" },
    {  653, "Swaziland" },
    {  654, "Comoros" },
    {  655, "South Africa" },
    {  656, "Unassigned" },
    {  657, "Unassigned" },
    {  658, "Saint Helena, Ascension and Tristan da Cunha" },
    {  659, "South Sudan" },
    {  660, "Unassigned" },
    {  661, "Unassigned" },
    {  662, "Unassigned" },
    {  663, "Unassigned" },
    {  664, "Unassigned" },
    {  665, "Unassigned" },
    {  666, "Unassigned" },
    {  667, "Unassigned" },
    {  668, "Unassigned" },
    {  669, "Unassigned" },
    {  670, "Unassigned" },
    {  671, "Unassigned" },
    {  672, "Unassigned" },
    {  673, "Unassigned" },
    {  674, "Unassigned" },
    {  675, "Unassigned" },
    {  676, "Unassigned" },
    {  677, "Unassigned" },
    {  678, "Unassigned" },
    {  679, "Unassigned" },
    {  680, "Unassigned" },
    {  681, "Unassigned" },
    {  682, "Unassigned" },
    {  683, "Unassigned" },
    {  684, "Unassigned" },
    {  685, "Unassigned" },
    {  686, "Unassigned" },
    {  687, "Unassigned" },
    {  688, "Unassigned" },
    {  689, "Unassigned" },
    {  690, "Unassigned" },
    {  691, "Unassigned" },
    {  692, "Unassigned" },
    {  693, "Unassigned" },
    {  694, "Unassigned" },
    {  695, "Unassigned" },
    {  696, "Unassigned" },
    {  697, "Unassigned" },
    {  698, "Unassigned" },
    {  699, "Unassigned" },
    {  700, "Unassigned" },
    {  701, "Unassigned" },
    {  702, "Belize" },
    {  703, "Unassigned" },
    {  704, "Guatemala" },
    {  705, "Unassigned" },
    {  706, "El Salvador" },
    {  707, "Unassigned" },
    {  708, "Honduras" },
    {  709, "Unassigned" },
    {  710, "Nicaragua" },
    {  711, "Unassigned" },
    {  712, "Costa Rica" },
    {  713, "Unassigned" },
    {  714, "Panama" },
    {  715, "Unassigned" },
    {  716, "Peru" },
    {  717, "Unassigned" },
    {  718, "Unassigned" },
    {  719, "Unassigned" },
    {  720, "Unassigned" },
    {  721, "Unassigned" },
    {  722, "Argentina" },
    {  723, "Unassigned" },
    {  724, "Brazil" },
    {  725, "Unassigned" },
    {  726, "Unassigned" },
    {  727, "Unassigned" },
    {  728, "Unassigned" },
    {  729, "Unassigned" },
    {  730, "Chile" },
    {  731, "Unassigned" },
    {  732, "Colombia" },
    {  733, "Unassigned" },
    {  734, "Venezuela (Bolivarian Republic of)" },
    {  735, "Unassigned" },
    {  736, "Bolivia (Plurinational State of)" },
    {  737, "Unassigned" },
    {  738, "Guyana" },
    {  739, "Unassigned" },
    {  740, "Ecuador" },
    {  741, "Unassigned" },
    {  742, "Unassigned" },
    {  743, "Unassigned" },
    {  744, "Paraguay" },
    {  745, "Unassigned" },
    {  746, "Suriname" },
    {  747, "Unassigned" },
    {  748, "Uruguay" },
    {  749, "Unassigned" },
    {  750, "Falkland Islands (Malvinas)" },
    {  751, "Unassigned" },
    {  752, "Unassigned" },
    {  753, "Unassigned" },
    {  754, "Unassigned" },
    {  755, "Unassigned" },
    {  756, "Unassigned" },
    {  757, "Unassigned" },
    {  758, "Unassigned" },
    {  759, "Unassigned" },
    {  760, "Unassigned" },
    {  761, "Unassigned" },
    {  762, "Unassigned" },
    {  763, "Unassigned" },
    {  764, "Unassigned" },
    {  765, "Unassigned" },
    {  766, "Unassigned" },
    {  767, "Unassigned" },
    {  768, "Unassigned" },
    {  769, "Unassigned" },
    {  770, "Unassigned" },
    {  771, "Unassigned" },
    {  772, "Unassigned" },
    {  773, "Unassigned" },
    {  774, "Unassigned" },
    {  775, "Unassigned" },
    {  776, "Unassigned" },
    {  777, "Unassigned" },
    {  778, "Unassigned" },
    {  779, "Unassigned" },
    {  780, "Unassigned" },
    {  781, "Unassigned" },
    {  782, "Unassigned" },
    {  783, "Unassigned" },
    {  784, "Unassigned" },
    {  785, "Unassigned" },
    {  786, "Unassigned" },
    {  787, "Unassigned" },
    {  788, "Unassigned" },
    {  789, "Unassigned" },
    {  790, "Unassigned" },
    {  791, "Unassigned" },
    {  792, "Unassigned" },
    {  793, "Unassigned" },
    {  794, "Unassigned" },
    {  795, "Unassigned" },
    {  796, "Unassigned" },
    {  797, "Unassigned" },
    {  798, "Unassigned" },
    {  799, "Unassigned" },
    {  800, "Unassigned" },
    {  801, "Unassigned" },
    {  802, "Unassigned" },
    {  803, "Unassigned" },
    {  804, "Unassigned" },
    {  805, "Unassigned" },
    {  806, "Unassigned" },
    {  807, "Unassigned" },
    {  808, "Unassigned" },
    {  809, "Unassigned" },
    {  810, "Unassigned" },
    {  811, "Unassigned" },
    {  812, "Unassigned" },
    {  813, "Unassigned" },
    {  814, "Unassigned" },
    {  815, "Unassigned" },
    {  816, "Unassigned" },
    {  817, "Unassigned" },
    {  818, "Unassigned" },
    {  819, "Unassigned" },
    {  820, "Unassigned" },
    {  821, "Unassigned" },
    {  822, "Unassigned" },
    {  823, "Unassigned" },
    {  824, "Unassigned" },
    {  825, "Unassigned" },
    {  826, "Unassigned" },
    {  827, "Unassigned" },
    {  828, "Unassigned" },
    {  829, "Unassigned" },
    {  830, "Unassigned" },
    {  831, "Unassigned" },
    {  832, "Unassigned" },
    {  833, "Unassigned" },
    {  834, "Unassigned" },
    {  835, "Unassigned" },
    {  836, "Unassigned" },
    {  837, "Unassigned" },
    {  838, "Unassigned" },
    {  839, "Unassigned" },
    {  840, "Unassigned" },
    {  841, "Unassigned" },
    {  842, "Unassigned" },
    {  843, "Unassigned" },
    {  844, "Unassigned" },
    {  845, "Unassigned" },
    {  846, "Unassigned" },
    {  847, "Unassigned" },
    {  848, "Unassigned" },
    {  849, "Unassigned" },
    {  850, "Unassigned" },
    {  851, "Unassigned" },
    {  852, "Unassigned" },
    {  853, "Unassigned" },
    {  854, "Unassigned" },
    {  855, "Unassigned" },
    {  856, "Unassigned" },
    {  857, "Unassigned" },
    {  858, "Unassigned" },
    {  859, "Unassigned" },
    {  860, "Unassigned" },
    {  861, "Unassigned" },
    {  862, "Unassigned" },
    {  863, "Unassigned" },
    {  864, "Unassigned" },
    {  865, "Unassigned" },
    {  866, "Unassigned" },
    {  867, "Unassigned" },
    {  868, "Unassigned" },
    {  869, "Unassigned" },
    {  870, "Unassigned" },
    {  871, "Unassigned" },
    {  872, "Unassigned" },
    {  873, "Unassigned" },
    {  874, "Unassigned" },
    {  875, "Unassigned" },
    {  876, "Unassigned" },
    {  877, "Unassigned" },
    {  878, "Unassigned" },
    {  879, "Unassigned" },
    {  880, "Unassigned" },
    {  881, "Unassigned" },
    {  882, "Unassigned" },
    {  883, "Unassigned" },
    {  884, "Unassigned" },
    {  885, "Unassigned" },
    {  886, "Unassigned" },
    {  887, "Unassigned" },
    {  888, "Unassigned" },
    {  889, "Unassigned" },
    {  890, "Unassigned" },
    {  891, "Unassigned" },
    {  892, "Unassigned" },
    {  893, "Unassigned" },
    {  894, "Unassigned" },
    {  895, "Unassigned" },
    {  896, "Unassigned" },
    {  897, "Unassigned" },
    {  898, "Unassigned" },
    {  899, "Unassigned" },
    {  900, "Unassigned" },
    {  901, "International Mobile, shared code" },
    { 1665, "Unset" },
    { 0, NULL }
};

value_string_ext E212_codes_ext = VALUE_STRING_EXT_INIT(E212_codes);

/*
 * Annex to ITU Operational Bulletin No. 1019 - 1.I.2013
 *
 * According to Recommendation ITU-T E.212 (05/2008)
 */

static const value_string mcc_mnc_2digits_codes[] = {
    { 20201, "Cosmote AE" },
    { 20202, "Cosmote AE" },
    { 20203, "OTE AE" },
    { 20204, "OTE AE" },
    { 20205, "Vodafone - Panafon" },
    { 20206, "COSMOLINE" },
    { 20207, "AMD TELECOM AE" },
    { 20209, "WIND HELLAS TELECOMMUNICATIONS" },
    { 20210, "WIND HELLAS TELECOMMUNICATIONS" },
    { 20211, "INTERCONNECT" },
    { 20212, "YUBOTO" },
    { 20213, "COMPATEL LIMITED" },
    { 20214, "CYTA (HELLAS)" },
    { 20215, "BWS" },
    { 20216, "INTER TELECOM" },
    { 20401, "RadioAccess Network Services B.V." },
    { 20402, "Tele2 Nederland B.V." },
    { 20403, "Voiceworks B.V." },
    { 20404, "Vodafone Libertel B.V." },
    { 20405, "Elephant Talk Communications Premium Rate Services" },
    { 20406, "Mundio Mobile (Netherlands) Ltd" },
    { 20407, "Teleena holding B.V." },
    { 20408, "KPN Mobile The Netherlands B.V." },
    { 20409, "Lycamobile Netherlands Limited" },
    { 20410, "KPN B.V." },
    { 20412, "KPN B.V." },
    { 20413, "Unica Installatietechniek B.V." },
    { 20415, "Ziggo B.V." },
    { 20416, "T-Mobile Netherlands B.V." },
    { 20417, "Intercity Mobile Communications B.V." },
    { 20418, "UPC Nederland B.V." },
    { 20419, "Mixe Communication Solutions B.V." },
    { 20420, "T-Mobile Netherlands B.V." },
    { 20421, "ProRail B.V." },
    { 20422, "Ministerie van Defensie" },
    { 20423, "ASpider Solutions Nederland B.V." },
    { 20424, "Private Mobility Nederland B.V." },
    { 20425, "CAPX B.V." },
    { 20426, "SpeakUp B.V." },
    { 20428, "Lancelot B.V." },
    { 20429, "Private Mobile Ltd" },
    { 20460, "Nextgen Mobile Ltd" },
    { 20461, "BodyTrace Netherlands B.V." },
    { 20464, "Zetacom B.V." },
    { 20465, "AGMS Netherlands B.V." },
    { 20466, "Utility Connect B.V." },
    { 20467, "RadioAccess B.V." },
    { 20468, "Roamware (Netherlands) B.V." },
    { 20469, "KPN Mobile The Netherlands B.V." },
    { 20600, "Proximus" },
    { 20601, "Proximus" },
    { 20602, "N.M.B.S" },
    { 20605, "Telenet nv" },
    { 20606, "Lycamobile sprl" },
    { 20607, "Mundio Mobile Belgium nv" },
    { 20610, "Mobistar sa" },
    { 20620, "Base Company nv" },
    { 20640, "Join Experience (Belgium)" },
    { 20801, "Orange" },
    { 20802, "Orange" },
    { 20803, "MobiquiThings" },
    { 20804, "Sisteer" },
    { 20805, "Globalstar Europe" },
    { 20806, "Globalstar Europe" },
    { 20807, "Globalstar Europe" },
    { 20808, "Completel" },
    { 20809, "S.F.R." },
    { 20810, "S.F.R." },
    { 20811, "S.F.R." },
    { 20812, "Hewlett-Packard France" },
    { 20813, "S.F.R." },
    { 20814, "RFF" },
    { 20815, "Free Mobile" },
    { 20816, "Free Mobile" },
    { 20817, "Legos" },
    { 20818, "Voxbone" },
    { 20820, "Bouygues Telecom" },
    { 20821, "Bouygues Telecom" },
    { 20822, "Transatel" },
    { 20823, "Omea Telecom" },
    { 20824, "MobiquiThings" },
    { 20825, "Lycamobile" },
    { 20826, "Euro-Information telecom" },
    { 20827, "Afone" },
    { 20828, "Airbus defense ans space SAS" },
    { 20829, "Societe International Mobile Communication" },
    { 20830, "Symacom" },
    { 20831, "Mundio Mobile" },
    { 20888, "Bouygues Telecom" },
    { 20889, "Omer Telecom Ltd" },
    { 20890, "Images & Reseaux" },
    { 20891, "Orange" },
    { 20892, "Plate-forme telecom" },
    { 20894, "Halys" },
    { 21303, "Mobiland" },
    { 21401, "Vodafone Espana, SAU" },
    { 21403, "France Telecom Espana, SA" },
    { 21404, "Xfera Moviles, S.A." },
    { 21405, "Telefonica Moviles Espana, SAU" },
    { 21406, "Vodafone Espana, SAU" },
    { 21407, "Telefonica Moviles Espana, SAU" },
    { 21408, "Euskaltel, SA" },
    { 21409, "France Telecom Espana, SA" },
    { 21410, "Operadora de Telecomunicaciones Opera SL" },
    { 21411, "France Telecom Espana, SA" },
    { 21412, "Contacta Servicios Avanzados de Telecomunicaciones SL" },
    { 21413, "Incotel Ingeniera y Consultaria SL" },
    { 21414, "Incotel Servicioz Avanzados SL" },
    { 21415, "BT Espana Compania de Servicios Globales de Telecomunicaciones, SAU" },
    { 21416, "Telecable de Asturias, SAU" },
    { 21417, "R Cable y Telecomunicaciones Galicia, SA" },
    { 21418, "Cableuropa, SAU" },
    { 21419, "E-Plus Moviles, SL" },
    { 21420, "Fonyou Telecom, SL" },
    { 21421, "ORANGE ESPAGNE, S.A. UNIPERSONAL" },
    { 21422, "Best Spain Telecom, SL" },
    { 21424, "Vizzavi Espana, S.L." },
    { 21425, "Lycamobile, SL" },
    { 21426, "Lleida Networks Serveis Telematics, SL" },
    { 21427, "SCN Truphone SL" },
    { 21428, "Consorcio de Telecomunicaciones Avanzadas, S.A." },
    { 21429, "NEO-SKY 2002, S.A." },
    { 21430, "Compatel Limited" },
    { 21431, "Red Digital De Telecomunicaciones de las Islas Baleares, S.L." },
    { 21432, "TUENTI TECHNOLOGIES, S.L." },
    { 21433, "EURONA WIRELESS TELECOM, S.A." },
    { 21434, "AIRE NETWORKS DEL MEDITERRANEO, S.L. UNIPERSONAL" },
    { 21435, "INGENIUM OUTSOURCING SERVICES, S.L." },
    { 21436, "OPEN CABLE TELECOMUNICACIONES, S.L."},
    { 21601, "Telenor Hungary Ltd" },
    { 21602, "MVM NET Ltd." },
    { 21603, "DIGI Telecommunication Ltd." },
    { 21630, "Magyar Telecom Plc" },
    { 21670, "Vodafone" },
    { 21671, "UPC Hungary Ltd" },
    { 21699, "MAV Co." },
    { 21803, "Eronet Mobile Communications Ltd." },
    { 21805, "MOBI'S (Mobilina Srpske)" },
    { 21890, "GSMBIH" },
    { 21901, "T-Mobile Hrvatska d.o.o./T-Mobile Croatia LLC" },
    { 21902, "Tele2/Tele2 d.o.o." },
    { 21910, "VIPnet/VIPnet d.o.o." },
    { 22001, "Telenor d.o.o." },
    { 22003, "Telekom Srbija a.d." },
    { 22005, "Vip mobile d.o.o." },
    { 22007, "Orion telekom d.o.o." },
    { 22009, "MUNDIO MOBILE d.o.o." },
    { 22011, "GLOBALTEL d.o.o." },
    { 22201, "Telecom Italia Mobile (TIM)" },
    { 22202, "Elsacom" },
    { 22210, "Omnitel Pronto Italia (OPI)" },
    { 22277, "IPSE 2000" },
    { 22288, "Wind" },
    { 22298, "Blu" },
    { 22299, "H3G" },
    { 22601, "Vodafone" },
    { 22602, "Romtelecom" },
    { 22603, "Cosmote" },
    { 22604, "Cosmote" },
    { 22605, "Digi.Mobil" },
    { 22606, "Cosmote" },
    { 22610, "Orange" },
    { 22611, "Enigma-System" },
    { 22801, "Swisscom Schweiz AG" },
    { 22802, "Sunrise Communications AG" },
    { 22803, "Salt Mobile SA" },
    { 22805, "Comfone AG" },
    { 22806, "SBB AG" },
    { 22808, "TelCommunication Services AG" },
    { 22809, "Comfone AG" },
    { 22812, "Sunrise Communications AG" },
    { 22851, "Bebbicell AG" },
    { 22853, "upc Cablecom GmbH" },
    { 22854, "Lycamobile AG" },
    { 22855, "WeMobile SA" },
    { 22856, "SMSRelay AG" },
    { 22857, "Mitto AG" },
    { 22858, "Beeone Communications SA" },
    { 22859, "Mundio Mobile Limited" },
    { 22861, "Compatel Ltd" },
    { 23001, "T-Mobile Czech Republic a.s." },
    { 23002, "Telefonica O2 Czech Republic a.s." },
    { 23003, "Vodafone Czech Republic a.s." },
    { 23004, "Air Telecom a.s." },
    { 23005, "Travel Telekommunikation, s.r.o." },
    { 23007, "ASTELNET s.r.o" },
    { 23008, "Compatel s.r.o" },
    { 23009, "Mundio Distribution Czech Republic s.r.o." },
    { 23098, "Sprava Zeleznicni Dopravni Cesty" },
    { 23101, "Orange, GSM" },
    { 23102, "Eurotel, GSM & NMT" },
    { 23104, "Eurotel, UMTS" },
    { 23105, "Orange, UMTS" },
    { 23201, "A1 Telekom Austria AG" },
    { 23202, "A1 Telekom Austria AG" },
    { 23203, "T-Mobile Austria GmbH" },
    { 23204, "T-Mobile Austria GmbH" },
    { 23205, "Hutchison Drei Austria GmbH" },
    { 23206, "Orange Austria Telecommunication GmbH" },
    { 23207, "T-Mobile Austria GmbH" },
    { 23208, "Lycamobile Austria Ltd" },
    { 23209, "A1 Telekom Austria AG" },
    { 23210, "Hutchison Drei Austria GmbH" },
    { 23211, "A1 Telekom Austria AG" },
    { 23212, "A1 Telekom Austria AG" },
    { 23213, "UPC Austria Services GmbH" },
    { 23214, "Hutchison Drei Austria GmbH" },
    { 23215, "Mundio Mobile (Austria) Ltd" },
    { 23216, "Hutchison Drei Austria GmbH" },
    { 23217, "MASS Response Service GmbH" },
    { 23218, "smartspace GmbH" },
    { 23219, "Tele2 Telecommunication GmbH" },
    { 23220, "Mtel Austrija GmbH" },
    { 23291, "OBB - Infrastruktur AG" },
    { 23400, "British Telecom" },
    { 23401, "Mundio Mobile Limited" },
    { 23402, "Telefonica UK Limited" },
    { 23403, "Jersey Airtel Limited" },
    { 23404, "FMS Solutions Ltd" },
    { 23405, "COLT Mobile Telecommunications Limited" },
    { 23406, "Internet Computer Bureau Limited" },
    { 23407, "Vodafone Ltd (C&W)" },
    { 23408, "BT OnePhone Limited" },
    { 23409, "Tismi B.V." },
    { 23410, "Telefonica UK Limited" },
    { 23411, "Telefonica UK Limited" },
    { 23412, "Network Rail Infrastructure Limited" },
    { 23413, "Network Rail Infrastructure Limited" },
    { 23414, "HAY SYSTEMS LIMITED" },
    { 23415, "Vodafone Uk Ltd" },
    { 23416, "TalkTalk Communications Limited" },
    { 23417, "FleXtel Limited" },
    { 23418, "Cloud9 Communications Limited" },
    { 23419, "TeleWare PLC" },
    { 23420, "Hutchison 3G UK Ltd" },
    { 23421, "LogicStar Ltd" },
    { 23422, "Telesign Mobile Limited" },
    { 23423, "Icron Network Limited" },
    { 23424, "Stour Marine Limited" },
    { 23425, "Truphone Ltd" },
    { 23426, "Lycamobile UK Limited" },
    { 23427, "Teleena UK Limited" },
    { 23428, "Marathon Telecom Limited" },
    { 23429, "(aq) Limited trading as aql" },
    { 23430, "EE Limited ( TM)" },
    { 23431, "EE Limited ( TM)" },
    { 23432, "EE Limited ( TM)" },
    { 23433, "Orange" },
    { 23434, "Orange" },
    { 23435, "JSC Ingenium (UK) Limited" },
    { 23436, "Sure (Isle of Man) Limited" },
    { 23437, "Synectiv Ltd" },
    { 23438, "Virgin Mobile Telecoms Limited" },
    { 23439, "Gamma Telecom Holdings Ltd" },
    { 23450, "Jersey Telecom" },
    { 23451, "UK Broadband Limited" },
    { 23452, "Shyam Telecom UK Ltd" },
    { 23453, "Limitless Mobile Ltd" },
    { 23454, "The Carphone Warehouse Limited" },
    { 23455, "Sure (Guernsey) Limited" },
    { 23456, "CESG" },
    { 23457, "Sky UK Limited" },
    { 23458, "Manx Telecom" },
    { 23459, "Limitless Mobile Ltd" },
    { 23470, "AMSUK Limited" },
    { 23471, "Home Office" },
    { 23476, "British Telecom" },
    { 23478, "Airwave Solutions Ltd" },
    { 23486, "EE Limited ( TM)" },
    { 23500, "Mundio Mobile Limited" },
    { 23501, "EE Limited ( TM)" },
    { 23502, "EE Limited ( TM)" },
    { 23503, "UK Broadband Limited" },
    { 23577, "British Telecom" },
    { 23591, "Vodafone Uk Ltd" },
    { 23592, "Vodafone Ltd (C&W)" },
    { 23594, "Hutchison 3G UK Ltd" },
    { 23595, "Network Rail Infrastructure Limited" },
    { 23801, "TDC Mobil" },
    { 23802, "Sonofon" },
    { 23803, "MIGway A/S" },
    { 23804, "NextGen Mobile Ltd T/A CardBoardFish" },
    { 23806, "Hi3G" },
    { 23807, "Mundio Mobile (Denmark) Limited" },
    { 23808, "Nordisk Mobiltelefon Danmark A/S" },
    { 23810, "TDC Mobil" },
    { 23811, "Dansk Beredskabskommunikation A/S" },
    { 23812, "Lycamobile Denmark" },
    { 23813, "Compatel Limited" },
    { 23814, "Monty UK Global Limited" },
    { 23815, "Ice Danmark ApS" },
    { 23816, "Tismi B.V." },
    { 23817, "Naka AG" },
    { 23818, "Cubic Telecom" },
    { 23820, "Telia" },
    { 23825, "Viahub (SMS Provider Corp.)" },
    { 23828, "CoolTEL" },
    { 23830, "Interactive Digital Media GmbH" },
    { 23842, "Brandtel ApS" },
    { 23843, "MobiWeb Limited" },
    { 23866, "TT-Netvaerket P/S" },
    { 23877, "Tele2" },
    { 24001, "Telia Sonera Sverige AB" },
    { 24002, "H3G Access AB" },
    { 24003, "Netett Sverige AB" },
    { 24004, "3G Infrastructure Services AB" },
    { 24005, "Svenska UMTS-Nat AB" },
    { 24006, "Telenor Sverige AB" },
    { 24008, "Telenor Sverige AB" },
    { 24009, "Com4 Sweden AB" },
    { 24010, "Tele2 Sverige AB" },
    { 24011, "ComHem AB" },
    { 24012, "Lycamobile Sweden Limited" },
    { 24013, "Alltele Foretag Sverige AB" },
    { 24014, "Tele2 Business AB" },
    { 24015, "Wireless Maingate Nordic AB" },
    { 24016, "42 Telecom AB" },
    { 24017, "Gotalandsnatet AB" },
    { 24018, "Generic Mobile Systems Sweden AB" },
    { 24019, "Mundio Mobile Sweden Ltd" },
    { 24020, "iMEZ AB" },
    { 24021, "Trafikverket ICT" },
    { 24022, "EuTel AB" },
    { 24023, "Infobip LTD (UK)" },
    { 24024, "Net4Mobility HB" },
    { 24025, "Monty UK Global Limited" },
    { 24026, "Beepsend A.B." },
    { 24027, "GlobeTouch AB" },
    { 24028, "CoolTEL Aps" },
    { 24029, "Mercury International Carrier Services" },
    { 24030, "NextGen Mobile Ltd" },
    { 24031, "RebTel Networks AB" },
    { 24032, "Compatel Ltd." },
    { 24033, "Mobile Arts AB" },
    { 24034, "Pro Net Telecommunications Services Ltd." },
    { 24035, "42 Telecom LTD" },
    { 24036, "Interactive Digital Media GmbH" },
    { 24037, "CLX Networks AB" },
    { 24038, "Voxbone SA" },
    { 24039, "Borderlight AB" },
    { 24040, "ReWiCom Scandinavia AB" },
    { 24041, "Shyam Telecom UK Ltd" },
    { 24042, "Telenor Connexion AB" },
    { 24043, "MobiWeb Ltd." },
    { 24044, "Limitless Mobile AB" },
    { 24045, "Spirius AB" },
    { 24046, "SMS Provider Corp." },
    { 24060, "Telefonaktiebolaget LM Ericsson" },
    { 24065, "shared use for closed networks" },
    { 24066, "shared use for closed networks" },
    { 24067, "shared use for test purpose" },
    { 24068, "shared use for test purpose" },
    { 24069, "crisis management after determination by the Swedish Post- and Telecom Authority" },
    { 24099, "Unassigned (may be used for test purposes)" }, /* Used in lab network */
    { 24201, "Telenor Norge AS" },
    { 24202, "Telia Norge AS" },
    { 24203, "Teletopia Gruppen AS" },
    { 24204, "Tele2 Norge AS" },
    { 24205, "Telia Norge AS" },
    { 24206, "ICE Norge AS" },
    { 24207, "Phonero AS" },
    { 24208, "TDC AS" },
    { 24209, "Com4 AS" },
    { 24210, "Norwegian Communications Authority" },
    { 24211, "Systemnet AS" },
    { 24212, "Telenor Norge AS" },
    { 24214, "ICE Communication Norge AS" },
    { 24220, "Jernbaneverket" },
    { 24221, "Jernbaneverket" },
    { 24222, "Network Norway AS" },
    { 24223, "Lycamobile Norway Ltd" },
    { 24224, "Mobile Norway AS" },
    { 24299, "Tampnet AS" },
    { 24403, "DNA Oy" },
    { 24404, "DNA Oy" },
    { 24405, "Elisa Oy" },
    { 24406, "Elisa Oy" },
    { 24407, "Nokia Solutions and Networks Oy" },
    { 24408, "Nokia Solutions and Networks Oy" },
    { 24409, "Nokia Siemens Networks Oy" },
    { 24410, "Viestintavirasto" },
    { 24411, "Viestintavirasto" },
    { 24412, "DNA Oy" },
    { 24413, "DNA Oy" },
    { 24414, "Alands Mobilteleofn Ab" },
    { 24415, "Satakunnan ammattikorkeakoulu Oy" },
    { 24416, "Oy Finland Tele2 AB" },
    { 24417, "Liikennevirasto" },
    { 24421, "Elisa Oyj" },
    { 24422, "EXFO Oy" },
    { 24423, "EXFO Oy" },
    { 24424, "TTY-saatio" },
    { 24426, "Compatel Limited" },
    { 24427, "Teknologian tutkimuskeskus VTT Oy" },
    { 24429, "SCNL TRUPHONE" },
    { 24431, "Kuiri Mobile Oy" },
    { 24432, "Voxbone S.A." },
    { 24433, "Virve Tuotteet ja Palvelut Oy" },
    { 24434, "Bittium Wireless Oy" },
    { 24435, "Ukkoverkot Oy" },
    { 24436, "TeliaSonera Finland Oyj" },
    { 24437, "Tismi BV" },
    { 24438, "Nokia Solutions and Networks Oy" },
    { 24439, "Nokia Solutions and Networks Oy" },
    { 24440, "Nokia Solutions and Networks Oy" },
    { 24441, "Nokia Solutions and Networks Oy" },
    { 24491, "TeliaSonera Finland Oyj" },
    { 24492, "TeliaSonera Finland Oyj" },
    { 24601, "Omnitel" },
    { 24602, "Bite GSM" },
    { 24603, "Tele2" },
    { 24701, "Latvijas Mobilais Telefons SIA" },
    { 24702, "Tele2" },
    { 24703, "Telekom Baltija" },
    { 24704, "Beta Telecom" },
    { 24705, "Bite Mobile" },
    { 24706, "Rigatta" },
    { 24707, "Master Telecom" },
    { 24708, "IZZI" },
    { 24709, "SIA \"Camel Mobile\"" },
    { 24801, "AS Eesti Telekom" },
    { 24802, "RLE" },
    { 24803, "Tele2" },
    { 24804, "OY Top Connect" },
    { 24805, "AS Bravocom Mobiil / Skycompiler OU" },
    { 24806, "ProGroup Holding OY" },
    { 24807, "Televorgu AS" },
    { 24808, "Vivex OU" },
    { 24809, "Bravo Telecom OU" },
    { 24810, "Telcotrade OU" },
    { 24871, "Siseministeerium (Ministry of Interior)" },
    { 25001, "Mobile Telesystems" },
    { 25002, "Megafon" },
    { 25003, "Nizhegorodskaya Cellular Communications" },
    { 25004, "Sibchallenge" },
    { 25005, "Mobile Comms System" },
    { 25007, "BM Telecom" },
    { 25010, "Don Telecom" },
    { 25011, "Orensot" },
    { 25012, "Baykal Westcom" },
    { 25013, "Kuban GSM" },
    { 25016, "New Telephone Company" },
    { 25017, "Ermak RMS" },
    { 25019, "Volgograd Mobile" },
    { 25020, "ECC" },
    { 25028, "Extel" },
    { 25039, "Uralsvyazinform" },
    { 25044, "Stuvtelesot" },
    { 25092, "Printelefone" },
    { 25093, "Telecom XXI" },
    { 25099, "Beeline" },
    { 25501, "Ukrainian Mobile Communication, UMC" },
    { 25502, "Ukranian Radio Systems, URS" },
    { 25503, "Kyivstar GSM" },
    { 25504, "International Telecommunications Ltd." },
    { 25505, "Golden Telecom" },
    { 25506, "Astelit" },
    { 25507, "Ukrtelecom" },
    { 25521, "CJSC - Telesystems of Ukraine" },
    { 25701, "MDC Velcom" },
    { 25702, "MTS" },
    { 25703, "BelCel Joint Venture (JV)" },
    { 25704, "Closed joint-stock company \"Belarusian telecommunication network\"" },
    { 25705, "Republican Unitary Telecommunication Enterprise (RUE) Beltelecom (National Telecommunications Operator of the Republic of Belarus)" },
    { 25706, "Belorussian Cloud Technologies" },
    { 25901, "Orange Moldova GSM" },
    { 25902, "Moldcell GSM" },
    { 25904, "Eventis Mobile GSM" },
    { 25905, "J.S.C. Moldtelecom/3G UMTS (W-CDMA)" },
    { 25999, "J.S.C. Moldtelecom" },
    { 26001, "Plus / Polkomtel S.A." },
    { 26002, "T-Mobile / PTC S.A." },
    { 26003, "Orange / PTK Centertel Sp. z o.o." },
    { 26004, "LTE / CenterNet S.A." },
    { 26005, "Orange(UMTS) / PTK Centertel Sp. z o.o." },
    { 26006, "Play / P4 Sp. z o.o." },
    { 26007, "Netia / Netia S.A." },
    { 26008, "E-Telko / E-Telko Sp. z o.o." },
    { 26009, "Lycamobile / Lycamobile Sp. z o.o." },
    { 26010, "Sferia / Sferia S.A." },
    { 26011, "Nordisk Polska / Nordisk Polska Sp. z o.o." },
    { 26012, "Cyfrowy Polsat / Cyfrowy Polsat S.A." },
    { 26013, "Sferia / Sferia S.A." },
    { 26014, "Sferia / Sferia S.A." },
    { 26015, "CenterNet / CenterNet S.A." },
    { 26016, "Mobyland / Mobyland Sp. z o.o." },
    { 26017, "Aero 2 / Aero 2 Sp. z o.o." },
    { 26018, "AMD Telecom / AMD Telecom S.A." },
    { 26019, "Teleena / Teleena Holding BV" },
    { 26020, "Mobile.Net / Mobile.Net Sp. z o.o." },
    { 26021, "Exteri / Exteri Sp. z o.o." },
    { 26022, "Arcomm / Arcomm Sp. z o.o." },
    { 26023, "Amicomm / Amicomm Sp. z o.o." },
    { 26024, "WideNet / WideNet Sp. z o.o." },
    { 26025, "BS&T / Best Solutions & Technology Sp. z o.o." },
    { 26026, "ATE / ATE-Advanced Technology & Experience Sp. z o.o." },
    { 26027, "Intertelcom / Intertelcom Sp. z o.o." },
    { 26028, "PhoneNet / PhoneNet Sp. z o.o." },
    { 26029, "Interfonica / Interfonica Sp. z o.o." },
    { 26030, "GrandTel / GrandTel Sp. z o.o." },
    { 26031, "Phone IT / Phone IT Sp. z o.o." },
    { 26032, "Compatel Ltd / COMPATEL LIMITED" },
    { 26033, "Truphone Poland / Truphone Poland Sp. Z o.o." },
    { 26034, "T-Mobile / PTC S.A." },
    { 26098, "Play (testowy) / P4 Sp. z o.o." },
    { 26201, "Telekom Deutschland GmbH" },
    { 26202, "Vodafone GmbH" },
    { 26203, "E-Plus Mobilfunk GmbH & Co. KG" },
    { 26204, "Vodafone GmbH" },
    { 26205, "E-Plus Mobilfunk GmbH & Co. KG" },
    { 26206, "Telekom Deutschland GmbH" },
    { 26207, "Telefonica Germany GmbH & Co. oHG" },
    { 26208, "Telefonica Germany GmbH & Co. oHG" },
    { 26209, "Vodafone GmbH" },
    { 26210, "DB Netz AG" },
    { 26211, "Telefonica Germany GmbH & Co. oHG" },
    { 26212, "E-Plus Mobilfunk GmbH & Co. KG" },
    { 26213, "Mobilcom Multimedia GmbH" },
    { 26214, "Quam GmbH" },
    { 26215, "AirData AG" },
    { 26216, "E-Plus Mobilfunk GmbH & Co. KG" },
    { 26217, "E-Plus Mobilfunk GmbH & Co. KG" },
    { 26218, "NetCologne Gesellschaft fur Telekommunikation mbH" },
    { 26219, "Inquam Deutschland GmbH" },
    { 26220, "E-Plus Mobilfunk GmbH & Co. KG" },
    { 26221, "Multiconnect GmbH" },
    { 26222, "sipgate Wireless GmbH" },
    { 26223, "Drillisch Netz AG" },
    { 26241, "First Telecom GmbH" },
    { 26242, "Vodafone GmbH" },
    { 26243, "Vodafone GmbH" },
    { 26272, "Ericsson GmbH" },
    { 26273, "Xantaro Deutschland GmbH" },
    { 26274, "Qualcomm CDMA Technologies GmbH" },
    { 26275, "Core Network Dynamics GmbH" },
    { 26277, "E-Plus Mobilfunk GmbH & Co. KG" },
    { 26278, "Telekom Deutschland GmbH" },
    { 26279, "ng4T GmbH" },
    { 26601, "Gibtelecom GSM" },
    { 26609, "Eazi Telecom Ltd (trading as \"Limba\")" },
    { 26801, "Vodafone Telecel - Comunicacoes Pessoais, S.A." },
    { 26803, "Optimus - Telecomunicacoes, S.A." },
    { 26805, "Oniway - Inforcomunicacoes, S.A." },
    { 26806, "TMN - Telecomunicacoes Moveis Nacionais, S.A." },
    { 27001, "P&T Luxembourg" },
    { 27002, "MTX Connect S.a r.l." },
    { 27010, "BLUE COMMUNICATIONS" },
    { 27077, "Tango" },
    { 27078, "Interactive Digital Media (IDM)" },
    { 27099, "Voxmobile S.A." },
    { 27201, "Vodafone Ireland Plc" },
    { 27202, "Telefonica Ltd" },
    { 27203, "Meteor Mobile Communications Ltd." },
    { 27207, "Eircom" },
    { 27209, "Clever Communications Ltd." },
    { 27401, "Iceland Telecom Ltd." },
    { 27402, "P/F Kall, reg. No 2868 (Vodafone FO) / Og fjarskipti hf (Vodafone Iceland)" },
    { 27403, "Og fjarskipti hf (Vodafone Iceland)" },
    { 27404, "IMC Islande ehf" },
    { 27407, "IceCell ehf" },
    { 27601, "Albanian Mobile Communications (AMC)" },
    { 27602, "Vodafone Albania" },
    { 27603, "Eagle Mobile" },
    { 27604, "Mobile 4 AL" },
    { 27801, "Vodafone Malta" },
    { 27811, "YOM" },
    { 27821, "go mobile" },
    { 27830, "GO Mobile" },
    { 27877, "3G Telecommunications Ltd" },
    { 28001, "CYTA" },
    { 28010, "Scancom (Cyprus) Ltd." },
    { 28020, "PrimeTel PLC" },
    { 28022, "Lemontel Ltd" },
    { 28023, "Mundio Mobile Cyprus Ltd" },
    { 28201, "Geocell Ltd." },
    { 28202, "Magti GSM Ltd." },
    { 28203, "Iberiatel Ltd." },
    { 28204, "Mobitel Ltd." },
    { 28205, "Silknet JSC" },
    { 28206, "JSC Compatel" },
    { 28207, "GLOBALCELL LTD" },
    { 28208, "Silknet GSC" },
    { 28209, "Gmobile LTD" },
    { 28401, "Mobiltel EAD" },
    { 28405, "Globul" },
    { 28601, "Turkcell" },
    { 28602, "Telsim GSM" },
    { 28603, "Aria" },
    { 28604, "Aycell" },
    { 28801, "Faroese Telecom - GSM" },
    { 28802, "Kall GSM" },
    { 29001, "Tele Greenland" },
    { 29002, "inu:it a/s" },
    { 29201, "Prima San Marino / San Marino Telecom" },
    { 29310, "Slovenske zeleznice-Infrastruktura d.o.o." },
    { 29340, "SI Mobil" },
    { 29341, "Mobitel" },
    { 29364, "T-2 d.o.o." },
    { 29370, "Tusmobil d.o.o." },
    { 29401, "T-Mobile" },
    { 29402, "Cosmofon" },
    { 29403, "Nov Operator" },
    { 29404, "Company for telecommunications LYCAMOBILE LLC-Skopje" },
    { 29410, "WTI Macedonia" },
    { 29411, "MOBIK TELEKOMUNIKACII DOOEL- Skopje" },
    { 29501, "Swisscom Schweiz AG" },
    { 29502, "Salt (Liechtenstein) AG" },
    { 29505, "Telecom Liechtenstein AG" },
    { 29506, "Cubic AG" },
    { 29507, "First Mobile AG" },
    { 29509, "Emnify GmbH" },
    { 29510, "SORACOM LI, LTD." },
    { 29701, "Telenor Montenegro" },
    { 29702, "Crnogorski Telekom" },
    { 29703, "Mtel Montenegro" },
    { 30801, "SAS SPM Telecom" },
    { 30802, "Globaltel" },
    { 34001, "Orange Caraibe" },
    { 34002, "Outremer Telecom" },
    { 34003, "United telecommunications services Caraibe" },
    { 34008, "Dauphin Telecom" },
    { 34010, "Guadeloupe Telephone Mobile" },
    { 34011, "Guyane Telephone Mobile" },
    { 34012, "Martinique Telephone Mobile" },
    { 34020, "Bouygues Telecom" },
    { 36251, "TELCELL GSM" },
    { 36269, "CT GSM" },
    { 36291, "SETEL GSM" },
    { 36301, "SETAR N.V." },
    { 36302, "DIGICEL ARUBA" },
    { 36439, "Bahamas Telecommunications Company Limited" },
    { 36449, "NewCo2015 Limited" },
    { 36801, "ETECSA" },
    { 37001, "Orange Dominicana, S.A." },
    { 37002, "Verizon Dominicana S.A." },
    { 37003, "Tricom S.A." },
    { 37004, "CentennialDominicana" },
    { 37201, "Comcel" },
    { 37202, "Digicel" },
    { 37203, "Rectel" },
    { 37412, "TSTT Mobile" },
    { 40001, "\"Azercell Telecom\" LLC" },
    { 40002, "\"Bakcell\" LLC" },
    { 40003, "\"Catel\" LLC" },
    { 40004, "\"Azerfon\" LLC" },
    { 40005, "Special State Protection Service of the Republic of Azerbaijan" },
    { 40006, "\"Nakhtel\" LLC" },
    { 40101, "Kar-Tel llc" },
    { 40102, "TSC Kazak Telecom" },
    { 40211, "Bhutan Telecom Ltd" },
    { 40217, "B-Mobile of Bhutan Telecom" },
    { 40400, "Dishnet Wireless Ltd, Madhya Pradesh" },
    { 40401, "Aircell Digilink India Ltd., Haryana" },
    { 40402, "Bharti Airtel Ltd., Punjab" },
    { 40403, "Bharti Airtel Ltd., H.P." },
    { 40404, "Idea Cellular Ltd., Delhi" },
    { 40405, "Fascel Ltd., Gujarat" },
    { 40406, "Bharti Airtel Ltd., Karnataka" },
    { 40407, "Idea Cellular Ltd., Andhra Pradesh" },
    { 40409, "Reliance Telecom Ltd., Assam" },
    { 40410, "Bharti Airtel Ltd., Delhi" },
    { 40411, "Hutchison Essar Mobile Services Ltd, Delhi" },
    { 40412, "Idea Mobile Communications Ltd., Haryana" },
    { 40413, "Hutchison Essar South Ltd., Andhra Pradesh" },
    { 40414, "Spice Communications PVT Ltd., Punjab" },
    { 40415, "Aircell Digilink India Ltd., UP (East)" },
    { 40416, "Bharti Airtel Ltd, North East" },
    { 40417, "Dishnet Wireless Ltd, West Bengal" },
    { 40418, "Reliance Telecom Ltd., H.P." },
    { 40419, "Idea Mobile Communications Ltd., Kerala" },
    { 40420, "Hutchison Essar Ltd, Mumbai" },
    { 40421, "BPL Mobile Communications Ltd., Mumbai" },
    { 40422, "Idea Cellular Ltd., Maharashtra" },
    { 40423, "Idea Cellular Ltd, Maharashtra" },
    { 40424, "Idea Cellular Ltd., Gujarat" },
    { 40425, "Dishnet Wireless Ltd, Bihar" },
    { 40427, "Hutchison Essar Cellular Ltd., Maharashtra" },
    { 40429, "Dishnet Wireless Ltd, Assam" },
    { 40430, "Hutchison Telecom East Ltd, Kolkata" },
    { 40431, "Bharti Airtel Ltd., Kolkata" },
    { 40433, "Dishnet Wireless Ltd, North East" },
    { 40434, "BSNL, Haryana" },
    { 40435, "Dishnet Wireless Ltd, Himachal Pradesh" },
    { 40436, "Reliance Telecom Ltd., Bihar" },
    { 40437, "Dishnet Wireless Ltd, J&K" },
    { 40438, "BSNL, Assam" },
    { 40440, "Bharti Airtel Ltd., Chennai" },
    { 40441, "Aircell Cellular Ltd, Chennai" },
    { 40442, "Aircel Ltd., Tamil Nadu" },
    { 40443, "Hutchison Essar Cellular Ltd., Tamil Nadu" },
    { 40444, "Spice Communications PVT Ltd., Karnataka" },
    { 40446, "Hutchison Essar Cellular Ltd., Kerala" },
    { 40448, "Dishnet Wireless Ltd, UP (West)" },
    { 40449, "Bharti Airtel Ltd., Andra Pradesh" },
    { 40450, "Reliance Telecom Ltd., North East" },
    { 40451, "BSNL, H.P." },
    { 40452, "Reliance Telecom Ltd., Orissa" },
    { 40453, "BSNL, Punjab" },
    { 40454, "BSNL, UP (West)" },
    { 40455, "BSNL, UP (East)" },
    { 40456, "Idea Mobile Communications Ltd., UP (West)" },
    { 40457, "BSNL, Gujarat" },
    { 40458, "BSNL, Madhya Pradesh" },
    { 40459, "BSNL, Rajasthan" },
    { 40460, "Aircell Digilink India Ltd., Rajasthan" },
    { 40461, "Dishnet Wireless Ltd, Punjab" },
    { 40462, "BSNL, J&K" },
    { 40463, "Dishnet Wireless Ltd, Haryana" },
    { 40464, "BSNL, Chennai" },
    { 40465, "Dishnet Wireless Ltd, UP (East)" },
    { 40466, "BSNL, Maharashtra" },
    { 40467, "Reliance Telecom Ltd., Madhya Pradesh" },
    { 40468, "MTNL, Delhi" },
    { 40469, "MTNL, Mumbai" },
    { 40470, "Bharti Hexacom Ltd, Rajasthan" },
    { 40471, "BSNL, Karnataka" },
    { 40472, "BSNL, Kerala" },
    { 40473, "BSNL, Andhra Pradesh" },
    { 40474, "BSNL, West Bengal" },
    { 40475, "BSNL, Bihar" },
    { 40476, "BSNL, Orissa" },
    { 40477, "BSNL, North East" },
    { 40478, "BTA Cellcom Ltd., Madhya Pradesh" },
    { 40479, "BSNL, Andaman & Nicobar" },
    { 40480, "BSNL, Tamil Nadu" },
    { 40481, "BSNL, Kolkata" },
    { 40482, "Idea Telecommunications Ltd, H.P." },
    { 40483, "Reliable Internet Services Ltd., Kolkata" },
    { 40484, "Hutchison Essar South Ltd., Chennai" },
    { 40485, "Reliance Telecom Ltd., W.B. & A.N." },
    { 40486, "Hutchison Essar South Ltd., Karnataka" },
    { 40487, "Idea Telecommunications Ltd, Rajasthan" },
    { 40488, "Hutchison Essar South Ltd, Punjab" },
    { 40489, "Idea Telecommunications Ltd, UP (East)" },
    { 40490, "Bharti Airtel Ltd., Maharashtra" },
    { 40491, "Dishnet Wireless Ltd, Kolkata" },
    { 40492, "Bharti Airtel Ltd., Mumbai" },
    { 40493, "Bharti Airtel Ltd., Madhya Pradesh" },
    { 40494, "Bharti Airtel Ltd., Tamil Nadu" },
    { 40495, "Bharti Airtel Ltd., Kerala" },
    { 40496, "Bharti Airtel Ltd., Haryana" },
    { 40497, "Bharti Airtel Ltd., UP (West)" },
    { 40498, "Bharti Airtel Ltd., Gujarat" },
    { 40499, "Dishnet Wireless Ltd, Kerala" },
    { 40508, "Reliance Infocomm Ltd, Himachal Pradesh" },
    { 40512, "Reliance Infocomm Ltd, Kolkata" },
    { 40515, "Reliance Infocomm Ltd, Mumbai" },
    { 40517, "Reliance Infocomm Ltd, Orissa" },
    { 40523, "Reliance Infocomm Ltd, West bengal" },
    { 40528, "Tata Teleservices Ltd, Chennai" },
    { 40552, "Bharti Airtel Ltd, Bihar" },
    { 40553, "Bharti Airtel Ltd, Orissa" },
    { 40554, "Bharti Airtel Ltd, UP (East)" },
    { 40555, "Bharti Airtel Ltd, J&K" },
    { 40556, "Bharti Airtel Ltd, Assam" },
    { 40566, "Hutchison Essar South Ltd, UP (West)" },
    { 40567, "Hutchison Essar South Ltd, Orissa" },
    { 40568, "Vodaphone/Hutchison, Madhya Pradesh" },
    { 40570, "Aditya Birla Telecom Ltd, Bihar" },
    { 40571, "Essar Spacetel Ltd, Himachal Pradesh" },
    { 40572, "Essar Spacetel Ltd, North East" },
    { 40573, "Essar Spacetel Ltd, Assam" },
    { 40574, "Essar Spacetel Ltd, J&K" },
    { 40576, "Essar Spacetel Ltd, Orissa" },
    { 40577, "Essar Spacetel Ltd, Maharashtra" },
    { 40581, "Aircell Ltd, Delhi" },
    { 40582, "Aircell Ltd, Andhra Pradesh" },
    { 40583, "Aircell Ltd, Gujarat" },
    { 40584, "Aircell Ltd, Maharashtra" },
    { 40585, "Aircell Ltd, Mumbai" },
    { 40586, "Aircell Ltd, Rajasthan" },
    { 41001, "Mobilink" },
    { 41003, "PAK Telecom Mobile Ltd. (UFONE)" },
    { 41004, "CMPak" },
    { 41006, "Telenor Pakistan" },
    { 41007, "Warid Telecom" },
    { 41201, "AWCC" },
    { 41220, "Roshan" },
    { 41240, "Areeba Afghanistan" },
    { 41250, "Etisalat" },
    { 41280, "Afghan Telecom" },
    { 41288, "Afghan Telecom" },
    { 41302, "MTN Network Ltd." },
    { 41303, "Celtel Lanka Ltd." },
    { 41401, "Myanmar Post and Telecommunication" },
    { 41505, "Ogero Telecom" },
    { 41532, "Cellis" },
    { 41533, "Cellis" },
    { 41534, "Cellis" },
    { 41535, "Cellis" },
    { 41536, "Libancell" },
    { 41537, "Libancell" },
    { 41538, "Libancell" },
    { 41539, "Libancell" },
    { 41601, "Fastlink" },
    { 41602, "Xpress" },
    { 41603, "Umniah" },
    { 41677, "MobileCom" },
    { 41701, "Syriatel" },
    { 41702, "Spacetel Syria" },
    { 41709, "Syrian Telecom" },
    { 41805, "Asia Cell" },
    { 41820, "Zain Iraq (previously Atheer)" },
    { 41830, "Zain Iraq (previously Iraqna)" },
    { 41840, "Korek Telecom" },
    { 41847, "Iraq Central Cooperative Association for Communication and Transportation" },
    { 41848, "ITC Fanoos" },
    { 41849, "Iraqtel" },
    { 41862, "Itisaluna" },
    { 41870, "Kalimat" },
    { 41880, "Iraqi Telecommunications & Post Company (ITPC)" },
    { 41881, "ITPC (Al-Mazaya)" },
    { 41883, "ITPC (Sader Al-Iraq)" },
    { 41884, "ITPC (Eaamar Albasrah)" },
    { 41885, "ITPC (Anwar Yagotat Alkhalee)" },
    { 41886, "ITPC (Furatfone)" },
    { 41887, "ITPC (Al-Seraj)" },
    { 41888, "ITPC (High Link)" },
    { 41889, "ITPC (Al-Shams)" },
    { 41891, "ITPC (Belad Babel)" },
    { 41892, "ITPC (Al Nakheel)" },
    { 41893, "ITPC (Iraqcell)" },
    { 41894, "ITPC (Shaly)" },
    { 41902, "ZAIN" },
    { 41903, "Wataniya Telecom" },
    { 41904, "Viva" },
    { 42001, "Saudi Telecom" },
    { 42003, "Etihad Etisalat Company (Mobily)" },
    { 42101, "Yemen Mobile Phone Company" },
    { 42102, "Spacetel Yemen" },
    { 42104, "Y-Telecom" },
    { 42202, "Oman Mobile Telecommunications Company (Oman Mobile)" },
    { 42203, "Oman Qatari Telecommunications Company (Nawras)" },
    { 42204, "Oman Telecommunications Company (Omantel)" },
    { 42402, "Etisalat" },
    { 42501, "Partner Communications Co. Ltd." },
    { 42502, "Cellcom Israel Ltd" },
    { 42503, "Pelephone Communications Ltd" },
    { 42504, "Globalsim Ltd" },
    { 42506, "Wataniya" },
    { 42507, "Mirs Ltd" },
    { 42508, "Golan Telecom Ltd" },
    { 42511, "365 Telecom (MVNO)" },
    { 42512, "Free Telecom (MVNO)" },
    { 42513, "Ituran Cellular Communications" },
    { 42514, "Alon Cellular Ltd." },
    { 42515, "Home Cellular (MVNO)" },
    { 42516, "Rami Levi (MVNO)" },
    { 42517, "Gale Phone (MVNO)" },
    { 42518, "Cellact Communications Ltd (MVNO)" },
    { 42519, "Azi Communications Ltd" },
    { 42520, "Bezeq Ltd" },
    { 42521, "B.I.P Communications Ltd." },
    { 42523, "Beezz Communication Solutions Ltd." },
    { 42524, "012 Telecom Ltd." },
    { 42525, "IMOD" },
    { 42528, "PHI Networks" },
    { 42601, "Bahrain Telecommunications Company (BATELCO)" },
    { 42602, "Zain Bahrain" },
    { 42603, "Civil Aviation Authority" },
    { 42604, "STC Bahrain" },
    { 42605, "Royal Court" },
    { 42701, "QATARNET" },
    { 42706, "Ooredoo Q.S.C./MOI LTE" },
    { 42899, "Mobicom" },
    { 42901, "Nepal Telecommunications" },
    { 43211, "Mobile Communication of Iran (MCI)" },
    { 43214, "Telecommunication Kish Co. (TKC/KFZO)" },
    { 43219, "Mobile Telecommunications Company of Esfahan JV-PJS (MTCE)" },
    { 43220, "RighTel" },
    { 43232, "Taliya" },
    { 43235, "MTN/Irancell" },
    { 43270, "Telephone Communications Company of Iran (TCI)" },
    { 43293, "Iraphone" },
    { 43401, "Buztel" },
    { 43402, "Uzmacom" },
    { 43404, "Daewoo Unitel" },
    { 43405, "Coscom" },
    { 43407, "Uzdunrobita" },
    { 43601, "JC Somoncom" },
    { 43602, "CJSC Indigo Tajikistan" },
    { 43603, "TT mobile" },
    { 43604, "Josa Babilon-T" },
    { 43605, "CTJTHSC Tajik-tel" },
    { 43701, "Bitel GSM" },
    { 43801, "Barash Communication Technologies (BCTI)" },
    { 43802, "TM-Cell" },
    { 44000, "Y!mobile" },
    { 44001, "UQ Communications Inc." },
    { 44002, "Hanshin Cable Engineering Co.,Ltd." },
    { 44003, "Internet Initiative Japan Inc." },
    { 44004, "SoftBank" },
    { 44005, "Wireless City Planning Inc." },
    { 44006, "SoftBank" },
    { 44007, "KDDI Corporation" },
    { 44008, "KDDI Corporation" },
    { 44009, "NTT DoCoMo Kansai Inc." },
    { 44010, "NTT DOCOMO, INC." },
    { 44011, "NTT DoCoMo Tokai Inc." },
    { 44012, "NTT DoCoMo Inc." },
    { 44013, "NTT DoCoMo Inc." },
    { 44014, "NTT DoCoMo Tohoku Inc." },
    { 44015, "NTT DoCoMo Inc." },
    { 44016, "NTT DoCoMo Inc." },
    { 44017, "NTT DoCoMo Inc." },
    { 44018, "NTT DoCoMo Tokai Inc." },
    { 44019, "NTT DoCoMo Hokkaido Inc." },
    { 44020, "SoftBank Corp." },
    { 44021, "SoftBank Corp." },
    { 44022, "NTT DoCoMo Kansai Inc." },
    { 44023, "NTT DoCoMo Tokai Inc." },
    { 44024, "NTT DoCoMo Chugoku Inc." },
    { 44025, "NTT DoCoMo Hokkaido Inc." },
    { 44026, "NTT DoCoMo Kyushu Inc." },
    { 44027, "NTT DoCoMo Tohoku Inc." },
    { 44028, "NTT DoCoMo Shikoku Inc." },
    { 44029, "NTT DoCoMo Inc." },
    { 44030, "NTT DoCoMo Inc." },
    { 44031, "NTT DoCoMo Kansai Inc." },
    { 44032, "NTT DoCoMo Inc." },
    { 44033, "NTT DoCoMo Tokai Inc." },
    { 44034, "NTT DoCoMo Kyushu Inc." },
    { 44035, "NTT DoCoMo Kansai Inc." },
    { 44036, "NTT DoCoMo Inc." },
    { 44037, "NTT DoCoMo Inc." },
    { 44038, "NTT DoCoMo Inc." },
    { 44039, "NTT DoCoMo Inc." },
    { 44040, "SoftBank" },
    { 44041, "SoftBank" },
    { 44042, "SoftBank" },
    { 44043, "SoftBank" },
    { 44044, "SoftBank" },
    { 44045, "SoftBank" },
    { 44046, "SoftBank" },
    { 44047, "SoftBank" },
    { 44048, "SoftBank" },
    { 44049, "NTT DoCoMo Inc." },
    { 44050, "KDDI Corporation" },
    { 44051, "KDDI Corporation" },
    { 44052, "KDDI Corporation" },
    { 44053, "KDDI Corporation" },
    { 44054, "KDDI Corporation" },
    { 44055, "KDDI Corporation" },
    { 44056, "KDDI Corporation" },
    { 44058, "NTT DoCoMo Kansai Inc." },
    { 44060, "NTT DoCoMo Kansai Inc." },
    { 44061, "NTT DoCoMo Chugoku Inc." },
    { 44062, "NTT DoCoMo Kyushu Inc." },
    { 44063, "NTT DoCoMo Inc." },
    { 44064, "NTT DoCoMo Inc." },
    { 44065, "NTT DoCoMo Shikoku Inc." },
    { 44066, "NTT DoCoMo Inc." },
    { 44067, "NTT DoCoMo Tohoku Inc." },
    { 44068, "NTT DoCoMo Kyushu Inc." },
    { 44069, "NTT DoCoMo Inc." },
    { 44070, "KDDI Corporation" },
    { 44071, "KDDI Corporation" },
    { 44072, "KDDI Corporation" },
    { 44073, "KDDI Corporation" },
    { 44074, "KDDI Corporation" },
    { 44075, "KDDI Corporation" },
    { 44076, "KDDI Corporation" },
    { 44077, "KDDI Corporation" },
    { 44078, "Okinawa Cellular Telephone Company" },
    { 44079, "KDDI Corporation" },
    { 44080, "TU-KA Cellular Tokyo Inc." },
    { 44081, "TU-KA Cellular Tokyo Inc." },
    { 44082, "TU-KA Phone Kansai Inc." },
    { 44083, "TU-KA Cellular Tokai Inc." },
    { 44084, "TU-KA Phone Kansai Inc." },
    { 44085, "TU-KA Cellular Tokai Inc." },
    { 44086, "TU-KA Cellular Tokyo Inc." },
    { 44087, "NTT DoCoMo Chugoku Inc." },
    { 44088, "KDDI Corporation" },
    { 44089, "KDDI Corporation" },
    { 44090, "SoftBank" },
    { 44092, "SoftBank" },
    { 44093, "SoftBank" },
    { 44094, "SoftBank" },
    { 44095, "SoftBank" },
    { 44096, "SoftBank" },
    { 44097, "SoftBank" },
    { 44098, "SoftBank" },
    { 44099, "NTT DoCoMo Inc." },
    { 44100, "Wireless City Planning Inc." },
    { 44101, "SoftBank Corp." },
    { 44110, "UQ Communications Inc." },
    { 44140, "NTT DoCoMo Inc." },
    { 44141, "NTT DoCoMo Inc." },
    { 44142, "NTT DoCoMo Inc." },
    { 44143, "NTT DoCoMo Kansai Inc." },
    { 44144, "NTT DoCoMo Chugoku Inc." },
    { 44145, "NTT DoCoMo Shikoku Inc." },
    { 44150, "TU-KA Cellular Tokyo Inc." },
    { 44151, "TU-KA Phone Kansai Inc." },
    { 44161, "SoftBank" },
    { 44162, "SoftBank" },
    { 44163, "SoftBank" },
    { 44164, "SoftBank" },
    { 44165, "SoftBank" },
    { 44170, "KDDI Corporation" },
    { 44190, "NTT DoCoMo Inc." },
    { 44191, "NTT DoCoMo Inc." },
    { 44192, "NTT DoCoMo Inc." },
    { 44193, "NTT DoCoMo Hokkaido Inc." },
    { 44194, "NTT DoCoMo Tohoku Inc." },
    { 44198, "NTT DoCoMo Kyushu Inc." },
    { 44199, "NTT DoCoMo Kyushu Inc." },
    { 45002, "KT" },
    { 45003, "SK Telecom" },
    { 45005, "SK Telecom" },
    { 45006, "LG U+" },
    { 45008, "KT" },
    { 45011, "SK Telecom (MVNO)" },
    { 45201, "Mobifone" },
    { 45202, "Vinaphone" },
    { 45203, "S Telecom (CDMA)" },
    { 45204, "Viettel" },
    { 45206, "EVN Telecom" },
    { 45207, "Beeline VN/GTEL Mobile JSC" },
    { 45208, "EVN Telecom" },
    { 45400, "Hong Kong Telecommunications (HKT) Ltd" },
    { 45401, "CITIC Telecom International Ltd" },
    { 45402, "Hong Kong Telecommunications (HKT) Ltd" },
    { 45403, "Hutchison Telephone Company Ltd" },
    { 45404, "Hutchison Telephone Company Ltd" },
    { 45405, "Hutchison Telephone Company Ltd" },
    { 45406, "SmarTone Mobile Communications Ltd" },
    { 45407, "China Unicom (Hong Kong) Operations Ltd" },
    { 45408, "Truphone (Hong Kong) Ltd" },
    { 45409, "China Motion Telecom (HK) Ltd" },
    { 45410, "Hong Kong Telecommunications (HKT) Ltd" },
    { 45411, "China-Hongkong Telecom Ltd" },
    { 45412, "China Mobile Hong Kong Co Ltd" },
    { 45413, "China Mobile Hong Kong Co Ltd" },
    { 45414, "Hutchison Telephone Company Ltd" },
    { 45415, "SmarTone Mobile Communications Ltd" },
    { 45416, "Hong Kong Telecommunications (HKT) Ltd" },
    { 45417, "SmarTone Mobile Communications Ltd" },
    { 45418, "Hong Kong Telecommunications (HKT) Ltd" },
    { 45419, "Hong Kong Telecommunications (HKT) Ltd" },
    { 45420, "Hong Kong Telecommunications (HKT) Ltd" },
    { 45421, "21Vianet Mobile Ltd" },
    { 45422, "Delcom (HK) Ltd" },
    { 45423, "Lycamobile Hong Kong Ltd" },
    { 45424, "Public Mobile Networks/Reserved" },
    { 45425, "Public Mobile Networks/Reserved" },
    { 45426, "Public Mobile Networks/Reserved" },
    { 45427, "Government use" },
    { 45428, "China Mobile Hong Kong Co Ltd" },
    { 45429, "Hong Kong Telecommunications (HKT) Ltd" },
    { 45430, "Public Mobile Networks/Reserved" },
    { 45431, "Public Mobile Networks/Reserved" },
    { 45432, "Public Mobile Networks/Reserved" },
    { 45433, "Public Mobile Networks/Reserved" },
    { 45434, "Public Mobile Networks/Reserved" },
    { 45435, "Public Mobile Networks/Reserved" },
    { 45436, "Public Mobile Networks/Reserved" },
    { 45437, "Public Mobile Networks/Reserved" },
    { 45438, "Public Mobile Networks/Reserved" },
    { 45439, "Public Mobile Networks/Reserved" },
    { 45500, "SmarTone - Comunicacoes Moveis, S.A." },
    { 45501, "Companhia de Telecomunicacoes de Macau S.A.R.L." },
    { 45502, "China Telecom (Macau) Limitada" },
    { 45503, "Hutchison - Telefone (Macau) Limitada" },
    { 45504, "Companhia de Telecomunicacoes de Macau S.A.R.L." },
    { 45505, "Hutchison - Telefone (Macau) Limitada" },
    { 45506, "SmarTone - Comunicacoes Moveis, S.A." },
    { 45507, "China Telecom (Macau) Limitada" },
    { 45601, "Mobitel (Cam GSM)" },
    { 45602, "Hello" },
    { 45603, "S Telecom (CDMA)" },
    { 45604, "Cadcomms" },
    { 45605, "Starcell" },
    { 45606, "Smart" },
    { 45608, "Viettel" },
    { 45618, "Mfone" },
    { 45701, "Lao Telecommunications" },
    { 45702, "ETL Mobile" },
    { 45708, "Millicom" },
    { 46000, "China Mobile" },
    { 46001, "China Unicom" },
    { 46002, "Liaoning PPTA" },
    { 46003, "China Unicom CDMA" },
    { 46004, "China Satellite Global Star Network" },
    { 46005, "China Telecom" },
    { 46006, "China Unicom" },
    { 46007, "China Mobile" },
    { 46020, "China Tietong (GSM-R)" },
    { 47001, "GramenPhone" },
    { 47002, "Aktel" },
    { 47003, "Mobile 2000" },
    { 47201, "DhiMobile" },
    { 50210, "DIGI Telecommunications" },
    { 50212, "Malaysian Mobile Services Sdn Bhd" },
    { 50213, "Celcom (Malaysia) Berhad" },
    { 50214, "Telekom Malaysia Berhad" },
    { 50216, "DIGI Telecommunications" },
    { 50217, "Malaysian Mobile Services Sdn Bhd" },
    { 50218, "U Mobile Sdn. Bhd." },
    { 50219, "Celcom (Malaysia) Berhad" },
    { 50220, "Electcoms Wireless Sdn Bhd" },
    { 50501, "Telstra Corporation Ltd." },
    { 50502, "Optus Mobile Pty. Ltd." },
    { 50503, "Vodafone Network Pty. Ltd." },
    { 50504, "Department of Defence" },
    { 50505, "The Ozitel Network Pty. Ltd." },
    { 50506, "Hutchison 3G Australia Pty. Ltd." },
    { 50507, "Vodafone Network Pty. Ltd." },
    { 50508, "One.Tel GSM 1800 Pty. Ltd." },
    { 50509, "Airnet Commercial Australia Ltd." },
    { 50510, "Norfolk Telecom" },
    { 50511, "Telstra Corporation Ltd." },
    { 50512, "Hutchison Telecommunications (Australia) Pty. Ltd." },
    { 50513, "Railcorp" },
    { 50514, "AAPT Ltd" },
    { 50515, "3GIS Pty Ltd. (Telstra & Hutchison 3G)" },
    { 50516, "Victorian Rail Track" },
    { 50517, "Vivid Wireless Pty Ltd" },
    { 50518, "Pactel International Pty Ltd" },
    { 50519, "Lycamobile Pty Ltd" },
    { 50520, "Ausgrid Corporation" },
    { 50521, "Queensland Rail Limited" },
    { 50522, "iiNet Ltd" },
    { 50523, "Challenge Networks Pty Ltd" },
    { 50524, "Advanced Communications Technologies Pty. Ltd." },
    { 50525, "Pilbara Iron Company Services Pty Ltd" },
    { 50526, "Dialogue Communications Pty Ltd" },
    { 50527, "Nexium Telecommunications" },
    { 50528, "RCOM International Pty Ltd" },
    { 50530, "Compatel Limited" },
    { 50531, "BHP Billiton" },
    { 50532, "Thales Australia" },
    { 50533, "CLX Networks Pty Ltd" },
    { 50534, "Santos Ltd" },
    { 50535, "MessageBird Pty Ltd" },
    { 50536, "Optus Mobile Pty. Ltd." },
    { 50537, "Yancoal Australia Ltd" },
    { 50538, "Truphone Pty Ltd" },
    { 50539, "Telstra Corporation Ltd." },
    { 50540, "CITIC PACIFIC MINING" },
    { 50541, "OTOC Australia Pty Ltd" },
    { 50542, "Groote Eylandt Mining Company Pty Ltd" },
    { 50561, "Commtel Network Solutions Pty Ltd" },
    { 50562, "NBNCo Limited" },
    { 50568, "NBNCo Limited" },
    { 50571, "Telstra Corporation Ltd." },
    { 50572, "Telstra Corporation Ltd." },
    { 50588, "Localstar Holding Pty. Ltd." },
    { 50590, "Optus Ltd." },
    { 50599, "One.Tel GSM 1800 Pty. Ltd." },
    { 51000, "PSN" },
    { 51001, "Satelindo" },
    { 51008, "Natrindo (Lippo Telecom)" },
    { 51010, "Telkomsel" },
    { 51011, "Excelcomindo" },
    { 51021, "Indosat - M3" },
    { 51028, "Komselindo" },
    { 51401, "Telin Timor-Leste" },
    { 51402, "Timor Telecom" },
    { 51403, "Viettel Timor-Leste" },
    { 51501, "Islacom" },
    { 51502, "Globe Telecom" },
    { 51503, "Smart Communications" },
    { 51505, "Digitel" },
    { 52000, "CAT CDMA" },
    { 52001, "AIS GSM" },
    { 52002, "CAT CDMA" },
    { 52003, "Advanced Wireless Network Company Limited" },
    { 52004, "Real Future Company Limited" },
    { 52005, "DTAC Network Company Limited" },
    { 52015, "TOT Public Company Limited" },
    { 52018, "Total Access Communications Public Company Limited" },
    { 52020, "ACes Regional Services Company Limited" },
    { 52023, "Digital Phone Company Limited" },
    { 52047, "TOT Public Company Limited" },
    { 52099, "True Move Company Limited" },
    { 52501, "SingTel ST GSM900" },
    { 52502, "SingTel ST GSM1800" },
    { 52503, "M1" },
    { 52505, "StarHub" },
    { 52508, "StarHub" },
    { 52512, "Digital Trunked Radio Network" },
    { 52811, "DST Com" },
    { 53000, "Reserved for AMPS MIN based IMSI's" },
    { 53001, "Vodafone New Zealand GSM Network" },
    { 53002, "Teleom New Zealand CDMA Network" },
    { 53003, "Woosh Wireless - CDMA Network" },
    { 53004, "TelstraClear - GSM Network" },
    { 53005, "Telecom New Zealand - UMTS Network" },
    { 53006, "FX Networks Ltd" },
    { 53007, "Bluereach Limited" },
    { 53024, "NZ Communications - UMTS Network" },
    { 53701, "Bmobile" },
    { 53702, "Telikom PNG Ltd" },
    { 53703, "Digicel Ltd" },
    { 53901, "Tonga Communications Corporation" },
    { 53943, "Digicel" },
    { 53988, "Digicel (Tonga) Ltd" },
    { 54002, "Bemobile (BMobile (SI) Ltd)" },
    { 54101, "SMILE" },
    { 54105, "Digicel Vanuatu" },
    { 54107, "WANTOK" },
    { 54201, "Vodafone (Fiji) Ltd" },
    { 54202, "Digicel (Fiji) Ltd" },
    { 54203, "Telecom Fiji Ltd (CDMA)" },
    { 54301, "Manuia" },
    { 54501, "ATHKL" },
    { 54502, "OceanLink" },
    { 54601, "OPT Mobilis" },
    { 54705, "VITI" },
    { 54710, "Mara Telecom" },
    { 54715, "Pacific Mobile Telecom" },
    { 54720, "Tikiphone" },
    { 54801, "Telecom Cook" },
    { 54901, "Telecom Samoa Cellular Ltd." },
    { 54927, "GoMobile SamoaTel Ltd" },
    { 55001, "FSM Telecom" },
    { 55201, "Palau National Communications Corp. (a.k.a. PNCC)" },
    { 55301, "Tuvalu Telecommunications Corporation" },
    { 55401, "Teletok/LTE 4G" },
    { 55501, "Telecom Niue" },
    { 60201, "Mobinil" },
    { 60202, "Vodafone" },
    { 60203, "Etisalat" },
    { 60301, "Algerie Telecom" },
    { 60302, "Orascom Telecom Algerie" },
    { 60400, "Meditelecom (GSM)" },
    { 60401, "Ittissalat Al Maghrid" },
    { 60502, "Tunisie Telecom" },
    { 60503, "Orascom Telecom" },
    { 60701, "Gamcel" },
    { 60702, "Africell" },
    { 60703, "COMIUM" },
    { 60704, "Qcell" },
    { 60705, "GAMTEL-Ecowan" },
    { 60706, "NETPAGE" },
    { 60801, "Sonatel (Orange)" },
    { 60802, "Sentel GSM (Tigo)" },
    { 60803, "Expresso Senegal" },
    { 60804, "CSU" },
    { 60901, "Mattel S.A." },
    { 60902, "Chinguitel S.A." },
    { 60910, "Mauritel Mobiles" },
    { 61001, "Malitel" },
    { 61101, "Orange Guinee" },
    { 61102, "Sotelgui" },
    { 61105, "Cellcom Guinee SA" },
    { 61202, "Atlantique Cellulaire" },
    { 61203, "Orange Cote d'Ivoire" },
    { 61204, "Comium Cote d'Ivoire" },
    { 61205, "Loteny Telecom" },
    { 61206, "Oricel Cote d'Ivoire" },
    { 61207, "Aircomm Cote d'Ivoire" },
    { 61302, "Celtel" },
    { 61303, "Telecel" },
    { 61401, "Sahel.Com" },
    { 61402, "Celtel" },
    { 61403, "Telecel" },
    { 61501, "Togo Telecom" },
    { 61601, "Libercom" },
    { 61602, "Telecel" },
    { 61603, "Spacetel Benin" },
    { 61701, "Cellplus" },
    { 61702, "Mahanagar Telephone (Mauritius) Ltd." },
    { 61703, "Mahanagar Telephone (Mauritius) Ltd." },
    { 61710, "Emtel" },
    { 61804, "Comium Liberia" },
    { 61901, "Celtel" },
    { 61902, "Millicom" },
    { 61903, "Africell" },
    { 61904, "Comium (Sierra Leone) Ltd" },
    { 61905, "Lintel (Sierra Leone) Ltd." },
    { 61909, "INTERGROUP TELECOM" },
    { 61925, "Mobitel" },
    { 61940, "Datatel (SL) Ltd GSM" },
    { 61950, "Datatel (SL) Ltd CDMA" },
    { 62001, "Spacefon" },
    { 62002, "Ghana Telecom Mobile" },
    { 62003, "Mobitel" },
    { 62004, "Kasapa Telecom Ltd." },
    { 62011, "Netafriques Dot Com Ltd" },
    { 62120, "Econet Wireless Nigeria Ltd." },
    { 62130, "MTN Nigeria Communications" },
    { 62140, "MTEL" },
    { 62150, "Globacom" },
    { 62160, "EMTS" },
    { 62201, "Celtel" },
    { 62202, "Tchad Mobile" },
    { 62301, "Centrafrique Telecom Plus (CTP)" },
    { 62302, "Telecel Centrafrique (TC)" },
    { 62303, "Celca (Socatel)" },
    { 62401, "Mobile Telephone Networks Cameroon" },
    { 62402, "Orange Cameroun" },
    { 62404, "NEXTTEL (ex VIETTEL CAMEROON)" },
    { 62501, "Cabo Verde Telecom" },
    { 62502, "T+Telecomunicacoes" },
    { 62601, "Companhia Santomese de Telecomunicacoes" },
    { 62701, "Guinea Ecuatorial de Telecomunicaciones Sociedad Anonima (GETESA)" },
    { 62801, "LIBERTIS" },
    { 62802, "MOOV" },
    { 62803, "CELTEL" },
    { 62804, "USAN GABON" },
    { 62805, "Reseau de l'Administration Gabonaise (RAG)" },
    { 62901, "Celtel" },
    { 62910, "Libertis Telecom" },
    { 63001, "Vodacom Congo RDC sprl" },
    { 63002, "AIRTEL sprl" },
    { 63005, "Supercell Sprl" },
    { 63086, "Congo-Chine Telecom s.a.r.l." },
    { 63088, "YOZMA TIMETURNS sprl" },
    { 63089, "OASIS sprl" },
    { 63090, "Africell RDC" },
    { 63102, "Unitel" },
    { 63104, "Movicel" },
    { 63201, "Guinetel S.A." },
    { 63202, "Spacetel Guine-Bissau S.A." },
    { 63301, "Cable & Wireless (Seychelles) Ltd." },
    { 63302, "Mediatech International Ltd." },
    { 63310, "Airtel (Seychelles) Ltd" },
    { 63401, "SD Mobitel" },
    { 63402, "Areeba-Sudan" },
    { 63403, "MTN Sudan" },
    { 63405, "Network of the World Ltd (NOW)" },
    { 63406, "Zain Sudan" },
    { 63499, "MTN Sudan" },
    { 63510, "MTN Rwandacell" },
    { 63513, "TIGO RWANDA LTD" },
    { 63514, "AIRTEL RWANDA Ltd" },
    { 63517, "Olleh Rwanda Networks (ORN)" },
    { 63601, "ETH MTN" },
    { 63801, "Evatis" },
    { 63902, "Safaricom Limited" },
    { 63903, "Kencell Communications Ltd." },
    { 64002, "MIC Tanzania Limited (Tigo)" },
    { 64003, "Zanzibar Telecom Limited (Zantel)" },
    { 64004, "Vodacom Tanzania Limited" },
    { 64005, "Airtel Tanzania Limited" },
    { 64007, "Tanzania Telecommunications Company Limited" },
    { 64008, "Benson Informatics Limited (Smart)" },
    { 64009, "Viettel Tanzania Limited (Halotel)" },
    { 64011, "Smile Communications Tanzania Ltd" },
    { 64013, "Wiafrica Tanzania Limited (Cootel)" },
    { 64101, "Airtel Uganda Limited" },
    { 64104, "Tangerine Uganda Limited" },
    { 64106, "Afrimax Uganda Limited" },
    { 64110, "MTN Uganda Limited" },
    { 64111, "Uganda Telecom Limited" },
    { 64114, "House of Integrated Technology and Systems Uganda Ltd (HiTs Telecom)" },
    { 64118, "Suretelecom Uganda Limited" },
    { 64122, "Airtel Uganda Limited" },
    { 64126, "Lycamobile Network services Uganda Limited" },
    { 64130, "Anupam Global Soft Uganda Limited" },
    { 64133, "Smile Communications Uganda Limited" },
    { 64140, "Civil Aviation Authority (CAA)" },
    { 64144, "K2 Telecom Limited" },
    { 64166, "i-Tel Ltd" },
    { 64201, "Econet" },
    { 64202, "Africell" },
    { 64203, "ONAMOB" },
    { 64207, "LACELL" },
    { 64282, "U.COM" },
    { 64301, "T.D.M. GSM" },
    { 64303, "Movitel" },
    { 64304, "VM Sarl" },
    { 64501, "Celtel Zambia Ltd." },
    { 64502, "Telecel Zambia Ltd." },
    { 64503, "Zamtel" },
    { 64601, "Celtel Madagascar (Zain), GSM" },
    { 64602, "Orange Madagascar, GSM" },
    { 64604, "Telecom Malagasy Mobile, GSM" },
    { 64700, "Orange La Reunion" },
    { 64701, "BJT Partners" },
    { 64702, "Telco OI" },
    { 64710, "Societe Reunionnaise du Radiotelephone" },
    { 64801, "Net One" },
    { 64803, "Telecel" },
    { 64804, "Econet" },
    { 64901, "Mobile Telecommunications Ltd." },
    { 64902, "Telecom Namibia" },
    { 64903, "Powercom Pty Ltd (leo)" },
    { 64904, "Paratus Telecommunications (Pty)" },
    { 64905, "Demshi Investments CC" },
    { 65001, "Telekom Network Ltd." },
    { 65010, "Celtel ltd." },
    { 65101, "Vodacom Lesotho (pty) Ltd." },
    { 65102, "Econet Ezin-cel" },
    { 65201, "Mascom Wireless (Pty) Ltd" },
    { 65202, "Orange Botswana (Pty) Ltd" },
    { 65204, "Botswana Telecommunications Corporation (BTC)" },
    { 65301, "SPTC" },
    { 65310, "Swazi MTN" },
    { 65401, "HURI / Comores Telecom" },
    { 65402, "TELMA / TELCO SA" },
    { 65501, "Vodacom (Pty) Ltd." },
    { 65502, "Telkom SA Ltd" },
    { 65504, "Sasol (Pty) Ltd" },
    { 65506, "Sentech (Pty) Ltd." },
    { 65507, "Cell C (Pty) Ltd." },
    { 65510, "Mobile Telephone Networks (MTN) Pty Ltd" },
    { 65511, "SAPS Gauteng" },
    { 65512, "Mobile Telephone Networks (MTN) Pty Ltd" },
    { 65513, "Neotel Pty Ltd" },
    { 65514, "Neotel Pty Ltd" },
    { 65517, "Sishen Iron Ore Company (Ltd) Pty" },
    { 65519, "Wireless Business Solutions (iBurst)" },
    { 65521, "Cape Town Metropolitan Council" },
    { 65524, "SMS Portal (Pty) Ltd" },
    { 65525, "Wirels Connect" },
    { 65527, "A to Z Vaal Industrial Supplies Pty Ltd" },
    { 65528, "Hymax Talking Solutions (Pty) Ltd" },
    { 65530, "Bokamoso Consortium Pty Ltd" },
    { 65531, "Karabo Telecoms (Pty) Ltd." },
    { 65532, "Ilizwi Telecommunications Pty Ltd" },
    { 65533, "Thinta Thinta Telecommunications Pty Ltd" },
    { 65534, "Bokone Telecoms Pty Ltd" },
    { 65535, "Kingdom Communications Pty Ltd" },
    { 65536, "Amatole Telecommunication Pty Ltd" },
    { 65538, "Wireless Business Solutions (Pty) Ltd" },
    { 65541, "South African Police Service" },
    { 65546, "SMS Cellular Services (Pty) Ltd" },
    { 65550, "Ericsson South Africa (Pty) Ltd" },
    { 65551, "Integrat (Pty) Ltd" },
    { 65573, "Wireless Business Solutions (Pty) Ltd" },
    { 65574, "Wireless Business Solutions (Pty) Ltd" },
    { 65801, "Sure South Atlantic Ltd. (Ascension)" },
    { 65912, "Sudani/Sudatel" },
    { 65991, "Zain-South Sudan" },
    { 65992, "MTN-South Sudan" },
    { 65995, "Vivacel/NOW" },
    { 65997, "Gemtel" },
    { 70267, "Belize Telecommunications Ltd., GSM 1900" },
    { 70268, "International Telecommunications Ltd. (INTELCO)" },
    { 70269, "SMART/Speednet Communications Ltd." },
    { 70401, "Servicios de Comunicaciones Personales Inalembricas, S.A. (SERCOM, S.A)" },
    { 70402, "Comunicaciones Celulares S.A." },
    { 70403, "Telefonica Centroamerica Guatemala S.A." },
    { 70601, "CTE Telecom Personal, S.A. de C.V." },
    { 70602, "Digicel, S.A. de C.V." },
    { 70603, "Telemovil El Salvador, S.A." },
    { 71021, "Empresa Nicaraguense de Telecomunicaciones, S.A. (ENITEL)" },
    { 71073, "Servicios de Comunicaciones, S.A. (SERCOM)" },
    { 71201, "Instituto Costarricense de Electricidad - ICE" },
    { 71202, "Instituto Costarricense de Electricidad - ICE" },
    { 71203, "CLARO CR Telecomunicaciones S.A." },
    { 71204, "Telefonica de Costa Rica TC, S.A." },
    { 71220, "Virtualis" },
    { 71401, "Cable & Wireless Panama S.A." },
    { 71402, "BSC de Panama S.A. / Telefonica Moviles Panama S.A." },
    { 71403, "Claro Panama, S.A." },
    { 71404, "Digicel (Panama), S.A." },
    { 71610, "TIM Peru" },
    { 72400, "NEXTEL" },
    { 72401, "SISTEER DO BRASIL TELECOMUNICACOES (MVNO)" },
    { 72402, "TIM REGIAO I" },
    { 72403, "TIM REGIAO III" },
    { 72404, "TIM REGIAO III" },
    { 72405, "CLARO" },
    { 72406, "VIVO REGIAO II" },
    { 72410, "VIVO REGIAO III" },
    { 72411, "VIVO REGIAO I" },
    { 72415, "SERCOMTEL" },
    { 72416, "BRT CELULAR" },
    { 72418, "DATORA (MVNO)" },
    { 72423, "TELEMIG CELULAR" },
    { 72424, "AMAZONIA CELULAR" },
    { 72430, "TNL PCS Oi" },
    { 72431, "TNL PCS Oi" },
    { 72432, "CTBC CELULAR R III" },
    { 72433, "CTBC CELULAR R II" },
    { 72434, "CTBC CELULAR R I" },
    { 72435, "TELCOM" },
    { 72436, "OPTIONS" },
    { 72437, "UNICEL" },
    { 72438, "CLARO" },
    { 72439, "NEXTEL (SMP)" },
    { 72454, "PORTO SEGURO TELECOMUNICACOES (MVNO)" },
    { 72499, "LOCAL (STFC)" },
    { 73001, "Entel Telefonica Movil" },
    { 73002, "Telefonica Movil" },
    { 73003, "Smartcom" },
    { 73004, "Centennial Cayman Corp. Chile S.A." },
    { 73005, "Multikom S.A." },
    { 73006, "Blue Two Chile SA" },
    { 73007, "Telefonica Moviles Chile S.A." },
    { 73008, "VTR Movil S.A." },
    { 73009, "Centennial Cayman Corp. Chile S.A." },
    { 73010, "Entel" },
    { 73011, "Celupago S.A." },
    { 73012, "Telestar Movil S.A." },
    { 73013, "TRIBE Mobile Chile SPA" },
    { 73014, "Netline Telefonica Movil Ltda" },
    { 73015, "CIBELES TELECOM S.A." },
    { 73016, "Nomade Telecomunicaciones S.A." },
    { 73017, "COMPATEL Chile Limitada" },
    { 73018, "Empresas Bunker S.A." },
    { 73019, "Sociedad Falabella Movil SPA" },
    { 73402, "Corporacion Digitel" },
    { 73403, "GALAXY ENTERTAINMENT DE VENEZUELA C.A." },
    { 73404, "Telcel, C.A." },
    { 73406, "Telecomunicaciones Movilnet, C.A." },
    { 73601, "Nuevatel S.A." },
    { 73602, "ENTEL S.A." },
    { 73603, "Telecel S.A." },
    { 73801, "U-Mobile (Cellular) Inc." },
    { 73805, "eGovernment Unit, Ministry of the Presidency" },
    { 74000, "Otecel S.A. - Bellsouth" },
    { 74001, "Porta GSM" },
    { 74002, "Telecsa S.A." },
    { 74401, "Hola Paraguay S.A." },
    { 74402, "Hutchison Telecom S.A." },
    { 74403, "Compania Privada de Comunicaciones S.A." },
    { 74602, "Telesur" },
    { 74603, "Digicel" },
    { 74604, "Intelsur" },
    { 74605, "Telesur (CDMA)" },
    { 74800, "Ancel - TDMA" },
    { 74801, "Ancel - GSM" },
    { 74803, "Ancel" },
    { 74807, "Movistar" },
    { 74810, "CTI Movil" },
    { 90101, "ICO Global Communications" },
    { 90103, "Iridium Communications Inc" },
    { 90105, "Thuraya RMSS Network" },
    { 90106, "Thuraya Satellite Telecommunications Company" },
    { 90110, "Asia Cellular Satellite (AceS)" },
    { 90111, "Inmarsat Ltd." },
    { 90112, "Maritime Communications Partner AS (MCP network)" },
    { 90113, "BebbiCell AG (Formerly Global Networks Switzerland AG)" },
    { 90114, "AeroMobile AS" },
    { 90115, "OnAir N.V. (Formerly SITA on behalf of Onair)" },
    { 90116, "Jasper Technologies Inc. [formerly Jasper Wireless, Inc]" },
    { 90117, "Jersey Telecom" },
    { 90118, "Cingular Wireless" },
    { 90119, "Vodafone Malta (Vodafone Group)" },
    { 90120, "Intermatica" },
    { 90121, "Wins Limited (Formerly Seanet Maritime Communications AB)" },
    { 90122, "MediaLincc Ltd" },
    { 90124, "Voxbone SA" },
    { 90126, "Telecom Italia" },
    { 90127, "Monaco Telecom" },
    { 90128, "Vodafone Group" },
    { 90129, "Telenor Connexion AB" },
    { 90131, "Orange [formerly France Telecom Orange]" },
    { 90132, "MegaFon" },
    { 90133, "Smart Communications , Inc" },
    { 90134, "Tyntec Limited" },
    { 90135, "Globecomm Network Services" },
    { 90136, "Azerfon LLC" },
    { 90137, "TRANSATEL" },
    { 90138, "Multiregional TransitTelecom (MTT)" },
    { 90139, "MTX Connect" },
    { 90140, "Deutsche Telekom AG" },
    { 90141, "BodyTrace Netherlands B.V." },
    { 90142, "DCN Hub ehf" },
    { 90143, "EMnify GmbH" },
    { 90144, "AT&T, Inc." },
    { 90145, "Advanced Wireless Network Company Limited" },
    { 90146, "Telecom26 AG" },
    { 90147, "Ooredoo" },
    { 90148, "Com4 Sweden AB" },
    { 90149, "Zain Kuwait" },
    { 90150, "Sawatch Limited/EchoStar Mobile Limited" },
    { 90151, "VisionNG" },
    { 90152, "Manx Telecom Trading Ltd." },
    { 90153, "Deutsche Telekom AG" },
    { 90154, "Teleena Holding B.V." },
    { 90155, "Beezz Communication Sol. Ltd." },
    { 90188, "UN Office for the Coordination of Humanitarian Affairs (OCHA)" },
    { 0, NULL }
};

value_string_ext mcc_mnc_2digits_codes_ext = VALUE_STRING_EXT_INIT(mcc_mnc_2digits_codes);

static const value_string mcc_mnc_3digits_codes[] = {
    {  302220, "Telus Mobility" },
    {  302221, "Telus Mobility" },
    {  302222, "Telus Mobility" },
    {  302250, "ALO Mobile Inc" },
    {  302270, "Bragg Communications" },
    {  302290, "Airtel Wireless" },
    {  302300, "ECOTEL inc." },
    {  302320, "Dave Wireless" },
    {  302340, "Execulink" },
    {  302360, "Telus Mobility" },
    {  302370, "Microcell" },
    {  302380, "Dryden Mobility" },
    {  302390, "Dryden Mobility" },
    {  302490, "Globalive Wireless" },
    {  302500, "Videotron Ltd" },
    {  302510, "Videotron Ltd" },
    {  302530, "Keewatinook Okimacinac" },
    {  302560, "Lynx Mobility" },
    {  302570, "Light Squared" },
    {  302590, "Quadro Communication" },
    {  302610, "Bell Mobility" },
    {  302620, "Ice Wireless" },
    {  302630, "Aliant Mobility" },
    {  302640, "Bell Mobility" },
    {  302650, "Tbaytel" },
    {  302656, "Tbaytel (formerly Tbay Mobility)" },
    {  302660, "MTS Mobility" },
    {  302670, "CityTel Mobility" },
    {  302680, "Sask Tel Mobility" },
    {  302690, "Bell Mobility" },
    {  302710, "Globalstar" },
    {  302720, "Rogers Wireless" },
    {  302730, "TerreStar Solutions" },
    {  302740, "Shaw Telecom G.P." },
    {  302760, "Public Mobile Inc" },
    {  302770, "Rural Com" },
    {  302780, "Sask Tel Mobility" },
    {  302860, "Telus Mobility" },
    {  302880, "Telus/Bell shared" },
    {  302940, "Wightman Telecom" },
    {  302990, "Test" },
    {  310010, "Verizon Wireless" },
    {  310012, "Verizon Wireless" },
    {  310013, "Verizon Wireless" },
    {  310014, "TEST IMSI HNI" },
    {  310016, "AT&T Mobility" },
    {  310017, "North Sight Communications Inc" },
    {  310020, "Union Telephone Company" },
    {  310030, "AT&T Mobility" },
    {  310035, "ETEX Communications, LP (d/b/a) ETEX Wireless" },
    {  310040, "MTA Communications dba MTA Wireless" },
    {  310050, "Alaska Wireless Networks" },
    {  310060, "Consolidated Telcom" },
    {  310070, "AT&T Mobility" },
    {  310080, "AT&T Mobility" },
    {  310090, "AT&T Mobility" },
    {  310100, "New Mexico RSA 4 East Ltd. Partnership" },
    {  310110, "Pacific Telecom Inc" },
    {  310120, "Sprint" },
    {  310130, "Carolina West Wireless" },
    {  310140, "GTA Wireless LLC" },
    {  310150, "AT&T Mobility" },
    {  310160, "T-Mobile USA" },
    {  310170, "AT&T Mobility" },
    {  310180, "West Central Wireless" },
    {  310190, "Alaska Wireless Networks" },
    {  310200, "T-Mobile USA" },
    {  310210, "T-Mobile USA" },
    {  310220, "T-Mobile USA" },
    {  310230, "T-Mobile USA" },
    {  310240, "T-Mobile USA" },
    {  310250, "T-Mobile USA" },
    {  310260, "T-Mobile USA" },
    {  310270, "T-Mobile USA" },
    {  310280, "AT&T Mobility" },
    {  310290, "NEP Cellcorp Inc." },
    {  310300, "Blanca Telephone Company" },
    {  310310, "T-Mobile USA" },
    {  310320, "Smith Bagley Inc, dba Cellular One" },
    {  310330, "AWCC" },
    {  310340, "HHigh Plains Midwest LLC, dba Westlink Communications" },
    {  310350, "Verizon Wireless" },
    {  310360, "Cellular Network Partnership dba Pioneer Cellular" },
    {  310370, "Docomo Pacific Inc" },
    {  310380, "AT&T Mobility" },
    {  310390, "TX-11 Acquistion LLC" },
    {  310400, "Wave Runner LLC" },
    {  310410, "AT&T Mobility" },
    {  310420, "Cincinnati Bell Wireless LLC" },
    {  310430, "GCI Communications Corp" },
    {  310440, "Numerex Corp" },
    {  310450, "North East Cellular Inc." },
    {  310460, "Newcore Wireless" },
    {  310470, "Shentel" },
    {  310480, "Wave Runner LLC" },
    {  310490, "T-Mobile USA" },
    {  310500, "Public Service Cellular, Inc." },
    {  310510, "Nsight" },
    {  310520, "Transactions Network Services (TNS)" },
    {  310530, "Iowa Wireless Services LLC dba | Wireless" },
    {  310540, "Oklahoma Western Telephone Company" },
    {  310550, "Syniverse Technologies" },
    {  310560, "AT&T" },
    {  310570, "MTPCS LLC" },
    {  310580, "Inland Cellular Telephone Company" },
    {  310590, "Verizon Wireless" },
    {  310600, "NewCell dba Cellcom" },
    {  310610, "Elkhart Telephone Co. Inc. dba Epic Touch Co." },
    {  310620, "Nsight" },
    {  310630, "Agri-Valley Broadband, Inc" },
    {  310640, "Numerex Corp" },
    {  310650, "JASPER TECHNOLOGIES INC." },
    {  310660, "T-Mobile USA" },
    {  310670, "AT&T Mobility" },
    {  310680, "AT&T Mobility" },
    {  310690, "Keystone Wireless LLC" },
    {  310700, "Cross Valiant Cellular Partnership" },
    {  310710, "Arctic Slope Telephone Association Cooperative" },
    {  310720, "Syniverse Technologies" },
    {  310730, "US Cellular" },
    {  310740, "Viaero Wireless" },
    {  310750, "East Kentucky Network LLC dba Appalachian Wireless" },
    {  310760, "Lynch 3G Communications Corporation" },
    {  310770, "Iowa Wireless Services LLC dba I Wireless" },
    {  310780, "D.D. Inc" },
    {  310790, "PinPoint Communications Inc." },
    {  310800, "T-Mobile USA" },
    {  310810, "LCFR LLC" },
    {  310820, "Verizon Wireless" },
    {  310830, "Sprint" },
    {  310840, "Telecom North America Mobile Inc" },
    {  310850, "Aeris Communications, Inc." },
    {  310860, "West Central Wireless" },
    {  310870, "Kaplan Telephone Company Inc." },
    {  310880, "Advantage Cellular Systems, Inc." },
    {  310890, "Verizon Wireless" },
    {  310900, "Mid-Rivers Telephone Cooperative" },
    {  310910, "Verizon Wireless" },
    {  310920, "James Valley Wireless LLC" },
    {  310930, "Copper Valley Wireless" },
    {  310940, "Iris Wireless LLC" },
    {  310950, "AT&T Mobility" },
    {  310960, "UBET Wireless" },
    {  310970, "Globalstar USA" },
    {  310980, "Texas RSA 7B3 dba Peoples Wireless Services" },
    {  310990, "Worldcall Interconnect" },
    {  311000, "West Central Wireless" },
    {  311010, "Chariton Valley Communications Corp., Inc." },
    {  311020, "Missouri RSA No. 5 Partnership" },
    {  311030, "Indigo Wireless, Inc." },
    {  311040, "Commnet Wireless LLC" },
    {  311050, "Thumb Cellular Limited Partnership" },
    {  311060, "Space Data Corporation" },
    {  311070, "AT&T Mobility" },
    {  311080, "Pine Telephone Company dba Pine Cellular" },
    {  311090, "AT&T Mobility" },
    {  311100, "Nex-Tech Wireless LLC" },
    {  311110, "Verizon Wireless" },
    {  311120, "Wave Runner LLC" },
    {  311130, "Light Squared LP" },
    {  311140, "Cross Telephone Company" },
    {  311150, "Wilkes Cellular Inc." },
    {  311160, "Light Squared LP" },
    {  311170, "Broadpoint, LLC" },
    {  311180, "AT&T Mobility" },
    {  311190, "AT&T Mobility" },
    {  311200, "ARINC" },
    {  311210, "Emery Telecom-Wireless Inc" },
    {  311220, "United States Cellular" },
    {  311230, "Cellular South Inc" },
    {  311240, "Cordova Wireless Communications Inc" },
    {  311250, "Wave Runner LLC" },
    {  311260, "Sprint" },
    {  311270, "Verizon Wireless" },
    {  311271, "Verizon Wireless" },
    {  311272, "Verizon Wireless" },
    {  311273, "Verizon Wireless" },
    {  311274, "Verizon Wireless" },
    {  311275, "Verizon Wireless" },
    {  311276, "Verizon Wireless" },
    {  311277, "Verizon Wireless" },
    {  311278, "Verizon Wireless" },
    {  311279, "Verizon Wireless" },
    {  311280, "Verizon Wireless" },
    {  311281, "Verizon Wireless" },
    {  311282, "Verizon Wireless" },
    {  311283, "Verizon Wireless" },
    {  311284, "Verizon Wireless" },
    {  311285, "Verizon Wireless" },
    {  311286, "Verizon Wireless" },
    {  311287, "Verizon Wireless" },
    {  311288, "Verizon Wireless" },
    {  311289, "Verizon Wireless" },
    {  311290, "Pinpoint Wireless Inc." },
    {  311300, "Nexus Communications Inc" },
    {  311310, "Leaco Rural Telephone Company Inc" },
    {  311320, "Commnet Wireless LLC" },
    {  311330, "Bug Tussel Wireless LLC" },
    {  311340, "Illinois Valley Cellular" },
    {  311350, "Sagebrush Cellular Inc dba Nemont" },
    {  311360, "Stelera Wireless LLC" },
    {  311370, "GCI Communications Corp" },
    {  311380, "New Dimension Wireless Ltd" },
    {  311390, "Verizon Wireless" },
    {  311400, "TEST IMSI HNI" },
    {  311410, "Iowa RSA No. 2 Limited Partnership" },
    {  311420, "Northwest Cell" },
    {  311430, "RSA 1 Limited Partnership dba Cellular 29 Plus" },
    {  311440, "Bluegrass Cellular LLC" },
    {  311450, "Panhandle Telecommunication Systems Inc." },
    {  311460, "Fisher Wireless Services Inc" },
    {  311470, "Vitelcom Cellular D/B/A Innovative Wireless" },
    {  311480, "Verizon Wireless" },
    {  311481, "Verizon Wireless" },
    {  311482, "Verizon Wireless" },
    {  311483, "Verizon Wireless" },
    {  311484, "Verizon Wireless" },
    {  311485, "Verizon Wireless" },
    {  311486, "Verizon Wireless" },
    {  311487, "Verizon Wireless" },
    {  311488, "Verizon Wireless" },
    {  311489, "Verizon Wireless" },
    {  311490, "Sprint" },
    {  311500, "Mosaic Telecom Inc" },
    {  311510, "Light Squared LP" },
    {  311520, "Light Squared LP" },
    {  311530, "Newcore Wireless LLC" },
    {  311540, "Proximiti Mobility Inc" },
    {  311550, "Commnet Midwest LLC" },
    {  311560, "OTZ Communications Inc" },
    {  311570, "Bend Cable Communications LLC" },
    {  311580, "U.S. Cellular" },
    {  311590, "Verizon Wireless" },
    {  311600, "Cox TMI Wireless LLC" },
    {  311610, "North Dakota Network Co." },
    {  311620, "Terrestar Networks Inc" },
    {  311630, "Cellular South Inc." },
    {  311640, "Standing Rock Telecommunications" },
    {  311650, "United Wireless Inc" },
    {  311660, "Metro PCS Wireless Inc" },
    {  311670, "Pine Belt Cellular Inc dba Pine Belt Wireless" },
    {  311680, "GreenFly LLC" },
    {  311690, "TeleBeeper of New Mexico Inc" },
    {  311700, "Aspenta International, Inc." },
    {  311710, "Northeast Wireless Networks LLC" },
    {  311720, "Maine PCS LLC" },
    {  311730, "Proximiti Mobility Inc" },
    {  311740, "Telalaska Cellular" },
    {  311750, "Flat Wireless, LLC" },
    {  311760, "Edigen Inc" },
    {  311770, "Altiostar Networks, Inc." },
    {  311790, "Coleman County Telephone Cooperative, Inc." },
    {  311800, "Bluegrass Cellular LLC" },
    {  311810, "Blegrass Cellular LLC" },
    {  311820, "Sonus Networks" },
    {  311830, "Thumb Cellular LLC" },
    {  311840, "Nsight" },
    {  311850, "Nsight" },
    {  311860, "Uintah Basin Electronic Telecommunications" },
    {  311870, "Sprint" },
    {  311880, "Sprint" },
    {  311890, "Globecomm Network Services Corporation" },
    {  311900, "Gigsky Inc." },
    {  311910, "SI Wireless LLC" },
    {  311920, "Missouri RSA No 5 Partnership dba Charlton Valley Wireless Services" },
    {  311940, "Sprint" },
    {  311950, "Sunman Telecommunications corp." },
    {  311960, "Lycamobile USA Inc" },
    {  311970, "Big River Broadband LLC" },
    {  311980, "LigTel Communications" },
    {  311990, "VTel Wireless" },
    {  312010, "Charlton Valley Communication Corporation Inc" },
    {  312020, "Infrastructure Networks LLC" },
    {  312030, "Cross Wireless" },
    {  312040, "Custer Telephone Cooperative Inc" },
    {  312050, "Fuego Wireless LLC" },
    {  312060, "CoverageCo" },
    {  312070, "Adams Networks Inc" },
    {  312080, "South Georgia Regional Information Technology Authority" },
    {  312090, "AT&T Mobility" },
    {  312100, "ClearSky Technologies Inc" },
    {  312110, "Texas Energy Network LLC" },
    {  312120, "East Kentucky Network LLC dba Appalachian Wireless" },
    {  312130, "East Kentucky Network LLC dba Appalachian Wireless" },
    {  312140, "Cleveland Unlimited Inc" },
    {  312150, "Northwest Cell" },
    {  312160, "RSA1 Limited Partnership dba Chat Mobility" },
    {  312170, "Iowa RSA No. 2 Limited Partnership" },
    {  312180, "Keystone Wireless LLC" },
    {  312190, "Sprint" },
    {  312200, "Voyager Mobility LLC" },
    {  312210, "ASPENTA, LLC" },
    {  312220, "Chariton Valley Communications Corporation, Inc." },
    {  312230, "SRT Communications Inc." },
    {  312240, "Sprint" },
    {  312250, "Sprint" },
    {  312260, "Central LTE Holdings" },
    {  312270, "Cellular Network Partnership dba Pioneer Cellular" },
    {  312280, "Cellular Network Partnership dba Pioneer Cellular" },
    {  312290, "Uintah Basin Electronic Telecommunications" },
    {  312300, "Telecom North America Mobile Inc." },
    {  312310, "Clear Stream Communications, LLC" },
    {  312320, "S and R Communications LLC" },
    {  312330, "Nemont Communications, Inc." },
    {  312340, "MTA Communications dba MTA Wireless" },
    {  312350, "Triangle Communication System Inc." },
    {  312360, "Wes-Tex Telecommunications, LTD" },
    {  312370, "Commnet Wireless" },
    {  312380, "Copper Valley Wireless" },
    {  312390, "FTC Communications LLC" },
    {  312400, "Mid-Rivers Telephone Cooperative" },
    {  312410, "Eltopia Communications, LLC" },
    {  312420, "Nex-Tech Wireless, LLC" },
    {  312430, "Silver Star Communications" },
    {  312440, "Consolidated Telcom" },
    {  312450, "Cable & Communications Corporation" },
    {  312460, "KPU Telecommunications Division" },
    {  312470, "Carolina West Wireless, Inc." },
    {  312480, "Sagebrush Cellular, Inc." },
    {  312490, "TrustComm, Inc" },
    {  312510, "Wue" },
    {  312530, "Sprint" },
    {  312550, "Great Plains Communications, Inc" },
    {  312570, "Buffalo-Lake Erie Wireless Systems Co., LLC" },
    {  312580, "Morgan, Lewis & Bockius LLP" },
    {  312590, "Northern Michigan University" },
    {  312600, "Sagebrush Cellular, Inc." },
    {  312620, "GlobeTouch Inc." },
    {  312630, "NetGenuity, Inc." },
    {  312650, "365 Wireless LLC" },
    {  312670, "AT&T Mobility" },
    {  312680, "AT&T Mobility" },
    {  312690, "Tecore Government Services, LLC" },
    {  312700, "Wireless Partners,LLC" },
    {  312710, "Great North Woods Wireless LLC" },
    {  312720, "Southern Communications Services, Inc. D/B/A SouthernLINC Wireless" },
    {  312730, "Triangle Communication System Inc." },
    {  312740, "KDDI America, Inc dba Locus Telecommunications" },
    {  312750, "Artemis" },
    {  312760, "ARCTIC SLOPE TELEPHONE ASSOCIATION COOPERATIVE" },
    {  312770, "Verizon Wireless" },
    {  312780, "RedZone Wireless LLC" },
    {  312790, "Gila Electronics" },
    {  312800, "Cirrus Core Networks" },
    {  312810, "Bristol Bay Telephone Cooperative" },
    {  312820, "Santel Communications Cooperative, Inc." },
    {  312830, "Kings County Office of Education" },
    {  312840, "South Georgia Regional Information Technology" },
    {  312850, "Onvoy Spectrum, LLC" },
    {  312860, "Flat Wireless, LLC" },
    {  312870, "GigSky Mobile, LLC" },
    {  312880, "Albemarle County Public Schools" },
    {  312890, "Circle Gx" },
    {  312900, "Flat West Wireless, LLC" },
    {  312910, "East Kentucky Network LLC dba Appalachian Wireless" },
    {  312920, "Northeast Wireless Networks" },
    {  312930, "Hewlett-Packard Communication Services, LLC" },
    {  312940, "Webformix" },
    {  312950, "Custer Telephone Cooperative, Inc" },
    {  312960, "M&A Technology, Inc." },
    {  312970, "IOSAZ Intellectual Property LLC" },
    {  312980, "Mark Twain Communications Company" },
    {  312990, "Premier Holdings LLC DBA Premier Broadband" },
    {  313000, "Tennessee Wireless" },
    {  313010, "Cross Wireless LLC dba Sprocket Wireless" },
    {  313020, "CTC Telecom, INC. dba CTC Wireless" },
    {  313030, "Eagle Telephone System, INC dba Snake River PCS" },
    {  313040, "Nucla-Naturita Telephone Company" },
    {  313050, "Manti Tele Communications Company, Inc. dba Breakaway Wireless" },
    {  313060, "Country Wireless" },
    {  313070, "Midwest Network Solutions Hub LLC" },
    {  313080, "Speedwavz LLP" },
    {  313090, "Vivint Wireless, Inc" },
    {  313100, "First Responder Network Authority" },
    {  313200, "Mercury Network Corporation" },
    {  313210, "AT&T" },
    {  316010, "Sprint" },
    {  316011, "Southern Communications Services Inc." },
    {  334001, "COMUNICACIONES DIGITALES DEL NORTE, S.A. DE C.V" },
    {  334010, "NII DIGITAL, S. DE R.L. DE C.V." },
    {  334020, "RADIOMOVIL DIPSA, S.A. DE C.V." },
    {  334030, "PEGASO COMUNICACIONES Y SISTEMAS, S.A. DE C.V." },
    {  334040, "IUSACELL PCS DE MEXICO, S.A. DE  C.V." },
    {  334050, "COMUNICACIONES CELULARES DE OCCIDENTE, S.A. DE C.V. / SISTEMAS TELEFONICOS PORTATILES CELULARES, S.A. DE C.V. / TELECOMUNICACIONES DEL GOLFO, S.A. DE C.V. / SOS TELECOMUNICACIONES, S.A. DE C.V. / PORTATEL DEL SURESTE, S.A. DE C.V." },
    {  334060, "SERVICIOS DE ACCESO INALAMBRICO, S.A DE C.V." },
    {  334066, "TELEFONOS DE MEXICO, S.A.B. DE C.V" },
    {  334070, "OPERADORA UNEFON, S. A. DE C.V." },
    {  334080, "OPERADORA UNEFON, S. A. DE C.V." },
    {  334090, "NII DIGITAL, S. DE R.L. DE C.V." },
    {  338020, "Cable & Wireless Jamaica Ltd." },
    {  338040, "Symbiote Investment Limited" },
    {  338050, "Digicel (Jamaica) Ltd." },
    {  338110, "Cable & Wireless Jamaica Ltd." },
    {  342600, "Cable & Wireless (Barbados) Ltd. trading as Lime" },
    {  342820, "Sunbeach Communications" },
    {  344030, "APUA PCS" },
    {  344920, "Cable & Wireless (Antigua) trading as Lime" },
    {  344930, "AT&T Wireless (Antigua)" },
    {  346140, "Cable & Wireless (Cayman) trading as Lime" },
    {  348170, "Cable & Wireless (BVI) Ltd trading as Lime" },
    {  348370, "BVI Cable TV Ltd" },
    {  348570, "Caribbean Cellular Telephone Ltd." },
    {  348770, "Digicel (BVI) Ltd" },
    {  350000, "Bermuda Digital Communications Ltd (CellOne)" },
    {  352110, "Cable & Wireless Grenada ltd trading as lime" },
    {  354860, "Cable & Wireless (West Indies) Ltd trading as Lime" },
    {  356110, "Cable & Wireless St Kitts & Nevis Ltd trading as Lime" },
    {  358110, "Cable & Wireless (St Lucia) Ltd trading as Lime" },
    {  360110, "Cable & Wireless St Vincent and the Grenadines Ltd trading as Lime" },
    {  365010, "Weblinks Limited" },
    {  365840, "Cable and Wireless (Anguilla) Ltd trading as Lime" },
    {  366110, "Cable & Wireless Dominica Ltd trading as Lime" },
    {  374130, "Digicel Trinidad and Tobago Ltd." },
    {  374140, "LaqTel Ltd." },
    {  376350, "Cable & Wireless (TCI) Ltd trading as Lime" },
    {  376352, "IslandCom Communications Ltd." },
    {  376360, "IslandCom Communication Ltd" },
    {  405000, "Shyam Telelink Ltd, Rajasthan" },
    {  405005, "Reliance Communications Ltd/GSM, Delhi" },
    {  405006, "Reliance Communications Ltd/GSM, Gujarat" },
    {  405007, "Reliance Communications Ltd/GSM, Haryana" },
    {  405009, "Reliance Communications Ltd/GSM, J&K" },
    {  405010, "Reliance Communications Ltd,/GSM Karnataka" },
    {  405011, "Reliance Communications Ltd/GSM, Kerala" },
    {  405012, "Reliance Infocomm Ltd, Andhra Pradesh" },
    {  405013, "Reliance Communications Ltd/GSM, Maharashtra" },
    {  405014, "Reliance Communications Ltd/GSM, Madhya Pradesh" },
    {  405018, "Reliance Communications Ltd/GSM, Punjab" },
    {  405020, "Reliance Communications Ltd/GSM, Tamilnadu" },
    {  405021, "Reliance Communications Ltd/GSM, UP (East)" },
    {  405022, "Reliance Communications Ltd/GSM, UP (West)" },
    {  405025, "Tata Teleservices Ltd/GSM, Andhra Pradesh" },
    {  405027, "Tata Teleservices Ltd/GSM, Bihar" },
    {  405029, "Tata Teleservices Ltd/GSM, Delhi" },
    {  405030, "Tata Teleservices Ltd/GSM, Gujarat" },
    {  405031, "Tata Teleservices Ltd/GSM, Haryana" },
    {  405032, "Tata Teleservices Ltd/GSM, Himachal Pradesh" },
    {  405033, "Reliance Infocomm Ltd, Bihar" },
    {  405034, "Tata Teleservices Ltd/GSM, Kamataka" },
    {  405035, "Tata Teleservices Ltd/GSM, Kerala" },
    {  405036, "Tata Teleservices Ltd/GSM, Kolkata" },
    {  405037, "Tata Teleservices Ltd/GSM, Maharashtra" },
    {  405038, "Tata Teleservices Ltd/GSM, Madhya Pradesh" },
    {  405039, "Tata Teleservices Ltd/GSM, Mumbai" },
    {  405040, "Reliance Infocomm Ltd, Chennai" },
    {  405041, "Tata Teleservices Ltd/GSM, Orissa" },
    {  405042, "Tata Teleservices Ltd/GSM, Punjab" },
    {  405043, "Tata Teleservices Ltd/GSM, Rajasthan" },
    {  405044, "Tata Teleservices Ltd/GSM, Tamilnadu" },
    {  405045, "Tata Teleservices Ltd/GSM, UP (East)" },
    {  405046, "Tata Teleservices Ltd/GSM, UP (West)" },
    {  405047, "Tata Teleservices Ltd/GSM, West Bengal" },
    {  405750, "Vodafone Essar Spacetel Ltd, J&K" },
    {  405751, "Vodafone Essar Spacetel Ltd, Assam" },
    {  405752, "Vodafone Essar Spacetel Ltd, Bihar" },
    {  405753, "Vodafone Essar Spacetel Ltd, Orissa" },
    {  405754, "Vodafone Essar Spacetel Ltd, Himachal Pradesh" },
    {  405755, "Vodafone Essar Spacetel Ltd, North East" },
    {  405799, "Idea Cellular Ltd, MUMBAI" },
    {  405800, "Aircell Ltd, Delhi" },
    {  405801, "Aircell Ltd, Andhra Pradesh" },
    {  405802, "Aircell Ltd, Gujarat" },
    {  405803, "Aircell Ltd, Kamataka" },
    {  405804, "Aircell Ltd, Maharashtra" },
    {  405805, "Aircell Ltd, Mumbai" },
    {  405806, "Aircell Ltd, Rajasthan" },
    {  405807, "Dishnet Wireless Ltd, Haryana" },
    {  405808, "Dishnet Wireless Ltd, Madhya Pradesh" },
    {  405809, "Dishnet Wireless Ltd, Kerala" },
    {  708001, "Megatel" },
    {  708002, "Celtel" },
    {  708040, "Digicel Honduras" },
    {  714020, "Telefonica Moviles Panama S.A." },
    {  722010, "Compania de Radiocomunicaciones Moviles S.A." },
    {  722020, "Nextel Argentina srl" },
    {  722070, "Telefonica Comunicaciones Personales S.A." },
    {  722310, "CTI PCS S.A." },
    {  722320, "Compania de Telefonos del Interior Norte S.A." },
    {  722330, "Compania de Telefonos del Interior S.A." },
    {  722341, "Telecom Personal S.A." },
    {  732001, "Colombia Telecomunicaciones S.A. - Telecom" },
    {  732002, "Edatel S.A." },
    {  732020, "Emtelsa" },
    {  732099, "Emcali" },
    {  732101, "Comcel S.A. Occel S.A./Celcaribe" },
    {  732102, "Bellsouth Colombia S.A." },
    {  732103, "Colombia Movil S.A." },
    {  732111, "Colombia Movil S.A." },
    {  732123, "Telefonica Moviles Colombia S.A." },
    {  732130, "Avantel" },
    {  738002, "Guyana Telephone & Telegraph Company Limited (Cellink)" },
    {  738003, "Quark Communications Inc." },
    {  750001, "Touch" },
    { 1666665, "Unset" },
    {  0, NULL }
};

value_string_ext mcc_mnc_3digits_codes_ext = VALUE_STRING_EXT_INIT(mcc_mnc_3digits_codes);

static int proto_e212   = -1;
static int hf_E212_imsi = -1;
static int hf_E212_mcc  = -1;
static int hf_E212_mcc_lai = -1;
static int hf_E212_mcc_sai = -1;
static int hf_E212_mcc_rai = -1;
static int hf_E212_mcc_cgi = -1;
static int hf_E212_mcc_ecgi = -1;
static int hf_E212_mcc_tai = -1;
static int hf_E212_mnc  = -1;
static int hf_E212_mnc_lai = -1;
static int hf_E212_mnc_sai = -1;
static int hf_E212_mnc_rai = -1;
static int hf_E212_mnc_cgi = -1;
static int hf_E212_mnc_ecgi = -1;
static int hf_E212_mnc_tai = -1;

static int ett_e212_imsi = -1;

static expert_field ei_E212_mcc_non_decimal = EI_INIT;
static expert_field ei_E212_mnc_non_decimal = EI_INIT;


/* static int hf_E212_msin = -1; */

/*
 * MCC/MNC dissection - little endian MNC encoding
 *
 * MNC of length 2:
 *
 *   8   7   6   5   4   3   2   1
 * +---+---+---+---+---+---+---+---+
 * |  MCC digit 2  |  MCC digit 1  |  octet x
 * +---------------+---------------+
 * |    Filler     |  MCC digit 3  |  octet x+1
 * +---------------+---------------+
 * |  MNC digit 2  |  MNC digit 1  |  octet x+2
 * +---------------+---------------+
 *
 * MNC of length 3:
 *
 *   8   7   6   5   4   3   2   1
 * +---+---+---+---+---+---+---+---+
 * |  MCC digit 2  |  MCC digit 1  |  octet x
 * +---------------+---------------+
 * |  MNC digit 3  |  MCC digit 3  |  octet x+1
 * +---------------+---------------+
 * |  MNC digit 2  |  MNC digit 1  |  octet x+2
 * +---------------+---------------+
 *
 *
 * MCC/MNC dissection - big endian MNC encoding
 *
 * MNC of length 2:
 *
 *   8   7   6   5   4   3   2   1
 * +---+---+---+---+---+---+---+---+
 * |  MCC digit 2  |  MCC digit 1  |  octet x
 * +---------------+---------------+
 * |    Filler     |  MCC digit 3  |  octet x+1
 * +---------------+---------------+
 * |  MNC digit 2  |  MNC digit 1  |  octet x+2
 * +---------------+---------------+
 *
 * MNC of length 3:
 *
 *   8   7   6   5   4   3   2   1
 * +---+---+---+---+---+---+---+---+
 * |  MCC digit 2  |  MCC digit 1  |  octet x
 * +---------------+---------------+
 * |  MNC digit 1  |  MCC digit 3  |  octet x+1
 * +---------------+---------------+
 * |  MNC digit 3  |  MNC digit 2  |  octet x+2
 * +---------------+---------------+
 */

/*
 * Return MCC MNC in a packet scope allocated string that can be used in labels.
 */
gchar *
dissect_e212_mcc_mnc_wmem_packet_str(tvbuff_t *tvb, packet_info *pinfo, proto_tree *tree, int offset, e212_number_type_t number_type, gboolean little_endian)
{

    int         start_offset, mcc_mnc;
    guint8      octet;
    guint16     mcc, mnc;
    guint8      mcc1, mcc2, mcc3, mnc1, mnc2, mnc3;
    proto_item *item;
    gchar      *mcc_mnc_str;
    gboolean    long_mnc = FALSE;
    int         hf_E212_mcc_id, hf_E212_mnc_id;

    switch(number_type){
    case E212_LAI:
        hf_E212_mcc_id = hf_E212_mcc_lai;
        hf_E212_mnc_id = hf_E212_mnc_lai;
        break;
    case E212_RAI:
        hf_E212_mcc_id = hf_E212_mcc_rai;
        hf_E212_mnc_id = hf_E212_mnc_rai;
        break;
    case E212_SAI:
        hf_E212_mcc_id = hf_E212_mcc_sai;
        hf_E212_mnc_id = hf_E212_mnc_sai;
        break;
    case E212_CGI:
        hf_E212_mcc_id = hf_E212_mcc_cgi;
        hf_E212_mnc_id = hf_E212_mnc_cgi;
        break;
    case E212_ECGI:
        hf_E212_mcc_id = hf_E212_mcc_ecgi;
        hf_E212_mnc_id = hf_E212_mnc_ecgi;
        break;
    case E212_TAI:
        hf_E212_mcc_id = hf_E212_mcc_tai;
        hf_E212_mnc_id = hf_E212_mnc_tai;
        break;
    default:
        hf_E212_mcc_id = hf_E212_mcc;
        hf_E212_mnc_id = hf_E212_mnc;
    }
    start_offset = offset;
    /* MCC + MNC */
    mcc_mnc = tvb_get_ntoh24(tvb,offset);
    /* Mobile country code MCC */
    octet = tvb_get_guint8(tvb,offset);
    mcc1 = octet & 0x0f;
    mcc2 = octet >> 4;
    offset++;
    octet = tvb_get_guint8(tvb,offset);
    mcc3 = octet & 0x0f;
    /* MNC, Mobile network code (octet 3 bits 5 to 8, octet 4)  */
    mnc3 = octet >> 4;
    offset++;
    octet = tvb_get_guint8(tvb,offset);
    mnc1 = octet & 0x0f;
    mnc2 = octet >> 4;

    mcc = 100 * mcc1 + 10 * mcc2 + mcc3;
    mnc = 10 * mnc1 + mnc2;
    if ((mnc3 != 0xf) || (mcc_mnc == 0xffffff)) {
        long_mnc = TRUE;
        if(little_endian)
            mnc = 10 * mnc + mnc3;
        else
            mnc = 100 * mnc3 + mnc;
    }
    item = proto_tree_add_uint(tree, hf_E212_mcc_id , tvb, start_offset, 2, mcc );
    if (((mcc1 > 9) || (mcc2 > 9) || (mcc3 > 9)) && (mcc_mnc != 0xffffff))
        expert_add_info(pinfo, item, &ei_E212_mcc_non_decimal);

    if (long_mnc) {
        item = proto_tree_add_uint_format_value(tree, hf_E212_mnc_id , tvb, start_offset + 1, 2, mnc,
                   "%s (%03u)",
                   val_to_str_ext_const(mcc * 1000 + mnc, &mcc_mnc_3digits_codes_ext, "Unknown"),
                   mnc);
        /* Prepare a string with the MCC and MNC including the country and Operator if
         * known, do NOT print unknown.
         */
        mcc_mnc_str = wmem_strdup_printf(wmem_packet_scope(), "MCC %u %s, MNC %03u %s",
            mcc,
            val_to_str_ext_const(mcc,&E212_codes_ext,""),
            mnc,
            val_to_str_ext_const(mcc * 1000 + mnc, &mcc_mnc_3digits_codes_ext, ""));
    } else {
        item = proto_tree_add_uint_format_value(tree, hf_E212_mnc_id , tvb, start_offset + 1, 2, mnc,
                   "%s (%02u)",
                   val_to_str_ext_const(mcc * 100 + mnc, &mcc_mnc_2digits_codes_ext, "Unknown"),
                   mnc);
        /* Prepare a string with the MCC and MNC including the country and Operator if
         * known, do NOT print unknown.
         */
        mcc_mnc_str = wmem_strdup_printf(wmem_packet_scope(), "MCC %u %s, MNC %02u %s",
            mcc,
            val_to_str_ext_const(mcc,&E212_codes_ext,""),
            mnc,
            val_to_str_ext_const(mcc * 100 + mnc, &mcc_mnc_2digits_codes_ext, ""));
    }

    if (((mnc1 > 9) || (mnc2 > 9) || ((mnc3 > 9) && (mnc3 != 0x0f))) && (mcc_mnc != 0xffffff))
        expert_add_info(pinfo, item, &ei_E212_mnc_non_decimal);

    return mcc_mnc_str;
}

int
dissect_e212_mcc_mnc(tvbuff_t *tvb, packet_info *pinfo, proto_tree *tree, int offset, e212_number_type_t number_type, gboolean little_endian)
{
    dissect_e212_mcc_mnc_wmem_packet_str(tvb, pinfo, tree, offset, number_type, little_endian);
    return offset +3;
}

/*
 * When we want to decode the MCC/MNC pair in an address that is encoded according to E.212
 * the main problem is that we do not know whether we are dealing with a length 2 or length 3
 * MNC. Initially, it was possible to find the length of the MNC by checking the MCC code.
 * Originally each country employed a pure 2 digit or 3 digit MNC scheme. However, it is possible
 * to find countries now that employ both lengths for the MNC (e.g. Canada, India).
 * Since in these cases we can be sure that an MNC cannot possible be a prefix for another MNC, we
 * initially assume that the MNC is 2 digits long. If the MCC/MNC combination gives a match in our
 * mcc_mnc_codes list then we can be sure that we deal with an MNC that is 2 digits long. Otherwise,
 * assume that the MNC is 3 digits long.
 *
 * MNC of length 2:
 *
 *     8   7   6   5   4   3   2   1
 *   +---+---+---+---+---+---+---+---+
 *   |  MCC digit 2  |  MCC digit 1  |  octet x
 *   +---------------+---------------+
 *   |  MNC digit 1  |  MCC digit 3  |  octet x+1
 *   +---------------+---------------+
 *   | addr digit 1  |  MNC digit 2  |  octet x+2
 *   +---------------+---------------+
 *
 * MNC of length 3:
 *
 *     8   7   6   5   4   3   2   1
 *   +---+---+---+---+---+---+---+---+
 *   |  MCC digit 2  |  MCC digit 1  |  octet x
 *   +---------------+---------------+
 *   |  MNC digit 1  |  MCC digit 3  |  octet x+1
 *   +---------------+---------------+
 *   |  MNC digit 3  |  MNC digit 2  |  octet x+2
 *   +---------------+---------------+
 *
 * This function will consume either 2.5 or 3 octets. For this reason it returns
 * the number of nibbles consumed, i.e. 5 or 6 (2 or 3 digits long MNC respectively)
 */
int
dissect_e212_mcc_mnc_in_address(tvbuff_t *tvb, packet_info *pinfo, proto_tree *tree, int offset)
{
    guint32     start_offset, mcc_mnc;
    guint8      octet;
    guint16     mcc, mnc;
    guint8      mcc1, mcc2, mcc3, mnc1, mnc2, mnc3;
    proto_item *item;
    gboolean    long_mnc;

    long_mnc = FALSE;
    start_offset = offset;

    /* MCC + MNC */
    mcc_mnc = tvb_get_ntoh24(tvb,offset);

    /* MCC digits 1 and 2 */
    octet = tvb_get_guint8(tvb,offset);
    mcc1  = octet & 0x0f;
    mcc2  = octet >> 4;
    offset++;

    /* MCC digit 3 and MNC digit 1 */
    octet = tvb_get_guint8(tvb,offset);
    mcc3  = octet & 0x0f;
    mnc1  = octet >> 4;
    offset++;

    /* MNC digits 2 and 3 */
    octet = tvb_get_guint8(tvb,offset);
    mnc2  = octet & 0x0f;
    mnc3  = octet >> 4;

    mcc   = 100 * mcc1 + 10 * mcc2 + mcc3;
    mnc   = 10 * mnc1 + mnc2;

    /* Try to match the MCC and 2 digits MNC with an entry in our list of operators */
    if (!try_val_to_str_ext(mcc * 100 + mnc, &mcc_mnc_2digits_codes_ext)) {
        mnc = 10 * mnc + mnc3;
        long_mnc = TRUE;
    }

    item = proto_tree_add_uint(tree, hf_E212_mcc , tvb, start_offset, 2, mcc );
    if (((mcc1 > 9) || (mcc2 > 9) || (mcc3 > 9)) & (mcc_mnc != 0xffffff))
        expert_add_info(pinfo, item, &ei_E212_mcc_non_decimal);

    if (long_mnc)
        item = proto_tree_add_uint_format_value(tree, hf_E212_mnc , tvb, start_offset + 1, 2, mnc,
                   "%s (%03u)",
                   val_to_str_ext_const(mcc * 1000 + mnc, &mcc_mnc_3digits_codes_ext, "Unknown"),
                   mnc);
    else
        item = proto_tree_add_uint_format_value(tree, hf_E212_mnc , tvb, start_offset + 1, 2, mnc,
                   "%s (%02u)",
                   val_to_str_ext_const(mcc * 100 + mnc, &mcc_mnc_2digits_codes_ext, "Unknown"),
                   mnc);

    if (((mnc1 > 9) || (mnc2 > 9) || (long_mnc && (mnc3 > 9))) && (mcc_mnc != 0xffffff))
        expert_add_info(pinfo, item, &ei_E212_mnc_non_decimal);

    if (long_mnc)
        return 6;
    else
        return 5;
}

/*
 * MNC of length 2:
 *
 *     8   7   6   5   4   3   2   1
 *   +---+---+---+---+---+---+---+---+
 *   |  MCC digit 1  |  Other data   |  octet x
 *   +---------------+---------------+
 *   |  MNC digit 1  |  MCC digit 2  |  octet x+1
 *   +---------------+---------------+
 *   | MNC digit  3  |  MNC digit 2  |  octet x+2
 *   +---------------+---------------+
 *
 * MNC of length 3:
 *
 *     8   7   6   5   4   3   2   1
 *   +---+---+---+---+---+---+---+---+
 *   |  MCC digit 1  |  Other data   |  octet x
 *   +---------------+---------------+
 *   |  MCC digit 3  |  MCC digit 2  |  octet x+1
 *   +---------------+---------------+
 *   |  MNC digit 2  |  MNC digit 1  |  octet x+2
 *   +---------------+---------------+
 *   |  .....        |  MNC digit 3  |  octet x+3
 *   +---------------+---------------+
 */
static int
dissect_e212_mcc_mnc_high_nibble(tvbuff_t *tvb, packet_info *pinfo _U_, proto_tree *tree, int offset)
{

    guint32     start_offset;
    guint8      octet;
    guint16     mcc, mnc;
    guint8      mcc1, mcc2, mcc3, mnc1, mnc2, mnc3;
    gboolean    long_mnc;

    long_mnc = FALSE;
    start_offset = offset;

    /* MCC digits 1 */
    octet = tvb_get_guint8(tvb,offset);
    mcc1  = octet >> 4;
    offset++;

    /* MCC digits 1 and 2 */
    octet = tvb_get_guint8(tvb,offset);
    mcc2  = octet & 0x0f;
    mcc3  = octet >> 4;
    offset++;

    /* MNC digit 1 and MNC digit 2 */
    octet = tvb_get_guint8(tvb,offset);
    mnc1  = octet & 0x0f;
    mnc2  = octet >> 4;
    offset++;

    /* MNC digits 3 */
    octet = tvb_get_guint8(tvb,offset);
    mnc3  = octet & 0x0f;

    mcc   = 100 * mcc1 + 10 * mcc2 + mcc3;
    mnc   = 10 * mnc1 + mnc2;

    /* Try to match the MCC and 2 digits MNC with an entry in our list of operators */
    if (!try_val_to_str_ext(mcc * 100 + mnc, &mcc_mnc_2digits_codes_ext)) {
        mnc = 10 * mnc + mnc3;
        long_mnc = TRUE;
    }

    proto_tree_add_uint(tree, hf_E212_mcc , tvb, start_offset, 2, mcc );

    if (long_mnc)
        proto_tree_add_uint_format_value(tree, hf_E212_mnc , tvb, start_offset + 2, 2, mnc,
                   "%s (%03u)",
                   val_to_str_ext_const(mcc * 1000 + mnc, &mcc_mnc_3digits_codes_ext, "Unknown"),
                   mnc);
    else
        proto_tree_add_uint_format_value(tree, hf_E212_mnc , tvb, start_offset + 2, 1, mnc,
                   "%s (%02u)",
                   val_to_str_ext_const(mcc * 100 + mnc, &mcc_mnc_2digits_codes_ext, "Unknown"),
                   mnc);

    if (long_mnc)
        return 7;
    else
        return 5;
}

int
dissect_e212_mcc_mnc_in_utf8_address(tvbuff_t *tvb, packet_info *pinfo _U_, proto_tree *tree, int offset)
{
    guint16 mcc = 0, mnc = 0;
    gboolean    long_mnc = FALSE;

    ws_strtou16(tvb_get_string_enc(wmem_packet_scope(), tvb, offset, 3, ENC_UTF_8),
        NULL, &mcc);
    ws_strtou16(tvb_get_string_enc(wmem_packet_scope(), tvb, offset + 3, 2, ENC_UTF_8),
        NULL, &mnc);

    /* Try to match the MCC and 2 digits MNC with an entry in our list of operators */
    if (!try_val_to_str_ext(mcc * 100 + mnc, &mcc_mnc_2digits_codes_ext)) {
        ws_strtou16(tvb_get_string_enc(wmem_packet_scope(), tvb, offset + 3, 3, ENC_UTF_8),
            NULL, &mnc);
        long_mnc = TRUE;
    }

    proto_tree_add_uint(tree, hf_E212_mcc, tvb, offset, 3, mcc );

    if (long_mnc)
        proto_tree_add_uint_format_value(tree, hf_E212_mnc, tvb, offset + 3, 3, mnc,
                   "%s (%03u)",
                   val_to_str_ext_const(mcc * 1000 + mnc, &mcc_mnc_3digits_codes_ext, "Unknown1"),
                   mnc);
    else
        proto_tree_add_uint_format_value(tree, hf_E212_mnc, tvb, offset + 3, 2, mnc,
                   "%s (%02u)",
                   val_to_str_ext_const(mcc * 100 + mnc, &mcc_mnc_2digits_codes_ext, "Unknown2"),
                   mnc);

    if (long_mnc)
        return 6;
    else
        return 5;
}

const gchar *
dissect_e212_imsi(tvbuff_t *tvb, packet_info *pinfo, proto_tree *tree, int offset, int length, gboolean skip_first)
{
    proto_item *item;
    proto_tree *subtree;
    const gchar *imsi_str;

    /* Fetch the BCD encoded digits from tvb indicated half byte, formating the digits according to
     * a default digit set of 0-9 returning "?" for overdecadic digits a pointer to the wmem
     * allocated string will be returned.
     */
    imsi_str = tvb_bcd_dig_to_wmem_packet_str( tvb, offset, length, NULL, skip_first);
    item = proto_tree_add_string(tree, hf_E212_imsi, tvb, offset, length, imsi_str);

    subtree = proto_item_add_subtree(item, ett_e212_imsi);

    if(skip_first) {
        dissect_e212_mcc_mnc_high_nibble(tvb, pinfo, subtree, offset);
    } else {
        dissect_e212_mcc_mnc_in_address(tvb, pinfo, subtree, offset);
    }

    return imsi_str;
}

const gchar *
dissect_e212_utf8_imsi(tvbuff_t *tvb, packet_info *pinfo, proto_tree *tree, int offset, int length)
{
    proto_item *item;
    proto_tree *subtree;
    const gchar *imsi_str;

    /* Fetch the UTF8-encoded IMSI */
    imsi_str = tvb_get_string_enc(wmem_packet_scope(), tvb, offset, length, ENC_UTF_8);
    item = proto_tree_add_string(tree, hf_E212_imsi, tvb, offset, length, imsi_str);

    subtree = proto_item_add_subtree(item, ett_e212_imsi);

    dissect_e212_mcc_mnc_in_utf8_address(tvb, pinfo, subtree, offset);

    return imsi_str;
}

/*
 * Register the protocol with Wireshark.
 *
 * This format is required because a script is used to build the C function
 * that calls all the protocol registration.
 */


void
proto_register_e212(void)
{

/* Setup list of header fields  See Section 1.6.1 for details */
    static hf_register_info hf[] = {
    { &hf_E212_imsi,
        { "IMSI","e212.imsi",
        FT_STRING, BASE_NONE, NULL, 0x0,
        "International mobile subscriber identity(IMSI)", HFILL }
    },
    { &hf_E212_mcc,
        { "Mobile Country Code (MCC)","e212.mcc",
        FT_UINT16, BASE_DEC|BASE_EXT_STRING, &E212_codes_ext, 0x0,
        "Mobile Country Code MCC", HFILL }
    },
    { &hf_E212_mcc_lai,
        { "Mobile Country Code (MCC)","e212.lai.mcc",
        FT_UINT16, BASE_DEC|BASE_EXT_STRING, &E212_codes_ext, 0x0,
        "Mobile Country Code MCC", HFILL }
    },
    { &hf_E212_mcc_rai,
        { "Mobile Country Code (MCC)","e212.rai.mcc",
        FT_UINT16, BASE_DEC|BASE_EXT_STRING, &E212_codes_ext, 0x0,
        "Mobile Country Code MCC", HFILL }
    },
    { &hf_E212_mcc_sai,
        { "Mobile Country Code (MCC)","e212.sai.mcc",
        FT_UINT16, BASE_DEC|BASE_EXT_STRING, &E212_codes_ext, 0x0,
        "Mobile Country Code MCC", HFILL }
    },
    { &hf_E212_mcc_cgi,
        { "Mobile Country Code (MCC)","e212.cgi.mcc",
        FT_UINT16, BASE_DEC|BASE_EXT_STRING, &E212_codes_ext, 0x0,
        "Mobile Country Code MCC", HFILL }
    },
    { &hf_E212_mcc_ecgi,
        { "Mobile Country Code (MCC)","e212.ecgi.mcc",
        FT_UINT16, BASE_DEC|BASE_EXT_STRING, &E212_codes_ext, 0x0,
        "Mobile Country Code MCC", HFILL }
    },
    { &hf_E212_mcc_tai,
        { "Mobile Country Code (MCC)","e212.tai.mcc",
        FT_UINT16, BASE_DEC|BASE_EXT_STRING, &E212_codes_ext, 0x0,
        "Mobile Country Code MCC", HFILL }
    },
    { &hf_E212_mnc,
        { "Mobile Network Code (MNC)","e212.mnc",
        FT_UINT16, BASE_DEC, NULL, 0x0,
        "Mobile network code", HFILL }
    },
    { &hf_E212_mnc_lai,
        { "Mobile Network Code (MNC)","e212.lai.mnc",
        FT_UINT16, BASE_DEC, NULL, 0x0,
        "Mobile network code", HFILL }
    },
    { &hf_E212_mnc_rai,
        { "Mobile Network Code (MNC)","e212.rai.mnc",
        FT_UINT16, BASE_DEC, NULL, 0x0,
        "Mobile network code", HFILL }
    },
    { &hf_E212_mnc_sai,
        { "Mobile Network Code (MNC)","e212.sai.mnc",
        FT_UINT16, BASE_DEC, NULL, 0x0,
        "Mobile network code", HFILL }
    },
    { &hf_E212_mnc_cgi,
        { "Mobile Network Code (MNC)","e212.cgi.mnc",
        FT_UINT16, BASE_DEC, NULL, 0x0,
        "Mobile network code", HFILL }
    },
    { &hf_E212_mnc_ecgi,
        { "Mobile Network Code (MNC)","e212.ecgi.mnc",
        FT_UINT16, BASE_DEC, NULL, 0x0,
        "Mobile network code", HFILL }
    },
    { &hf_E212_mnc_tai,
        { "Mobile Network Code (MNC)","e212.tai.mnc",
        FT_UINT16, BASE_DEC, NULL, 0x0,
        "Mobile network code", HFILL }
    },
#if 0
    { &hf_E212_msin,
        { "Mobile Subscriber Identification Number (MSIN)", "e212.msin",
        FT_STRING, BASE_NONE, NULL, 0,
        "Mobile Subscriber Identification Number(MSIN)", HFILL }},
#endif
    };


    static gint *ett_e212_array[] = {
        &ett_e212_imsi,
    };

    static ei_register_info ei[] = {
        { &ei_E212_mcc_non_decimal, { "e212.mcc.non_decimal", PI_MALFORMED, PI_WARN, "MCC contains non-decimal digits", EXPFILL }},
        { &ei_E212_mnc_non_decimal, { "e212.mnc.non_decimal", PI_MALFORMED, PI_WARN, "MNC contains non-decimal digits", EXPFILL }},
    };

    expert_module_t* expert_e212;

    /*
     * Register the protocol name and description
     */
    proto_e212 = proto_register_protocol(
            "ITU-T E.212 number",
            "E.212",
            "e212");

    /*
     * Required function calls to register
     * the header fields and subtrees used.
     */
    proto_register_field_array(proto_e212, hf, array_length(hf));
    proto_register_subtree_array(ett_e212_array, array_length(ett_e212_array));
    expert_e212 = expert_register_protocol(proto_e212);
    expert_register_field_array(expert_e212, ei, array_length(ei));

}

/*
 * Editor modelines  -  http://www.wireshark.org/tools/modelines.html
 *
 * Local variables:
 * c-basic-offset: 4
 * tab-width: 8
 * indent-tabs-mode: nil
 * End:
 *
 * vi: set shiftwidth=4 tabstop=8 expandtab:
 * :indentSize=4:tabSize=8:noTabs=true:
 */
