/* packet-e212.c
 * Routines for output and filtering of E.212 numbers common
 * to many dissectors.
 * Copyright 2006, Anders Broman <anders.broman@ericsson.com>
 *
 * Wireshark - Network traffic analyzer
 * By Gerald Combs <gerald@wireshark.org>
 * Copyright 1998 Gerald Combs
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 */
/*
 * Updates and corrections for Polish networks MCC=260
 * according to www.uke.gov.pl status from January 2011
 * Copyright 2011, Grzegorz Szczytowski <grzegorz.szczytowski@gmail.com>
 */
/* SoftBank PLMN codes updated according to http://ja.wikipedia.org/wiki/Mobile_Network_Code
 * as E.212 is still not updated
 */

#include "config.h"

#include <stdlib.h>

#include <epan/packet.h>

#include "packet-e212.h"
#include "expert.h"

void proto_register_e212(void);

/*
 * Annex to ITU Operational Bulletin
 * No. 1056 - 15.VII.2014
 *
 * COMPLEMENT TO ITU-T RECOMMENDATION E.212 (05/2008)
 *
 * Amendment No. 2 ITU Operational Bulletin No. 1058 - 15.VIII.2014
 *
 * Find the bulletins here:
 * http://www.itu.int/pub/T-SP-OB
 */
static const value_string E212_codes[] = {
    {  202, "Greece" },
    {  203, "Unassigned" },
    {  204, "Netherlands" },
    {  205, "Unassigned" },
    {  206, "Belgium" },
    {  207, "Unassigned" },
    {  208, "France" },
    {  209, "Unassigned" },
    {  210, "Unassigned" },
    {  211, "Unassigned" },
    {  212, "Unassigned" },
    {  213, "Andorra" },
    {  214, "Spain" },
    {  215, "Unassigned" },
    {  216, "Hungary" },
    {  217, "Unassigned" },
    {  218, "Bosnia and Herzegovina" },
    {  219, "Croatia" },
    {  220, "Serbia" },
    {  221, "Unassigned" },
    {  222, "Italy" },
    {  223, "Unassigned" },
    {  224, "Unassigned" },
    {  225, "Unassigned" },
    {  226, "Romania" },
    {  227, "Unassigned" },
    {  228, "Switzerland / Liechtenstein" },
    {  229, "Unassigned" },
    {  230, "Czech Rep." },
    {  231, "Slovakia" },
    {  232, "Austria" },
    {  233, "Unassigned" },
    {  234, "United Kingdom" },
    {  235, "United Kingdom" },
    {  236, "Unassigned" },
    {  237, "Unassigned" },
    {  238, "Denmark" },
    {  239, "Unassigned" },
    {  240, "Sweden" },
    {  241, "Unassigned" },
    {  242, "Norway" },
    {  243, "Unassigned" },
    {  244, "Finland" },
    {  245, "Unassigned" },
    {  246, "Lithuania" },
    {  247, "Latvia" },
    {  248, "Estonia" },
    {  249, "Unassigned" },
    {  250, "Russian Federation" },
    {  251, "Unassigned" },
    {  252, "Unassigned" },
    {  253, "Unassigned" },
    {  254, "Unassigned" },
    {  255, "Ukraine" },
    {  256, "Unassigned" },
    {  257, "Belarus" },
    {  258, "Unassigned" },
    {  259, "Moldova (Republic of)" },
    {  260, "Poland" },
    {  261, "Unassigned" },
    {  262, "Germany" },
    {  263, "Unassigned" },
    {  264, "Unassigned" },
    {  265, "Unassigned" },
    {  266, "Gibraltar" },
    {  267, "Unassigned" },
    {  268, "Portugal" },
    {  269, "Unassigned" },
    {  270, "Luxembourg" },
    {  271, "Unassigned" },
    {  272, "Ireland" },
    {  273, "Unassigned" },
    {  274, "Iceland / Faroe Islands" },
    {  275, "Unassigned" },
    {  276, "Albania" },
    {  277, "Unassigned" },
    {  278, "Malta" },
    {  279, "Unassigned" },
    {  280, "Cyprus" },
    {  281, "Unassigned" },
    {  282, "Georgia" },
    {  283, "Unassigned" },
    {  284, "Bulgaria" },
    {  285, "Unassigned" },
    {  286, "Turkey" },
    {  287, "Unassigned" },
    {  288, "Faroe Islands" },
    {  289, "Unassigned" },
    {  290, "Greenland" },
    {  291, "Unassigned" },
    {  292, "San Marino" },
    {  293, "Slovenia" },
    {  294, "The Former Yugoslav Republic of Macedonia" },
    {  295, "Liechtenstein" },
    {  296, "Unassigned" },
    {  297, "Montenegro" },
    {  298, "Unassigned" },
    {  299, "Unassigned" },
    {  300, "Unassigned" },
    {  301, "Unassigned" },
    {  302, "Canada" },
    {  303, "Unassigned" },
    {  304, "Unassigned" },
    {  305, "Unassigned" },
    {  306, "Unassigned" },
    {  307, "Unassigned" },
    {  308, "Saint Pierre and Miquelon" },
    {  309, "Unassigned" },
    {  310, "United States" },
    {  311, "United States" },
    {  312, "United States" },
    {  313, "United States" },
    {  314, "Unassigned" },
    {  315, "Unassigned" },
    {  316, "United States" },
    {  317, "Unassigned" },
    {  318, "Unassigned" },
    {  319, "Unassigned" },
    {  320, "Unassigned" },
    {  321, "Unassigned" },
    {  322, "Unassigned" },
    {  323, "Unassigned" },
    {  324, "Unassigned" },
    {  325, "Unassigned" },
    {  326, "Unassigned" },
    {  327, "Unassigned" },
    {  328, "Unassigned" },
    {  329, "Unassigned" },
    {  330, "Unassigned" },
    {  331, "Unassigned" },
    {  332, "Unassigned" },
    {  333, "Unassigned" },
    {  334, "Mexico" },
    {  335, "Unassigned" },
    {  336, "Unassigned" },
    {  337, "Unassigned" },
    {  338, "Jamaica" },
    {  339, "Unassigned" },
    {  340, "French Guiana / Guadeloupe / Martinique" },
    {  341, "Unassigned" },
    {  342, "Barbados" },
    {  343, "Unassigned" },
    {  344, "Antigua and Barbuda" },
    {  345, "Unassigned" },
    {  346, "Cayman Islands" },
    {  347, "Unassigned" },
    {  348, "British Virgin Islands" },
    {  349, "Unassigned" },
    {  350, "Bermuda" },
    {  351, "Unassigned" },
    {  352, "Grenada" },
    {  353, "Unassigned" },
    {  354, "Montserrat" },
    {  355, "Unassigned" },
    {  356, "Saint Kitts and Nevis" },
    {  357, "Unassigned" },
    {  358, "Saint Lucia" },
    {  359, "Unassigned" },
    {  360, "Saint Vincent and the Grenadines" },
    {  361, "Unassigned" },
    {  362, "Curacao" },
    {  363, "Aruba" },
    {  364, "Unassigned" },
    {  365, "Anguilla" },
    {  366, "Dominica" },
    {  367, "Unassigned" },
    {  368, "Cuba" },
    {  369, "Unassigned" },
    {  370, "Dominican Rep." },
    {  371, "Unassigned" },
    {  372, "Haiti" },
    {  373, "Unassigned" },
    {  374, "Trinidad and Tobago" },
    {  375, "Unassigned" },
    {  376, "Turks and Caicos Islands" },
    {  377, "Unassigned" },
    {  378, "Unassigned" },
    {  379, "Unassigned" },
    {  380, "Unassigned" },
    {  381, "Unassigned" },
    {  382, "Unassigned" },
    {  383, "Unassigned" },
    {  384, "Unassigned" },
    {  385, "Unassigned" },
    {  386, "Unassigned" },
    {  387, "Unassigned" },
    {  388, "Unassigned" },
    {  389, "Unassigned" },
    {  390, "Unassigned" },
    {  391, "Unassigned" },
    {  392, "Unassigned" },
    {  393, "Unassigned" },
    {  394, "Unassigned" },
    {  395, "Unassigned" },
    {  396, "Unassigned" },
    {  397, "Unassigned" },
    {  398, "Unassigned" },
    {  399, "Unassigned" },
    {  400, "Azerbaijan" },
    {  401, "Kazakhstan" },
    {  402, "Bhutan" },
    {  403, "Unassigned" },
    {  404, "India" },
    {  405, "India" },
    {  406, "Unassigned" },
    {  407, "Unassigned" },
    {  408, "Unassigned" },
    {  409, "Unassigned" },
    {  410, "Pakistan" },
    {  411, "Unassigned" },
    {  412, "Afghanistan" },
    {  413, "Sri Lanka" },
    {  414, "Myanmar" },
    {  415, "Lebanon" },
    {  416, "Jordan" },
    {  417, "Syrian Arab Republic" },
    {  418, "Iraq" },
    {  419, "Kuwait" },
    {  420, "Saudi Arabia" },
    {  421, "Yemen" },
    {  422, "Oman" },
    {  423, "Unassigned" },
    {  424, "United Arab Emirates" },
    {  425, "Israel" },
    {  426, "Bahrain" },
    {  427, "Qatar" },
    {  428, "Mongolia" },
    {  429, "Nepal" },
    {  430, "Unassigned" },
    {  431, "Unassigned" },
    {  432, "Iran (Islamic Republic of)" },
    {  433, "Unassigned" },
    {  434, "Uzbekistan" },
    {  435, "Unassigned" },
    {  436, "Tajikistan" },
    {  437, "Kyrgyzstan" },
    {  438, "Turkmenistan" },
    {  439, "Unassigned" },
    {  440, "Japan" },
    {  441, "Japan" },
    {  442, "Unassigned" },
    {  443, "Unassigned" },
    {  444, "Unassigned" },
    {  445, "Unassigned" },
    {  446, "Unassigned" },
    {  447, "Unassigned" },
    {  448, "Unassigned" },
    {  449, "Unassigned" },
    {  450, "Korea (Rep. of)" },
    {  451, "Unassigned" },
    {  452, "Viet Nam" },
    {  453, "Unassigned" },
    {  454, "Hong Kong, China" },
    {  455, "Macao, China" },
    {  456, "Cambodia" },
    {  457, "Lao P.D.R." },
    {  458, "Unassigned" },
    {  459, "Unassigned" },
    {  460, "China" },
    {  461, "Unassigned" },
    {  462, "Unassigned" },
    {  463, "Unassigned" },
    {  464, "Unassigned" },
    {  465, "Unassigned" },
    {  466, "Unassigned" },
    {  467, "Unassigned" },
    {  468, "Unassigned" },
    {  469, "Unassigned" },
    {  470, "Bangladesh" },
    {  471, "Unassigned" },
    {  472, "Maldives" },
    {  473, "Unassigned" },
    {  474, "Unassigned" },
    {  475, "Unassigned" },
    {  476, "Unassigned" },
    {  477, "Unassigned" },
    {  478, "Unassigned" },
    {  479, "Unassigned" },
    {  480, "Unassigned" },
    {  481, "Unassigned" },
    {  482, "Unassigned" },
    {  483, "Unassigned" },
    {  484, "Unassigned" },
    {  485, "Unassigned" },
    {  486, "Unassigned" },
    {  487, "Unassigned" },
    {  488, "Unassigned" },
    {  489, "Unassigned" },
    {  490, "Unassigned" },
    {  491, "Unassigned" },
    {  492, "Unassigned" },
    {  493, "Unassigned" },
    {  494, "Unassigned" },
    {  495, "Unassigned" },
    {  496, "Unassigned" },
    {  497, "Unassigned" },
    {  498, "Unassigned" },
    {  499, "Unassigned" },
    {  500, "Unassigned" },
    {  501, "Unassigned" },
    {  502, "Malaysia" },
    {  503, "Unassigned" },
    {  504, "Unassigned" },
    {  505, "Australia" },
    {  506, "Unassigned" },
    {  507, "Unassigned" },
    {  508, "Unassigned" },
    {  509, "Unassigned" },
    {  510, "Indonesia" },
    {  511, "Unassigned" },
    {  512, "Unassigned" },
    {  513, "Unassigned" },
    {  514, "Timor-Leste" },
    {  515, "Philippines" },
    {  516, "Unassigned" },
    {  517, "Unassigned" },
    {  518, "Unassigned" },
    {  519, "Unassigned" },
    {  520, "Thailand" },
    {  521, "Unassigned" },
    {  522, "Unassigned" },
    {  523, "Unassigned" },
    {  524, "Unassigned" },
    {  525, "Singapore" },
    {  526, "Unassigned" },
    {  527, "Unassigned" },
    {  528, "Brunei Darussalam" },
    {  529, "Unassigned" },
    {  530, "New Zealand" },
    {  531, "Unassigned" },
    {  532, "Unassigned" },
    {  533, "Unassigned" },
    {  534, "Unassigned" },
    {  535, "Unassigned" },
    {  536, "Unassigned" },
    {  537, "Papua New Guinea" },
    {  538, "Unassigned" },
    {  539, "Tonga" },
    {  540, "Solomon Islands" },
    {  541, "Vanuatu" },
    {  542, "Fiji / Nauru" },
    {  543, "Unassigned" },
    {  544, "Unassigned" },
    {  545, "Unassigned" },
    {  546, "New Caledonia" },
    {  547, "French Polynesia" },
    {  548, "Cook Islands" },
    {  549, "Samoa" },
    {  550, "Micronesia" },
    {  551, "Unassigned" },
    {  552, "Palau" },
    {  553, "Tuvalu" },
    {  554, "Unassigned" },
    {  555, "Niue" },
    {  556, "Unassigned" },
    {  557, "Unassigned" },
    {  558, "Unassigned" },
    {  559, "Unassigned" },
    {  560, "Unassigned" },
    {  561, "Unassigned" },
    {  562, "Unassigned" },
    {  563, "Unassigned" },
    {  564, "Unassigned" },
    {  565, "Unassigned" },
    {  566, "Unassigned" },
    {  567, "Unassigned" },
    {  568, "Unassigned" },
    {  569, "Unassigned" },
    {  570, "Unassigned" },
    {  571, "Unassigned" },
    {  572, "Unassigned" },
    {  573, "Unassigned" },
    {  574, "Unassigned" },
    {  575, "Unassigned" },
    {  576, "Unassigned" },
    {  577, "Unassigned" },
    {  578, "Unassigned" },
    {  579, "Unassigned" },
    {  580, "Unassigned" },
    {  581, "Unassigned" },
    {  582, "Unassigned" },
    {  583, "Unassigned" },
    {  584, "Unassigned" },
    {  585, "Unassigned" },
    {  586, "Unassigned" },
    {  587, "Unassigned" },
    {  588, "Unassigned" },
    {  589, "Unassigned" },
    {  590, "Unassigned" },
    {  591, "Unassigned" },
    {  592, "Unassigned" },
    {  593, "Unassigned" },
    {  594, "Unassigned" },
    {  595, "Unassigned" },
    {  596, "Unassigned" },
    {  597, "Unassigned" },
    {  598, "Unassigned" },
    {  599, "Unassigned" },
    {  600, "Unassigned" },
    {  601, "Unassigned" },
    {  602, "Egypt" },
    {  603, "Algeria" },
    {  604, "Morocco" },
    {  605, "Tunisia" },
    {  606, "Unassigned" },
    {  607, "Gambia" },
    {  608, "Senegal" },
    {  609, "Mauritania" },
    {  610, "Mali" },
    {  611, "Guinea" },
    {  612, "Cote d'Ivoire" },
    {  613, "Burkina Faso" },
    {  614, "Niger" },
    {  615, "Togo" },
    {  616, "Benin" },
    {  617, "Mauritius" },
    {  618, "Liberia" },
    {  619, "Sierra Leone" },
    {  620, "Ghana" },
    {  621, "Nigeria" },
    {  622, "Chad" },
    {  623, "Central African Rep." },
    {  624, "Cameroon" },
    {  625, "Cabo Verde" },
    {  626, "Sao Tome and Principe" },
    {  627, "Equatorial Guinea" },
    {  628, "Gabon" },
    {  629, "Congo" },
    {  630, "Dem. Rep. of the Congo" },
    {  631, "Angola" },
    {  632, "Guinea-Bissau" },
    {  633, "Seychelles" },
    {  634, "Sudan" },
    {  635, "Rwanda" },
    {  636, "Ethiopia" },
    {  637, "Unassigned" },
    {  638, "Djibouti" },
    {  639, "Kenya" },
    {  640, "Tanzania" },
    {  641, "Uganda" },
    {  642, "Burundi" },
    {  643, "Mozambique" },
    {  644, "Unassigned" },
    {  645, "Zambia" },
    {  646, "Madagascar" },
    {  647, "French Departments and Territories in the Indian Ocean" },
    {  648, "Zimbabwe" },
    {  649, "Namibia" },
    {  650, "Malawi" },
    {  651, "Lesotho" },
    {  652, "Botswana" },
    {  653, "Swaziland" },
    {  654, "Comoros" },
    {  655, "South Africa" },
    {  656, "Unassigned" },
    {  657, "Unassigned" },
    {  658, "Unassigned" },
    {  659, "South Sudan" },
    {  660, "Unassigned" },
    {  661, "Unassigned" },
    {  662, "Unassigned" },
    {  663, "Unassigned" },
    {  664, "Unassigned" },
    {  665, "Unassigned" },
    {  666, "Unassigned" },
    {  667, "Unassigned" },
    {  668, "Unassigned" },
    {  669, "Unassigned" },
    {  670, "Unassigned" },
    {  671, "Unassigned" },
    {  672, "Unassigned" },
    {  673, "Unassigned" },
    {  674, "Unassigned" },
    {  675, "Unassigned" },
    {  676, "Unassigned" },
    {  677, "Unassigned" },
    {  678, "Unassigned" },
    {  679, "Unassigned" },
    {  680, "Unassigned" },
    {  681, "Unassigned" },
    {  682, "Unassigned" },
    {  683, "Unassigned" },
    {  684, "Unassigned" },
    {  685, "Unassigned" },
    {  686, "Unassigned" },
    {  687, "Unassigned" },
    {  688, "Unassigned" },
    {  689, "Unassigned" },
    {  690, "Unassigned" },
    {  691, "Unassigned" },
    {  692, "Unassigned" },
    {  693, "Unassigned" },
    {  694, "Unassigned" },
    {  695, "Unassigned" },
    {  696, "Unassigned" },
    {  697, "Unassigned" },
    {  698, "Unassigned" },
    {  699, "Unassigned" },
    {  700, "Unassigned" },
    {  701, "Unassigned" },
    {  702, "Belize" },
    {  703, "Unassigned" },
    {  704, "Guatemala" },
    {  705, "Unassigned" },
    {  706, "El Salvador" },
    {  707, "Unassigned" },
    {  708, "Honduras" },
    {  709, "Unassigned" },
    {  710, "Nicaragua" },
    {  711, "Unassigned" },
    {  712, "Costa Rica" },
    {  713, "Unassigned" },
    {  714, "Panama" },
    {  715, "Unassigned" },
    {  716, "Peru" },
    {  717, "Unassigned" },
    {  718, "Unassigned" },
    {  719, "Unassigned" },
    {  720, "Unassigned" },
    {  721, "Unassigned" },
    {  722, "Argentina" },
    {  723, "Unassigned" },
    {  724, "Brazil" },
    {  725, "Unassigned" },
    {  726, "Unassigned" },
    {  727, "Unassigned" },
    {  728, "Unassigned" },
    {  729, "Unassigned" },
    {  730, "Chile" },
    {  731, "Unassigned" },
    {  732, "Colombia" },
    {  733, "Unassigned" },
    {  734, "Venezuela (Bolivarian Republic of)" },
    {  735, "Unassigned" },
    {  736, "Bolivia (Plurinational State of)" },
    {  737, "Unassigned" },
    {  738, "Guyana" },
    {  739, "Unassigned" },
    {  740, "Ecuador" },
    {  741, "Unassigned" },
    {  742, "Unassigned" },
    {  743, "Unassigned" },
    {  744, "Paraguay" },
    {  745, "Unassigned" },
    {  746, "Suriname" },
    {  747, "Unassigned" },
    {  748, "Uruguay" },
    {  749, "Unassigned" },
    {  750, "Falkland Islands (Malvinas)" },
    {  751, "Unassigned" },
    {  752, "Unassigned" },
    {  753, "Unassigned" },
    {  754, "Unassigned" },
    {  755, "Unassigned" },
    {  756, "Unassigned" },
    {  757, "Unassigned" },
    {  758, "Unassigned" },
    {  759, "Unassigned" },
    {  760, "Unassigned" },
    {  761, "Unassigned" },
    {  762, "Unassigned" },
    {  763, "Unassigned" },
    {  764, "Unassigned" },
    {  765, "Unassigned" },
    {  766, "Unassigned" },
    {  767, "Unassigned" },
    {  768, "Unassigned" },
    {  769, "Unassigned" },
    {  770, "Unassigned" },
    {  771, "Unassigned" },
    {  772, "Unassigned" },
    {  773, "Unassigned" },
    {  774, "Unassigned" },
    {  775, "Unassigned" },
    {  776, "Unassigned" },
    {  777, "Unassigned" },
    {  778, "Unassigned" },
    {  779, "Unassigned" },
    {  780, "Unassigned" },
    {  781, "Unassigned" },
    {  782, "Unassigned" },
    {  783, "Unassigned" },
    {  784, "Unassigned" },
    {  785, "Unassigned" },
    {  786, "Unassigned" },
    {  787, "Unassigned" },
    {  788, "Unassigned" },
    {  789, "Unassigned" },
    {  790, "Unassigned" },
    {  791, "Unassigned" },
    {  792, "Unassigned" },
    {  793, "Unassigned" },
    {  794, "Unassigned" },
    {  795, "Unassigned" },
    {  796, "Unassigned" },
    {  797, "Unassigned" },
    {  798, "Unassigned" },
    {  799, "Unassigned" },
    {  800, "Unassigned" },
    {  801, "Unassigned" },
    {  802, "Unassigned" },
    {  803, "Unassigned" },
    {  804, "Unassigned" },
    {  805, "Unassigned" },
    {  806, "Unassigned" },
    {  807, "Unassigned" },
    {  808, "Unassigned" },
    {  809, "Unassigned" },
    {  810, "Unassigned" },
    {  811, "Unassigned" },
    {  812, "Unassigned" },
    {  813, "Unassigned" },
    {  814, "Unassigned" },
    {  815, "Unassigned" },
    {  816, "Unassigned" },
    {  817, "Unassigned" },
    {  818, "Unassigned" },
    {  819, "Unassigned" },
    {  820, "Unassigned" },
    {  821, "Unassigned" },
    {  822, "Unassigned" },
    {  823, "Unassigned" },
    {  824, "Unassigned" },
    {  825, "Unassigned" },
    {  826, "Unassigned" },
    {  827, "Unassigned" },
    {  828, "Unassigned" },
    {  829, "Unassigned" },
    {  830, "Unassigned" },
    {  831, "Unassigned" },
    {  832, "Unassigned" },
    {  833, "Unassigned" },
    {  834, "Unassigned" },
    {  835, "Unassigned" },
    {  836, "Unassigned" },
    {  837, "Unassigned" },
    {  838, "Unassigned" },
    {  839, "Unassigned" },
    {  840, "Unassigned" },
    {  841, "Unassigned" },
    {  842, "Unassigned" },
    {  843, "Unassigned" },
    {  844, "Unassigned" },
    {  845, "Unassigned" },
    {  846, "Unassigned" },
    {  847, "Unassigned" },
    {  848, "Unassigned" },
    {  849, "Unassigned" },
    {  850, "Unassigned" },
    {  851, "Unassigned" },
    {  852, "Unassigned" },
    {  853, "Unassigned" },
    {  854, "Unassigned" },
    {  855, "Unassigned" },
    {  856, "Unassigned" },
    {  857, "Unassigned" },
    {  858, "Unassigned" },
    {  859, "Unassigned" },
    {  860, "Unassigned" },
    {  861, "Unassigned" },
    {  862, "Unassigned" },
    {  863, "Unassigned" },
    {  864, "Unassigned" },
    {  865, "Unassigned" },
    {  866, "Unassigned" },
    {  867, "Unassigned" },
    {  868, "Unassigned" },
    {  869, "Unassigned" },
    {  870, "Unassigned" },
    {  871, "Unassigned" },
    {  872, "Unassigned" },
    {  873, "Unassigned" },
    {  874, "Unassigned" },
    {  875, "Unassigned" },
    {  876, "Unassigned" },
    {  877, "Unassigned" },
    {  878, "Unassigned" },
    {  879, "Unassigned" },
    {  880, "Unassigned" },
    {  881, "Unassigned" },
    {  882, "Unassigned" },
    {  883, "Unassigned" },
    {  884, "Unassigned" },
    {  885, "Unassigned" },
    {  886, "Unassigned" },
    {  887, "Unassigned" },
    {  888, "Unassigned" },
    {  889, "Unassigned" },
    {  890, "Unassigned" },
    {  891, "Unassigned" },
    {  892, "Unassigned" },
    {  893, "Unassigned" },
    {  894, "Unassigned" },
    {  895, "Unassigned" },
    {  896, "Unassigned" },
    {  897, "Unassigned" },
    {  898, "Unassigned" },
    {  899, "Unassigned" },
    {  900, "Unassigned" },
    {  901, "International Mobile, shared code" },
    { 1665, "Unset" },
    { 0, NULL }
};

value_string_ext E212_codes_ext = VALUE_STRING_EXT_INIT(E212_codes);

/*
 * Annex to ITU Operational Bulletin No. 1019 - 1.I.2013
 *
 * According to Recommendation ITU-T E.212 (05/2008)
 *
 * The value parts of the list below are equal to:
 *   1000 * MCC + 10 * MNC,  in case of 2 digits long MNCs
 *   1000 * MCC + MNC,       in case of 3 digits long MNCs
 * with this encoding, we are sure that even in cases of mixed length
 * MNCs under the same MCC, it is impossible for 2 MNCs to get mapped to
 * the same entry. Aliasing to the same entry in the list below occurs
 * only in case of an invalid combination of MNCs, i.e. in the same MCC a
 * 2 digits long MNC is a prefix of a 3 digits long MNC
 */
static const value_string mcc_mnc_codes[] = {
    {  202010, "Cosmote" },
    {  202020, "Cosmote" },
    {  202030, "OTE" },
    {  202040, "EDISY" },
    {  202050, "Vodafone - Panafon" },
    {  202060, "COSMOLINE" },
    {  202070, "AMD TELECOM" },
    {  202090, "WIND" },
    {  202100, "WIND" },
    {  204010, "RadioAccess Network Services B.V." },
    {  204020, "Tele2 Nederland B.V." },
    {  204030, "Voiceworks B.V." },
    {  204040, "Vodafone Libertel B.V." },
    {  204050, "Elephant Talk Communications Premium Rate Services" },
    {  204060, "Mundio Mobile (Netherlands) Ltd" },
    {  204070, "Teleena holding B.V." },
    {  204080, "KPN Mobile The Netherlands B.V." },
    {  204090, "Lycamobile Netherlands Limited" },
    {  204100, "KPN B.V." },
    {  204120, "KPN B.V." },
    {  204130, "Unica Installatietechniek B.V." },
    {  204150, "Ziggo B.V." },
    {  204160, "T-Mobile Netherlands B.V." },
    {  204170, "Intercity Mobile Communications B.V." },
    {  204180, "UPC Nederland B.V." },
    {  204190, "Mixe Communication Solutions B.V." },
    {  204200, "T-Mobile Netherlands B.V." },
    {  204210, "ProRail B.V." },
    {  204220, "Ministerie van Defensie" },
    {  204230, "ASpider Solutions Nederland B.V." },
    {  204240, "Private Mobility Nederland B.V." },
    {  204250, "CAPX B.V." },
    {  204260, "SpeakUp B.V." },
    {  204280, "Lancelot B.V." },
    {  204290, "Private Mobile Ltd" },
    {  204600, "Nextgen Mobile Ltd" },
    {  204610, "BodyTrace Netherlands B.V." },
    {  204640, "Zetacom B.V." },
    {  204650, "AGMS Netherlands B.V." },
    {  204660, "Utility Connect B.V." },
    {  204670, "RadioAccess B.V." },
    {  204680, "Roamware (Netherlands) B.V." },
    {  204690, "KPN Mobile The Netherlands B.V." },
    {  206010, "Belgacom nv" },
    {  206020, "N.M.B.S" },
    {  206050, "Telenet nv" },
    {  206060, "Lycamobile sprl" },
    {  206070, "Mundio Mobile Belgium nv" },
    {  206100, "Mobistar sa" },
    {  206200, "Base Company nv" },
    {  208010, "Orange France" },
    {  208020, "Orange France" },
    {  208030, "MobiquiThings" },
    {  208040, "Sisteer" },
    {  208050, "Globalstar Europe" },
    {  208060, "Globalstar Europe" },
    {  208070, "Globalstar Europe" },
    {  208090, "S.F.R." },
    {  208100, "S.F.R." },
    {  208110, "S.F.R." },
    {  208130, "S.F.R." },
    {  208140, "RFF" },
    {  208150, "Free Mobile" },
    {  208200, "Bouygues Telecom" },
    {  208210, "Bouygues Telecom" },
    {  208220, "Transatel" },
    {  208230, "Omer Telecom Ltd" },
    {  208240, "MobiquiThings" },
    {  208250, "Lycamobile" },
    {  208260, "NRJ Mobile" },
    {  208270, "Afone" },
    {  208280, "Astrium" },
    {  208290, "Societe International Mobile Communication" },
    {  208300, "Symacom" },
    {  208310, "Mundio Mobile" },
    {  208880, "Bouygues Telecom" },
    {  208890, "Omer Telecom Ltd" },
    {  208900, "Images & Reseaux" },
    {  208910, "Orange France" },
    {  213030, "Mobiland" },
    {  214010, "Vodafone Espana, SAU" },
    {  214030, "France Telecom Espana, SA" },
    {  214040, "Xfera Moviles, S.A." },
    {  214050, "Telefonica Moviles Espana, SAU" },
    {  214060, "Vodafone Espana, SAU" },
    {  214070, "Telefonica Moviles Espana, SAU" },
    {  214080, "Euskaltel, SA" },
    {  214090, "France Telecom Espana, SA" },
    {  214100, "Operadora de Telecomunicaciones Opera SL" },
    {  214110, "France Telecom Espana, SA" },
    {  214120, "Contacta Servicios Avanzados de Telecomunicaciones SL" },
    {  214130, "Incotel Ingeniera y Consultaria SL" },
    {  214140, "Incotel Servicioz Avanzados SL" },
    {  214150, "BT Espana Compania de Servicios Globales de Telecomunicaciones, SAU" },
    {  214160, "Telecable de Asturias, SAU" },
    {  214170, "R Cable y Telecomunicaciones Galicia, SA" },
    {  214180, "Cableuropa, SAU" },
    {  214190, "E-Plus Moviles, SL" },
    {  214200, "Fonyou Telecom, SL" },
    {  214210, "Jazz Telecom, SAU" },
    {  214220, "Best Spain Telecom, SL" },
    {  214240, "Vizzavi Espana, S.L." },
    {  214250, "Lycamobile, SL" },
    {  214260, "Lleida Networks Serveis Telematics, SL" },
    {  214270, "SCN Truphone SL" },
    {  214280, "Consorcio de Telecomunicaciones Avanzadas, S.A." },
    {  214290, "NEO-SKY 2002, S.A." },
    {  214300, "Compatel Limited" },
    {  214310, "Red Digital De Telecomunicaciones de las Islas Baleares, S.L." },
    {  214320, "TUENTI TECHNOLOGIES, S.L." },
    {  216010, "Telenor Hungary Ltd" },
    {  216300, "Magyar Telecom Plc" },
    {  216700, "Vodafone" },
    {  216710, "UPC Hungary Ltd" },
    {  218030, "Eronet Mobile Communications Ltd." },
    {  218050, "MOBI'S (Mobilina Srpske)" },
    {  218900, "GSMBIH" },
    {  219010, "T-Mobile Hrvatska d.o.o./T-Mobile Croatia LLC" },
    {  219020, "Tele2/Tele2 d.o.o." },
    {  219100, "VIPnet/VIPnet d.o.o." },
    {  220010, "Telenor d.o.o." },
    {  220030, "Telekom Srbija a.d." },
    {  220050, "Vip mobile d.o.o." },
    {  222010, "Telecom Italia Mobile (TIM)" },
    {  222020, "Elsacom" },
    {  222100, "Omnitel Pronto Italia (OPI)" },
    {  222770, "IPSE 2000" },
    {  222880, "Wind" },
    {  222980, "Blu" },
    {  222990, "H3G" },
    {  226010, "Vodafone" },
    {  226020, "Romtelecom" },
    {  226030, "Cosmote" },
    {  226040, "Cosmote" },
    {  226050, "Digi.Mobil" },
    {  226060, "Cosmote" },
    {  226100, "Orange" },
    {  226110, "Enigma-System" },
    {  228010, "Swisscom Schweiz AG" },
    {  228020, "Sunrise Communications AG" },
    {  228030, "Orange Communications SA" },
    {  228050, "Comfone AG" },
    {  228060, "SBB AG" },
    {  228080, "Tele2 Telecommunications AG" },
    {  228120, "Sunrise Communications AG" },
    {  228510, "Bebbicell AG" },
    {  230010, "T-Mobile Czech Republic a.s." },
    {  230020, "Telefonica O2 Czech Republic a.s." },
    {  230030, "Vodafone Czech Republic a.s." },
    {  230040, "Air Telecom a.s." },
    {  230050, "Travel Telekommunikation, s.r.o." },
    {  230070, "ASTELNET s.r.o" },
    {  230080, "Compatel s.r.o" },
    {  230980, "Sprava Zeleznicni Dopravni Cesty" },
    {  231010, "Orange, GSM" },
    {  231020, "Eurotel, GSM & NMT" },
    {  231040, "Eurotel, UMTS" },
    {  231050, "Orange, UMTS" },
    {  232010, "Mobilkom Austria Aktiengesellschaft" },
    {  232020, "Mobilkom Austria Aktiengesellschaft" },
    {  232030, "T-Mobile Austria GmbH" },
    {  232040, "T-Mobile Austria GmbH" },
    {  232050, "Orange Austria Telecommunication GmbH" },
    {  232060, "Orange Austria Telecommunication GmbH" },
    {  232070, "T-Mobile Austria GmbH" },
    {  232090, "Mobilkom Austria Aktiengesellschaft" },
    {  232100, "Hutchison 3G Austria GmbH" },
    {  232110, "Mobilkom Austria Aktiengesellschaft" },
    {  232120, "Orange Austria Telecommunication GmbH" },
    {  232140, "Hutchison 3G Austria GmbH" },
    {  232150, "Barablu Mobile Austria Ltd" },
    {  232910, "OBB - Infrastruktur Bau AG" },
    {  234000, "British Telecom" },
    {  234010, "Mapesbury Communications Ltd" },
    {  234020, "O2 UK Ltd." },
    {  234030, "Jersey Airtel Ltd" },
    {  234040, "FMS Solutions Ltd" },
    {  234050, "Colt Mobile Telecommunications Ltd" },
    {  234060, "Internet Computer Bureau Ltd" },
    {  234070, "Cable & Wireless UK" },
    {  234080, "OnePhone (UK) Ltd" },
    {  234090, "Tismi BV" },
    {  234100, "O2 UK Ltd." },
    {  234110, "O2 UK Ltd." },
    {  234120, "Network Rail Infrastructure Ltd" },
    {  234130, "Network Rail Infrastructure Ltd" },
    {  234140, "Hay Systems Ltd" },
    {  234150, "Vodafone Ltd" },
    {  234160, "Opal Telecom Ltd" },
    {  234170, "Flextel Ltd" },
    {  234180, "Cloud9" },
    {  234190, "Teleware plc" },
    {  234200, "Hutchison 3G UK Ltd" },
    {  234210, "LogicStar Ltd" },
    {  234220, "Routo Telecommunications Ltd" },
    {  234230, "Vectone Network Ltd" },
    {  234240, "Stour Marine Ltd" },
    {  234250, "Software Cellular Network Ltd" },
    {  234260, "Lycamobile UK Limited" },
    {  234270, "Teleena UK Limited" },
    {  234280, "Marathon Telecom Limited" },
    {  234290, "(aq) Limited T/A aql" },
    {  234300, "T-Mobile UK" },
    {  234310, "T-Mobile UK" },
    {  234320, "T-Mobile UK" },
    {  234330, "Orange" },
    {  234340, "Orange" },
    {  234350, "JSC Ingenium (UK) Limited" },
    {  234360, "Sure (Isle of Man) Limited" },
    {  234370, "Synectiv Ltd" },
    {  234380, "Virgin Mobile Telecoms Limited" },
    {  234390, "SSE Energy Supply Limited" },
    {  234500, "Jersey Telecom" },
    {  234510, "UK Broadband Limited" },
    {  234520, "Shyam Telecom UK Ltd" },
    {  234530, "Limitless Mobile Ltd" },
    {  234550, "Cable and Wireless Guensey Ltd" },
    {  234580, "Manx Telecom" },
    {  234760, "British Telecom" },
    {  234780, "Airwave mmO2 Ltd" },
    {  234860, "EE Limited ( TM)" },
    {  235000, "Mundlo Mobile Limited" },
    {  235010, "EE Limited ( TM)" },
    {  235020, "EE Limited ( TM)" },
    {  235770, "British Telecom" },
    {  235910, "Vodafone Ltd" },
    {  235920, "Cable & Wireless UK" },
    {  235940, "Hutchison 3G UK Ltd" },
    {  235950, "Network Rail Infrastructure Ltd" },
    {  238010, "TDC Mobil" },
    {  238020, "Sonofon" },
    {  238030, "MIGway A/S" },
    {  238040, "NextGen Mobile Ltd T/A CardBoardFish" },
    {  238060, "Hi3G" },
    {  238080, "Nordisk Mobiltelefon Danmark A/S" },
    {  238100, "TDC Mobil" },
    {  238120, "Lycamobile Denmark" },
    {  238130, "Compatel Limited" },
    {  238200, "Telia" },
    {  238280, "CoolTEL" },
    {  238300, "Interactive Digital Media GmbH" },
    {  238430, "MobiWeb Limited" },
    {  238660, "TT-Netvaerket P/S" },
    {  238770, "Tele2" },
    {  240010, "Telia Sonera Sverige AB" },
    {  240020, "H3G Access AB" },
    {  240030, "Netett Sverige AB" },
    {  240040, "3G Infrastructure Services AB" },
    {  240050, "Svenska UMTS-Nat AB" },
    {  240060, "Telenor Sverige AB" },
    {  240080, "Telenor Sverige AB" },
    {  240100, "Spring Mobil AB" },
    {  240120, "Lycamobile Sweden Limited" },
    {  240130, "Alltele Foretag Sverige AB" },
    {  240140, "TDC Sverige AB" },
    {  240150, "Wireless Maingate Nordic AB" },
    {  240160, "42 Telecom AB" },
    {  240170, "Gotalandsnatet AB" },
    {  240180, "Generic Mobile Systems Sweden AB" },
    {  240190, "Mundio Mobile Sweden Ltd" },
    {  240200, "iMEZ AB" },
    {  240210, "Trafikverket ICT" },
    {  240220, "EuTel AB" },
    {  240230, "Infobip LTD" },
    {  240240, "Net4Mobility HB" },
    {  240260, "Beepsend A.B." },
    {  240270, "Fogg Mobile AB" },
    {  240280, "CoolTEL Aps" },
    {  240290, "Mercury International Carrier Services" },
    {  240300, "NextGen Mobile Ltd" },
    {  240320, "Compatel Ltd." },
    {  240330, "Mobile Arts AB" },
    {  240340, "Pro Net Telecommunications Services Ltd." },
    {  240350, "42 Telecom LTD" },
    {  240360, "Interactive Digital Media GmbH" },
    {  240370, "CLX Networks AB" },
    {  240380, "Voxbone SA" },
    {  240400, "ReWiCom Scandinavia AB" },
    {  240410, "Shyam Telecom UK Ltd" },
    {  240420, "Telenor Connexion AB" },
    {  240430, "MobiWeb Ltd." },
    {  240440, "Limitless Mobile AB" },
    {  240450, "Spirius AB" },
    {  240650, "shared use for closed networks" },
    {  240660, "shared use for closed networks" },
    {  240670, "shared use for test purpose" },
    {  240680, "shared use for test purpose" },
    {  240690, "crisis management after determination by the Swedish Post- and Telecom Authority" },
    {  240990, "Unassigned(May be used for test purposes)" }, /* Used in lab network */
    {  242010, "Telenor Norge AS" },
    {  242020, "NetCom AS" },
    {  242030, "Teletopia Gruppen AS" },
    {  242040, "Tele2 Norge AS" },
    {  242050, "Network Norway AS" },
    {  242070, "Ventelo Bedrift AS" },
    {  242080, "TDC AS" },
    {  242090, "Com4 AS" },
    {  242100, "Post-og teletilsynet" },
    {  242110, "Systemnet AS" },
    {  242120, "Telenor Norge AS" },
    {  242200, "Jernbaneverket" },
    {  242210, "Jernbaneverket" },
    {  242220, "Network Norway AS" },
    {  242230, "Lycamobile Norway Ltd" },
    {  242240, "Mobile Norway AS" },
    {  244030, "DNA Oy" },
    {  244040, "DNA Oy" },
    {  244050, "Elisa Oy" },
    {  244090, "Nokia Siemens Networks Oy" },
    {  244100, "TDC Oy FINLAND" },
    {  244120, "DNA Oy" },
    {  244130, "DNA Oy" },
    {  244140, "Alands Mobiltelefon AB" },
    {  244160, "Oy Finland Tele2 AB" },
    {  244210, "Saunalahti Group Oyj" },
    {  244290, "SCNL TRUPHONE" },
    {  244910, "TeliaSonera Finland Oyj" },
    {  246010, "Omnitel" },
    {  246020, "Bite GSM" },
    {  246030, "Tele2" },
    {  247010, "Latvijas Mobilais Telefons SIA" },
    {  247020, "Tele2" },
    {  247030, "Telekom Baltija" },
    {  247040, "Beta Telecom" },
    {  247050, "Bite Mobile" },
    {  247060, "Rigatta" },
    {  247070, "Master Telecom" },
    {  247080, "IZZI" },
    {  247090, "SIA \"Camel Mobile\"" },
    {  248010, "EMT GSM" },
    {  248020, "RLE" },
    {  248030, "Tele2" },
    {  248040, "OY Top Connect" },
    {  248050, "AS Bravocom Mobiil" },
    {  248060, "ProGroup Holding OY" },
    {  248070, "Televorgu AS" },
    {  248080, "Vivex OU" },
    {  248090, "Bravo Telecom OU" },
    {  248710, "Siseministeerium (Ministry of Interior)" },
    {  250010, "Mobile Telesystems" },
    {  250020, "Megafon" },
    {  250030, "Nizhegorodskaya Cellular Communications" },
    {  250040, "Sibchallenge" },
    {  250050, "Mobile Comms System" },
    {  250070, "BM Telecom" },
    {  250100, "Don Telecom" },
    {  250110, "Orensot" },
    {  250120, "Baykal Westcom" },
    {  250130, "Kuban GSM" },
    {  250160, "New Telephone Company" },
    {  250170, "Ermak RMS" },
    {  250190, "Volgograd Mobile" },
    {  250200, "ECC" },
    {  250280, "Extel" },
    {  250390, "Uralsvyazinform" },
    {  250440, "Stuvtelesot" },
    {  250920, "Printelefone" },
    {  250930, "Telecom XXI" },
    {  250990, "Beeline" },
    {  255010, "Ukrainian Mobile Communication, UMC" },
    {  255020, "Ukranian Radio Systems, URS" },
    {  255030, "Kyivstar GSM" },
    {  255040, "International Telecommunications Ltd." },
    {  255050, "Golden Telecom" },
    {  255060, "Astelit" },
    {  255070, "Ukrtelecom" },
    {  255210, "CJSC - Telesystems of Ukraine" },
    {  257010, "MDC Velcom" },
    {  257020, "MTS" },
    {  257030, "BelCel Joint Venture (JV)" },
    {  257040, "Closed joint-stock company \"Belarusian telecommunication network\"" },
    {  257050, "Republican Unitary Telecommunication Enterprise (RUE) Beltelecom (National Telecommunications Operator of the Republic of Belarus)" },
    {  257060, "Belorussian Cloud Technologies" },
    {  259010, "Orange Moldova GSM" },
    {  259020, "Moldcell GSM" },
    {  259040, "Eventis Mobile GSM" },
    {  259050, "J.S.C. Moldtelecom/3G UMTS (W-CDMA)" },
    {  259990, "J.S.C. Moldtelecom" },
    {  260010, "Plus / Polkomtel S.A." },
    {  260020, "T-Mobile / PTC S.A." },
    {  260030, "Orange / PTK Centertel Sp. z o.o." },
    {  260040, "LTE / CenterNet S.A." },
    {  260050, "Orange(UMTS) / PTK Centertel Sp. z o.o." },
    {  260060, "Play / P4 Sp. z o.o." },
    {  260070, "Netia / Netia S.A." },
    {  260080, "E-Telko / E-Telko Sp. z o.o." },
    {  260090, "Lycamobile / Lycamobile Sp. z o.o." },
    {  260100, "Sferia / Sferia S.A." },
    {  260110, "Nordisk Polska / Nordisk Polska Sp. z o.o." },
    {  260120, "Cyfrowy Polsat / Cyfrowy Polsat S.A." },
    {  260130, "Sferia / Sferia S.A." },
    {  260140, "Sferia / Sferia S.A." },
    {  260150, "CenterNet / CenterNet S.A." },
    {  260160, "Mobyland / Mobyland Sp. z o.o." },
    {  260170, "Aero 2 / Aero 2 Sp. z o.o." },
    {  260180, "AMD Telecom / AMD Telecom S.A." },
    {  260190, "Teleena / Teleena Holding BV" },
    {  260200, "Mobile.Net / Mobile.Net Sp. z o.o." },
    {  260210, "Exteri / Exteri Sp. z o.o." },
    {  260220, "Arcomm / Arcomm Sp. z o.o." },
    {  260230, "Amicomm / Amicomm Sp. z o.o." },
    {  260240, "WideNet / WideNet Sp. z o.o." },
    {  260250, "BS&T / Best Solutions & Technology Sp. z o.o." },
    {  260260, "ATE / ATE-Advanced Technology & Experience Sp. z o.o." },
    {  260270, "Intertelcom / Intertelcom Sp. z o.o." },
    {  260280, "PhoneNet / PhoneNet Sp. z o.o." },
    {  260290, "Interfonica / Interfonica Sp. z o.o." },
    {  260300, "GrandTel / GrandTel Sp. z o.o." },
    {  260310, "Phone IT / Phone IT Sp. z o.o." },
    {  260320, "Compatel Ltd / COMPATEL LIMITED" },
    {  260330, "Truphone Poland / Truphone Poland Sp. Z o.o." },
    {  260340, "T-Mobile / PTC S.A." },
    {  260980, "Play (testowy) / P4 Sp. z o.o." },
    {  262010, "Telekom Deutschland GmbH" },
    {  262020, "Vodafone D2 GmbH" },
    {  262030, "E-Plus Mobilfunk GmbH & Co. KG" },
    {  262040, "Vodafone D2 GmbH" },
    {  262050, "E-Plus Mobilfunk GmbH & Co. KG" },
    {  262060, "Telekom Deutschland GmbH" },
    {  262070, "Telefonica Germany GmbH & Co. oHG" },
    {  262080, "Telefonica Germany GmbH & Co. oHG" },
    {  262090, "Vodafone D2 GmbH" },
    {  262100, "DB Netz AG" },
    {  262110, "Telefonica Germany GmbH & Co. oHG" },
    {  262120, "E-Plus Mobilfunk GmbH & Co. KG" },
    {  262130, "Mobilcom Multimedia GmbH" },
    {  262140, "Quam GmbH" },
    {  262150, "AirData AG" },
    {  262160, "E-Plus Mobilfunk GmbH & Co. KG" },
    {  262170, "E-Plus Mobilfunk GmbH & Co. KG" },
    {  262180, "NetCologne Gesellschaft fur Telekommunikation mbH" },
    {  262190, "Inquam Deutschland GmbH" },
    {  262200, "E-Plus Mobilfunk GmbH & Co. KG" },
    {  262410, "First Telecom GmbH" },
    {  262420, "Vodafone D2 GmbH" },
    {  262430, "Vodafone D2 GmbH" },
    {  262770, "E-Plus Mobilfunk GmbH & Co. KG" },
    {  262780, "Telekom Deutschland GmbH" },
    {  262790, "ng4T GmbH" },
    {  266010, "Gibtelecom GSM" },
    {  266090, "Eazi Telecom Limited" },
    {  268010, "Vodafone Telecel - Comunicacoes Pessoais, S.A." },
    {  268030, "Optimus - Telecomunicacoes, S.A." },
    {  268050, "Oniway - Inforcomunicacoes, S.A." },
    {  268060, "TMN - Telecomunicacoes Moveis Nacionais, S.A." },
    {  270010, "P&T Luxembourg" },
    {  270020, "MTX Connect S.a r.l." },
    {  270100, "BLUE COMMUNICATIONS" },
    {  270770, "Tango" },
    {  270780, "Interactive Digital Media (IDM)" },
    {  270990, "Voxmobile S.A." },
    {  272010, "Vodafone Ireland Plc" },
    {  272020, "Telefonica Ltd" },
    {  272030, "Meteor Mobile Communications Ltd." },
    {  272070, "Eircom" },
    {  272090, "Clever Communications Ltd." },
    {  274010, "Iceland Telecom Ltd." },
    {  274020, "P/F Kall, reg. No 2868 (Vodafone FO) / Og fjarskipti hf (Vodafone Iceland)" },
    {  274030, "Og fjarskipti hf (Vodafone Iceland)" },
    {  274040, "IMC Islande ehf" },
    {  274070, "IceCell ehf" },
    {  276010, "Albanian Mobile Communications (AMC)" },
    {  276020, "Vodafone Albania" },
    {  276030, "Eagle Mobile" },
    {  276040, "Mobile 4 AL" },
    {  278010, "Vodafone Malta" },
    {  278210, "go mobile" },
    {  278770, "3G Telecommunications Ltd" },
    {  280010, "CYTA" },
    {  280100, "Scancom (Cyprus) Ltd." },
    {  280200, "PrimeTel PLC" },
    {  280220, "Lemontel Ltd" },
    {  282010, "Geocell Ltd." },
    {  282020, "Magti GSM Ltd." },
    {  282030, "Iberiatel Ltd." },
    {  282040, "Mobitel Ltd." },
    {  282050, "Silknet JSC" },
    {  282060, "JSC Compatel" },
    {  282070, "GLOBALCELL  LTD" },
    {  284010, "Mobiltel EAD" },
    {  284050, "Globul" },
    {  286010, "Turkcell" },
    {  286020, "Telsim GSM" },
    {  286030, "Aria" },
    {  286040, "Aycell" },
    {  288010, "Faroese Telecom - GSM" },
    {  288020, "Kall GSM" },
    {  290010, "Tele Greenland" },
    {  292010, "Prima San Marino / San Marino Telecom" },
    {  293400, "SI Mobil" },
    {  293410, "Mobitel" },
    {  293640, "T-2 d.o.o." },
    {  293700, "Tusmobil d.o.o." },
    {  294010, "T-Mobile" },
    {  294020, "Cosmofon" },
    {  294030, "Nov Operator" },
    {  294100, "WTI Macedonia" },
    {  294110, "MOBIK TELEKOMUNIKACII DOOEL- Skopje" },
    {  295010, "Swisscom Schweiz AG" },
    {  295020, "Orange (Liechtenstein) AG" },
    {  295050, "Mobilkom (Liechtenstein) AG" },
    {  295060, "Cubic AG" },
    {  297010, "Telenor Montenegro" },
    {  297020, "Crnogorski Telekom" },
    {  297030, "Mtel Montenegro" },
    {  302220, "Telus Mobility" },
    {  302221, "Telus Mobility" },
    {  302222, "Telus Mobility" },
    {  302250, "ALO Mobile Inc" },
    {  302270, "Bragg Communications" },
    {  302290, "Airtel Wireless" },
    {  302320, "Dave Wireless" },
    {  302340, "Execulink" },
    {  302360, "Telus Mobility" },
    {  302370, "Microcell" },
    {  302380, "Dryden Mobility" },
    {  302390, "Dryden Mobility" },
    {  302490, "Globalive Wireless" },
    {  302500, "Videotron Ltd" },
    {  302510, "Videotron Ltd" },
    {  302530, "Keewatinook Okimacinac" },
    {  302560, "Lynx Mobility" },
    {  302570, "Light Squared" },
    {  302590, "Quadro Communication" },
    {  302610, "Bell Mobility" },
    {  302620, "Ice Wireless" },
    {  302630, "Aliant Mobility" },
    {  302640, "Bell Mobility" },
    {  302656, "Tbay Mobility" },
    {  302660, "MTS Mobility" },
    {  302670, "CityTel Mobility" },
    {  302680, "Sask Tel Mobility" },
    {  302690, "Bell Mobility" },
    {  302710, "Globalstar" },
    {  302720, "Rogers Wireless" },
    {  302730, "TerreStar Solutions" },
    {  302740, "Shaw Telecom G.P." },
    {  302760, "Public Mobile Inc" },
    {  302770, "Rural Com" },
    {  302780, "Sask Tel Mobility" },
    {  302860, "Telus Mobility" },
    {  302880, "Telus/Bell shared" },
    {  302940, "Wightman Telecom" },
    {  302990, "Test" },
    {  308010, "St. Pierre-et-Miquelon Telecom" },
    {  310010, "Verizon Wireless" },
    {  310012, "Verizon Wireless" },
    {  310013, "Verizon Wireless" },
    {  310016, "Cricket Communications" },
    {  310017, "North Sight Communications Inc" },
    {  310020, "Union Telephone Company" },
    {  310030, "Centennial Communications" },
    {  310035, "ETEX Communications dba ETEX Wireless" },
    {  310040, "MTA Communications dba MTA Wireless" },
    {  310050, "Alaska Communications" },
    {  310060, "Consolidated Telcom" },
    {  310070, "AT&T" },
    {  310080, "Corr Wireless Communications LLC" },
    {  310090, "Cricket Communications" },
    {  310100, "New Mexico RSA 4 East Ltd. Partnership" },
    {  310110, "Pacific Telecom Inc" },
    {  310120, "Sprintcom Inc" },
    {  310130, "Carolina West Wireless" },
    {  310140, "GTA Wireless LLC" },
    {  310150, "Cingular Wireless" },
    {  310160, "T-Mobile USA" },
    {  310170, "Cingular Wireless" },
    {  310180, "West Central Wireless" },
    {  310190, "Alaska Wireless Communications LLC" },
    {  310200, "T-Mobile USA" },
    {  310210, "T-Mobile USA" },
    {  310220, "T-Mobile USA" },
    {  310230, "T-Mobile USA" },
    {  310240, "T-Mobile USA" },
    {  310250, "T-Mobile USA" },
    {  310260, "T-Mobile USA" },
    {  310270, "T-Mobile USA" },
    {  310280, "Contennial Puerto Rio License Corp." },
    {  310290, "NEP Cellcorp Inc." },
    {  310300, "Blanca Telephone Company" },
    {  310310, "T-Mobile USA" },
    {  310320, "Smith Bagley Inc, dba Cellular One" },
    {  310330, "AWCC" },
    {  310340, "High Plains Midwest LLC, dba Westlink Communications" },
    {  310350, "Mohave Cellular L.P." },
    {  310360, "Cellular Network Partnership dba Pioneer Cellular" },
    {  310370, "Docomo Pacific Inc" },
    {  310380, "New Cingular Wireless PCS, LLC" },
    {  310390, "TX-11 Acquistion LLC" },
    {  310400, "Wave Runner LLC" },
    {  310410, "Cingular Wireless" },
    {  310420, "Cincinnati Bell Wireless LLC" },
    {  310430, "GCI Communications Corp" },
    {  310440, "Numerex Corp" },
    {  310450, "North East Cellular Inc." },
    {  310460, "Newcore Wireless" },
    {  310470, "nTELOS Communications Inc" },
    {  310480, "Choice Phone LLC" },
    {  310490, "T-Mobile USA" },
    {  310500, "Public Service Cellular, Inc." },
    {  310510, "Nsighttel Wireless Inc" },
    {  310520, "Transactions Network Services" },
    {  310530, "Iowa Wireless Services LLC" },
    {  310540, "Oklahoma Western Telephone Company" },
    {  310550, "Wireless Solutions International" },
    {  310560, "AT&T" },
    {  310570, "MTPCS LLC" },
    {  310580, "Inland Cellular Telephone Company" },
    {  310590, "Verizon Wireless" },
    {  310600, "New Cell Inc. dba Cellcom" },
    {  310610, "Elkhart Telephone Co. Inc. dba Epic Touch Co." },
    {  310620, "Nsighttel Wireless Inc" },
    {  310640, "Airadigm Communications" },
    {  310650, "Jasper Wireless Inc." },
    {  310660, "T-Mobile USA" },
    {  310670, "AT&T Mobility Vanguard Services" },
    {  310680, "AT&T" },
    {  310690, "Keystone Wireless LLC" },
    {  310700, "Cross Valiant Cellular Partnership" },
    {  310710, "Arctic Slope Telephone Association Cooperative" },
    {  310720, "Wireless Solutions International Inc." },
    {  310730, "US Cellular" },
    {  310740, "Convey Communications Inc" },
    {  310750, "East Kentucky Network LLC dba Appalachian Wireless" },
    {  310760, "Lynch 3G Communications Corporation" },
    {  310770, "Iowa Wireless Services LLC dba I Wireless" },
    {  310780, "D.D. Inc" },
    {  310790, "PinPoint Communications Inc." },
    {  310800, "T-Mobile USA" },
    {  310810, "LCFR LLC" },
    {  310820, "South Canaan Cellular Communications Co. LP" },
    {  310830, "Clearwire Corporation" },
    {  310840, "Telecom North America Mobile Inc" },
    {  310850, "Aeris Communications, Inc." },
    {  310860, "TX RSA 15B2, LP dba Five Star Wireless" },
    {  310870, "Kaplan Telephone Company Inc." },
    {  310880, "Advantage Cellular Systems, Inc." },
    {  310890, "Verizon Wireless" },
    {  310900, "Cable & Communications Corporation dba Mid-Rivers Wireless" },
    {  310910, "Verizon Wireless" },
    {  310920, "James Valley Wireless LLC" },
    {  310930, "Copper Valley Wireless" },
    {  310940, "Iris Wireless LLC" },
    {  310950, "Texas RSA 1 dba XIT Wireless" },
    {  310960, "UBET Wireless" },
    {  310970, "Globalstar USA" },
    {  310980, "Texas RSA 7B3 dba Peoples Wireless Services" },
    {  310990, "Worldcall Interconnect" },
    {  311000, "Mid-Tex Cellular Ltd." },
    {  311010, "Chariton Valley Communications Corp., Inc." },
    {  311020, "Missouri RSA No. 5 Partnership" },
    {  311030, "Indigo Wireless, Inc." },
    {  311040, "Commnet Wireless LLC" },
    {  311050, "Thumb Cellular Limited Partnership" },
    {  311060, "Space Data Corporation" },
    {  311070, "Wisconsin RSA #7 Limited Partnership" },
    {  311080, "Pine Telephone Company dba Pine Cellular" },
    {  311090, "LongLines Wireless" },
    {  311100, "Nex-Tech Wireless LLC" },
    {  311110, "Verizon Wireless" },
    {  311120, "Choice Phone LLC" },
    {  311130, "Light Squared LP" },
    {  311140, "Cross Telephone Company" },
    {  311150, "Wilkes Cellular Inc." },
    {  311160, "Light Squared LP" },
    {  311170, "PetroCom LLC" },
    {  311180, "Cingular Wireless, Licensee Pacific Telesis Mobile Services, LLC" },
    {  311190, "Cellular Properties Inc." },
    {  311200, "ARINC" },
    {  311210, "Emery Telecom-Wireless Inc" },
    {  311220, "United States Cellular" },
    {  311230, "Cellular South Inc" },
    {  311240, "Cordova Wireless Communications Inc" },
    {  311250, "Wave Runner LLC" },
    {  311260, "Clearwire Corporation" },
    {  311270, "Verizon Wireless" },
    {  311271, "Verizon Wireless" },
    {  311272, "Verizon Wireless" },
    {  311273, "Verizon Wireless" },
    {  311274, "Verizon Wireless" },
    {  311275, "Verizon Wireless" },
    {  311276, "Verizon Wireless" },
    {  311277, "Verizon Wireless" },
    {  311278, "Verizon Wireless" },
    {  311279, "Verizon Wireless" },
    {  311280, "Verizon Wireless" },
    {  311281, "Verizon Wireless" },
    {  311282, "Verizon Wireless" },
    {  311283, "Verizon Wireless" },
    {  311284, "Verizon Wireless" },
    {  311285, "Verizon Wireless" },
    {  311286, "Verizon Wireless" },
    {  311287, "Verizon Wireless" },
    {  311288, "Verizon Wireless" },
    {  311289, "Verizon Wireless" },
    {  311290, "Pinpoint Wireless Inc." },
    {  311300, "Nexus Communications Inc" },
    {  311310, "Leaco Rural Telephone Company Inc" },
    {  311320, "Commnet Wireless LLC" },
    {  311330, "Bug Tussel Wireless LLC" },
    {  311340, "Illinois Valley Cellular" },
    {  311350, "Sagebrush Cellular Inc dba Nemont" },
    {  311360, "Stelera Wireless LLC" },
    {  311370, "GCI Communications Corp" },
    {  311380, "New Dimension Wireless Ltd" },
    {  311390, "Verizon Wireless" },
    {  311410, "Iowa RSA No. 2 Limited Partnership" },
    {  311420, "Northwest Missouri Cellular Limited Partnership" },
    {  311430, "RSA 1 Limited Partnership dba Cellular 29 Plus" },
    {  311440, "Bluegrass Cellular LLC" },
    {  311450, "Panhandle Telecommunication Systems Inc." },
    {  311460, "Fisher Wireless Services Inc" },
    {  311470, "Vitelcom Cellular Inc dba Innovative Wireless" },
    {  311480, "Verizon Wireless" },
    {  311481, "Verizon Wireless" },
    {  311482, "Verizon Wireless" },
    {  311483, "Verizon Wireless" },
    {  311484, "Verizon Wireless" },
    {  311485, "Verizon Wireless" },
    {  311486, "Verizon Wireless" },
    {  311487, "Verizon Wireless" },
    {  311488, "Verizon Wireless" },
    {  311489, "Verizon Wireless" },
    {  311490, "Sprintcom Inc" },
    {  311500, "Mosaic Telecom Inc" },
    {  311510, "Light Squared LP" },
    {  311520, "Light Squared LP" },
    {  311530, "Newcore Wireless LLC" },
    {  311540, "Proximiti Mobility Inc" },
    {  311550, "Commnet Midwest LLC" },
    {  311560, "OTZ Communications Inc" },
    {  311570, "Bend Cable Communications LLC" },
    {  311580, "United States Cellular" },
    {  311590, "California RSA No3 Ltd Partnership dba Golden State Cellular" },
    {  311600, "Cox TMI Wireless LLC" },
    {  311610, "North Dakota Network Co." },
    {  311620, "Terrestar Networks Inc" },
    {  311630, "Corr Wireless Communications" },
    {  311640, "Standing Rock Telecommunications" },
    {  311650, "United Wireless Inc" },
    {  311660, "Metro PCS Wireless Inc" },
    {  311670, "Pine Belt Cellular Inc dba Pine Belt Wireless" },
    {  311680, "GreenFly LLC" },
    {  311690, "TeleBeeper of New Mexico Inc" },
    {  311700, "TotalSolutions Telecom LLC" },
    {  311710, "Northeast Wireless Networks LLC" },
    {  311720, "Maine PCS LLC" },
    {  311730, "Proximiti Mobility Inc" },
    {  311740, "Telalaska Cellular" },
    {  311750, "NetAmerica Alliance LLC" },
    {  311760, "Edigen Inc" },
    {  311770, "Radio Mobile Access Inc" },
    {  311800, "Bluegrass Cellular LLC" },
    {  311810, "Bluegrass Cellular LLC" },
    {  311820, "Kineto Wireless Inc" },
    {  311830, "Thumb Cellular LLC" },
    {  311840, "Nsight Spectrum LLC" },
    {  311850, "Nsight Spectrum LLC" },
    {  311860, "Uintah Basin Electronic Telecommunications" },
    {  311870, "Sprintcom Inc" },
    {  311880, "Sprintcom Inc" },
    {  311890, "Globecom Network Services Corporation" },
    {  311900, "Gigsky Inc." },
    {  311910, "SI Wireless LLC" },
    {  311920, "Missouri RSA No 5 Partnership dba Charlton Valley Wireless Services" },
    {  311940, "Clearwire Corporation" },
    {  311950, "Sunman Telecommunications corp." },
    {  311960, "Lycamobile USA Inc" },
    {  311970, "Big River Broadband LLC" },
    {  311980, "LigTel Communications" },
    {  311990, "VTel Wireless" },
    {  312010, "Charlton Valley Communication Corporation Inc" },
    {  312020, "Infrastructure Networks LLC" },
    {  312030, "Cross Wireless" },
    {  312040, "Custer Telephone Cooperative Inc" },
    {  312050, "Fuego Wireless LLC" },
    {  312060, "CoverageCo" },
    {  312070, "Adams Networks Inc" },
    {  312080, "South Georgia Regional Information Technology Authority" },
    {  312090, "Allied Wireless Communications Corporation" },
    {  312100, "ClearSky Technologies Inc" },
    {  312110, "Texas Energy Network LLC" },
    {  312120, "East Kentucky Network LLC dba Appalachian Wireless" },
    {  312130, "East Kentucky Network LLC dba Appalachian Wireless" },
    {  312140, "Cleveland Unlimited Inc" },
    {  312150, "Northwest Cell" },
    {  312160, "RSA1 Limited Partnership dba Chat Mobility" },
    {  312170, "Iowa RSA No. 2 Limited Partnership" },
    {  312180, "Keystone Wireless LLC" },
    {  312190, "Sprint-Nextel Communications Inc" },
    {  312200, "Voyager Mobility LLC" },
    {  313100, "Assigned to Public Safety" },
    {  316010, "Sprint-Nextel Communications Inc" },
    {  316011, "Southern Communications Services Inc." },
    {  334001, "COMUNICACIONES DIGITALES DEL NORTE, S.A. DE C.V" },
    {  334010, "NII DIGITAL, S. DE R.L. DE C.V." },
    {  334020, "RADIOMOVIL DIPSA, S.A. DE C.V." },
    {  334030, "PEGASO COMUNICACIONES Y SISTEMAS, S.A. DE C.V." },
    {  334040, "IUSACELL PCS DE MEXICO, S.A. DE  C.V." },
    {  334050, "COMUNICACIONES CELULARES DE OCCIDENTE, S.A. DE C.V. / SISTEMAS TELEFONICOS PORTATILES CELULARES, S.A. DE C.V. / TELECOMUNICACIONES DEL GOLFO, S.A. DE C.V. / SOS TELECOMUNICACIONES, S.A. DE C.V. / PORTATEL DEL SURESTE, S.A. DE C.V." },
    {  334060, "SERVICIOS DE ACCESO INALAMBRICO, S.A DE C.V." },
    {  334066, "TELEFONOS DE MEXICO, S.A.B. DE C.V" },
    {  334070, "OPERADORA UNEFON, S. A. DE C.V." },
    {  334080, "OPERADORA UNEFON, S. A. DE C.V." },
    {  334090, "NII DIGITAL, S. DE R.L. DE C.V." },
    {  338020, "Cable & Wireless Jamaica Ltd." },
    {  338050, "Digicel (Jamaica) Ltd." },
    {  338110, "Cable & Wireless Jamaica Ltd trading as Lime" },
    {  340010, "Orange Caraibe Mobiles" },
    {  340020, "Outremer Telecom" },
    {  340030, "Saint Martin et Saint Barthelemy Telcell Sarl" },
    {  340080, "Dauphin Telecom" },
    {  340100, "Guadeloupe Telephone Mobile" },
    {  340110, "Guyane Telephone Mobile" },
    {  340120, "Martinique Telephone Mobile" },
    {  340200, "Bouygues Telecom Caraibe" },
    {  342600, "Cable & Wireless (Barbados) Ltd. trading as Lime" },
    {  342820, "Sunbeach Communications" },
    {  344030, "APUA PCS" },
    {  344920, "Cable & Wireless (Antigua) trading as Lime" },
    {  344930, "AT&T Wireless (Antigua)" },
    {  346140, "Cable & Wireless (Cayman) trading as Lime" },
    {  348170, "Cable & Wireless (BVI) Ltd trading as Lime" },
    {  348370, "BVI Cable TV Ltd" },
    {  348570, "Caribbean Cellular Telephone Ltd." },
    {  348770, "Digicel (BVI) Ltd" },
    {  350000, "Bermuda Digital Communications Ltd (CellOne)" },
    {  352110, "Cable & Wireless Grenada ltd trading as lime" },
    {  354860, "Cable & Wireless (West Indies) Ltd trading as Lime" },
    {  356110, "Cable & Wireless St Kitts & Nevis Ltd trading as Lime" },
    {  358110, "Cable & Wireless (St Lucia) Ltd trading as Lime" },
    {  360110, "Cable & Wireless St Vincent and the Grenadines Ltd trading as Lime" },
    {  362510, "TELCELL GSM" },
    {  362690, "CT GSM" },
    {  362910, "SETEL GSM" },
    {  363010, "Setar GSM" },
    {  365010, "Weblinks Limited" },
    {  365840, "Cable and Wireless (Anguilla) Ltd trading as Lime" },
    {  366110, "Cable & Wireless Dominica Ltd trading as Lime" },
    {  368010, "ETECSA" },
    {  370010, "Orange Dominicana, S.A." },
    {  370020, "Verizon Dominicana S.A." },
    {  370030, "Tricom S.A." },
    {  370040, "CentennialDominicana" },
    {  372010, "Comcel" },
    {  372020, "Digicel" },
    {  372030, "Rectel" },
    {  374120, "TSTT Mobile" },
    {  374130, "Digicel Trinidad and Tobago Ltd." },
    {  374140, "LaqTel Ltd." },
    {  376350, "Cable & Wireless (TCI) Ltd trading as Lime" },
    {  376352, "IslandCom Communications Ltd." },
    {  376360, "IslandCom Communication Ltd" },
    {  400010, "\"Azercell Telecom\" LLC" },
    {  400020, "\"Bakcell\" LLC" },
    {  400030, "\"Catel\" LLC" },
    {  400040, "\"Azerfon\" LLC" },
    {  401010, "Kar-Tel llc" },
    {  401020, "TSC Kazak Telecom" },
    {  402110, "Bhutan Telecom Ltd" },
    {  402170, "B-Mobile of Bhutan Telecom" },
    {  404000, "Dishnet Wireless Ltd, Madhya Pradesh" },
    {  404010, "Aircell Digilink India Ltd., Haryana" },
    {  404020, "Bharti Airtel Ltd., Punjab" },
    {  404030, "Bharti Airtel Ltd., H.P." },
    {  404040, "Idea Cellular Ltd., Delhi" },
    {  404050, "Fascel Ltd., Gujarat" },
    {  404060, "Bharti Airtel Ltd., Karnataka" },
    {  404070, "Idea Cellular Ltd., Andhra Pradesh" },
    {  404090, "Reliance Telecom Ltd., Assam" },
    {  404100, "Bharti Airtel Ltd., Delhi" },
    {  404110, "Hutchison Essar Mobile Services Ltd, Delhi" },
    {  404120, "Idea Mobile Communications Ltd., Haryana" },
    {  404130, "Hutchinson Essar South Ltd., Andhra Pradesh" },
    {  404140, "Spice Communications PVT Ltd., Punjab" },
    {  404150, "Aircell Digilink India Ltd., UP (East)" },
    {  404160, "Bharti Airtel Ltd, North East" },
    {  404170, "Dishnet Wireless Ltd, West Bengal" },
    {  404180, "Reliance Telecom Ltd., H.P." },
    {  404190, "Idea Mobile Communications Ltd., Kerala" },
    {  404200, "Hutchinson Essar Ltd, Mumbai" },
    {  404210, "BPL Mobile Communications Ltd., Mumbai" },
    {  404220, "Idea Cellular Ltd., Maharashtra" },
    {  404230, "Idea Cellular Ltd, Maharashtra" },
    {  404240, "Idea Cellular Ltd., Gujarat" },
    {  404250, "Dishnet Wireless Ltd, Bihar" },
    {  404270, "Hutchison Essar Cellular Ltd., Maharashtra" },
    {  404290, "Dishnet Wireless Ltd, Assam" },
    {  404300, "Hutchison Telecom East Ltd, Kolkata" },
    {  404310, "Bharti Airtel Ltd., Kolkata" },
    {  404330, "Dishnet Wireless Ltd, North East" },
    {  404340, "BSNL, Haryana" },
    {  404350, "Dishnet Wireless Ltd, Himachal Pradesh" },
    {  404360, "Reliance Telecom Ltd., Bihar" },
    {  404370, "Dishnet Wireless Ltd, J&K" },
    {  404380, "BSNL, Assam" },
    {  404400, "Bharti Airtel Ltd., Chennai" },
    {  404410, "Aircell Cellular Ltd, Chennai" },
    {  404420, "Aircel Ltd., Tamil Nadu" },
    {  404430, "Hutchison Essar Cellular Ltd., Tamil Nadu" },
    {  404440, "Spice Communications PVT Ltd., Karnataka" },
    {  404460, "Hutchison Essar Cellular Ltd., Kerala" },
    {  404480, "Dishnet Wireless Ltd, UP (West)" },
    {  404490, "Bharti Airtel Ltd., Andra Pradesh" },
    {  404500, "Reliance Telecom Ltd., North East" },
    {  404510, "BSNL, H.P." },
    {  404520, "Reliance Telecom Ltd., Orissa" },
    {  404530, "BSNL, Punjab" },
    {  404540, "BSNL, UP (West)" },
    {  404550, "BSNL, UP (East)" },
    {  404560, "Idea Mobile Communications Ltd., UP (West)" },
    {  404570, "BSNL, Gujarat" },
    {  404580, "BSNL, Madhya Pradesh" },
    {  404590, "BSNL, Rajasthan" },
    {  404600, "Aircell Digilink India Ltd., Rajasthan" },
    {  404610, "Dishnet Wireless Ltd, Punjab" },
    {  404620, "BSNL, J&K" },
    {  404630, "Dishnet Wireless Ltd, Haryana" },
    {  404640, "BSNL, Chennai" },
    {  404650, "Dishnet Wireless Ltd, UP (East)" },
    {  404660, "BSNL, Maharashtra" },
    {  404670, "Reliance Telecom Ltd., Madhya Pradesh" },
    {  404680, "MTNL, Delhi" },
    {  404690, "MTNL, Mumbai" },
    {  404700, "Bharti Hexacom Ltd, Rajasthan" },
    {  404710, "BSNL, Karnataka" },
    {  404720, "BSNL, Kerala" },
    {  404730, "BSNL, Andhra Pradesh" },
    {  404740, "BSNL, West Bengal" },
    {  404750, "BSNL, Bihar" },
    {  404760, "BSNL, Orissa" },
    {  404770, "BSNL, North East" },
    {  404780, "BTA Cellcom Ltd., Madhya Pradesh" },
    {  404790, "BSNL, Andaman & Nicobar" },
    {  404800, "BSNL, Tamil Nadu" },
    {  404810, "BSNL, Kolkata" },
    {  404820, "Idea Telecommunications Ltd, H.P." },
    {  404830, "Reliable Internet Services Ltd., Kolkata" },
    {  404840, "Hutchinson Essar South Ltd., Chennai" },
    {  404850, "Reliance Telecom Ltd., W.B. & A.N." },
    {  404860, "Hutchinson Essar South Ltd., Karnataka" },
    {  404870, "Idea Telecommunications Ltd, Rajasthan" },
    {  404880, "Hutchison Essar South Ltd, Punjab" },
    {  404890, "Idea Telecommunications Ltd, UP (East)" },
    {  404900, "Bharti Airtel Ltd., Maharashtra" },
    {  404910, "Dishnet Wireless Ltd, Kolkata" },
    {  404920, "Bharti Airtel Ltd., Mumbai" },
    {  404930, "Bharti Airtel Ltd., Madhya Pradesh" },
    {  404940, "Bharti Airtel Ltd., Tamil Nadu" },
    {  404950, "Bharti Airtel Ltd., Kerala" },
    {  404960, "Bharti Airtel Ltd., Haryana" },
    {  404970, "Bharti Airtel Ltd., UP (West)" },
    {  404980, "Bharti Airtel Ltd., Gujarat" },
    {  404990, "Dishnet Wireless Ltd, Kerala" },
    {  405000, "Shyam Telelink Ltd, Rajasthan" },
    {  405005, "Reliance Communications Ltd/GSM, Delhi" },
    {  405006, "Reliance Communications Ltd/GSM, Gujarat" },
    {  405007, "Reliance Communications Ltd/GSM, Haryana" },
    {  405009, "Reliance Communications Ltd/GSM, J&K" },
    {  405010, "Reliance Communications Ltd,/GSM Karnataka" },
    {  405011, "Reliance Communications Ltd/GSM, Kerala" },
    {  405012, "Reliance Infocomm Ltd, Andhra Pradesh" },
    {  405013, "Reliance Communications Ltd/GSM, Maharashtra" },
    {  405014, "Reliance Communications Ltd/GSM, Madhya Pradesh" },
    {  405018, "Reliance Communications Ltd/GSM, Punjab" },
    {  405020, "Reliance Communications Ltd/GSM, Tamilnadu" },
    {  405021, "Reliance Communications Ltd/GSM, UP (East)" },
    {  405022, "Reliance Communications Ltd/GSM, UP (West)" },
    {  405025, "Tata Teleservices Ltd/GSM, Andhra Pradesh" },
    {  405027, "Tata Teleservices Ltd,/GSM Bihar" },
    {  405029, "Tata Teleservices Ltd/GSM, Delhi" },
    {  405030, "Tata Teleservices Ltd/GSM, Gujarat" },
    {  405031, "Tata Teleservices Ltd/GSM, Haryana" },
    {  405032, "Tata Teleservices Ltd/GSM, Himachal Pradesh" },
    {  405033, "Reliance Infocomm Ltd, Bihar" },
    {  405034, "Tata Teleservices Ltd/GSM, Kamataka" },
    {  405035, "Tata Teleservices Ltd/GSM, Kerala" },
    {  405036, "Tata Teleservices Ltd/GSM, Kolkata" },
    {  405037, "Tata Teleservices Ltd/GSM, Maharashtra" },
    {  405038, "Tata Teleservices Ltd/GSM, Madhya Pradesh" },
    {  405039, "Tata Teleservices Ltd/GSM, Mumbai" },
    {  405040, "Reliance Infocomm Ltd, Chennai" },
    {  405041, "Tata Teleservices Ltd/GSM, Orissa" },
    {  405042, "Tata Teleservices Ltd/GSM, Punjab" },
    {  405043, "Tata Teleservices Ltd/GSM, Rajasthan" },
    {  405044, "Tata Teleservices Ltd/GSM, Tamilnadu" },
    {  405045, "Tata Teleservices Ltd/GSM, UP (East)" },
    {  405046, "Tata Teleservices Ltd/GSM, UP (West)" },
    {  405047, "Tata Teleservices Ltd/GSM, West Bengal" },
    {  405080, "Reliance Infocomm Ltd, Himachal Pradesh" },
    {  405120, "Reliance Infocomm Ltd, Kolkata" },
    {  405150, "Reliance Infocomm Ltd, Mumbai" },
    {  405170, "Reliance Infocomm Ltd, Orissa" },
    {  405230, "Reliance Infocomm Ltd, West bengal" },
    {  405280, "Tata Teleservices Ltd, Chennai" },
    {  405520, "Bharti Airtel Ltd, Bihar" },
    {  405530, "Bharti Airtel Ltd, Orissa" },
    {  405540, "Bharti Airtel Ltd, UP (East)" },
    {  405550, "Bharti Airtel Ltd, J&K" },
    {  405560, "Bharti Airtel Ltd, Assam" },
    {  405660, "Hutchison Essar South Ltd, UP (West)" },
    {  405670, "Hutchison Essar South Ltd, Orissa" },
    {  405680, "Vodaphone/Hutchison, Madhya Pradesh" },
    {  405700, "Aditya Birla Telecom Ltd, Bihar" },
    {  405710, "Essar Spacetel Ltd, Himachal Pradesh" },
    {  405720, "Essar Spacetel Ltd, North East" },
    {  405730, "Essar Spacetel Ltd, Assam" },
    {  405740, "Essar Spacetel Ltd, J&K" },
    {  405750, "Vodafone Essar Spacetel Ltd, J&K" },
    {  405751, "Vodafone Essar Spacetel Ltd, Assam" },
    {  405752, "Vodafone Essar Spacetel Ltd, Bihar" },
    {  405753, "Vodafone Essar Spacetel Ltd, Orissa" },
    {  405754, "Vodafone Essar Spacetel Ltd, Himachal Pradesh" },
    {  405755, "Vodafone Essar Spacetel Ltd, North East" },
    {  405760, "Essar Spacetel Ltd, Orissa" },
    {  405770, "Essar Spacetel Ltd, Maharashtra" },
    {  405799, "Idea Cellular Ltd, MUMBAI" },
    {  405800, "Aircell Ltd, Delhi" },
    {  405801, "Aircell Ltd, Andhra Pradesh" },
    {  405802, "Aircell Ltd, Gujarat" },
    {  405803, "Aircell Ltd, Kamataka" },
    {  405804, "Aircell Ltd, Maharashtra" },
    {  405805, "Aircell Ltd, Mumbai" },
    {  405806, "Aircell Ltd, Rajasthan" },
    {  405807, "Dishnet Wireless Ltd, Haryana" },
    {  405808, "Dishnet Wireless Ltd, Madhya Pradesh" },
    {  405809, "Dishnet Wireless Ltd, Kerala" },
    {  405810, "Aircell Ltd, Delhi" },
    {  405820, "Aircell Ltd, Andhra Pradesh" },
    {  405830, "Aircell Ltd, Gujarat" },
    {  405840, "Aircell Ltd, Maharashtra" },
    {  405850, "Aircell Ltd, Mumbai" },
    {  405860, "Aircell Ltd, Rajasthan" },
    {  410010, "Mobilink" },
    {  410030, "PAK Telecom Mobile Ltd. (UFONE)" },
    {  410040, "CMPak" },
    {  410060, "Telenor Pakistan" },
    {  410070, "Warid Telecom" },
    {  412010, "AWCC" },
    {  412200, "Roshan" },
    {  412400, "Areeba Afghanistan" },
    {  412500, "Etisalat" },
    {  412800, "Afghan Telecom" },
    {  412880, "Afghan Telecom" },
    {  413020, "MTN Network Ltd." },
    {  413030, "Celtel Lanka Ltd." },
    {  414010, "Myanmar Post and Telecommunication" },
    {  415050, "Ogero Telecom" },
    {  415320, "Cellis" },
    {  415330, "Cellis" },
    {  415340, "Cellis" },
    {  415350, "Cellis" },
    {  415360, "Libancell" },
    {  415370, "Libancell" },
    {  415380, "Libancell" },
    {  415390, "Libancell" },
    {  416010, "Fastlink" },
    {  416020, "Xpress" },
    {  416030, "Umniah" },
    {  416770, "MobileCom" },
    {  417010, "Syriatel" },
    {  417020, "Spacetel Syria" },
    {  417090, "Syrian Telecom" },
    {  418050, "Asia Cell" },
    {  418200, "Zain Iraq (previously Atheer)" },
    {  418300, "Zain Iraq (previously Iraqna)" },
    {  418400, "Korek Telecom" },
    {  418470, "Iraq Central Cooperative Association for Communication and Transportation" },
    {  418480, "ITC Fanoos" },
    {  418490, "Iraqtel" },
    {  418620, "Itisaluna" },
    {  418700, "Kalimat" },
    {  418800, "Iraqi Telecommunications & Post Company (ITPC)" },
    {  418810, "ITPC (Al-Mazaya)" },
    {  418830, "ITPC (Sader Al-Iraq)" },
    {  418840, "ITPC (Eaamar Albasrah)" },
    {  418850, "ITPC (Anwar Yagotat Alkhalee)" },
    {  418860, "ITPC (Furatfone)" },
    {  418870, "ITPC (Al-Seraj)" },
    {  418880, "ITPC (High Link)" },
    {  418890, "ITPC (Al-Shams)" },
    {  418910, "ITPC (Belad Babel)" },
    {  418920, "ITPC (Al Nakheel)" },
    {  418930, "ITPC (Iraqcell)" },
    {  418940, "ITPC (Shaly)" },
    {  419020, "ZAIN" },
    {  419030, "Wataniya Telecom" },
    {  419040, "Viva" },
    {  420010, "Saudi Telecom" },
    {  420030, "Etihad Etisalat Company (Mobily)" },
    {  421010, "Yemen Mobile Phone Company" },
    {  421020, "Spacetel Yemen" },
    {  421040, "Y-Telecom" },
    {  422020, "Oman Mobile Telecommunications Company (Oman Mobile)" },
    {  422030, "Oman Qatari Telecommunications Company (Nawras)" },
    {  422040, "Oman Telecommunications Company (Omantel)" },
    {  424020, "Etisalat" },
    {  425010, "Partner Communications Co. Ltd." },
    {  425020, "Cellcom Israel Ltd" },
    {  425030, "Pelephone Communications Ltd" },
    {  425040, "Globalsim Ltd" },
    {  425060, "Wataniya" },
    {  425070, "Mirs Ltd" },
    {  425080, "Golan Telecom Ltd" },
    {  425110, "365 Telecom (MVNO)" },
    {  425120, "Free Telecom (MVNO)" },
    {  425130, "Ituran Cellular Communications" },
    {  425140, "Alon Cellular Ltd." },
    {  425150, "Home Cellular (MVNO)" },
    {  425160, "Rami Levi (MVNO)" },
    {  425170, "Gale Phone (MVNO)" },
    {  425180, "Cellact Communications Ltd (MVNO)" },
    {  425190, "Azi Communications Ltd" },
    {  425200, "Bezeq Ltd" },
    {  425210, "B.I.P Communications Ltd." },
    {  425230, "Beezz Communication Solutions Ltd." },
    {  426010, "Bahrain Telecommunications Company (BATELCO)" },
    {  426020, "Zain Bahrain" },
    {  426030, "Civil Aviation Authority" },
    {  426040, "STC Bahrain" },
    {  426050, "Royal Court" },
    {  427010, "QATARNET" },
    {  427060, "Ooredoo Q.S.C./MOI LTE" },
    {  428990, "Mobicom" },
    {  429010, "Nepal Telecommunications" },
    {  432110, "Telecommunication Company of Iran (TCI)" },
    {  432140, "Telecommunication Kish Co. (KIFZO)" },
    {  432190, "Telecommunication Company of Iran (TCI) - Isfahan Celcom GSM" },
    {  434010, "Buztel" },
    {  434020, "Uzmacom" },
    {  434040, "Daewoo Unitel" },
    {  434050, "Coscom" },
    {  434070, "Uzdunrobita" },
    {  436010, "JC Somoncom" },
    {  436020, "CJSC Indigo Tajikistan" },
    {  436030, "TT mobile" },
    {  436040, "Josa Babilon-T" },
    {  436050, "CTJTHSC Tajik-tel" },
    {  437010, "Bitel GSM" },
    {  438010, "Barash Communication Technologies (BCTI)" },
    {  438020, "TM-Cell" },
    {  440010, "NTT DoCoMo Inc." },
    {  440020, "NTT DoCoMo Kansai Inc." },
    {  440030, "NTT DoCoMo Hokuriku Inc." },
    {  440040, "SoftBank" },
    {  440060, "SoftBank" },
    {  440070, "KDDI Corporation" },
    {  440080, "KDDI Corporation" },
    {  440090, "NTT DoCoMo Kansai Inc." },
    {  440100, "NTT DoCoMo Kansai Inc." },
    {  440110, "NTT DoCoMo Tokai Inc." },
    {  440120, "NTT DoCoMo Inc." },
    {  440130, "NTT DoCoMo Inc." },
    {  440140, "NTT DoCoMo Tohoku Inc." },
    {  440150, "NTT DoCoMo Inc." },
    {  440160, "NTT DoCoMo Inc." },
    {  440170, "NTT DoCoMo Inc." },
    {  440180, "NTT DoCoMo Tokai Inc." },
    {  440190, "NTT DoCoMo Hokkaido Inc." },
    {  440200, "SoftBank" },
    {  440210, "NTT DoCoMo Inc." },
    {  440220, "NTT DoCoMo Kansai Inc." },
    {  440230, "NTT DoCoMo Tokai Inc." },
    {  440240, "NTT DoCoMo Chugoku Inc." },
    {  440250, "NTT DoCoMo Hokkaido Inc." },
    {  440260, "NTT DoCoMo Kyushu Inc." },
    {  440270, "NTT DoCoMo Tohoku Inc." },
    {  440280, "NTT DoCoMo Shikoku Inc." },
    {  440290, "NTT DoCoMo Inc." },
    {  440300, "NTT DoCoMo Inc." },
    {  440310, "NTT DoCoMo Kansai Inc." },
    {  440320, "NTT DoCoMo Inc." },
    {  440330, "NTT DoCoMo Tokai Inc." },
    {  440340, "NTT DoCoMo Kyushu Inc." },
    {  440350, "NTT DoCoMo Kansai Inc." },
    {  440360, "NTT DoCoMo Inc." },
    {  440370, "NTT DoCoMo Inc." },
    {  440380, "NTT DoCoMo Inc." },
    {  440390, "NTT DoCoMo Inc." },
    {  440400, "SoftBank" },
    {  440410, "SoftBank" },
    {  440420, "SoftBank" },
    {  440430, "SoftBank" },
    {  440440, "SoftBank" },
    {  440450, "SoftBank" },
    {  440460, "SoftBank" },
    {  440470, "SoftBank" },
    {  440480, "SoftBank" },
    {  440490, "NTT DoCoMo Inc." },
    {  440500, "KDDI Corporation" },
    {  440510, "KDDI Corporation" },
    {  440520, "KDDI Corporation" },
    {  440530, "KDDI Corporation" },
    {  440540, "KDDI Corporation" },
    {  440550, "KDDI Corporation" },
    {  440560, "KDDI Corporation" },
    {  440580, "NTT DoCoMo Kansai Inc." },
    {  440600, "NTT DoCoMo Kansai Inc." },
    {  440610, "NTT DoCoMo Chugoku Inc." },
    {  440620, "NTT DoCoMo Kyushu Inc." },
    {  440630, "NTT DoCoMo Inc." },
    {  440640, "NTT DoCoMo Inc." },
    {  440650, "NTT DoCoMo Shikoku Inc." },
    {  440660, "NTT DoCoMo Inc." },
    {  440670, "NTT DoCoMo Tohoku Inc." },
    {  440680, "NTT DoCoMo Kyushu Inc." },
    {  440690, "NTT DoCoMo Inc." },
    {  440700, "KDDI Corporation" },
    {  440710, "KDDI Corporation" },
    {  440720, "KDDI Corporation" },
    {  440730, "KDDI Corporation" },
    {  440740, "KDDI Corporation" },
    {  440750, "KDDI Corporation" },
    {  440760, "KDDI Corporation" },
    {  440770, "KDDI Corporation" },
    {  440780, "Okinawa Cellular Telephone" },
    {  440790, "KDDI Corporation" },
    {  440800, "TU-KA Cellular Tokyo Inc." },
    {  440810, "TU-KA Cellular Tokyo Inc." },
    {  440820, "TU-KA Phone Kansai Inc." },
    {  440830, "TU-KA Cellular Tokai Inc." },
    {  440840, "TU-KA Phone Kansai Inc." },
    {  440850, "TU-KA Cellular Tokai Inc." },
    {  440860, "TU-KA Cellular Tokyo Inc." },
    {  440870, "NTT DoCoMo Chugoku Inc." },
    {  440880, "KDDI Corporation" },
    {  440890, "KDDI Corporation" },
    {  440900, "SoftBank" },
    {  440920, "SoftBank" },
    {  440930, "SoftBank" },
    {  440940, "SoftBank" },
    {  440950, "SoftBank" },
    {  440960, "SoftBank" },
    {  440970, "SoftBank" },
    {  440980, "SoftBank" },
    {  440990, "NTT DoCoMo Inc." },
    {  441400, "NTT DoCoMo Inc." },
    {  441410, "NTT DoCoMo Inc." },
    {  441420, "NTT DoCoMo Inc." },
    {  441430, "NTT DoCoMo Kansai Inc." },
    {  441440, "NTT DoCoMo Chugoku Inc." },
    {  441450, "NTT DoCoMo Shikoku Inc." },
    {  441500, "TU-KA Cellular Tokyo Inc." },
    {  441510, "TU-KA Phone Kansai Inc." },
    {  441610, "SoftBank" },
    {  441620, "SoftBank" },
    {  441630, "SoftBank" },
    {  441640, "SoftBank" },
    {  441650, "SoftBank" },
    {  441700, "KDDI Corporation" },
    {  441900, "NTT DoCoMo Inc." },
    {  441910, "NTT DoCoMo Inc." },
    {  441920, "NTT DoCoMo Inc." },
    {  441930, "NTT DoCoMo Hokkaido Inc." },
    {  441940, "NTT DoCoMo Tohoku Inc." },
    {  441980, "NTT DoCoMo Kyushu Inc." },
    {  441990, "NTT DoCoMo Kyushu Inc." },
    {  450020, "KT" },
    {  450030, "SK Telecom" },
    {  450050, "SK Telecom" },
    {  450060, "LG U+" },
    {  450080, "KT" },
    {  450110, "SK Telecom (MVNO)" },
    {  452010, "Mobifone" },
    {  452020, "Vinaphone" },
    {  452030, "S Telecom (CDMA)" },
    {  452040, "Viettel" },
    {  452060, "EVN Telecom" },
    {  452070, "Beeline VN/GTEL Mobile JSC" },
    {  452080, "EVN Telecom" },
    {  454000, "GSM900/HKCSL" },
    {  454010, "MVNO/CITIC" },
    {  454020, "3G Radio System/HKCSL3G" },
    {  454030, "3G Radio System/Hutchison 3G" },
    {  454040, "GSM900/GSM1800/Hutchison" },
    {  454050, "CDMA/Hutchison" },
    {  454060, "GSM900/SmarTone" },
    {  454070, "MVNO/China Unicom International Ltd." },
    {  454080, "MVNO/Trident" },
    {  454090, "MVNO/China Motion Telecom (HK) Ltd." },
    {  454100, "GSM1800New World PCS Ltd." },
    {  454110, "MVNO/CHKTL" },
    {  454120, "GSM1800/Peoples Telephone Company Ltd." },
    {  454150, "3G Radio System/SMT3G" },
    {  454160, "GSM1800/Mandarin Communications Ltd." },
    {  454180, "GSM7800/Hong Kong CSL Ltd." },
    {  454190, "3G Radio System/Sunday3G" },
    {  454200, "Public Mobile Networks/Reserved" },
    {  454210, "Public Mobile Networks/Reserved" },
    {  454220, "Public Mobile Networks/Reserved" },
    {  454230, "Public Mobile Networks/Reserved" },
    {  454240, "Public Mobile Networks/Reserved" },
    {  454250, "Public Mobile Networks/Reserved" },
    {  454260, "Public Mobile Networks/Reserved" },
    {  454270, "Public Mobile Networks/Reserved" },
    {  454280, "Public Mobile Networks/Reserved" },
    {  454290, "Public Mobile Networks/Reserved" },
    {  454300, "Public Mobile Networks/Reserved" },
    {  454310, "Public Mobile Networks/Reserved" },
    {  454320, "Public Mobile Networks/Reserved" },
    {  454330, "Public Mobile Networks/Reserved" },
    {  454340, "Public Mobile Networks/Reserved" },
    {  454350, "Public Mobile Networks/Reserved" },
    {  454360, "Public Mobile Networks/Reserved" },
    {  454370, "Public Mobile Networks/Reserved" },
    {  454380, "Public Mobile Networks/Reserved" },
    {  454390, "Public Mobile Networks/Reserved" },
    {  455000, "SmarTone - Comunicacoes Moveis, S.A." },
    {  455010, "Companhia de Telecomunicacoes de Macau S.A.R.L." },
    {  455020, "China Telecom (Macau) Limitada" },
    {  455030, "Hutchison - Telefone (Macau) Limitada" },
    {  455040, "Companhia de Telecomunicacoes de Macau S.A.R.L." },
    {  455050, "Hutchison - Telefone (Macau) Limitada" },
    {  455060, "SmarTone - Comunicacoes Moveis, S.A." },
    {  456010, "Mobitel (Cam GSM)" },
    {  456020, "Hello" },
    {  456030, "S Telecom (CDMA)" },
    {  456040, "Cadcomms" },
    {  456050, "Starcell" },
    {  456060, "Smart" },
    {  456080, "Viettel" },
    {  456180, "Mfone" },
    {  457010, "Lao Telecommunications" },
    {  457020, "ETL Mobile" },
    {  457080, "Millicom" },
    {  460000, "China Mobile" },
    {  460010, "China Unicom" },
    {  460030, "China Unicom CDMA" },
    {  460040, "China Satellite Global Star Network" },
    {  470010, "GramenPhone" },
    {  470020, "Aktel" },
    {  470030, "Mobile 2000" },
    {  472010, "DhiMobile" },
    {  502100, "DIGI Telecommunications" },
    {  502120, "Malaysian Mobile Services Sdn Bhd" },
    {  502130, "Celcom (Malaysia) Berhad" },
    {  502140, "Telekom Malaysia Berhad" },
    {  502160, "DIGI Telecommunications" },
    {  502170, "Malaysian Mobile Services Sdn Bhd" },
    {  502180, "U Mobile Sdn. Bhd." },
    {  502190, "Celcom (Malaysia) Berhad" },
    {  502200, "Electcoms Wireless Sdn Bhd" },
    {  505010, "Telstra Corporation Ltd." },
    {  505020, "Optus Mobile Pty. Ltd." },
    {  505030, "Vodafone Network Pty. Ltd." },
    {  505040, "Department of Defence" },
    {  505050, "The Ozitel Network Pty. Ltd." },
    {  505060, "Hutchison 3G Australia Pty. Ltd." },
    {  505070, "Vodafone Network Pty. Ltd." },
    {  505080, "One.Tel GSM 1800 Pty. Ltd." },
    {  505090, "Airnet Commercial Australia Ltd." },
    {  505100, "Norfolk Telecom" },
    {  505110, "Telstra Corporation Ltd." },
    {  505120, "Hutchison Telecommunications (Australia) Pty. Ltd." },
    {  505130, "Railcorp" },
    {  505140, "AAPT Ltd" },
    {  505150, "3GIS Pty Ltd. (Telstra & Hutchison 3G)" },
    {  505160, "Victorian Rail Track" },
    {  505170, "Vivid Wireless Pty Ltd" },
    {  505180, "Pactel International Pty Ltd" },
    {  505190, "Lycamobile Pty Ltd" },
    {  505200, "Ausgrid Corporation" },
    {  505210, "Queensland Rail Limited" },
    {  505220, "iiNet Ltd" },
    {  505230, "Challenge Networks Pty Ltd" },
    {  505240, "Advanced Communications Technologies Pty. Ltd." },
    {  505250, "Pilbara Iron Company Services Pty Ltd" },
    {  505260, "Dialogue Communications Pty Ltd" },
    {  505270, "Nexium Telecommunications" },
    {  505280, "RCOM International Pty Ltd" },
    {  505300, "Compatel Limited" },
    {  505310, "BHP Billiton" },
    {  505620, "NBNCo Limited" },
    {  505680, "NBNCo Limited" },
    {  505710, "Telstra Corporation Ltd." },
    {  505720, "Telstra Corporation Ltd." },
    {  505880, "Localstar Holding Pty. Ltd." },
    {  505900, "Optus Ltd." },
    {  505990, "One.Tel GSM 1800 Pty. Ltd." },
    {  510000, "PSN" },
    {  510010, "Satelindo" },
    {  510080, "Natrindo (Lippo Telecom)" },
    {  510100, "Telkomsel" },
    {  510110, "Excelcomindo" },
    {  510210, "Indosat - M3" },
    {  510280, "Komselindo" },
    {  514010, "Telin Timor-Leste" },
    {  514020, "Timor Telecom" },
    {  514030, "Viettel Timor-Leste" },
    {  515010, "Islacom" },
    {  515020, "Globe Telecom" },
    {  515030, "Smart Communications" },
    {  515050, "Digitel" },
    {  520000, "CAT CDMA" },
    {  520010, "AIS GSM" },
    {  520020, "CAT CDMA" },
    {  520030, "Advanced Wireless Network Company Limited" },
    {  520040, "Real Future Company Limited" },
    {  520050, "DTAC Network Company Limited" },
    {  520150, "TOT Public Company Limited" },
    {  520180, "Total Access Communications Public Company Limited" },
    {  520200, "ACes Regional Services Company Limited" },
    {  520230, "Digital Phone Company Limited" },
    {  520470, "TOT Public Company Limited" },
    {  520990, "True Move Company Limited" },
    {  525010, "SingTel ST GSM900" },
    {  525020, "SingTel ST GSM1800" },
    {  525030, "MobileOne" },
    {  525050, "Starhub" },
    {  525120, "Digital Trunked Radio Network" },
    {  528110, "DST Com" },
    {  530000, "Reserved for AMPS MIN based IMSI's" },
    {  530010, "Vodafone New Zealand GSM Network" },
    {  530020, "Teleom New Zealand CDMA Network" },
    {  530030, "Woosh Wireless - CDMA Network" },
    {  530040, "TelstraClear - GSM Network" },
    {  530050, "Telecom New Zealand - UMTS Network" },
    {  530060, "FX Networks Ltd" },
    {  530070, "Bluereach Limited" },
    {  530240, "NZ Communications - UMTS Network" },
    {  537010, "Bmobile" },
    {  537020, "Telikom PNG Ltd" },
    {  537030, "Digicel Ltd" },
    {  539010, "Tonga Communications Corporation" },
    {  539430, "Digicel" },
    {  539880, "Digicel (Tonga) Ltd" },
    {  540020, "Bemobile (BMobile (SI) Ltd)" },
    {  541010, "SMILE" },
    {  541050, "Digicel Vanuatu" },
    {  541070, "WANTOK" },
    {  542010, "Vodafone (Fiji) Ltd" },
    {  542020, "Digicel (Fiji) Ltd" },
    {  542030, "Telecom Fiji Ltd (CDMA)" },
    {  546010, "OPT Mobilis" },
    {  547050, "VITI" },
    {  547100, "Mara Telecom" },
    {  547150, "Pacific Mobile Telecom" },
    {  547200, "Tikiphone" },
    {  548010, "Telecom Cook" },
    {  549010, "Telecom Samoa Cellular Ltd." },
    {  549270, "GoMobile SamoaTel Ltd" },
    {  550010, "FSM Telecom" },
    {  552010, "Palau National Communications Corp. (a.k.a. PNCC)" },
    {  553010, "Tuvalu Telecommunications Corporation" },
    {  555010, "Telecom Niue" },
    {  602010, "Mobinil" },
    {  602020, "Vodafone" },
    {  602030, "Etisalat" },
    {  603010, "Algerie Telecom" },
    {  603020, "Orascom Telecom Algerie" },
    {  604000, "Meditelecom (GSM)" },
    {  604010, "Ittissalat Al Maghrid" },
    {  605020, "Tunisie Telecom" },
    {  605030, "Orascom Telecom" },
    {  607010, "Gamcel" },
    {  607020, "Africell" },
    {  607030, "Comium Services Ltd" },
    {  607040, "Qcell" },
    {  608010, "Sonatel Mobiles (Orange)" },
    {  608020, "Sentel GSM (Tigo)" },
    {  608030, "Expresso Senegal" },
    {  608040, "CSU" },
    {  609010, "Mattel S.A." },
    {  609020, "Chinguitel S.A." },
    {  609100, "Mauritel Mobiles" },
    {  610010, "Malitel" },
    {  611010, "Orange Guinee" },
    {  611020, "Sotelgui" },
    {  611050, "Cellcom Guinee SA" },
    {  612020, "Atlantique Cellulaire" },
    {  612030, "Orange Cote d'Ivoire" },
    {  612040, "Comium Cote d'Ivoire" },
    {  612050, "Loteny Telecom" },
    {  612060, "Oricel Cote d'Ivoire" },
    {  612070, "Aircomm Cote d'Ivoire" },
    {  613020, "Celtel" },
    {  613030, "Telecel" },
    {  614010, "Sahel.Com" },
    {  614020, "Celtel" },
    {  614030, "Telecel" },
    {  615010, "Togo Telecom" },
    {  616010, "Libercom" },
    {  616020, "Telecel" },
    {  616030, "Spacetel Benin" },
    {  617010, "Cellplus" },
    {  617020, "Mahanagar Telephone (Mauritius) Ltd." },
    {  617030, "Mahanagar Telephone (Mauritius) Ltd." },
    {  617100, "Emtel" },
    {  618040, "Comium Liberia" },
    {  619010, "Celtel" },
    {  619020, "Millicom" },
    {  619030, "Africell" },
    {  619040, "Comium (Sierra Leone) Ltd" },
    {  619050, "Lintel (Sierra Leone) Ltd." },
    {  619250, "Mobitel" },
    {  619400, "Datatel (SL) Ltd GSM" },
    {  619500, "Datatel (SL) Ltd CDMA" },
    {  620010, "Spacefon" },
    {  620020, "Ghana Telecom Mobile" },
    {  620030, "Mobitel" },
    {  620040, "Kasapa Telecom Ltd." },
    {  620110, "Netafriques Dot Com Ltd" },
    {  621200, "Econet Wireless Nigeria Ltd." },
    {  621300, "MTN Nigeria Communications" },
    {  621400, "MTEL" },
    {  621500, "Globacom" },
    {  621600, "EMTS" },
    {  622010, "Celtel" },
    {  622020, "Tchad Mobile" },
    {  623010, "Centrafrique Telecom Plus (CTP)" },
    {  623020, "Telecel Centrafrique (TC)" },
    {  623030, "Celca (Socatel)" },
    {  624010, "Mobile Telephone Networks Cameroon" },
    {  624020, "Orange Cameroun" },
    {  625010, "Cabo Verde Telecom" },
    {  625020, "T+Telecomunicacoes" },
    {  626010, "Companhia Santomese de Telecomunicacoes" },
    {  627010, "Guinea Ecuatorial de Telecomunicaciones Sociedad Anonima (GETESA)" },
    {  628010, "LIBERTIS" },
    {  628020, "MOOV" },
    {  628030, "CELTEL" },
    {  628040, "USAN GABON" },
    {  628050, "Reseau de l'Administration Gabonaise (RAG)" },
    {  629010, "Celtel" },
    {  629100, "Libertis Telecom" },
    {  630010, "Vodacom Congo RDC sprl" },
    {  630020, "AIRTEL sprl" },
    {  630050, "Supercell Sprl" },
    {  630860, "Congo-Chine Telecom s.a.r.l." },
    {  630880, "YOZMA TIMETURNS sprl" },
    {  630890, "OASIS sprl" },
    {  630900, "Africell RDC" },
    {  631020, "Unitel" },
    {  631040, "Movicel" },
    {  632010, "Guinetel S.A." },
    {  632020, "Spacetel Guine-Bissau S.A." },
    {  633010, "Cable & Wireless (Seychelles) Ltd." },
    {  633020, "Mediatech International Ltd." },
    {  633100, "Telecom (Seychelles) Ltd." },
    {  634010, "SD Mobitel" },
    {  634020, "Areeba-Sudan" },
    {  634050, "Network of the World Ltd (NOW)" },
    {  634060, "Zain Sudan" },
    {  634990, "MTN Sudan" },
    {  635100, "MTN Rwandacell" },
    {  635130, "TIGO RWANDA LTD" },
    {  635140, "AIRTEL RWANDA Ltd" },
    {  635170, "Olleh Rwanda Networks (ORN)" },
    {  636010, "ETH MTN" },
    {  638010, "Evatis" },
    {  639020, "Safaricom Ltd." },
    {  639030, "Kencell Communications Ltd." },
    {  640020, "MIC (T) Ltd." },
    {  640030, "Zantel" },
    {  640040, "Vodacom (T) Ltd." },
    {  640050, "Celtel (T) Ltd." },
    {  641010, "Celtel Uganda" },
    {  641100, "MTN Uganda Ltd." },
    {  641110, "Uganda Telecom Ltd." },
    {  641140, "House of Integrated Technology and Systems Uganda Ltd (HiTs Telecom)" },
    {  641180, "Sure Telecom Uganda Limited" },
    {  641220, "Warid Telecom Uganda Ltd." },
    {  641300, "Anupam Global Soft Uganda Limited" },
    {  641330, "Smile Communications Uganda Limited" },
    {  641400, "Civil Aviation Authority (CAA)" },
    {  641440, "K2 Telecom Ltd" },
    {  641660, "i-Tel Ltd" },
    {  642010, "Econet" },
    {  642020, "Africell" },
    {  642030, "ONAMOB" },
    {  642070, "LACELL" },
    {  642820, "U.COM" },
    {  643010, "T.D.M. GSM" },
    {  643030, "Movitel" },
    {  643040, "VM Sarl" },
    {  645010, "Celtel Zambia Ltd." },
    {  645020, "Telecel Zambia Ltd." },
    {  645030, "Zamtel" },
    {  646010, "Celtel Madagascar (Zain), GSM" },
    {  646020, "Orange Madagascar, GSM" },
    {  646040, "Telecom Malagasy Mobile, GSM" },
    {  647000, "Orange La Reunion" },
    {  647020, "Outremer Telecom" },
    {  647100, "Societe Reunionnaise du Radiotelephone" },
    {  648010, "Net One" },
    {  648030, "Telecel" },
    {  648040, "Econet" },
    {  649010, "Mobile Telecommunications Ltd." },
    {  649020, "Telecom Namibia" },
    {  649030, "Powercom Pty Ltd (leo)" },
    {  650010, "Telekom Network Ltd." },
    {  650100, "Celtel ltd." },
    {  651010, "Vodacom Lesotho (pty) Ltd." },
    {  651020, "Econet Ezin-cel" },
    {  652010, "Mascom Wireless (Pty) Ltd" },
    {  652020, "Orange Botswana (Pty) Ltd" },
    {  652040, "Botswana Telecommunications Corporation (BTC)" },
    {  653010, "SPTC" },
    {  653100, "Swazi MTN" },
    {  654010, "HURI - SNPT" },
    {  655010, "Vodacom (Pty) Ltd." },
    {  655020, "Telkom SA Ltd" },
    {  655040, "Sasol (Pty) Ltd" },
    {  655060, "Sentech (Pty) Ltd." },
    {  655070, "Cell C (Pty) Ltd." },
    {  655100, "Mobile Telephone Networks (MTN) Pty Ltd" },
    {  655110, "SAPS Gauteng" },
    {  655120, "Mobile Telephone Networks (MTN) Pty Ltd" },
    {  655130, "Neotel Pty Ltd" },
    {  655140, "Neotel Pty Ltd" },
    {  655160, "Phoenix Systems Integration (Pty) Ltd" },
    {  655190, "Wireless Business Solutions (iBurst)" },
    {  655210, "Cape Town Metropolitan Council" },
    {  655250, "Wirels Connect" },
    {  655270, "A to Z Vaal Industrial Supplies Pty Ltd" },
    {  655300, "Bokamoso Consortium Pty Ltd" },
    {  655310, "Karabo Telecoms (Pty) Ltd." },
    {  655320, "Ilizwi Telecommunications Pty Ltd" },
    {  655330, "Thinta Thinta Telecommunications Pty Ltd" },
    {  655340, "Bokone Telecoms Pty Ltd" },
    {  655350, "Kingdom Communications Pty Ltd" },
    {  655360, "Amatole Telecommunication Pty Ltd" },
    {  655410, "South African Police Service" },
    {  655500, "Ericsson South Africa (Pty) Ltd" },
    {  655510, "Integrat (Pty) Ltd" },
    {  659120, "Sudani/Sudatel" },
    {  659910, "Zain-South Sudan" },
    {  659920, "MTN-South Sudan" },
    {  659950, "Vivacel/NOW" },
    {  659970, "Gemtel" },
    {  702670, "Belize Telecommunications Ltd., GSM 1900" },
    {  702680, "International Telecommunications Ltd. (INTELCO)" },
    {  704010, "Servicios de Comunicaciones Personales Inalembricas, S.A. (SERCOM, S.A)" },
    {  704020, "Comunicaciones Celulares S.A." },
    {  704030, "Telefonica Centroamerica Guatemala S.A." },
    {  706010, "CTE Telecom Personal, S.A. de C.V." },
    {  706020, "Digicel, S.A. de C.V." },
    {  706030, "Telemovil El Salvador, S.A." },
    {  708001, "Megatel" },
    {  708002, "Celtel" },
    {  708040, "Digicel Honduras" },
    {  710210, "Empresa Nicaraguense de Telecomunicaciones, S.A. (ENITEL)" },
    {  710730, "Servicios de Comunicaciones, S.A. (SERCOM)" },
    {  712010, "Instituto Costarricense de Electricidad - ICE" },
    {  712020, "Instituto Costarricense de Electricidad - ICE" },
    {  712030, "CLARO CR Telecomunicaciones S.A." },
    {  712040, "Telefonica de Costa Rica TC, S.A." },
    {  712200, "Virtualis" },
    {  714010, "Cable & Wireless Panama S.A." },
    {  714020, "BSC de Panama S.A. / Telefonica Moviles Panama S.A." },
    {  714030, "Claro Panama, S.A." },
    {  714040, "Digicel (Panama), S.A." },
    {  716100, "TIM Peru" },
    {  722010, "Compania de Radiocomunicaciones Moviles S.A." },
    {  722020, "Nextel Argentina srl" },
    {  722070, "Telefonica Comunicaciones Personales S.A." },
    {  722310, "CTI PCS S.A." },
    {  722320, "Compania de Telefonos del Interior Norte S.A." },
    {  722330, "Compania de Telefonos del Interior S.A." },
    {  722341, "Telecom Personal S.A." },
    {  724000, "NEXTEL" },
    {  724010, "SISTEER DO BRASIL TELECOMUNICACOES (MVNO)" },
    {  724020, "TIM REGIAO I" },
    {  724030, "TIM REGIAO III" },
    {  724040, "TIM REGIAO III" },
    {  724050, "CLARO" },
    {  724060, "VIVO REGIAO II" },
    {  724100, "VIVO REGIAO III" },
    {  724110, "VIVO REGIAO I" },
    {  724150, "SERCOMTEL" },
    {  724160, "BRT CELULAR" },
    {  724180, "DATORA (MVNO)" },
    {  724230, "TELEMIG CELULAR" },
    {  724240, "AMAZONIA CELULAR" },
    {  724300, "TNL PCS Oi" },
    {  724310, "TNL PCS Oi" },
    {  724320, "CTBC CELULAR R III" },
    {  724330, "CTBC CELULAR R II" },
    {  724340, "CTBC CELULAR R I" },
    {  724350, "TELCOM" },
    {  724360, "OPTIONS" },
    {  724370, "UNICEL" },
    {  724380, "CLARO" },
    {  724390, "NEXTEL (SMP)" },
    {  724540, "PORTO SEGURO TELECOMUNICACOES (MVNO)" },
    {  724990, "LOCAL (STFC)" },
    {  730010, "Entel Telefonica Movil" },
    {  730020, "Telefonica Movil" },
    {  730030, "Smartcom" },
    {  730040, "Centennial Cayman Corp. Chile S.A." },
    {  730050, "Multikom S.A." },
    {  730060, "Blue Two Chile SA" },
    {  730070, "Telefonica Moviles Chile S.A." },
    {  730080, "VTR Movil S.A." },
    {  730090, "Centennial Cayman Corp. Chile S.A." },
    {  730100, "Entel" },
    {  730110, "Celupago S.A." },
    {  730120, "Telestar Movil S.A." },
    {  730130, "TRIBE Mobile Chile SPA" },
    {  730140, "Netline Telefonica Movil Ltda" },
    {  730150, "CIBELES TELECOM S.A." },
    {  732001, "Colombia Telecomunicaciones S.A. - Telecom" },
    {  732002, "Edatel S.A." },
    {  732020, "Emtelsa" },
    {  732099, "Emcali" },
    {  732101, "Comcel S.A. Occel S.A./Celcaribe" },
    {  732102, "Bellsouth Colombia S.A." },
    {  732103, "Colombia Movil S.A." },
    {  732111, "Colombia Movil S.A." },
    {  732123, "Telefonica Moviles Colombia S.A." },
    {  732130, "Avantel" },
    {  734020, "Corporacion Digitel" },
    {  734040, "Telcel, C.A." },
    {  734060, "Telecomunicaciones Movilnet, C.A." },
    {  736010, "Nuevatel S.A." },
    {  736020, "ENTEL S.A." },
    {  736030, "Telecel S.A." },
    {  738002, "Guyana Telephone & Telegraph Company Limited (Cellink)" },
    {  738010, "Cel*Star (Guyana) Inc." },
    {  740000, "Otecel S.A. - Bellsouth" },
    {  740010, "Porta GSM" },
    {  740020, "Telecsa S.A." },
    {  744010, "Hola Paraguay S.A." },
    {  744020, "Hutchison Telecom S.A." },
    {  744030, "Compania Privada de Comunicaciones S.A." },
    {  746020, "Telesur" },
    {  746030, "Digicel" },
    {  746040, "Intelsur" },
    {  746050, "Telesur (CDMA)" },
    {  748000, "Ancel - TDMA" },
    {  748010, "Ancel - GSM" },
    {  748030, "Ancel" },
    {  748070, "Movistar" },
    {  748100, "CTI Movil" },
    {  750001, "Touch" },
    {  901010, "ICO Global Communications" },
    {  901030, "Iridium Communications Inc" },
    {  901050, "Thuraya RMSS Network" },
    {  901060, "Thuraya Satellite Telecommunications Company" },
    {  901100, "Asia Cellular Satellite (AceS)" },
    {  901110, "Inmarsat Ltd." },
    {  901120, "Maritime Communications Partner AS (MCP network)" },
    {  901130, "BebbiCell AG (Formerly Global Networks Switzerland AG)" },
    {  901140, "Telenor" },
    {  901150, "OnAir N.V. (Formerly SITA on behalf of Onair)" },
    {  901160, "Jasper Wireless, Inc" },
    {  901170, "Jersey Telecom" },
    {  901180, "Cingular Wireless" },
    {  901190, "Vodafone Malta (Vodafone Group)" },
    {  901200, "Intermatica" },
    {  901210, "Wins Limited (Formerly Seanet Maritime Communications AB)" },
    {  901220, "MediaLincc Ltd" },
    {  901240, "Voxbone SA" },
    {  901260, "Telecom Italia" },
    {  901270, "OnAir N.V. (Formerly SITA on behalf of Onair)" },
    {  901280, "Vodafone Group" },
    {  901290, "Telenor Connexion AB" },
    {  901310, "France Telecom Orange" },
    {  901320, "MegaFon" },
    {  901330, "Smart Communications , Inc" },
    {  901340, "Tyntec GmbH" },
    {  901350, "Globecomm Network Services" },
    {  901360, "Azerfon LLC" },
    {  901370, "TRANSATEL" },
    {  901380, "Multiregional TransitTelecom (MTT)" },
    {  901390, "MTX Connect" },
    {  901400, "Deutsche Telekom AG" },
    {  901410, "BodyTrace Netherlands B.V." },
    {  901880, "UN Office for the Coordination of Humanitarian Affairs (OCHA)" },
    { 1666665, "Unset" },
    {  0, NULL }
};

static value_string_ext mcc_mnc_codes_ext = VALUE_STRING_EXT_INIT(mcc_mnc_codes);


static int proto_e212   = -1;
static int hf_E212_imsi = -1;
static int hf_E212_mcc  = -1;
static int hf_E212_mcc_lai = -1;
static int hf_E212_mcc_sai = -1;
static int hf_E212_mcc_rai = -1;
static int hf_E212_mnc  = -1;
static int hf_E212_mnc_lai = -1;
static int hf_E212_mnc_sai = -1;
static int hf_E212_mnc_rai = -1;

static int ett_e212_imsi = -1;

static expert_field ei_E212_mcc_non_decimal = EI_INIT;
static expert_field ei_E212_mnc_non_decimal = EI_INIT;


/* static int hf_E212_msin = -1; */

/*
 * MCC/MNC dissection - little endian MNC encoding
 *
 * MNC of length 2:
 *
 *   8   7   6   5   4   3   2   1
 * +---+---+---+---+---+---+---+---+
 * |  MCC digit 2  |  MCC digit 1  |  octet x
 * +---------------+---------------+
 * |    Filler     |  MCC digit 3  |  octet x+1
 * +---------------+---------------+
 * |  MNC digit 2  |  MNC digit 1  |  octet x+2
 * +---------------+---------------+
 *
 * MNC of length 3:
 *
 *   8   7   6   5   4   3   2   1
 * +---+---+---+---+---+---+---+---+
 * |  MCC digit 2  |  MCC digit 1  |  octet x
 * +---------------+---------------+
 * |  MNC digit 3  |  MCC digit 3  |  octet x+1
 * +---------------+---------------+
 * |  MNC digit 2  |  MNC digit 1  |  octet x+2
 * +---------------+---------------+
 *
 *
 * MCC/MNC dissection - big endian MNC encoding
 *
 * MNC of length 2:
 *
 *   8   7   6   5   4   3   2   1
 * +---+---+---+---+---+---+---+---+
 * |  MCC digit 2  |  MCC digit 1  |  octet x
 * +---------------+---------------+
 * |    Filler     |  MCC digit 3  |  octet x+1
 * +---------------+---------------+
 * |  MNC digit 2  |  MNC digit 1  |  octet x+2
 * +---------------+---------------+
 *
 * MNC of length 3:
 *
 *   8   7   6   5   4   3   2   1
 * +---+---+---+---+---+---+---+---+
 * |  MCC digit 2  |  MCC digit 1  |  octet x
 * +---------------+---------------+
 * |  MNC digit 1  |  MCC digit 3  |  octet x+1
 * +---------------+---------------+
 * |  MNC digit 3  |  MNC digit 2  |  octet x+2
 * +---------------+---------------+
 */

/*
 * Return MCC MNC in a packet scope allocated string that can be used in labels.
 */
gchar *
dissect_e212_mcc_mnc_wmem_packet_str(tvbuff_t *tvb, packet_info *pinfo, proto_tree *tree, int offset, e212_number_type_t number_type, gboolean little_endian)
{

    int         start_offset, mcc_mnc;
    guint8      octet;
    guint16     mcc, mnc;
    guint8      mcc1, mcc2, mcc3, mnc1, mnc2, mnc3;
    proto_item *item;
    gchar      *mcc_mnc_str;
    gboolean    long_mnc = FALSE;
    int         hf_E212_mcc_id, hf_E212_mnc_id;

    switch(number_type){
    case E212_LAI:
        hf_E212_mcc_id = hf_E212_mcc_lai;
        hf_E212_mnc_id = hf_E212_mnc_lai;
        break;
    case E212_RAI:
        hf_E212_mcc_id = hf_E212_mcc_rai;
        hf_E212_mnc_id = hf_E212_mnc_rai;
        break;
    case E212_SAI:
        hf_E212_mcc_id = hf_E212_mcc_sai;
        hf_E212_mnc_id = hf_E212_mnc_sai;
        break;
    default:
        hf_E212_mcc_id = hf_E212_mcc;
        hf_E212_mnc_id = hf_E212_mnc;
    }
    start_offset = offset;
    /* MCC + MNC */
    mcc_mnc = tvb_get_ntoh24(tvb,offset);
    /* Mobile country code MCC */
    octet = tvb_get_guint8(tvb,offset);
    mcc1 = octet & 0x0f;
    mcc2 = octet >> 4;
    offset++;
    octet = tvb_get_guint8(tvb,offset);
    mcc3 = octet & 0x0f;
    /* MNC, Mobile network code (octet 3 bits 5 to 8, octet 4)  */
    mnc3 = octet >> 4;
    offset++;
    octet = tvb_get_guint8(tvb,offset);
    mnc1 = octet & 0x0f;
    mnc2 = octet >> 4;

    mcc = 100 * mcc1 + 10 * mcc2 + mcc3;
    mnc = 10 * mnc1 + mnc2;
    if ((mnc3 != 0xf) || (mcc_mnc == 0xffffff)) {
        long_mnc = TRUE;
        if(little_endian)
            mnc = 10 * mnc + mnc3;
        else
            mnc = 100 * mnc3 + mnc;
    }
    item = proto_tree_add_uint(tree, hf_E212_mcc_id , tvb, start_offset, 2, mcc );
    if (((mcc1 > 9) || (mcc2 > 9) || (mcc3 > 9)) && (mcc_mnc != 0xffffff))
        expert_add_info(pinfo, item, &ei_E212_mcc_non_decimal);

    if (long_mnc) {
        item = proto_tree_add_uint_format_value(tree, hf_E212_mnc_id , tvb, start_offset + 1, 2, mnc,
                   "%s (%03u)",
                   val_to_str_ext_const(mcc * 1000 + mnc, &mcc_mnc_codes_ext, "Unknown"),
                   mnc);
        /* Prepare a string with the MCC and MNC including the country and Operator if
         * known, do NOT print unknown.
         */
        mcc_mnc_str = wmem_strdup_printf(wmem_packet_scope(), "MCC %u %s, MNC %03u %s",
            mcc,
            val_to_str_ext_const(mcc,&E212_codes_ext,""),
            mnc,
            val_to_str_ext_const(mcc * 1000 + mnc, &mcc_mnc_codes_ext, ""));
    } else {
        item = proto_tree_add_uint_format_value(tree, hf_E212_mnc , tvb, start_offset + 1, 2, mnc,
                   "%s (%02u)",
                   val_to_str_ext_const(mcc * 1000 + 10 * mnc, &mcc_mnc_codes_ext, "Unknown"),
                   mnc);
        /* Prepare a string with the MCC and MNC including the country and Operator if
         * known, do NOT print unknown.
         */
        mcc_mnc_str = wmem_strdup_printf(wmem_packet_scope(), "MCC %u %s, MNC %02u %s",
            mcc,
            val_to_str_ext_const(mcc,&E212_codes_ext,""),
            mnc,
            val_to_str_ext_const(mcc * 1000 + mnc, &mcc_mnc_codes_ext, ""));
    }

    if (((mnc1 > 9) || (mnc2 > 9) || ((mnc3 > 9) && (mnc3 != 0x0f))) && (mcc_mnc != 0xffffff))
        expert_add_info(pinfo, item, &ei_E212_mnc_non_decimal);

    return mcc_mnc_str;
}

int
dissect_e212_mcc_mnc(tvbuff_t *tvb, packet_info *pinfo, proto_tree *tree, int offset, e212_number_type_t number_type, gboolean little_endian)
{
    dissect_e212_mcc_mnc_wmem_packet_str(tvb, pinfo, tree, offset, number_type, little_endian);
    return offset +3;
}

/*
 * When we want to decode the MCC/MNC pair in an address that is encoded according to E.212
 * the main problem is that we do not know whether we are dealing with a length 2 or length 3
 * MNC. Initially, it was possible to find the length of the MNC by checking the MCC code.
 * Originally each country employed a pure 2 digit or 3 digit MNC scheme. However, it is possible
 * to find countries now that employ both lengths for the MNC (e.g. Canada, India).
 * Since in these cases we can be sure that an MNC cannot possible be a prefix for another MNC, we
 * initially assume that the MNC is 2 digits long. If the MCC/MNC combination gives a match in our
 * mcc_mnc_codes list then we can be sure that we deal with an MNC that is 2 digits long. Otherwise,
 * assume that the MNC is 3 digits long.
 *
 * MNC of length 2:
 *
 *     8   7   6   5   4   3   2   1
 *   +---+---+---+---+---+---+---+---+
 *   |  MCC digit 2  |  MCC digit 1  |  octet x
 *   +---------------+---------------+
 *   |  MNC digit 1  |  MCC digit 3  |  octet x+1
 *   +---------------+---------------+
 *   | addr digit 1  |  MNC digit 2  |  octet x+2
 *   +---------------+---------------+
 *
 * MNC of length 3:
 *
 *     8   7   6   5   4   3   2   1
 *   +---+---+---+---+---+---+---+---+
 *   |  MCC digit 2  |  MCC digit 1  |  octet x
 *   +---------------+---------------+
 *   |  MNC digit 1  |  MCC digit 3  |  octet x+1
 *   +---------------+---------------+
 *   |  MNC digit 3  |  MNC digit 2  |  octet x+2
 *   +---------------+---------------+
 *
 * This function will consume either 2.5 or 3 octets. For this reason it returns
 * the number of nibbles consumed, i.e. 5 or 6 (2 or 3 digits long MNC respectively)
 */
int
dissect_e212_mcc_mnc_in_address(tvbuff_t *tvb, packet_info *pinfo, proto_tree *tree, int offset)
{
    guint32     start_offset, mcc_mnc;
    guint8      octet;
    guint16     mcc, mnc;
    guint8      mcc1, mcc2, mcc3, mnc1, mnc2, mnc3;
    proto_item *item;
    gboolean    long_mnc;

    long_mnc = FALSE;
    start_offset = offset;

    /* MCC + MNC */
    mcc_mnc = tvb_get_ntoh24(tvb,offset);

    /* MCC digits 1 and 2 */
    octet = tvb_get_guint8(tvb,offset);
    mcc1  = octet & 0x0f;
    mcc2  = octet >> 4;
    offset++;

    /* MCC digit 3 and MNC digit 1 */
    octet = tvb_get_guint8(tvb,offset);
    mcc3  = octet & 0x0f;
    mnc1  = octet >> 4;
    offset++;

    /* MNC digits 2 and 3 */
    octet = tvb_get_guint8(tvb,offset);
    mnc2  = octet & 0x0f;
    mnc3  = octet >> 4;

    mcc   = 100 * mcc1 + 10 * mcc2 + mcc3;
    mnc   = 10 * mnc1 + mnc2;

    /* Try to match the MCC and 2 digits MNC with an entry in our list of operators */
    if (!try_val_to_str_ext(mcc * 1000 + 10 * mnc, &mcc_mnc_codes_ext)) {
        mnc = 10 * mnc + mnc3;
        long_mnc = TRUE;
    }

    item = proto_tree_add_uint(tree, hf_E212_mcc , tvb, start_offset, 2, mcc );
    if (((mcc1 > 9) || (mcc2 > 9) || (mcc3 > 9)) & (mcc_mnc != 0xffffff))
        expert_add_info(pinfo, item, &ei_E212_mcc_non_decimal);

    if (long_mnc)
        item = proto_tree_add_uint_format_value(tree, hf_E212_mnc , tvb, start_offset + 1, 2, mnc,
                   "%s (%03u)",
                   val_to_str_ext_const(mcc * 1000 + mnc, &mcc_mnc_codes_ext, "Unknown"),
                   mnc);
    else
        item = proto_tree_add_uint_format_value(tree, hf_E212_mnc , tvb, start_offset + 1, 2, mnc,
                   "%s (%02u)",
                   val_to_str_ext_const(mcc * 1000 + 10 * mnc, &mcc_mnc_codes_ext, "Unknown"),
                   mnc);

    if (((mnc1 > 9) || (mnc2 > 9) || (long_mnc && (mnc3 > 9))) && (mcc_mnc != 0xffffff))
        expert_add_info(pinfo, item, &ei_E212_mnc_non_decimal);

    if (long_mnc)
        return 6;
    else
        return 5;
}

/*
 * MNC of length 2:
 *
 *     8   7   6   5   4   3   2   1
 *   +---+---+---+---+---+---+---+---+
 *   |  MCC digit 1  |  Other data   |  octet x
 *   +---------------+---------------+
 *   |  MNC digit 1  |  MCC digit 2  |  octet x+1
 *   +---------------+---------------+
 *   | MNC digit  3  |  MNC digit 2  |  octet x+2
 *   +---------------+---------------+
 *
 * MNC of length 3:
 *
 *     8   7   6   5   4   3   2   1
 *   +---+---+---+---+---+---+---+---+
 *   |  MCC digit 1  |  Other data   |  octet x
 *   +---------------+---------------+
 *   |  MCC digit 3  |  MCC digit 2  |  octet x+1
 *   +---------------+---------------+
 *   |  MNC digit 2  |  MNC digit 1  |  octet x+2
 *   +---------------+---------------+
 *   |  .....        |  MNC digit 3  |  octet x+3
 *   +---------------+---------------+
 */
static int
dissect_e212_mcc_mnc_high_nibble(tvbuff_t *tvb, packet_info *pinfo _U_, proto_tree *tree, int offset)
{

    guint32     start_offset;
    guint8      octet;
    guint16     mcc, mnc;
    guint8      mcc1, mcc2, mcc3, mnc1, mnc2, mnc3;
    gboolean    long_mnc;

    long_mnc = FALSE;
    start_offset = offset;

    /* MCC digits 1 */
    octet = tvb_get_guint8(tvb,offset);
    mcc1  = octet >> 4;
    offset++;

    /* MCC digits 1 and 2 */
    octet = tvb_get_guint8(tvb,offset);
    mcc2  = octet & 0x0f;
    mcc3  = octet >> 4;
    offset++;

    /* MNC digit 1 and MNC digit 2 */
    octet = tvb_get_guint8(tvb,offset);
    mnc1  = octet & 0x0f;
    mnc2  = octet >> 4;
    offset++;

    /* MNC digits 3 */
    octet = tvb_get_guint8(tvb,offset);
    mnc3  = octet & 0x0f;

    mcc   = 100 * mcc1 + 10 * mcc2 + mcc3;
    mnc   = 10 * mnc1 + mnc2;

    /* Try to match the MCC and 2 digits MNC with an entry in our list of operators */
    if (!try_val_to_str_ext(mcc * 1000 + 10 * mnc, &mcc_mnc_codes_ext)) {
        mnc = 10 * mnc + mnc3;
        long_mnc = TRUE;
    }

    proto_tree_add_uint(tree, hf_E212_mcc , tvb, start_offset, 2, mcc );

    if (long_mnc)
        proto_tree_add_uint_format_value(tree, hf_E212_mnc , tvb, start_offset + 2, 2, mnc,
                   "%s (%03u)",
                   val_to_str_ext_const(mcc * 1000 + mnc, &mcc_mnc_codes_ext, "Unknown"),
                   mnc);
    else
        proto_tree_add_uint_format_value(tree, hf_E212_mnc , tvb, start_offset + 2, 1, mnc,
                   "%s (%02u)",
                   val_to_str_ext_const(mcc * 1000 + 10 * mnc, &mcc_mnc_codes_ext, "Unknown"),
                   mnc);

    if (long_mnc)
        return 7;
    else
        return 5;
}

static int
dissect_e212_mcc_mnc_in_utf8_address(tvbuff_t *tvb, packet_info *pinfo _U_, proto_tree *tree, int offset)
{
    guint16 mcc, mnc;
    gboolean    long_mnc = FALSE;

    mcc = atoi(tvb_get_string_enc(wmem_packet_scope(), tvb, offset, 3, ENC_UTF_8));
    mnc = atoi(tvb_get_string_enc(wmem_packet_scope(), tvb, offset + 3, 2, ENC_UTF_8));

    /* Try to match the MCC and 2 digits MNC with an entry in our list of operators */
    if (!try_val_to_str_ext(mcc * 1000 + 10 * mnc, &mcc_mnc_codes_ext)) {
            mnc = atoi(tvb_get_string_enc(wmem_packet_scope(), tvb, offset + 3, 3, ENC_UTF_8));
            long_mnc = TRUE;
    }

    proto_tree_add_uint(tree, hf_E212_mcc, tvb, offset, 3, mcc );

    if (long_mnc)
        proto_tree_add_uint_format_value(tree, hf_E212_mnc, tvb, offset + 3, 3, mnc,
                   "%s (%03u)",
                   val_to_str_ext_const(mcc * 1000 + mnc, &mcc_mnc_codes_ext, "Unknown1"),
                   mnc);
    else
        proto_tree_add_uint_format_value(tree, hf_E212_mnc, tvb, offset + 3, 2, mnc,
                   "%s (%02u)",
                   val_to_str_ext_const(mcc * 1000 + 10 * mnc, &mcc_mnc_codes_ext, "Unknown2"),
                   mnc);

    if (long_mnc)
        return 6;
    else
        return 5;
}

const gchar *
dissect_e212_imsi(tvbuff_t *tvb, packet_info *pinfo, proto_tree *tree, int offset, int length, gboolean skip_first)
{
    proto_item *item;
    proto_tree *subtree;
    const gchar *imsi_str;

    /* Fetch the BCD encoded digits from tvb indicated half byte, formating the digits according to
     * a default digit set of 0-9 returning "?" for overdecadic digits a pointer to the wmem
     * allocated string will be returned.
     */
    imsi_str = tvb_bcd_dig_to_wmem_packet_str( tvb, offset, length, NULL, skip_first);
    item = proto_tree_add_string(tree, hf_E212_imsi, tvb, offset, length, imsi_str);

    subtree = proto_item_add_subtree(item, ett_e212_imsi);

    if(skip_first) {
        dissect_e212_mcc_mnc_high_nibble(tvb, pinfo, subtree, offset);
    } else {
        dissect_e212_mcc_mnc_in_address(tvb, pinfo, subtree, offset);
    }

    return imsi_str;
}

const gchar *
dissect_e212_utf8_imsi(tvbuff_t *tvb, packet_info *pinfo, proto_tree *tree, int offset, int length)
{
    proto_item *item;
    proto_tree *subtree;
    const gchar *imsi_str;

    /* Fetch the UTF8-encoded IMSI */
    imsi_str = tvb_get_string_enc(wmem_packet_scope(), tvb, offset, length, ENC_UTF_8);
    item = proto_tree_add_string(tree, hf_E212_imsi, tvb, offset, length, imsi_str);

    subtree = proto_item_add_subtree(item, ett_e212_imsi);

    dissect_e212_mcc_mnc_in_utf8_address(tvb, pinfo, subtree, offset);

    return imsi_str;
}

/*
 * Register the protocol with Wireshark.
 *
 * This format is required because a script is used to build the C function
 * that calls all the protocol registration.
 */


void
proto_register_e212(void)
{

/* Setup list of header fields  See Section 1.6.1 for details */
    static hf_register_info hf[] = {
    { &hf_E212_imsi,
        { "IMSI","e212.imsi",
        FT_STRING, BASE_NONE, NULL, 0x0,
        "International mobile subscriber identity(IMSI)", HFILL }
    },
    { &hf_E212_mcc,
        { "Mobile Country Code (MCC)","e212.mcc",
        FT_UINT16, BASE_DEC|BASE_EXT_STRING, &E212_codes_ext, 0x0,
        "Mobile Country Code MCC", HFILL }
    },
    { &hf_E212_mcc_lai,
        { "Mobile Country Code (MCC)","e212.lai.mcc",
        FT_UINT16, BASE_DEC|BASE_EXT_STRING, &E212_codes_ext, 0x0,
        "Mobile Country Code MCC", HFILL }
    },
    { &hf_E212_mcc_rai,
        { "Mobile Country Code (MCC)","e212.rai.mcc",
        FT_UINT16, BASE_DEC|BASE_EXT_STRING, &E212_codes_ext, 0x0,
        "Mobile Country Code MCC", HFILL }
    },
    { &hf_E212_mcc_sai,
        { "Mobile Country Code (MCC)","e212.sai.mcc",
        FT_UINT16, BASE_DEC|BASE_EXT_STRING, &E212_codes_ext, 0x0,
        "Mobile Country Code MCC", HFILL }
    },
    { &hf_E212_mnc,
        { "Mobile Network Code (MNC)","e212.mnc",
        FT_UINT16, BASE_DEC, NULL, 0x0,
        "Mobile network code", HFILL }
    },
    { &hf_E212_mnc_lai,
        { "Mobile Network Code (MNC)","e212.lai.mnc",
        FT_UINT16, BASE_DEC, NULL, 0x0,
        "Mobile network code", HFILL }
    },
    { &hf_E212_mnc_rai,
        { "Mobile Network Code (MNC)","e212.rai.mnc",
        FT_UINT16, BASE_DEC, NULL, 0x0,
        "Mobile network code", HFILL }
    },
    { &hf_E212_mnc_sai,
        { "Mobile Network Code (MNC)","e212.sai.mnc",
        FT_UINT16, BASE_DEC, NULL, 0x0,
        "Mobile network code", HFILL }
    },
#if 0
    { &hf_E212_msin,
        { "Mobile Subscriber Identification Number (MSIN)", "e212.msin",
        FT_STRING, BASE_NONE, NULL, 0,
        "Mobile Subscriber Identification Number(MSIN)", HFILL }},
#endif
    };


    static gint *ett_e212_array[] = {
        &ett_e212_imsi,
    };

    static ei_register_info ei[] = {
        { &ei_E212_mcc_non_decimal, { "e212.mcc.non_decimal", PI_MALFORMED, PI_WARN, "MCC contains non-decimal digits", EXPFILL }},
        { &ei_E212_mnc_non_decimal, { "e212.mnc.non_decimal", PI_MALFORMED, PI_WARN, "MNC contains non-decimal digits", EXPFILL }},
    };

    expert_module_t* expert_e212;

    /*
     * Register the protocol name and description
     */
    proto_e212 = proto_register_protocol(
            "ITU-T E.212 number",
            "E.212",
            "e212");

    /*
     * Required function calls to register
     * the header fields and subtrees used.
     */
    proto_register_field_array(proto_e212, hf, array_length(hf));
    proto_register_subtree_array(ett_e212_array, array_length(ett_e212_array));
    expert_e212 = expert_register_protocol(proto_e212);
    expert_register_field_array(expert_e212, ei, array_length(ei));

}

/*
 * Editor modelines  -  http://www.wireshark.org/tools/modelines.html
 *
 * Local variables:
 * c-basic-offset: 4
 * tab-width: 8
 * indent-tabs-mode: nil
 * End:
 *
 * vi: set shiftwidth=4 tabstop=8 expandtab:
 * :indentSize=4:tabSize=8:noTabs=true:
 */
