/* packet-e212.c
 * Routines for output and filtering of E.212 numbers common
 * to many dissectors.
 * Copyright 2006, Anders Broman <anders.broman@ericsson.com>
 *
 * $Id: packet-e212.c 32508 2010-04-18 19:34:57Z dimeg $
 *
 * Wireshark - Network traffic analyzer
 * By Gerald Combs <gerald@wireshark.org>
 * Copyright 1998 Gerald Combs
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */

#ifdef HAVE_CONFIG_H
# include "config.h"
#endif

#include <glib.h>
#include <epan/packet.h>

#include "packet-e212.h"
#include "expert.h"


/*
 * Annex to ITU Operational Bulletin
 * No. 953 - 1.IV.2010
 * including amendments up to ITU Operational Bulletin No. 954 - 15.IV.2010
 *
 * COMPLEMENT TO ITU-T RECOMMENDATION E.212 (05/2008)
 */
const value_string E212_codes[] = {
	{  202,	"Greece" },
	{  204,	"Netherlands (Kingdom of the)" },
	{  206,	"Belgium" },
	{  208,	"France" },
	{  212,	"Monaco (Principality of)" },
	{  213,	"Andorra (Principality of)" },
	{  214,	"Spain" },
	{  216,	"Hungary (Republic of)" },
	{  218,	"Bosnia and Herzegovina" },
	{  219,	"Croatia (Republic of)" },
	{  220,	"Serbia (Republic of)" },
	{  222,	"Italy" },
	{  225,	"Vatican City State" },
	{  226,	"Romania" },
	{  228,	"Switzerland (Confederation of)" },
	{  230,	"Czech Republic" },
	{  231,	"Slovak Republic" },
	{  232,	"Austria" },
	{  234,	"United Kingdom of Great Britain and Northern Ireland" },
	{  235,	"United Kingdom of Great Britain and Northern Ireland" },
	{  238,	"Denmark" },
	{  240,	"Sweden" },
	{  242,	"Norway" },
	{  244,	"Finland" },
	{  246,	"Lithuania (Republic of)" },
	{  247,	"Latvia (Republic of)" },
	{  248,	"Estonia (Republic of)" },
	{  250,	"Russian Federation" },
	{  255,	"Ukraine" },
	{  257,	"Belarus (Republic of)" },
	{  259,	"Moldova (Republic of)" },
	{  260,	"Poland (Republic of)" },
	{  262,	"Germany (Federal Republic of)" },
	{  266,	"Gibraltar" },
	{  268,	"Portugal" },
	{  270,	"Luxembourg" },
	{  272,	"Ireland" },
	{  274,	"Iceland" },
	{  276,	"Albania (Republic of)" },
	{  278,	"Malta" },
	{  280,	"Cyprus (Republic of)" },
	{  282,	"Georgia" },
	{  283,	"Armenia (Republic of)" },
	{  284,	"Bulgaria (Republic of)" },
	{  286,	"Turkey" },
	{  288,	"Faroe Islands" },
	{  290,	"Greenland (Denmark)" },
	{  292,	"San Marino (Republic of)" },
	{  293,	"Slovenia (Republic of)" },
	{  294,	"The Former Yugoslav Republic of Macedonia" },
	{  295,	"Liechtenstein (Principality of)" },
	{  297, "Montenegro (Republic of)" },
	{  302,	"Canada" },
	{  308,	"Saint Pierre and Miquelon (Collectivite territoriale de la Republique francaise)" },
	{  310,	"United States of America" },
	{  311,	"United States of America" },
	{  312,	"United States of America" },
	{  313,	"United States of America" },
	{  314,	"United States of America" },
	{  315,	"United States of America" },
	{  316,	"United States of America" },
	{  330,	"Puerto Rico" },
	{  332,	"United States Virgin Islands" },
	{  334,	"Mexico" },
	{  338,	"Jamaica" },
	{  340,	"Martinique/Guadeloupe (French Departments of)" },
	{  342,	"Barbados" },
	{  344,	"Antigua and Barbuda" },
	{  346,	"Cayman Islands" },
	{  348,	"British Virgin Islands" },
	{  350,	"Bermuda" },
	{  352,	"Grenada" },
	{  354,	"Montserrat" },
	{  356,	"Saint Kitts and Nevis" },
	{  358,	"Saint Lucia" },
	{  360,	"Saint Vincent and the Grenadines" },
	{  362,	"Netherlands Antilles" },
	{  363,	"Aruba" },
	{  364,	"Bahamas (Commonwealth of the)" },
	{  365,	"Anguilla" },
	{  366,	"Dominica (Commonwealth of)" },
	{  368,	"Cuba" },
	{  370,	"Dominican Republic" },
	{  372,	"Haiti (Republic of)" },
	{  374,	"Trinidad and Tobago" },
	{  376,	"Turks and Caicos Islands" },
	{  400,	"Azerbaijani Republic" },
	{  401,	"Kazakhstan (Republic of)" },
	{  402,	"Bhutan (Kingdom of)" },
	{  404,	"India (Republic of)" },
	{  405,	"India (Republic of)" },
	{  406,	"India (Republic of)" },
	{  410,	"Pakistan (Islamic Republic of)" },
	{  412,	"Afghanistan" },
	{  413,	"Sri Lanka (Democratic Socialist Republic of)" },
	{  414,	"Myanmar (Union of)" },
	{  415,	"Lebanon" },
	{  416,	"Jordan (Hashemite Kingdom of)" },
	{  417,	"Syrian Arab Republic" },
	{  418,	"Iraq (Republic of)" },
	{  419,	"Kuwait (State of)" },
	{  420,	"Saudi Arabia (Kingdom of)" },
	{  421,	"Yemen (Republic of)" },
	{  422,	"Oman (Sultanate of)" },
	{  424,	"United Arab Emirates" },
	{  425,	"Israel (State of)" },
	{  426,	"Bahrain (Kingdom of)" },
	{  427,	"Qatar (State of)" },
	{  428,	"Mongolia" },
	{  429,	"Nepal" },
	{  430,	"United Arab Emirates (Abu Dhabi)" },
	{  431,	"United Arab Emirates (Dubai)" },
	{  432,	"Iran (Islamic Republic of)" },
	{  434,	"Uzbekistan (Republic of)" },
	{  436,	"Tajikistan (Republic of)" },
	{  437,	"Kyrgyz Republic" },
	{  438,	"Turkmenistan" },
	{  440,	"Japan" },
	{  441,	"Japan" },
	{  450,	"Korea (Republic of)" },
	{  452,	"Viet Nam (Socialist Republic of)" },
	{  454,	"Hongkong, China" },
	{  455,	"Macao, China" },
	{  456,	"Cambodia (Kingdom of)" },
	{  457,	"Lao People's Democratic Republic" },
	{  460,	"China (People's Republic of)" },
	{  461,	"China (People's Republic of)" },
	{  466,	"Taiwan, China" },
	{  467,	"Democratic People's Republic of Korea" },
	{  470,	"Bangladesh (People's Republic of)" },
	{  472,	"Maldives (Republic of)" },
	{  502,	"Malaysia" },
	{  505,	"Australia" },
	{  510,	"Indonesia (Republic of)" },
	{  514,	"Democratic Republic of Timor-Leste" },
	{  515,	"Philippines (Republic of the)" },
	{  520,	"Thailand" },
	{  525,	"Singapore (Republic of)" },
	{  528,	"Brunei Darussalam" },
	{  530,	"New Zealand" },
	{  534,	"Previously assigned to Northern Mariana Islands (Commonwealth of the)" },
	{  535,	"Previously assigned to Guam" },
	{  536,	"Nauru (Republic of)" },
	{  537,	"Papua New Guinea" },
	{  539,	"Tonga (Kingdom of)" },
	{  540,	"Solomon Islands" },
	{  541,	"Vanuatu (Republic of)" },
	{  542,	"Fiji (Republic of)" },
	{  543,	"Wallis and Futuna (Territoire francais d'outre-mer)" },
	{  544,	"American Samoa" },
	{  545,	"Kiribati (Republic of)" },
	{  546,	"New Caledonia (Territoire francais d'outre-mer)" },
	{  547,	"French Polynesia (Territoire francais d'outre-mer)" },
	{  548,	"Cook Islands" },
	{  549,	"Samoa (Independent State of)" },
	{  550,	"Micronesia (Federated States of)" },
	{  551,	"Marshall Islands (Republic of the)" },
	{  552,	"Palau (Republic of)" },
	{  602,	"Egypt (Arab Republic of)" },
	{  603,	"Algeria (People's Democratic Republic of)" },
	{  604,	"Morocco (Kingdom of)" },
	{  605,	"Tunisia" },
	{  606,	"Libya (Socialist People's Libyan Arab Jamahiriya)" },
	{  607,	"Gambia (Republic of the)" },
	{  608,	"Senegal (Republic of)" },
	{  609,	"Mauritania (Islamic Republic of)" },
	{  610,	"Mali (Republic of)" },
	{  611,	"Guinea (Republic of)" },
	{  612,	"Cote d'Ivoire (Republic of)" },
	{  613,	"Burkina Faso" },
	{  614,	"Niger (Republic of the)" },
	{  615,	"Togolese Republic" },
	{  616,	"Benin (Republic of)" },
	{  617,	"Mauritius (Republic of)" },
	{  618,	"Liberia (Republic of)" },
	{  619,	"Sierra Leone" },
	{  620,	"Ghana" },
	{  621,	"Nigeria (Federal Republic of)" },
	{  622,	"Chad (Republic of)" },
	{  623,	"Central African Republic" },
	{  624,	"Cameroon (Republic of)" },
	{  625,	"Cape Verde (Republic of)" },
	{  626,	"Sao Tome and Principe (Democratic Republic of)" },
	{  627,	"Equatorial Guinea (Republic of)" },
	{  628,	"Gabonese Republic" },
	{  629,	"Congo (Republic of the)" },
	{  630,	"Democratic Republic of the Congo" },
	{  631,	"Angola (Republic of)" },
	{  632,	"Guinea-Bissau (Republic of)" },
	{  633,	"Seychelles (Republic of)" },
	{  634,	"Sudan (Republic of the)" },
	{  635,	"Rwanda (Republic of)" },
	{  636,	"Ethiopia (Federal Democratic Republic of)" },
	{  637,	"Somali Democratic Republic" },
	{  638,	"Djibouti (Republic of)" },
	{  639,	"Kenya (Republic of)" },
	{  640,	"Tanzania (United Republic of)" },
	{  641,	"Uganda (Republic of)" },
	{  642,	"Burundi (Republic of)" },
	{  643,	"Mozambique (Republic of)" },
	{  645,	"Zambia (Republic of)" },
	{  646,	"Madagascar (Republic of)" },
	{  647,	"French Departments and Territories in the Indian Ocean" },
	{  648,	"Zimbabwe (Republic of)" },
	{  649,	"Namibia (Republic of)" },
	{  650,	"Malawi" },
	{  651,	"Lesotho (Kingdom of)" },
	{  652,	"Botswana (Republic of)" },
	{  653,	"Swaziland (Kingdom of)" },
	{  654,	"Comoros (Union of the)" },
	{  655,	"South Africa (Republic of)" },
	{  657,	"Eritrea" },
	{  702,	"Belize" },
	{  704,	"Guatemala (Republic of)" },
	{  706,	"El Salvador (Republic of)" },
	{  708,	"Honduras (Republic of)" },
	{  710,	"Nicaragua" },
	{  712,	"Costa Rica" },
	{  714,	"Panama (Republic of)" },
	{  716,	"Peru" },
	{  722,	"Argentine Republic" },
	{  724,	"Brazil (Federative Republic of)" },
	{  730,	"Chile" },
	{  732,	"Colombia (Republic of)" },
	{  734,	"Venezuela (Bolivarian Republic of)" },
	{  736,	"Bolivia (Republic of)" },
	{  738,	"Guyana" },
	{  740,	"Ecuador" },
	{  742,	"French Guiana (French Department of)" },
	{  744,	"Paraguay (Republic of)" },
	{  746,	"Suriname (Republic of)" },
	{  748,	"Uruguay (Eastern Republic of)" },
	{  750, "Falkland Islands (Malvinas)" },
	{  901,	"International Mobile, shared code" },
	{ 0, NULL }
};

/*
 * Annex to ITU Operational Bulletin No. 932 - 15.V.2009
 * including amendments up to ITU Operational Bulletin No. 954 - 15.IV.2010
 * 
 * According to Recommendation ITU-T E.212 (05/2008)
 *
 * The value parts of the list below are equal to:
 *   1000 * MCC + 10 * MNC,  in case of 2 digits long MNCs
 *   1000 * MCC + MNC,       in case of 3 digits long MNCs
 * with this encoding, we are sure that even in cases of mixed length
 * MNCs under the same MCC, it is impossible for 2 MNCs to get mapped to
 * the same entry. Aliasing to the same entry in the list below occurs 
 * only in case of an invalid combination of MNCs, i.e. in the same MCC a
 * 2 digits long MNC is a prefix of a 3 digits long MNC
 */
const value_string mcc_mnc_codes[] = {
	{  202010, "Cosmote" },
	{  202050, "Vodafone - Panafon" },
	{  202090, "Info Quest S.A." },
	{  202100, "WIND Hellas" },
	{  204020, "Tele2 (Netherlands) B.V." },
	{  204030, "Blyk N.V." },
	{  204040, "Vodafone Libertel N.V." },
	{  204050, "Elephant Talk Comm. Premium Rate Serv. Neth. B.V." },
	{  204060, "Barablu Mobile Benelux Ltd" },
	{  204070, "Teleena holding B.V." },
	{  204080, "KPN Mobile The Netherlands B.V." },
	{  204100, "KPN B.V." },
	{  204120, "Telfort B.V." },
	{  204140, "INMO B.V." },
	{  204160, "T-Mobile Netherlands B.V." },
	{  204180, "Telfort B.V." },
	{  204200, "Orange Nederland N.V." },
	{  204210, "ProRail B.V." },
	{  204600, "KPN B.V." },
	{  204690, "KPN Mobile The Netherlands B.V." },
	{  206010, "Proximus" },
	{  206100, "Mobistar" },
	{  206200, "Base" },
	{  208010, "Orange France" },
	{  208020, "Orange France" },
	{  208050, "Globalstar Europe" },
	{  208060, "Globalstar Europe" },
	{  208070, "Globalstar Europe" },
	{  208100, "S.F.R." },
	{  208110, "S.F.R. (UMTS)" },
	{  208130, "SFR (Zones Blanches)" },
	{  208200, "Bouygues Telecom" },
	{  208210, "Bouygues Telecom" },
	{  208220, "Transatel" },
	{  208880, "Bouygues Telecom (Zones Blanches)" },
	{  213030, "Mobiland" },
	{  214010, "Vodafone Espana, S.A.U." },
	{  214030, "France Telecom Espana, S.A." },
	{  214040, "Xfera Moviles, S.A." },
	{  214050, "Telefonica Moviles Espana, S.A.U." },
	{  214060, "Vodafone Espana, S.A.U." },
	{  214070, "Telefonica Moviles Espana, S.A.U." },
	{  214080, "Euskaltel, S.A." },
	{  214090, "France Telecom Espana, S.A." },
	{  214150, "BT Espana Compania de Servicios Globales de Telecomunicaciones, S.A.U." },
	{  214160, "Telecable de Asturias, S.A.U." },
	{  214170, "R Cable y Telecomunicaciones Galicia, S.A." },
	{  214180, "Cableuropa, S.A.U." },
	{  214190, "E-plus Moviles, S.L." },
	{  214200, "Fonyou Telecom, S.L." },
	{  214210, "Jazz Telecom, S.A.U." },
	{  214220, "Best Spain Telecom, S.L." },
	{  214230, "Barablu Movil Espana, S.L.U." },
	{  214240, "Vizzavi Espana, S.L." },
	{  214250, "Lycamobile, S.L." },
	{  214260, "Lleida Networks Serveis Telematics, S.L." },
	{  216010, "Pannon GSM" },
	{  216300, "Westel Mobile" },
	{  216700, "Vodafone" },
	{  218030, "Eronet Mobile Communications Ltd." },
	{  218050, "MOBI'S (Mobilina Srpske)" },
	{  218900, "GSMBIH" },
	{  219010, "T-Mobile Hrvatska d.o.o./T-Mobile Croatia LLC" },
	{  219020, "Tele2/Tele2 d.o.o." },
	{  219100, "VIPnet/VIPnet d.o.o." },
	{  220010, "Telenor d.o.o." },
	{  220030, "Telekom Srbija a.d." },
	{  220050, "Vip mobile d.o.o." },
	{  222010, "Telecom Italia Mobile (TIM)" },
	{  222020, "Elsacom" },
	{  222100, "Omnitel Pronto Italia (OPI)" },
	{  222770, "IPSE 2000" },
	{  222880, "Wind" },
	{  222980, "Blu" },
	{  222990, "H3G" },
	{  226010, "Vodafone Romania SA" },
	{  226020, "Romtelecom (CDMA)" },
	{  226030, "Cosmorom" },
	{  226100, "Orange Romania" },
	{  228010, "Swisscom Mobile AG" },
	{  228020, "TDC Switzerland AG" },
	{  228030, "Orange Communications SA" },
	{  228050, "Comfone AG" },
	{  228060, "SBB AG" },
	{  228070, "IN&Phone SA" },
	{  228080, "Tele2 Telecommunications AG" },
	{  228120, "TDC Switzerland AG" },
	{  228510, "Bebbicell AG" },
	{  230010, "T-Mobile Czech Republic a.s." },
	{  230020, "Telefonica O2 Czech Republic a.s." },
	{  230030, "Vodafone Czech Republic a.s." },
	{  230040, "Mobilkom a.s." },
	{  230980, "Sprava Zeleznicni Dopravni Cesty" },
	{  230990, "Vodafone Czech Republic a.s. R&D Centre" },
	{  231010, "Orange, GSM" },
	{  231020, "Eurotel, GSM & NMT" },
	{  231040, "Eurotel, UMTS" },
	{  231050, "Orange, UMTS" },
	{  232010, "Mobilkom Austria Aktiengesellschaft" },
	{  232020, "Mobilkom Austria Aktiengesellschaft" },
	{  232030, "T-Mobile Austria GmbH" },
	{  232040, "T-Mobile Austria GmbH" },
	{  232050, "Orange Austria Telecommunication GmbH" },
	{  232060, "Orange Austria Telecommunication GmbH" },
	{  232070, "T-Mobile Austria GmbH" },
	{  232090, "Mobilkom Austria Aktiengesellschaft" },
	{  232100, "Hutchison 3G Austria GmbH" },
	{  232110, "Mobilkom Austria Aktiengesellschaft" },
	{  232120, "Orange Austria Telecommunication GmbH" },
	{  232140, "Hutchison 3G Austria GmbH" },
	{  232150, "Barablu Mobile Austria Ltd" },
	{  232910, "OBB - Infrastruktur Bau AG" },
	{  234000, "British Telecom" },
	{  234010, "Mapesbury Communications Ltd" },
	{  234020, "O2 UK Ltd" },
	{  234030, "Jersey Airtel Ltd" },
	{  234040, "FMS Solutions Ltd" },
	{  234050, "COLT Mobile Telecommunications Ltd" },
	{  234060, "Internet Computer Bureau Ltd" },
	{  234070, "Cable & Wireless PLC" },
	{  234080, "OnePhone (UK) Ltd" },
	{  234090, "Wire9 Telecom PLC" },
	{  234120, "Network Rail Infrastructure Ltd" },
	{  234130, "Network Rail Infrastructure Ltd" },
	{  234140, "Hay Systems Ltd" },
	{  234160, "Opal Telecom Limited" },
	{  234170, "FleXtel Limited" },
	{  234180, "Wire9 Telecom PLC" },
	{  234190, "TeleWare PLC" },
	{  234200, "Hutchison 3G UK Ltd" },
	{  234210, "LogicStar Ltd" },
	{  234220, "Routo Telecommunications Limited" },
	{  234230, "Vectone Network Limited" },
	{  234240, "Stour Marine Limited" },
	{  234250, "Software Cellular Network Limited" },
	{  234320, "T-Mobile (UK) Limited" },
	{  234760, "British Telecom" },
	{  234780, "Airwave Solutions" },
	{  235770, "British Telecom" },
	{  235910, "Vodafone Ltd" },
	{  235920, "Cable & Wireless PLC" },
	{  235940, "Hutchison 3G UK Ltd" },
	{  235950, "Network Rail Infrastructure Ltd" },
	{  238010, "TDC Mobil" },
	{  238020, "Sonofon" },
	{  238030, "MIGway A/S" },
	{  238060, "Hi3G" },
	{  238070, "Barablu Mobile Ltd." },
	{  238100, "TDC Mobil" },
	{  238120, "Lycamobile Denmark" },
	{  238200, "Telia" },
	{  238300, "Telia Mobile" },
	{  238770, "Tele2" },
	{  240010, "TeliaSonera Sverige AB" },
	{  240020, "HI3G Access AB" },
	{  240030, "AINMT Sverige AB" },
	{  240040, "3G Infrastructure Services AB" },
	{  240050, "Svenska UMTS-Nat AB" },
	{  240060, "Telenor Sverige AB" },
	{  240070, "Tele2 Sverige AB" },
	{  240080, "Telenor Sverige AB" },
	{  240090, "Djuice Mobile Sweden, filial till Telenor Mobile Sweden AS" },
	{  240100, "Spring Mobil AB" },
	{  240110, "Linholmen Science Park AB" },
	{  240120, "Barablu Mobile Scandinavia Ltd" },
	{  240130, "Ventelo Sverige AB" },
	{  240140, "TDC Mobil A/S" },
	{  240150, "Wireless Maingate Nordic AB" },
	{  240160, "42IT AB" },
	{  240170, "Gotalandsnatet AB" },
	{  240180, "Generic Mobile Systems Sweden AB" },
	{  240190, "Mundio Mobile Sweden Limited" },
	{  240200, "iMEZ AB" },
	{  240210, "Banverket" },
	{  240220, "EuTel AB" },
	{  240240, "Net4Mobility HB" },
	{  242010, "Telenor Mobil AS" },
	{  242020, "Netcom GSM AS" },
	{  242030, "Teletopia Mobile Communications AS" },
	{  242040, "Tele2 Norge AS" },
	{  244040, "Finnet Networks Ltd." },
	{  244050, "Elisa Matkapuhelinpalvelut Ltd." },
	{  244090, "Finnet Group" },
	{  244120, "Finnet Networks Ltd." },
	{  244140, "Alands Mobiltelefon AB" },
	{  244160, "Oy Finland Tele2 AB" },
	{  244210, "Saunalahti Group Ltd." },
	{  244910, "Sonera Carrier Networks Oy" },
	{  246010, "Omnitel" },
	{  246020, "Bite GSM" },
	{  246030, "Tele2" },
	{  247010, "Latvijas Mobilais Telefons SIA" },
	{  247020, "Tele2" },
	{  247030, "Telekom Baltija" },
	{  247040, "Beta Telecom" },
	{  247050, "Bite Mobile" },
	{  247060, "Rigatta" },
	{  247070, "Master Telecom" },
	{  247080, "IZZI" },
	{  248010, "EMT GSM" },
	{  248020, "RLE" },
	{  248030, "Tele2" },
	{  248040, "OY Top Connect" },
	{  248050, "AS Bravocom Mobiil" },
	{  248060, "ProGroup Holding OY" },
	{  248070, "Televorgu AS" },
	{  248710, "Siseministeerium (Ministry of Interior)" },
	{  250010, "Mobile Telesystems" },
	{  250020, "Megafon" },
	{  250030, "Nizhegorodskaya Cellular Communications" },
	{  250040, "Sibchallenge" },
	{  250050, "Mobile Comms System" },
	{  250070, "BM Telecom" },
	{  250100, "Don Telecom" },
	{  250110, "Orensot" },
	{  250120, "Baykal Westcom" },
	{  250130, "Kuban GSM" },
	{  250160, "New Telephone Company" },
	{  250170, "Ermak RMS" },
	{  250190, "Volgograd Mobile" },
	{  250200, "ECC" },
	{  250280, "Extel" },
	{  250390, "Uralsvyazinform" },
	{  250440, "Stuvtelesot" },
	{  250920, "Printelefone" },
	{  250930, "Telecom XXI" },
	{  250990, "Bec Line GSM" },
	{  255010, "Ukrainian Mobile Communication, UMC" },
	{  255020, "Ukranian Radio Systems, URS" },
	{  255030, "Kyivstar GSM" },
	{  255040, "International Telecommunications Ltd." },
	{  255050, "Golden Telecom" },
	{  255060, "Astelit" },
	{  255070, "Ukrtelecom" },
	{  255210, "CJSC - Telesystems of Ukraine" },
	{  257010, "MDC Velcom" },
	{  257020, "MTS" },
	{  259010, "Orange Moldova GSM" },
	{  259020, "Moldcell GSM" },
	{  259040, "Eventis Mobile GSM" },
	{  260010, "Plus GSM (Polkomtel S.A.)" },
	{  260020, "ERA GSM (Polska Telefonia Cyfrowa Sp. Z.o.o.)" },
	{  260030, "Idea (Polska Telefonia Komorkowa Centertel Sp. Z.o.o)" },
	{  260040, "Tele2 Polska (Tele2 Polska Sp. Z.o.o.)" },
	{  260050, "IDEA (UMTS)/PTK Centertel sp. Z.o.o." },
	{  260060, "Netia Mobile" },
	{  260070, "Premium internet" },
	{  260080, "E-Telko" },
	{  260090, "Telekomunikacja Kolejowa (GSM-R)" },
	{  260100, "Telefony Opalenickie" },
	{  262010, "T-Mobile Deutschland GmbH" },
	{  262020, "Vodafone D2 GmbH" },
	{  262030, "E-Plus Mobilfunk GmbH & Co. KG" },
	{  262040, "Vodafone D2 GmbH" },
	{  262050, "E-Plus Mobilfunk GmbH & Co. KG" },
	{  262060, "T-Mobile Deutschland GmbH" },
	{  262070, "O2 (Germany) GmbH & Co. OHG" },
	{  262080, "O2 (Germany) GmbH & Co. OHG" },
	{  262090, "Vodafone D2 GmbH" },
	{  262100, "Arcor AG & Co." },
	{  262110, "O2 (Germany) GmbH & Co. OHG" },
	{  262120, "Dolphin Telecom (Deutschland) GmbH" },
	{  262130, "Mobilcom Multimedia GmbH" },
	{  262140, "Group 3G UMTS GmbH (Quam)" },
	{  262150, "Airdata AG" },
	{  262760, "Siemens AG, ICMNPGUSTA" },
	{  262770, "E-Plus Mobilfunk GmbH & Co. KG" },
	{  266010, "Gibtelecom GSM" },
	{  266060, "CTS" },
	{  266090, "Cloud9 Mobile Communications" },
	{  268010, "Vodafone Telecel - Comunicacoes Pessoais, S.A." },
	{  268030, "Optimus - Telecomunicacoes, S.A." },
	{  268050, "Oniway - Inforcomunicacoes, S.A." },
	{  268060, "TMN - Telecomunicacoes Moveis Nacionais, S.A." },
	{  270010, "P&T Luxembourg" },
	{  270770, "Tango" },
	{  270990, "Voxmobile S.A." },
	{  272010, "Vodafone Ireland Plc" },
	{  272020, "Digifone mm02 Ltd." },
	{  272030, "Meteor Mobile Communications Ltd." },
	{  272070, "Eircom" },
	{  272090, "Clever Communications Ltd." },
	{  274010, "Iceland Telecom Ltd." },
	{  274020, "Og fjarskipti hf (Vodafone Iceland)" },
	{  274030, "Og fjarskipti hf (Vodafone Iceland)" },
	{  274040, "IMC Islande ehf" },
	{  274070, "IceCell ehf" },
	{  276010, "AMC" },
	{  276020, "Vodafone Albania" },
	{  276030, "Eagle Mobile" },
	{  276040, "Mobile 4 AL" },
	{  278010, "Vodafone Malta" },
	{  278210, "go mobile" },
	{  278770, "3G Telecommunications Ltd" },
	{  280010, "CYTA" },
	{  280100, "Scancom (Cyprus) Ltd." },
	{  282010, "Geocell Ltd." },
	{  282020, "Magti GSM Ltd." },
	{  282030, "Iberiatel Ltd." },
	{  282040, "Mobitel Ltd." },
	{  282050, "United Telecom" },
	{  284010, "M-Tel GSM BG" },
	{  284050, "Globul" },
	{  286010, "Turkcell" },
	{  286020, "Telsim GSM" },
	{  286030, "Aria" },
	{  286040, "Aycell" },
	{  288010, "Faroese Telecom - GSM" },
	{  288020, "Kall GSM" },
	{  290010, "Tele Greenland" },
	{  292010, "SMT - San Marino Telecom" },
	{  293400, "SI Mobil" },
	{  293410, "Mobitel" },
	{  293640, "T-2 d.o.o." },
	{  293700, "Tusmobil d.o.o." },
	{  294010, "T-Mobile" },
	{  294020, "Cosmofon" },
	{  294030, "Nov Operator" },
	{  295010, "Telecom FL AG" },
	{  295020, "Viag Europlatform AG" },
	{  295050, "Mobilkom (Liechstein) AG" },
	{  295770, "Tele2 AG" },
	{  297030, "MTEL d.o.o. Podgorica" },
	{  302360, "Clearnet" },
	{  302370, "Microcell" },
	{  302620, "Ice Wireless" },
	{  302630, "Aliant Mobility" },
	{  302640, "Bell Mobility" },
	{  302656, "Tbay Mobility" },
	{  302660, "MTS Mobility" },
	{  302670, "CityTel Mobility" },
	{  302680, "Sask Tel Mobility" },
	{  302710, "Globalstar" },
	{  302720, "Rogers Wireless" },
	{  302860, "Telus Mobility" },
	{  308010, "St. Pierre-et-Miquelon Telecom" },
	{  310010, "MCI" },
	{  310012, "Verizon Wireless" },
	{  310013, "ALLTEL Communications Inc" },
	{  310016, "Cricket Communications" },
	{  310017, "North Sight Communications Inc" },
	{  310020, "Union Telephone Company" },
	{  310030, "Centennial Communications" },
	{  310035, "ETEX Communications" },
	{  310040, "MTA Communications dba MTA Wireless" },
	{  310050, "ACS Wireless Inc" },
	{  310060, "Consolidated Telcom" },
	{  310070, "Cingular Wireless" },
	{  310080, "Corr Wireless Communications LLC" },
	{  310090, "Cingular Wireless" },
	{  310100, "New Mexico RSA 4 East Ltd. Partnership" },
	{  310110, "Pacific Telecom Inc" },
	{  310130, "Carolina West Wireless" },
	{  310140, "GTA Wireless LLC" },
	{  310150, "Cingular Wireless" },
	{  310160, "T-Mobile USA" },
	{  310170, "Cingular Wireless" },
	{  310180, "West Central Wireless" },
	{  310190, "Alaska Wireless Communications LLC" },
	{  310200, "T-Mobile USA" },
	{  310210, "T-Mobile USA" },
	{  310220, "T-Mobile USA" },
	{  310230, "T-Mobile USA" },
	{  310240, "T-Mobile USA" },
	{  310250, "T-Mobile USA" },
	{  310260, "T-Mobile USA" },
	{  310270, "T-Mobile USA" },
	{  310280, "Contennial Puerto Rio License Corp." },
	{  310290, "Nep Cellcorp Inc." },
	{  310300, "Blanca Telephone Company" },
	{  310310, "T-Mobile USA" },
	{  310320, "Smith Bagley Inc, dba Cellular One" },
	{  310330, "Oklahoma Independent RSA5 Partnership" },
	{  310340, "High Plains Midwest LLC, dba Wetlink Communications" },
	{  310350, "Mohave Cellular L.P." },
	{  310360, "Cellular Network Partnership dba Pioneer Cellular" },
	{  310370, "Guamcell Cellular and Paging" },
	{  310380, "New Cingular Wireless PCS, LLC" },
	{  310390, "TX-11 Acquistion LLC" },
	{  310400, "Wave Runner LLC" },
	{  310410, "Cingular Wireless" },
	{  310420, "Cincinnati Bell Wireless LLC" },
	{  310430, "Alaska Digitel LLC" },
	{  310440, "United Wireless Inc" },
	{  310450, "North East Cellular Inc." },
	{  310460, "TMP Corporation" },
	{  310470, "INTELOS Communications" },
	{  310480, "Choice Phone LLC" },
	{  310490, "T-Mobile USA" },
	{  310500, "Public Service Cellular, Inc." },
	{  310510, "Airtel Wireless LLC" },
	{  310520, "VeriSign" },
	{  310530, "Iowa Wireless Services LLC" },
	{  310540, "Oklahoma Western Telephone Company" },
	{  310550, "Wireless Solutions International" },
	{  310560, "Cingular Wireless" },
	{  310570, "MTPCS LLC" },
	{  310580, "Inland Cellular Telephone Company" },
	{  310590, "Western Wireless Corporation" },
	{  310600, "New Cell Inc. dba Cellcom" },
	{  310610, "Elkhart Telephone Co. Inc. dba Epic Touch Co." },
	{  310620, "Coleman County Telecommunications Inc. (Trans Texas PCS)" },
	{  310630, "Texas-10 LLC" },
	{  310640, "Airadigm Communications" },
	{  310650, "Jasper Wireless Inc." },
	{  310660, "T-Mobile USA" },
	{  310670, "AT&T Mobility Vanguard Services" },
	{  310680, "Cingular Wireless" },
	{  310690, "Keystane Wireless LLC" },
	{  310700, "Cross Valiant Cellular Partnership" },
	{  310710, "Arctic Slope Telephone Association Cooperative" },
	{  310720, "Wireless Solutions International Inc." },
	{  310730, "Sea Mobile" },
	{  310740, "Convey Communications Inc" },
	{  310750, "East Kentucky Network LLC dba Appalachian Wireless" },
	{  310760, "Panhandle Telecommunications Systems Inc." },
	{  310770, "Iowa Wireless Services LLC dba I Wireless" },
	{  310780, "Connect Net Inc" },
	{  310790, "PinPoint Communications Inc." },
	{  310800, "T-Mobile USA" },
	{  310820, "South Canaan Cellular Communications Co. LP" },
	{  310830, "Caprock Cellular Ltd. Partnership" },
	{  310850, "Aeris Communications, Inc." },
	{  310860, "TX RSA 15B2, LP dba Five Star Wireless" },
	{  310870, "Kaplan Telephone Company Inc." },
	{  310880, "Advantage Cellular Systems, Inc." },
	{  310890, "Rural Cellular Corporation" },
	{  310910, "ALLTEL Communications Inc" },
	{  310930, "Copper Valley Wireless" },
	{  310940, "Iris Wireless LLC" },
	{  310950, "Texas RSA 1 dba XIT Cellular" },
	{  310960, "UBET Wireless" },
	{  310970, "Globalstar USA" },
	{  310990, "Worldcall Interconnect" },
	{  311000, "Mid-Tex Cellular Ltd." },
	{  311010, "Chariton Valley Communications Corp., Inc." },
	{  311020, "Missouri RSA No. 5 Partnership" },
	{  311030, "Indigo Wireless, Inc." },
	{  311040, "Commnet Wireless, LLC" },
	{  311050, "Thumb Cellular Limited Partnership" },
	{  311060, "Space Data Corporation" },
	{  311080, "Pine Telephone Company dba Pine Cellular" },
	{  311090, "Siouxland PCS" },
	{  311100, "Nex-Tech Wireless LLC" },
	{  311110, "Alltel Communications Inc" },
	{  311120, "Choice Phone LLC" },
	{  311140, "MBO Wireless Inc./Cross Telephone Company" },
	{  311150, "Wilkes Cellular Inc." },
	{  311170, "PetroCom LLC" },
	{  311180, "Cingular Wireless, Licensee Pacific Telesis Mobile Services, LLC" },
	{  311190, "Cellular Properties Inc." },
	{  311200, "ARINC" },
	{  311230, "Cellular South Inc." },
	{  311240, "Cordova Wireless Communications Inc" },
	{  311250, "Wave Runner LLC" },
	{  311260, "SLO Cellular Inc. dba CellularOne of San Luis Obispo" },
	{  311270, "Alltel Communications Inc." },
	{  311271, "Alltel Communications Inc." },
	{  311272, "Alltel Communications Inc." },
	{  311273, "Alltel Communications Inc." },
	{  311274, "Alltel Communications Inc." },
	{  311275, "Alltel Communications Inc." },
	{  311276, "Alltel Communications Inc." },
	{  311277, "Alltel Communications Inc." },
	{  311278, "Alltel Communications Inc." },
	{  311279, "Alltel Communications Inc." },
	{  311280, "Verizon Wireless" },
	{  311281, "Verizon Wireless" },
	{  311282, "Verizon Wireless" },
	{  311283, "Verizon Wireless" },
	{  311284, "Verizon Wireless" },
	{  311285, "Verizon Wireless" },
	{  311286, "Verizon Wireless" },
	{  311287, "Verizon Wireless" },
	{  311288, "Verizon Wireless" },
	{  311289, "Verizon Wireless" },
	{  311290, "Pinpoint Wireless Inc." },
	{  311300, "Rutal Cellular Corporation" },
	{  311310, "Leaco Rural Telephone Company Inc" },
	{  311320, "Commnet Wireless LLC" },
	{  311330, "Bug Tussel Wireless LLC" },
	{  311340, "Illinois Valley Cellular" },
	{  311350, "Sagebrush Cellular Inc" },
	{  311360, "Stelera Wireless LLC" },
	{  311370, "GCI Communications Corp." },
	{  311380, "GreenFly LLC" },
	{  311390, "ALLTEL Communications Inc" },
	{  311410, "Iowa RSA No.2 Ltd Partnership" },
	{  311420, "Northwest Missouri Cellular Limited Partnership" },
	{  311430, "RSA 1 Limited Partnership dba Cellular 29 Plus" },
	{  311440, "Bluegrass Cellular LLC" },
	{  311450, "Panhandle Telecommunication Systems Inc." },
	{  311460, "Fisher Wireless Services Inc" },
	{  311470, "Vitelcom Cellular Inc dba Innovative Wireless" },
	{  311480, "Verizon Wireless" },
	{  311481, "Verizon Wireless" },
	{  311482, "Verizon Wireless" },
	{  311483, "Verizon Wireless" },
	{  311484, "Verizon Wireless" },
	{  311485, "Verizon Wireless" },
	{  311486, "Verizon Wireless" },
	{  311487, "Verizon Wireless" },
	{  311488, "Verizon Wireless" },
	{  311489, "Verizon Wireless" },
	{  311490, "Strata8 Networks Inc/Wirefree Partners LLC" },
	{  311500, "CTC Telecom Inc" },
	{  311510, "Benton-Linn Wireless" },
	{  311520, "Crossroads Wireless Inc" },
	{  311530, "Wireless Communications Venture" },
	{  311540, "Keystone Wireless Inc" },
	{  311550, "Commnet Midwest LLC" },
	{  311560, "OTZ Communications Inc" },
	{  311570, "Bend Cable Communications LLC" },
	{  311580, "United States Cellular" },
	{  311590, "California RSA No3 Ltd Partnership" },
	{  311600, "Cox TMI Wireless LLC" },
	{  311610, "North Dakota Network Co." },
	{  316010, "Nextel Communications Inc" },
	{  316011, "Southern Communications Services Inc." },
	{  334020, "Telcel" },
	{  338020, "Cable & Wireless Jamaica Ltd." },
	{  338050, "Mossel (Jamaica) Ltd." },
	{  340010, "Orange Caraibe Mobiles" },
	{  340020, "Outremer Telecom" },
	{  340030, "Saint Martin et Saint Barthelemy Telcell Sarl" },
	{  340080, "Dauphin Telecom" },
	{  340100, "Guadeloupe Telephone Mobile" },
	{  340110, "Guyane Telephone Mobile" },
	{  340120, "Martinique Telephone Mobile" },
	{  340200, "Bouygues Telecom Caraibe" },
	{  342600, "Cable & Wireless (Barbados) Ltd." },
	{  342820, "Sunbeach Communications" },
	{  344030, "APUA PCS" },
	{  344920, "Cable & Wireless (Antigua)" },
	{  344930, "AT&T Wireless (Antigua)" },
	{  346140, "Cable & Wireless (Cayman)" },
	{  348170, "Cable & Wireless (BVI) Ltd" },
	{  348370, "BVI Cable TV Ltd" },
	{  348570, "Caribbean Cellular Telephone Ltd." },
	{  348770, "Digicel (BVI) Ltd" },
	{  362510, "TELCELL GSM" },
	{  362690, "CT GSM" },
	{  362910, "SETEL GSM" },
	{  363010, "Setar GSM" },
	{  365010, "Weblinks Limited" },
	{  368010, "ETECSA" },
	{  370010, "Orange Dominicana, S.A." },
	{  370020, "Verizon Dominicana S.A." },
	{  370030, "Tricom S.A." },
	{  370040, "CentennialDominicana" },
	{  372010, "Comcel" },
	{  372020, "Digicel" },
	{  372030, "Rectel" },
	{  374120, "TSTT Mobile" },
	{  374130, "Digicel Trinidad and Tobago Ltd." },
	{  374140, "LaqTel Ltd." },
	{  376352, "IslandCom Communications Ltd." },
	{  376360, "IslandCom Communication Ltd" },
	{  400010, "Azercell Limited Liability Joint Venture" },
	{  400020, "Bakcell Limited Liabil ity Company" },
	{  400030, "Catel JV" },
	{  400040, "Azerphone LLC" },
	{  401010, "Kar-Tel llc" },
	{  401020, "TSC Kazak Telecom" },
	{  402110, "Bhutan Telecom Ltd" },
	{  402170, "B-Mobile of Bhutan Telecom" },
	{  404000, "Dishnet Wireless Ltd, Madhya Pradesh" },
	{  404010, "Aircell Digilink India Ltd., Haryana" },
	{  404020, "Bharti Airtel Ltd., Punjab" },
	{  404030, "Bharti Airtel Ltd., H.P." },
	{  404040, "Idea Cellular Ltd., Delhi" },
	{  404050, "Fascel Ltd., Gujarat" },
	{  404060, "Bharti Airtel Ltd., Karnataka" },
	{  404070, "Idea Cellular Ltd., Andhra Pradesh" },
	{  404090, "Reliance Telecom Ltd., Assam" },
	{  404100, "Bharti Airtel Ltd., Delhi" },
	{  404110, "Hutchison Essar Mobile Services Ltd, Delhi" },
	{  404120, "Idea Mobile Communications Ltd., Haryana" },
	{  404130, "Hutchinson Essar South Ltd., Andhra Pradesh" },
	{  404140, "Spice Communications PVT Ltd., Punjab" },
	{  404150, "Aircell Digilink India Ltd., UP (East)" },
	{  404160, "Bharti Airtel Ltd, North East" },
	{  404170, "Dishnet Wireless Ltd, West Bengal" },
	{  404180, "Reliance Telecom Ltd., H.P." },
	{  404190, "Idea Mobile Communications Ltd., Kerala" },
	{  404200, "Hutchinson Essar Ltd, Mumbai" },
	{  404210, "BPL Mobile Communications Ltd., Mumbai" },
	{  404220, "Idea Cellular Ltd., Maharashtra" },
	{  404230, "Idea Cellular Ltd, Maharashtra" },
	{  404240, "Idea Cellular Ltd., Gujarat" },
	{  404250, "Dishnet Wireless Ltd, Bihar" },
	{  404270, "Hutchison Essar Cellular Ltd., Maharashtra" },
	{  404290, "Dishnet Wireless Ltd, Assam" },
	{  404300, "Hutchison Telecom East Ltd, Kolkata" },
	{  404310, "Bharti Airtel Ltd., Kolkata" },
	{  404330, "Dishnet Wireless Ltd, North East" },
	{  404340, "BSNL, Haryana" },
	{  404350, "Dishnet Wireless Ltd, Himachal Pradesh" },
	{  404360, "Reliance Telecom Ltd., Bihar" },
	{  404370, "Dishnet Wireless Ltd, J&K" },
	{  404380, "BSNL, Assam" },
	{  404400, "Bharti Airtel Ltd., Chennai" },
	{  404410, "Aircell Cellular Ltd, Chennai" },
	{  404420, "Aircel Ltd., Tamil Nadu" },
	{  404430, "Hutchison Essar Cellular Ltd., Tamil Nadu" },
	{  404440, "Spice Communications PVT Ltd., Karnataka" },
	{  404460, "Hutchison Essar Cellular Ltd., Kerala" },
	{  404480, "Dishnet Wireless Ltd, UP (West)" },
	{  404490, "Bharti Airtel Ltd., Andra Pradesh" },
	{  404500, "Reliance Telecom Ltd., North East" },
	{  404510, "BSNL, H.P." },
	{  404520, "Reliance TelecomLtd., Orissa" },
	{  404530, "BSNL, Punjab" },
	{  404540, "BSNL, UP (West)" },
	{  404550, "BSNL, UP (East)" },
	{  404560, "Idea Mobile Communications Ltd., UP (West)" },
	{  404570, "BSNL, Gujarat" },
	{  404580, "BSNL, Madhya Pradesh" },
	{  404590, "BSNL, Rajasthan" },
	{  404600, "Aircell Digilink India Ltd., Rajasthan" },
	{  404610, "Dishnet Wireless Ltd, Punjab" },
	{  404620, "BSNL, J&K" },
	{  404630, "Dishnet Wireless Ltd, Haryana" },
	{  404640, "BSNL, Chennai" },
	{  404650, "Dishnet Wireless Ltd, UP (East)" },
	{  404660, "BSNL, Maharashtra" },
	{  404670, "Reliance Telecom Ltd., Madhya Pradesh" },
	{  404680, "MTNL, Delhi" },
	{  404690, "MTNL, Mumbai" },
	{  404700, "Bharti Hexacom Ltd, Rajasthan" },
	{  404710, "BSNL, Karnataka" },
	{  404720, "BSNL, Kerala" },
	{  404730, "BSNL, Andhra Pradesh" },
	{  404740, "BSNL, West Bengal" },
	{  404750, "BSNL, Bihar" },
	{  404760, "BSNL, Orissa" },
	{  404770, "BSNL, North East" },
	{  404780, "BTA Cellcom Ltd., Madhya Pradesh" },
	{  404790, "BSNL, Andaman & Nicobar" },
	{  404800, "BSNL, Tamil Nadu" },
	{  404810, "BSNL, Kolkata" },
	{  404820, "Idea Telecommunications Ltd, H.P." },
	{  404830, "Reliable Internet Services Ltd., Kolkata" },
	{  404840, "Hutchinson Essar South Ltd., Chennai" },
	{  404850, "Reliance Telecom Ltd., W.B. & A.N." },
	{  404860, "Hutchinson Essar South Ltd., Karnataka" },
	{  404870, "Idea Telecommunications Ltd, Rajasthan" },
	{  404880, "Hutchison Essar South Ltd, Punjab" },
	{  404890, "Idea Telecommunications Ltd, UP (East)" },
	{  404900, "Bharti Airtel Ltd., Maharashtra" },
	{  404910, "Dishnet Wireless Ltd, Kolkata" },
	{  404920, "Bharti Airtel Ltd., Mumbai" },
	{  404930, "Bharti Airtel Ltd., Madhya Pradesh" },
	{  404940, "Bharti Airtel Ltd., Tamil Nadu" },
	{  404950, "Bharti Airtel Ltd., Kerala" },
	{  404960, "Bharti Airtel Ltd., Haryana" },
	{  404970, "Bharti Airtel Ltd., UP (West)" },
	{  404980, "Bharti Airtel Ltd., Gujarat" },
	{  404990, "Dishnet Wireless Ltd, Kerala" },
	{  405000, "Shyam Telelink Ltd, Rajasthan" },
	{  405010, "Reliance Infocomm Ltd, Andhra Pradesh" },
	{  405030, "Reliance Infocomm Ltd, Bihar" },
	{  405040, "Reliance Infocomm Ltd, Chennai" },
	{  405050, "Reliance Infocomm Ltd, Delhi" },
	{  405060, "Reliance Infocomm Ltd, Gujarat" },
	{  405070, "Reliance Infocomm Ltd, Haryana" },
	{  405080, "Reliance Infocomm Ltd, Himachal Pradesh" },
	{  405090, "Reliance Infocomm Ltd, J&K" },
	{  405100, "Reliance Infocomm Ltd, Karnataka" },
	{  405110, "Reliance Infocomm Ltd, Kerala" },
	{  405120, "Reliance Infocomm Ltd, Kolkata" },
	{  405130, "Reliance Infocomm Ltd, Maharashtra" },
	{  405140, "Reliance Infocomm Ltd, Madhya Pradesh" },
	{  405150, "Reliance Infocomm Ltd, Mumbai" },
	{  405170, "Reliance Infocomm Ltd, Orissa" },
	{  405180, "Reliance Infocomm Ltd, Punjab" },
	{  405200, "Reliance Infocomm Ltd, Tamilnadu" },
	{  405210, "Reliance Infocomm Ltd, UP (East)" },
	{  405220, "Reliance Infocomm Ltd, UP (West)" },
	{  405230, "Reliance Infocomm Ltd, West bengal" },
	{  405250, "Tata Teleservices Ltd, Andhra Pradesh" },
	{  405270, "Tata Teleservices Ltd, Bihar" },
	{  405280, "Tata Teleservices Ltd, Chennai" },
	{  405290, "Tata Teleservices Ltd, Delhi" },
	{  405300, "Tata Teleservices Ltd, Gujarat" },
	{  405310, "Tata Teleservices Ltd, Haryana" },
	{  405320, "Tata Teleservices Ltd, Himachal Pradesh" },
	{  405340, "Tata Teleservices Ltd, Karnataka" },
	{  405350, "Tata Teleservices Ltd, Kerala" },
	{  405360, "Tata Teleservices Ltd, Kolkata" },
	{  405370, "Tata Teleservices Ltd, Maharashtra" },
	{  405380, "Tata Teleservices Ltd, Madhya Pradesh" },
	{  405390, "Tata Teleservices Ltd, Mumbai" },
	{  405410, "Tata Teleservices Ltd, Orissa" },
	{  405420, "Tata Teleservices Ltd, Punjab" },
	{  405430, "Tata Teleservices Ltd, Rajasthan" },
	{  405440, "Tata Teleservices Ltd, Tamilnadu" },
	{  405450, "Tata Teleservices Ltd, UP (East)" },
	{  405460, "Tata Teleservices Ltd, UP (West)" },
	{  405470, "Tata Teleservices Ltd, West Bengal" },
	{  405520, "Bharti Airtel Ltd, Bihar" },
	{  405530, "Bharti Airtel Ltd, Orissa" },
	{  405540, "Bharti Airtel Ltd, UP (East)" },
	{  405550, "Bharti Airtel Ltd, J&K" },
	{  405560, "Bharti Airtel Ltd, Assam" },
	{  405660, "Hutchison Essar South Ltd, UP (West)" },
	{  405670, "Hutchison Essar South Ltd, Orissa" },
	{  405680, "Vodaphone/Hutchison, Madhya Pradesh" },
	{  405700, "Aditya Birla Telecom Ltd, Bihar" },
	{  405710, "Essar Spacetel Ltd, Himachal Pradesh" },
	{  405720, "Essar Spacetel Ltd, North East" },
	{  405730, "Essar Spacetel Ltd, Assam" },
	{  405740, "Essar Spacetel Ltd, J&K" },
	{  405750, "Essar Spacetel Ltd, Bihar" },
	{  405760, "Essar Spacetel Ltd, Orissa" },
	{  405770, "Essar Spacetel Ltd, Maharashtra" },
	{  405800, "Aircell Ltd, Karnataka" },
	{  405810, "Aircell Ltd, Delhi" },
	{  405820, "Aircell Ltd, Andhra Pradesh" },
	{  405830, "Aircell Ltd, Gujarat" },
	{  405840, "Aircell Ltd, Maharashtra" },
	{  405850, "Aircell Ltd, Mumbai" },
	{  405860, "Aircell Ltd, Rajasthan" },
	{  410010, "Mobilink" },
	{  410030, "PAK Telecom Mobile Ltd. (UFONE)" },
	{  410040, "CMPak" },
	{  410060, "Telenor Pakistan" },
	{  410070, "Warid Telecom" },
	{  412010, "AWCC" },
	{  412200, "Roshan" },
	{  412300, "New1" },
	{  412400, "Areeba Afghanistan" },
	{  412880, "Afghan Telecom" },
	{  413020, "MTN Network Ltd." },
	{  413030, "Celtel Lanka Ltd." },
	{  414010, "Myanmar Post and Telecommunication" },
	{  415050, "Ogero Telecom" },
	{  415320, "Cellis" },
	{  415330, "Cellis" },
	{  415340, "Cellis" },
	{  415350, "Cellis" },
	{  415360, "Libancell" },
	{  415370, "Libancell" },
	{  415380, "Libancell" },
	{  415390, "Libancell" },
	{  416010, "Fastlink" },
	{  416020, "Xpress" },
	{  416030, "Umniah" },
	{  416770, "Mobilecom" },
	{  417010, "Syriatel" },
	{  417020, "Spacetel Syria" },
	{  417090, "Syrian Telecom" },
	{  418050, "Asia Cell" },
	{  418200, "Zain Iraq (previously Atheer)" },
	{  418300, "Zain Iraq (previously Iraqna)" },
	{  418400, "Korek Telecom" },
	{  418470, "Iraq Central Cooperative Association for Communication and Transportation" },
	{  418480, "ITC Fanoos" },
	{  418490, "Iraqtel" },
	{  418620, "Itisaluna" },
	{  418700, "Kalimat" },
	{  418800, "Iraqi Telecommunications & Post Company (ITPC)" },
	{  418810, "ITPC (Al-Mazaya)" },
	{  418830, "ITPC (Sader Al-Iraq)" },
	{  418840, "ITPC (Eaamar Albasrah)" },
	{  418850, "ITPC (Anwar Yagotat Alkhalee)" },
	{  418860, "ITPC (Furatfone)" },
	{  418870, "ITPC (Al-Seraj)" },
	{  418880, "ITPC (High Link)" },
	{  418890, "ITPC (Al-Shams)" },
	{  418910, "ITPC (Belad Babel)" },
	{  418920, "ITPC (Al Nakheel)" },
	{  418930, "ITPC (Iraqcell)" },
	{  418940, "ITPC (Shaly)" },
	{  419020, "Mobile Telecommunications Company" },
	{  419030, "Wataniya Telecom" },
	{  419040, "Viva" },
	{  420010, "Saudi Telecom" },
	{  420030, "Etihad Etisalat Company (Mobily)" },
	{  421010, "Yemen Mobile Phone Company" },
	{  421020, "Spacetel Yemen" },
	{  422020, "Oman Mobile Telecommunications Company (Oman Mobile)" },
	{  422030, "Oman Qatari Telecommunications Company (Nawras)" },
	{  422040, "Oman Telecommunications Company (Omantel)" },
	{  424020, "Etisalat" },
	{  425010, "Partner Communications Co. Ltd." },
	{  425020, "Cellcom Israel Ltd." },
	{  425030, "Pelephone Communications Ltd." },
	{  425040, "Globalsim Ltd" },
	{  425060, "Wataniya" },
	{  426010, "BHR Mobile Plus" },
	{  427010, "QATARNET" },
	{  428990, "Mobicom" },
	{  429010, "Nepal Telecommunications" },
	{  432110, "Telecommunication Company of Iran (TCI)" },
	{  432140, "Telecommunication Kish Co. (KIFZO)" },
	{  432190, "Telecommunication Company of Iran (TCI) - Isfahan Celcom" },
	{  434010, "Buztel" },
	{  434020, "Uzmacom" },
	{  434040, "Daewoo Unitel" },
	{  434050, "Coscom" },
	{  434070, "Uzdunrobita" },
	{  436010, "JC Somoncom" },
	{  436020, "CJSC Indigo Tajikistan" },
	{  436030, "TT mobile" },
	{  436040, "Josa Babilon-T" },
	{  436050, "CTJTHSC Tajik-tel" },
	{  437010, "Bitel GSM" },
	{  438010, "Barash Communication Technologies (BCTI)" },
	{  438020, "TM-Cell" },
	{  440010, "NTT DoCoMo, Inc." },
	{  440020, "NTT DoCoMo Kansai, Inc." },
	{  440030, "NTT DoCoMo Hokuriku, Inc." },
	{  440040, "Vodafone" },
	{  440060, "Vodafone" },
	{  440070, "KDDI Corporation" },
	{  440080, "KDDI Corporation" },
	{  440090, "NTT DoCoMo Kansai Inc." },
	{  440100, "NTT DoCoMo Kansai Inc." },
	{  440110, "NTT DoCoMo Tokai Inc." },
	{  440120, "NTT DoCoMo Inc." },
	{  440130, "NTT DoCoMo Inc." },
	{  440140, "NTT DoCoMo Tohoku Inc." },
	{  440150, "NTT DoCoMo Inc." },
	{  440160, "NTT DoCoMo Inc." },
	{  440170, "NTT DoCoMo Inc." },
	{  440180, "NTT DoCoMo Tokai Inc." },
	{  440190, "NTT DoCoMo Hokkaido" },
	{  440200, "NTT DoCoMo Hokuriku Inc." },
	{  440210, "NTT DoCoMo Inc." },
	{  440220, "NTT DoCoMo Kansai Inc." },
	{  440230, "NTT DoCoMo Tokai Inc." },
	{  440240, "NTT DoCoMo Chugoku Inc." },
	{  440250, "NTT DoCoMo Hokkaido Inc." },
	{  440260, "NTT DoCoMo Kyushu Inc." },
	{  440270, "NTT DoCoMoTohoku Inc." },
	{  440280, "NTT DoCoMo Shikoku Inc." },
	{  440290, "NTT DoCoMo Inc." },
	{  440300, "NTT DoCoMo Inc." },
	{  440310, "NTT DoCoMo Kansai Inc." },
	{  440320, "NTT DoCoMo Inc." },
	{  440330, "NTT DoCoMo Tokai Inc." },
	{  440340, "NTT DoCoMo Kyushu Inc." },
	{  440350, "NTT DoCoMo Kansai Inc." },
	{  440360, "NTT DoCoMo Inc." },
	{  440370, "NTT DoCoMo Inc." },
	{  440380, "NTT DoCoMo Inc." },
	{  440390, "NTT DoCoMo Inc." },
	{  440400, "Vodafone" },
	{  440410, "Vodafone" },
	{  440420, "Vodafone" },
	{  440430, "Vodafone" },
	{  440440, "Vodafone" },
	{  440450, "Vodafone" },
	{  440460, "Vodafone" },
	{  440470, "Vodafone" },
	{  440480, "Vodafone" },
	{  440490, "NTT DoCoMo Inc." },
	{  440500, "KDDI Corporation" },
	{  440510, "KDDI Corporation" },
	{  440520, "KDDI Corporation" },
	{  440530, "KDDI Corporation" },
	{  440540, "KDDI Corporation" },
	{  440550, "KDDI Corporation" },
	{  440560, "KDDI Corporation" },
	{  440580, "NTT DoCoMo Kansai Inc." },
	{  440600, "NTT DoCoMo Kansai Inc." },
	{  440610, "NTT DoCoMo Chugoku Inc." },
	{  440620, "NTT DoCoMo Kyushu Inc." },
	{  440630, "NTT DoCoMo Inc." },
	{  440640, "NTT DoCoMo Inc." },
	{  440650, "NTT DoCoMo Shikoku Inc." },
	{  440660, "NTT DoCoMo Inc." },
	{  440670, "NTT DoCoMo Tohoku Inc." },
	{  440680, "NTT DoCoMo Kyushu Inc." },
	{  440690, "NTT DoCoMo Inc." },
	{  440700, "KDDI Corporation" },
	{  440710, "KDDI Corporation" },
	{  440720, "KDDI Corporation" },
	{  440730, "KDDI Corporation" },
	{  440740, "KDDI Corporation" },
	{  440750, "KDDI Corporation" },
	{  440760, "KDDI Corporation" },
	{  440770, "KDDI Corporation" },
	{  440780, "Okinawa Cellular Telephone" },
	{  440790, "KDDI Corporation" },
	{  440800, "TU-KA Cellular Tokyo Inc." },
	{  440810, "TU-KA Cellular Tokyo Inc." },
	{  440820, "TU-KA Phone Kansai Inc." },
	{  440830, "TU-KA Cellular Tokai Inc." },
	{  440840, "TU-KA Phone Kansai Inc." },
	{  440850, "TU-KA Cellular Tokai Inc." },
	{  440860, "TU-KA Cellular Tokyo Inc." },
	{  440870, "NTT DoCoMo Chugoku Inc." },
	{  440880, "KDDI Corporation" },
	{  440890, "KDDI Corporation" },
	{  440900, "Vodafone" },
	{  440920, "Vodafone" },
	{  440930, "Vodafone" },
	{  440940, "Vodafone" },
	{  440950, "Vodafone" },
	{  440960, "Vodafone" },
	{  440970, "Vodafone" },
	{  440980, "Vodafone" },
	{  440990, "NTT DoCoMo Inc." },
	{  441400, "NTT DoCoMo Inc." },
	{  441410, "NTT DoCoMo Inc." },
	{  441420, "NTT DoCoMo Inc." },
	{  441430, "NTT DoCoMo Kansai Inc." },
	{  441440, "NTT DoCoMo Chugoku Inc." },
	{  441450, "NTT DoCoMo Shikoku Inc." },
	{  441500, "TU-KA Cellular Tokyo Inc." },
	{  441510, "TU-KA Phone Kansai Inc." },
	{  441610, "Vodafone" },
	{  441620, "Vodafone" },
	{  441630, "Vodafone" },
	{  441640, "Vodafone" },
	{  441650, "Vodafone" },
	{  441700, "KDDI Corporation" },
	{  441900, "NTT DoCoMo Inc." },
	{  441910, "NTT DoCoMo Inc." },
	{  441920, "NTT DoCoMo Inc." },
	{  441930, "NTT DoCoMo Hokkaido Inc." },
	{  441940, "NTT DoCoMo Tohoku Inc." },
	{  441980, "NTT DoCoMo Kyushu Inc." },
	{  441990, "NTT DoCoMo Kyushu Inc." },
	{  450020, "KT Freetel" },
	{  450030, "SK Telecom" },
	{  452010, "Mobifone" },
	{  452020, "Vinaphone" },
	{  452030, "S Telecom (CDMA)" },
	{  452040, "Viettel" },
	{  452070, "Beeline VN/GTEL Mobile JSC" },
	{  454000, "GSM900/HKCSL" },
	{  454010, "MVNO/CITIC" },
	{  454020, "3G Radio System/HKCSL3G" },
	{  454030, "3G Radio System/Hutchison 3G" },
	{  454040, "GSM900/GSM1800/Hutchison" },
	{  454050, "CDMA/Hutchison" },
	{  454060, "GSM900/SmarTone" },
	{  454070, "MVNO/China Unicom International Ltd." },
	{  454080, "MVNO/Trident" },
	{  454090, "MVNO/China Motion Telecom (HK) Ltd." },
	{  454100, "GSM1800New World PCS Ltd." },
	{  454110, "MVNO/CHKTL" },
	{  454120, "GSM1800/Peoples Telephone Company Ltd." },
	{  454150, "3G Radio System/SMT3G" },
	{  454160, "GSM1800/Mandarin Communications Ltd." },
	{  454180, "GSM7800/Hong Kong CSL Ltd." },
	{  454190, "3G Radio System/Sunday3G" },
	{  455000, "Smartone Mobile Communications (Macao) Ltd." },
	{  455010, "CTM GSM" },
	{  455030, "Hutchison Telecom" },
	{  456010, "Mobitel (Cam GSM)" },
	{  456020, "Hello" },
	{  456030, "S Telecom (CDMA)" },
	{  456040, "Cadcomms" },
	{  456050, "Starcell" },
	{  456060, "Smart" },
	{  456080, "Viettel" },
	{  456180, "Mfone" },
	{  457010, "Lao Telecommunications" },
	{  457020, "ETL Mobile" },
	{  457080, "Millicom" },
	{  460000, "China Mobile" },
	{  460010, "China Unicom" },
	{  460030, "China Unicom CDMA" },
	{  460040, "China Satellite Global Star Network" },
	{  470010, "GramenPhone" },
	{  470020, "Aktel" },
	{  470030, "Mobile 2000" },
	{  472010, "DhiMobile" },
	{  502100, "DIGI Telecommunications" },
	{  502120, "Malaysian Mobile Services Sdn Bhd" },
	{  502130, "Celcom (Malaysia) Berhad" },
	{  502140, "Telekom Malaysia Berhad" },
	{  502160, "DIGI Telecommunications" },
	{  502170, "Malaysian Mobile Services Sdn Bhd" },
	{  502180, "U Mobile Sdn. Bhd." },
	{  502190, "CelCom (Malaysia) Berhad" },
	{  502200, "Electcoms Wireless Sdn Bhd" },
	{  505010, "Telstra Corporation Ltd." },
	{  505020, "Optus Mobile Pty. Ltd." },
	{  505030, "Vodafone Network Pty. Ltd." },
	{  505040, "Department of Defence" },
	{  505050, "The Ozitel Network Pty. Ltd." },
	{  505060, "Hutchison 3G Australia Pty. Ltd." },
	{  505070, "Vodafone Network Pty. Ltd." },
	{  505080, "One.Tel GSM 1800 Pty. Ltd." },
	{  505090, "Airnet Commercial Australia Ltd." },
	{  505100, "Norfolk Telecom" },
	{  505110, "Telstra Corporation Ltd." },
	{  505120, "Hutchison Telecommunications (Australia) Pty. Ltd." },
	{  505130, "Railcorp" },
	{  505140, "AAPT Ltd." },
	{  505150, "3GIS Pty Ltd. (Telstra & Hutchison 3G)" },
	{  505160, "Victorian Rail Track" },
	{  505170, "Vivid Wireless Pty Ltd" },
	{  505240, "Advanced Communications Technologies Pty. Ltd." },
	{  505710, "Telstra Corporation Ltd." },
	{  505720, "Telstra Corporation Ltd." },
	{  505880, "Localstar Holding Pty. Ltd." },
	{  505900, "Optus Ltd." },
	{  505990, "One.Tel GSM 1800 Pty. Ltd." },
	{  510000, "PSN" },
	{  510010, "Satelindo" },
	{  510080, "Natrindo (Lippo Telecom)" },
	{  510100, "Telkomsel" },
	{  510110, "Excelcomindo" },
	{  510210, "Indosat - M3" },
	{  510280, "Komselindo" },
	{  515010, "Islacom" },
	{  515020, "Globe Telecom" },
	{  515030, "Smart Communications" },
	{  515050, "Digitel" },
	{  520000, "CAT CDMA" },
	{  520010, "AIS GSM" },
	{  520150, "ACT Mobile" },
	{  525010, "SingTel ST GSM900" },
	{  525020, "SingTel ST GSM1800" },
	{  525030, "MobileOne" },
	{  525050, "Starhub" },
	{  525120, "Digital Trunked Radio Network" },
	{  528110, "DST Com" },
	{  530000, "Reserved for AMPS MIN based IMSI's" },
	{  530010, "Vodafone New Zealand GSM Network" },
	{  530020, "Teleom New Zealand CDMA Network" },
	{  530030, "Woosh Wireless - CDMA Network" },
	{  530040, "TelstraClear - GSM Network" },
	{  530050, "Telecom New Zealand - UMTS Ntework" },
	{  530240, "NZ Communications - UMTS Network" },
	{  537010, "Bmobile" },
	{  537020, "Greencom" },
	{  537030, "Digicel Ltd" },
	{  539010, "Tonga Communications Corporation" },
	{  539430, "Digicel" },
	{  539880, "Digicel (Tonga) Ltd" },
	{  541010, "SMILE" },
	{  541050, "Digicel Vanuatu" },
	{  542010, "Vodafone (Fiji) Ltd" },
	{  542020, "Digicel (Fiji) Ltd" },
	{  542030, "Telecom Fiji Ltd (CDMA)" },
	{  546010, "OPT Mobilis" },
	{  547100, "Mara Telecom" },
	{  547200, "Tikiphone" },
	{  548010, "Telecom Cook" },
	{  549010, "Telecom Samoa Cellular Ltd." },
	{  549270, "GoMobile SamoaTel Ltd" },
	{  550010, "FSM Telecom" },
	{  552010, "Palau National Communications Corp. (a.k.a. PNCC)" },
	{  602010, "Mobinil" },
	{  602020, "Vodafone" },
	{  602030, "Etisalat" },
	{  603010, "Algerie Telecom" },
	{  603020, "Orascom Telecom Algerie" },
	{  604000, "Meditelecom (GSM)" },
	{  604010, "Ittissalat Al Maghrid" },
	{  605020, "Tunisie Telecom" },
	{  605030, "Orascom Telecom" },
	{  607010, "Gamcel" },
	{  607020, "Africell" },
	{  607030, "Comium Services Ltd" },
	{  608010, "Sonatel" },
	{  608020, "Sentel GSM" },
	{  609010, "Mattel S.A." },
	{  609020, "Chinguitel S.A." },
	{  609100, "Mauritel Mobiles" },
	{  610010, "Malitel" },
	{  611010, "Orange Guinee" },
	{  611020, "Sotelgui" },
	{  611050, "Cellcom Guinee SA" },
	{  612020, "Atlantique Cellulaire" },
	{  612030, "Orange Cote d'Ivoire" },
	{  612040, "Comium Cote d'Ivoire" },
	{  612050, "Loteny Telecom" },
	{  612060, "Oricel Cote d'Ivoire" },
	{  612070, "Aircomm Cote d'Ivoire" },
	{  613020, "Celtel" },
	{  613030, "Telecel" },
	{  614010, "Sahel.Com" },
	{  614020, "Celtel" },
	{  614030, "Telecel" },
	{  615010, "Togo Telecom" },
	{  616010, "Libercom" },
	{  616020, "Telecel" },
	{  616030, "Spacetel Benin" },
	{  617010, "Cellplus" },
	{  617020, "Mahanagar Telephone (Mauritius) Ltd." },
	{  617100, "Emtel" },
	{  618040, "Comium Liberia" },
	{  619010, "Celtel" },
	{  619020, "Millicom" },
	{  619030, "Africell" },
	{  619040, "Comium (Sierra Leone) Ltd." },
	{  619050, "Lintel (Sierra Leone) Ltd." },
	{  619250, "Mobitel" },
	{  619400, "Datatel (SL) Ltd GSM" },
	{  619500, "Datatel (SL) Ltd CDMA" },
	{  620010, "Spacefon" },
	{  620020, "Ghana Telecom Mobile" },
	{  620030, "Mobitel" },
	{  620040, "Kasapa Telecom Ltd." },
	{  620110, "Netafriques Dot Com Ltd" },
	{  621200, "Econet Wireless Nigeria Ltd." },
	{  621300, "MTN Nigeria Communications" },
	{  621400, "MTEL" },
	{  621500, "Globacom" },
	{  621600, "EMTS" },
	{  622010, "Celtel" },
	{  622020, "Tchad Mobile" },
	{  623010, "Centrafrique Telecom Plus (CTP)" },
	{  623020, "Telecel Centrafrique (TC)" },
	{  623030, "Celca (Socatel)" },
	{  624010, "Mobile Telephone Networks Cameroon" },
	{  624020, "Orange Cameroun" },
	{  625010, "Cabo Verde Telecom" },
	{  625020, "T+Telecomunicacoes" },
	{  626010, "Companhia Santomese de Telecomunicacoes" },
	{  627010, "Guinea Ecuatorial de Telecomunicaciones Sociedad Anonima" },
	{  628010, "Libertis S.A." },
	{  628020, "Telecel Gabon S.A." },
	{  628030, "Celtel Gabon S.A." },
	{  628040, "USAN Gabon" },
	{  629010, "Celtel" },
	{  629100, "Libertis Telecom" },
	{  630010, "Vodacom Congo RDC sprl" },
	{  630050, "Supercell Sprl" },
	{  630860, "Congo-Chine Telecom s.a.r.l." },
	{  631020, "Unitel" },
	{  632010, "Guinetel S.A." },
	{  632020, "Spacetel Guine-Bissau S.A." },
	{  633010, "Cable & Wireless (Seychelles) Ltd." },
	{  633020, "Mediatech International Ltd." },
	{  633100, "Telecom (Seychelles) Ltd." },
	{  634010, "SD Mobitel" },
	{  634020, "Areeba-Sudan" },
	{  634050, "Network of the World Ltd (NOW)" },
	{  634990, "MTN Sudan" },
	{  635100, "MTN Rwandacell" },
	{  636010, "ETH MTN" },
	{  637300, "Golis Telecommunications Company" },
	{  638010, "Evatis" },
	{  639020, "Safaricom Ltd." },
	{  639030, "Kencell Communications Ltd." },
	{  640020, "MIC (T) Ltd." },
	{  640030, "Zantel" },
	{  640040, "Vodacom (T) Ltd." },
	{  640050, "Celtel (T) Ltd." },
	{  641010, "Celtel Uganda" },
	{  641100, "MTN Uganda Ltd." },
	{  641110, "Uganda Telecom Ltd." },
	{  641140, "House of Integrated Technology and Systems Uganda Ltd" },
	{  641220, "Warid Telecom Uganda Ltd." },
	{  641660, "i-Tel Ltd" },
	{  642010, "ECONET" },
	{  642020, "AFRICELL" },
	{  642030, "ONAMOB" },
	{  642070, "LACELL" },
	{  642080, "HITS TELECOM" },
	{  642820, "U.COM" },
	{  643010, "T.D.M. GSM" },
	{  643040, "VM Sarl" },
	{  645010, "Celtel Zambia Ltd." },
	{  645020, "Telecel Zambia Ltd." },
	{  645030, "Zamtel" },
	{  646010, "Celtel Madagascar (Zain), GSM" },
	{  646020, "Orange Madagascar, GSM" },
	{  646030, "Madamobil, CDMA 2000" },
	{  646040, "Telecom Malagasy Mobile, GSM" },
	{  647000, "Orange La Reunion" },
	{  647020, "Outremer Telecom" },
	{  647100, "Societe Reunionnaise du Radiotelephone" },
	{  648010, "Net One" },
	{  648030, "Telecel" },
	{  648040, "Econet" },
	{  649010, "Mobile Telecommunications Ltd." },
	{  649030, "Powercom Pty Ltd" },
	{  650010, "Telekom Network Ltd." },
	{  650100, "Celtel ltd." },
	{  651010, "Vodacom Lesotho (pty) Ltd." },
	{  651020, "Econet Ezin-cel" },
	{  652010, "Mascom Wireless (Pty) Ltd." },
	{  652020, "Orange Botswana (Pty) Ltd." },
	{  653010, "SPTC" },
	{  653100, "Swazi MTN" },
	{  654010, "HURI - SNPT" },
	{  655010, "Vodacom (Pty) Ltd." },
	{  655020, "Telkom SA Ltd" },
	{  655060, "Sentech (Pty) Ltd." },
	{  655070, "Cell C (Pty) Ltd." },
	{  655100, "Mobile Telephone Networks (MTN) Pty Ltd" },
	{  655110, "SAPS Gauteng" },
	{  655120, "Mobile Telephone Networks (MTN) Pty Ltd" },
	{  655130, "Neotel Pty Ltd" },
	{  655210, "Cape Town Metropolitan Council" },
	{  655300, "Bokamoso Consortium Pty Ltd" },
	{  655310, "Karabo Telecoms (Pty) Ltd." },
	{  655320, "Ilizwi Telecommunications Pty Ltd" },
	{  655330, "Thinta Thinta Telecommunications Pty Ltd" },
	{  655340, "Bokone Telecoms Pty Ltd" },
	{  655350, "Kingdom Communications Pty Ltd" },
	{  655360, "Amatole Telecommunication Pty Ltd" },
	{  702670, "Belize Telecommunications Ltd., GSM 1900" },
	{  702680, "International Telecommunications Ltd. (INTELCO)" },
	{  704010, "Servicios de Comunicaciones Personales Inalambricas, S.A." },
	{  704020, "Comunicaciones Celulares S.A." },
	{  704030, "Telefonica Centroamerica Guatemala S.A." },
	{  706010, "CTE Telecom Personal, S.A. de C.V." },
	{  706020, "Digicel, S.A. de C.V." },
	{  706030, "Telemovil El Salvador, S.A." },
	{  708001, "Megatel" },
	{  708002, "Celtel" },
	{  708040, "Digicel Honduras" },
	{  710210, "Empresa Nicaraguense de Telecomunicaciones, S.A. (ENITEL)" },
	{  710730, "Servicios de Comunicaciones, S.A. (SERCOM)" },
	{  712010, "Instituto Costarricense de Electricidad - ICE" },
	{  714010, "Cable & Wireless Panama S.A." },
	{  714020, "Telefonica Moviles Panama S.A." },
	{  714030, "Claro Panama, S.A." },
	{  714040, "Digicel (Panama), S.A." },
	{  716100, "TIM Peru" },
	{  722010, "Compania de Radiocomunicaciones Moviles S.A." },
	{  722020, "Nextel Argentina srl" },
	{  722070, "Telefonica Comunicaciones Personales S.A." },
	{  722310, "CTI PCS S.A." },
	{  722320, "Compania de Telefonos del Interior Norte S.A." },
	{  722330, "Compania de Telefonos del Interior S.A." },
	{  722341, "Telecom Personal S.A." },
	{  724000, "Telet" },
	{  724010, "CRT Cellular" },
	{  724020, "Global Telecom" },
	{  724030, "CTMR Cel" },
	{  724040, "BCP" },
	{  724050, "Telesc Cel" },
	{  724060, "Tess" },
	{  724070, "Sercontel Cel" },
	{  724080, "Maxitel MG" },
	{  724090, "Telepar Cel" },
	{  724100, "ATL Algar" },
	{  724110, "Telems Cel" },
	{  724120, "Americel" },
	{  724130, "Telesp Cel" },
	{  724140, "Maxitel BA" },
	{  724150, "CTBC Cel" },
	{  724160, "BSE" },
	{  724170, "Ceterp Cel" },
	{  724180, "Norte Brasil Tel" },
	{  724190, "Telemig Cel" },
	{  724210, "Telerj Cel" },
	{  724230, "Telest Cel" },
	{  724250, "Telebrasilia Cel" },
	{  724270, "Telegoias Cel" },
	{  724290, "Telemat Cel" },
	{  724310, "Teleacre Cel" },
	{  724330, "Teleron Cel" },
	{  724350, "Telebahia Cel" },
	{  724370, "Telergipe Cel" },
	{  724390, "Telasa Cel" },
	{  724410, "Telpe Cel" },
	{  724430, "Telepisa Cel" },
	{  724450, "Telpa Cel" },
	{  724470, "Telern Cel" },
	{  724480, "Teleceara Cel" },
	{  724510, "Telma Cel" },
	{  724530, "Telepara Cel" },
	{  724550, "Teleamazon Cel" },
	{  724570, "Teleamapa Cel" },
	{  724590, "Telaima Cel" },
	{  730010, "Entel Telefonica Movil" },
	{  730020, "Telefonica Movil" },
	{  730030, "Smartcom" },
	{  730040, "Centennial Cayman Corp. Chile S.A." },
	{  730050, "Multikom S.A." },
	{  730060, "Blue Two Chile S.A." },
	{  730070, "Telefonica Moviles Chile S.A." },
	{  730100, "Entel" },
	{  732001, "Colombia Telecomunicaciones S.A. - Telecom" },
	{  732002, "Edatel S.A." },
	{  732020, "Emtelsa" },
	{  732099, "Emcali" },
	{  732101, "Comcel S.A. Occel S.A./Celcaribe" },
	{  732102, "Bellsouth Colombia S.A." },
	{  732103, "Colombia Movil S.A." },
	{  732111, "Colombia Movil S.A." },
	{  732123, "Telefonica Moviles Colombia S.A." },
	{  732130, "Avantel" },
	{  734010, "Infonet" },
	{  734020, "Corporacion Digitel" },
	{  734030, "Digicel" },
	{  734040, "Telcel, C.A." },
	{  734060, "Telecomunicaciones Movilnet, C.A." },
	{  736010, "Nuevatel S.A." },
	{  736020, "ENTEL S.A." },
	{  736030, "Telecel S.A." },
	{  738010, "Cel*Star (Guyana) Inc." },
	{  740000, "Otecel S.A. - Bellsouth" },
	{  740010, "Porta GSM" },
	{  740020, "Telecsa S.A." },
	{  744010, "Hola Paraguay S.A." },
	{  744020, "Hutchison Telecom S.A." },
	{  744030, "Compania Privada de Comunicaciones S.A." },
	{  746020, "Telesur" },
	{  746030, "Digicel" },
	{  746040, "Intelsur" },
	{  746050, "Telesur (CDMA)" },
	{  748000, "Ancel - TDMA" },
	{  748010, "Ancel - GSM" },
	{  748030, "Ancel" },
	{  748070, "Movistar" },
	{  748100, "CTI Movil" },
	{  901010, "ICO Global Communications" },
	{  901020, "Sense Communications International AS" },
	{  901030, "Iridium Satellite, LLC (GMSS)" },
	{  901050, "Thuraya RMSS Network" },
	{  901060, "Thuraya Satellite Telecommunications Company" },
	{  901090, "Tele1 Europe" },
	{  901100, "Asia Cellular Satellite (AceS)" },
	{  901110, "Inmarsat Ltd." },
	{  901120, "Maritime Communications Partner AS (MCP network)" },
	{  901130, "Global Networks, Inc." },
	{  901140, "Telenor GSM - services in aircraft" },
	{  901150, "OnAir" },
	{  901160, "Jasper Systems, Inc." },
	{  901170, "Jersey Telecom" },
	{  901180, "Cingular Wireless" },
	{  901190, "Vodaphone Malta Vodafone Group)" },
	{  901200, "Intermatica" },
	{  901210, "Seanet Maritime Communications" },
	{  901220, "MediaLincc Ltd" },
	{  901230, "Ukrainian Radiosystems" },
	{  901240, "Voxbone SA" },
	{  901250, "In & Phone" },
	{  901260, "Telecom Italia" },
	{  901270, "Onair" },
	{  901280, "Vodafone Group" },
	{  901290, "Telenor Connexion AB" },
	{  901300, "VP Network Engineering, Terrestar Networks" },
	{  901880, "Telecommunications for Disaster Relief (TDR) (OCHA)" },
	{  0, NULL }
};

static int proto_e212						= -1;
static int hf_E212_mcc						= -1;
static int hf_E212_mnc						= -1;
static int hf_E212_msin						= -1;

/*
 * MCC/MNC dissection - little endian MNC encoding
 *
 * MNC of length 2:
 * 
 *	 8   7   6   5   4   3   2   1
 * +---+---+---+---+---+---+---+---+
 * |  MCC digit 2  |  MCC digit 1  |  octet x
 * +---------------+---------------+
 * |    Filler     |  MCC digit 3  |  octet x+1
 * +---------------+---------------+
 * |  MNC digit 2  |  MNC digit 1  |  octet x+2
 * +---------------+---------------+
 *
 * MNC of length 3:
 *
 *	 8   7   6   5   4   3   2   1
 * +---+---+---+---+---+---+---+---+
 * |  MCC digit 2  |  MCC digit 1  |  octet x
 * +---------------+---------------+
 * |  MNC digit 3  |  MCC digit 3  |  octet x+1
 * +---------------+---------------+
 * |  MNC digit 2  |  MNC digit 1  |  octet x+2
 * +---------------+---------------+
 *
 *
 * MCC/MNC dissection - big endian MNC encoding
 *
 * MNC of length 2:
 * 
 *	 8   7   6   5   4   3   2   1
 * +---+---+---+---+---+---+---+---+
 * |  MCC digit 2  |  MCC digit 1  |  octet x
 * +---------------+---------------+
 * |    Filler     |  MCC digit 3  |  octet x+1
 * +---------------+---------------+
 * |  MNC digit 2  |  MNC digit 1  |  octet x+2
 * +---------------+---------------+
 *
 * MNC of length 3:
 *
 *	 8   7   6   5   4   3   2   1
 * +---+---+---+---+---+---+---+---+
 * |  MCC digit 2  |  MCC digit 1  |  octet x
 * +---------------+---------------+
 * |  MNC digit 1  |  MCC digit 3  |  octet x+1
 * +---------------+---------------+
 * |  MNC digit 3  |  MNC digit 2  |  octet x+2
 * +---------------+---------------+
 */
int
dissect_e212_mcc_mnc(tvbuff_t *tvb, packet_info *pinfo, proto_tree *tree, int offset, gboolean little_endian){

	int			start_offset;	
	guint8		octet;
	guint16		mcc, mnc;
	guint8		mcc1, mcc2, mcc3, mnc1, mnc2, mnc3;
	proto_item	*item;

	start_offset = offset;
	/* Mobile country code MCC */
	octet = tvb_get_guint8(tvb,offset);
	mcc1 = octet & 0x0f;
	mcc2 = octet >> 4;
	offset++;
	octet = tvb_get_guint8(tvb,offset);
	mcc3 = octet & 0x0f;
	/* MNC, Mobile network code (octet 3 bits 5 to 8, octet 4)  */
	mnc3 = octet >> 4;
	offset++;
	octet = tvb_get_guint8(tvb,offset);
	mnc1 = octet & 0x0f;
	mnc2 = octet >> 4;

	mcc = 100 * mcc1 + 10 * mcc2 + mcc3;
	mnc = 10 * mnc1 + mnc2;
	if (mnc3 != 0xf) {
		if(little_endian)
			mnc = 10 * mnc + mnc3;
		else
			mnc = 100 * mnc3 + mnc;
	}
	item = proto_tree_add_uint(tree, hf_E212_mcc , tvb, start_offset, 2, mcc );
	if ((mcc1 > 9) || (mcc2 > 9) || (mcc3 > 9))
		expert_add_info_format(pinfo, item, PI_MALFORMED, PI_WARN, "MCC contains non-decimal digits");

	if(mnc3 != 0x0f)
		item = proto_tree_add_uint_format(tree, hf_E212_mnc , tvb, start_offset + 1, 2, mnc,
				   "Mobile Network Code (MNC): %s (%03u)",
				   val_to_str(mcc * 1000 + mnc, mcc_mnc_codes, "Unknown"),
				   mnc);
	else
		item = proto_tree_add_uint_format(tree, hf_E212_mnc , tvb, start_offset + 1, 2, mnc,
				   "Mobile Network Code (MNC): %s (%02u)",
				   val_to_str(mcc * 1000 + 10 * mnc, mcc_mnc_codes, "Unknown"),
				   mnc);

	if ((mnc1 > 9) || (mnc2 > 9) || ((mnc3 > 9) && (mnc3 != 0x0f)))
		expert_add_info_format(pinfo, item, PI_MALFORMED, PI_WARN, "MNC contains non-decimal digits");

	offset++;
	return offset;
}


/*
 * When we want to decode the MCC/MNC pair in an address that is encoded according to E.212
 * the main problem is that we do not know whether we are dealing with a length 2 or length 3 
 * MNC. Initially, it was possible to find the length of the MNC by checking the MCC code. 
 * Originally each country employed a pure 2 digit or 3 digit MNC scheme. However, it is possible
 * to find countries now that employ both lengths for the MNC (e.g. Canada, India).
 * Since in these cases we can be sure that an MNC cannot possible be a prefix for another MNC, we
 * initially assume that the MNC is 2 digits long. If the MCC/MNC combination gives a match in our 
 * mcc_mnc_codes list then we can be sure that we deal with an MNC that is 2 digits long. Otherwise,
 * assume that the MNC is 3 digits long.
 * 
 * MNC of length 2:
 * 
 *     8   7   6   5   4   3   2   1
 *   +---+---+---+---+---+---+---+---+
 *   |  MCC digit 2  |  MCC digit 1  |  octet x
 *   +---------------+---------------+
 *   |  MNC digit 1  |  MCC digit 3  |  octet x+1
 *   +---------------+---------------+
 *   | addr digit 1  |  MNC digit 2  |  octet x+2
 *   +---------------+---------------+
 *
 * MNC of length 3:
 * 
 *     8   7   6   5   4   3   2   1
 *   +---+---+---+---+---+---+---+---+
 *   |  MCC digit 2  |  MCC digit 1  |  octet x
 *   +---------------+---------------+
 *   |  MNC digit 1  |  MCC digit 3  |  octet x+1
 *   +---------------+---------------+
 *   |  MNC digit 3  |  MNC digit 2  |  octet x+2
 *   +---------------+---------------+
 *
 * This function will consume either 2.5 or 3 octets. For this reason it returns
 * the number of nibbles consumed, i.e. 5 or 6 (2 or 3 digits long MNC respectively)
 */
int
dissect_e212_mcc_mnc_in_address(tvbuff_t *tvb, packet_info *pinfo, proto_tree *tree, int offset)
{
	guint32		start_offset;
	guint8		octet;
	guint16		mcc, mnc;
	guint8		mcc1, mcc2, mcc3, mnc1, mnc2, mnc3;
	proto_item	*item;
	gboolean	long_mnc;

	long_mnc = FALSE;
	start_offset = offset;

	/* MCC digits 1 and 2 */
	octet = tvb_get_guint8(tvb,offset);
	mcc1 = octet & 0x0f;
	mcc2 = octet >> 4;
	offset++;

	/* MCC digit 3 and MNC digit 1 */
	octet = tvb_get_guint8(tvb,offset);
	mcc3 = octet & 0x0f;
	mnc1 = octet >> 4;
	offset++;

	/* MNC digits 2 and 3 */
	octet = tvb_get_guint8(tvb,offset);
	mnc2 = octet & 0x0f;
	mnc3 = octet >> 4;

	mcc = 100 * mcc1 + 10 * mcc2 + mcc3;
	mnc = 10 * mnc1 + mnc2;

	/* Try to match the MCC and 2 digits MNC with an entry in our list of operators */
	if (!match_strval(mcc * 1000 + 10 * mnc, mcc_mnc_codes)) {
		mnc = 10 * mnc + mnc3;
		long_mnc = TRUE;
	}

	item = proto_tree_add_uint(tree, hf_E212_mcc , tvb, start_offset, 2, mcc );
	if ((mcc1 > 9) || (mcc2 > 9) || (mcc3 > 9))
		expert_add_info_format(pinfo, item, PI_MALFORMED, PI_WARN, "MCC contains non-decimal digits");

	if(long_mnc)
		item = proto_tree_add_uint_format(tree, hf_E212_mnc , tvb, start_offset + 1, 2, mnc,
				   "Mobile Network Code (MNC): %s (%03u)",
				   val_to_str(mcc * 1000 + mnc, mcc_mnc_codes, "Unknown"),
				   mnc);
	else
		item = proto_tree_add_uint_format(tree, hf_E212_mnc , tvb, start_offset + 1, 2, mnc,
				   "Mobile Network Code (MNC): %s (%02u)",
				   val_to_str(mcc * 1000 + 10 * mnc, mcc_mnc_codes, "Unknown"),
				   mnc);

	if ((mnc1 > 9) || (mnc2 > 9) || (long_mnc && (mnc3 > 9)))
		expert_add_info_format(pinfo, item, PI_MALFORMED, PI_WARN, "MNC contains non-decimal digits");

	if(long_mnc)
		return 6;
	else
		return 5;
}

/*
 * Register the protocol with Wireshark.
 *
 * This format is required because a script is used to build the C function
 * that calls all the protocol registration.
 */


void
proto_register_e212(void)
{

/* Setup list of header fields  See Section 1.6.1 for details */
	static hf_register_info hf[] = {
	{ &hf_E212_mcc,
		{ "Mobile Country Code (MCC)","e212.mcc",
		FT_UINT16, BASE_DEC, VALS(E212_codes), 0x0,          
		"Mobile Country Code MCC", HFILL }
	},
	{ &hf_E212_mnc,
		{ "Mobile network code (MNC)","e212.mnc",
		FT_UINT16, BASE_DEC, NULL, 0x0,          
		"Mobile network code", HFILL }
	},
	{ &hf_E212_msin,
      { "Mobile Subscriber Identification Number (MSIN)", "e212.msin",
        FT_STRING, BASE_NONE, NULL, 0,
        "Mobile Subscriber Identification Number(MSIN)", HFILL }},
	};

	/*
	 * Register the protocol name and description
	 */
	proto_e212 = proto_register_protocol(
			"ITU-T E.212 number",
			"E.212",
			"e212");

	/*
	 * Required function calls to register
	 * the header fields and subtrees used.
	 */
	proto_register_field_array(proto_e212, hf, array_length(hf));

}
