/* packet-nas_eps.c
 * Routines for Non-Access-Stratum (NAS) protocol for Evolved Packet System (EPS) dissection
 *
 * Copyright 2008 - 2009, Anders Broman <anders.broman@ericsson.com>
 *
 * $Id: packet-nas_eps.c 27802 2009-03-19 22:03:33Z etxrab $
 *
 * Wireshark - Network traffic analyzer
 * By Gerald Combs <gerald@wireshark.org>
 * Copyright 1998 Gerald Combs
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 * References: 3GPP TS 24.301 V8.0.0 (2008-12) and V8.1.0 Draft v5
 */

#ifdef HAVE_CONFIG_H
# include "config.h"
#endif

#include <glib.h>
#include <epan/packet.h>

#include "packet-gsm_a_common.h"
#include "packet-e212.h"

#define PNAME  "Non-Access-Stratum (NAS)PDU"
#define PSNAME "NAS-EPS"
#define PFNAME "nas-eps"

/* Initialize the protocol and registered fields */
static int proto_nas_eps = -1;

static int hf_nas_eps_msg_emm_type = -1;
int hf_nas_eps_common_elem_id = -1;
int hf_nas_eps_emm_elem_id = -1;
static int hf_nas_eps_bearer_id = -1;
static int hf_nas_eps_spare_bits = -1;
static int hf_nas_eps_security_header_type = -1;
static int hf_nas_eps_emm_eps_att_type = -1;
static int hf_nas_eps_emm_nas_key_set_id = -1;
static int hf_nas_eps_emm_odd_even = -1;
static int hf_nas_eps_emm_type_of_id = -1;
static int hf_nas_eps_emm_mme_grp_id = -1;
static int hf_nas_eps_emm_mme_code = -1;
static int hf_nas_eps_emm_m_tmsi = -1;
static int hf_nas_eps_esm_msg_cont = -1;
static int hf_nas_eps_emm_EPS_attach_result = -1;
static int hf_nas_eps_emm_spare_half_octet = -1;
static int hf_nas_eps_emm_res = -1;
static int hf_nas_eps_emm_cause = -1;
static int hf_nas_eps_emm_id_type2 = -1;
static int hf_nas_eps_emm_short_mac = -1;
static int hf_nas_eps_emm_128eea0 = -1;
static int hf_nas_eps_emm_128eea1 = -1;
static int hf_nas_eps_emm_128eea2 = -1;
static int hf_nas_eps_emm_128eea3 = -1;
static int hf_nas_eps_emm_128eea4 = -1;
static int hf_nas_eps_emm_128eea5 = -1;
static int hf_nas_eps_emm_128eea6 = -1;
static int hf_nas_eps_emm_128eea7 = -1;
static int hf_nas_eps_emm_128eia1 = -1;
static int hf_nas_eps_emm_128eia2 = -1;
static int hf_nas_eps_emm_128eia3 = -1;
static int hf_nas_eps_emm_128eia4 = -1;
static int hf_nas_eps_emm_128eia5 = -1;
static int hf_nas_eps_emm_128eia6 = -1;
static int hf_nas_eps_emm_128eia7 = -1;
static int hf_nas_eps_emm_128uea0 = -1;
static int hf_nas_eps_emm_128uea1 = -1;
static int hf_nas_eps_emm_128uea2 = -1;
static int hf_nas_eps_emm_128uea3 = -1;
static int hf_nas_eps_emm_128uea4 = -1;
static int hf_nas_eps_emm_128uea5 = -1;
static int hf_nas_eps_emm_128uea6 = -1;
static int hf_nas_eps_emm_128uea7 = -1;


static int hf_nas_eps_active_flg = -1;
static int hf_nas_eps_eps_update_result_value = -1;
static int hf_nas_eps_eps_update_type_value = -1;
static int hf_nas_eps_service_type = -1;

/* ESM */
static int hf_nas_eps_msg_esm_type = -1;
int hf_nas_eps_esm_elem_id = -1;
static int hf_nas_eps_esm_proc_trans_id = -1;
static int hf_nas_eps_esm_request_type = -1;
static int hf_nas_eps_esm_pdn_type = -1;

/* Initialize the subtree pointers */
static int ett_nas_eps = -1;
static int ett_nas_eps_esm_msg_cont = -1;

/* Global variables */
packet_info *gpinfo;

/* Forward declarations */
static void dissect_nas_eps(tvbuff_t *tvb, packet_info *pinfo, proto_tree *tree);

/* Table 9.8.1: Message types for EPS mobility management
 *	0	1	-	-	-	-	-	-		EPS mobility management messages
 */
static const value_string nas_msg_emm_strings[] = {									
	{ 0x41,	"Attach request"},
	{ 0x42,	"Attach accept"},
	{ 0x43,	"Attach complete"},
	{ 0x44,	"Attach reject"},
	{ 0x45,	"Detach request"},
	{ 0x46,	"Detach accept"},
							
	{ 0x48,	"Tracking area update request"},
	{ 0x49,	"Tracking area update accept"},
	{ 0x4a,	"Tracking area update complete"},
	{ 0x4b,	"Tracking area update reject"},
							
	{ 0x4c,	"Extended service request"},
	{ 0x4e,	"Service reject"},
									
	{ 0x50,	"GUTI reallocation command"},
	{ 0x51,	"GUTI reallocation complete"},
	{ 0x52,	"Authentication request"},
	{ 0x53,	"Authentication response"},
	{ 0x54,	"Authentication reject"},
	{ 0x5a,	"Authentication failure"},
	{ 0x55,	"Identity request"},
	{ 0x56,	"Identity response"},
	{ 0x5d,	"Security mode command"},
	{ 0x5e,	"Security mode complete"},
	{ 0x5f,	"Security mode reject"},
									
	{ 0x60,	"EMM status"},
	{ 0x61,	"EMM information"},
	{ 0x62,	"Downlink NAS transport"},
	{ 0x63,	"Uplink NAS transport"},
	{ 0x64, "CS Service notification"},
	{ 0,	NULL }
};

/* Table 9.8.2: Message types for EPS session management */

static const value_string nas_msg_esm_strings[] = {	
	{ 0xc1,	"Activate default EPS bearer context request"},
	{ 0xc2,	"Activate default EPS bearer context accept"},
	{ 0xc3,	"Activate default EPS bearer context reject"},
	{ 0xc5,	"Activate dedicated EPS bearer context request"},
	{ 0xc6,	"Activate dedicated EPS bearer context accept"},
	{ 0xc7,	"Activate dedicated EPS bearer context reject"},
	{ 0xc9,	"Modify EPS bearer context request"},
	{ 0xca,	"Modify EPS bearer context accept"},
	{ 0xcb,	"Modify EPS bearer context reject"},
	{ 0xcd,	"Deactivate EPS bearer context request"},
	{ 0xce,	"Deactivate EPS bearer context accept"},
	{ 0xd0,	"PDN connectivity request"},
	{ 0xd1,	"PDN connectivity reject"},
	{ 0xd2,	"PDN disconnect request"},
	{ 0xd3,	"PDN disconnect reject"},
	{ 0xd4,	"Bearer resource allocation request"},
	{ 0xd5,	"Bearer resource allocation reject"},
	{ 0xd6,	"Bearer resource modification request"},
	{ 0xd7,	"Bearer resource modification reject"},
	{ 0xd9,	"ESM information request"},
	{ 0xda,	"ESM information response"},
	{ 0xe8,	"ESM status"},
	{ 0,	NULL }
};

static const value_string security_header_type_vals[] = {
	{ 0,	"Not security protected, plain NAS message"},
	{ 1,	"Security protected NAS message"},
	{ 2,	"Reserved"},
	{ 3,	"Reserved"},
	{ 4,	"Reserved"},
	{ 5,	"Reserved"},
	{ 6,	"Reserved"},
	{ 7,	"Reserved"},
	{ 8,	"Reserved"},
	{ 9,	"Reserved"},
	{ 10,	"Reserved"},
	{ 11,	"Reserved"},
	{ 12,	"Security header for the SERVICE REQUEST message "},
	{ 13,	"These values are not used in this version of the protocol. If received they shall be interpreted as \"1100\". (NOTE)"},
	{ 14,	"These values are not used in this version of the protocol. If received they shall be interpreted as \"1100\". (NOTE)"},
	{ 15,	"These values are not used in this version of the protocol. If received they shall be interpreted as \"1100\". (NOTE)"},
	{ 0,	NULL }
};

const value_string nas_eps_common_elem_strings[] = {
	{ 0x00,	"EPS bearer context status" },		/* 9.9.2.1	EPS bearer context status */
	{ 0x00,	"Location area identification" },	/* 9.9.2.2	Location area identification */
	{ 0x00,	"Mobile identity" },				/* 9.9.2.3	Mobile identity */
	{ 0x00, "Mobile station classmark 2" },		/* 9.9.2.4	Mobile station classmark 2 */
	{ 0x00, "Mobile station classmark 3" },		/* 9.9.2.5	Mobile station classmark 3 */
	{ 0x00,	"PLMN list" },						/* 9.9.2.5	PLMN list */
	{ 0x00, "Supported codec list" },			/* 9.9.2.8	Supported codec list */
	{ 0, NULL }
};
#define	NUM_NAS_EPS_COMMON_ELEM (sizeof(nas_eps_common_elem_strings)/sizeof(value_string))
gint ett_nas_eps_common_elem[NUM_NAS_EPS_COMMON_ELEM];

typedef enum
{
	DE_EPS_CMN_EPS_BE_CTX_STATUS,				/* 9.9.2.1	EPS bearer context status */
	DE_EPS_CMN_LOC_AREA_ID,						/* 9.9.2.2	Location area identification */
	DE_EPS_CMN_MOB_ID,							/* 9.9.2.3	Mobile identity */
	DE_EPS_MS_CM_2,								/* 9.9.2.4	Mobile station classmark 2 */
	DE_EPS_MS_CM_3,								/* 9.9.2.5	Mobile station classmark 3 */
	DE_EPS_CMN_PLM_LST,							/* 9.9.2.6	PLMN list */
	DE_EPS_CMN_SUP_CODEC_LST,					/* 9.9.2.6	9.9.2.8	Supported codec list */
	DE_EPS_COMMON_NONE							/* NONE */
}
nas_eps_common_elem_idx_t;
/* 
 * 9.9.2	Common information elements
 */

/*
 * 9.9.2.1	EPS bearer context status
 */
static guint16
de_eps_cmn_eps_be_ctx_status(tvbuff_t *tvb, proto_tree *tree, guint32 offset, guint len _U_, gchar *add_string _U_, int string_len _U_)
{
	guint32	curr_offset;

	curr_offset = offset;

	proto_tree_add_text(tree, tvb, curr_offset, len , "Not decoded yet");

	return len;
}
/*
 * 9.9.2.2	Location area identification
 * See subclause 10.5.1.3 in 3GPP TS 24.008 [6].
 */
/*
 * 9.9.2.3	Mobile identity
 * See subclause 10.5.1.4 in 3GPP TS 24.008 [6].
 * exported from gsm_a_common
 */

/*
 * 9.9.2.4	Mobile station classmark 2
 * See subclause 10.5.1.6 in 3GPP TS 24.008 [13].
 */
/*
 * 9.9.2.5	Mobile station classmark 3
 * See subclause 10.5.1.7 in 3GPP TS 24.008 [13].
 */
/*
 * 9.9.2.6	PLMN list
 * See subclause 10.5.1.13 in 3GPP TS 24.008 [6].
 */
/*
 * 9.9.2.7	Spare half octet
 * This element is used in the description of EMM and ESM messages when an odd number of 
 * half octet type 1 information elements are used. This element is filled with spare bits 
 * set to zero and is placed in bits 5 to 8 of the octet unless otherwise specified.
 *
 */
/*
 * 9.9.2.8	Supported codec list
 * See subclause 10.5.4.32 in 3GPP TS 24.008 [13].
 * Dissectecd in packet-gsm_a_dtap.c
 */

guint16 (*nas_eps_common_elem_fcn[])(tvbuff_t *tvb, proto_tree *tree, guint32 offset, guint len, gchar *add_string, int string_len) = {
	/* 9.9.2	Common information elements */
	de_eps_cmn_eps_be_ctx_status,	/* 9.9.2.1	EPS bearer context status */
	de_lai,							/* 9.9.2.2	Location area identification */
	de_mid,							/* 9.9.2.3	Mobile identity See subclause 10.5.1.4 in 3GPP TS 24.008*/
	de_ms_cm_2,						/* 9.9.2.4	Mobile station classmark 2 */
	de_ms_cm_3,						/* 9.9.2.5	Mobile station classmark 3 */
	de_plmn_list,					/* 9.9.2.6	PLMN list */
	NULL,							/* 9.9.2.8	Supported codec list (packet-gsm_a_dtap.c) */
	NULL,	/* NONE */
};

const value_string nas_emm_elem_strings[] = {
	/* 9.9.3	EPS Mobility Management (EMM) information elements */
	{ 0x00,	"Authentication failure parameter" },	/* 9.9.3.1	Authentication failure parameter */
	{ 0x00,	"Authentication parameter AUTN" },		/* 9.9.3.2	Authentication parameter AUTN */
	{ 0x00,	"Authentication parameter RAND" },		/* 9.9.3.3	Authentication parameter RAND */
	{ 0x00,	"Authentication response parameter" },	/* 9.9.3.4	Authentication response parameter */
	{ 0x00,	"CSFB response" },						/* 9.9.3.5	CSFB response */
	{ 0x00,	"Daylight saving time" },				/* 9.9.3.6	Daylight saving time */
	{ 0x00,	"Detach type" },						/* 9.9.3.7	Detach type */
	{ 0x00,	"DRX parameter" },						/* 9.9.3.8	DRX parameter */
	{ 0x00,	"EMM cause" },							/* 9.9.3.9	EMM cause */
	{ 0x00,	"EPS attach result" },					/* 9.9.3.10	EPS attach result */
	{ 0x00,	"EPS attach type" },					/* 9.9.3.11	EPS attach type */
	{ 0x00,	"EPS mobile identity" },				/* 9.9.3.12	EPS mobile identity */
	{ 0x00,	"EPS update resul" },					/* 9.9.3.13	EPS update result */
	{ 0x00,	"EPS update type" },					/* 9.9.3.14	EPS update type */
	{ 0x00,	"ESM message container" },				/* 9.9.3.15	ESM message conta */
	{ 0x00,	"GPRS timer" },							/* 9.9.3.16	GPRS timer ,See subclause 10.5.7.3 in 3GPP TS 24.008 [6]. */
	{ 0x00,	"Identity type 2" },					/* 9.9.3.17	Identity type 2 ,See subclause 10.5.5.9 in 3GPP TS 24.008 [6]. */
	{ 0x00,	"IMEISV request" },						/* 9.9.3.18	IMEISV request ,See subclause 10.5.5.10 in 3GPP TS 24.008 [6]. */
	{ 0x00,	"KSI and sequence number" },			/* 9.9.3.19	KSI and sequence number */
	{ 0x00,	"MS network capability" },				/* 9.9.3.20	MS network capability ,See subclause 10.5.5.12 in 3GPP TS 24.008 [6]. */
	{ 0x00,	"NAS key set identifier" },				/* 9.9.3.21	NAS key set identifier */
	{ 0x00, "NAS message container" },				/* 9.9.3.22	NAS message container */
	{ 0x00,	"NAS security algorithms" },			/* 9.9.3.23	NAS security algorithms */
	{ 0x00,	"Network name" },						/* 9.9.3.24	Network name, See subclause 10.5.3.5a in 3GPP TS 24.008 [6]. */
	{ 0x00,	"Nonce" },								/* 9.9.3.25	Nonce */
	{ 0x00,	"P-TMSI signature" },					/* 9.9.3.26	P-TMSI signature, See subclause 10.5.5.8 in 3GPP TS 24.008 [6]. */
	{ 0x00,	"Service type" },						/* 9.9.3.27	Service type ,See subclause 10.5.5.15 in 3GPP TS 24.008 [6]. */
	{ 0x00,	"Short MAC" },							/* 9.9.3.28	Short MAC */
	{ 0x00,	"Time zone" },							/* 9.9.3.29	Time zone, See subclause 10.5.3.8 in 3GPP TS 24.008 [6]. */
	{ 0x00,	"Time zone and time" },					/* 9.9.3.30	Time zone and time, See subclause 10.5.3.9 in 3GPP TS 24.008 [6]. */
	{ 0x00,	"TMSI status" },						/* 9.9.3.31	TMSI status, See subclause 10.5.5.4 in 3GPP TS 24.008 [6]. */
	{ 0x00,	"Tracking area identity" },				/* 9.9.3.32	Tracking area identity */
	{ 0x00,	"Tracking area identity list" },		/* 9.9.3.33	Tracking area identity list */
	{ 0x00,	"UE network capability" },				/* 9.9.3.34	UE network capability */
	{ 0x00,	"UE radio capability information update needed" },	/* 9.9.3.35	UE radio capability information update needed */
	{ 0x00,	"UE security capability" },				/* 9.9.3.36	UE security capability */
	{ 0, NULL }
};
#define	NUM_NAS_EMM_ELEM (sizeof(nas_emm_elem_strings)/sizeof(value_string))
gint ett_nas_eps_emm_elem[NUM_NAS_EMM_ELEM];

typedef enum
{
	/* 9.9.3	EPS Mobility Management (EMM) information elements */
	DE_EMM_AUTH_FAIL_PAR,		/* 9.9.3.1	Authentication failure parameter (dissected in packet-gsm_a_dtap.c)*/
	DE_EMM_AUTN,				/* 9.9.3.2	Authentication parameter AUTN */
	DE_EMM_AUTH_PAR_RAND,		/* 9.9.3.3	Authentication parameter RAND */
	DE_EMM_AUTH_RESP_PAR,		/* 9.9.3.4	Authentication response parameter */
	DE_EMM_CSFB_RESP,			/* 9.9.3.5	CSFB response */
	DE_EMM_DAYL_SAV_T,			/* 9.9.3.6	Daylight saving time */
	DE_EMM_DET_TYPE,			/* 9.9.3.7	Detach type */
	DE_EMM_DRX_PAR,				/* 9.9.3.8	DRX parameter (dissected in packet-gsm_a_gm.c)*/
	DE_EMM_CAUSE,				/* 9.9.3.9	EMM cause */
	DE_EMM_ATT_RES,				/* 9.9.3.10	EPS attach result (Coded inline */
	DE_EMM_ATT_TYPE,			/* 9.9.3.11	EPS attach type (Coded Inline)*/
	DE_EMM_EPS_MID,				/* 9.9.3.12	EPS mobile identity */
	DE_EMM_EPS_UPD_RES,			/* 9.9.3.13	EPS update result ( Coded inline)*/
	DE_EMM_EPS_UPD_TYPE,		/* 9.9.3.14	EPS update type */
	DE_EMM_ESM_MSG_CONT,		/* 9.9.3.15	ESM message conta */
	DE_EMM_GPRS_TIMER,			/* 9.9.3.16	GPRS timer ,See subclause 10.5.7.3 in 3GPP TS 24.008 [6]. */
	DE_EMM_ID_TYPE_2,			/* 9.9.3.17	Identity type 2 ,See subclause 10.5.5.9 in 3GPP TS 24.008 [6]. */
	DE_EMM_IMEISV_REQ,			/* 9.9.3.18	IMEISV request ,See subclause 10.5.5.10 in 3GPP TS 24.008 [6]. */
	DE_EMM_KSI_AND_SEQ_NO,		/* 9.9.3.19	KSI and sequence number */
	DE_EMM_MS_NET_CAP,			/* 9.9.3.20	MS network capability ,See subclause 10.5.5.12 in 3GPP TS 24.008 [6]. */
	DE_EMM_NAS_KEY_SET_ID,		/* 9.9.3.21	NAS key set identifier (coded inline)*/
	DE_EMM_NAS_MSG_CONT,		/* 9.9.3.22	NAS message container */
	DE_EMM_NAS_SEC_ALGS,		/* 9.9.3.23	NAS security algorithms */
	DE_EMM_NET_NAME,			/* 9.9.3.24	Network name, See subclause 10.5.3.5a in 3GPP TS 24.008 [6]. */
	DE_EMM_NONCE,				/* 9.9.3.25	Nonce */
	DE_EMM_P_TMSI_SIGN,			/* 9.9.3.26	P-TMSI signature, See subclause 10.5.5.8 in 3GPP TS 24.008 [6]. */
	DE_EMM_SERV_TYPE,			/* 9.9.3.27	Service type */
	DE_EMM_SHORT_MAC,			/* 9.9.3.28	Short MAC */
	DE_EMM_TZ,					/* 9.9.3.29	Time zone, See subclause 10.5.3.8 in 3GPP TS 24.008 [6]. */
	DE_EMM_TZ_AND_T,			/* 9.9.3.30	Time zone and time, See subclause 10.5.3.9 in 3GPP TS 24.008 [6]. */
	DE_EMM_TMSI_STAT,			/* 9.9.3.31	TMSI status, See subclause 10.5.5.4 in 3GPP TS 24.008 [6]. */
	DE_EMM_TRAC_AREA_ID,		/* 9.9.3.32	Tracking area identity */
	DE_EMM_TRAC_AREA_ID_LST,	/* 9.9.3.33	Tracking area identity list */
	DE_EMM_UE_NET_CAP,			/* 9.9.3.34	UE network capability */
	DE_EMM_UE_RA_CAP_INF_UPD_NEED,	/* 9.9.3.35	UE radio capability information update needed */
	DE_EMM_UE_SEC_CAP,			/* 9.9.3.36	UE security capability */
	DE_EMM_NONE					/* NONE */
}
nas_emm_elem_idx_t;

/* TODO: Update to latest spec */
/* 9.9.3	EPS Mobility Management (EMM) information elements
 * 9.9.3.1	Authentication failure parameter
 * See subclause 10.5.3.2.2 in 3GPP TS 24.008 [6].
 * (dissected in packet-gsm_a_dtap.c)
 */
/*
 * 9.9.3.2	Authentication parameter AUTN
 * See subclause 10.5.3.1.1 in 3GPP TS 24.008 [6].
 */
/*
 * 9.9.3.3	Authentication parameter RAND
 * See subclause 10.5.3.1 in 3GPP TS 24.008 [6].
 */
/*
 * 9.9.3.4	Authentication response parameter
 */
static guint16
de_emm_auth_resp_par(tvbuff_t *tvb, proto_tree *tree, guint32 offset, guint len _U_, gchar *add_string _U_, int string_len _U_)
{
	guint32	curr_offset;

	curr_offset = offset;

	proto_tree_add_item(tree, hf_nas_eps_emm_res, tvb, curr_offset, len, FALSE);

	return len;
}
/*
 * 9.9.3.5	CSFB response
 */
static guint16
de_emm_csfb_resp(tvbuff_t *tvb, proto_tree *tree, guint32 offset, guint len _U_, gchar *add_string _U_, int string_len _U_)
{
	guint32	curr_offset;

	curr_offset = offset;


	proto_tree_add_text(tree, tvb, curr_offset, 1, "Not decoded yet");
	curr_offset++;

	return(curr_offset-offset);
}
/*
 * 9.9.3.6	Daylight saving time
 * See subclause 10.5.3.12 in 3GPP TS 24.008 [6].
 */
/*
 * 9.9.3.7	Detach type
 */
/*
Type of detach (octet 1)

In the UE to network direction:
Bits
3	2	1		
0	0	1		EPS detach
0	1	0		IMSI detach
0	1	1		combined EPS/IMSI detach

All other values are interpreted as "combined EPS/IMSI detach" in this version of the protocol.

In the network to UE direction:
Bits
3	2	1		
0	0	1		re-attach required
0	1	0		re-attach not required
0	1	1		IMSI detach

All other values are interpreted as "re-attach not required" in this version of the protocol.

Switch off (octet 1)

In the UE to network direction:
Bit
4				
0				normal detach
1				switch off

In the network to UE direction bit 4 is spare. The network shall set this bit to zero.
*/
/*
 * 9.9.3.8	DRX parameter
 * See subclause 10.5.5.6 in 3GPP TS 24.008 [13].
 */
/*
 * 9.9.3.9	EMM cause
 */
static const value_string nas_eps_emm_cause_values[] = {
	{ 0x2,	"IMSI unknown in HLR"},
	{ 0x3,	"Illegal MS"},
	{ 0x6,	"Illegal ME"},
	{ 0x7,	"EPS services not allowed"},
	{ 0x8,	"EPS services and non-EPS services not allowed"},
	{ 0x9,	"UE identity cannot be derived by the network"},
	{ 0xa,	"Implicitly detached"},
	{ 0xb,	"PLMN not allowed"},
	{ 0xc,	"Tracking Area not allowed"},
	{ 0xd,	"Roaming not allowed in this tracking area"},
	{ 0xe,	"EPS services not allowed in this PLMN"},
	{ 0xf,	"No Suitable Cells In tracking area"},
	{ 0x10,	"MSC temporarily not reachable"},
	{ 0x11,	"Network failure"},
	{ 0x12,	"CS domain not available"},
	{ 0x13,	"ESM failure"},
	{ 0x14,	"MAC failure"},
	{ 0x15,	"Synch failure"},
	{ 0x16,	"Congestion"},
	{ 0x17,	"UE security capabilities mismatch"},
	{ 0x18,	"Security mode rejected, unspecified"},
	{ 0x19,	"Not authorized for this CSG"},
	{ 0x26,	"CS fallback call establishment not allowed"},
	{ 0x27,	"CS domain temporarily not available"},
	{ 0x28,	"No EPS bearer context activated"},
	{ 0x5f,	"Semantically incorrect message"},
	{ 0x60,	"Invalid mandatory information"},
	{ 0x61,	"Message type non-existent or not implemented"},
	{ 0x62,	"Message type not compatible with the protocol state"},
	{ 0x63,	"Information element non-existent or not implemented"},
	{ 0x64,	"Conditional IE error"},
	{ 0x65,	"Message not compatible with the protocol state"},
	{ 0x6f,	"Protocol error, unspecified"},
	{ 0, NULL }
};

static guint16
de_emm_cause(tvbuff_t *tvb, proto_tree *tree, guint32 offset, guint len _U_, gchar *add_string _U_, int string_len _U_)
{
	guint32	curr_offset;

	curr_offset = offset;

	proto_tree_add_item(tree, hf_nas_eps_emm_cause, tvb, curr_offset, 1, FALSE);
	curr_offset++;

	return curr_offset-offset;}
/*
 * 9.9.3.10	EPS attach result
 */

static const value_string nas_eps_emm_EPS_attach_result_values[] = {
	{ 0,	"reserved"},
	{ 1,	"EPS only"},
	{ 2,	"Combined EPS/IMSI attach"},
	{ 3,	"reserved"},
	{ 4,	"reserved"},
	{ 5,	"reserved"},
	{ 6,	"reserved"},
	{ 7,	"reserved"},
	{ 0, NULL }
};
/* Coded inline */

/*
 * 9.9.3.11	EPS attach type
 */

static const value_string nas_eps_emm_eps_att_type_vals[] = {
	{ 0,	"EPS attach(unused)"},
	{ 1,	"EPS attach"},
	{ 2,	"EPS attach(unused)"},
	{ 3,	"EPS attach(unused)"},
	{ 4,	"Combined handover EPS/IMSI attach"},
	{ 5,	"EPS attach(unused)"},
	{ 6,	"EPS attach(unused)"},
	{ 7,	"EPS attach(unused)"},
	{ 0, NULL }
};
/* Coded inline */

/*
 * 9.9.3.12	EPS mobile identity
 */

static const value_string nas_eps_emm_type_of_id_vals[] = {
	{ 0,	"reserved"},
	{ 1,	"IMSI"},
	{ 2,	"reserved"},
	{ 3,	"reserved"},
	{ 4,	"reserved"},
	{ 5,	"reserved"},
	{ 6,	"GUTI"},
	{ 7,	"reserved"},
	{ 0, NULL }
};
static guint16
de_emm_eps_mid(tvbuff_t *tvb, proto_tree *tree, guint32 offset, guint len _U_, gchar *add_string _U_, int string_len _U_)
{
	guint32	curr_offset;
	guint8 octet;

	curr_offset = offset;

	octet = tvb_get_guint8(tvb,offset);
	/* Type of identity (octet 3) */
	proto_tree_add_item(tree, hf_nas_eps_emm_odd_even, tvb, curr_offset, 1, FALSE);
	proto_tree_add_item(tree, hf_nas_eps_emm_type_of_id, tvb, curr_offset, 1, FALSE);
	curr_offset++;
	switch (octet&0x7){
		case 1:
			/* IMSI */
			proto_tree_add_text(tree, tvb, curr_offset, len - 1, "Not decoded yet");
			break;
		case 6:
			/* GUTI */
			curr_offset = dissect_e212_mcc_mnc(tvb, tree, curr_offset);
			/* MME Group ID octet 7 - 8 */
			proto_tree_add_item(tree, hf_nas_eps_emm_mme_grp_id, tvb, curr_offset, 2, FALSE);
			curr_offset+=2;
			/* MME Code Octet 9 */
			proto_tree_add_item(tree, hf_nas_eps_emm_mme_code, tvb, curr_offset, 1, FALSE);
			offset++;
			/* M-TMSI Octet 10 - 13 */
			proto_tree_add_item(tree, hf_nas_eps_emm_m_tmsi, tvb, curr_offset, 4, FALSE);
			offset+=3;
			break;
		default:
			proto_tree_add_text(tree, tvb, curr_offset, len - 1, "Type of identity not known");
			break;
	}
	
	return(len);
}
/*
 * 9.9.3.13	EPS update result
 */
static const value_string nas_eps_emm_eps_update_result_vals[] = {
	{ 0,	"TA updated"},
	{ 1,	"Combined TA/LA updated"},
	{ 2,	"TA updated and ISR activated"},
	{ 3,	"Combined TA/LA updated and ISR activated"},
	{ 0, NULL }
};

/*
 * 9.9.3.14	EPS update type
 */
static const true_false_string  nas_eps_emm_active_flg_value = {
	"Bearer establishment requested",
	"No bearer establishment requested"
};

static const value_string nas_eps_emm_eps_update_type_vals[] = {
	{ 0,	"TA updating"},
	{ 1,	"Combined TA/LA updating"},
	{ 2,	"Combined TA/LA updating with IMSI attach"},
	{ 3,	"Periodic updating"},
	{ 0, NULL }
};

/*
 * 9.9.3.15	ESM message container
 */
static guint16
de_emm_esm_msg_cont(tvbuff_t *tvb, proto_tree *tree, guint32 offset, guint len, gchar *add_string _U_, int string_len _U_)
{
	proto_item *item;
	proto_tree *sub_tree;
	tvbuff_t	*new_tvb;
	guint32	curr_offset;

	curr_offset = offset;


	item = proto_tree_add_item(tree, hf_nas_eps_esm_msg_cont, tvb, curr_offset, len, FALSE);
	sub_tree = proto_item_add_subtree(item, ett_nas_eps_esm_msg_cont);

	/* This IE can contain any ESM PDU as defined in subclause 8.3. */
	new_tvb = tvb_new_subset(tvb, curr_offset, len, len );
	dissect_nas_eps(new_tvb, gpinfo, sub_tree);

	return(len);
}
/*
 * 9.9.3.16	GPRS timer
 * See subclause 10.5.7.3 in 3GPP TS 24.008 [6].
 * packet-gsm_a_gm.c
 */
/*
 * 9.9.3.17	Identity type 2
 * See subclause 10.5.5.9 in 3GPP TS 24.008 [6].
 */
static const value_string nas_eps_emm_id_type2_vals[] = {
	{ 1,	"IMSI"},
	{ 2,	"IMEI"},
	{ 3,	"IMEISV"},
	{ 4,	"TMSI"},
	{ 0, NULL }
};

/*
 * 9.9.3.18	IMEISV request
 * See subclause 10.5.5.10 in 3GPP TS 24.008 [6].
 */
/*
 * 9.9.3.19	KSI and sequence number
 */

/*
 * 9.9.3.20	MS network capability
 * See subclause 10.5.5.12 in 3GPP TS 24.008 [6].
 */
/*
 * 9.9.3.21	NAS key set identifier
 */
/*
Bit
4			
0			cached security context
1			mapped security context
*/

static const value_string nas_eps_emm_NAS_key_set_identifier_vals[] = {
	{ 0,	""},
	{ 1,	""},
	{ 2,	""},
	{ 3,	""},
	{ 4,	""},
	{ 5,	""},
	{ 6,	""},
	{ 7,	"No key is available"},
	{ 0, NULL }
};
/* Coded Inline */

/*
 * 9.9.3.22	NAS message container
 */
static guint16
de_emm_nas_msg_cont(tvbuff_t *tvb, proto_tree *tree, guint32 offset, guint len _U_, gchar *add_string _U_, int string_len _U_)
{
	guint32	curr_offset;

	curr_offset = offset;


	proto_tree_add_text(tree, tvb, curr_offset, len , "Not decoded yet");
	

	return(len);
}
/*
 * 9.9.3.23	NAS security algorithms
 */
static guint16
de_emm_nas_sec_alsgs(tvbuff_t *tvb, proto_tree *tree, guint32 offset, guint len _U_, gchar *add_string _U_, int string_len _U_)
{
	guint32	curr_offset;

	curr_offset = offset;


	proto_tree_add_text(tree, tvb, curr_offset, 1 , "Not decoded yet");
	curr_offset++;

	return(curr_offset-offset);
}
/*
 * 9.9.3.24	Network name
 * See subclause 10.5.3.5a in 3GPP TS 24.008 [6].
 */
/*
 * 9.9.3.25	Nonce
 * Editor's note: The coding of this information element is FFS.
 */
static guint16
de_emm_nonce(tvbuff_t *tvb, proto_tree *tree, guint32 offset, guint len _U_, gchar *add_string _U_, int string_len _U_)
{
	guint32	curr_offset;

	curr_offset = offset;


	proto_tree_add_text(tree, tvb, curr_offset, 5 , "Not decoded yet");
	curr_offset+=5;

	return(len);
}
/*
 * 9.9.3.26	P-TMSI signature
 * See subclause 10.5.5.8 in 3GPP TS 24.008 [6].
 */
/*
 * 9.9.3.27	Service type
 */
static const value_string nas_eps_service_type_vals[] = {
	{ 0,	"Mobile originating CS fallback"},
	{ 1,	"Mobile terminating CS fallback"},
	{ 2,	"Mobile originating CS fallback emergency call"},
	{ 0, NULL }
};

/*
 * 9.9.3.28	Short MAC
 */
static guint16
de_emm_nas_short_mac(tvbuff_t *tvb, proto_tree *tree, guint32 offset, guint len _U_, gchar *add_string _U_, int string_len _U_)
{
	guint32	curr_offset;

	curr_offset = offset;


	proto_tree_add_item(tree, hf_nas_eps_emm_short_mac, tvb, curr_offset, 2, FALSE);
	curr_offset+=2;

	return(curr_offset-offset);
}
/*
 * 9.9.3.29	Time zone
 * See subclause 10.5.3.8 in 3GPP TS 24.008 [6].
 */
/*
 * 9.9.3.30	Time zone and time
 * See subclause 10.5.3.9 in 3GPP TS 24.008 [6].
 */
/*
 * 9.9.3.31	TMSI status
 * See subclause 10.5.5.4 in 3GPP TS 24.008 [6].
 */
/*
 * 9.9.3.32	Tracking area identity
 */
static guint16
de_emm_trac_area_id(tvbuff_t *tvb, proto_tree *tree, guint32 offset, guint len _U_, gchar *add_string _U_, int string_len _U_)
{
	guint32	curr_offset;

	curr_offset = offset;


	proto_tree_add_text(tree, tvb, curr_offset, 6 , "Not decoded yet");
	curr_offset+=6;

	return(curr_offset-offset);
}
/*
 * 9.9.3.33	Tracking area identity list
 */
static guint16
de_emm_trac_area_id_lst(tvbuff_t *tvb, proto_tree *tree, guint32 offset, guint len _U_, gchar *add_string _U_, int string_len _U_)
{
	guint32	curr_offset;

	curr_offset = offset;


	proto_tree_add_text(tree, tvb, curr_offset, len , "Not decoded yet");

	return(len);
}
/*
 * 9.9.3.34	UE network capability 
 */
/*
EPS integrity algorithms supported (octet 4)
Bit 8 of octet 4 is spare and shall be coded as zero.
EPS integrity algorithm 128-EIA1 supported (octet 4, bit 7)
0 EPS integrity algorithm 128-EIA1 not supported
1 EPS integrity algorithm 128-EIA1 supported
EPS integrity algorithm 128-EIA2 supported (octet 4, bit 6)
0 EPS integrity algorithm 128-EIA2 not supported
1 EPS integrity algorithm 128-EIA2 supported
EPS integrity algorithm EIA3 supported (octet 4, bit 5)
0 EPS integrity algorithm EIA3 not supported
1 EPS integrity algorithm EIA3 supported
EPS integrity algorithm EIA4 supported (octet 4, bit 4)
0 EPS integrity algorithm EIA4 not supported
1 EPS integrity algorithm EIA4 supported
EPS integrity algorithm EIA5 supported (octet 4, bit 3)
0 EPS integrity algorithm EIA5 not supported
1 EPS integrity algorithm EIA5 supported
EPS integrity algorithm EIA6 supported (octet 4, bit 2)
0 EPS integrity algorithm EIA6 not supported
1 EPS integrity algorithm EIA6 supported
EPS integrity algorithm EIA7 supported (octet 4, bit 1)
0 EPS integrity algorithm EIA7 not supported
1 EPS integrity algorithm EIA7 supported
*/
static const true_false_string  nas_eps_emm_supported_flg_value = {
	"Supported",
	"Not Supported"
};
static guint16
de_emm_ue_net_cap(tvbuff_t *tvb, proto_tree *tree, guint32 offset, guint len _U_, gchar *add_string _U_, int string_len _U_)
{
	guint32	curr_offset;

	curr_offset = offset;


	/* EPS encryption algorithms supported (octet 3) */
	/* EPS encryption algorithm 128-EEA0 supported (octet 3, bit 8) */
	proto_tree_add_item(tree, hf_nas_eps_emm_128eea0, tvb, curr_offset, 1, FALSE);
	/* EPS encryption algorithm 128-EEA1 supported (octet 3, bit 7) */
	proto_tree_add_item(tree, hf_nas_eps_emm_128eea1, tvb, curr_offset, 1, FALSE);
	/* EPS encryption algorithm 128-EEA2 supported (octet 3, bit 6) */
	proto_tree_add_item(tree, hf_nas_eps_emm_128eea2, tvb, curr_offset, 1, FALSE);
	/* EPS encryption algorithm 128-EEA3 supported (octet 3, bit 5) */
	proto_tree_add_item(tree, hf_nas_eps_emm_128eea3, tvb, curr_offset, 1, FALSE);
	/* EPS encryption algorithm 128-EEA4 supported (octet 3, bit 4) */
	proto_tree_add_item(tree, hf_nas_eps_emm_128eea4, tvb, curr_offset, 1, FALSE);
	/* EPS encryption algorithm 128-EEA5 supported (octet 3, bit 5) */
	proto_tree_add_item(tree, hf_nas_eps_emm_128eea5, tvb, curr_offset, 1, FALSE);
	/* EPS encryption algorithm 128-EEA6 supported (octet 3, bit 6) */
	proto_tree_add_item(tree, hf_nas_eps_emm_128eea6, tvb, curr_offset, 1, FALSE);
	/* EPS encryption algorithm 128-EEA7 supported (octet 3, bit 7) */
	proto_tree_add_item(tree, hf_nas_eps_emm_128eea7, tvb, curr_offset, 1, FALSE);
	curr_offset++;


	/* EPS integrity algorithms supported (octet 4)
	* Bit 8 of octet 4 is spare and shall be coded as zero.
	* EPS integrity algorithm 128-EIA1 supported (octet 4, bit 7)
	*/
	proto_tree_add_item(tree, hf_nas_eps_emm_128eia1, tvb, curr_offset, 1, FALSE);
	/* EPS integrity algorithm 128-EIA2 supported (octet 4, bit 6) */
	proto_tree_add_item(tree, hf_nas_eps_emm_128eia2, tvb, curr_offset, 1, FALSE);
	/* EPS integrity algorithm EIA3 supported (octet 4, bit 5) */
	proto_tree_add_item(tree, hf_nas_eps_emm_128eia3, tvb, curr_offset, 1, FALSE);
	/* EPS integrity algorithm EIA4 supported (octet 4, bit 4) */
	proto_tree_add_item(tree, hf_nas_eps_emm_128eia4, tvb, curr_offset, 1, FALSE);
	/* EPS integrity algorithm EIA5 supported (octet 4, bit 3) */
	proto_tree_add_item(tree, hf_nas_eps_emm_128eia5, tvb, curr_offset, 1, FALSE);
	/* EPS integrity algorithm EIA6 supported (octet 4, bit 2) */
	proto_tree_add_item(tree, hf_nas_eps_emm_128eia6, tvb, curr_offset, 1, FALSE);
	/* EPS integrity algorithm EIA7 supported (octet 4, bit 1) */
	proto_tree_add_item(tree, hf_nas_eps_emm_128eia7, tvb, curr_offset, 1, FALSE);
	curr_offset++;


	/* UMTS encryption algorithms supported (octet 5)
	 * UMTS encryption algorithm UEA0 supported (octet 5, bit 8)
	 */
	/* UMTS encryption algorithm 128-UEA0 supported (octet 5, bit 8) */
	proto_tree_add_item(tree, hf_nas_eps_emm_128uea0, tvb, curr_offset, 1, FALSE);
	/* UMTS encryption algorithm 128-UEA0 supported (octet 5, bit 7) */
	proto_tree_add_item(tree, hf_nas_eps_emm_128uea1, tvb, curr_offset, 1, FALSE);
	/* UMTS encryption algorithm 128-UEA0 supported (octet 5, bit 6) */
	proto_tree_add_item(tree, hf_nas_eps_emm_128uea2, tvb, curr_offset, 1, FALSE);
	/* UMTS encryption algorithm 128-UEA0 supported (octet 5, bit 5) */
	proto_tree_add_item(tree, hf_nas_eps_emm_128uea3, tvb, curr_offset, 1, FALSE);
	/* EPS encryption algorithm 128-UEA0 supported (octet 5, bit 4) */
	proto_tree_add_item(tree, hf_nas_eps_emm_128uea4, tvb, curr_offset, 1, FALSE);
	/* UMTS encryption algorithm 128-UEA0 supported (octet 5, bit 5) */
	proto_tree_add_item(tree, hf_nas_eps_emm_128uea5, tvb, curr_offset, 1, FALSE);
	/* UMTS encryption algorithm 128-UEA0 supported (octet 5, bit 6) */
	proto_tree_add_item(tree, hf_nas_eps_emm_128uea6, tvb, curr_offset, 1, FALSE);
	/* UMTS encryption algorithm 128-UEA0 supported (octet 5, bit 7) */
	proto_tree_add_item(tree, hf_nas_eps_emm_128uea7, tvb, curr_offset, 1, FALSE);
	curr_offset++;

	/* UCS2 support (UCS2) (octet 6, bit 8)
	 * This information field indicates the likely treatment of UCS2 encoded character strings
	 * by the UE.
	 */
	proto_tree_add_text(tree, tvb, curr_offset, len-3 , "Not decoded yet");

	return(len);
}
/*
 * 9.9.3.35	UE radio capability information update needed
 */

static guint16
de_emm_ue_ra_cap_inf_upd_need(tvbuff_t *tvb, proto_tree *tree, guint32 offset, guint len _U_, gchar *add_string _U_, int string_len _U_)
{
	guint32	curr_offset;

	curr_offset = offset;


	proto_tree_add_text(tree, tvb, curr_offset, len , "Not decoded yet");

	return(len);
}
/*
 * 9.9.3.36	UE security capability
 */

static guint16
de_emm_ue_sec_cap(tvbuff_t *tvb, proto_tree *tree, guint32 offset, guint len _U_, gchar *add_string _U_, int string_len _U_)
{
	guint32	curr_offset;

	curr_offset = offset;


	proto_tree_add_text(tree, tvb, curr_offset, len , "Not decoded yet");

	return(len);
}
/*
 * 9.9.3.37	Emergency Number List
 * See subclause 10.5.3.13 in 3GPP TS 24.008 [13].
 */

/*
 * 9.9.3.38	CLI
 */
/*
 * The coding of the CLI value part is the same as for octets 3 to 14
 * of the Calling party BCD number information element defined in 
 * subclause 10.5.4.9 of 3GPP TS 24.008
 */

/*
 * 9.9.3.39	SS Code
 */
/*
SS Code value

The coding of the SS Code value is given in subclause 17.7.5 of 3GPP TS 29.002 [15B].
*/
/*
 * 9.9.3.40	LCS indicator
 */

/*
 * 9.9.3.41	LCS client identity
 */
/*
LCS client identity (value part)

The coding of the value part of the LCS client identity is given in subclause 17.7.13 of 3GPP TS 29.002 [15B].
*/

/*
 * 9.9.4	EPS Session Management (ESM) information elements
 */

/*
 * 9.9.4.1 Access point name
 * See subclause 10.5.6.1 in 3GPP TS 24.008 [6].
 */
/*
 * 9.9.4.2 APN aggregate maximum bit rate
 * 9.9.4.3 EPS quality of service
 * 9.9.4.1 Access point name
 * 9.9.4.2 APN aggregate maximum bit rate 
 * 9.9.4.3 EPS quality of service 
 * 9.9.4.4 ESM cause
 */
/*
 * 9.9.4.5 ESM information transfer flag 
 */
static guint16
de_esm_inf_trf_flg(tvbuff_t *tvb, proto_tree *tree, guint32 offset, guint len _U_, gchar *add_string _U_, int string_len _U_)
{
	guint32	curr_offset;

	curr_offset = offset;


	proto_tree_add_text(tree, tvb, curr_offset, len , "Not decoded yet");

	return(len);
}
/*
 * 9.9.4.6 Linked EPS bearer identity 
 * 9.9.4.7 LLC service access point identifier 
 * See subclause 10.5.6.9 in 3GPP TS 24.008
 * 9.9.4.8 Packet flow identifier 
 * See subclause 10.5.6.11 in 3GPP TS 24.008 
 * 9.9.4.9 PDN address
 *
 * 9.9.4.10 PDN type
 * Coded inline 1/2 octet
 */
static const value_string nas_eps_esm_pdn_type_values[] = {
	{ 0x1,	"IPv4" },
	{ 0x2,	"IPv6" },
	{ 0x3,	"IPv4v6" },
	{ 0, NULL }
};

/*
 * 9.9.4.11 Protocol configuration options 
 * See subclause 10.5.6.3 in 3GPP TS 24.008
 */
/*
 * 9.9.4.12 Quality of service
 * See subclause 10.5.6.5 in 3GPP TS 24.008
 */
/*
 * 9.9.4.13 Radio priority 
 * See subclause 10.5.7.2 in 3GPP TS 24.008
 */
/*
 * 9.9.4.14 Request type
 * Coded inline 1/2 octet
 */
static const value_string nas_eps_esm_request_type_values[] = {
	{ 0x1,	"Initial attach" },
	{ 0x2,	"Handover" },
	{ 0, NULL }
};
/*
 * 9.9.4.15 Traffic flow aggregate description 
 * The Traffic flow aggregate description information element is encoded using the same format as the Traffic flow
 * template information element (see subclause 10.5.6.12 in 3GPP TS 24.008 [13]). When sending this IE, the UE shall
 * assign the packet filter identifier values so that they are unique across all packet filters for the PDN connection.
 */
/*
 * 9.9.4.16 Traffic flow template
 * See subclause 10.5.6.12 in 3GPP TS 24.008
 */
/*
 * 9.9.4.17 Transaction identifier 
 * The Transaction identifier information element is coded as the Linked TI information element in 3GPP TS 24.008 [13],
 * subclause 10.5.6.7.
 */

guint16 (*emm_elem_fcn[])(tvbuff_t *tvb, proto_tree *tree, guint32 offset, guint len, gchar *add_string, int string_len) = {
	/* 9.9.3	EPS Mobility Management (EMM) information elements */
	NULL,						/* 9.9.3.1	Authentication failure parameter(dissected in packet-gsm_a_dtap.c) */
	NULL,						/* 9.9.3.2	Authentication parameter AUTN(packet-gsm_a_dtap.c) */
	NULL,						/* 9.9.3.3	Authentication parameter RAND */
	de_emm_auth_resp_par,		/* 9.9.3.4	Authentication response parameter */
	de_emm_csfb_resp,			/* 9.9.3.5	CSFB response */
	NULL,						/* 9.9.3.6	Daylight saving time (packet-gsm_a_dtap.c)*/
	NULL,						/* 9.9.3.7	Detach type */
	NULL,						/* 9.9.3.8	DRX parameter */
	de_emm_cause,				/* 9.9.3.9	EMM cause */
	NULL,						/* 9.9.3.10	EPS attach result (coded inline) */
	NULL,						/* 9.9.3.11	EPS attach type(Coded Inline) */
	de_emm_eps_mid,				/* 9.9.3.12	EPS mobile identity */
	NULL,						/* 9.9.3.13	EPS update result (Coded Inline)*/
	NULL,						/* 9.9.3.14	EPS update type (Inline)*/
	de_emm_esm_msg_cont,		/* 9.9.3.15	ESM message conta */
	NULL,						/* 9.9.3.16	GPRS timer ,See subclause 10.5.7.3 in 3GPP TS 24.008 [6]. (packet-gsm_a_gm.c)*/
	NULL,						/* 9.9.3.17	Identity type 2 ,See subclause 10.5.5.9 in 3GPP TS 24.008 [6]. */
	NULL,						/* 9.9.3.18	IMEISV request ,See subclause 10.5.5.10 in 3GPP TS 24.008 [6]. */
	NULL,						/* 9.9.3.19	KSI and sequence number */
	NULL,						/* 9.9.3.20	MS network capability ,See subclause 10.5.5.12 in 3GPP TS 24.008 [6].(packet-gsm_a_gm.c) */
	NULL,						/* 9.9.3.21	NAS key set identifier (Coded Inline) */
	de_emm_nas_msg_cont,		/* 9.9.3.22	NAS message container */
	de_emm_nas_sec_alsgs,		/* 9.9.3.23	NAS security algorithms */
	NULL,						/* 9.9.3.24	Network name, See subclause 10.5.3.5a in 3GPP TS 24.008 [6]. (packet-gsm_a_dtap.c)*/
	de_emm_nonce,				/* 9.9.3.25	Nonce */
	NULL,						/* 9.9.3.26	P-TMSI signature, See subclause 10.5.5.8 in 3GPP TS 24.008 [6]. (packet-gsm_a_gm.c)*/
	NULL,						/* 9.9.3.27	Service type  */
	de_emm_nas_short_mac,		/* 9.9.3.28	Short MAC */
	NULL,						/* 9.9.3.29	Time zone, See subclause 10.5.3.8 in 3GPP TS 24.008 [6]. (packet-gsm_a_dtap.c)*/
	NULL,						/* 9.9.3.30	Time zone and time, See subclause 10.5.3.9 in 3GPP TS 24.008 [6]. (packet-gsm_a_dtap.c)*/
	NULL,						/* 9.9.3.31	TMSI status, See subclause 10.5.5.4 in 3GPP TS 24.008 [6]. (packet-gsm_a_gm.c)*/
	de_emm_trac_area_id,		/* 9.9.3.32	Tracking area identity */
	de_emm_trac_area_id_lst,	/* 9.9.3.33	Tracking area identity list */
	de_emm_ue_net_cap,			/* 9.9.3.34	UE network capability */
	de_emm_ue_ra_cap_inf_upd_need, /* 9.9.3.35	UE radio capability information update needed */
	de_emm_ue_sec_cap,			/* 9.9.3.36	UE security capability */
	NULL,	/* NONE */
};

/* 9.9.4 EPS Session Management (ESM) information elements */
const value_string nas_esm_elem_strings[] = {
	{ 0x00,	"Access point name" },						/* 9.9.4.1 Access point name */
	{ 0x00,	"APN aggregate maximum bit rate" },			/* 9.9.4.2 APN aggregate maximum bit rate */ 
	{ 0x00,	"EPS quality of service" },					/* 9.9.4.3 EPS quality of service */
	{ 0x00,	"ESM cause" },								/* 9.9.4.4 ESM cause */
	{ 0x00,	"ESM information transfer flag" },			/* 9.9.4.5 ESM information transfer flag */ 
	{ 0x00,	"Linked EPS bearer identity" },				/* 9.9.4.6 Linked EPS bearer identity */
	{ 0x00,	"LLC service access point identifier" },	/* 9.9.4.7 LLC service access point identifier */ 
	{ 0x00,	"Packet flow identifier" },					/* 9.9.4.8 Packet flow identifier */
	{ 0x00,	"PDN address" },							/* 9.9.4.9 PDN address */
	{ 0x00,	"PDN type" },								/* 9.9.4.10 PDN type */
	{ 0x00,	"Protocol configuration options" },			/* 9.9.4.11 Protocol configuration options */ 
	{ 0x00,	"Quality of service" },						/* 9.9.4.12 Quality of service */
	{ 0x00,	"Radio priority" },							/* 9.9.4.13 Radio priority */
	{ 0x00,	"Request type" },							/* 9.9.4.14 Request type */
	{ 0x00,	"Traffic flow aggregate description" },		/* 9.9.4.15 Traffic flow aggregate description */ 
	{ 0x00,	"Traffic flow templat" },					/* 9.9.4.16 Traffic flow template */
	{ 0x00,	"Transaction identifier" },					/* 9.9.4.17 Transaction identifier */
	{ 0, NULL }
};


#define	NUM_NAS_ESM_ELEM (sizeof(nas_esm_elem_strings)/sizeof(value_string))
gint ett_nas_eps_esm_elem[NUM_NAS_ESM_ELEM];

typedef enum
{
	DE_ESM_APN,						/* 9.9.4.1 Access point name */
	DE_ESM_APN_AGR_MAX_BR,			/* 9.9.4.2 APN aggregate maximum bit rate */
	DE_ESM_EPS_QOS,					/* 9.9.4.3 EPS quality of service */
	DE_ESM_CAUSE,					/* 9.9.4.4 ESM cause */
	DE_ESM_INF_TRF_FLG,				/* 9.9.4.5 ESM information transfer flag */ 
	DE_ESM_LNKED_EPS_B_ID,			/* 9.9.4.6 Linked EPS bearer identity  */
	DE_ESM_LLC_SAPI,				/* 9.9.4.7 LLC service access point identifier */ 
	DE_ESM_P_FLW_ID,				/* 9.9.4.8 Packet flow identifier  */
	DE_ESM_PDN_ADDR,				/* 9.9.4.9 PDN address */
	DE_ESM_PDN_TYPE,				/* 9.9.4.10 PDN type */
	DE_ESM_PROT_CONF_OPT,			/* 9.9.4.11 Protocol configuration options */ 
	DE_ESM_QOS,						/* 9.9.4.12 Quality of service */
	DE_ESM_RA_PRI,					/* 9.9.4.13 Radio priority  */
	DE_ESM_REQ_TYPE,				/* 9.9.4.14 Request type */
	DE_ESM_TRAF_FLOW_AGR_DESC,		/* 9.9.4.15 Traffic flow aggregate description */ 
	DE_ESM_TRAF_FLOW_TEMPL,			/* 9.9.4.16 Traffic flow template */
	DE_ESM_TID,						/* 9.9.4.17 Transaction identifier  */
	DE_ESM_NONE						/* NONE */
}

nas_esm_elem_idx_t;

guint16 (*esm_elem_fcn[])(tvbuff_t *tvb, proto_tree *tree, guint32 offset, guint len, gchar *add_string, int string_len) = {
	NULL,						/* 9.9.4.1 Access point name */
	NULL,			/* 9.9.4.2 APN aggregate maximum bit rate */
	NULL,					/* 9.9.4.3 EPS quality of service */
	NULL,					/* 9.9.4.4 ESM cause */
	de_esm_inf_trf_flg,				/* 9.9.4.5 ESM information transfer flag */ 
	NULL,			/* 9.9.4.6 Linked EPS bearer identity  */
	NULL,				/* 9.9.4.7 LLC service access point identifier */ 
	NULL,				/* 9.9.4.8 Packet flow identifier  */
	NULL,				/* 9.9.4.9 PDN address */
	NULL,				/* 9.9.4.10 PDN type */
	NULL,			/* 9.9.4.11 Protocol configuration options */ 
	NULL,						/* 9.9.4.12 Quality of service */
	NULL,					/* 9.9.4.13 Radio priority  */
	NULL,				/* 9.9.4.14 Request type */
	NULL,		/* 9.9.4.15 Traffic flow aggregate description */ 
	NULL,			/* 9.9.4.16 Traffic flow template */
	NULL,						/* 9.9.4.17 Transaction identifier  */
	NULL,	/* NONE */
};

/* MESSAGE FUNCTIONS */

/*
 * 8.2.1	Attach accept
 */

static void
nas_emm_attach_acc(tvbuff_t *tvb, proto_tree *tree, guint32 offset, guint len)
{
	guint32	curr_offset, bit_offset;
	guint32	consumed;
	guint	curr_len;

	curr_offset = offset;
	curr_len = len;

	/* 	EPS attach result	EPS attach result 9.9.3.10	M	V	1/2 */
	bit_offset = curr_offset<<3;
	proto_tree_add_bits_item(tree, hf_nas_eps_spare_bits, tvb, bit_offset, 1, FALSE);
	bit_offset++;
	proto_tree_add_bits_item(tree, hf_nas_eps_emm_EPS_attach_result, tvb, bit_offset, 3, FALSE);
	bit_offset+=3;
	/* 	Spare half octet	Spare half octet 9.9.2.7	M	V	1/2 */
	proto_tree_add_bits_item(tree, hf_nas_eps_emm_spare_half_octet, tvb, bit_offset, 4, FALSE);
	bit_offset+=4;
	/* Fix up the lengths */
	curr_len--;
	curr_offset++;
	/* 	T3412 value	GPRS timer 9.9.3.16	M	V	1 */
	ELEM_MAND_V(GSM_A_PDU_TYPE_GM, DE_GPRS_TIMER);
	/* 	Tracking area identity list 9.9.3.33	M	LV	7-97 */
	ELEM_MAND_LV(NAS_PDU_TYPE_EMM, DE_EMM_TRAC_AREA_ID_LST, " - TAI list");
	/* 	ESM message container 9.9.3.15	M	LV-E	2-n */
	ELEM_MAND_LV_E(NAS_PDU_TYPE_EMM, DE_EMM_ESM_MSG_CONT, "");
	/* 50	GUTI	EPS mobile identity 9.9.3.12	O	TLV	13 */
	ELEM_OPT_TLV(0x50, NAS_PDU_TYPE_EMM, DE_EMM_EPS_MID, "GUTI");
	/* 13	Location area identification	Location area identification 9.9.2.2	O	TV	6 */
	ELEM_OPT_TV(0x13, NAS_PDU_TYPE_COMMON, DE_EPS_CMN_LOC_AREA_ID, "Location area identification");
	/* 23	MS identity 	Mobile identity 9.9.2.3	O	TLV	7-10 */
	ELEM_OPT_TLV(0x23, NAS_PDU_TYPE_COMMON, DE_EPS_CMN_MOB_ID, "MS identity");
	/* 53	EMM cause	EMM cause 9.9.3.9	O	TV	2 */
	ELEM_OPT_TV(0x53, NAS_PDU_TYPE_EMM, DE_EMM_CAUSE, "");
	/* 17	T3402 value	GPRS timer 9.9.3.16	O	TV	2 */
	ELEM_OPT_TV(0x17, GSM_A_PDU_TYPE_GM, DE_GPRS_TIMER, "T3402 value");
	/* 59	T3423 value	GPRS timer 9.9.3.16	O	TV	2 */
	ELEM_OPT_TV(0x59, GSM_A_PDU_TYPE_GM, DE_GPRS_TIMER, "T3423 value");
	/* 4A	Equivalent PLMNs	PLMN list 9.9.2.6	O	TLV	5-47 */
	ELEM_OPT_TLV(0x4a, NAS_PDU_TYPE_COMMON, DE_EPS_CMN_PLM_LST, "Equivalent PLMNs");

	EXTRANEOUS_DATA_CHECK(curr_len, 0);

}
/*
 * 8.2.2	Attach complete
 */
static void
nas_emm_attach_comp(tvbuff_t *tvb, proto_tree *tree, guint32 offset, guint len)
{
	guint32	curr_offset;
	guint32	consumed;
	guint	curr_len;

	curr_offset = offset;
	curr_len = len;

	/* ESM message container	ESM message container 9.9.3.15	M	LV-E	2-n */
	ELEM_MAND_LV_E(NAS_PDU_TYPE_EMM, DE_EMM_ESM_MSG_CONT, "");

	EXTRANEOUS_DATA_CHECK(curr_len, 0);

}

/*
 * 8.2.3	Attach reject
 */
static void
nas_emm_attach_rej(tvbuff_t *tvb, proto_tree *tree, guint32 offset, guint len)
{
	guint32	curr_offset;
	guint32	consumed;
	guint	curr_len;

	curr_offset = offset;
	curr_len = len;

	/* * EMM cause	EMM cause 9.9.3.9	M	V	1 */
	ELEM_MAND_V(NAS_PDU_TYPE_EMM, DE_EMM_CAUSE);
	/* 78 ESM message container	ESM message container 9.9.3.15	O	TLV-E	4-n */
	ELEM_OPT_TLV(0x78, NAS_PDU_TYPE_EMM, DE_EMM_ESM_MSG_CONT, "");

	EXTRANEOUS_DATA_CHECK(curr_len, 0);

}
/*
 * 8.2.4	Attach request
 */
static void
nas_emm_attach_req(tvbuff_t *tvb, proto_tree *tree, guint32 offset, guint len)
{
	guint32	curr_offset, bit_offset;
	guint32	consumed;
	guint	curr_len;

	curr_offset = offset;
	curr_len = len;

	bit_offset = curr_offset<<3;
	
	/* NAS key set identifier	NAS key set identifier 9.9.3.21	M	V	1/2 */
	proto_tree_add_bits_item(tree, hf_nas_eps_spare_bits, tvb, bit_offset, 1, FALSE);
	bit_offset++;
	proto_tree_add_bits_item(tree, hf_nas_eps_emm_nas_key_set_id, tvb, bit_offset, 3, FALSE);
	bit_offset+=3;
	/* EPS attach type	EPS attach type 9.9.3.11	M	V	1/2  
	 * Inline:
	 */
	proto_tree_add_bits_item(tree, hf_nas_eps_spare_bits, tvb, bit_offset, 1, FALSE);
	bit_offset++;
	proto_tree_add_bits_item(tree, hf_nas_eps_emm_eps_att_type, tvb, bit_offset, 3, FALSE);
	bit_offset+=3;
	/* Fix up the lengths */
	curr_len--;
	curr_offset++;
	/* Old GUTI or IMSI	EPS mobile identity 9.9.3.12	M	LV	5-12 */
	ELEM_MAND_LV(NAS_PDU_TYPE_EMM, DE_EMM_EPS_MID, " - Old GUTI or IMSI");
	/* UE network capability	UE network capability 9.9.3.34	M	LV	3-14 */
	ELEM_MAND_LV(NAS_PDU_TYPE_EMM, DE_EMM_UE_NET_CAP, "");
	/* ESM message container	ESM message container 9.9.3.15	M	LV-E	2-n */
	ELEM_MAND_LV_E(NAS_PDU_TYPE_EMM, DE_EMM_ESM_MSG_CONT, "");
	/* 52 Last visited registered TAI	Tracking area identity 9.9.3.32	O	TV	6 */
	ELEM_OPT_TV(0x52, NAS_PDU_TYPE_EMM, DE_EMM_TRAC_AREA_ID, "Last visited registered TAI");
	/* 5c DRX parameter	DRX parameter 9.9.3.8	O	TV	3 */
	ELEM_OPT_TV(0x5c, GSM_A_PDU_TYPE_GM, DE_DRX_PARAM, "" );
	/* 31 MS network capability	MS network capability 9.9.3.20	M	LV	3-9 */
	ELEM_OPT_TLV( 0x31, GSM_A_PDU_TYPE_GM, DE_MS_NET_CAP , "" );
	/* 13 Old location area identification	Location area identification 9.9.2.2	O	TV	6 */
	ELEM_OPT_TV(0x13, NAS_PDU_TYPE_COMMON, DE_EPS_CMN_LOC_AREA_ID, "Old location area identification");
	/* 9- TMSI status	TMSI status 9.9.3.31	O	TV	1 */
	ELEM_OPT_TV_SHORT( 0x90 , GSM_A_PDU_TYPE_GM, DE_TMSI_STAT , "" );
	/* 11	Mobile station classmark 2	Mobile station classmark 2 9.9.2.5	O	TLV	5 */
	ELEM_OPT_TLV( 0x11, NAS_PDU_TYPE_COMMON, DE_EPS_MS_CM_2 , "" );
	/* 20	Mobile station classmark 3	Mobile station classmark 3 9.9.2.5	O	TLV	2-34 */
	ELEM_OPT_TLV( 0x20, NAS_PDU_TYPE_COMMON, DE_EPS_MS_CM_3 , "" );
	/* 40	Supported Codecs	Supported Codec List 9.9.2.8	O	TLV	5-n */
	ELEM_OPT_TLV(0x40, GSM_A_PDU_TYPE_DTAP, DE_SUP_CODEC_LIST, " - Supported Codecs");

	EXTRANEOUS_DATA_CHECK(curr_len, 0);
}
/*
 * 8.2.5	Authentication failure 
 */
static void
nas_emm_attach_fail(tvbuff_t *tvb, proto_tree *tree, guint32 offset, guint len)
{
	guint32	curr_offset;
	guint32	consumed;
	guint	curr_len;

	curr_offset = offset;
	curr_len = len;

	 /* EMM cause	EMM cause 9.9.3.9	M	V	1 */
	ELEM_MAND_V(NAS_PDU_TYPE_EMM, DE_EMM_CAUSE);
	/* 30 Authentication failure parameter	Authentication failure parameter 9.9.3.1	O	TLV	1 */
	ELEM_OPT_TLV(0x30, GSM_A_PDU_TYPE_DTAP, DE_AUTH_FAIL_PARAM, "");

	EXTRANEOUS_DATA_CHECK(curr_len, 0);
}
/*
 * 8.2.6	Authentication reject
 * No IE:s
 */
/*
 * 8.2.7	Authentication request
 */

static void
nas_emm_auth_req(tvbuff_t *tvb, proto_tree *tree, guint32 offset, guint len)
{
	guint32	curr_offset, bit_offset;
	guint32	consumed;
	guint	curr_len;

	curr_offset = offset;
	curr_len = len;

	/* 
	 * NAS key set identifierASME 	NAS key set identifier 9.9.3.21	M	V	1/2  
	 */
	bit_offset = curr_offset<<3;
	proto_tree_add_bits_item(tree, hf_nas_eps_spare_bits, tvb, bit_offset, 1, FALSE);
	bit_offset++;
	proto_tree_add_bits_item(tree, hf_nas_eps_emm_nas_key_set_id, tvb, bit_offset, 3, FALSE);
	bit_offset+=3;
	
	/* 	Spare half octet	Spare half octet 9.9.2.7	M	V	1/2 */
	proto_tree_add_bits_item(tree, hf_nas_eps_emm_spare_half_octet, tvb, bit_offset, 4, FALSE);
	bit_offset+=4;
	/* Fix up the lengths */
	curr_len--;
	curr_offset++;

	/*
	 * Authentication parameter RAND (EPS challenge) 9.9.3.3	M	V	16
	 */
	ELEM_MAND_V(GSM_A_PDU_TYPE_DTAP, DE_AUTH_PARAM_RAND);
	/*
	 * Authentication parameter AUTN (EPS challenge) 9.9.3.2	M	LV	17
	 */
	ELEM_MAND_LV(GSM_A_PDU_TYPE_COMMON, DE_AUTH_PARAM_AUTN, " - EPS challenge");

	EXTRANEOUS_DATA_CHECK(curr_len, 0);

}
/*
 * 8.2.8	Authentication response
 */
static void
nas_emm_auth_resp(tvbuff_t *tvb, proto_tree *tree, guint32 offset, guint len)
{
	guint32	curr_offset;
	guint32	consumed;
	guint	curr_len;

	curr_offset = offset;
	curr_len = len;

	/*
	 * Authentication response parameter 9.9.3.4	M	LV	5-17
	 */
	ELEM_MAND_LV(NAS_PDU_TYPE_EMM, DE_EMM_AUTH_RESP_PAR, "");

	EXTRANEOUS_DATA_CHECK(curr_len, 0);
}
/*
 * 8.2.9	Detach accept
 * 8.2.9.1	Detach accept (UE originating detach)
 * No further IE's
 * 8.2.9.2	Detach accept (UE terminated detach)
 * No further IE's
 */
/*
 * 8.2.10	Detach request
 * 8.2.10.1	Detach request (UE originating detach)
 * Detach type	Detach type 9.9.3.6	M	V	1/2
 * Spare half octet	Spare half octet 9.9.2.7	M	V	1/2
 * GUTI or IMSI	EPS mobile identity 9.9.3.12	M	LV	5-12
 *ELEM_MAND_LV(NAS_PDU_TYPE_EMM, DE_EMM_EPS_MID, " - GUTI or IMSI");
 */
/*
 * 8.2.10.2	Detach request (UE terminated detach)
 * Detach type	Detach type 9.9.3.6	M	V	1/2
 * Spare half octet	Spare half octet 9.9.2.7	M	V	1/2
 * EMM cause	EMM cause 9.9.3.9	O	TV	2
 * ELEM_OPT_TV(0x53, NAS_PDU_TYPE_EMM, DE_EMM_CAUSE, "");
 */


/*
 * 8.2.11	Downlink NAS Transport
 */
static void
nas_emm_dl_nas_trans(tvbuff_t *tvb, proto_tree *tree, guint32 offset, guint len)
{
	guint32	curr_offset;
	guint32	consumed;
	guint	curr_len;

	curr_offset = offset;
	curr_len = len;

	/* NAS message container	NAS message container 9.9.3.22	M	LV	3-252 */
	ELEM_MAND_LV(NAS_PDU_TYPE_EMM, DE_EMM_NAS_MSG_CONT, "");

	EXTRANEOUS_DATA_CHECK(curr_len, 0);
}
/*
 * 8.2.12	EMM information
 */
static void
nas_emm_emm_inf(tvbuff_t *tvb, proto_tree *tree, guint32 offset, guint len)
{
	guint32	curr_offset;
	guint32	consumed;
	guint	curr_len;

	curr_offset = offset;
	curr_len = len;

	/* 43	Full name for network	Network name 9.9.3.24	O	TLV	3-? */
	ELEM_OPT_TLV(0x43, GSM_A_PDU_TYPE_DTAP, DE_NETWORK_NAME, " - Full name for network");
	/* 45	Short name for network	Network name 9.9.3.24	O	TLV	3-? */
	ELEM_OPT_TLV(0x45, GSM_A_PDU_TYPE_DTAP, DE_NETWORK_NAME, " - Short Name");
	/* 46	Local time zone	Time zone 9.9.3.29	O	TV	2 */
	ELEM_OPT_TV(0x46, GSM_A_PDU_TYPE_DTAP, DE_TIME_ZONE, " - Local");
	/* 47	Universal time and local time zone	Time zone and time 9.9.3.30	O	TV	8 */
	ELEM_OPT_TV(0x47, GSM_A_PDU_TYPE_DTAP, DE_TIME_ZONE_TIME, " - Universal Time and Local Time Zone");
	/* 49	Network daylight saving time	Daylight saving time 9.9.3.6	O	TLV	3 */
	ELEM_OPT_TLV(0x49, GSM_A_PDU_TYPE_DTAP, DE_DAY_SAVING_TIME, "");

	EXTRANEOUS_DATA_CHECK(curr_len, 0);
}


/*
 * 8.2.13	EMM status
 */
static void
nas_emm_emm_status(tvbuff_t *tvb, proto_tree *tree, guint32 offset, guint len)
{
	guint32	curr_offset;
	guint32	consumed;
	guint	curr_len;

	curr_offset = offset;
	curr_len = len;

	/* EMM cause	EMM cause 9.9.3.9	M	V	1 */
	ELEM_MAND_V(NAS_PDU_TYPE_EMM, DE_EMM_CAUSE);

	EXTRANEOUS_DATA_CHECK(curr_len, 0);
}

/*
 * 8.2.14	Extended service request
 */
static void
nas_emm_ext_serv_req(tvbuff_t *tvb, proto_tree *tree, guint32 offset, guint len)
{
	guint32	curr_offset,bit_offset;
	guint32	consumed;
	guint	curr_len;

	curr_offset = offset;
	curr_len = len;

	/* Service type	Service type 9.9.3.27	M	V	1/2 Service type*/
	bit_offset = curr_offset<<3;
	proto_tree_add_bits_item(tree, hf_nas_eps_service_type, tvb, bit_offset, 4, FALSE);
	bit_offset+=4;
	/* NAS key set identifier	NAS key set identifier 9.9.3.21	M	V	1/2 */
	proto_tree_add_bits_item(tree, hf_nas_eps_spare_bits, tvb, bit_offset, 1, FALSE);
	bit_offset++;
	proto_tree_add_bits_item(tree, hf_nas_eps_emm_nas_key_set_id, tvb, bit_offset, 3, FALSE);
	bit_offset+=3;
	/* Fix up the lengths */
	curr_len--;
	curr_offset++;

	/* M-TMSI	Mobile identity 9.9.2.3	M	LV	6 */
	ELEM_MAND_LV(NAS_PDU_TYPE_COMMON, DE_EPS_CMN_MOB_ID, "M-TMSI");
	/* B-	CSFB response	CSFB response 9.9.3.5	C	TV	1 */
	ELEM_OPT_TV_SHORT(0xb0, NAS_PDU_TYPE_EMM, DE_EMM_CSFB_RESP, "");

	EXTRANEOUS_DATA_CHECK(curr_len, 0);
}
/*
 * 8.2.15	GUTI reallocation command
 */
static void
nas_emm_guti_realloc_cmd(tvbuff_t *tvb, proto_tree *tree, guint32 offset, guint len)
{
	guint32	curr_offset;
	guint32	consumed;
	guint	curr_len;

	curr_offset = offset;
	curr_len = len;

	/* GUTI	EPS mobile identity 9.9.3.12	M	LV	12 */
	ELEM_MAND_LV(NAS_PDU_TYPE_EMM, DE_EMM_EPS_MID, " - GUTI");
	
	/* 54	TAI list	Tracking area identity list 9.9.3.33	O	TLV	8-98 */
	ELEM_OPT_TLV(0x54, NAS_PDU_TYPE_EMM, DE_EMM_TRAC_AREA_ID_LST, "");

	EXTRANEOUS_DATA_CHECK(curr_len, 0);
}

/*
 * 8.2.16	GUTI reallocation complete
 * No more IE's
 */
/*
 * 8.2.17	Identity request
 */

static void
nas_emm_id_req(tvbuff_t *tvb, proto_tree *tree, guint32 offset, guint len)
{
	guint32	curr_offset, bit_offset;
	guint32	consumed;
	guint	curr_len;

	curr_offset = offset;
	curr_len = len;


	bit_offset=curr_offset<<3;

	/* Spare half octet	Spare half octet 9.9.2.7	M	V	1/2 */
	proto_tree_add_bits_item(tree, hf_nas_eps_emm_spare_half_octet, tvb, bit_offset, 4, FALSE);
	bit_offset+=4;

	/* Identity type	Identity type 2 9.9.3.17	M	V	1/2 */
	proto_tree_add_bits_item(tree, hf_nas_eps_emm_id_type2, tvb, bit_offset, 4, FALSE);
	bit_offset+=4;
	consumed = 1;


	/* Fix up the lengths */
	curr_len--;
	curr_offset++;

	EXTRANEOUS_DATA_CHECK(curr_len, 0);
}
/*
 * 8.2.18	Identity response
 */
static void
nas_emm_id_res(tvbuff_t *tvb, proto_tree *tree, guint32 offset, guint len)
{
	guint32	curr_offset;
	guint32	consumed;
	guint	curr_len;

	curr_offset = offset;
	curr_len = len;

	/* Mobile identity	Mobile identity 9.9.2.3	M	LV	4-10 */
	ELEM_MAND_LV(NAS_PDU_TYPE_COMMON, DE_EPS_CMN_MOB_ID, "");
	
	EXTRANEOUS_DATA_CHECK(curr_len, 0);
}

/*
 * 8.2.19	NAS CS service notification
 */

/*
 * 8.2.20	Security mode command
 */
static void
nas_emm_sec_mode_cmd(tvbuff_t *tvb, proto_tree *tree, guint32 offset, guint len)
{
	proto_item *item;
	guint32	curr_offset, bit_offset;
	guint32	consumed;
	guint	curr_len;

	curr_offset = offset;
	curr_len = len;

	/* 	Selected NAS security algorithms	NAS security algorithms 9.9.3.23	M	V	1  */
	ELEM_MAND_V(NAS_PDU_TYPE_EMM, DE_EMM_NAS_SEC_ALGS);
	/* 	NAS key set identifierASME	NAS key set identifier 9.9.3.21	M	V	1/2 */
	bit_offset = curr_offset<<3;
	proto_tree_add_bits_item(tree, hf_nas_eps_spare_bits, tvb, bit_offset, 1, FALSE);
	bit_offset++;
	item = proto_tree_add_bits_item(tree, hf_nas_eps_emm_nas_key_set_id, tvb, bit_offset, 3, FALSE);
	proto_item_append_text(item," - ASME");
	bit_offset+=3;
	/* 	NAS key set identifierSGSN	NAS key set identifier 9.9.3.21	M	V	1/2 */
	proto_tree_add_bits_item(tree, hf_nas_eps_spare_bits, tvb, bit_offset, 1, FALSE);
	bit_offset++;
	item = proto_tree_add_bits_item(tree, hf_nas_eps_emm_nas_key_set_id, tvb, bit_offset, 3, FALSE);
	proto_item_append_text(item," - SGSN");
	bit_offset+=3;

	/* Fix up the lengths */
	curr_len--;
	curr_offset++;

	/* 	Replayed UE security capabilities	UE security capability 9.9.3.36	M	LV	3-6 */
	ELEM_MAND_LV(NAS_PDU_TYPE_EMM, DE_EMM_UE_SEC_CAP, " - Replayed UE security capabilities");
	/* C-	IMEISV request	IMEISV request 9.9.3.18	O	TV	1 */
	ELEM_OPT_TV_SHORT( 0xC0 , GSM_A_PDU_TYPE_GM, DE_IMEISV_REQ , "" );
	/* 55	Replayed NonceUE	Nonce 9.9.3.25	O	TV	5 */
	ELEM_OPT_TV(0x55, GSM_A_PDU_TYPE_GM, DE_EMM_NONCE, " - Replayed NonceUE");
	/* 56	NonceMME	Nonce 9.9.3.25	O	TV	5 */
	ELEM_OPT_TV(0x55, GSM_A_PDU_TYPE_GM, DE_EMM_NONCE, " - NonceMME");

	EXTRANEOUS_DATA_CHECK(curr_len, 0);
}
/*
 * 8.2.21	Security mode complete
 */
static void
nas_emm_sec_mode_comp(tvbuff_t *tvb, proto_tree *tree, guint32 offset, guint len)
{
	guint32	curr_offset;
	guint32	consumed;
	guint	curr_len;

	curr_offset = offset;
	curr_len = len;

	/* 23	IMEISV	Mobile identity 9.9.2.3	O	TLV	11 DE_EPS_CMN_MOB_ID*/
	ELEM_OPT_TLV(0x23, NAS_PDU_TYPE_EMM, DE_EPS_CMN_MOB_ID, "IMEISV");
 
	EXTRANEOUS_DATA_CHECK(curr_len, 0);
}
/*
 * 8.2.22	Security mode reject
 */
static void
nas_emm_sec_mode_rej(tvbuff_t *tvb, proto_tree *tree, guint32 offset, guint len)
{
	guint32	curr_offset;
	guint32	consumed;
	guint	curr_len;

	curr_offset = offset;
	curr_len = len;

	/* EMM cause	EMM cause 9.9.3.9	M	V	1 */
	ELEM_MAND_V(NAS_PDU_TYPE_EMM, DE_EMM_CAUSE);

	EXTRANEOUS_DATA_CHECK(curr_len, 0);
}
/*
 * 8.2.23	Security protected NAS message
 */
#if 0

static void
nas_emm_sec_prot_msg(tvbuff_t *tvb, proto_tree *tree, guint32 offset, guint len)
{
	proto_item *item;
	guint32	curr_offset;
	guint32	consumed;
	guint	curr_len;

	curr_offset = offset;
	curr_len = len;

	/* Message authentication code	Message authentication code 9.5	M	V	4 */
	/* Sequence number	Sequence number 9.6	M	V	1 */
	/* NAS message	NAS message 9.7	M	V	1-n  */
	EXTRANEOUS_DATA_CHECK(curr_len, 0);
}
#endif
/*
 * 8.2.24	Service reject
 */
static void
nas_emm_serv_rej(tvbuff_t *tvb, proto_tree *tree, guint32 offset, guint len)
{
	guint32	curr_offset;
	guint32	consumed;
	guint	curr_len;

	curr_offset = offset;
	curr_len = len;

	/* EMM cause	EMM cause 9.9.3.9	M	V	1 */
	ELEM_MAND_V(NAS_PDU_TYPE_EMM, DE_EMM_CAUSE);

	/* 5B	T3442 value	GPRS timer 9.9.3.16	C	TV	2 */
	ELEM_OPT_TV(0x5b, GSM_A_PDU_TYPE_GM, DE_GPRS_TIMER, " - T3442 value");

	EXTRANEOUS_DATA_CHECK(curr_len, 0);
}
/*
 * 8.2.25	Service request
 * This message is sent by the UE to the network to request the establishment
 * of a NAS signalling connection and of the radio and S1 bearers. 
 * Its structure does not follow the structure of a standard layer 3 message. See table 8.2.22.1.
 * Protocol discriminator	Protocol discriminator 9.2	M	V	1/2
 * Security header type	Security header type 9.3.1	M	V	1/2
 * KSI and sequence number	KSI and sequence number 9.9.3.17	M	V	1
 * Message authentication code (short)	Short MAC 9.9.3.25	M	V	2
 */
/*
 * 8.2.26	Tracking area update accept
 */
static void
nas_emm_trac_area_upd_acc(tvbuff_t *tvb, proto_tree *tree, guint32 offset, guint len)
{
	guint32	curr_offset, bit_offset;
	guint32	consumed;
	guint	curr_len;

	curr_offset = offset;
	curr_len = len;

	/* 	EPS update result	EPS update result 9.9.3.13	M	V	1/2 */
	bit_offset = curr_offset<<3;
	proto_tree_add_bits_item(tree, hf_nas_eps_spare_bits, tvb, bit_offset, 1, FALSE);
	bit_offset++;
	proto_tree_add_bits_item(tree, hf_nas_eps_eps_update_result_value, tvb, bit_offset, 3, FALSE);
	bit_offset+=3;
	/* 	Spare half octet	Spare half octet 9.9.2.7	M	V	1/2 */
	proto_tree_add_bits_item(tree, hf_nas_eps_emm_spare_half_octet, tvb, bit_offset, 4, FALSE);
	bit_offset+=4;
	/* Fix up the lengths */
	curr_len--;
	curr_offset++;
	/* No more mandatory elements */
	if (curr_len==0)
		return;
	/* 5A	T3412 value	GPRS timer 9.9.3.16	O	TV	2 */
	ELEM_OPT_TV(0x5a, GSM_A_PDU_TYPE_GM, DE_GPRS_TIMER, "T3412 value");
	/* 50	GUTI	EPS mobile identity 9.9.3.12	O	TLV	13 */
	ELEM_OPT_TLV(0x50, NAS_PDU_TYPE_EMM, DE_EMM_EPS_MID, "GUTI"); 
	/* 54	TAI list	Tracking area identity list 9.9.3.33	O	TLV	8-98 */
	ELEM_OPT_TLV(0x54, NAS_PDU_TYPE_EMM, DE_EMM_TRAC_AREA_ID_LST, ""); 
	/* 57	EPS bearer context status	EPS bearer context status 9.9.2.1	O	TLV	4 */
	ELEM_OPT_TLV(0x57, NAS_PDU_TYPE_COMMON, DE_EPS_CMN_EPS_BE_CTX_STATUS, "");
	/* 13	Location area identification	Location area identification 9.9.2.2	O	TV	6 */
	ELEM_OPT_TLV(0x13, NAS_PDU_TYPE_COMMON, DE_EPS_CMN_LOC_AREA_ID, "");
	/* 23	MS identity	Mobile identity 9.9.2.3	O	TLV	7-10  */
	ELEM_OPT_TLV(0x23, NAS_PDU_TYPE_COMMON, DE_EPS_CMN_MOB_ID, "MS identity");
	/* 53	EMM cause	EMM cause 9.9.3.9	O	TV	2  */
	ELEM_OPT_TV(0x53, NAS_PDU_TYPE_EMM, DE_EMM_CAUSE, "");
	/* 17	T3402 value	GPRS timer 9.9.3.16	O	TV	2  */
	ELEM_OPT_TV(0x17, GSM_A_PDU_TYPE_GM, DE_GPRS_TIMER, "T3402 value");
	/* 59	T3423 value	GPRS timer 9.9.3.16	O	TV	2 */
	ELEM_OPT_TV(0x59, GSM_A_PDU_TYPE_GM, DE_GPRS_TIMER, "T3423 value");
	/* 4A	Equivalent PLMNs	PLMN list 9.9.2.6	O	TLV	5-47 */
	ELEM_OPT_TLV(0x4a, NAS_PDU_TYPE_COMMON, DE_EPS_CMN_PLM_LST, "Equivalent PLMNs");

	EXTRANEOUS_DATA_CHECK(curr_len, 0);
}
/*
 * 8.2.27	Tracking area update complete
 * No more IE's
 */
/*
 * 8.2.28	Tracking area update reject
 */
static void
nas_emm_trac_area_upd_rej(tvbuff_t *tvb, proto_tree *tree, guint32 offset, guint len)
{
	guint32	curr_offset;
	guint32	consumed;
	guint	curr_len;

	curr_offset = offset;
	curr_len = len;

	/* EMM cause	EMM cause 9.9.3.9	M	V	1 */
	ELEM_MAND_V(NAS_PDU_TYPE_EMM, DE_EMM_CAUSE);

	EXTRANEOUS_DATA_CHECK(curr_len, 0);
}
/*
 * 8.2.29	Tracking area update request
 */
static void
nas_emm_trac_area_upd_req(tvbuff_t *tvb, proto_tree *tree, guint32 offset, guint len)
{
	proto_item *item;
	guint32	curr_offset, bit_offset;
	guint32	consumed;
	guint	curr_len;

	curr_offset = offset;
	curr_len = len;

	/* 	EPS update type	EPS update type 9.9.3.14	M	V	1/2 */
	bit_offset = curr_offset<<3;
	proto_tree_add_bits_item(tree, hf_nas_eps_active_flg, tvb, bit_offset, 1, FALSE);
	bit_offset++;
	proto_tree_add_bits_item(tree, hf_nas_eps_eps_update_type_value, tvb, bit_offset, 3, FALSE);
	bit_offset+=3;

	/* 	Spare half octet	Spare half octet 9.9.2.7	M	V	1/2 */
	proto_tree_add_bits_item(tree, hf_nas_eps_emm_spare_half_octet, tvb, bit_offset, 4, FALSE);
	bit_offset+=4;
	/* Fix up the lengths */
	curr_len--;
	curr_offset++;
	/* 	Old GUTI 	EPS mobile identity 9.9.3.12	M	LV	12 */
	ELEM_MAND_LV(NAS_PDU_TYPE_EMM, DE_EMM_EPS_MID, " - Old GUTI");
	/* 	NAS key set identifierASME	NAS key set identifier 9.9.3.21	M	V	1/2 */
	bit_offset = curr_offset<<3;
	proto_tree_add_bits_item(tree, hf_nas_eps_spare_bits, tvb, bit_offset, 1, FALSE);
	bit_offset++;
	item = proto_tree_add_bits_item(tree, hf_nas_eps_emm_nas_key_set_id, tvb, bit_offset, 3, FALSE);
	proto_item_append_text(item," - ASME");
	bit_offset+=3;
	/* 	NAS key set identifierSGSN	NAS key set identifier 9.9.3.21	M	V	1/2 */
	proto_tree_add_bits_item(tree, hf_nas_eps_spare_bits, tvb, bit_offset, 1, FALSE);
	bit_offset++;
	item = proto_tree_add_bits_item(tree, hf_nas_eps_emm_nas_key_set_id, tvb, bit_offset, 3, FALSE);
	proto_item_append_text(item," - SGSN");
	bit_offset+=3;
	/* Fix up the lengths */
	curr_len--;
	curr_offset++;
	/* No more Mandatory elements */
	if (curr_len==0)
		return;
	/* 19	Old P-TMSI signature	P-TMSI signature 9.9.3.26	O	TV	4 */
	ELEM_OPT_TV( 0x19 , GSM_A_PDU_TYPE_GM, DE_P_TMSI_SIG, " - Old P-TMSI Signature");
	/* 50	Additional GUTI	EPS mobile identity 9.9.3.12	O	TLV	13 */
	ELEM_OPT_TLV(0x50, NAS_PDU_TYPE_EMM, DE_EMM_EPS_MID, " - Additional GUTI");
	/* 55	NonceUE	Nonce 9.9.3.25	O	TV	5 */
	ELEM_OPT_TV(0x55, GSM_A_PDU_TYPE_GM, DE_EMM_NONCE, " - NonceUE");
	/* 58	UE network capability	UE network capability 9.9.3.34	O	TLV	4-15 */
	ELEM_OPT_TLV(0x58, NAS_PDU_TYPE_EMM, DE_EMM_UE_NET_CAP, "");
	/* 52	Last visited registered TAI	Tracking area identity 9.9.3.32	O	TV	6 */
	ELEM_OPT_TV(0x52, NAS_PDU_TYPE_EMM, DE_EMM_TRAC_AREA_ID, "Last visited registered TAI");
	/* 5C	DRX parameter	DRX parameter 9.9.3.8	O	TV	3 */
	ELEM_OPT_TV(0x5c, GSM_A_PDU_TYPE_GM, DE_DRX_PARAM, "" );
	/* A-	UE radio capability information update needed	UE radio capability information update needed 9.9.3.35	O	TV	1 */
	ELEM_OPT_TV_SHORT( 0xA0 , NAS_PDU_TYPE_EMM, DE_EMM_UE_RA_CAP_INF_UPD_NEED , "" );
	/* 57	EPS bearer context status	EPS bearer context status 9.9.2.1	O	TLV	4 */
	ELEM_OPT_TLV(0x57, NAS_PDU_TYPE_COMMON, DE_EPS_CMN_EPS_BE_CTX_STATUS, "");
	/* 31	MS network capability	MS network capability 9.9.3.20	O	TLV	4-10 */
	ELEM_OPT_TLV( 0x31 , GSM_A_PDU_TYPE_GM, DE_MS_NET_CAP , "" );
	/* 13	Old location area identification	Location area identification 9.9.2.2	O	TV	6 */
	ELEM_OPT_TV(0x13, NAS_PDU_TYPE_COMMON, DE_EPS_CMN_LOC_AREA_ID, "Old location area identification");
 	/* 9-	TMSI status	TMSI status 9.9.3.31	O	TV	1  */
	ELEM_OPT_TV_SHORT( 0x90 , GSM_A_PDU_TYPE_GM, DE_TMSI_STAT , "" );
	/* 11	Mobile station classmark 2	Mobile station classmark 2 9.9.2.5	O	TLV	5 */
	ELEM_OPT_TLV( 0x11, NAS_PDU_TYPE_COMMON, DE_EPS_MS_CM_2 , "" );
	/* 20	Mobile station classmark 3	Mobile station classmark 3 9.9.2.5	O	TLV	2-34 */
	ELEM_OPT_TLV( 0x20, NAS_PDU_TYPE_COMMON, DE_EPS_MS_CM_3 , "" );
	/* 40	Supported Codecs	Supported Codec List 9.9.2.8	O	TLV	5-n */
	ELEM_OPT_TLV(0x40, GSM_A_PDU_TYPE_DTAP, DE_SUP_CODEC_LIST, " - Supported Codecs");
 
	EXTRANEOUS_DATA_CHECK(curr_len, 0);
}

/*
 * 8.2.30	Uplink NAS Transport
 */
static void
nas_emm_ul_nas_trans(tvbuff_t *tvb, proto_tree *tree, guint32 offset, guint len)
{
	guint32	curr_offset;
	guint32	consumed;
	guint	curr_len;

	curr_offset = offset;
	curr_len = len;

	/* NAS message container	NAS message container 9.9.3.22	M	LV	3-252*/
	ELEM_MAND_LV(NAS_PDU_TYPE_EMM, DE_EMM_NAS_MSG_CONT, "");
 
	EXTRANEOUS_DATA_CHECK(curr_len, 0);
}
/*
 * 8.2.9	CS service notification
 */

static void
nas_emm_cs_serv_not(tvbuff_t *tvb, proto_tree *tree, guint32 offset, guint len)
{
	guint32	curr_offset;
	guint32	consumed;
	guint	curr_len;

	curr_offset = offset;
	curr_len = len;

	consumed = 0;

	/* 60	CLI	CLI 9.9.3.38	O	TLV	3-12 */
	/* 61	SS Code	SS Code 9.9.3.39	O	TV	2 */
	/* 62	LCS indicator	LCS indicator 9.9.3.40	O	TV	2 */
	/* 63	LCS client identity	LCS client identity 9.9.3.41	O	TLV	3-257 */
 
	EXTRANEOUS_DATA_CHECK(curr_len, 0);
}
/*
 * 8.3	EPS session management messages
 */
/*
 * 8.3.1	Activate dedicated EPS bearer context accept
 * 27	Protocol configuration options	Protocol configuration options 9.9.4.8	O	TLV	3-253
 */
/*
 * 8.3.2	Activate dedicated EPS bearer context reject
 * 	ESM cause	ESM cause 9.9.4.2	M	V	1
 * 27	Protocol configuration options	Protocol configuration options 9.9.4.8	O	TLV	3-253
 */
/*
 * 8.3.3	Activate dedicated EPS bearer context request
 	Linked EPS bearer identity	Linked EPS bearer identity
9.9.4.6	M	V	1/2
	Spare half octet	Spare half octet
9.9.2.7	M	V	1/2
	EPS QoS	EPS quality of service
9.9.4.3	M	LV	2-10
	TFT	Traffic flow template
9.9.4.16	M	LV	2-256
5D	Transaction identifier	Transaction identifier
9.9.4.17	O	TLV	3-4
30	Negotiated QoS	Quality of service
9.9.4.12	O	TLV	14-18
32	Negotiated LLC SAPI	LLC service access point identifier
9.9.4.7	O	TV	2
8-	Radio priority	Radio priority
9.9.4.13	O	TV	1
34	Packet flow Identifier	Packet flow Identifier
9.9.4.8	O	TLV	3
27	Protocol configuration options	Protocol configuration options
9.9.4.11	O	TLV	3-253

 */
/*
 * 8.3.3	Activate dedicated EPS bearer context request
 *
 */

/*
 * 8.3.4	Activate default EPS bearer context accept
 */
/*
 * 8.3.5	Activate default EPS bearer context reject
 * 8.3.6 Activate default EPS bearer context request
 * 8.3.7 Bearer resource modification reject
 * 8.3.8 Bearer resource modification request
 * 8.3.9 Deactivate EPS bearer context accept
 * 8.3.10 Deactivate EPS bearer context request
 * 8.3.11 ESM information request
 * 8.3.12 ESM information response
 * 8.3.13 ESM status
 * 8.3.14 Modify EPS bearer context accept
 * 8.3.15 Modify EPS bearer context reject
 * 8.3.16 Modify EPS bearer context request
 */
/*
 * 8.3.17 PDN connectivity reject
 */
/*
 * 8.3.18 PDN connectivity request
 */
static void
nas_esm_pdn_con_req(tvbuff_t *tvb, proto_tree *tree, guint32 offset, guint len)
{
	guint32	curr_offset, bit_offset;
	guint32	consumed;
	guint	curr_len;

	curr_offset = offset;
	curr_len = len;

	bit_offset=curr_offset<<3;
	/* PDN type PDN type 9.9.4.10 M V 1/2 */
	proto_tree_add_bits_item(tree, hf_nas_eps_esm_pdn_type, tvb, bit_offset, 4, FALSE);
	bit_offset+=4;

	/* Request type 9.9.4.14 M V 1/2 */
	proto_tree_add_bits_item(tree, hf_nas_eps_esm_request_type, tvb, bit_offset, 4, FALSE);
	bit_offset+=4;
	
	/* Fix up the lengths */
	curr_len--;
	curr_offset++;
	if (curr_len==0)
		return;

	/* D- ESM information transfer flag 9.9.4.5 O TV 1 */
	ELEM_OPT_TV_SHORT( 0xd0 , NAS_PDU_TYPE_ESM, DE_ESM_INF_TRF_FLG , "" );
	/* 28 Access point name 9.9.4.1 O TLV 3-102 */
	ELEM_OPT_TLV( 0x28 , GSM_A_PDU_TYPE_GM, DE_ACC_POINT_NAME , "" );
	/* 27 Protocol configuration options 9.9.4.11 O TLV 3-253 */
	ELEM_OPT_TLV( 0x27 , GSM_A_PDU_TYPE_GM, DE_PRO_CONF_OPT , "" );

	EXTRANEOUS_DATA_CHECK(curr_len, 0);
}
/*
 * 8.3.19 PDN disconnect reject
 * 8.3.20 PDN disconnect request
 */


#define	NUM_NAS_MSG_ESM (sizeof(nas_msg_esm_strings)/sizeof(value_string))
static gint ett_nas_msg_esm[NUM_NAS_MSG_ESM];
static void (*nas_msg_esm_fcn[])(tvbuff_t *tvb, proto_tree *tree, guint32 offset, guint len) = {
	NULL,	/* Activate default EPS bearer context request*/
	NULL,	/* Activate default EPS bearer context accept*/
	NULL,	/* Activate default EPS bearer context reject*/
	NULL,	/* Activate dedicated EPS bearer context request*/
	NULL,	/* Activate dedicated EPS bearer context accept*/
	NULL,	/* Activate dedicated EPS bearer context reject*/
	NULL,	/* Modify EPS bearer context request*/
	NULL,	/* Modify EPS bearer context accept*/
	NULL,	/* Modify EPS bearer context reject*/
	NULL,	/* Deactivate EPS bearer context request*/
	NULL,	/* Deactivate EPS bearer context accept*/
	nas_esm_pdn_con_req,	/* 8.3.18 PDN connectivity request */
	NULL,	/* PDN connectivity reject*/
	NULL,	/* PDN disconnect request*/
	NULL,	/* PDN disconnect reject*/
	NULL,	/* Bearer resource allocation request*/
	NULL,	/* Bearer resource allocation reject*/
	NULL,	/* Bearer resource modification request*/
	NULL,	/* Bearer resource modification reject*/
	NULL,	/* ESM information request*/
	NULL,	/* ESM information response*/
	NULL,	/* ESM status */

	NULL,	/* NONE */
};

void get_nas_esm_msg_params(guint8 oct, const gchar **msg_str, int *ett_tree, int *hf_idx, msg_fcn *msg_fcn)
{
	gint			idx;

	*msg_str = match_strval_idx((guint32) (oct & 0xff), nas_msg_esm_strings, &idx);
	*ett_tree = ett_nas_msg_esm[idx];
	*hf_idx = hf_nas_eps_msg_esm_type;
	*msg_fcn = nas_msg_esm_fcn[idx];

	return;
}



#define	NUM_NAS_MSG_EMM (sizeof(nas_msg_emm_strings)/sizeof(value_string))
static gint ett_nas_msg_emm[NUM_NAS_MSG_EMM];
static void (*nas_msg_emm_fcn[])(tvbuff_t *tvb, proto_tree *tree, guint32 offset, guint len) = {
	nas_emm_attach_req,			/* Attach request */
	nas_emm_attach_acc,			/* Attach accept */
	nas_emm_attach_comp,		/* Attach complete */
	nas_emm_attach_rej,			/* Attach reject */
	NULL,	/* Detach request */
	NULL,	/* Detach accept */
							
	nas_emm_trac_area_upd_req,	/* Tracking area update request */
	nas_emm_trac_area_upd_acc,	/* Tracking area update accept */
	NULL,						/* Tracking area update complete (No IE's)*/
	nas_emm_trac_area_upd_rej,	/* Tracking area update reject */
			
	nas_emm_ext_serv_req,		/* Extended service request */
	nas_emm_serv_rej,			/* Service reject */
									
	nas_emm_guti_realloc_cmd,	/* GUTI reallocation command */
	NULL,						/* GUTI reallocation complete (No IE's) */
	nas_emm_auth_req,			/* Authentication request */
	nas_emm_auth_resp,			/* Authentication response */
	NULL,						/* Authentication reject (No IE:s)*/
	nas_emm_attach_fail,		/* Authentication failure */
	nas_emm_id_req,				/* Identity request */
	nas_emm_id_res,				/* Identity response */
/* 	NULL,						8.2.19	NAS CS service notification */
	nas_emm_sec_mode_cmd,		/* Security mode command */
	nas_emm_sec_mode_comp,		/* Security mode complete */
	nas_emm_sec_mode_rej,		/* Security mode reject */
									
	nas_emm_emm_status,			/* EMM status */
	nas_emm_emm_inf,			/* EMM information */
	nas_emm_dl_nas_trans,		/* Downlink NAS transport */
	nas_emm_ul_nas_trans,		/* Uplink NAS transport */
	nas_emm_cs_serv_not,		/* CS Service notification */
	NULL,	/* NONE */

};

void get_nas_emm_msg_params(guint8 oct, const gchar **msg_str, int *ett_tree, int *hf_idx, msg_fcn *msg_fcn)
{
	gint			idx;

	*msg_str = match_strval_idx((guint32) (oct & 0xff), nas_msg_emm_strings, &idx);
	*ett_tree = ett_nas_msg_emm[idx];
	*hf_idx = hf_nas_eps_msg_emm_type;
	*msg_fcn = nas_msg_emm_fcn[idx];

	return;
}

static void
disect_nas_eps_esm_msg(tvbuff_t *tvb, packet_info *pinfo, proto_tree *tree)
{
	const gchar		*msg_str;
	guint32			len;
	gint			ett_tree;
	int				hf_idx;
	void			(*msg_fcn)(tvbuff_t *tvb, proto_tree *tree, guint32 offset, guint len);
	guint8			oct;
	int				offset = 0;


	len = tvb_length(tvb);

	/* EPS bearer identity 9.3.2 */
	proto_tree_add_item(tree, hf_nas_eps_bearer_id, tvb, offset, 1, FALSE);
	/* Protocol discriminator 9.2 */
	proto_tree_add_item(tree, hf_gsm_a_L3_protocol_discriminator, tvb, offset, 1, FALSE);
	offset++;

	/* Procedure transaction identity 9.4 
	 * The procedure transaction identity and its use are defined in 3GPP TS 24.007
	 */
	proto_tree_add_item(tree, hf_nas_eps_esm_proc_trans_id, tvb, offset, 1, FALSE);
	offset++;

	/*messge type IE*/
	oct = tvb_get_guint8(tvb,offset);
	msg_fcn = NULL;
	ett_tree = -1;
	hf_idx = -1;
	msg_str = NULL;

	get_nas_esm_msg_params(oct, &msg_str, &ett_tree, &hf_idx, &msg_fcn);

	if(msg_str){
		if (check_col(pinfo->cinfo, COL_INFO)){
			col_append_fstr(pinfo->cinfo, COL_INFO, " %s ", msg_str);
		}
	}

	/*
	 * Add NAS message name
	 */
	proto_tree_add_item(tree, hf_idx, tvb, offset, 1, FALSE);
	offset++;


	/*
	 * decode elements
	 */
	if (msg_fcn == NULL)
	{
		proto_tree_add_text(tree, tvb, offset, len - offset,
			"Message Elements");
	}
	else
	{
		(*msg_fcn)(tvb, tree, offset, len - offset);
	}

}

static void
dissect_nas_eps_emm_msg(tvbuff_t *tvb, packet_info *pinfo, proto_tree *tree)
{
	const gchar		*msg_str;
	guint32			len;
	gint			ett_tree;
	int				hf_idx;
	void			(*msg_fcn)(tvbuff_t *tvb, proto_tree *tree, guint32 offset, guint len);
	guint8			security_header_type, oct;
	int offset = 0;

	len = tvb_length(tvb);

	/* 9.3.1	Security header type */
	security_header_type = tvb_get_guint8(tvb,offset)>>4;
	proto_tree_add_item(tree, hf_nas_eps_security_header_type, tvb, 0, 1, FALSE);
	if (security_header_type !=0)
		/* XXX Add further decoding here? */
		return;
	proto_tree_add_item(tree, hf_gsm_a_L3_protocol_discriminator, tvb, 0, 1, FALSE);
	offset++;
	/*messge type IE*/
	oct = tvb_get_guint8(tvb,offset);
	msg_fcn = NULL;
	ett_tree = -1;
	hf_idx = -1;
	msg_str = NULL;

	get_nas_emm_msg_params(oct, &msg_str, &ett_tree, &hf_idx, &msg_fcn);

	if(msg_str){
		if (check_col(pinfo->cinfo, COL_INFO)){
			col_append_fstr(pinfo->cinfo, COL_INFO, " %s ", msg_str);
		}
	}

	/*
	 * Add NAS message name
	 */
	proto_tree_add_item(tree, hf_idx, tvb, offset, 1, FALSE);
	offset++;


	/*
	 * decode elements
	 */
	if (msg_fcn == NULL)
	{
		proto_tree_add_text(tree, tvb, offset, len - offset,
			"Message Elements");
	}
	else
	{
		(*msg_fcn)(tvb, tree, offset, len - offset);
	}

}

static void
dissect_nas_eps(tvbuff_t *tvb, packet_info *pinfo, proto_tree *tree)
{
	proto_item *item;
	proto_tree *nas_eps_tree;
	guint8			pd;
	int offset = 0;

	/* Save pinfo */
	gpinfo = pinfo;

	/* make entry in the Protocol column on summary display */
	if (check_col(pinfo->cinfo, COL_PROTOCOL))
		col_append_str(pinfo->cinfo, COL_PROTOCOL, "/NAS-EPS");

	item = proto_tree_add_item(tree, proto_nas_eps, tvb, 0, -1, FALSE);
	nas_eps_tree = proto_item_add_subtree(item, ett_nas_eps);

	pd = tvb_get_guint8(tvb,offset)&0x0f;
	switch (pd){
		case 2:
			/* EPS session management messages. 
			 * Ref 3GPP TS 24.007 version 8.0.0 Release 8, Table 11.2: Protocol discriminator values 
			 */
			disect_nas_eps_esm_msg(tvb, pinfo, nas_eps_tree);
			break;
		case 7:
			/* EPS mobility management messages. 
			 * Ref 3GPP TS 24.007 version 8.0.0 Release 8, Table 11.2: Protocol discriminator values 
			 */
			dissect_nas_eps_emm_msg(tvb, pinfo, nas_eps_tree);
			break;
		default:
			proto_tree_add_text(nas_eps_tree, tvb, offset, -1, "Not a NAS EPS PD %u(%s)",pd,val_to_str(pd, protocol_discriminator_vals,"unknown"));
			break;
	}

}

void proto_register_nas_eps(void) {
	guint		i;
	guint		last_offset;

	/* List of fields */

  static hf_register_info hf[] = {
	{ &hf_nas_eps_msg_emm_type,
		{ "NAS EPS Mobility Management Message Type",	"nas_eps.nas_msg_emm_type",
		FT_UINT8, BASE_HEX, VALS(nas_msg_emm_strings), 0x0,
		"", HFILL }
	},
	{ &hf_nas_eps_common_elem_id,
		{ "Element ID",	"nas_eps.common.elem_id",
		FT_UINT8, BASE_DEC, NULL, 0,
		"", HFILL }
	},
	{ &hf_nas_eps_emm_elem_id,
		{ "Element ID",	"nas_eps.emm.elem_id",
		FT_UINT8, BASE_DEC, NULL, 0,
		"", HFILL }
	},
	{ &hf_nas_eps_bearer_id,
		{ "EPS bearer identity",	"nas_eps.bearer_id",
		FT_UINT8, BASE_HEX, NULL, 0xf0,
		"", HFILL }
	},
	{ &hf_nas_eps_spare_bits,
		{ "Spare bit(s)", "nas_eps.spare_bits",
		FT_UINT8, BASE_HEX, NULL, 0x0,
		"Spare bit(s)", HFILL }
	},
	{ &hf_nas_eps_security_header_type,
		{ "Security header type","nas_eps.security_header_type",
		FT_UINT8,BASE_DEC, VALS(security_header_type_vals), 0xf0,
		"Security_header_type", HFILL }
	},
	{ &hf_nas_eps_emm_eps_att_type,
		{ "EPS attach type","nas_eps.emm.eps_att_type",
		FT_UINT8,BASE_DEC, VALS(nas_eps_emm_eps_att_type_vals), 0x0,
		"EPS attach type", HFILL }
	},
	{ &hf_nas_eps_emm_nas_key_set_id,
		{ "NAS key set identifier","nas_eps.emm.nas_key_set_id",
		FT_UINT8,BASE_DEC, VALS(nas_eps_emm_NAS_key_set_identifier_vals), 0x0,
		"NAS key set identifier", HFILL }
	},
	{ &hf_nas_eps_emm_odd_even,
		{ "odd/even indic","nas_eps.emm.odd_even",
		FT_UINT8,BASE_DEC, NULL, 0x8,
		"odd/even indic", HFILL }
	},
	{ &hf_nas_eps_emm_type_of_id,
		{ "Type of identity","nas_eps.emm.type_of_id",
		FT_UINT8,BASE_DEC, VALS(nas_eps_emm_type_of_id_vals), 0x07,
		"Type of identity", HFILL }
	},
	{ &hf_nas_eps_emm_mme_grp_id,
		{ "MME Group ID","nas_eps.emm.mme_grp_id",
		FT_UINT16, BASE_DEC, NULL, 0x0,
		NULL, HFILL }
	},
	{ &hf_nas_eps_emm_mme_code,
		{ "MME Code","nas_eps.emm.mme_code",
		FT_UINT8, BASE_DEC, NULL, 0x0,
		NULL, HFILL }
	},
	{ &hf_nas_eps_emm_m_tmsi,
		{ "M-TMSI","nas_eps.emm.m_tmsi",
		FT_UINT32, BASE_HEX, NULL, 0x0,
		NULL, HFILL }
	},
	{ &hf_nas_eps_esm_msg_cont,
		{ "ESM message container contents","nas_eps.emm.esm_msg_cont",
		FT_BYTES, BASE_NONE, NULL, 0x0,
		"ESM message container contents", HFILL }
	},
	{ &hf_nas_eps_emm_EPS_attach_result,
		{ "Type of identity","nas_eps.emm.EPS_attach_result",
		FT_UINT8,BASE_DEC, VALS(nas_eps_emm_EPS_attach_result_values), 0x0,
		"Type of identity", HFILL }
	},
	{ &hf_nas_eps_emm_spare_half_octet,
		{ "Spare half octet","nas_eps.emm.EPS_attach_result",
		FT_UINT8,BASE_DEC, NULL, 0x0,
		"Spare half octet", HFILL }
	},
	{ &hf_nas_eps_emm_res,
		{ "RES","nas_eps.emm.res",
		FT_BYTES, BASE_HEX, NULL, 0x0,
		"RES", HFILL }
	},
	{ &hf_nas_eps_emm_cause,
		{ "Cause","nas_eps.emm.cause",
		FT_UINT8,BASE_DEC, VALS(nas_eps_emm_cause_values), 0x0,
		"Cause", HFILL }
	},
	{ &hf_nas_eps_emm_id_type2,
		{ "Identity type 2","nas_eps.emm.id_type2",
		FT_UINT8,BASE_DEC, VALS(nas_eps_emm_id_type2_vals), 0x0,
		NULL, HFILL }
	},
	{ &hf_nas_eps_emm_short_mac,
		{ "Short MAC value","nas_eps.emm.short_mac",
		FT_BYTES, BASE_HEX, NULL, 0x0,
		"Short MAC value", HFILL }
	},
	{ &hf_nas_eps_emm_128eea0,
		{ "128-EEA0","nas_eps.emm.128eea0",
		FT_BOOLEAN, 8, TFS(&nas_eps_emm_supported_flg_value), 0x80,
		NULL, HFILL }
	},
	{ &hf_nas_eps_emm_128eea1,
		{ "128-EEA1","nas_eps.emm.128eea1",
		FT_BOOLEAN, 8, TFS(&nas_eps_emm_supported_flg_value), 0x40,
		NULL, HFILL }
	},
	{ &hf_nas_eps_emm_128eea2,
		{ "128-EEA2","nas_eps.emm.128eea2",
		FT_BOOLEAN, 8, TFS(&nas_eps_emm_supported_flg_value), 0x20,
		NULL, HFILL }
	},
	{ &hf_nas_eps_emm_128eea3,
		{ "128-EEA3","nas_eps.emm.128eea3",
		FT_BOOLEAN, 8, TFS(&nas_eps_emm_supported_flg_value), 0x10,
		NULL, HFILL }
	},
	{ &hf_nas_eps_emm_128eea4,
		{ "128-EEA4","nas_eps.emm.128eea4",
		FT_BOOLEAN, 8, TFS(&nas_eps_emm_supported_flg_value), 0x08,
		NULL, HFILL }
	},
	{ &hf_nas_eps_emm_128eea5,
		{ "128-EEA5","nas_eps.emm.128eea5",
		FT_BOOLEAN, 8, TFS(&nas_eps_emm_supported_flg_value), 0x04,
		NULL, HFILL }
	},
	{ &hf_nas_eps_emm_128eea6,
		{ "128-EEA6","nas_eps.emm.128eea6",
		FT_BOOLEAN, 8, TFS(&nas_eps_emm_supported_flg_value), 0x02,
		NULL, HFILL }
	},
	{ &hf_nas_eps_emm_128eea7,
		{ "128-EEA7","nas_eps.emm.128eea7",
		FT_BOOLEAN, 8, TFS(&nas_eps_emm_supported_flg_value), 0x01,
		NULL, HFILL }
	},
	{ &hf_nas_eps_emm_128eia1,
		{ "128-EIA1","nas_eps.emm.128eia1",
		FT_BOOLEAN, 8, TFS(&nas_eps_emm_supported_flg_value), 0x40,
		NULL, HFILL }
	},
	{ &hf_nas_eps_emm_128eia2,
		{ "128-EIA2","nas_eps.emm.128eia2",
		FT_BOOLEAN, 8, TFS(&nas_eps_emm_supported_flg_value), 0x20,
		NULL, HFILL }
	},
	{ &hf_nas_eps_emm_128eia3,
		{ "128-EIA3","nas_eps.emm.128eia3",
		FT_BOOLEAN, 8, TFS(&nas_eps_emm_supported_flg_value), 0x10,
		NULL, HFILL }
	},
	{ &hf_nas_eps_emm_128eia4,
		{ "128-EIA4","nas_eps.emm.128eia4",
		FT_BOOLEAN, 8, TFS(&nas_eps_emm_supported_flg_value), 0x08,
		NULL, HFILL }
	},
	{ &hf_nas_eps_emm_128eia5,
		{ "128-EIA5","nas_eps.emm.128eia5",
		FT_BOOLEAN, 8, TFS(&nas_eps_emm_supported_flg_value), 0x04,
		NULL, HFILL }
	},
	{ &hf_nas_eps_emm_128eia6,
		{ "128-EIA6","nas_eps.emm.128eia6",
		FT_BOOLEAN, 8, TFS(&nas_eps_emm_supported_flg_value), 0x02,
		NULL, HFILL }
	},
	{ &hf_nas_eps_emm_128eia7,
		{ "128-EIA7","nas_eps.emm.128eia7",
		FT_BOOLEAN, 8, TFS(&nas_eps_emm_supported_flg_value), 0x01,
		NULL, HFILL }
	},


	{ &hf_nas_eps_emm_128uea0,
		{ "128-UEA0","nas_eps.emm.128uea0",
		FT_BOOLEAN, 8, TFS(&nas_eps_emm_supported_flg_value), 0x80,
		NULL, HFILL }
	},
	{ &hf_nas_eps_emm_128uea1,
		{ "128-UEA1","nas_eps.emm.128uea1",
		FT_BOOLEAN, 8, TFS(&nas_eps_emm_supported_flg_value), 0x40,
		NULL, HFILL }
	},
	{ &hf_nas_eps_emm_128uea2,
		{ "128-UEA2","nas_eps.emm.128uea2",
		FT_BOOLEAN, 8, TFS(&nas_eps_emm_supported_flg_value), 0x20,
		NULL, HFILL }
	},
	{ &hf_nas_eps_emm_128uea3,
		{ "128-UEA3","nas_eps.emm.128uea3",
		FT_BOOLEAN, 8, TFS(&nas_eps_emm_supported_flg_value), 0x10,
		NULL, HFILL }
	},
	{ &hf_nas_eps_emm_128uea4,
		{ "128-UEA4","nas_eps.emm.128uea4",
		FT_BOOLEAN, 8, TFS(&nas_eps_emm_supported_flg_value), 0x08,
		NULL, HFILL }
	},
	{ &hf_nas_eps_emm_128uea5,
		{ "128-UEA5","nas_eps.emm.128uea5",
		FT_BOOLEAN, 8, TFS(&nas_eps_emm_supported_flg_value), 0x04,
		NULL, HFILL }
	},
	{ &hf_nas_eps_emm_128uea6,
		{ "128-UEA6","nas_eps.emm.128uea6",
		FT_BOOLEAN, 8, TFS(&nas_eps_emm_supported_flg_value), 0x02,
		NULL, HFILL }
	},
	{ &hf_nas_eps_emm_128uea7,
		{ "128-UEA7","nas_eps.emm.128uea7",
		FT_BOOLEAN, 8, TFS(&nas_eps_emm_supported_flg_value), 0x01,
		NULL, HFILL }
	},

	{ &hf_nas_eps_active_flg,
		{ "Active flag", "nas_eps.emm.active_flg",
		FT_BOOLEAN, 8, TFS(&nas_eps_emm_active_flg_value), 0x0,
		"Active flag", HFILL }
	},
	{ &hf_nas_eps_eps_update_result_value,
		{ "EPS update result value","nas_eps.emm.eps_update_result_value",
		FT_UINT8,BASE_DEC, VALS(nas_eps_emm_eps_update_result_vals), 0x0,
		"EPS update result value", HFILL }
	},
	{ &hf_nas_eps_eps_update_type_value,
		{ "EPS update type value", "nas_eps.emm.update_type_value",
		FT_UINT8,BASE_DEC, VALS(nas_eps_emm_eps_update_type_vals), 0x0,
		"EPS update type value", HFILL }
	},
	{ &hf_nas_eps_service_type,
		{ "Service type", "nas_eps.emm.service_type",
		FT_UINT8,BASE_DEC, VALS(nas_eps_service_type_vals), 0x0,
		"Service type", HFILL }
	},
	/* ESM hf cvariables */
	{ &hf_nas_eps_msg_esm_type, 
		{ "NAS EPS session management messages",	"nas_eps.nas_msg_esm_type",
		FT_UINT8, BASE_HEX, VALS(nas_msg_esm_strings), 0x0,
		"", HFILL }
	},
	{ &hf_nas_eps_esm_elem_id,
		{ "Element ID",	"nas_eps.esm.elem_id",
		FT_UINT8, BASE_DEC, NULL, 0,
		"", HFILL }
	},
	{ &hf_nas_eps_esm_proc_trans_id,
		{ "Procedure transaction identity",	"nas_eps.esm.proc_trans_id",
		FT_UINT8, BASE_DEC, NULL, 0,
		"", HFILL }
	},
	{ &hf_nas_eps_esm_pdn_type,
		{ "PDN type",	"nas_eps.nas_eps_esm_pdn_type",
		FT_UINT8, BASE_DEC, VALS(nas_eps_esm_pdn_type_values), 0x0,
		NULL, HFILL }
	},
	{ &hf_nas_eps_esm_request_type,
		{ "Request type",	"nas_eps.esm_request_type",
		FT_UINT8, BASE_HEX, VALS(nas_eps_esm_request_type_values), 0x0,
		NULL, HFILL }
	},
  };

	/* Setup protocol subtree array */
#define	NUM_INDIVIDUAL_ELEMS	2
	static gint *ett[NUM_INDIVIDUAL_ELEMS +
		NUM_NAS_EPS_COMMON_ELEM +
		NUM_NAS_MSG_EMM + NUM_NAS_EMM_ELEM+
		NUM_NAS_MSG_ESM + NUM_NAS_ESM_ELEM];

	ett[0] = &ett_nas_eps;
	ett[1] = &ett_nas_eps_esm_msg_cont;

	last_offset = NUM_INDIVIDUAL_ELEMS;

	for (i=0; i < NUM_NAS_EPS_COMMON_ELEM; i++, last_offset++)
	{
		ett_nas_eps_common_elem[i] = -1;
		ett[last_offset] = &ett_nas_eps_common_elem[i];
	}

	/* EMM */
	for (i=0; i < NUM_NAS_MSG_EMM; i++, last_offset++)
	{
		ett_nas_msg_emm[i] = -1;
		ett[last_offset] = &ett_nas_msg_emm[i];
	}

	for (i=0; i < NUM_NAS_EMM_ELEM; i++, last_offset++)
	{
		ett_nas_eps_emm_elem[i] = -1;
		ett[last_offset] = &ett_nas_eps_emm_elem[i];
	}
	/* EPS */
	for (i=0; i < NUM_NAS_MSG_ESM; i++, last_offset++)
	{
		ett_nas_msg_esm[i] = -1;
		ett[last_offset] = &ett_nas_msg_esm[i];
	}

	for (i=0; i < NUM_NAS_ESM_ELEM; i++, last_offset++)
	{
		ett_nas_eps_esm_elem[i] = -1;
		ett[last_offset] = &ett_nas_eps_esm_elem[i];
	}

	/* Register protocol */
	proto_nas_eps = proto_register_protocol(PNAME, PSNAME, PFNAME);
	/* Register fields and subtrees */
	proto_register_field_array(proto_nas_eps, hf, array_length(hf));
	proto_register_subtree_array(ett, array_length(ett));
 
	/* Register dissector */
	register_dissector(PFNAME, dissect_nas_eps, proto_nas_eps);
}
